/*
 ************************************************************************
 *                                                                      *
 * udpTrain1 -- training UDP                                            *
 *                                                                      *
 *             this makes a box, plate, or wire (centered at origin)    *
 *             and returns its area and volume (for training purposes)  *
 *             sensitivities are computed analytically                  *
 *                                                                      *
 *             Written by John Dannenhoffer @ Geocentric Technologies   *
 *                                                                      *
 ************************************************************************
 */

/*
 * Copyright (C) 2025  John F. Dannenhoffer, III (Geocentric Technologies))
 *
 * This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *     MA  02110-1301  USA
 */

/* uncomment the following to get DEBUG printouts */
//#define DEBUG 1

/* the number of "input" Bodys

   this only needs to be specified if this is a UDF (user-defined
   function) that consumes Bodys from OpenCSM's stack.  (the default
   value is 0).

   if NUMUDPINPUTBODYS>0  then exactly NUMUDPINPUTBODYS are in emodel
   if NUMUDPINPUTBODYS<0  then up to  -NUMUDPINPUTBODYS are in emodel
*/
#define NUMUDPINPUTBODYS 0

/*  the name of the routine to clean up private data

    this only needs to be specified if the UDP hangs private
    data onto udps[iudp].data (which is a void*), and that data
    could not be freed by a simple call to EG_free().  cases where
    this happens is when udps[iudp].data points to a structure that
    contains components that had be allocated via separate calls
    to EG_alloc (or malloc) or which used an allocator other
    than EG_alloc
*/
#define FREEUDPDATA(A) freePrivateData(A)
static int freePrivateData(void *data);

#define COPYUDPDATA(SRC,TGT) copyPrivateData(SRC,TGT)
static int copyPrivateData(/*@null@*/void *src, void **tgt);

/* the number of arguments (specified below) */
#define NUMUDPARGS 6

/* set up the necessary structures (uses NUMUDPARGS) */
#include "udpUtilities.h"

/* shorthand macros for accessing argument values and velocities */
#define LENX(      IUDP  )  ((double *) (udps[IUDP].arg[0].val))[0]
#define LENX_DOT(  IUDP  )  ((double *) (udps[IUDP].arg[0].dot))[0]
#define LENX_SIZ(  IUDP  )               udps[IUDP].arg[0].size

#define LENY(      IUDP  )  ((double *) (udps[IUDP].arg[1].val))[0]
#define LENY_DOT(  IUDP  )  ((double *) (udps[IUDP].arg[1].dot))[0]
#define LENY_SIZ(  IUDP  )               udps[IUDP].arg[1].size

#define LENZ(      IUDP  )  ((double *) (udps[IUDP].arg[2].val))[0]
#define LENZ_DOT(  IUDP  )  ((double *) (udps[IUDP].arg[2].dot))[0]
#define LENZ_SIZ(  IUDP  )               udps[IUDP].arg[2].size

#define CENTER(    IUDP,I)  ((double *) (udps[IUDP].arg[3].val))[I]
#define CENTER_DOT(IUDP,I)  ((double *) (udps[IUDP].arg[3].dot))[I]
#define CENTER_SIZ(IUDP  )               udps[IUDP].arg[3].size

#define AREA(      IUDP  )  ((double *) (udps[IUDP].arg[4].val))[0]
#define AREA_DOT(  IUDP  )  ((double *) (udps[IUDP].arg[4].dot))[0]

#define VOLUME(    IUDP  )  ((double *) (udps[IUDP].arg[5].val))[0]
#define VOLUME_DOT(IUDP  )  ((double *) (udps[IUDP].arg[5].dot))[0]

/* data about possible arguments
      argNames: argument name (must be all lower case)
      argTypes: argument type: +ATTRINT      integer input
                               -ATTRINT      integer output
                               +ATTRREAL     double input  (no sensitivities)
                               -ATTRREAL     double output (no sensitivities)
                               +ATTRREALSEN  double input  (with sensitivities)
                               -ATTRREALSEN  double output (with sensitivities)
                               +ATTRSTRING   string input
                               -ATTRSTRING   *** cannot be used ***
                               +ATTRFILE     input file
                               -ATTRFILE     *** cannot be used ***
                               +ATTRREBUILD  forces rebuild if any variable in
                                             semi-colon-separated list has been changed
                               -ATTRREBUILD  *** cannot be used ***
                               +ATTRRECYCLE  forces rebuild (always) by blocking recycling
                               -ATTRRECYCLE  *** cannot be used ***
      argIdefs: default value for ATTRINT
      argDdefs: default value for ATTRREAL or ATTRREALSEN */
static char  *argNames[NUMUDPARGS] = {"lenx",      "leny",       "lenz",
                                      "center",    "area",       "volume",    };
static int    argTypes[NUMUDPARGS] = {ATTRREALSEN,  ATTRREALSEN, ATTRREALSEN,
                                      ATTRREALSEN, -ATTRREALSEN, -ATTRREALSEN,};
static int    argIdefs[NUMUDPARGS] = {0,           0,            0,
                                      0,           0,            0,           };
static double argDdefs[NUMUDPARGS] = {0.,          0.,           0.,
                                      0.,          0.,           0.,          };

/* get utility routines: udpErrorStr, udpInitialize, udpReset, udpSet,
                         udpGet, udpVel, udpClean, udpMesh */
#include "udpUtilities.c"


/*
 ************************************************************************
 *                                                                      *
 *   udpExecute - execute the primitive                                 *
 *                                                                      *
 ************************************************************************
 */

int
udpExecute(ego  context,                /* (in)  EGADS context */
           ego  *ebody,                 /* (out) Body (or model) pointer */
           int  *nMesh,                 /* (out) number of associated meshes */
           char *string[])              /* (out) error message */
{
    int     status = EGADS_SUCCESS;
    int     type=0, sense[4];
    double  node1[3], node2[3], node3[3], node4[3];
    double  data[18], trange[2];
    char    *message=NULL;
    ego     enodes[4], tempNodes[2], ecurve, eedges[8], eloop, eface;
    udp_T   *udps = *Udps;

    ROUTINE(udpExecute);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    /* debug printing of the input arguments */
    printf("udpExecute(context=%llx)\n", (long long)context);
    printf("lenx(0)       = %f\n", LENX(    0));
    printf("lenx_dot(0)   = %f\n", LENX_DOT(0));
    printf("leny(0)       = %f\n", LENY(    0));
    printf("leny_dot(0)   = %f\n", LENY_DOT(0));
    printf("lenz(0)       = %f\n", LENZ(    0));
    printf("lenz_dot(0)   = %f\n", LENZ_DOT(0));
    if (CENTER_SIZ(0) == 3) {
        printf("center(0)     = %f %f %f\n", CENTER(    0,0), CENTER(    0,1), CENTER(    0,2));
        printf("center_dot(0) = %f %f %f\n", CENTER_DOT(0,0), CENTER_DOT(0,1), CENTER_DOT(0,2));
    }
#endif

    /* default return values */
    *ebody  = NULL;
    *nMesh  = 0;
    *string = NULL;

    /* the place where messages to the user are placed */
    MALLOC(message, char, 100);
    message[0] = '\0';

    /* check arguments */
    if (LENX_SIZ(0) > 1) {
        snprintf(message, 100, "\"lenx\" should be a scalar");
        status  = EGADS_RANGERR;
        goto cleanup;

    } else if (LENX(0) < 0) {
        snprintf(message, 100, "\"lenx\" (=%f) should be non-negative", LENX(0));
        status  = EGADS_RANGERR;
        goto cleanup;

    } else if (LENY_SIZ(0) > 1) {
        snprintf(message, 100, "\"leny\" should be a scalar");
        status  = EGADS_RANGERR;
        goto cleanup;

    } else if (LENY(0) < 0) {
        snprintf(message, 100, "\"leny\" (-%f) should be non-negative", LENY(0));
        status  = EGADS_RANGERR;
        goto cleanup;

    } else if (LENZ_SIZ(0) > 1) {
        snprintf(message, 100, "\"lenz\" should be a scalar");
        status  = EGADS_RANGERR;
        goto cleanup;

    } else if (LENZ(0) < 0) {
        snprintf(message, 100, "\"lenz\" (=%f) should be non-negative", LENZ(0));
        status  = EGADS_RANGERR;
        goto cleanup;

    } else if (LENX(0) <= 0 && LENY(0) <= 0 && LENZ(0) <= 0) {
        snprintf(message, 100, "cannot have \"lenx\"=\"leny\"=\"lenz\"=0");
        status  = EGADS_GEOMERR;
        goto cleanup;

    } else if (CENTER_SIZ(0) == 1) {
        // CENTER is not used

    } else if (CENTER_SIZ(0) != 3) {
        snprintf(message, 100, "\"center\" should contain 3 entries");
        status  = EGADS_GEOMERR;
        goto cleanup;
    }

    /* make private data (not needed here, but included to
       show how one would do this */
    if (numUdp == 0) {
        MALLOC(udps[0].data, char, 30);
    }

    /* if not after a ocsmCopy (which happens when making finite difference
       senstivities), create the string in  the private data */
    if (udps[0].data != NULL) {
        strcpy((char*)(udps[0].data), "this is test private data");
    }

    /* cache copy of arguments for future use */
    status = cacheUdp(NULL);
    CHECK_STATUS(cacheUdp);

#ifdef DEBUG
    /* debug printing of cached input arguments */
    printf("lenx[%d]       = %f\n", numUdp, LENX(    numUdp));
    printf("lenx_dot[%d]   = %f\n", numUdp, LENX_DOT(numUdp));
    printf("leny[%d]       = %f\n", numUdp, LENY(    numUdp));
    printf("leny_dot[%d]   = %f\n", numUdp, LENY_DOT(numUdp));
    printf("lenz[%d]       = %f\n", numUdp, LENZ(    numUdp));
    printf("lenz_dot[%d]   = %f\n", numUdp, LENZ_DOT(numUdp));
    if (CENTER_SIZ(numUdp) == 3) {
        printf("center[%d]     = %f %f %f\n", numUdp, CENTER(    numUdp,0), CENTER(    numUdp,1), CENTER(    numUdp,2));
        printf("center_dot[%d] = %f %f %f\n", numUdp, CENTER_DOT(numUdp,0), CENTER_DOT(numUdp,1), CENTER_DOT(numUdp,2));
    }
#endif

    /* check for 3D SolidBody (and make if requested) */
    if (LENX(0) > 0 && LENY(0) > 0 && LENZ(0) > 0)  {

        /*
                ^ Y
                |
                4----11----8
               /:         /|
              3 :        7 |
             /  4       /  8
            3----12----7   |
            |   2-----9|---6  --> X
            |  '       |  /
            2 1        6 5
            |'         |/
            1----10----5
           /
          Z

        */

        data[0] = -LENX(0)/2;
        data[1] = -LENY(0)/2;
        data[2] = -LENZ(0)/2;
        data[3] =  LENX(0);
        data[4] =  LENY(0);
        data[5] =  LENZ(0);

        /* move the Body if CENTER has 3 values */
        if (CENTER_SIZ(0) == 3) {
            data[0] += CENTER(0,0);
            data[1] += CENTER(0,1);
            data[2] += CENTER(0,2);
        }

        /* make SolidBody */
        status = EG_makeSolidBody(context, BOX, data, ebody);
        CHECK_STATUS(EG_makeSolidBody);

        SPLINT_CHECK_FOR_NULL(*ebody);

        /* set the output value(s) */
        status = EG_getMassProperties(*ebody, data);
        CHECK_STATUS(EG_getMassProperties);

        AREA(  numUdp) = data[1];
        VOLUME(numUdp) = data[0];

        /* remember this model (Body) */
        udps[numUdp].ebody = *ebody;

        goto cleanup;

    /* WireBody parallel to X axis */
    } else if (LENY(0) == 0 && LENZ(0) == 0) {
        type = OCSM_WIRE_BODY;
        node1[0] = -LENX(0)/2;   node1[1] = 0;            node1[2] = 0;
        node2[0] = +LENX(0)/2;   node2[1] = 0;            node2[2] = 0;

    /* WireBody parallel to Y axis */
    } else if (LENZ(0) == 0 && LENX(0) == 0) {
        type = OCSM_WIRE_BODY;
        node1[0] = 0;            node1[1] = -LENY(0)/2;   node1[2] = 0;
        node2[0] = 0;            node2[1] = +LENY(0)/2;   node2[2] = 0;

    /* WireBody parallel to Z axis */
    } else if (LENX(0) == 0 && LENY(0) == 0) {
        type = OCSM_WIRE_BODY;
        node1[0] = 0;            node1[1] = 0;            node1[2] = -LENZ(0)/2;
        node2[0] = 0;            node2[1] = 0;            node2[2] = +LENZ(0)/2;

    /* SheetBody parallel to XY plane */
    } else if (LENZ(0) == 0) {
        type = OCSM_SHEET_BODY;
        node1[0] = -LENX(0)/2;   node1[1] = -LENY(0)/2;   node1[2] = 0;
        node2[0] = +LENX(0)/2;   node2[1] = -LENY(0)/2;   node2[2] = 0;
        node3[0] = +LENX(0)/2;   node3[1] = +LENY(0)/2;   node3[2] = 0;
        node4[0] = -LENX(0)/2;   node4[1] = +LENY(0)/2;   node4[2] = 0;

    /* SheetBody parallel to YZ plane */
    } else if (LENX(0) == 0) {
        type = OCSM_SHEET_BODY;
        node1[0] = 0;            node1[1] = -LENY(0)/2;   node1[2] = -LENZ(0)/2;
        node2[0] = 0;            node2[1] = +LENY(0)/2;   node2[2] = -LENZ(0)/2;
        node3[0] = 0;            node3[1] = +LENY(0)/2;   node3[2] = +LENZ(0)/2;
        node4[0] = 0;            node4[1] = -LENY(0)/2;   node4[2] = +LENZ(0)/2;

    /* SheetBody parallel to ZX plane */
    } else if (LENY(0) == 0) {
        type = OCSM_SHEET_BODY;
        node1[0] = -LENX(0)/2;   node1[1] = 0;            node1[2] = -LENZ(0)/2;
        node2[0] = -LENX(0)/2;   node2[1] = 0;            node2[2] = +LENZ(0)/2;
        node3[0] = +LENX(0)/2;   node3[1] = 0;            node3[2] = +LENZ(0)/2;
        node4[0] = +LENX(0)/2;   node4[1] = 0;            node4[2] = -LENZ(0)/2;
    }

    /* make the WireBody if required */
    if (type == OCSM_WIRE_BODY) {

        /* move the Nodes if CENTER has 3 values */
        if (CENTER_SIZ(0) == 3) {
            node1[0] += CENTER(0,0);   node1[1] += CENTER(0,1);   node1[2] += CENTER(0,2);
            node2[0] += CENTER(0,0);   node2[1] += CENTER(0,1);   node2[2] += CENTER(0,2);
        }

        /* make Nodes */
        status = EG_makeTopology(context, NULL, NODE, 0, node1, 0, NULL, NULL, &(enodes[0]));
        CHECK_STATUS(EG_makeTopology);

        status = EG_makeTopology(context, NULL, NODE, 0, node2, 0, NULL, NULL, &(enodes[1]));
        CHECK_STATUS(EG_makeTopology);

        /* make the Line */
        data[0] = node1[0];
        data[1] = node1[1];
        data[2] = node1[2];
        data[3] = node2[0] - node1[0];
        data[4] = node2[1] - node1[1];
        data[5] = node2[2] - node1[2];
        status = EG_makeGeometry(context, CURVE, LINE, NULL, NULL, data, &ecurve);
        CHECK_STATUS(EG_makeGeometry);

        /* get the parameter range */
        status = EG_invEvaluate(ecurve, node1, &(trange[0]), data);
        CHECK_STATUS(EG_invEvaluate);

        status = EG_invEvaluate(ecurve, node2, &(trange[1]), data);
        CHECK_STATUS(EG_invEvaluate);

        /* make Edge */
        status = EG_makeTopology(context, ecurve, EDGE, TWONODE, trange,
                                 2, enodes, NULL, &(eedges[0]));
        CHECK_STATUS(EG_makeTopology);

        /* make Loop from this Edge */
        sense[0] = SFORWARD;
        status = EG_makeTopology(context, NULL, LOOP, OPEN, NULL, 1, eedges, sense, &eloop);
        CHECK_STATUS(EG_makeTopology);

        /* create the WireBody (which will be returned) */
        status = EG_makeTopology(context, NULL, BODY, WIREBODY, NULL, 1, &eloop, NULL, ebody);
        CHECK_STATUS(EG_makeTopology);
        if (*ebody == NULL) goto cleanup;    // needed for splint

        /* set the output value(s) */
        AREA(  numUdp) = 0;
        VOLUME(numUdp) = 0;

        /* remember this model (Body) */
        udps[numUdp].ebody = *ebody;

    /* make the SheetBody if required */
    } else if (type == OCSM_SHEET_BODY) {

        /*
                  y,z,x
                    ^
                    :
              4----<3----3
              |     :    |
              4v    +- - 2^ - -> x,y,z
              |          |
              1----1>----2
        */

        /* move the Nodes if CENTER has 3 values */
        if (CENTER_SIZ(0) == 3) {
            node1[0] += CENTER(0,0);   node1[1] += CENTER(0,1);   node1[2] += CENTER(0,2);
            node2[0] += CENTER(0,0);   node2[1] += CENTER(0,1);   node2[2] += CENTER(0,2);
            node3[0] += CENTER(0,0);   node3[1] += CENTER(0,1);   node3[2] += CENTER(0,2);
            node4[0] += CENTER(0,0);   node4[1] += CENTER(0,1);   node4[2] += CENTER(0,2);
        }

       /* make Nodes */
        status = EG_makeTopology(context, NULL, NODE, 0, node1, 0, NULL, NULL, &(enodes[0]));
        CHECK_STATUS(EG_makeTopology);

        status = EG_makeTopology(context, NULL, NODE, 0, node2, 0, NULL, NULL, &(enodes[1]));
        CHECK_STATUS(EG_makeTopology);

        status = EG_makeTopology(context, NULL, NODE, 0, node3, 0, NULL, NULL, &(enodes[2]));
        CHECK_STATUS(EG_makeTopology);

        status = EG_makeTopology(context, NULL, NODE, 0, node4, 0, NULL, NULL, &(enodes[3]));
        CHECK_STATUS(EG_makeTopology);

        /* make the Line 1 */
        data[0] = node1[0];
        data[1] = node1[1];
        data[2] = node1[2];
        data[3] = node2[0] - node1[0];
        data[4] = node2[1] - node1[1];
        data[5] = node2[2] - node1[2];
        status = EG_makeGeometry(context, CURVE, LINE, NULL, NULL, data, &ecurve);
        CHECK_STATUS(EG_makeGeometry);

        /* get the parameter range */
        status = EG_invEvaluate(ecurve, node1, &(trange[0]), data);
        CHECK_STATUS(EG_invEvaluate);

        status = EG_invEvaluate(ecurve, node2, &(trange[1]), data);
        CHECK_STATUS(EG_invEvaluate);

        /* make Edge */
        tempNodes[0] = enodes[0];
        tempNodes[1] = enodes[1];

        status = EG_makeTopology(context, ecurve, EDGE, TWONODE, trange,
                                 2, tempNodes, NULL, &(eedges[0]));
        CHECK_STATUS(EG_makeTopology);

        /* make the Line 2 */
        data[0] = node2[0];
        data[1] = node2[1];
        data[2] = node2[2];
        data[3] = node3[0] - node2[0];
        data[4] = node3[1] - node2[1];
        data[5] = node3[2] - node2[2];
        status = EG_makeGeometry(context, CURVE, LINE, NULL, NULL, data, &ecurve);
        CHECK_STATUS(EG_makeGeometry);

        /* get the parameter range */
        status = EG_invEvaluate(ecurve, node2, &(trange[0]), data);
        CHECK_STATUS(EG_invEvaluate);

        status = EG_invEvaluate(ecurve, node3, &(trange[1]), data);
        CHECK_STATUS(EG_invEvaluate);

        /* make Edge */
        tempNodes[0] = enodes[1];
        tempNodes[1] = enodes[2];

        status = EG_makeTopology(context, ecurve, EDGE, TWONODE, trange,
                                 2, tempNodes, NULL, &(eedges[1]));
        CHECK_STATUS(EG_makeTopology);

        /* make the Line 3 */
        data[0] = node3[0];
        data[1] = node3[1];
        data[2] = node3[2];
        data[3] = node4[0] - node3[0];
        data[4] = node4[1] - node3[1];
        data[5] = node4[2] - node3[2];
        status = EG_makeGeometry(context, CURVE, LINE, NULL, NULL, data, &ecurve);
        CHECK_STATUS(EG_makeGeometry);

        /* get the parameter range */
        status = EG_invEvaluate(ecurve, node3, &(trange[0]), data);
        CHECK_STATUS(EG_invEvaluate);

        status = EG_invEvaluate(ecurve, node4, &(trange[1]), data);
        CHECK_STATUS(EG_invEvaluate);

        /* make Edge */
        tempNodes[0] = enodes[2];
        tempNodes[1] = enodes[3];

        status = EG_makeTopology(context, ecurve, EDGE, TWONODE, trange,
                                 2, tempNodes, NULL, &(eedges[2]));
        CHECK_STATUS(EG_makeTopology);

        /* make the Line 4 */
        data[0] = node4[0];
        data[1] = node4[1];
        data[2] = node4[2];
        data[3] = node1[0] - node4[0];
        data[4] = node1[1] - node4[1];
        data[5] = node1[2] - node4[2];
        status = EG_makeGeometry(context, CURVE, LINE, NULL, NULL, data, &ecurve);
        CHECK_STATUS(EG_makeGeometry);

        /* get the parameter range */
        status = EG_invEvaluate(ecurve, node4, &(trange[0]), data);
        CHECK_STATUS(EG_invEvaluate);

        status = EG_invEvaluate(ecurve, node1, &(trange[1]), data);
        CHECK_STATUS(EG_invEvaluate);

        /* make Edge */
        tempNodes[0] = enodes[3];
        tempNodes[1] = enodes[0];

        status = EG_makeTopology(context, ecurve, EDGE, TWONODE, trange,
                                 2, tempNodes, NULL, &(eedges[3]));
        CHECK_STATUS(EG_makeTopology);

        /* make Loop from this Edge */
        sense[0] = SFORWARD;   sense[1] = SFORWARD;
        sense[2] = SFORWARD;   sense[3] = SFORWARD;

        status = EG_makeTopology(context, NULL, LOOP, CLOSED, NULL, 4, eedges, sense, &eloop);
        CHECK_STATUS(EG_makeTopology);

        /* make Face from the loop */
        status = EG_makeFace(eloop, SREVERSE, NULL, &eface);
        CHECK_STATUS(EG_makeFace);

        /* create the FaceBody (which will be returned) */
        status = EG_makeTopology(context, NULL, BODY, FACEBODY, NULL, 1, &eface, NULL, ebody);
        CHECK_STATUS(EG_makeTopology);
        if (*ebody == NULL) goto cleanup;    // needed for splint

        /* set the output value(s) */
        status = EG_getMassProperties(*ebody, data);
        CHECK_STATUS(EG_getMassProperties);

        AREA(  numUdp) = data[1];
        VOLUME(numUdp) = 0;

        /* remember this model (Body) */
        udps[numUdp].ebody = *ebody;
    }

cleanup:
#ifdef DEBUG
    printf("udpExecute -> numUdp=%d, *ebody=%llx\n", numUdp, (long long)(*ebody));
#endif

    if (strlen(message) > 0) {
        *string = message;
        printf("%s\n", message);
    } else if (status != EGADS_SUCCESS) {
        FREE(message);
        *string = udpErrorStr(status);
    } else {
        FREE(message);
    }

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   udpSensitivity - return sensitivity derivatives for the "real" argument *
 *                                                                      *
 ************************************************************************
 */

int
udpSensitivity(ego    ebody,            /* (in)  Body pointer */
               int    npnt,             /* (in)  number of points */
               int    entType,          /* (in)  OCSM entity type */
               int    entIndex,         /* (in)  OCSM entity index (bias-1) */
   /*@unused@*/double uvs[],            /* (in)  parametric coordinates for evaluation */
               double vels[])           /* (out) velocities */
{
    int    status = EGADS_SUCCESS;

    int    iudp, judp, i, inode, iedge, iface;
    double lenx_dot, leny_dot, lenz_dot, xcent_dot, ycent_dot, zcent_dot;
    double area_dot=0, volume_dot=0;

    ROUTINE(udpSensitivity);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    if (uvs != NULL) {
        printf("udpSensitivity(ebody=%llx, npnt=%d, entType=%d, entIndex=%d, uvs=%f %f)\n",
               (long long)ebody, npnt, entType, entIndex, uvs[0], uvs[1]);
    } else {
        printf("udpSensitivity(ebody=%llx, npnt=%d, entType=%d, entIndex=%d, uvs=NULL)\n",
               (long long)ebody, npnt, entType, entIndex);
    }
#endif

    /* the following line should be included if sensitivities
       are not computed analytically */
//$$$    return EGADS_NOLOAD;

    /* check that ebody matches one of the ebodys */
    iudp = 0;
    for (judp = 1; judp <= numUdp; judp++) {
        if (ebody == udps[judp].ebody) {
            iudp = judp;
            break;
        }
    }
    if (iudp <= 0) {
        status = EGADS_NOTMODEL;
        goto cleanup;
    }

    /* remember the velocity of center */
    if (CENTER_SIZ(iudp) == 3) {
        xcent_dot = CENTER_DOT(iudp,0);
        ycent_dot = CENTER_DOT(iudp,1);
        zcent_dot = CENTER_DOT(iudp,2);
    } else {
        xcent_dot = 0;
        ycent_dot = 0;
        zcent_dot = 0;
    }

    /* WireBody in X direction */
    if        (LENY(iudp) <= 0 && LENZ(iudp) <= 0) {
        if (entType == OCSM_NODE) {
            inode = entIndex;

            if        (inode == 1) {
                lenx_dot = -LENX_DOT(iudp)/2;   leny_dot = 0;   lenz_dot = 0;
            } else if (inode == 2) {
                lenx_dot = +LENX_DOT(iudp)/2;   leny_dot = 0;   lenz_dot = 0;
            } else {
                status = OCSM_NODE_NOT_FOUND;
                goto cleanup;
            }
        } else if (entType == OCSM_EDGE) {
            iedge = entIndex;

            if (iedge == 1) {
                lenx_dot = 0;   leny_dot = 0;   lenz_dot = 0;
            } else {
                status = OCSM_EDGE_NOT_FOUND;
                goto cleanup;
            }
        } else {
            status = OCSM_ILLEGAL_VALUE;
            goto cleanup;
        }

        /* return (constant) velocities, accounting for the movement
           of the CENTER if it set */
        for (i = 0; i < npnt; i++) {
            vels[3*i  ] = lenx_dot + xcent_dot;
            vels[3*i+1] = leny_dot + ycent_dot;
            vels[3*i+2] = lenz_dot + zcent_dot;
        }

    /* WireBody in Y direction */
    } else if (LENZ(iudp) <= 0 && LENX(iudp) <= 0) {
        if (entType == OCSM_NODE) {
            inode = entIndex;

            if        (inode == 1) {
                lenx_dot = 0;   leny_dot = -LENY_DOT(iudp)/2;   lenz_dot = 0;
            } else if (inode == 2) {
                lenx_dot = 0;   leny_dot = +LENY_DOT(iudp)/2;   lenz_dot = 0;
            } else {
                status = OCSM_NODE_NOT_FOUND;
                goto cleanup;
            }
        } else if (entType == OCSM_EDGE) {
            iedge = entIndex;

            if (iedge == 1) {
                lenx_dot = 0;   leny_dot = 0;   lenz_dot = 0;
            } else {
                status = OCSM_EDGE_NOT_FOUND;
                goto cleanup;
            }
        } else {
            status = OCSM_ILLEGAL_VALUE;
            goto cleanup;
        }

        /* return (constant) velocities, accounting for the movement
           of the CENTER if it set */
        for (i = 0; i < npnt; i++) {
            vels[3*i  ] = lenx_dot + xcent_dot;
            vels[3*i+1] = leny_dot + ycent_dot;
            vels[3*i+2] = lenz_dot + zcent_dot;
        }

    /* WireBody in Z direction */
    } else if (LENX(iudp) <= 0 && LENY(iudp) <= 0) {
        if (entType == OCSM_NODE) {
            inode = entIndex;

            if        (inode == 1) {
                lenx_dot = 0;   leny_dot = 0;   lenz_dot = -LENZ_DOT(iudp)/2;
            } else if (inode == 2) {
                lenx_dot = 0;   leny_dot = 0;   lenz_dot = +LENZ_DOT(iudp)/2;
            } else {
                status = OCSM_NODE_NOT_FOUND;
                goto cleanup;
            }
        } else if (entType == OCSM_EDGE) {
            iedge = entIndex;

            if (iedge == 1) {
                lenx_dot = 0;   leny_dot = 0;   lenz_dot = 0;
            } else {
                status = OCSM_EDGE_NOT_FOUND;
                goto cleanup;
            }
        } else {
            status = OCSM_ILLEGAL_VALUE;
            goto cleanup;
        }

        /* return (constant) velocities, accounting for the movement
           of the CENTER if it set */
        for (i = 0; i < npnt; i++) {
            vels[3*i  ] = lenx_dot + xcent_dot;
            vels[3*i+1] = leny_dot + ycent_dot;
            vels[3*i+2] = lenz_dot + zcent_dot;
        }

    /* SheetBody in XY plane (since the velocity on each Node and Edge is
                              a constant, we need to just compute one value) */
    } else if (LENZ(iudp) <= 0) {
        if        (entType == OCSM_NODE) {
            inode = entIndex;

            if        (inode == 1) {
                lenx_dot = -LENX_DOT(iudp)/2;   leny_dot = -LENY_DOT(iudp)/2;   lenz_dot = 0;
            } else if (inode == 2) {
                lenx_dot = +LENX_DOT(iudp)/2;   leny_dot = -LENY_DOT(iudp)/2;   lenz_dot = 0;
            } else if (inode == 3) {
                lenx_dot = +LENX_DOT(iudp)/2;   leny_dot = +LENY_DOT(iudp)/2;   lenz_dot = 0;
            } else if (inode == 4) {
                lenx_dot = -LENX_DOT(iudp)/2;   leny_dot = +LENY_DOT(iudp)/2;   lenz_dot = 0;
            } else {
                status = OCSM_NODE_NOT_FOUND;
                goto cleanup;
            }
        } else if (entType == OCSM_EDGE) {
            iedge = entIndex;

            if        (iedge == 1) {
                lenx_dot = 0;   leny_dot = -LENY_DOT(iudp)/2;   lenz_dot = 0;
            } else if (iedge == 2) {
                lenx_dot = +LENX_DOT(iudp)/2;   leny_dot = 0;   lenz_dot = 0;
            } else if (iedge == 3) {
                lenx_dot = 0;   leny_dot = +LENY_DOT(iudp)/2;   lenz_dot = 0;
            } else if (iedge == 4) {
                lenx_dot = -LENX_DOT(iudp)/2;   leny_dot = 0;   lenz_dot = 0;
            } else {
                status = OCSM_EDGE_NOT_FOUND;
                goto cleanup;
            }
        } else if (entType == OCSM_FACE) {
            iface = entIndex;

            if (iface == 1) {
                lenx_dot = 0;   leny_dot = 0;   lenz_dot = 0;
            } else {
                status = OCSM_FACE_NOT_FOUND;
                goto cleanup;
            }
        } else {
            status = OCSM_ILLEGAL_VALUE;
            goto cleanup;
        }

        /* return (constant) velocities, accounting for the movement
           of the CENTER if it set */
        for (i = 0; i < npnt; i++) {
            vels[3*i  ] = lenx_dot + xcent_dot;
            vels[3*i+1] = leny_dot + ycent_dot;
            vels[3*i+2] = lenz_dot + zcent_dot;
        }

        /* compute the sensitivities of the area */
        area_dot = LENX_DOT(iudp) * LENY(iudp) + LENY_DOT(iudp) * LENX(iudp);

    /* SheetBody in YZ plane (since the velocity on each Node and Edge is
                              a constant, we need to just compute one value) */
    } else if (LENX(iudp) <= 0) {
        if        (entType == OCSM_NODE) {
            inode = entIndex;

            if        (inode == 1) {
                lenx_dot = 0;   leny_dot = -LENY_DOT(iudp)/2;   lenz_dot = -LENZ_DOT(iudp)/2;
            } else if (inode == 2) {
                lenx_dot = 0;   leny_dot = +LENY_DOT(iudp)/2;   lenz_dot = -LENZ_DOT(iudp)/2;
            } else if (inode == 3) {
                lenx_dot = 0;   leny_dot = +LENY_DOT(iudp)/2;   lenz_dot = +LENZ_DOT(iudp)/2;
            } else if (inode == 4) {
                lenx_dot = 0;   leny_dot = -LENY_DOT(iudp)/2;   lenz_dot = +LENZ_DOT(iudp)/2;
            } else {
                status = OCSM_NODE_NOT_FOUND;
                goto cleanup;
            }
        } else if (entType == OCSM_EDGE) {
            iedge = entIndex;

            if        (iedge == 1) {
                lenx_dot = 0;   leny_dot = 0;   lenz_dot = -LENZ_DOT(iudp)/2;
            } else if (iedge == 2) {
                lenx_dot = 0;   leny_dot = +LENY_DOT(iudp)/2;   lenz_dot = 0;
            } else if (iedge == 3) {
                lenx_dot = 0;   leny_dot = 0;   lenz_dot = +LENZ_DOT(iudp)/2;
            } else if (iedge == 4) {
                lenx_dot = 0;   leny_dot = -LENY_DOT(iudp)/2;   lenz_dot = 0;
            } else {
                status = OCSM_EDGE_NOT_FOUND;
                goto cleanup;
            }
        } else if (entType == OCSM_FACE) {
            iface = entIndex;

            if (iface == 1) {
                lenx_dot = 0;   leny_dot = 0;   lenz_dot = 0;
            } else {
                status = OCSM_FACE_NOT_FOUND;
                goto cleanup;
            }
        } else {
            status = OCSM_ILLEGAL_VALUE;
            goto cleanup;
        }

        /* return (constant) velocities, accounting for the movement
           of the CENTER if it set */
        for (i = 0; i < npnt; i++) {
            vels[3*i  ] = lenx_dot + xcent_dot;
            vels[3*i+1] = leny_dot + ycent_dot;
            vels[3*i+2] = lenz_dot + zcent_dot;
        }

        /* compute the sensitivities of the area */
        area_dot = LENY_DOT(iudp) * LENZ(iudp) + LENZ_DOT(iudp) * LENY(iudp);

    /* SheetBody in ZX plane (since the velocity on each Node and Edge is
                              a constant, we need to just compute one value) */
    } else if (LENY(iudp) <= 0) {
        if        (entType == OCSM_NODE) {
            inode = entIndex;

            if        (inode == 1) {
                lenx_dot = -LENX_DOT(iudp)/2;   leny_dot = 0;   lenz_dot = -LENZ_DOT(iudp)/2;
            } else if (inode == 2) {
                lenx_dot = -LENX_DOT(iudp)/2;   leny_dot = 0;   lenz_dot = +LENZ_DOT(iudp)/2;
            } else if (inode == 3) {
                lenx_dot = +LENX_DOT(iudp)/2;   leny_dot = 0;   lenz_dot = +LENZ_DOT(iudp)/2;
            } else if (inode == 4) {
                lenx_dot = +LENX_DOT(iudp)/2;   leny_dot = 0;   lenz_dot = -LENZ_DOT(iudp)/2;
            } else {
                status = OCSM_NODE_NOT_FOUND;
                goto cleanup;
            }
        } else if (entType == OCSM_EDGE) {
            iedge = entIndex;

            if        (iedge == 1) {
                lenx_dot = -LENX_DOT(iudp)/2;   leny_dot = 0;   lenz_dot = 0;
            } else if (iedge == 2) {
                lenx_dot = 0;   leny_dot = 0;   lenz_dot = +LENZ_DOT(iudp)/2;
            } else if (iedge == 3) {
                lenx_dot = +LENX_DOT(iudp)/2;   leny_dot = 0;   lenz_dot = 0;
        } else if (iedge == 4) {
                lenx_dot = 0;   leny_dot = 0;   lenz_dot = -LENZ_DOT(iudp)/2;
            } else {
                status = OCSM_EDGE_NOT_FOUND;
                goto cleanup;
            }
        } else if (entType == OCSM_FACE) {
            iface = entIndex;

            if (iface == 1) {
                lenx_dot = 0;   leny_dot = 0;   lenz_dot = 0;
            } else {
                status = OCSM_FACE_NOT_FOUND;
                goto cleanup;
            }

        } else {
            status = OCSM_ILLEGAL_VALUE;
            goto cleanup;
        }

        /* return (constant) velocities, accounting for the movement
           of the CENTER if it set */
        for (i = 0; i < npnt; i++) {
            vels[3*i  ] = lenx_dot + xcent_dot;
            vels[3*i+1] = leny_dot + ycent_dot;
            vels[3*i+2] = lenz_dot + zcent_dot;
        }

        /* compute the sensitivities of the area */
        area_dot = LENZ_DOT(iudp) * LENX(iudp) + LENX_DOT(iudp) * LENZ(iudp);

    /* SolidBody (since the velocity on each Node, Edge, and Face is a
                  constant, we need just to compute one vaue) */
    } else {
        if        (entType == OCSM_NODE) {
            inode = entIndex;

            if        (inode == 1) {
                lenx_dot = -LENX_DOT(iudp)/2;   leny_dot = -LENY_DOT(iudp)/2;   lenz_dot = +LENZ_DOT(iudp)/2;
            } else if (inode == 2) {
                lenx_dot = -LENX_DOT(iudp)/2;   leny_dot = -LENY_DOT(iudp)/2;   lenz_dot = -LENZ_DOT(iudp)/2;
            } else if (inode == 3) {
                lenx_dot = -LENX_DOT(iudp)/2;   leny_dot = +LENY_DOT(iudp)/2;   lenz_dot = +LENZ_DOT(iudp)/2;
            } else if (inode == 4) {
                lenx_dot = -LENX_DOT(iudp)/2;   leny_dot = +LENY_DOT(iudp)/2;   lenz_dot = -LENZ_DOT(iudp)/2;
            } else if (inode == 5) {
                lenx_dot = +LENX_DOT(iudp)/2;   leny_dot = -LENY_DOT(iudp)/2;   lenz_dot = +LENZ_DOT(iudp)/2;
            } else if (inode == 6) {
                lenx_dot = +LENX_DOT(iudp)/2;   leny_dot = -LENY_DOT(iudp)/2;   lenz_dot = -LENZ_DOT(iudp)/2;
            } else if (inode == 7) {
                lenx_dot = +LENX_DOT(iudp)/2;   leny_dot = +LENY_DOT(iudp)/2;   lenz_dot = +LENZ_DOT(iudp)/2;
            } else if (inode == 8) {
                lenx_dot = +LENX_DOT(iudp)/2;   leny_dot = +LENY_DOT(iudp)/2;   lenz_dot = -LENZ_DOT(iudp)/2;
            } else {
                status = OCSM_NODE_NOT_FOUND;
                goto cleanup;
            }
        } else if (entType == OCSM_EDGE) {
            iedge = entIndex;

            if        (iedge == 1) {
                lenx_dot = -LENX_DOT(iudp)/2;   leny_dot = -LENY_DOT(iudp)/2;   lenz_dot =  0;
            } else if (iedge == 2) {
                lenx_dot = -LENX_DOT(iudp)/2;   leny_dot =  0;   lenz_dot = +LENZ_DOT(iudp)/2;
            } else if (iedge == 3) {
                lenx_dot = -LENX_DOT(iudp)/2;   leny_dot = +LENY_DOT(iudp)/2;   lenz_dot =  0;
            } else if (iedge == 4) {
                lenx_dot = -LENX_DOT(iudp)/2;   leny_dot =  0;   lenz_dot = -LENZ_DOT(iudp)/2;
            } else if (iedge == 5) {
                lenx_dot = +LENX_DOT(iudp)/2;   leny_dot = -LENY_DOT(iudp)/2;   lenz_dot =  0;
            } else if (iedge == 6) {
                lenx_dot = +LENX_DOT(iudp)/2;   leny_dot =  0;   lenz_dot = +LENZ_DOT(iudp)/2;
            } else if (iedge == 7) {
                lenx_dot = +LENX_DOT(iudp)/2;   leny_dot = +LENY_DOT(iudp)/2;   lenz_dot =  0;
            } else if (iedge == 8) {
                lenx_dot = +LENX_DOT(iudp)/2;   leny_dot =  0;   lenz_dot = -LENZ_DOT(iudp)/2;
            } else if (iedge == 9) {
                lenx_dot =  0;   leny_dot = -LENY_DOT(iudp)/2;   lenz_dot = -LENZ_DOT(iudp)/2;
            } else if (iedge == 10) {
                lenx_dot =  0;   leny_dot = -LENY_DOT(iudp)/2;   lenz_dot = +LENZ_DOT(iudp)/2;
            } else if (iedge == 11) {
                lenx_dot =  0;   leny_dot = +LENY_DOT(iudp)/2;   lenz_dot = -LENZ_DOT(iudp)/2;
            } else if (iedge == 12) {
                lenx_dot =  0;   leny_dot = +LENY_DOT(iudp)/2;   lenz_dot = +LENZ_DOT(iudp)/2;
            } else {
                status = OCSM_EDGE_NOT_FOUND;
                goto cleanup;
            }
        } else if (entType == OCSM_FACE) {
            iface = entIndex;

            if        (iface == 1) {
                lenx_dot = -LENX_DOT(iudp)/2;   leny_dot = 0;   lenz_dot = 0;
            } else if (iface == 2) {
                lenx_dot = +LENX_DOT(iudp)/2;   leny_dot = 0;   lenz_dot = 0;
            } else if (iface == 3) {
                lenx_dot = 0;   leny_dot = -LENY_DOT(iudp)/2;   lenz_dot = 0;
            } else if (iface == 4) {
                lenx_dot = 0;   leny_dot = +LENY_DOT(iudp)/2;   lenz_dot = 0;
            } else if (iface == 5) {
                lenx_dot = 0;   leny_dot = 0;   lenz_dot = -LENZ_DOT(iudp)/2;
            } else if (iface == 6) {
                lenx_dot = 0;   leny_dot = 0;   lenz_dot = +LENZ_DOT(iudp)/2;
            } else {
                status = OCSM_FACE_NOT_FOUND;
                goto cleanup;
            }

        } else {
            status = OCSM_ILLEGAL_VALUE;
            goto cleanup;
        }

        /* return (constant) velocities, accounting for the movement
           of the CENTER if it set */
        for (i = 0; i < npnt; i++) {
            vels[3*i  ] = lenx_dot + xcent_dot;
            vels[3*i+1] = leny_dot + ycent_dot;
            vels[3*i+2] = lenz_dot + zcent_dot;
        }

        /* compute the sensitivities of the area and volume */
        area_dot = 2 * (LENX_DOT(iudp) * (LENY(iudp) + LENZ(iudp))
                     +  LENY_DOT(iudp) * (LENZ(iudp) + LENX(iudp))
                     +  LENZ_DOT(iudp) * (LENX(iudp) + LENY(iudp)));
        volume_dot = LENX_DOT(iudp) * LENY(iudp) * LENZ(iudp)
                   + LENY_DOT(iudp) * LENZ(iudp) * LENX(iudp)
                   + LENZ_DOT(iudp) * LENX(iudp) * LENY(iudp);
    }

    AREA_DOT(  iudp) = area_dot;
    VOLUME_DOT(iudp) = volume_dot;

    status = EGADS_SUCCESS;

cleanup:
#ifdef DEBUG
    printf("udpSensitivity -> vels=%f %f %f\n", vels[0], vels[1], vels[2]);
#endif
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   freePrivateData - free private data (just an example)              *
 *                                                                      *
 ************************************************************************
 */

static int
freePrivateData(void  *data)            /* (in)  pointer to private data */
{
    int    status = EGADS_SUCCESS;

#ifdef DEBUG
    printf("freePrivateData(%s)\n", (char*)(data));
#endif

    /* note: this function would not be necessary if we are only calling
             EG_free (and then FREEUDPDATA would not be defined above).
             It is simply included here to show how one would write
             such a function if the allocation was more complicated
             than a simple EG_alloc() */

    EG_free(data);

//cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   copyPrivateData - copy private data (just an example)              *
 *                                                                      *
 ************************************************************************
 */

static int
copyPrivateData(
      /*@null@*/void  *src,             /* (in)  pointer to source private data */
                void  **tgt)            /* (in)  pointer to target private data */
{
    int    status = EGADS_SUCCESS;

    *tgt = NULL;

    if (src == NULL) goto cleanup;

    /* note: this function would not be necessary if we are only calling
             EG_free (and then COPYUDPDATA would not be defined above).
             It is simply included here to show how one would write
             such a function if the allocation was more complicated
             than a simple EG_alloc() */

    *tgt = (void *) malloc(sizeof(char)*(strlen((char*)src)+1));
    if (*tgt == NULL) {
        status = EGADS_MALLOC;
        goto cleanup;
    }

#ifdef DEBUG
    printf("copyPrivateData(src=%llx, tgt=%llx)\n", (long long)(src), (long long)(*tgt));
#endif

    strcpy(*tgt, src);

cleanup:
    return status;
}
