/*
 *      CAPS: Computational Aircraft Prototype Syntheses
 *
 *             Session09: Example of AIM that calls an analysis program
 *
 *      Copyright 2014-2025, Massachusetts Institute of Technology
 *      Licensed under The GNU Lesser General Public License, version 2.1
 *      See http://www.opensource.org/licenses/lgpl-2.1.php
 *
 */

#include <string.h>
#include "aimUtil.h"

/* uncomment to track the calls that CAPS makes to these various routines */
//#define DEBUG

enum aimInputs
{
  InputVariable = 1,           /* index is 1-based */
  NUMINPUT = InputVariable     /* Total number of inputs */
};

enum aimOutputs
{
  OutputVariable = 1,          /* index is 1-based */
  NUMOUT  = OutputVariable     /* Total number of outputs */
};

typedef struct {
  int nBody;
  ego *tess;
} aimStorage;


/****************** exposed AIM entry points -- Analysis **********************/

/* aimInitialize: Initialization Information for the AIM */
int
aimInitialize(int inst, /*@unused@*/ const char *unitSys, void *aimInfo,
              void **instStore, int *major, int *minor, int *nIn, int *nOut,
              int *nFields, /*@unused@*/ char ***fnames,
              /*@unused@*/ int **franks, /*@unused@*/ int **fInOut)
{
  int        status;
  aimStorage *aimStore = NULL;

#ifdef DEBUG
/*@-nullpass@*/
  printf("\n theAIM/aimInitialize  instance = %d  unitSys = %s!\n",
         inst, unitSys);
/*@+nullpass@*/
#endif

  /* specify the rev of this AIM */
  *major = 1;
  *minor = 0;

  /* specify the number of analysis inputs  defined in aimInputs
   *     and the number of analysis outputs defined in aimOutputs */
  *nIn  = NUMINPUT;
  *nOut = NUMOUT;

  /* return if "query" only */
  if (inst == -1) return CAPS_SUCCESS;

  /* specify the field variables this analysis can generate and consume */
  *nFields = 0;

  /* setup our AIM specific state */
  AIM_ALLOC(aimStore, 1, aimStorage, aimInfo, status);
  *instStore = aimStore;

  /* populate our instance storage */
  aimStore->nBody = 0;
  aimStore->tess  = NULL;

  return CAPS_SUCCESS;

cleanup:
  return status;
}


/* aimInputs: Input Information for the AIM */
int
aimInputs(/*@unused@*/ void *instStore, void *aimInfo, int index, char **ainame,
          capsValue *defval)
{
  int status = CAPS_SUCCESS;
  
#ifdef DEBUG
  printf(" theAIM/aimInputs  instance = %d  index  = %d!\n",
         aim_getInstance(aimInfo), index);
#endif

  /* fill in the required members based on the index */
 if (index == InputVariable) {
    *ainame              = AIM_NAME(InputVariable);
    defval->type         = Integer;
    defval->vals.integer = 0;
  } else {
    printf(" theAIM/aimInputs: Unknown index = %d!\n", index);
    return CAPS_BADINDEX;
  }

  AIM_NOTNULL(*ainame, aimInfo, status);

cleanup:
  return status;
}


/* aimUpdateState: The always the first call in the execution sequence */
int
aimUpdateState(/*@unused@*/ void *instStore, void *aimInfo,
               /*@unused@*/ capsValue *inputs)
{
#ifdef DEBUG
  printf(" theAIM/aimUpdateState        instance = %d!\n",
         aim_getInstance(aimInfo));
#endif

  return CAPS_SUCCESS;
}


/* aimPreAnalysis: Parse Inputs, Generate Input File(s) */
int
aimPreAnalysis(/*@unused@*/ const void *instStore, void *aimInfo,
               capsValue *inputs)
{
#ifdef DEBUG
  printf("\n theAIM/aimPreAnalysis instance = %d!\n", aim_getInstance(aimInfo));
#endif
  
  return CAPS_SUCCESS;
}


/* aimExecute: runs the Analysis & specifies the AIM does the execution */
int
aimExecute(/*@unused@*/ const void *instStor, /*@unused@*/ void *aimInfo,
           int *state)
{
#ifdef DEBUG
  printf(" theAIM/aimExecute instance = %d!\n", aim_getInstance(aimInfo));
#endif

  *state = 0;
  return CAPS_SUCCESS;
}


/* aimPostAnalysis: Perform any processing after the Analysis is run */
int
aimPostAnalysis(/*@unused@*/ void *instStore, void *aimInfo, int restart,
                /*@unused@*/ capsValue *inputs)
{
#ifdef DEBUG
  printf(" theAIM/aimPostAnalysis instance = %d!\n", aim_getInstance(aimInfo));
#endif

  return CAPS_SUCCESS;
}


/* aimOutputs: Output Information for the AIM */
int
aimOutputs(/*@unused@*/ void *instStore, /*@unused@*/ void *aimInfo, int index,
           char **aoname, capsValue *form)
{
  int status = CAPS_SUCCESS;
#ifdef DEBUG
  printf(" theAIM/aimOutputs instance = %d  index  = %d!\n",
         aim_getInstance(aimInfo), index);
#endif
  
  if (index == OutputVariable) {
    *aoname = AIM_NAME(OutputVariable);
    form->type = Double;
  } else {
    printf(" theAIM/aimOutputs: Unknown index = %d!\n", index);
    return CAPS_BADINDEX;
  }

  AIM_NOTNULL(*aoname, aimInfo, status);

cleanup:
  return status;
}


/* aimCalcOutput: Calculate/Retrieve Output Information */
int
aimCalcOutput(/*@unused@*/ void *instStore, void *aimInfo, int index,
              capsValue *val)
{
  int        myNum, status = CAPS_SUCCESS;
  capsValue  *myVal;
#ifdef DEBUG
  const char *name;

  status = aim_getName(aimInfo, index, ANALYSISOUT, &name);
  printf(" theAIM/aimCalcOutput instance = %d  index = %d %s %d!\n",
         aim_getInstance(aimInfo), index, name, status);
#endif

  if (index == OutputVariable) {
    /* default return */
    val->vals.real = -1;

    /* get the input num */
    status = aim_getValue(aimInfo, InputVariable, ANALYSISIN, &myVal);
    AIM_STATUS(aimInfo, status);
    if ((myVal->type != Integer) || (myVal->length != 1)) {
      printf("ERROR:: aim_getValue -> type=%d, len = %d\n",
             myVal->type, myVal->length);
      status = CAPS_BADTYPE;
      goto cleanup;
    }

    myNum = myVal->vals.integer;
    val->vals.real = myNum*myNum;
  } else {
    status = CAPS_BADVALUE;
  }

cleanup:
  return status;
}


/* aimCleanup: Free up the AIMs storage */
void aimCleanup(void *instStore)
{
  aimStorage *aimStore = NULL;
#ifdef DEBUG
  printf(" theAIM/aimCleanup!\n");
#endif

  /* clean up any allocated data 
   * tessellation objects are deleted by CAPS 
   */

  aimStore = (aimStorage *) instStore;
  if (aimStore == NULL) return;
  aimStore->nBody = 0;
  AIM_FREE(aimStore->tess);
  AIM_FREE(aimStore);
}
