/*
 *******************************************************************
 *                                                                 *
 * tankCalc -- compute tank mass given densities and tessellation  *
 *                                                                 *
 *              Written by John Dannenhoffer @ Syracuse University *
 *                                                                 *
 *******************************************************************
 */

/*
 * Copyright (C) 2013/2016  John F. Dannenhoffer, III (Syracuse University)
 *
 * This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later
 *     version.
 *
 * This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free
 *    Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *    Boston, MA  02110-1301  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

/******************************************************************/
/*                                                                */
/*   main - main program                                          */
/*                                                                */
/******************************************************************/

int
main(int       argc,               /* (in)  number of arguments */
     char      *argv[])            /* (in)  array  of arguments */
{
    int       haveDots, nread;
    double    wallDens,     fuelDens,     area,     volume,     mass;
    double    wallDens_dot, fuelDens_dot, area_dot, volume_dot, mass_dot;
    double    x0,     y0,     z0,     x1,     y1,     z1,     x2,     y2,     z2;
    double    x0_dot, y0_dot, z0_dot, x1_dot, y1_dot, z1_dot, x2_dot, y2_dot, z2_dot;
    double    areax,     areay,     areaz,     xcen,     ycen,     zcen, aa;
    double    areax_dot, areay_dot, areaz_dot, xcen_dot, ycen_dot, zcen_dot;
    char      filename_in[256], filename_out[256];
    FILE      *fp_in, *fp_out;

    /* ---------------------------------------------------------- */

    /* initialize all the dots (in case we do not have any) */
    wallDens_dot = 0;
    fuelDens_dot = 0;
    
    x0_dot = 0;
    y0_dot = 0;
    z0_dot = 0;
    x1_dot = 0;
    y1_dot = 0;
    z1_dot = 0;
    x2_dot = 0;
    y2_dot = 0;
    z2_dot = 0;
    
    /* get the names of the input and output files */
    if (argc <= 1) {
        strncpy(filename_in, "tankCalc.in", 255);
    } else {
        strncpy(filename_in, argv[1],       255);
    }

    if (argc <= 2) {
        strncpy(filename_out, "tankCalc.out", 255);
    } else {
        strncpy(filename_out, argv[2],        255);
    }

    /* open the input file */
    fp_in = fopen(filename_in, "r");
    if (fp_in == NULL) {
        printf("error: \"%s\" could not be opened for reading\n", filename_in);
        exit(EXIT_FAILURE);
    }

    /* determine if we have dots (sensitivities) */
    fscanf(fp_in, "%d", &haveDots);

    /* read the densities */
    if (haveDots == 0) {
        fscanf(fp_in, "%lf %lf", &wallDens, &fuelDens);
    } else {
        fscanf(fp_in, "%lf %lf %lf %lf", &wallDens, &wallDens_dot,
               &fuelDens, &fuelDens_dot);
    }

    /* initialize the area and volume */
    area       = 0;
    area_dot   = 0;
    volume     = 0;
    volume_dot = 0;

    /* read the values until we get to  the end of the file */
    nread = 0;
    while (1) {
        if (haveDots == 0 || haveDots == 1) {
            fscanf(fp_in, "%lf %lf %lf %lf %lf %lf %lf %lf %lf",
                   &x0, &y0, &z0, &x1, &y1, &z1, &x2, &y2, &z2);
        } else {
            fscanf(fp_in, "%lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf",
                   &x0, &y0, &z0, &x0_dot, &y0_dot, &z0_dot,
                   &x1, &y1, &z1, &x1_dot, &y1_dot, &z1_dot,
                   &x2, &y2, &z2, &x2_dot, &y2_dot, &z2_dot);
        }
        
        if (feof(fp_in) == 0) {
            nread++;
            
            areax = (y1 - y0) * (z2 - z0) - (z1 - z0) * (y2 - y0);
            areay = (z1 - z0) * (x2 - x0) - (x1 - x0) * (z2 - z0);
            areaz = (x1 - x0) * (y2 - y0) - (y1 - y0) * (x2 - x0);

            areax_dot = ((y1_dot - y0_dot) * (z2 - z0) + (z2_dot - z0_dot) * (y1 - y0))
                      - ((z1_dot - z0_dot) * (y2 - y0) + (y2_dot - y0_dot) * (z1 - z0));
            areay_dot = ((z1_dot - z0_dot) * (x2 - x0) + (x2_dot - x0_dot) * (z1 - z0))
                      - ((x1_dot - x0_dot) * (z2 - z0) + (z2_dot - z0_dot) * (x1 - x0));
            areaz_dot = ((x1_dot - x0_dot) * (y2 - y0) + (y2_dot - y0_dot) * (x1 - x0))
                      - ((y1_dot - y0_dot) * (x2 - x0) + (x2_dot - x0_dot) * (y1 - y0));

            aa        = sqrt(areax * areax + areay * areay + areaz * areaz);
            area     += aa;
            area_dot += (areax_dot * areax + areay_dot * areay + areaz_dot * areaz) / aa;

            xcen = (x0 + x1 + x2);
            ycen = (y0 + y1 + y2);
            zcen = (z0 + z1 + z2);

            xcen_dot = (x0_dot + x1_dot + x2_dot);
            ycen_dot = (y0_dot + y1_dot + y2_dot);
            zcen_dot = (z0_dot + z1_dot + z2_dot);

            volume     += xcen * areax + ycen * areay + zcen * areaz;
            volume_dot += xcen_dot * areax + areax_dot * xcen
                        + ycen_dot * areay + areay_dot * ycen
                        + zcen_dot * areaz + areaz_dot * zcen;
        } else {
            break;
        }
    }

    fclose(fp_in);
    printf("   %d triangles were read\n", nread);

    /* compute the output and write the output file */
    area       /= 2;
    area_dot   /= 2;
    volume     /= 18;
    volume_dot /= 18;
    
    mass     = wallDens     * area   + fuelDens   * volume;
    mass_dot = wallDens_dot * area   + area_dot   * wallDens
             + fuelDens_dot * volume + volume_dot * fuelDens;

    fp_out = fopen(filename_out, "w");
    if (fp_out == NULL) {
        printf("error: \"%s\" could not be opened for writing\n", filename_out);
        exit(EXIT_FAILURE);
    }

    if (haveDots == 0) {
        fprintf(fp_out, "%12.6lf %12.6lf %12.6lf\n",
                area, volume, mass);
    } else {
        fprintf(fp_out, "%12.6lf %12.6lf %12.6lf %12.6lf %12.6lf %12.6lf\n",
                area, area_dot, volume, volume_dot, mass, mass_dot);
    }
    
    fclose(fp_out);

    exit(EXIT_SUCCESS);
}
 
