/*
 *      CAPS: Computational Aircraft Prototype Syntheses
 *
 *             Session13 AIM Example Mesh Helper
 *
 *      Copyright 2014-2025, Massachusetts Institute of Technology
 *      Licensed under The GNU Lesser General Public License, version 2.1
 *      See http://www.opensource.org/licenses/lgpl-2.1.php
 *
 */

#include <string.h>
#include <math.h>
#include "aimUtil.h"
#include "aimMesh.h"


int fillMyMesh(void *aimInfo, int nTess, ego *tess, aimMesh *mesh)
{
  int          i, j, n, iTess, ibnd, stat, pty, pin, status = CAPS_SUCCESS;
  int          nTrias, nPts, nGroup, nFace, nVerts, nTri, nPoints, base, it[3];
  const double *points, *uv;
  const int    *ptype, *pindex, *tris, *tric;
  aimMeshRef   *meshRef;
  aimMeshData  *meshData;
  ego          body;
  
  meshRef  = mesh->meshRef;
  meshData = mesh->meshData;
  
  /* for simplicity Groups are Faces */
  for (nTrias = nVerts = nGroup = iTess = 0; iTess < nTess; iTess++) {
    status = EG_statusTessBody(tess[iTess], &body, &stat, &nPoints);
    AIM_STATUS(aimInfo, status);
    status = EG_getBodyTopos(body, NULL, FACE, &nFace, NULL);
    AIM_STATUS(aimInfo, status);
    nTri = 0;
    for (i = 1; i <= nFace; i++) {
      status = EG_getTessFace(tess[iTess], i, &nPts, &points, &uv, &ptype,
                              &pindex, &n, &tris, &tric);
      AIM_STATUS(aimInfo, status);
      nTri += n;
    }
    nGroup += nFace;
    nVerts += nPoints;
    nTrias += nTri;
  }
/*
  printf(" fillMyMesh: nTess = %d  nGroup = %d  nVerts = %d  nTrias = %d\n",
         nTess, nGroup, nVerts, nTrias);
 */
  
  /* don't need to fill meshRef->maps -- don't have a Volume mesh */
  
  /* fill Boundary Groups -- leave names as NULL */
  meshRef->nbnd = nGroup;
  AIM_ALLOC(meshRef->bnds, meshRef->nbnd, aimMeshBnd, aimInfo, status);
  for (ibnd = 0; ibnd < meshRef->nbnd; ibnd++) {
    meshRef->bnds[ibnd].groupName = NULL;
    meshRef->bnds[ibnd].ID = ibnd;
  }
  
  /* deal with the Mesh Data */
  meshData->nVertex     = nVerts;
  meshData->nTotalElems = nTrias;
  AIM_ALLOC(meshData->verts, meshData->nVertex, aimMeshCoords, aimInfo, status);
  for (n = iTess = 0; iTess < nTess; iTess++) {
    status = EG_statusTessBody(tess[iTess], &body, &stat, &nPoints);
    AIM_STATUS(aimInfo, status);
    for (i = 1; i <= nPoints; i++, n++) {
      status = EG_getGlobal(tess[iTess], i, &pty, &pin, meshData->verts[n]);
      AIM_STATUS(aimInfo, status);
    }
  }
  
  /* set Element Groups */
  meshData->nElemGroup = nGroup;
  AIM_ALLOC(meshData->elemGroups, meshData->nElemGroup, aimMeshElemGroup,
            aimInfo, status);
  for (i = 0; i < meshData->nElemGroup; i++) {
    meshData->elemGroups[i].groupName = NULL;
    meshData->elemGroups[i].ID = i;
    meshData->elemGroups[i].elementTopo = aimTri;
    meshData->elemGroups[i].order = 1;
    meshData->elemGroups[i].nPoint = 3;
    meshData->elemGroups[i].nElems = 0;
    meshData->elemGroups[i].elements = NULL;
  }
  for (base = n = iTess = 0; iTess < nTess; iTess++) {
    status = EG_statusTessBody(tess[iTess], &body, &stat, &nPoints);
    AIM_STATUS(aimInfo, status);
    status = EG_getBodyTopos(body, NULL, FACE, &nFace, NULL);
    AIM_STATUS(aimInfo, status);
    nTri = 0;
    for (i = 1; i <= nFace; i++, n++) {
      status = EG_getTessFace(tess[iTess], i, &nPts, &points, &uv, &ptype,
                              &pindex, &nTri, &tris, &tric);
      AIM_STATUS(aimInfo, status);
      meshData->elemGroups[n].nElems = nTri;
      AIM_ALLOC(meshData->elemGroups[n].elements,
                3*meshData->elemGroups[n].nElems, int, aimInfo, status);
      for (j = 0; j < nTri; j++) {
        status = EG_localToGlobal(tess[iTess], i, tris[3*j  ], &it[0]);
        AIM_STATUS(aimInfo, status);
        status = EG_localToGlobal(tess[iTess], i, tris[3*j+1], &it[1]);
        AIM_STATUS(aimInfo, status);
        status = EG_localToGlobal(tess[iTess], i, tris[3*j+2], &it[2]);
        AIM_STATUS(aimInfo, status);
        meshData->elemGroups[n].elements[3*j  ] = base + it[0];
        meshData->elemGroups[n].elements[3*j+1] = base + it[1];
        meshData->elemGroups[n].elements[3*j+2] = base + it[2];
      }
    }
    base += nPoints;
  }
  
  /* don't need to fill meshData->elemMaps */

cleanup:
  if (status != CAPS_SUCCESS) printf(" fillMyMesh: Status = %d!\n", status);
  return status;
}
