/*
 *      CAPS: Computational Aircraft Prototype Syntheses
 *
 *             Session13: Binary STL Surface Mesh Writer Example
 *
 *      Copyright 2014-2025, Massachusetts Institute of Technology
 *      Licensed under The GNU Lesser General Public License, version 2.1
 *      See http://www.opensource.org/licenses/lgpl-2.1.php
 *
 */

#include <string.h>
#include <math.h>
#include "aimUtil.h"
#include "aimMesh.h"

#include "myMeshWriter.h"

#define UINT16 unsigned short int
#define UINT32 unsigned int
#define REAL32 float


static void triNormal(double *p1, double *p2, double *p3, double norm[])
{
    double a[3]={0.0,0.0,0.0};
    double b[3]={0.0,0.0,0.0};
    double mag = 0.0;

    // a x b

    // Create two vectors from points
    a[0]= p2[0]-p1[0];
    a[1]= p2[1]-p1[1];
    a[2]= p2[2]-p1[2];

    b[0]= p3[0]-p1[0];
    b[1]= p3[1]-p1[1];
    b[2]= p3[2]-p1[2];

    // Take the cross product
    norm[0] = a[1]*b[2]-a[2]*b[1];

    norm[1] = a[2]*b[0]-a[0]*b[2];

    norm[2] = a[0]*b[1]-a[1]*b[0];

    // Normalize vector
    mag = sqrt(norm[0]*norm[0] + norm[1]*norm[1] + norm[2]*norm[2]);

    norm[0] = norm[0]/mag;
    norm[1] = norm[1]/mag;
    norm[2] = norm[2]/mag;
}


const char *meshExtension()
{
/*@-observertrans@*/
  return MESHEXTENSION;
/*@+observertrans@*/
}


int meshWrite(void *aimInfo, aimMesh *mesh)
{
  int         status, igroup, ielem, nPoint, itri;
  char        filename[PATH_MAX];
  double      norm[3] = {0,0,0};
  double      *p1, *p2, *p3;
  REAL32      coors[12];
  UINT32      totTri;
  UINT16      icolor;
  FILE        *fp = NULL;
  aimMeshData *meshData = NULL;
  char        header[80] = "CAPS AIM Development Session13 binary STL Writer";

  if (mesh == NULL)           return CAPS_NULLVALUE;
  if (mesh->meshRef  == NULL) return CAPS_NULLVALUE;
  if (mesh->meshData == NULL) return CAPS_NULLVALUE;

  if (mesh->meshData->dim != 3) {
    AIM_ERROR(aimInfo, "meshData dim = %d must be 3!!!", mesh->meshData->dim);
    return CAPS_BADVALUE;
  }

  snprintf(filename, PATH_MAX, "%s%s", mesh->meshRef->fileName, MESHEXTENSION);
  fp = fopen(filename, "wb");
  if (fp == NULL) {
    AIM_ERROR(aimInfo, "Cannot open file: %s\n", filename);
    return CAPS_IOERR;
  }
  
  meshData = mesh->meshData;

  totTri = 0;
  for (igroup = 0; igroup < meshData->nElemGroup; igroup++) {
    if (meshData->elemGroups[igroup].order != 1) {
      AIM_ERROR(aimInfo, "STL only supports linear meshes! group %d order = %d",
                igroup, meshData->elemGroups[igroup].order);
      status = CAPS_IOERR;
      goto cleanup;
    }
    if (meshData->elemGroups[igroup].elementTopo == aimTri) {
      totTri +=   meshData->elemGroups[igroup].nElems;
    } else {
      totTri += 2*meshData->elemGroups[igroup].nElems;
    }
  }

  printf("\nWriting Binary STL file ....\n");

  (void) fwrite(header,  sizeof(char),  80, fp);
  (void) fwrite(&totTri, sizeof(UINT32), 1, fp);

  /* Write Normals and Points */
  icolor = 0;
  for (igroup = 0; igroup < meshData->nElemGroup; igroup++, icolor++) {

    if (meshData->elemGroups[igroup].elementTopo == aimTri) {

      nPoint = meshData->elemGroups[igroup].nPoint;

      for (ielem = 0; ielem < meshData->elemGroups[igroup].nElems; ielem++) {

        p1 = &meshData->verts[meshData->elemGroups[igroup].elements[nPoint*ielem+0]-1][0];
        p2 = &meshData->verts[meshData->elemGroups[igroup].elements[nPoint*ielem+1]-1][0];
        p3 = &meshData->verts[meshData->elemGroups[igroup].elements[nPoint*ielem+2]-1][0];
        triNormal(p1, p2, p3, norm);
        coors[ 0] = norm[0];
        coors[ 1] = norm[1];
        coors[ 2] = norm[2];
        coors[ 3] = p1[0];
        coors[ 4] = p1[1];
        coors[ 5] = p1[2];
        coors[ 6] = p2[0];
        coors[ 7] = p2[1];
        coors[ 8] = p2[2];
        coors[ 9] = p3[0];
        coors[10] = p3[1];
        coors[11] = p3[2];
        (void) fwrite(coors,   sizeof(REAL32), 12, fp);
        (void) fwrite(&icolor, sizeof(UINT16),  1, fp);

      }

    } else if (meshData->elemGroups[igroup].elementTopo == aimQuad) {

      nPoint = meshData->elemGroups[igroup].nPoint;

      for (ielem = 0; ielem < meshData->elemGroups[igroup].nElems; ielem++) {

        for (itri = 0; itri < 2; itri++) {
          if (itri == 0) {
            p1 = &meshData->verts[meshData->elemGroups[igroup].elements[nPoint*ielem+0]-1][0];
            p2 = &meshData->verts[meshData->elemGroups[igroup].elements[nPoint*ielem+1]-1][0];
            p3 = &meshData->verts[meshData->elemGroups[igroup].elements[nPoint*ielem+2]-1][0];
          } else {
            p1 = &meshData->verts[meshData->elemGroups[igroup].elements[nPoint*ielem+0]-1][0];
            p2 = &meshData->verts[meshData->elemGroups[igroup].elements[nPoint*ielem+2]-1][0];
            p3 = &meshData->verts[meshData->elemGroups[igroup].elements[nPoint*ielem+3]-1][0];
          }
          triNormal(p1, p2, p3, norm);
          coors[ 0] = norm[0];
          coors[ 1] = norm[1];
          coors[ 2] = norm[2];
          coors[ 3] = p1[0];
          coors[ 4] = p1[1];
          coors[ 5] = p1[2];
          coors[ 6] = p2[0];
          coors[ 7] = p2[1];
          coors[ 8] = p2[2];
          coors[ 9] = p3[0];
          coors[10] = p3[1];
          coors[11] = p3[2];
          (void) fwrite(coors,   sizeof(REAL32), 12, fp);
          (void) fwrite(&icolor, sizeof(UINT16),  1, fp);
        }
      }
    }
  }

  printf("Finished writing Binary STL file\n\n");

  status = CAPS_SUCCESS;

cleanup:
/*@-dependenttrans@*/
  if (fp != NULL) fclose(fp);
/*@+dependenttrans@*/
  return status;
}
