/*
 *      EGADS: Engineering Geometry Aircraft Design System
 *
 *             ESP Plugin Development Training Exercise #1
 *
 *      Copyright 2011-2025, Massachusetts Institute of Technology
 *      Licensed under The GNU Lesser General Public License, version 2.1
 *      See http://www.opensource.org/licenses/lgpl-2.1.php
 *
 */

#include "egads.h"
#include <math.h>
#include <string.h>


static void printAttr(const char *name, int type, int length,
                      const int *ints, const double *reals, const char *str)
{
  int i;
  
  printf(" Name = %s", name);
  if (type == ATTRSTRING) {
    printf("  String = %s", str);
  } else if (type == ATTRINT) {
    printf("  Integer(s) = %d", ints[0]);
    for (i = 1; i < length; i++) printf(" %d", ints[i]);
  } else if (type == ATTRREAL) {
    printf("  Double(s) = %lf", reals[0]);
    for (i = 1; i < length; i++) printf(" %lf", reals[i]);
  } else if (type == ATTRCSYS) {
    printf("  is a Coordinate System");
  } else {
    printf("  is a Pointer");
  }
  printf("\n");
}


static double tessVolume(const ego body, const ego tess)
{
  int          i, itri, status, nFace, nPts, nTri, ip0, ip1, ip2;
  double       xa, ya, za, xb, yb, zb, xbar, ybar, zbar, areax, areay, areaz;
  double       volume = 0.0;
  const int    *ptype, *pindex, *tris, *tric;
  const double *xyz, *uv;
  
  status = EG_getBodyTopos(body, NULL, FACE, &nFace, NULL);
  if (status != EGADS_SUCCESS) {
    printf(" EG_getBodyTopos = %d (tessVolume)\n", status);
    return 0.0;
  }
  
  for (i = 1; i <= nFace; i++) {
    status = EG_getTessFace(tess, i, &nPts, &xyz, &uv, &ptype, &pindex,
                            &nTri, &tris, &tric);
    if (status != EGADS_SUCCESS) {
      printf(" EG_getTessFace = %d, %d (tessVolume)\n", i, status);
      return 0.0;
    }
    for (itri = 0; itri < nTri; itri++) {
      ip0 = 3 * (tris[3*itri  ] - 1);
      ip1 = 3 * (tris[3*itri+1] - 1);
      ip2 = 3 * (tris[3*itri+2] - 1);
      
      xa = xyz[ip1  ] - xyz[ip0  ];
      ya = xyz[ip1+1] - xyz[ip0+1];
      za = xyz[ip1+2] - xyz[ip0+2];
      
      xb = xyz[ip2  ] - xyz[ip0  ];
      yb = xyz[ip2+1] - xyz[ip0+1];
      zb = xyz[ip2+2] - xyz[ip0+2];
      
      xbar = xyz[ip0  ] + xyz[ip1  ] + xyz[ip2  ];
      ybar = xyz[ip0+1] + xyz[ip1+1] + xyz[ip2+1];
      zbar = xyz[ip0+2] + xyz[ip1+2] + xyz[ip2+2];
      
      areax = ya * zb - za * yb;
      areay = za * xb - xa * zb;
      areaz = xa * yb - ya * xb;

      volume += (xbar*areax + ybar*areay + zbar*areaz)/18;
    }
  }
  
  return volume;
}


int main(int argc, char *argv[])
{
  int          i, j, status, oclass, mtype, nbody, nvert, atype, alen;
  int          inode, iedge, iface, iattr;
  int          numNodes=0, numEdges=0, numFaces=0, numAttrs=0;
  int          *triang;
  float        arg;
  double       size, vol, params[3], box[6], props[14];
  ego          context, model, geom, *bodies, tess;
  ego          *nodes=NULL, *edges=NULL, *faces=NULL;
  const char   *OCCrev, *str;
  const char   *attrName;
  const int    *ints;
  const double *reals;

  if ((argc != 1) && (argc != 4)) {
    printf(" Usage: myExample [angle relSide relSag]\n\n");
    return 1;
  }

  /* look at EGADS revision */
  EG_revision(&i, &j, &OCCrev);
  printf("\n Using EGADS %2d.%02d %s\n\n", i, j, OCCrev);

  /* initialize */
  status = EG_open(&context);
  if (status != EGADS_SUCCESS) {
    printf(" EG_open = %d!\n\n", status);
    return 1;
  }
  /* load our EGADS file */
  status = EG_loadModel(context, 0, "case.egads", &model);
  if (status != EGADS_SUCCESS) {
    printf(" EG_loadModel = %d\n\n", status);
    EG_close(context);
    return 1;
  }
  /* get the bounding box for the model */
  status = EG_getBoundingBox(model, box);
  if (status != EGADS_SUCCESS) {
    printf(" EG_getBoundingBox = %d\n\n", status);
    EG_deleteObject(model);
    EG_close(context);
    return 1;
  }
  size = sqrt((box[0]-box[3])*(box[0]-box[3]) + (box[1]-box[4])*(box[1]-box[4]) +
              (box[2]-box[5])*(box[2]-box[5]));

  /* get all bodies */
  status = EG_getTopology(model, &geom, &oclass, &mtype, NULL, &nbody,
                          &bodies, &triang);
  if (status != EGADS_SUCCESS) {
    printf(" EG_getTopology = %d\n\n", status);
    EG_deleteObject(model);
    EG_close(context);
    return 1;
  }

  params[0] =  0.025*size;
  params[1] =  0.001*size;
  params[2] = 15.0;
  if (argc == 4) {
    sscanf(argv[1], "%f", &arg);
    params[2] = arg;
    sscanf(argv[2], "%f", &arg);
    params[0] = arg;
    sscanf(argv[3], "%f", &arg);
    params[1] = arg;
    printf(" Using angle = %lf,  relSide = %lf,  relSag = %lf\n",
           params[2], params[0], params[1]);
    params[0] *= size;
    params[1] *= size;
  }

  printf(" Number of Bodies = %d\n\n", nbody);

  for (i = 0; i < nbody; i++) {
    status = EG_attributeRet(bodies[i], "capsLength", &atype, &alen,
                             &ints, &reals, &str);
    if (status == EGADS_SUCCESS) {
      printf(" Body %d:", i);
      printAttr("capsLength", atype, alen, ints, reals, str);
    }
    status = EG_makeTessBody(bodies[i], params, &tess);
    if (status != EGADS_SUCCESS) {
      printf(" EG_makeTessBody %d = %d\n", i, status);
      continue;
    }
    status = EG_statusTessBody(tess, &geom, &j, &nvert);
    printf("    Tessellation  %d  npts = %d", status, nvert);
    if (bodies[i]->mtype == SOLIDBODY) {
      printf(" (Solid)\n");
      status = EG_getMassProperties(bodies[i], props);
      if (status != EGADS_SUCCESS) props[0] = 0.0;
      vol = tessVolume(bodies[i], tess);
      printf("    Volumes = %le  %le\n", vol, props[0]);
    } else {
      printf("\n");
    }

    /**************************************************************************/
    
    // Faces
    printf("\n------------ FACE Attributes ----------\n");
    status = EG_getBodyTopos(bodies[i], NULL, FACE, &numFaces, &faces);
    if (status == EGADS_SUCCESS) {
      for (iface = 0; iface < numFaces; iface++) {
        status = EG_attributeNum(faces[iface], &numAttrs);
        if (status == EGADS_SUCCESS) {
          printf("   Attribute on Face %d: \n", iface+1);
          
          for (iattr = 0; iattr < numAttrs; iattr++) {
            status = EG_attributeGet(faces[iface], iattr+1, &attrName, &atype,
                                     &alen, &ints, &reals, &str);
            
            if (status == EGADS_SUCCESS)
              printAttr(attrName, atype, alen, ints, reals, str);
          }
          printf("----------------------\n");
        }
      }
      EG_free(faces); faces = NULL;
    }

    // Edges
    printf("\n------------ EDGE Attributes ----------\n");
    status = EG_getBodyTopos(bodies[i], NULL, EDGE, &numEdges, &edges);
    if (status == EGADS_SUCCESS) {
      for (iedge = 0; iedge < numEdges; iedge++) {
        status = EG_attributeNum(edges[iedge], &numAttrs);
        if (status == EGADS_SUCCESS) {
          printf("   Attribute on Edge %d: \n", iedge+1);
          
          for (iattr = 0; iattr < numAttrs; iattr++) {
            status = EG_attributeGet(edges[iedge], iattr+1, &attrName, &atype,
                                     &alen, &ints, &reals, &str);
            if (status == EGADS_SUCCESS)
              printAttr(attrName, atype, alen, ints, reals, str);
          }
          printf("----------------------\n");
        }
      }
      EG_free(edges); edges = NULL;
    }

    // Nodes
    printf("\n------------ NODE Attributes ----------\n");
    status = EG_getBodyTopos(bodies[i], NULL, NODE, &numNodes, &nodes);
    if (status == EGADS_SUCCESS) {
      for (inode = 0; inode < numNodes; inode++) {
        status = EG_attributeNum(nodes[inode], &numAttrs);
        if (status == EGADS_SUCCESS) {
          printf("   Attribute on Node %d: \n", inode+1);
          
          for (iattr = 0; iattr < numAttrs; iattr++) {
            status = EG_attributeGet(nodes[inode], iattr+1, &attrName, &atype,
                                     &alen, &ints, &reals, &str);
            if (status == EGADS_SUCCESS)
              printAttr(attrName, atype, alen, ints, reals, str);
          }
          printf("----------------------\n");
        }
      }
      EG_free(nodes); nodes = NULL;
    }
    /**************************************************************************/
    
    status = EG_deleteObject(tess);
    if (status != EGADS_SUCCESS)
      printf(" EG_deleteObject tess %d = %d\n", i, status);
  }
  printf("\n");

  status = EG_deleteObject(model);
  if (status != EGADS_SUCCESS) printf(" EG_deleteObject Model = %d\n", status);
  EG_close(context);

  return 0;
}
