/*
 ************************************************************************
 *                                                                      *
 * udpExercise2 -- exercise UDP                                         *
 *                                                                      *
 *             this makes a pyramid (centered at origin)                *
 *             and returns its area and volume (for training purposes)  *
 *             sensitivities are computed analytically                  *
 *                                                                      *
 *             Written by John Dannenhoffer @ Geocentric Technologies   *
 *                                                                      *
 ************************************************************************
 */

/*
 * Copyright (C) 2025  John F. Dannenhoffer, III (Geocentric Technologies))
 *
 * This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *     MA  02110-1301  USA
 */

/* uncomment the following to get DEBUG printouts */
//#define DEBUG 1

/* the number of "input" Bodys

   this only needs to be specified if this is a UDF (user-defined
   function) that consumes Bodys from OpenCSM's stack.  (the default
   value is 0).

   if NUMUDPINPUTBODYS>0  then exactly NUMUDPINPUTBODYS are in emodel
   if NUMUDPINPUTBODYS<0  then up to  -NUMUDPINPUTBODYS are in emodel
*/
#define NUMUDPINPUTBODYS 0

/* the number of arguments (specified below) */
#define NUMUDPARGS 5

/* set up the necessary structures (uses NUMUDPARGS) */
#include "udpUtilities.h"

/* shorthand macros for accessing argument values and velocities */
#define LENGTH(    IUDP)  ((double *) (udps[IUDP].arg[0].val))[0]
#define LENGTH_DOT(IUDP)  ((double *) (udps[IUDP].arg[0].dot))[0]
#define LENGTH_SIZ(IUDP)               udps[IUDP].arg[0].size

#define WIDTH(     IUDP)  ((double *) (udps[IUDP].arg[1].val))[0]
#define WIDTH_DOT( IUDP)  ((double *) (udps[IUDP].arg[1].dot))[0]
#define WIDTH_SIZ( IUDP)               udps[IUDP].arg[1].size

#define HEIGHT(    IUDP)  ((double *) (udps[IUDP].arg[2].val))[0]
#define HEIGHT_DOT(IUDP)  ((double *) (udps[IUDP].arg[2].dot))[0]
#define HEIGHT_SIZ(IUDP)               udps[IUDP].arg[2].size

#define AREA(      IUDP)  ((double *) (udps[IUDP].arg[3].val))[0]
#define AREA_DOT(  IUDP)  ((double *) (udps[IUDP].arg[3].dot))[0]

#define VOLUME(    IUDP)  ((double *) (udps[IUDP].arg[4].val))[0]
#define VOLUME_DOT(IUDP)  ((double *) (udps[IUDP].arg[4].dot))[0]

/* data about possible arguments
      argNames: argument name (must be all lower case)
      argTypes: argument type: +ATTRINT      integer input
                               -ATTRINT      integer output
                               +ATTRREAL     double input  (no sensitivities)
                               -ATTRREAL     double output (no sensitivities)
                               +ATTRREALSEN  double input  (with sensitivities)
                               -ATTRREALSEN  double output (with sensitivities)
                               +ATTRSTRING   string input
                               -ATTRSTRING   *** cannot be used ***
                               +ATTRFILE     input file
                               -ATTRFILE     *** cannot be used ***
                               +ATTRREBUILD  forces rebuild if any variable in
                                             semi-colon-separated list has been changed
                               -ATTRREBUILD  *** cannot be used ***
                               +ATTRRECYCLE  forces rebuild (always) by blocking recycling
                               -ATTRRECYCLE  *** cannot be used ***
      argIdefs: default value for ATTRINT
      argDdefs: default value for ATTRREAL or ATTRREALSEN */
static char  *argNames[NUMUDPARGS] = {"length",    "width",      "height",
                                      "area",       "volume",    };
static int    argTypes[NUMUDPARGS] = {ATTRREALSEN,  ATTRREALSEN, ATTRREALSEN,
                                     -ATTRREALSEN, -ATTRREALSEN, };
static int    argIdefs[NUMUDPARGS] = {0,           0,            0,
                                      0,           0,            };
static double argDdefs[NUMUDPARGS] = {1.,          1.,           1.,
                                      0.,          0.,           };

/* get utility routines: udpErrorStr, udpInitialize, udpReset, udpSet,
                         udpGet, udpVel, udpClean, udpMesh */
#include "udpUtilities.c"

static int makeEdge(ego enode0, ego enode1, ego *eedge);
static int makeFace(ego eedge0, ego eedge1, ego eedge2, ego eedge3, ego *eface);


/*
 ************************************************************************
 *                                                                      *
 *   udpExecute - execute the primitive                                 *
 *                                                                      *
 ************************************************************************
 */

int
udpExecute(ego  context,                /* (in)  EGADS context */
           ego  *ebody,                 /* (out) Body (or model) pointer */
           int  *nMesh,                 /* (out) number of associated meshes */
           char *string[])              /* (out) error message */
{
    int     status = EGADS_SUCCESS;

    double  data[18], temp1, temp2;
    char    *message=NULL;
    ego     enodes[5], eedges[8], efaces[5], eshell;
    udp_T   *udps = *Udps;

    ROUTINE(udpExecute);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    /* debug printing of the input arguments */
    printf("udpExecute(context=%llx)\n", (long long)context);
    printf("length(0)  = %f\n", LENGTH(0));
    printf("width( 0)  = %f\n", WIDTH( 0));
    printf("height(0)  = %f\n", HEIGHT(0));
#endif

    /* default return values */
    *ebody  = NULL;
    *nMesh  = 0;
    *string = NULL;

    /* the place where messages to the user are placed */
    MALLOC(message, char, 100);
    message[0] = '\0';

    /* check arguments */
    if (LENGTH_SIZ(0) > 1) {
        snprintf(message, 100, "\"length\" should be a scalar");
        status  = EGADS_RANGERR;
        goto cleanup;

    } else if (LENGTH(0) < 0) {
        snprintf(message, 100, "\"length\" (=%f) should be non-negative", LENGTH(0));
        status  = EGADS_RANGERR;
        goto cleanup;

    } else if (WIDTH_SIZ(0) > 1) {
        snprintf(message, 100, "\"width\" should be a scalar");
        status  = EGADS_RANGERR;
        goto cleanup;

    } else if (WIDTH(0) < 0) {
        snprintf(message, 100, "\"width\" (-%f) should be non-negative", WIDTH(0));
        status  = EGADS_RANGERR;
        goto cleanup;

    } else if (HEIGHT_SIZ(0) > 1) {
        snprintf(message, 100, "\"height\" should be a scalar");
        status  = EGADS_RANGERR;
        goto cleanup;

    } else if (HEIGHT(0) < 0) {
        snprintf(message, 100, "\"height\" (=%f) should be non-negative", HEIGHT(0));
        status  = EGADS_RANGERR;
        goto cleanup;

    }

    /* cache copy of arguments for future use */
    status = cacheUdp(NULL);
    CHECK_STATUS(cacheUdp);

#ifdef DEBUG
    /* debug printing of cached input arguments */
    printf("length[%d] = %f\n", numUdp, LENGTH(numUdp));
    printf("width[ %d] = %f\n", numUdp, WIDTH( numUdp));
    printf("height[%d] = %f\n", numUdp, HEIGHT(numUdp));
#endif

    /*
        y
        |
     3--<2---2
     |\     /|
     | 7   6 |
     |  v v  ^
     3   4   1 -->x
     v  ^ ^  |
     | 4   5 |
     |/     \|
     0---0>--1

    */

    /* create the Nodes */
    data[0] = -LENGTH(0) / 2;
    data[1] = -WIDTH( 0) / 2;
    data[2] = 0;

    status = EG_makeTopology(context, NULL, NODE, 0, data,
                             0, NULL, NULL, &(enodes[0]));
    CHECK_STATUS(EG_makeTopology);

    data[0] = +LENGTH(0) / 2;
    data[1] = -WIDTH( 0) / 2;
    data[2] = 0;

    status = EG_makeTopology(context, NULL, NODE, 0, data,
                             0, NULL, NULL, &(enodes[1]));
    CHECK_STATUS(EG_makeTopology);

    data[0] = +LENGTH(0) / 2;
    data[1] = +WIDTH( 0) / 2;
    data[2] = 0;

    status = EG_makeTopology(context, NULL, NODE, 0, data,
                             0, NULL, NULL, &(enodes[2]));
    CHECK_STATUS(EG_makeTopology);

    data[0] = -LENGTH(0) / 2;
    data[1] = +WIDTH( 0) / 2;
    data[2] = 0;

    status = EG_makeTopology(context, NULL, NODE, 0, data,
                             0, NULL, NULL, &(enodes[3]));
    CHECK_STATUS(EG_makeTopology);

    data[0] = 0;
    data[1] = 0;
    data[2] = HEIGHT(0);

    status = EG_makeTopology(context, NULL, NODE, 0, data,
                             0, NULL, NULL, &(enodes[4]));
    CHECK_STATUS(EG_makeTopology);

    /* make the Edges */
    status = makeEdge(enodes[0], enodes[1], &(eedges[0]));
    CHECK_STATUS(makeEdge);

    status = makeEdge(enodes[1], enodes[2], &(eedges[1]));
    CHECK_STATUS(makeEdge);

    status = makeEdge(enodes[2], enodes[3], &(eedges[2]));
    CHECK_STATUS(makeEdge);

    status = makeEdge(enodes[3], enodes[0], &(eedges[3]));
    CHECK_STATUS(makeEdge);

    status = makeEdge(enodes[0], enodes[4], &(eedges[4]));
    CHECK_STATUS(makeEdge);

    status = makeEdge(enodes[1], enodes[4], &(eedges[5]));
    CHECK_STATUS(makeEdge);

    status = makeEdge(enodes[2], enodes[4], &(eedges[6]));
    CHECK_STATUS(makeEdge);

    status = makeEdge(enodes[3], enodes[4], &(eedges[7]));
    CHECK_STATUS(makeEdge);

    /* make the Faces */
    status = makeFace(eedges[0], eedges[1], eedges[2], eedges[3], &(efaces[0]));
    CHECK_STATUS(makeFace);

    status = makeFace(eedges[0], eedges[5], NULL,      eedges[4], &(efaces[1]));
    CHECK_STATUS(makeFace);

    status = makeFace(eedges[1], eedges[6], NULL,      eedges[5], &(efaces[2]));
    CHECK_STATUS(makeFace);

    status = makeFace(eedges[2], eedges[7], NULL,      eedges[6], &(efaces[3]));
    CHECK_STATUS(makeFace);

    status = makeFace(eedges[3], eedges[4], NULL,      eedges[7], &(efaces[4]));
    CHECK_STATUS(makeFace);

    /* make the Shell and SolidBody */
    status = EG_makeTopology(context, NULL, SHELL, CLOSED, NULL,
                             5, efaces, NULL, &eshell);
    CHECK_STATUS(EG_makeTopology);

    status = EG_makeTopology(context, NULL, BODY, SOLIDBODY, NULL,
                             1, &eshell, NULL, ebody);
    CHECK_STATUS(EG_makeTopology);

    SPLINT_CHECK_FOR_NULL(*ebody);

    /* set the output value(s) */
    temp1     = sqrt(LENGTH(0)*LENGTH(0)/4 + HEIGHT(0)*HEIGHT(0));
    temp2     = sqrt(WIDTH( 0)*WIDTH( 0)/4 + HEIGHT(0)*HEIGHT(0));

    AREA(  numUdp) = LENGTH(0) * WIDTH( 0) + LENGTH(0) * temp2 + WIDTH( 0) * temp1;
    VOLUME(numUdp) = LENGTH(0) * WIDTH(0) * HEIGHT(0) / 3;

    /* remember this model (Body) */
    udps[numUdp].ebody = *ebody;

cleanup:
#ifdef DEBUG
    printf("udpExecute -> numUdp=%d, *ebody=%llx\n", numUdp, (long long)(*ebody));
#endif

    if (strlen(message) > 0) {
        *string = message;
        printf("%s\n", message);
    } else if (status != EGADS_SUCCESS) {
        FREE(message);
        *string = udpErrorStr(status);
    } else {
        FREE(message);
    }

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   udpSensitivity - return sensitivity derivatives for the "real" argument *
 *                                                                      *
 ************************************************************************
 */

int
udpSensitivity(ego    ebody,            /* (in)  Body pointer */
               int    npnt,             /* (in)  number of points */
               int    entType,          /* (in)  OCSM entity type */
               int    entIndex,         /* (in)  OCSM entity index (bias-1) */
               double uvs[],            /* (in)  parametric coordinates for evaluation */
               double vels[])           /* (out) velocities */
{
    int    status = EGADS_SUCCESS;

    int     iudp, judp, ipnt, inode, nnode, iedge, nedge, iface, nface;
    double  data[18], temp1, temp1_dot, temp2, temp2_dot;
    ego     *enodes=NULL, myNode, *eedges=NULL, myEdge, *efaces=NULL, myFace;

    ROUTINE(udpSensitivity);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    if (uvs != NULL) {
        printf("udpSensitivity(ebody=%llx, npnt=%d, entType=%d, entIndex=%d, uvs=%f %f)\n",
               (long long)ebody, npnt, entType, entIndex, uvs[0], uvs[1]);
    } else {
        printf("udpSensitivity(ebody=%llx, npnt=%d, entType=%d, entIndex=%d, uvs=NULL)\n",
               (long long)ebody, npnt, entType, entIndex);
    }
#endif

    /* check that ebody matches one of the ebodys */
    iudp = 0;
    for (judp = 1; judp <= numUdp; judp++) {
        if (ebody == udps[judp].ebody) {
            iudp = judp;
            break;
        }
    }
    if (iudp <= 0) {
        status = EGADS_NOTMODEL;
        goto cleanup;
    }

    /* initialize the returns to zero */
    for (ipnt = 0; ipnt < npnt; ipnt++) {
        vels[3*ipnt  ] = 0;
        vels[3*ipnt+1] = 0;
        vels[3*ipnt+2] = 0;
    }

    /* velocities of Nodes */
    if (entType == OCSM_NODE) {
        inode = entIndex;

        status = EG_getBodyTopos(ebody, NULL, NODE, &nnode, &enodes);
        CHECK_STATUS(EG_getBodyTopos);

        if (inode >= 1 && inode <= nnode) {
            myNode = enodes[inode-1];
        } else {
            status = OCSM_NODE_NOT_FOUND;
            goto cleanup;
        }

        status = EG_evaluate(myNode, 0, data);
        CHECK_STATUS(EG_evaluate);

        vels[0] = data[0] / LENGTH(iudp) * LENGTH_DOT(iudp);
        vels[1] = data[1] / WIDTH( iudp) * WIDTH_DOT( iudp);
        vels[2] = data[2] / HEIGHT(iudp) * HEIGHT_DOT(iudp);
        
    /* velocities of Edges */
    } else if (entType == OCSM_EDGE) {
        iedge = entIndex;

        status = EG_getBodyTopos(ebody, NULL, EDGE, &nedge, &eedges);
        CHECK_STATUS(EG_getBodyTopos);

        if (iedge >= 1 && iedge <= nedge) {
            myEdge = eedges[iedge-1];
        } else {
            status = OCSM_EDGE_NOT_FOUND;
            goto cleanup;
        }

        for (ipnt = 0; ipnt < npnt; ipnt++) {
            status = EG_evaluate(myEdge, &(uvs[ipnt]), data);
            CHECK_STATUS(EG_evaluate);

            vels[3*ipnt  ] = data[0] / LENGTH(iudp) * LENGTH_DOT(iudp);
            vels[3*ipnt+1] = data[1] / WIDTH( iudp) * WIDTH_DOT( iudp);
            vels[3*ipnt+2] = data[2] / HEIGHT(iudp) * HEIGHT_DOT(iudp);
        }

    /* velocities of Faces */
    } else {
        iface = entIndex;

        status = EG_getBodyTopos(ebody, NULL, FACE, &nface, &efaces);
        CHECK_STATUS(EG_getBodyTopos);

        if (iface >= 1 && iface <= nface) {
            myFace = efaces[iface-1];
        } else {
            status = OCSM_FACE_NOT_FOUND;
            goto cleanup;
        }

        for (ipnt = 0; ipnt < npnt; ipnt++) {
            status = EG_evaluate(myFace, &(uvs[2*ipnt]), data);
            CHECK_STATUS(EG_evaluate);

            vels[3*ipnt  ] = data[0] / LENGTH(iudp) * LENGTH_DOT(iudp);
            vels[3*ipnt+1] = data[1] / WIDTH( iudp) * WIDTH_DOT( iudp);
            vels[3*ipnt+2] = data[2] / HEIGHT(iudp) * HEIGHT_DOT(iudp);
        }
    }

    /* set velocities of the output value(s) */
    temp1     = sqrt(LENGTH(iudp)*LENGTH(iudp)/4 + HEIGHT(iudp)*HEIGHT(iudp));
    temp2     = sqrt(WIDTH( iudp)*WIDTH( iudp)/4 + HEIGHT(iudp)*HEIGHT(iudp));

    temp1_dot = (LENGTH_DOT(iudp) * LENGTH(iudp) / 4 + HEIGHT_DOT(iudp) * HEIGHT(iudp)) / temp1;
    temp2_dot = (WIDTH_DOT( iudp) * WIDTH( iudp) / 4 + HEIGHT_DOT(iudp) * HEIGHT(iudp)) / temp2;
    
    AREA_DOT(  iudp) = LENGTH_DOT(iudp) * WIDTH( iudp) + WIDTH_DOT( iudp) * LENGTH(iudp)
                     + WIDTH_DOT( iudp) * temp1 + WIDTH( iudp) * temp1_dot
                     + LENGTH_DOT(iudp) * temp2 + LENGTH(iudp) * temp2_dot;
    VOLUME_DOT(iudp) = (LENGTH_DOT(iudp) * WIDTH( iudp) * HEIGHT(iudp)
                      + WIDTH_DOT( iudp) * HEIGHT(iudp) * LENGTH(iudp)
                      + HEIGHT_DOT(iudp) * LENGTH(iudp) * WIDTH( iudp)) / 3;
        
cleanup:
    if (enodes != NULL) EG_free(enodes);
    if (eedges != NULL) EG_free(eedges);
    if (efaces != NULL) EG_free(efaces);
    
#ifdef DEBUG
    printf("udpSensitivity -> vels=%f %f %f\n", vels[0], vels[1], vels[2]);
#endif
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   makeEdge - helper to make linear Edge between 2 Nodes              *
 *                                                                      *
 ************************************************************************
 */

static int
makeEdge(ego enode0,                    /* (in)  first  Node */
         ego enode1,                    /* (in)  second Node */
         ego *eedge)                    /* (out) linear Edge */
{
    int     status = EGADS_SUCCESS;

    int     oclass, mtype, nchild, *senses;
    double  xyz0[3], xyz1[3], data[18], trange[2];
    ego     context, eref, ecurve, enodes[2], *echilds;

    ROUTINE(makeEdge);

    /* --------------------------------------------------------------- */

    /* get the context from one of the Nodes */
    status = EG_getContext(enode0, &context);

    /* get the coordinates of the Nodes */
    status = EG_getTopology(enode0, &eref, &oclass, &mtype, xyz0,
                            &nchild, &echilds, &senses);
    CHECK_STATUS(EG_getTopology);

    status = EG_getTopology(enode1, &eref, &oclass, &mtype, xyz1,
                            &nchild, &echilds, &senses);
    CHECK_STATUS(EG_getTopology);

    /* make a line between the Nodes */
    data[0] = xyz0[0];
    data[1] = xyz0[1];
    data[2] = xyz0[2];
    data[3] = xyz1[0] - xyz0[0];
    data[4] = xyz1[1] - xyz0[1];
    data[5] = xyz1[2] - xyz0[2];

    status = EG_makeGeometry(context, CURVE, LINE, NULL, NULL, data, &ecurve);
    CHECK_STATUS(EG_makeGeometry);

    /* get the trange associated with the Nodes */
    trange[0] = 0;
    trange[1] = sqrt(data[3]*data[3] + data[4]*data[4] + data[5]*data[5]);

    /* make the Edge */
    enodes[0] = enode0;
    enodes[1] = enode1;

    status = EG_makeTopology(context, ecurve, EDGE, TWONODE, trange,
                             2, enodes, NULL, eedge);
    CHECK_STATUS(EG_makeTopology);

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   makeFace - helper to make planar Face given 3 or 4 Edges           (
 *                                                                      *
 ************************************************************************
 */

static int makeFace(ego eedge0,         /* (in)  first  Edge */
                    ego eedge1,         /* (in)  second Edge */
                    ego eedge2,         /* (in)  third  Edge (or NULL) */
                    ego eedge3,         /* (in)  fourth Edge */
                    ego *eface)         /* (out) planar Face */
{
    int     status = EGADS_SUCCESS;

    int     nedge;
    ego     eedges[8], eloop;

    ROUTINE(makeFace);

    /* --------------------------------------------------------------- */

    /* set up array with either 3 or 4 Edges */
    if (eedge2 != NULL) {
        nedge     = 4;
        
        eedges[0] = eedge0;
        eedges[1] = eedge1;
        eedges[2] = eedge2;
        eedges[3] = eedge3;
    } else {
        nedge     = 3;
        
        eedges[0] = eedge0;
        eedges[1] = eedge1;
        eedges[2] = eedge3;
    }

    /* the fast way to do this is to call EG_makeLoop (which will automatically
       reorder and assign senses so that a contiguous Loop is made) and then
       EG_makeFace (which works because these Faces are planar) */
    status = EG_makeLoop(nedge, eedges, NULL, 0., &eloop);
    CHECK_STATUS(EG_makeLoop);

    status = EG_makeFace(eloop, SFORWARD, NULL, eface);
    CHECK_STATUS(EG_makeFace);

cleanup:
    return status;
}
