/*
 ************************************************************************
 *                                                                      *
 * udfExercise3 -- training UDF                                         *
 *                                                                      *
 *             this modified attributes on a Body                       *
 *             sensitivities are not computed                           *
 *                                                                      *
 *             Written by John Dannenhoffer @ Geocentric Technologies   *
 *                                                                      *
 ************************************************************************
 */

/*
 * Copyright (C) 2025  John F. Dannenhoffer, III (Geocentric Technologies))
 *
 * This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *     MA  02110-1301  USA
 */

/* uncomment the following to get DEBUG printouts */
//#define DEBUG 1

/* the number of "input" Bodys

   this only needs to be specified if this is a UDF (user-defined
   function) that consumes Bodys from OpenCSM's stack.  (the default
   value is 0).

   if NUMUDPINPUTBODYS>0  then exactly NUMUDPINPUTBODYS are in emodel
   if NUMUDPINPUTBODYS<0  then up to  -NUMUDPINPUTBODYS are in emodel
*/
#define NUMUDPINPUTBODYS 1

/* the number of arguments (specified below) */
#define NUMUDPARGS 1

/* set up the necessary structures (uses NUMUDPARGS) */
#include "udpUtilities.h"

/* shorthand macros for accessing argument values and velocities */
#define ATTRNAME(IUDP)  ((char   *) (udps[IUDP].arg[0].val))

/* data about possible arguments
      argNames: argument name (must be all lower case)
      argTypes: argument type: +ATTRINT      integer input
                               -ATTRINT      integer output
                               +ATTRREAL     double input  (no sensitivities)
                               -ATTRREAL     double output (no sensitivities)
                               +ATTRREALSEN  double input  (with sensitivities)
                               -ATTRREALSEN  double output (with sensitivities)
                               +ATTRSTRING   string input
                               -ATTRSTRING   *** cannot be used ***
                               +ATTRFILE     input file
                               -ATTRFILE     *** cannot be used ***
                               +ATTRREBUILD  forces rebuild if any variable in
                                             semi-colon-separated list has been changed
                               -ATTRREBUILD  *** cannot be used ***
                               +ATTRRECYCLE  forces rebuild (always) by blocking recycling
                               -ATTRRECYCLE  *** cannot be used ***
      argIdefs: default value for ATTRINT
      argDdefs: default value for ATTRREAL or ATTRREALSEN */
static char  *argNames[NUMUDPARGS] = {"attrname", };
static int    argTypes[NUMUDPARGS] = {ATTRSTRING, };
static int    argIdefs[NUMUDPARGS] = {0,          };
static double argDdefs[NUMUDPARGS] = {0.,         };

/* get utility routines: udpErrorStr, udpInitialize, udpReset, udpSet,
                         udpGet, udpVel, udpClean, udpMesh */
#include "udpUtilities.c"

#ifdef DEBUG
static int printAttributes(ego eobj);
#endif


/*
 ************************************************************************
 *                                                                      *
 *   udpExecute - execute the primitive                                 *
 *                                                                      *
 ************************************************************************
 */

int
udpExecute(ego  emodel,                 /* (in)  Model containing Body */
           ego  *ebody,                 /* (out) Body (or model) pointer */
           int  *nMesh,                 /* (out) number of associated meshes */
           char *string[])              /* (out) error message */
{
    int     status = EGADS_SUCCESS;

    char    *message=NULL;
    udp_T   *udps = *Udps;

    int     attrTypeN, attrTypeE, attrLenN, attrLenE;
    int     nnode, inode, nedge, iedge;
    CINT    *tempIlistN, *tempIlistE;
    CDOUBLE *tempRlistN, *tempRlistE;
    CCHAR   *tempClistN, *tempClistE;
    ego     *ebodys, eref, *enodes=NULL, *eedges=NULL, *theNodes;

    int     oclass, mtype, nchild, *senses, ival;
    double  data[18];

    ROUTINE(udpExecute);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    /* debug printing of the input arguments */
    printf("udpExecute(emodel=%llx)\n", (long long)emodel);
    printf("attrname(0) = %s\n", ATTRNAME(0));
#endif

    /* default return values */
    *ebody  = NULL;
    *nMesh  = 0;
    *string = NULL;

    /* the place where messages to the user are placed */
    MALLOC(message, char, 100);
    message[0] = '\0';

    /* check that Model was input that contains one Body */
    status = EG_getTopology(emodel, &eref, &oclass, &mtype,
                            data, &nchild, &ebodys, &senses);
    CHECK_STATUS(EG_getTopology);

    if (oclass != MODEL) {
        snprintf(message, 100, "expecting a Model, but found oclass=%d", oclass);
        status = EGADS_NOTMODEL;
        goto cleanup;
    } else if (nchild != 1) {
        snprintf(message, 100, "expecting Model to contain one Body (not %d)", nchild);
        status = EGADS_NOTBODY;
        goto cleanup;
    }

    /* check arguments */
    if (STRLEN(ATTRNAME(0)) <= 0) {
        snprintf(message, 100, "\"attrname\" must be specified");
        status = OCSM_ILLEGAL_VALUE;
        goto cleanup;
    }

    /* cache copy of arguments for future use */
    status = cacheUdp(NULL);
    CHECK_STATUS(cacheUdp);

#ifdef DEBUG
    /* debug printing of cached input arguments */
    printf("attrname(%d) = %s\n", numUdp, ATTRNAME(numUdp));
#endif

    /* make a copy of the Body (so that it does not get removed
     when OpenCSM deletes emodel) */
    status = EG_copyObject(ebodys[0], NULL, ebody);
    CHECK_STATUS(EG_copyObject);

    SPLINT_CHECK_FOR_NULL(*ebody);

    /* get arrays of the Nodes and Edges in the Body */
    status = EG_getBodyTopos(*ebody, NULL, NODE, &nnode, &enodes);
    CHECK_STATUS(EG_getBodyTopos);

    status = EG_getBodyTopos(*ebody, NULL, EDGE, &nedge, &eedges);
    CHECK_STATUS(EG_getBodyTopos);

    /* debug print of attributes before processing */
#ifdef DEBUG
    printf("Attributes before processing\n");

    for (inode = 0; inode < nnode; inode++) {
        printf("Node %3d\n", inode+1);
        printAttributes(enodes[inode]);
    }

    for (iedge = 0; iedge < nedge; iedge++) {
        printf("Edge %3d\n", iedge+1);
        printAttributes(eedges[iedge]);
    }
#endif

    /* loop through the Edges, looking for those that have attrname */
    for (iedge = 0; iedge < nedge; iedge++) {
        SPLINT_CHECK_FOR_NULL(eedges);
        
        status = EG_attributeRet(eedges[iedge], ATTRNAME(0), &attrTypeE, &attrLenE,
                                 &tempIlistE, &tempRlistE, &tempClistE);
        if (status == EGADS_NOTFOUND) continue;

        /* find the Nodes at the end of this Edge */
        status = EG_getTopology(eedges[iedge], &eref, &oclass, &mtype, data,
                                &nnode, &theNodes, &senses);
        CHECK_STATUS(EG_getTopology);

        /* check the attribute at the two Nodes */
        for (inode = 0; inode < nnode; inode++) {
            status = EG_attributeRet(theNodes[inode], ATTRNAME(0), &attrTypeN, &attrLenN,
                                     &tempIlistN, &tempRlistN, &tempClistN);

            /* if the attribute does not exist, copy the Edge's attribute */
            if (status == EGADS_NOTFOUND) {
                status = EG_attributeAdd(theNodes[inode], ATTRNAME(0), attrTypeE, attrLenE,
                                         tempIlistE, tempRlistE, tempClistE);
                CHECK_STATUS(EG_attributeAdd);

            /* if it exists but the type or length disagree, put **conflict**
               on the Node */
            } else if (attrTypeN != attrTypeE || attrLenN != attrLenE) {
                status = EG_attributeAdd(theNodes[inode], ATTRNAME(0), ATTRSTRING, 0,
                                         NULL, NULL, "**conflict**");
                CHECK_STATUS(EG_attributeAdd);

            /* if an ATTRINT, make sure all the values match */
            } else if (attrTypeN == ATTRINT) {
                for (ival = 0; ival < attrLenN; ival++) {
                    if (tempIlistN[ival] != tempIlistE[ival]) {
                        status = EG_attributeAdd(theNodes[inode], ATTRNAME(0), ATTRSTRING, 0,
                                                 NULL, NULL, "**conflict**");
                        CHECK_STATUS(EG_attributeAdd);
                        break;
                    }
                }

            /* if an ATTRREAL, make sure all the values match */
            } else if (attrTypeN == ATTRREAL) {
                for (ival = 0; ival < attrLenN; ival++) {
                    if (tempRlistN[ival] != tempRlistE[ival]) {
                        status = EG_attributeAdd(theNodes[inode], ATTRNAME(0), ATTRSTRING, 0,
                                                 NULL, NULL, "**conflict**");
                        CHECK_STATUS(EG_attributeAdd);
                        break;
                    }
                }

            /* if an ATTRSTRING, make sure the values match */
            } else if (attrTypeN == ATTRSTRING) {
                if (strcmp(tempClistN, tempClistE) != 0) {
                    status = EG_attributeAdd(theNodes[inode], ATTRNAME(0), ATTRSTRING, 0,
                                             NULL, NULL, "**conflict**");
                    CHECK_STATUS(EG_attributeAdd);
                }
            }
        }
    }

    /* debug print of attributes after processing */
#ifdef DEBUG
    printf("Attributes after processing\n");

    for (inode = 0; inode < nnode; inode++) {
        printf("Node %3d\n", inode+1);
        printAttributes(enodes[inode]);
    }

    for (iedge = 0; iedge < nedge; iedge++) {
        printf("Edge %3d\n", iedge+1);
        printAttributes(eedges[iedge]);
    }
#endif

    /* add a special Attribute to the Body to tell OpenCSM that there
       is no topological change and hence it should not adjust the
       Attributes on the Body in finishBody() */
    status = EG_attributeAdd(*ebody, "__noTopoChange__", ATTRSTRING,
                             0, NULL, NULL, "udfTrain2");
    CHECK_STATUS(EG_attributeAdd);

    /* set the output value(s) */

    /* remember this model (Body) */
    udps[numUdp].ebody = *ebody;

cleanup:
#ifdef DEBUG
    printf("udpExecute -> numUdp=%d, *ebody=%llx\n", numUdp, (long long)(*ebody));
#endif

    if (enodes != NULL) EG_free(enodes);
    if (eedges != NULL) EG_free(eedges);

    if (strlen(message) > 0) {
        *string = message;
        printf("%s\n", message);
    } else if (status != EGADS_SUCCESS) {
        FREE(message);
        *string = udpErrorStr(status);
    } else {
        FREE(message);
    }

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   udpSensitivity - return sensitivity derivatives for the "real" argument *
 *                                                                      *
 ************************************************************************
 */

int
udpSensitivity(ego    ebody,            /* (in)  Body pointer */
               int    npnt,             /* (in)  number of points */
               int    entType,          /* (in)  OCSM entity type */
               int    entIndex,         /* (in)  OCSM entity index (bias-1) */
               double uvs[],            /* (in)  parametric coordinates for evaluation */
               double vels[])           /* (out) velocities */
{
    int    status = EGADS_SUCCESS;

    int    iudp, judp;

    ROUTINE(udpSensitivity);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    if (uvs != NULL) {
        printf("udpSensitivity(ebody=%llx, npnt=%d, entType=%d, entIndex=%d, uvs=%f %f)\n",
               (long long)ebody, npnt, entType, entIndex, uvs[0], uvs[1]);
    } else {
        printf("udpSensitivity(ebody=%llx, npnt=%d, entType=%d, entIndex=%d, uvs=NULL)\n",
               (long long)ebody, npnt, entType, entIndex);
    }
#endif

    /* check that ebody matches one of the ebodys */
    iudp = 0;
    for (judp = 1; judp <= numUdp; judp++) {
        if (ebody == udps[judp].ebody) {
            iudp = judp;
            break;
        }
    }
    if (iudp <= 0) {
        status = EGADS_NOTMODEL;
        goto cleanup;
    }

    /* the following line should be included if sensitivities
       are not computed analytically */
    status = EGADS_NOLOAD;

cleanup:
#ifdef DEBUG
    printf("udpSensitivity -> vels=%f %f %f\n", vels[0], vels[1], vels[2]);
#endif
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   printAttributes - print attributes associated with an ego          *
 *                                                                      *
 ************************************************************************
 */

#ifdef DEBUG
static int
printAttributes(ego eobj)               /* (in)  ego object */
{
    int     status=EGADS_SUCCESS;       /* (out)  return status */

    int     attrType, attrLen, nattr, iattr, i;
    CINT    *tempIlist;
    CDOUBLE *tempRlist;
    CCHAR   *tempClist, *attrName;

    ROUTINE(printAttributes);

    /* --------------------------------------------------------------- */

    status = EG_attributeNum(eobj, &nattr);
    CHECK_STATUS(EG_attributeNum);

    for (iattr = 1; iattr <= nattr; iattr++) {
        status = EG_attributeGet(eobj, iattr, &attrName, &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        CHECK_STATUS(EG_attributeGet);

        printf("   %-20s ", attrName);
        if (attrType == ATTRINT) {
            for (i = 0; i < attrLen; i++) {
                printf(" %6d", tempIlist[i]);
            }
            printf("\n");
        } else if (attrType == ATTRREAL) {
            for (i = 0; i < attrLen; i++) {
                printf(" %13.7f", tempRlist[i]);
            }
            printf("\n");
        } else if (attrType == ATTRSTRING) {
            printf(" %s\n", tempClist);
        }
    }

cleanup:
    return status;
}
#endif
