/*
 *      CAPS: Computational Aircraft Prototype Syntheses
 *
 *             Session08 AIM Example Code -- solution
 *
 *      Copyright 2014-2025, Massachusetts Institute of Technology
 *      Licensed under The GNU Lesser General Public License, version 2.1
 *      See http://www.opensource.org/licenses/lgpl-2.1.php
 *
 */

#include <string.h>
#include <math.h>
#include "aimUtil.h"

//#define DEBUG
//#define EXPLICITFILES

enum aimInputs
{
  InputVariable = 1,           /* index is 1-based */
  num,
  Mach,
  Mesh_Format,
  FileIn,
  Table,
  NUMINPUT = Table             /* Total number of inputs */
};

enum aimOutputs
{
  sqrtNum = 1,                 /* index is 1-based */
  FileOut,
  NUMOUT  = FileOut            /* Total number of outputs */
};

typedef struct {
  int nBody;
  ego *tess;
} aimStorage;


static int
printValues(void *aimInfo, capsValue *inputs)
{
  int        i, j, n, status = CAPS_SUCCESS;
  const char *name;
  capsValue  *val;

  /* look at the CSM DESPMTRs and CFGPMTRs */
  printf("......GeometryIn:\n");
  n = aim_getIndex(aimInfo, NULL, GEOMETRYIN);
  for (i = 0; i < n; i++) {
    status = aim_getName(aimInfo, i+1, GEOMETRYIN, &name);
    if (status != CAPS_SUCCESS) continue;

    status = aim_getValue(aimInfo, i+1, GEOMETRYIN, &val);
    if (status == CAPS_SUCCESS) {
      printf(".........%2d: %-20s  %d  (%d*%d)\n", i+1, name, val->type,
             val->nrow, val->ncol);
      printf("............");
      if (val->length == 1) {
        printf("%10.5lf", val->vals.real);
      } else {
        for (j = 0; j < val->length; j++) {
          printf("%10.5lf ", val->vals.reals[j]);
        }
      }
      printf("\n");
    }
  }

  /* look at the CSM OUTPMTRs */
  printf("\n......GeometryOut:\n");
  n = aim_getIndex(aimInfo, NULL, GEOMETRYOUT);
  for (i = 0; i < n; i++) {
    status = aim_getName(aimInfo, i+1, GEOMETRYOUT, &name);
    if (status != CAPS_SUCCESS) continue;

    status = aim_getValue(aimInfo, i+1, GEOMETRYOUT, &val);
    if (status == CAPS_SUCCESS) {
      printf(".........%2d: %-20s  %d  (%d*%d)\n",
             i+1, name, val->type, val->nrow, val->ncol);
      printf("............");
      if (val->length == 1) {
        printf("%10.5lf", val->vals.real);
      } else {
        for (j = 0; j < val->length; j++) {
          printf("%10.5lf ", val->vals.reals[j]);
        }
      }
      printf("\n");
    }
  }

  /* write out input list of CAPS values */
  if (inputs != NULL) {
    printf("\n......AnalysisIn:\n");
    for (i = 0; i < NUMINPUT; i++) {
      status = aim_getName(aimInfo, i+1, ANALYSISIN, &name);
      AIM_STATUS(aimInfo, status);

      printf(".........%2d: %-20s  %d  (%d*%d) %s\n", i+1, name,
             inputs[i].type, inputs[i].nrow, inputs[i].ncol, inputs[i].units);
      printf("............");
      if (inputs[i].type == Double) {
        if (inputs[i].length == 1) {
          printf("%10.5lf", inputs[i].vals.real);
        } else {
          for (j = 0; j < inputs[i].length; j++) {
            printf("%10.5lf ", inputs[i].vals.reals[j]);
          }
        }
      } else if ((inputs[i].type == Integer) || (inputs[i].type == Boolean)) {
        if (inputs[i].length == 1) {
          printf(" %d", inputs[i].vals.integer);
        } else {
          for (j = 0; j < inputs[i].length; j++) {
            printf(" %d", inputs[i].vals.integers[j]);
          }
        }
      } else if (inputs[i].type == String) {
        printf(" %s", inputs[i].vals.string);
      } else if (inputs[i].type == Tuple) {
        for (j = 0; j < inputs[i].length; j++)
          if (j == inputs[i].length-1) {
            printf(" %s -- %s", inputs[i].vals.tuple[j].name,
                                inputs[i].vals.tuple[j].value);
          } else {
            printf(" %s -- %s\n", inputs[i].vals.tuple[j].name,
                                  inputs[i].vals.tuple[j].value);
            printf("............");
        }
      }
      printf("\n");
    }
  }
  printf("\n");

cleanup:
  return status;
}


/****************** exposed AIM entry points -- Analysis **********************/

/* aimInitialize: Initialization Information for the AIM */
int
aimInitialize(int inst, /*@unused@*/ const char *unitSys, void *aimInfo,
              void **instStore, int *major, int *minor, int *nIn, int *nOut,
              int *nFields, char ***fnames, int **franks, int **fInOut)
{
  int        status;
  aimStorage *aimStore = NULL;

#ifdef DEBUG
/*@-nullpass@*/
  printf("\n myAIM/aimInitialize  instance = %d  unitSys = %s!\n",
         inst, unitSys);
/*@+nullpass@*/
#endif
  
  /* specify the rev of this AIM */
  *major = 1;
  *minor = 0;

  /* specify the number of analysis inputs  defined in aimInputs
   *     and the number of analysis outputs defined in aimOutputs */
  *nIn  = NUMINPUT;
  *nOut = NUMOUT;

  /* return if "query" only */
  if (inst == -1) return CAPS_SUCCESS;

  /* specify the field variables this analysis can generate and consume */
  *nFields = 0;
  
  /* setup our AIM specific state */
  AIM_ALLOC(aimStore, 1, aimStorage, aimInfo, status);
  *instStore = aimStore;

  /* populate our instance storage */
  aimStore->nBody = 0;
  aimStore->tess  = NULL;

  return CAPS_SUCCESS;
  
cleanup:
  return status;
}


/* aimInputs: Input Information for the AIM */
int
aimInputs(/*@unused@*/ void *instStore, /*@unused@*/ void *aimInfo, int index,
          char **ainame, capsValue *defval)
{
  int        i, status = CAPS_SUCCESS;
  const char *fullPath = NULL;
  capsTuple  *tuple = NULL;
  
#ifdef DEBUG
  printf(" myAIM/aimInputs  instance = %d  index  = %d!\n",
         aim_getInstance(aimInfo), index);
#endif
  if ((index == 1) && (aim_getInstance(aimInfo) == 0)) {
    status = aim_getRootPath(aimInfo, &fullPath);
    if (status == CAPS_SUCCESS) {
/*@-nullpass@*/
      printf("\n myAIM/aimInputs: RootPath = %s\n\n", fullPath);
/*@+nullpass@*/
    } else {
      printf(" myAIM/aimInputs: aim_getRootPath = %d\n", status);
    }
  }

  /* fill in the required members based on the index */
 if (index == InputVariable) {
    *ainame              = AIM_NAME(InputVariable);
    defval->type         = Boolean;
    defval->vals.integer = false;

  } else if (index == num) {
    *ainame           = AIM_NAME(num);
    defval->type      = Double;
    defval->vals.real = 8.0;

  } else if (index == Mach) {
    *ainame              = AIM_NAME(Mach); // Mach number
    defval->type         = Double;
    defval->nullVal      = IsNull;
    defval->units        = NULL;
    defval->lfixed       = Change;
    defval->dim          = Scalar;

  } else if (index == Mesh_Format) {
    *ainame              = AIM_NAME(Mesh_Format);
    defval->type         = String;
    defval->vals.string  = NULL;
    defval->nullVal      = IsNull;
    defval->dim          = Vector;
    defval->lfixed       = Change;
    
  } else if (index == FileIn) {
    *ainame              = AIM_NAME(FileIn);
    defval->type         = String;
    defval->lfixed       = Change;
    defval->nullVal      = IsNull;

  } else if (index == Table) {

    /* an example of filling in a Tuple */
    *ainame = AIM_NAME(Table);

    AIM_ALLOC( tuple, 3, capsTuple, aimInfo, status);
    for (i = 0; i < 3; i++) {
      tuple[i].name  = NULL;
      tuple[i].value = NULL;
    }
    AIM_STRDUP(tuple[0].name , "Entry1", aimInfo, status);
    AIM_STRDUP(tuple[1].name , "Entry2", aimInfo, status);
    AIM_STRDUP(tuple[2].name , "Entry3", aimInfo, status);
    AIM_STRDUP(tuple[0].value, "Value1", aimInfo, status);
    AIM_STRDUP(tuple[1].value, "Value2", aimInfo, status);
    AIM_STRDUP(tuple[2].value, "Value3", aimInfo, status);

    defval->type       = Tuple;
    defval->dim        = Vector;
    defval->nrow       = 1;
    defval->ncol       = 3;
    defval->vals.tuple = tuple;
  
  }

  AIM_NOTNULL(*ainame, aimInfo, status);

cleanup:
  return status;
}


/* aimUpdateState: The always the first call in the execution sequence */
int
aimUpdateState(void *instStore, void *aimInfo, /*@unused@*/ capsValue *inputs)
{
  int        i, nBody, status = CAPS_SUCCESS;
  double     size, box[6], params[3], props[14];
  const char *intents;
  ego        *bodies;
  aimStorage *aimStore;
  
#ifdef DEBUG
  printf(" myAIM/aimUpdateState        instance = %d!\n",
         aim_getInstance(aimInfo));
#endif
  aimStore = (aimStorage *) instStore;
  
  status = aim_newGeometry(aimInfo);
  printf(" myAIM/aimUpdateState instance = %d   aim_newGeometry = %d!\n",
         aim_getInstance(aimInfo), status);
  if (status == CAPS_CLEAN)   return CAPS_SUCCESS;
  if (status != CAPS_SUCCESS) return status;
  
  /* create the tessellation */
  
  status = aim_getBodies(aimInfo, &intents, &nBody, &bodies);
  AIM_STATUS(aimInfo, status, "aim_getBodies");
  if ((bodies == NULL) || (nBody == 0)) return CAPS_SUCCESS;

  aimStore->nBody = nBody;
  AIM_REALL(aimStore->tess, aimStore->nBody, ego, aimInfo, status);

  for (i = 0; i < nBody; i++) {
    /* tessellate with EGADS tessellation in this example */
    status = EG_getBoundingBox(bodies[i], box);
    AIM_STATUS(aimInfo, status);

                              size = box[3]-box[0];
    if (size < box[4]-box[1]) size = box[4]-box[1];
    if (size < box[5]-box[2]) size = box[5]-box[2];
    params[0] =  0.025*size;
    params[1] =  0.001*size;
    params[2] = 15.0;
    status = EG_makeTessBody(bodies[i], params, &aimStore->tess[i]);
    AIM_STATUS(aimInfo, status);
    
    status = EG_getMassProperties(bodies[i], props);
    AIM_STATUS(aimInfo, status);
    printf(" myAIM/aimUpdateState intents = %s  vol = %6.2lf  tess %d complete!\n",
           intents, props[0], i);
    
    /* store the tessellation in CAPS */
    status = aim_newTess(aimInfo, aimStore->tess[i]);
    AIM_STATUS(aimInfo, status);
  }
  printf("\n");

cleanup:
  return status;
}


/* aimPreAnalysis: Parse Inputs, Generate Input File(s) */
int
aimPreAnalysis(/*@unused@*/ const void *instStore, void *aimInfo,
               capsValue *inputs)
{
  int      inst, status = CAPS_SUCCESS;
  double mach;
  FILE   *fp;

  inst = aim_getInstance(aimInfo);
#ifdef DEBUG
  printf("\n myAIM/aimPreAnalysis instance = %d!\n", inst);
#endif

  printValues(aimInfo, inputs);

  /* examine input values for correctness */
  if (inputs != NULL) {
    mach = inputs[Mach-1].vals.real;
    if (mach < 0) {
      AIM_ANALYSISIN_ERROR(aimInfo, Mach, "Mach number must be >= 0\n");
      AIM_ADDLINE(aimInfo, "Negative mach = %f is non-physical\n", mach);
      status = CAPS_BADVALUE;
      goto cleanup;
    }
  }
  
  /* read our file */
#ifdef EXPLICITFILES
  if (inst == 1) {
    if (inputs[FileIn-1].nullVal != IsNull) {
      printf(" Error: FileIn has a value!\n");
      status = CAPS_NOTFOUND;
      goto cleanup;
    }
    fp = aim_fopen(aimInfo, "../myAIM0/file.txt", "r");
    if (fp == NULL) {
      printf("Error: '../myAIM0/file.txt' -- file not found!\n");
      status = CAPS_NOTFOUND;
    } else {
      /* read file contents */
      printf(" ...Reading...\n");
      fclose(fp);
    }
  }
#else
  if (inst == 1) {
    if (inputs[FileIn-1].nullVal == IsNull) {
      printf(" Error: no FileIn!\n");
      status = CAPS_NOTFOUND;
      goto cleanup;
    }
    printf(" FileIn = %s\n", inputs[FileIn-1].vals.string);
    fp = fopen(inputs[FileIn-1].vals.string, "r");
    if (fp == NULL) {
      printf("Error: FileIn = %s  -- file not found!\n",
             inputs[FileIn-1].vals.string);
      status = CAPS_NOTFOUND;
    } else {
      /* read file contents */
      printf(" ...Reading...\n");
      fclose(fp);
    }
  }
#endif
  
cleanup:
  return status;
}


/* aimExecute: runs the Analysis & specifies the AIM does the execution */
int
aimExecute(/*@unused@*/ const void *instStor, /*@unused@*/ void *aimInfo,
           int *state)
{
#ifdef DEBUG
  printf(" myAIM/aimExecute instance = %d!\n", aim_getInstance(aimInfo));
#endif

  *state = 0;
  return CAPS_SUCCESS;
}


/* aimPostAnalysis: Perform any processing after the Analysis is run */
int
aimPostAnalysis(/*@unused@*/ void *instStore, void *aimInfo, int restart,
                /*@unused@*/ capsValue *inputs)
{
  int       status = CAPS_SUCCESS;
  capsValue dynOut;
#ifdef EXPLICITFILES
  FILE      *fp;
#endif

#ifdef DEBUG
  printf(" myAIM/aimPostAnalysis instance = %d!\n", aim_getInstance(aimInfo));
#endif
  
  if (restart == 0) {
    status = aim_initValue(&dynOut);
    AIM_STATUS(aimInfo, status);
    dynOut.vals.integer = 42;
    status = aim_makeDynamicOutput(aimInfo, "Everything", &dynOut);
    AIM_STATUS(aimInfo, status);
  }

  /* write our file */
#ifdef EXPLICITFILES
  if (aim_getInstance(aimInfo) == 0) {
    fp = aim_fopen(aimInfo, "file.txt", "w");
    if (fp == NULL) {
      printf("Error: 'file.txt' -- cannot open!\n");
      return CAPS_DIRERR;
    } else {
      /* write file contents */
      printf(" ...Writing file.txt in Post...\n");
      fclose(fp);
    }
  }
#endif
  
cleanup:
  return status;
}


/* aimOutputs: Output Information for the AIM */
int
aimOutputs(/*@unused@*/ void *instStore, /*@unused@*/ void *aimInfo, int index,
           char **aoname, capsValue *form)
{
  int status = CAPS_SUCCESS;
#ifdef DEBUG
  printf(" myAIM/aimOutputs instance = %d  index  = %d!\n",
         aim_getInstance(aimInfo), index);
#endif
  
  if (index == sqrtNum) {
    *aoname = AIM_NAME(sqrtNum);
    form->type = Double;
  } else if (index == FileOut) {
    *aoname = AIM_NAME(FileOut);
    form->type    = String;
    form->nullVal = IsNull;
  } else {
    printf(" myAIM/aimOutputs: Unknown index = %d!\n", index);
    status = CAPS_BADINDEX;
  }

  return status;
}


/* aimCalcOutput: Calculate/Retrieve Output Information */
int
aimCalcOutput(/*@unused@*/ void *instStore, void *aimInfo, int index,
              capsValue *val)
{
  int       status = CAPS_SUCCESS;
  double    myNum;
  capsValue *myVal;
  char      aPath[PATH_MAX];
  FILE      *fp;
#ifdef DEBUG
  const char *name;

  status = aim_getName(aimInfo, index, ANALYSISOUT, &name);
  printf(" myAIM/aimCalcOutput instance = %d  index = %d %s %d!\n",
         aim_getInstance(aimInfo), index, name, status);
#endif

  if (index == sqrtNum) {
    /* default return */
    val->vals.real = -1;
    
    /* get the input num */
    status = aim_getValue(aimInfo, num, ANALYSISIN, &myVal);
    AIM_STATUS(aimInfo, status);
    if ((myVal->type != Double) || (myVal->length != 1)) {
      printf("ERROR:: aim_getValue -> type=%d, len = %d\n",
             myVal->type, myVal->length);
      status = CAPS_BADTYPE;
      goto cleanup;
    }
    
    myNum = myVal->vals.real;
    val->vals.real = sqrt(myNum);
    
  } else if (index == FileOut) {
    if (aim_getInstance(aimInfo) == 0) {
      status = aim_file(aimInfo, "linked.txt", aPath);
      AIM_STATUS(aimInfo, status);
      fp = fopen(aPath, "w");
      if (fp == NULL) {
        printf("Error: aPath = %s  -- cannot open!\n", aPath);
        return CAPS_DIRERR;
      } else {
        /* write file contents */
        printf(" ...Writing linked.txt in CalcOutput...\n");
        fclose(fp);
      }
      AIM_STRDUP(val->vals.string, aPath, aimInfo, status);
    }
  } else {
    status = CAPS_BADVALUE;
  }

cleanup:
  return status;
}


/* aimCleanup: Free up the AIMs storage */
void aimCleanup(void *instStore)
{
  aimStorage *aimStore = NULL;
#ifdef DEBUG
  printf(" myAIM/aimCleanup!\n");
#endif

  /* clean up any allocated data 
   * tessellation objects are deleted by CAPS 
   */

  aimStore = (aimStorage *) instStore;
  if (aimStore == NULL) return;
  aimStore->nBody = 0;
  AIM_FREE(aimStore->tess);
  AIM_FREE(aimStore);
}
