/*
 *      CAPS: Computational Aircraft Prototype Syntheses
 *
 *             Session09: Example of AIM that calls an analysis program
 *
 *      Copyright 2014-2025, Massachusetts Institute of Technology
 *      Licensed under The GNU Lesser General Public License, version 2.1
 *      See http://www.opensource.org/licenses/lgpl-2.1.php
 *
 */

#include <string.h>
#include "aimUtil.h"

/* uncomment to track the calls that CAPS makes to these various routines */
//#define DEBUG

/* list of inputs that the AIM accepts */
enum aimInputs
{
    WALL_DENSITY = 1,           /* index is bias-1 */
    FUEL_DENSITY,
    NUMINPUT = FUEL_DENSITY     /* total number of inputs */
};

/* list of outputs that the AIM produces */
enum aimOutputs
{
    AREA = 1,                  /* index is bias-1 */
    VOL,
    MASS,
    NUMOUTPUT = MASS           /* total number of outputs */
};

/* structure to hold instance-specific information */
typedef struct {
    int    nBody;              /* number of Bodys */
    ego    *tess;              /* array  of tessellation egos */
    double area;               /* surface area computed by analysis */
    double volume;             /* volume       computed by analysis */
    double mass;               /* mass         computed by analysis */
} aimStorage;


/****************** exposed AIM entry points -- Analysis **********************/


/* *******************************************************/
/* aimInitialize: Initialization Information for the AIM */
/* *******************************************************/
int
aimInitialize(int   inst,               /* (in)  -1 to indicate a query or instance # */
 /*@unused@*/ const char *unitSys,      /* (in)  string declaring unit system */
              void  *aimInfo,           /* (out) new AIM context */
              void  **instStore,        /* (out) pointer to storage for this instance */
              int   *major,             /* (out) major version number */
              int   *minor,             /* (out) minor version number */
              int   *nIn,               /* (out) number of analysis inputs */
              int   *nOut,              /* (out) number of analysis outputs */
              int   *nFields,           /* (out) number of fields for DataSet filling */
 /*@unused@*/ char  ***fnames,          /* (out) field/DataSet names */
 /*@unused@*/ int   **franks,           /* (out) field/DataSet ranks */
 /*@unused@*/ int   **fInOut)           /* (out) field types */
{
    int        status = CAPS_SUCCESS;

    aimStorage *aimStore = NULL;

#ifdef DEBUG
/*@-nullpass@*/
    printf("...theAIM/aimInitialize (inst=%d, unitSys=%s)\n", inst, unitSys);
/*@+nullpass@*/
#endif

    /* specify the rev of this AIM */
    *major = 1;
    *minor = 0;

    /* specify the number of analysis inputs  defined in aimInputs
     *     and the number of analysis outputs defined in aimOutputs */
    *nIn  = NUMINPUT;
    *nOut = NUMOUTPUT;

    /* return if "query" only */
    if (inst == -1) return CAPS_SUCCESS;

    /* specify the field variables this analysis can generate and consume */
    *nFields = 0;

    /* setup our AIM specific state */
    AIM_ALLOC(aimStore, 1, aimStorage, aimInfo, status);
    *instStore = aimStore;

    /* populate our instance storage */
    aimStore->nBody  = 0;
    aimStore->tess   = NULL;
    aimStore->area   = -1;    /* bad values */
    aimStore->volume = -2;
    aimStore->mass   = -3;

    return CAPS_SUCCESS;

cleanup:
    return status;
}


/* **************************************************************/
/* aimInputs: Information about the inputs that the AIM accepts */
/* **************************************************************/
int
aimInputs(
/*@unused@*/ void      *instStore,    /* (in)  pointer to instance storage */
             void      *aimInfo,      /* (in)  AIM context (or NULL) */
             int       index,         /* (in)  index of analysis input (bias 1) */
             char      **ainame,      /* (out) name of index'th analysis input */
             capsValue *defval)       /* (out) default value and units */
{
    int status = CAPS_SUCCESS;

#ifdef DEBUG
    printf("...theAIM/aimInputs (inst=%d, index=%d)\n", aim_getInstance(aimInfo),
           index);
#endif

    /* fill in the required members based on the index */
    if        (index == WALL_DENSITY) {
        *ainame           = AIM_NAME(wallDensity);
        defval->type      = Double;
        defval->vals.real = 1;

    } else if (index == FUEL_DENSITY) {
        *ainame           = AIM_NAME(fuelDensity);
        defval->type      = Double;
        defval->vals.real = 1;

    } else {
        printf("......unknown index=%d\n", index);
        status = CAPS_BADINDEX;
        goto cleanup;
    }

    AIM_NOTNULL(*ainame, aimInfo, status);

cleanup:
    return status;
}


/* *****************************************************/
/* aimUpdateState: First call in an execution sequence */
/* *****************************************************/
int
aimUpdateState(void      *instStore,    /* (in)  instance storage */
               void      *aimInfo,      /* (in)  AIM context */
  /*@unused@*/ capsValue *inputs)       /* (in)  array of analysis inputs (bias 0) */
{
    int        status = CAPS_SUCCESS;

    int        i, nBody;
    double     size, box[6], params[3];
    const char *intents;
    ego        *bodies;
    aimStorage *aimStore = (aimStorage *) instStore;

#ifdef DEBUG
    printf("...theAIM/aimUpdateState (inst=%d)\n", aim_getInstance(aimInfo));
#endif

    /* determine if the geometry has been changed since previous call */
    status = aim_newGeometry(aimInfo);
#ifdef DEBUG
    printf("......aim_newGeometry() -> status=%d\n", status);
#endif
    if (status == CAPS_CLEAN || status != CAPS_SUCCESS) {
        status = CAPS_SUCCESS;
        goto cleanup;
    }

    /* if it has been changed, create a tessellation */
    status = aim_getBodies(aimInfo, &intents, &nBody, &bodies);
    AIM_STATUS(aimInfo, status, "aim_getBodies");

    if ((bodies == NULL) || (nBody == 0)) {
        status = CAPS_SUCCESS;
        goto cleanup;
    }

    /* make room for the tessellations */
    aimStore->nBody = nBody;
    AIM_REALL(aimStore->tess, aimStore->nBody, ego, aimInfo, status);

    for (i = 0; i < nBody; i++) {
        /* tessellate with EGADS tessellation in this example */
        status = EG_getBoundingBox(bodies[i], box);
        AIM_STATUS(aimInfo, status);

                                  size = box[3]-box[0];
        if (size < box[4]-box[1]) size = box[4]-box[1];
        if (size < box[5]-box[2]) size = box[5]-box[2];
        params[0] =  0.025*size;
        params[1] =  0.001*size;
        params[2] = 15.0;
        status = EG_makeTessBody(bodies[i], params, &aimStore->tess[i]);
        AIM_STATUS(aimInfo, status);

        /* store the tessellation in CAPS */
        status = aim_newTess(aimInfo, aimStore->tess[i]);
        AIM_STATUS(aimInfo, status);
    }

cleanup:
    return status;
}


/* *****************************************************/
/* aimPreAnalysis: Generate input file(s) for analysis */
/* *****************************************************/
int
aimPreAnalysis(const void *instStore,  /* (in)  instance storage */
               void       *aimInfo,    /* (in)  AIM context */
               capsValue  *inputs)     /* (in)  array of analysis inputs (bias 0) */
{
    int          status = CAPS_SUCCESS;

    int          stat, npts, nface, iface, npnt, ntri, itri, ip0, ip1, ip2;
    const int    *ptype, *pindx, *tris, *tric;
    double       wallDens, fuelDens;
    const double *xyz, *uv;
    ego          ebody;
    FILE         *fp;

    aimStorage *aimStore = (aimStorage *) instStore;

#ifdef DEBUG
    printf("...theAIM/aimPreAnalysis (inst=%d)\n", aim_getInstance(aimInfo));
#endif
  
    /* see if we have a single body/tessellation */
    if (aimStore->nBody != 1) {
        AIM_ERROR(aimInfo, "Only works with a single Body (nBody = %d)!",
                  aimStore->nBody);
        status = CAPS_NOBODIES;
        goto cleanup;
    }

    /* write a file that contains header info followed by the
       tessellation for the first Body */
    fp = aim_fopen(aimInfo, "tankCalc.in", "w");
    if (fp == NULL) {
        printf("......could not open 'tankCalc.in' for input\n");
        status = CAPS_IOERR;
        goto cleanup;
    }

    /* put the values of the AIM inputs into the file */
    wallDens = inputs[WALL_DENSITY-1].vals.real;
    fuelDens = inputs[FUEL_DENSITY-1].vals.real;

    fprintf(fp, "    0\n");
    fprintf(fp, "%12.6lf %12.6lf\n\n", wallDens, fuelDens);

    /* get the number of Faces in the Body */
    status = EG_statusTessBody(aimStore->tess[0], &ebody, &stat, &npts);
    AIM_STATUS(aimInfo, status);

    status = EG_getBodyTopos(ebody, NULL, FACE, &nface, NULL);
    AIM_STATUS(aimInfo, status);

    /* get the tessellation for each Face and write to file */
    for (iface = 1; iface <= nface; iface++) {
        status = EG_getTessFace(aimStore->tess[0], iface,
                                &npnt, &xyz, &uv, &ptype, &pindx,
                                &ntri, &tris, &tric);
        AIM_STATUS(aimInfo, status);

        for (itri = 0; itri < ntri; itri++) {
            ip0 = tris[3*itri  ] - 1;
            ip1 = tris[3*itri+1] - 1;
            ip2 = tris[3*itri+2] - 1;

            fprintf(fp, "%12.6lf %12.6lf %12.6lf\n",   xyz[3*ip0],
                    xyz[3*ip0+1], xyz[3*ip0+2]);
            fprintf(fp, "%12.6lf %12.6lf %12.6lf\n",   xyz[3*ip1],
                    xyz[3*ip1+1], xyz[3*ip1+2]);
            fprintf(fp, "%12.6lf %12.6lf %12.6lf\n\n", xyz[3*ip2],
                    xyz[3*ip2+1], xyz[3*ip2+2]);
        }
    }

    fclose(fp);

cleanup:
    return status;
}


/* ***************************************************************************/
/* aimExecute: runs the Analysis & specifies the AIM that does the execution */
/* ***************************************************************************/
int
aimExecute(
/*@unused@*/ const void *instStor,      /* (in)  instance storage */
             void       *aimInfo,       /* (in)  AIM context */
             int        *state)         /* (out) status (0=done, 1-running) */
{
    int  status = CAPS_SUCCESS;
    char infile[PATH_MAX], outfile[PATH_MAX];
    char *command = NULL;
    
#ifdef DEBUG
    printf("...theAIM/aimExecute (inst=%d)\n", aim_getInstance(aimInfo));
#endif

    /* get the path for the input and output files */
    status = aim_file(aimInfo, "tankCalc.in",  infile);
    AIM_STATUS(aimInfo, status);
    status = aim_file(aimInfo, "tankCalc.out", outfile);
    AIM_STATUS(aimInfo, status);
    
    AIM_ALLOC(command, strlen(infile)+strlen(outfile)+20, char, aimInfo, status);

    /* construct the command-line  */
    sprintf(command, "tankCalc %s %s", infile, outfile);
#ifdef DEBUG
    printf("......command=%s\n", command);
#endif
    
    /* run the analysis */
    status = aim_system(aimInfo, NULL, command);
    AIM_STATUS(aimInfo, status);

    *state = 0;

cleanup:
    AIM_FREE(command);
    
    return status;
}


/* *******************************************************************/
/* aimPostAnalysis: Perform any processing after the Analysis is run */
/* *******************************************************************/
int
aimPostAnalysis(void      *instStore,   /* (in)  instance storage */
                void      *aimInfo,     /* (in)  AIM context */
   /*@unused@*/ int       restart,      /* (in)  restart state (0=normal, 1=restart) */
   /*@unused@*/ capsValue *inputs)      /* (in)  array of analysis inputs (bias 0) */
{
    FILE       *fp;
    aimStorage *aimStore = (aimStorage *) instStore;

#ifdef DEBUG
    printf("...theAIM/aimPostAnalysis (inst=%d)\n", aim_getInstance(aimInfo));
#endif

    /* read the analysis output file and store the values in the
       instance storage */
    fp = aim_fopen(aimInfo, "tankCalc.out", "r");
    if (fp == NULL) {
        printf("......could not open 'tankCalc.out' for reading\n");
        return CAPS_IOERR;
    }
    
    fscanf(fp, "%lf %lf %lf", &aimStore->area, &aimStore->volume,
                              &aimStore->mass);
    fclose(fp);

    return CAPS_SUCCESS;
}


/* *****************************************************************/
/* aimOutputs: Information about the outputs that the AIM produces */
/* *****************************************************************/
int
aimOutputs(
/*@unused@*/ void      *instStore,   /* (in)  instance storage */
             void      *aimInfo,     /* (in)  AIM context */
             int       index,        /* (in)  index of analysis output to return (bias 1) */
             char      **aoname,     /* (out) analysis output name */
             capsValue *form)        /* (out) analysis output value/units */
{
    int status = CAPS_SUCCESS;

#ifdef DEBUG
    printf("...theAIM/aimOutputs (inst=%d, index=%d)\n",
           aim_getInstance(aimInfo), index);
#endif

    if        (index == AREA) {
        *aoname    = AIM_NAME(area);
        form->type = Double;

    } else if (index == VOL) {
        *aoname    = AIM_NAME(vol);
        form->type = Double;

    } else if (index == MASS) {
        *aoname    = AIM_NAME(mass);
        form->type = Double;

    } else {
        printf("......unknown index=%d\n", index);
        status = CAPS_BADINDEX;
        goto cleanup;
    }
  
    AIM_NOTNULL(*aoname, aimInfo, status);

cleanup:
    return status;
}


/* *************************************************************************/
/* aimCalcOutput: Fill specific output variable (possibly from information */
/*                that was created in aimPostAnalysis)                     */
/* *************************************************************************/
int
aimCalcOutput(void      *instStore,  /* (in)  instance storage */
 /*@unused@*/ void      *aimInfo,    /* (in)  AIM context */
              int       index,       /* (in)  index of analysis output to return (bias 1) */
              capsValue *val)        /* (out) returned value/units */
{
    int        status = CAPS_SUCCESS;

    aimStorage *aimStore = (aimStorage *) instStore;

#ifdef DEBUG
    const char *name;

    status = aim_getName(aimInfo, index, ANALYSISOUT, &name);
    printf("...theAIM/aimCalcOutput (inst=%d, index=%d (%s) -> status=%d\n",
           aim_getInstance(aimInfo), index, name, status);
#endif

    if        (index == AREA) {
        val->vals.real = aimStore->area;

    } else if (index == VOL) {
        val->vals.real = aimStore->volume;

    } else if (index == MASS) {
        val->vals.real = aimStore->mass;

    } else {
        status = CAPS_BADVALUE;
        goto cleanup;
    }

cleanup:
    return status;
}


/* **************************************/
/* aimCleanup: Free up the AIMs storage */
/* **************************************/
void aimCleanup(void *instStore)        /* (both) instance storage */
{
    aimStorage *aimStore = (aimStorage *) instStore;

#ifdef DEBUG
    printf("...theAIM/aimCleanup\n");
#endif

    /* clean up any allocated data
     * tessellation objects are deleted by CAPS
     */

    if (aimStore == NULL) return;

    aimStore->nBody = 0;
    AIM_FREE(aimStore->tess);
    AIM_FREE(aimStore);
}
