###################################################################
#                                                                 #
# runAvl --- run AVL on wing geometry; plot CL(alpha) and CL(CD)  #
#                                                                 #
#              Written by John Dannenhoffer @ Syracuse University #
#                     and Marshall Galbraith @ MIT                #
#                                                                 #
###################################################################

# import pyCAPS module
import pyCAPS
from   pyOCSM import esp

import os

#------------------------------------------------------------------

# make a semi-colon-separated string from a list
def makeString(array):
    out = ""
    for i in array:
        out += str(i) + ";"
    return out

#------------------------------------------------------------------

# load geometry [.csm] file
filename = os.path.join(__file__, "..", "wingAvl.csm")

capsProblem = pyCAPS.Problem(problemName = "runAvl",
                             capsFile    = filename,
                             outLevel    = 0)

# make sure there is no tip treatment
capsProblem.geometry.despmtr["wing:tiptreat"].value = 0

# setup AIM for AVL
print ("\n==> Create avlAIM")
avl = capsProblem.analysis.create(aim  = "avlAIM",
                                  name = "avl")

# view the geometry with ESP
print ("\n==> Viewing AVL Wing bodies...")
avl.geometry.view()

print ("\n==> Setting analysis values")
avl.input.Alpha = 1.0

# set meshing parameters for each surface
wing = {"numChord"     : 4,
        "numSpanTotal" : 24}

# Associate the surface parameters 
# with capsGroups defined on the geometry
avl.input.AVL_Surface = {"Wing" : wing }

# initialize storage of results
Alpha = []
CL    = []
CD    = []

# run a sweep of angles of attack
for alpha in range(-10, 11):
    print("--> alpha", alpha)
    
    try:
        avl.input.Alpha = alpha

        CL.append(avl.output.CLtot)
        CD.append(avl.output.CDtot)
        Alpha.append(alpha)
    except pyCAPS.CAPSError:
        print("    *** did not work ***")

# load the plotter
esp.TimLoad("plotter", esp.GetEsp("pyscript"), "")

# plot the lift curve
esp.TimMesg("plotter", "new|AVL results for wing|alpha (deg)|CL|")
esp.TimMesg("plotter", "add|"+makeString(Alpha)+"|"+makeString(CL)+"|k-+|")
esp.TimMesg("plotter", "add|"+str(Alpha[0])+";"+str(Alpha[-1])+"|0;0|k:|")
esp.TimMesg("plotter", "add|-.1;+.1|0;0|k:|")
esp.TimMesg("plotter", "show")

# plot the drag polar
esp.TimMesg("plotter", "new|AVL results for wing|CL|CD|")
esp.TimMesg("plotter", "add|"+makeString(CL)+"|"+makeString(CD)+"|k-+|")
esp.TimMesg("plotter", "add|"+str(CL[0])+";"+str(CL[-1])+"|0;0|k:|")
esp.TimMesg("plotter", "show")

# exit the plotter
esp.TimQuit("plotter")

# close the capsProblem (required if you want to run another pyscript)
capsProblem.close()
