###################################################################
#                                                                 #
# runAvlFlaps --- run AVL on wing geometry; CL(flap) and CD(CL)   #
#                                                                 #
#              Written by John Dannenhoffer @ Syracuse University #
#                     and Marshall Galbraith @ MIT                #
#                                                                 #
###################################################################

# Import pyCAPS module
import pyCAPS
from   pyOCSM import esp

import os

#------------------------------------------------------------------

# make a semi-colon-separated string from a list
def makeString(array):
    out = ""
    for i in array:
        out += str(i) + ";"
    return out

#------------------------------------------------------------------

# load geometry [.csm] file
filename = os.path.join(__file__, "..", "wingAvl.csm")

capsProblem = pyCAPS.Problem(problemName = "runAvlFlaps",
                             capsFile    = filename,
                             outLevel    = 0)

# make sure there is no tip treatment
capsProblem.geometry.despmtr["wing:tiptreat"].value = 0

# make sure the controls are on
capsProblem.geometry.cfgpmtr["COMP:controls"].value = 1

# setup AIM for AVL
print ("\n==> Create avlAIM")
avl = capsProblem.analysis.create(aim  = "avlAIM",
                                  name = "avl")

# view the geometry with ESP
print ("\n==> Viewing AVL Wing bodies...")
avl.geometry.view()

print ("\n==> Setting analysis values")
avl.input.Alpha = 4.0

# set meshing parameters for each surface
wing = {"numChord"     : 4,
        "numSpanTotal" : 80}

# Associate the surface parameters 
# with capsGroups defined on the geometry
avl.input.AVL_Surface = {"Wing" : wing }

# initialize storage of results
Delta = []
CL    = []
CD    = []

# run a sweep of flap deflections
for delta in range(0, 30, 5):
    print("--> delta", delta)
    
    try:
        # set up control surface deflections
        controls = {}

        hinge = capsProblem.geometry.despmtr["wing:hinge"].value
        for i in range(len(hinge)):
            controls["WingControl"+str(int(hinge[i][8]))] = {"deflectionAngle": delta}
        avl.input.AVL_Control = controls

        # explicitly trigger execution
        avl.runAnalysis()

        CL.append(avl.output.CLtot)
        CD.append(avl.output.CDtot)
        Delta.append(delta)
    except pyCAPS.CAPSError:
        print("    *** did not work ***")

# load the plotter
esp.TimLoad("plotter", esp.GetEsp("pyscript"), "")

# plot the lift curve
esp.TimMesg("plotter", "new|AVL results for wing|delta (deg)|CL|")
esp.TimMesg("plotter", "add|"+makeString(Delta)+"|"+makeString(CL)+"|k-+|")
esp.TimMesg("plotter", "add|"+str(Delta[0])+";"+str(Delta[-1])+"|0;0|k:|")
esp.TimMesg("plotter", "add|-.1;+.1|0;0|k:|")
esp.TimMesg("plotter", "show")

# plot the drag polar
esp.TimMesg("plotter", "new|AVL results for wing|CL|CD|")
esp.TimMesg("plotter", "add|"+makeString(CL)+"|"+makeString(CD)+"|k-+|")
esp.TimMesg("plotter", "add|"+str(CL[0])+";"+str(CL[-1])+"|0;0|k:|")
esp.TimMesg("plotter", "show")

# exit the plotter
esp.TimQuit("plotter")

# close the capsProblem (required if you want to run another pyscript)
capsProblem.close()
