###################################################################
#                                                                 #
# runSu2Inv --- run Su2 on wing geometry                          #
#                                                                 #
#              Written by John Dannenhoffer @ Syracuse University #
#                     and Marshall Galbraith @ MIT                #
#                                                                 #
###################################################################

# import pyCAPS module
import pyCAPS
from   pyOCSM import esp

import os
import time

# import SU2 python environment
from parallel_computation import parallel_computation as su2Run

#------------------------------------------------------------------

# load geometry [.csm] file
filename = os.path.join(__file__, "..", "wingCfdInv.csm")

capsProblem = pyCAPS.Problem(problemName = "runSu2Inv",
                             capsFile    = filename,
                             outLevel    = 0)

# create aflr4 AIM
aflr4 = capsProblem.analysis.create(aim  = "aflr4AIM",
                                    name = "aflr4")

# farfield growth factor
aflr4.input.ff_cdfr = 1.4

# Curvature spacing angle in degrees
aflr4.input.curv_angle = 15

# Global edge mesh spacing scale factor flag
aflr4.input.mer_all = True

# Global edge mesh spacing refinement weight
aflr4.input.erw_all = 0.05

# scaling factor to compute AFLR4 'ref_len' parameter
aflr4.input.Mesh_Length_Factor = 4

# edge mesh spacing discontinuity scaled interpolant and farfield meshing BC
aflr4.input.Mesh_Sizing = {"leftWing": {"edgeWeight":1.0},
                           "riteWing": {"edgeWeight":1.0},
                           "Farfield": {"bcType":"Farfield"}}

# create AFLR3 AIM to generate the volume mesh
aflr3 = capsProblem.analysis.create(aim  = "aflr3AIM",
                                  name = "aflr3")

# link the aflr4 Surface_Mesh as input to aflr3
aflr3.input["Surface_Mesh"].link(aflr4.output["Surface_Mesh"])

# runtime counter begin
tic = time.perf_counter()

# show the surface meshes
aflr4.geometry.view()

# create SU2 AIM
su2 = capsProblem.analysis.create(aim  = "su2AIM",
                                  name = "su2")

# link the aflr3 Volume_Mesh as input to su2
su2.input["Mesh"].link(aflr3.output["Volume_Mesh"])

# set project name. Files written to analysisDir will have this name
su2.input.Proj_Name = "inviscidWing"

su2.input.Alpha         = 1.0            # AoA
su2.input.Mach          = 0.5            # Mach number
su2.input.Equation_Type = "Compressible" # Equation type
su2.input.Num_Iter      = 10             # Number of iterations

# set number of inner iterations and convergence
su2.input.Input_String = ["INNER_ITER= 10",
                          "CONV_FIELD= RMS_DENSITY",
                          "CONV_RESIDUAL_MINVAL= -8"]

# set boundary conditions via capsGroup
inviscidBC = {"bcType" : "Inviscid"}
su2.input.Boundary_Condition = {"Wing"    : inviscidBC,
                                "Farfield": "farfield"}

# specifcy the boundares used to compute forces
su2.input.Surface_Monitor = ["Wing"]

# set SU2 Version
su2.input.SU2_Version = "Blackbird"

# run AIM pre-analysis
su2.preAnalysis()

####### Run SU2 #######################
print ("\n\nRunning SU2......")
currentDirectory = os.getcwd() # Get our current working directory

os.chdir(su2.analysisDir) # Move into test directory

# run SU2 with specified number of partitions
su2Run(su2.input.Proj_Name + ".cfg", partitions = 1)

os.chdir(currentDirectory) # Move back to top directory
#######################################

# run AIM post-analysis
su2.postAnalysis()

# get Lift and Drag coefficients
print ("\n==> Total Forces and Moments")

print ("--> Cl = ", su2.output.CLtot,
           "Cd = ", su2.output.CDtot)

# Get Cmx, Cmy, and Cmz coefficients
print ("--> Cmx = ", su2.output.CMXtot,
           "Cmy = ", su2.output.CMYtot,
           "Cmz = ", su2.output.CMZtot)

# Get Cx, Cy, Cz coefficients
print ("--> Cx = ", su2.output.CXtot,
           "Cy = ", su2.output.CYtot,
           "Cz = ", su2.output.CZtot)

# runtime counter end
toc = time.perf_counter()
print(f"\nTotal time {toc-tic:0.3f} seconds")
