#------------------------------------------------------------------------------#

# Import pyCAPS module
import pyCAPS
from   pyOCSM import esp

# Import os
import os

#------------------------------------------------------------------------------#

# Load geometry [.csm] file
filename = os.path.join(__file__, "..", "..", "transport.csm")
print ("\n==> Loading geometry from file \"" + filename + "\"...")
myProblem = pyCAPS.Problem(problemName = "WorkDirAvl",
                           capsFile = filename,
                           outLevel = 0)
myProblem.setOutLevel("minimal")

# Set geometry variables to enable Vortex Lattice bodies
print ("\n==> Setting Build Variables and Geometry Values...")

# Change to VLM view
myProblem.geometry.cfgpmtr.VIEW.Concept = 0
myProblem.geometry.cfgpmtr.VIEW.Vlm     = 1

# Enable lifting surfaces and controls
myProblem.geometry.cfgpmtr.COMP.wing     = 1
myProblem.geometry.cfgpmtr.COMP.fuse     = 0
myProblem.geometry.cfgpmtr.COMP.htail    = 1
myProblem.geometry.cfgpmtr.COMP.vtail    = 1

# Create AVL AIM (if it does not already exist)
if ("avl" in myProblem.analysis):
    print ("\n==> Reusing AVL aim...")
    avl = myProblem.analysis["avl"]
else:
    print ("\n==> Creating AVL aim...")
    avl = myProblem.analysis.create(aim = "avlAIM",
                                    name = "avl")

# Check the geometry
avl.geometry.view()

# Set meshing parameters for each surface
wing = {"numChord"     : 4,
        "numSpanTotal" : 80}

htail = {"numChord"     : 4,
         "numSpanTotal" : 30}

vtail = {"numChord"     : 4,
         "numSpanTotal" : 10}

# Associate the surface parameters with capsGroups defined on the geometry
avl.input.AVL_Surface = {"Wing" : wing ,
                         "Htail": htail,
                         "Vtail": vtail}

# Set up control surface deflections based on the information in the csm file
controls = {}

hinge = myProblem.geometry.despmtr.wing.hinge
for i in range(len(hinge)):
    controls["WingControl_"+str(int(hinge[i][8]))] = {"deflectionAngle": hinge[i][0]}

hinge = myProblem.geometry.despmtr.htail.hinge
for i in range(len(hinge)):
    controls["HtailControl_"+str(int(hinge[i][8]))] = {"deflectionAngle": hinge[i][0]}

hinge = myProblem.geometry.despmtr.vtail.hinge
controls["VtailControl_"+str(int(hinge[8]))] = {"deflectionAngle": hinge[0]}

avl.input.AVL_Control = controls

# Set analysis specific variables
avl.input.Mach  =  0.5
avl.input.Beta  =  0.0

# initialize _data
alpha_data = ""
cl_data    = ""
cd_data    = ""
cm_data    = ""

# loop over the angles of attack
for alpha in range(-10, 16, 2):
    print ("executing AVL for alpha =", alpha)

    # compute performance (getting outputs automagically runs AVL)
    avl.input.Alpha = alpha
    cl = avl.output["CLtot"].value
    cd = avl.output["CDtot"].value
    cm = avl.output["Cmtot"].value

    # keep track of plotter data
    alpha_data += str(alpha) + ";"
    cl_data    += str(cl)    + ";"
    cd_data    += str(cd)    + ";"
    cm_data    += str(cm)    + ";"

# plot the lift curve_data
esp.TimLoad("plotter", esp.GetEsp("pyscript"), "")
esp.TimMesg("plotter", "new|AVL predictions for Transport|alpha (deg)|CL|")
esp.TimMesg("plotter", "add|"+alpha_data+"|"+cl_data+"|k-+|")
esp.TimMesg("plotter", "show")
esp.TimQuit("plotter")

# plot the drag polar
esp.TimLoad("plotter", esp.GetEsp("pyscript"), "")
esp.TimMesg("plotter", "new|AVL predictions for Transport|CL|CD|")
esp.TimMesg("plotter", "add|"+cl_data+"|"+cd_data+"|k-+|")
esp.TimMesg("plotter", "show")
esp.TimQuit("plotter")

# plot the moment
esp.TimLoad("plotter", esp.GetEsp("pyscript"), "")
esp.TimMesg("plotter", "new|AVL predictions for Transport|CL|CM|")
esp.TimMesg("plotter", "add|"+cl_data+"|"+cm_data+"|k-+|")
esp.TimMesg("plotter", "show")
esp.TimQuit("plotter")
