###################################################################
#                                                                 #
# runModal --- run modal analysis through mASTROS                 #
#                                                                 #
#              Written by John Dannenhoffer @ Syracuse University #
#                     and Marshall Galbraith @ MIT                #
#                                                                 #
###################################################################

# import pyCAPS module
import pyCAPS

# import os module
import os
import shutil

#------------------------------------------------------------------

# load geometry [.csm] file
capsProblem = pyCAPS.Problem(problemName = "runModal",
                             capsFile    = "wing.csm",
                             outLevel    = 0)

# alias the geometry
wing = capsProblem.geometry

# reduce number of ribs for mASTROS
wing.cfgpmtr.wing.nrib1       = 1
wing.cfgpmtr.wing.nrib2       = 2
wing.cfgpmtr.wing.nrib3       = 4

#------------------------------------------------------------------

# create EGADS tess aim
tess = capsProblem.analysis.create(aim  = "egadsTessAIM",
                                   name = "tess")

# no Tess vertexes on edges (minimial mesh)
tess.input.Edge_Point_Min = 2
tess.input.Edge_Point_Max = 2

# use regularized quads
tess.input.Mesh_Elements = "Quad"

#------------------------------------------------------------------

# create astros aim
astros = capsProblem.analysis.create(aim      = "astrosAIM",
                                     name     = "astros",
                                     autoExec = False)

# link the EGADS tess Surface_Mesh to astros
astros.input["Mesh"].link(tess.output["Surface_Mesh"])

# set analysis specific variables
astros.input.Proj_Name        = "astros_modal"
astros.input.Mesh_File_Format = "Large"

# set constraints
constraint = {"dofConstraint" : 123456}
astros.input.Constraint = {"rootConstraint": constraint}

# set analysis type
eigen = { "analysisType"         : "Modal",
          "extractionMethod"     : "SINV",
          "frequencyRange"       : [0, 10],
          "numEstEigenvalue"     : 1,
          "numDesiredEigenvalue" : 10,
          "eigenNormalization"   : "MASS"}
astros.input.Analysis = {"EigenAnalysis": eigen}

# set materials
unobtainium  = {"youngModulus" : 2.2E6 ,
                "poissonRatio" : .5,
                "density"      : 7850}
madeupium    = {"materialType" : "isotropic",
                "youngModulus" : 1.2E5 ,
                "poissonRatio" : .5,
                "density"      : 7850}
astros.input.Material = {"Unobtainium": unobtainium,
                         "Madeupium"  : madeupium}

# set properties
skinShell = {"propertyType"        : "Shell",
             "material"            : "unobtainium",
             "bendingInertiaRatio" : 1.0,
             "shearMembraneRatio"  : 0, # Turn of shear - no materialShear
             "membraneThickness"   : 0.05}
ribShell  = {"propertyType"        : "Shell",
             "material"            : "unobtainium",
             "bendingInertiaRatio" : 1.0,
             "shearMembraneRatio"  : 0, # Turn of shear - no materialShear
             "membraneThickness"   : 0.1}
sparShell = {"propertyType"        : "Shell",
             "material"            : "madeupium",
             "bendingInertiaRatio" : 1.0,
             "shearMembraneRatio"  : 0, # Turn of shear - no materialShear
             "membraneThickness"   : 0.2}
astros.input.Property = {"leftWingSkin": skinShell,
                         "riteWingSkin": skinShell,
                         "wingRib"     :  ribShell,
                         "wingSpar1"   : sparShell,
                         "wingSpar2"   : sparShell}

# run AIM pre-analysis
print ("\n==> Running mASTROS pre-analysis...")
astros.preAnalysis()

####### Run mASTROS ####################
# declare ASTROS install directory
astrosInstallDir = os.environ['ESP_ROOT'] + os.sep + "bin" + os.sep

# copy files needed to run astros
files = ["ASTRO.D01", "ASTRO.IDX"]
for file in files:
    try:
        shutil.copy2(astrosInstallDir + file, astros.analysisDir + os.sep + file)
    except:
        print ('Unable to copy "' + file + '"')
        raise

# run micro-ASTROS via system call
print ("\n==> Running mASTROS...")
astros.system("mastros.exe < " + astros.input.Proj_Name +  ".dat > " + astros.input.Proj_Name + ".out")

# remove temporary files
for file in files:
    if os.path.isfile(astros.analysisDir + os.sep + file):
        os.remove(astros.analysisDir + os.sep + file)
#######################################

# run AIM post-analysis
print ("\n==> Running mASTROS post-analysis...")
astros.postAnalysis()

# get list of Eigen-frequencies
freqs = astros.output.EigenFrequency

print ("\n--> Eigen-frequencies:")
for i in range(len(freqs)):
    print ("    " + repr(i+1).ljust(2) + ": " + str(freqs[i]))
