###################################################################
#                                                                 #
# runStatic --- run static structural deformations through mASTROS#
#                                                                 #
#              Written by John Dannenhoffer @ Syracuse University #
#                     and Marshall Galbraith @ MIT                #
#                                                                 #
###################################################################

# import pyCAPS module
import pyCAPS

# import os module
import os
import shutil

#------------------------------------------------------------------

# load geometry [.csm] file
capsProblem = pyCAPS.Problem(problemName = "runStatic",
                             capsFile    = "wing.csm",
                             outLevel    = 0)

# alias the geometry
wing = capsProblem.geometry

# reduce number of ribs for mASTROS
wing.cfgpmtr.wing.nrib1       = 1
wing.cfgpmtr.wing.nrib2       = 2
wing.cfgpmtr.wing.nrib3       = 4

#------------------------------------------------------------------

# create EGADS tess aim
tess = capsProblem.analysis.create(aim = "egadsTessAIM",
                                   name = "tess")

# no Tess vertexes on edges (minimial mesh)
tess.input.Edge_Point_Max = 2

# use regularized quads
tess.input.Mesh_Elements = "Quad"

#------------------------------------------------------------------

# create astros aim
astros = capsProblem.analysis.create(aim      = "astrosAIM",
                                     name     = "astros",
                                     autoExec = False)

# link the EGADS tess Surface_Mesh to astros
astros.input["Mesh"].link(tess.output["Surface_Mesh"])

# set analysis specific variables
astros.input.Proj_Name        = "astros_modal"
astros.input.Mesh_File_Format = "Large"

# set constraints
constraint = {"dofConstraint" : 123456}
astros.input.Constraint = {"rootConstraint": constraint}

# define loads
leftLoad = {"loadType"         : "GridForce",
            "forceScaleFactor" : 1.e6,
            "directionVector"  : [0.0, 0.0, 1.0]}
riteLoad = {"loadType"         : "GridForce",
            "forceScaleFactor" : 2.e6,
            "directionVector"  : [0.0, 0.0, 1.0]}
astros.input.Load = {"leftPointLoad": leftLoad,
                     "ritePointLoad": riteLoad}

# set analysis type
astros.input.Analysis_Type = "Static"

# set materials
unobtainium  = {"youngModulus" : 2.2e6 ,
                "poissonRatio" : .5,
                "density"      : 7850}
madeupium    = {"materialType" : "isotropic",
                "youngModulus" : 1.2e5 ,
                "poissonRatio" : .5,
                "density"      : 7850}
astros.input.Material = {"Unobtainium": unobtainium,
                         "Madeupium"  : madeupium}

# set properties
skinShell = {"propertyType"        : "Shell",
             "material"            : "unobtainium",
             "bendingInertiaRatio" : 1.0,
             "shearMembraneRatio"  : 0, # Turn of shear - no materialShear
             "membraneThickness"   : 0.05}
ribShell  = {"propertyType"        : "Shell",
             "material"            : "unobtainium",
             "bendingInertiaRatio" : 1.0,
             "shearMembraneRatio"  : 0, # Turn of shear - no materialShear
             "membraneThickness"   : 0.1}
sparShell = {"propertyType"        : "Shell",
             "material"            : "madeupium",
             "bendingInertiaRatio" : 1.0,
             "shearMembraneRatio"  : 0, # Turn of shear - no materialShear
             "membraneThickness"   : 0.2}
astros.input.Property = {"leftWingSkin": skinShell,
                         "riteWingSkin": skinShell,
                         "wingRib"     :  ribShell,
                         "wingSpar1"   : sparShell,
                         "wingSpar2"   : sparShell}

# run AIM pre-analysis
print ("\n==> Running mASTROS pre-analysis...")
astros.preAnalysis()

####### Run mASTROS ####################
# declare ASTROS install directory
astrosInstallDir = os.environ['ESP_ROOT'] + os.sep + "bin" + os.sep

# copy files needed to run astros
files = ["ASTRO.D01", "ASTRO.IDX"]
for file in files:
    try:
        shutil.copy2(astrosInstallDir + file, astros.analysisDir + os.sep + file)
    except:
        print ('Unable to copy "' + file + '"')
        raise

# run micro-ASTROS via system call
print ("\n==> Running mASTROS...")
astros.system("mastros.exe < " + astros.input.Proj_Name +  ".dat > " + astros.input.Proj_Name + ".out")

# remove temporary files
for file in files:
    if os.path.isfile(astros.analysisDir + os.sep + file):
        os.remove(astros.analysisDir + os.sep + file)
#######################################

# run AIM post-analysis
print ("\n==> Running mASTROS post-analysis...")
astros.postAnalysis()

# print results
print ("\n--> Maximum displacements:")
print ("--> Tmax" , astros.output.Tmax )
print ("--> T1max", astros.output.T1max)
print ("--> T2max", astros.output.T2max)
print ("--> T3max", astros.output.T3max)
