###################################################################
#                                                                 #
# varyCamber --- MSES airfoil analysis a a function of camber     #
#                                                                 #
#              Written by John Dannenhoffer @ Syracuse University #
#                     and Marshall Galbraith @ MIT                #
#                                                                 #
###################################################################

# import pyCAPS module
import pyCAPS
from   pyOCSM import esp

import os

#------------------------------------------------------------------

# make a semi-colon-separated string from a list
def makeString(array):
    out = ""
    for i in array:
        out += str(i) + ";"
    return out

#------------------------------------------------------------------

# load geometry [.csm] file or link to model in ESP
filename = os.path.join(__file__, "..", "naca.csm")

capsProblem = pyCAPS.Problem(problemName = "varyCamber",
                             capsFile    = filename,
                             outLevel    = 1)

# setup AIM for MSES
mses = capsProblem.analysis.create(aim  = "msesAIM",
                                   name = "mses")

# set flow condition
mses.input.Alpha = 0.0
mses.input.Mach  = 0.1
mses.input.Re    = 5e6

# set meshing parameters
mses.input.GridAlpha      = 0
mses.input.Airfoil_Points = 201

# trip the flow near the leading edge to get smooth gradient
mses.input.xTransition_Upper = 0.1
mses.input.xTransition_Lower = 0.1

# use camber as the design variable
mses.input.Design_Variable = {"camber":{}}                  # new

# plot the functional and gradient
Camber    = []                                              # changed
CL        = []
CD        = []
CD_Camber = []                                              # changed

# run MSES and generate the data
for i in range(21):                                         # changed
    camber = 0.01 * (i - 10)                                # changed
    print("--> camber", camber)                             # changed

    try:
        # asking for the outputs below run MSES
        capsProblem.geometry.despmtr.camber = camber        # changed

        CL.append(      mses.output["CL"].value)
        CD.append(      mses.output["CD"].value)
        CD_Camber.append(mses.output["CD"].deriv("camber")) # changed
        Camber.append(camber)                               # changed

    except:
        print("    *** did not converge ***")

# compute finite difference derivatives
FD_Camber    = []
FD_CD_Camber = []

for i in range(len(Camber)):
    FD_Camber.append(Camber[i])
    if i == 0:
        FD_CD_Camber.append((CD[i+1]-CD[i  ])/(Camber[i+1]-Camber[i  ]))
    elif i == len(CD)-1:
        FD_CD_Camber.append((CD[i  ]-CD[i-1])/(Camber[i  ]-Camber[i-1]))
    else:
        FD_CD_Camber.append((CD[i+1]-CD[i-1])/(Camber[i+1]-Camber[i-1]))
    
# load the plotter
esp.TimLoad("plotter", esp.GetEsp("pyscript"), "")

# plot the lift curve
esp.TimMesg("plotter", "new|MSES results for NACA airfoil|camber|CL|")
esp.TimMesg("plotter", "add|"+makeString(Camber)+"|"+makeString(CL)+"|k-+|")
esp.TimMesg("plotter", "add|-.1;+.1|0;0|k:|")
esp.TimMesg("plotter", "show")

# plot the drag polar
esp.TimMesg("plotter", "new|MSES results for NACA airfoil|camber|CD|")
esp.TimMesg("plotter", "add|"+makeString(Camber)+"|"+makeString(CD)+"|k-+|")
esp.TimMesg("plotter", "show")

# plot the derivative of the lift curve
esp.TimMesg("plotter", "new|MSES results for NACA airfoil|camber|d(CD)/d(camber)|")
esp.TimMesg("plotter", "add|"+makeString(Camber)+"|"+makeString(CD_Camber)+"|k-+|")
esp.TimMesg("plotter", "add|"+makeString(FD_Camber)+"|"+makeString(FD_CD_Camber)+"|go|")
esp.TimMesg("plotter", "add|-.1;+.1|0;0|k:|")
esp.TimMesg("plotter", "show")

# exit the plotter
esp.TimQuit("plotter")

# close the capsProblem (required if you want to run another pyscript)
capsProblem.close()
