# Import pyCAPS module
import pyCAPS

# Import modules
import os
import sys
import shutil
import argparse

# Setup and read command line options. Please note that this isn't required for pyCAPS
parser = argparse.ArgumentParser(description = 'Astros Multiple Load Pytest Example',
                                 prog = 'astros_MultipleLoadCase_PyTest',
                                 formatter_class = argparse.ArgumentDefaultsHelpFormatter)

#Setup the available commandline options
parser.add_argument('-workDir', default = ["."+os.sep], nargs=1, type=str, help = 'Set working/run directory')
parser.add_argument('-noAnalysis', action='store_true', default = False, help = "Don't run analysis code")
parser.add_argument("-outLevel", default = 1, type=int, choices=[0, 1, 2], help="Set output verbosity")
args = parser.parse_args()

#------------------------------------------------------------------
# ASTROS_ROOT should be the path containing ASTRO.D01 and ASTRO.IDX
#------------------------------------------------------------------
if args.noAnalysis == False:
    try:
       ASTROS_ROOT = os.environ["ASTROS_ROOT"]
       os.putenv("PATH", ASTROS_ROOT + os.pathsep + os.getenv("PATH"))
    except KeyError:
       print("Please set the environment variable ASTROS_ROOT")
       sys.exit(1)
#------------------------------------------------------------------

# Create project name
projectName = "AstrosMultipleLoadPlate"

workDir = os.path.join(str(args.workDir[0]), projectName)

# Load CSM file
geometryScript = os.path.join("..","csmData","feaSimplePlate.csm")
capsProblem = pyCAPS.Problem(problemName=workDir,
                             capsFile=geometryScript, 
                             outLevel=args.outLevel)


# Load Astros aim
astros = capsProblem.analysis.create( aim = "astrosAIM",
                                      name = "astros",
                                      autoExec = not args.noAnalysis )

# Set project name so a mesh file is generated
astros.input.Proj_Name = projectName

# Set meshing parameters
astros.input.Edge_Point_Max = 4
astros.input.Edge_Point_Min = 4

astros.input.Quad_Mesh = True

astros.input.Tess_Params = [.25,.01,15]

# Set analysis type
astros.input.Analysis_Type = "Static"

# Set materials
madeupium    = {"materialType" : "isotropic",
                "youngModulus" : 72.0E9 ,
                "poissonRatio": 0.33,
                "density" : 2.8E3}

astros.input.Material = {"Madeupium": madeupium}

# Set properties
shell  = {"propertyType" : "Shell",
          "membraneThickness" : 0.006,
          "material"        : "madeupium",
          "bendingInertiaRatio" : 1.0, # Default
          "shearMembraneRatio"  : 5.0/6.0} # Default

astros.input.Property = {"plate": shell}

# Set constraints
constraint = {"groupName" : "plateEdge",
              "dofConstraint" : 123456}

astros.input.Constraint = {"edgeConstraint": constraint}

# Create multiple loads
numLoad = 5
loads = {}
for i in range(numLoad):
    # Set load name
    name = "load_"+ str(i)

    # Set load values
    value =  {"groupName" : "plate",
              "loadType" : "Pressure",
              "pressureForce" : 1.e6*(i+1)}

    # Add load to dict
    loads[name] = value

# Set loads
astros.input.Load = loads

# Create multiple analysis cases
analysisCases = {}
for name in loads.keys():

    # set analysis value s
    value = {"analysisType" : "Static",
             "analysisLoad" : name}

    # Add to analysis cases
    analysisCases[name] = value

# Set analysis
astros.input.Analysis = analysisCases

if args.noAnalysis:
    astros.preAnalysis()
    exit()

# Run AIM 
astros.runAnalysis()
