# Import pyCAPS class file
import pyCAPS

# Import os module
import os
import argparse

# CBAERO_ROOT should be the path containing tps_libs (e.g. /path/to/cbaero_distribution.August.2022)
try:
   TPSSIZER_LIB = os.environ["TPSSIZER_LIB"]
except KeyError:
   print("Please set the environment variable TPSSIZER_LIB")
   exit(1)

# Setup and read command line options. Please note that this isn't required for pyCAPS
parser = argparse.ArgumentParser(description = 'CBAero TPS Pytest Example',
                                 prog = 'cbaero_PyTest',
                                 formatter_class = argparse.ArgumentDefaultsHelpFormatter)

#Setup the available commandline options
parser.add_argument('-workDir', default = "./", nargs=1, type=str, help = 'Set working/run directory')
parser.add_argument("-outLevel", default = 1, type=int, choices=[0, 1, 2], help="Set output verbosity")
args = parser.parse_args()

# Create working directory variable
projectName = "CBAeroTPSAnalysisTest"
workDir = os.path.join(str(args.workDir[0]), projectName)

# Load CSM file
geometryScript = os.path.join("..","csmData","cfdX43a.csm")
myProblem = pyCAPS.Problem(problemName=workDir,
                           capsFile=geometryScript,
                           outLevel=args.outLevel)

# Turn of the farfield domain
myProblem.geometry.cfgpmtr.makeBoundBox = 0

# Load egadsTess aim to create surface mesh
meshAIM = myProblem.analysis.create(aim = "egadsTessAIM",
                                    name = "egadsTess" )

# Set meshing parameters
meshAIM.input.Tess_Params = [0.025, 0.01, 10.0]

# Load CBAero aim
cbaero = myProblem.analysis.create(aim = "cbaeroAIM",
                                   name = "cbaero")

deg = pyCAPS.Unit("degree")
psi = pyCAPS.Unit("psi")
second = pyCAPS.Unit("second")
kelvin = pyCAPS.Unit("kelvin")

# Link mesh
cbaero.input["Surface_Mesh"].link(meshAIM.output["Surface_Mesh"])

# Set number of cases to run simultaneously
cbaero.input.NumParallelCase = 3;

# Set number of threads to use to solve each case
cbaero.input.NumThreadPerCase = 2;

# Set AoA number
cbaero.input.Alpha = [1.0, 3.0, 5.0] * deg

# Set Mach number
cbaero.input.Mach = [2.0, 2.5, 4.0]

# Set Dynamic pressure
cbaero.input.Dynamic_Pressure = 10 * psi

# Set Mangler setting
cbaero.input.Mangler_Setting = "2D"

# Set Flow Type
cbaero.input.Flow_Type = "Turbulent" # Needs to be turbulent or laminar for cbtps and tpssizer to work

# Set aero surfaces
cbaero.input.Aero_Surface = {"fuselage_base":"Base",
                             "horizontalCS" :"Wing",
                             "verticalCS"   :"Wing",
                             "inlet"        :"Body",
                             "hollowInlet"  :"Body"}

# Set materials
cbaero.input.Material_Group = {"mat1":{"surfaceType":"Fully Catalytic", "emissivity":0.8,"groupName":["fuselage", "fuselage_base"]},
                               "mat2":{"surfaceType":"RCG"            , "emissivity":0.6,"groupName":["horizontalCS", "verticalCS"]},
                               "mat3":{"surfaceType":"8"              , "emissivity":0.7,"groupName":"inlet"},
                               "mat4":{"surfaceType":"Fully Catalytic", "emissivity":0.5,"groupName":"hollowInlet"}}

# Trajectory files written in this example, but could be referenced elsewhere
filename = os.path.join(cbaero.analysisDir, "hl20.trj")
with open(filename, "w") as f:
    f.write( \
"""NOMINAL
  280
  2.  24.6341705  8.53497186E-05  30.1782799
  4.  24.6804695  8.95557678E-05  29.0947304
  6.  24.7262802  9.38090743E-05  28.0348396
  8.  24.7715607  9.87240128E-05  26.9997101
  10.  24.8162708  0.00010401701  25.9904709
  12.  24.8603497  0.000109357279  25.0082207
  14.  24.9037609  0.000115075614  24.0540791
  16.  24.9464607  0.00012178639  23.1291695
  18.  24.98839  0.000128591681  22.2346001
  20.  25.0294991  0.00013553875  21.3714809
  22.  25.0697403  0.000143714555  20.5054398
  24.  25.1090603  0.000152504726  19.5946407
  26.  25.1473999  0.000161767486  18.6648293
  28.  25.1847095  0.000172117201  17.7426796
  30.  25.2209301  0.000183081283  16.8547993
  32.  25.2560005  0.000194848762  16.0278492
  34.  25.2898693  0.000207844991  15.2884598
  36.  25.3224697  0.000221786395  14.6632795
  38.  25.3537502  0.000236672975  14.1789503
  40.  25.3836308  0.000253071834  13.8621101
  42.  25.4120598  0.000270841207  13.6894398
  44.  25.4389706  0.000289981108  13.5348501
  46.  25.4642906  0.000310633273  13.3909597
  48.  25.4879608  0.000333128555  13.25842
  50.  25.5098896  0.000357797719  13.1378603
  52.  25.5300293  0.000384451792  13.0299101
  54.  25.5482998  0.000413279748  12.9352198
  56.  25.5646305  0.000444470701  12.8544197
  58.  25.5789299  0.000478922477  12.7881498
  60.  25.5911407  0.000516304339  12.7370396
  62.  25.60116  0.000556994288  12.7017403
  64.  25.6089306  0.000601086998  12.6828804
  66.  25.6143494  0.000649054826  12.6805696
  68.  25.6150894  0.000701465062  12.6902304
  70.  25.6171093  0.000758742914  12.7106504
  72.  25.6191502  0.00082169188  12.7417803
  74.  25.6212006  0.000890784489  12.7835903
  76.  25.62327  0.00096639886  12.83605
  78.  25.6253605  0.00104919658  12.8991203
  80.  25.6274605  0.00114002835  12.9727602
  82.  25.62957  0.00124069001  13.0569601
  84.  25.6316891  0.00135023624  13.15166
  86.  25.6338291  0.00147164462  13.2568302
  88.  25.6359692  0.0016049149  13.3724403
  90.  25.6110191  0.00174702273  13.4984598
  92.  25.5471592  0.00189806253  13.7974596
  94.  25.4831009  0.00206389418  14.5676003
  96.  25.4188309  0.00224451814  15.7570601
  98.  25.3543491  0.00244191871  17.3038406
  100.  25.2896099  0.00265902653  19.1459408
  102.  25.2246094  0.00289494335  21.2213707
  104.  25.1592999  0.00315354462  23.4681492
  106.  25.0936394  0.00343681453  25.8242607
  108.  25.0275803  0.0037457468  28.2277298
  110.  24.9610691  0.0040833177  30.6165504
  112.  24.8940506  0.00445052003  33.0717812
  114.  24.8263302  0.00485222088  36.5104713
  116.  24.7575397  0.00528747635  40.5626717
  118.  24.6873398  0.00576129463  44.5613213
  120.  24.6154499  0.0062745274  47.8393593
  122.  24.5418301  0.00683317566  49.7297211
  124.  24.4668694  0.00744120963  49.9500008
  126.  24.3908291  0.00810642727  49.9500008
  128.  24.31353  0.00883388426  49.9500008
  130.  24.2347603  0.00963137951  49.9500008
  132.  24.1543293  0.0105118146  49.9500008
  134.  24.07201  0.0114791114  49.9500008
  136.  23.9875908  0.0125352079  49.9500008
  138.  23.9008904  0.0137196593  49.9500008
  140.  23.8117294  0.0150126657  49.9500008
  142.  23.7012405  0.0164241008  49.9500008
  144.  23.5795708  0.0179638937  49.9500008
  146.  23.4555092  0.0196517017  49.9500008
  148.  23.3291302  0.0214875713  49.9500008
  150.  23.2005005  0.0234912112  49.9500008
  152.  23.0697098  0.0256626662  49.9500008
  154.  22.9368896  0.028001938  49.9500008
  156.  22.8022709  0.0304991025  49.9500008
  158.  22.6661091  0.0331443287  49.9500008
  160.  22.5286694  0.0359277427  49.9500008
  162.  22.3902092  0.0388098769  49.9500008
  164.  22.2509003  0.0417906903  49.9500008
  166.  22.1108799  0.0448109657  49.9500008
  168.  21.9702091  0.0478707477  49.9500008
  170.  21.8289795  0.0509206541  49.9500008
  172.  21.6872597  0.0539310984  49.9500008
  174.  21.5451508  0.0568823256  49.9500008
  176.  21.40271  0.0597348325  49.9500008
  178.  21.2600498  0.0624688566  49.9500008
  180.  21.1172199  0.0650548637  49.9500008
  182.  20.9742699  0.067482993  49.9500008
  184.  20.8311501  0.0697432458  49.9500008
  186.  20.6875401  0.0718554854  49.9500008
  188.  20.5429192  0.0738690421  49.9500008
  190.  20.3965397  0.0758430958  49.9500008
  192.  20.24753  0.0778467357  49.9500008
  194.  20.0949802  0.0799491033  49.9500008
  196.  19.9380493  0.0822291598  49.9500008
  198.  19.7760792  0.0847263262  49.9500008
  200.  19.6087093  0.0874702707  49.9500008
  202.  19.43573  0.0904510841  49.9500008
  204.  19.2571106  0.0936786383  49.9500008
  206.  19.0729408  0.0971036404  49.9500008
  208.  18.8833904  0.100676656  49.9500008
  210.  18.6887493  0.104427367  49.9500008
  212.  18.4893398  0.108178072  49.9500008
  214.  18.2854996  0.111928731  49.9500008
  216.  18.0774899  0.11558076  49.9500008
  218.  17.86549  0.119232744  49.9500008
  220.  17.6496391  0.12278606  49.9500008
  222.  17.4299698  0.126339331  49.9500008
  224.  17.2065296  0.129695222  49.9500008
  226.  16.9792995  0.133051127  49.9500008
  228.  16.78195  0.136900514  49.9500008
  230.  16.5803795  0.140749916  49.9500008
  232.  16.3731194  0.144599333  49.9500008
  234.  16.16012  0.148547441  49.9500008
  236.  15.94135  0.152495518  49.9404411
  238.  15.7172298  0.156443626  49.8589706
  240.  15.4888401  0.159799531  49.7036896
  242.  15.2580099  0.161872253  49.4859314
  244.  15.0275698  0.161970973  49.2170601
  246.  14.8009501  0.159996927  48.9084282
  248.  14.5810604  0.156739742  48.5713997
  250.  14.3692703  0.153383836  48.2173309
  252.  14.1651402  0.151113704  47.8575592
  254.  13.9716501  0.150817588  47.5034409
  256.  13.7982597  0.15308775  47.1663017
  258.  13.6319704  0.15782547  46.8416405
  260.  13.4688597  0.165030763  46.5158501
  262.  13.3039799  0.174506187  46.1824188
  264.  13.1314201  0.185067385  45.8348694
  266.  12.9457397  0.195036337  45.4666901
  268.  12.7447796  0.202142909  45.0713882
  270.  12.5317297  0.205794945  44.6424789
  272.  12.3138905  0.206781954  44.1734695
  274.  12.0989304  0.207078069  43.6578484
  276.  11.8916597  0.208756  43.0891304
  278.  11.6929903  0.213098913  42.4605904
  280.  11.50214  0.220501602  41.5924301
  282.  11.3196001  0.231556296  40.3815689
  284.  11.1459398  0.246559128  38.8993492
  286.  10.9815903  0.265806079  37.2170715
  288.  10.8257303  0.289790839  35.4060593
  290.  10.67735  0.318612009  33.5376282
  292.  10.5357704  0.352368295  31.6830997
  294.  10.4012203  0.390862286  29.9137897
  296.  10.2715597  0.433600545  28.3010101
  298.  10.1429396  0.479398519  26.9160805
  300.  10.0120897  0.526775777  25.8303204
  302.  9.87481976  0.573758245  25.0845604
  304.  9.70715046  0.615706861  24.4241695
  306.  9.53141975  0.65341121  23.7819595
  308.  9.3544302  0.686772704  23.1655998
  310.  9.17776966  0.715495169  22.5827694
  312.  9.00269985  0.739479899  22.0411491
  314.  8.83016968  0.758924365  21.5484009
  316.  8.66069031  0.774321914  21.1122208
  318.  8.49427032  0.78596884  20.7402706
  320.  8.33055973  0.794555962  20.4402294
  322.  8.16893005  0.800182045  20.2197704
  324.  8.00852966  0.803241789  20.0865803
  326.  7.84827995  0.803636611  20.0478897
  328.  7.68723011  0.801366448  20.0701294
  330.  7.52513981  0.796135187  20.1269398
  332.  7.36194992  0.787449419  20.2161903
  334.  7.19776011  0.774716735  20.3357601
  336.  7.03281021  0.757246375  20.4835205
  338.  6.86745024  0.734643459  20.6573391
  340.  6.70215988  0.706513286  20.8550892
  342.  6.53782988  0.673250496  21.0746498
  344.  6.37552977  0.635348678  21.3138809
  346.  6.21636009  0.593794882  21.5706596
  348.  6.06141996  0.54987222  21.8428707
  350.  5.91146994  0.504863858  22.1283607
  352.  5.76640987  0.459954113  22.4349194
  354.  5.62572002  0.416327596  22.9652309
  356.  5.4870801  0.374280304  23.7606297
  358.  5.34896994  0.334404439  24.7685795
  360.  5.21084976  0.297094852  25.9365101
  362.  5.07287979  0.262647629  27.2118797
  364.  4.93567991  0.231260195  28.5421391
  366.  4.80011988  0.203129962  29.8747292
  368.  4.66810989  0.178059503  31.1570892
  370.  4.54203987  0.156147495  32.336689
  372.  4.4228301  0.1371966  33.3609581
  374.  4.3111701  0.120812006  34.1773491
  376.  4.20760012  0.106697492  34.7333107
  378.  4.10521984  0.094339937  34.9762993
  380.  4.01172018  0.0838478729  34.9760284
  382.  3.92655993  0.0749448985  34.9510307
  384.  3.84833002  0.0672954097  34.9104004
  386.  3.77612996  0.0606922507  34.8571701
  388.  3.70918989  0.0549378544  34.7943497
  390.  3.64685988  0.0498843081  34.7249718
  392.  3.58866  0.0454327986  34.65205
  394.  3.53422999  0.0415143184  34.5786018
  396.  3.48328996  0.0380597375  34.5076599
  398.  3.43565011  0.0350197069  34.4422417
  400.  3.39119005  0.0323744342  34.3853607
  402.  3.34977007  0.0300647914  34.3400497
  404.  3.31125998  0.0280611534  34.3093185
  406.  3.2755301  0.0263338368  34.2961998
  408.  3.24238992  0.0248434301  34.3229294
  410.  3.21162009  0.0235701799  34.4190292
  412.  3.18298006  0.0224646982  34.5742798
  414.  3.15627003  0.0215171557  34.7777405
  416.  3.13126993  0.0206979197  35.0184402
  418.  3.10782003  0.0199971646  35.28545
  420.  3.08578992  0.0193950851  35.5677986
  422.  3.06506991  0.0188818052  35.8545609
  424.  3.04561996  0.0184573736  36.1347694
  426.  3.02741003  0.0181020796  36.3974686
  428.  3.0104599  0.0178257097  36.6317215
  430.  2.99479008  0.0176381376  36.8265686
  432.  2.98042011  0.0175197069  36.9710617
  434.  2.96738005  0.0174901243  37.0542488
  436.  2.95567012  0.0175394602  37.0708694
  438.  2.94525003  0.0176677685  37.0724411
  440.  2.93604994  0.017884925  37.0739098
  442.  2.92796993  0.0181810018  37.0752716
  444.  2.92092991  0.0185659733  37.0765114
  446.  2.91481996  0.0190397445  37.0776405
  448.  2.90953994  0.0196023155  37.0786514
  450.  2.90497994  0.0202735346  37.0795593
  452.  2.90101004  0.0210433844  37.080349
  454.  2.89749002  0.021921834  37.0810204
  456.  2.89427996  0.022908885  37.0815887
  458.  2.89122009  0.0240241978  37.0820389
  460.  2.88812995  0.0252678636  37.0823784
  462.  2.88483  0.0266497172  37.0826111
  464.  2.88109994  0.0281697083  37.0827293
  466.  2.87674999  0.02982793  37.0672112
  468.  2.87161994  0.0316440463  36.9372482
  470.  2.86562991  0.0336082205  36.6909485
  472.  2.85868001  0.0357303396  36.3471413
  474.  2.8506999  0.0380005203  35.9246712
  476.  2.84157991  0.04042859  35.442379
  478.  2.83123994  0.0429948457  34.9191093
  480.  2.81954002  0.0456992909  34.3736801
  482.  2.80439997  0.0484531187  33.8249588
  484.  2.78469992  0.0511674397  33.2917786
  486.  2.76348996  0.0539310984  32.7929688
  488.  2.74063993  0.056714505  32.3473816
  490.  2.71601009  0.0594683364  31.9738407
  492.  2.68941998  0.0621530227  31.6912098
  494.  2.66069007  0.0647094548  31.5183105
  496.  2.62959003  0.0670980141  31.4720898
  498.  2.59596992  0.0692398846  31.4907894
  500.  2.55980992  0.0710856393  31.5348206
  502.  2.52120996  0.072585918  31.6013794
  504.  2.48029995  0.0737209842  31.6876793
  506.  2.43726993  0.0744414926  31.7909203
  508.  2.39236999  0.0747672021  31.9083099
  510.  2.34589005  0.0747079924  32.0370598
  512.  2.29816008  0.0742934272  32.1743698
  514.  2.24954009  0.0735729188  32.31744
  516.  2.20040011  0.0726155043  32.4634895
  518.  2.15109992  0.0714705586  32.6097183
  520.  2.10198998  0.0702268928  32.7533417
  522.  2.05339003  0.0689338893  32.8915405
  524.  2.00556993  0.0676606372  33.0215492
  526.  1.95872998  0.0664465949  33.1405602
  528.  1.91299999  0.065331243  33.2492714
  530.  1.86843002  0.0643343553  33.3612404
  532.  1.82492995  0.0634559095  33.4771385
  534.  1.78249002  0.0627156422  33.5958099
  536.  1.74112999  0.0621234402  33.7160683
  538.  1.70085001  0.0616792552  33.83675
  540.  1.66163003  0.0614028834  33.9566689
  542.  1.62345004  0.0612844527  34.0746613
  544.  1.58629  0.0613436699  34.1895485
  546.  1.55007994  0.061590407  34.3001709
  548.  1.51477003  0.0620148405  34.4053383
  550.  1.48029006  0.062626794  34.503891
  552.  1.44652998  0.0634262785  34.5946503
  554.  1.41340005  0.0644232035  34.6764412
  556.  1.38079  0.0656076074  34.7480888
  558.  1.34856999  0.0669598281  34.8084297
  560.  1.34856999  0.0669598281  34.8084297
""")

filename = os.path.join(cbaero.analysisDir, "hl20_2.trj")
with open(filename, "w") as f:
    f.write( \
"""ABORT
  280
  2.  24.6341705  8.53497186E-05  30.1782799
  4.  24.6804695  8.95557678E-05  29.0947304
  6.  24.7262802  9.38090743E-05  28.0348396
  8.  24.7715607  9.87240128E-05  26.9997101
  10.  24.8162708  0.00010401701  25.9904709
  12.  24.8603497  0.000109357279  25.0082207
  14.  24.9037609  0.000115075614  24.0540791
  16.  24.9464607  0.00012178639  23.1291695
  18.  24.98839  0.000128591681  22.2346001
  20.  25.0294991  0.00013553875  21.3714809
  22.  25.0697403  0.000143714555  20.5054398
  24.  25.1090603  0.000152504726  19.5946407
  26.  25.1473999  0.000161767486  18.6648293
  28.  25.1847095  0.000172117201  17.7426796
  30.  25.2209301  0.000183081283  16.8547993
  32.  25.2560005  0.000194848762  16.0278492
  34.  25.2898693  0.000207844991  15.2884598
  36.  25.3224697  0.000221786395  14.6632795
  38.  25.3537502  0.000236672975  14.1789503
  40.  25.3836308  0.000253071834  13.8621101
  42.  25.4120598  0.000270841207  13.6894398
  44.  25.4389706  0.000289981108  13.5348501
  46.  25.4642906  0.000310633273  13.3909597
  48.  25.4879608  0.000333128555  13.25842
  50.  25.5098896  0.000357797719  13.1378603
  52.  25.5300293  0.000384451792  13.0299101
  54.  25.5482998  0.000413279748  12.9352198
  56.  25.5646305  0.000444470701  12.8544197
  58.  25.5789299  0.000478922477  12.7881498
  60.  25.5911407  0.000516304339  12.7370396
  62.  25.60116  0.000556994288  12.7017403
  64.  25.6089306  0.000601086998  12.6828804
  66.  25.6143494  0.000649054826  12.6805696
  68.  25.6150894  0.000701465062  12.6902304
  70.  25.6171093  0.000758742914  12.7106504
  72.  25.6191502  0.00082169188  12.7417803
  74.  25.6212006  0.000890784489  12.7835903
  76.  25.62327  0.00096639886  12.83605
  78.  25.6253605  0.00104919658  12.8991203
  80.  25.6274605  0.00114002835  12.9727602
  82.  25.62957  0.00124069001  13.0569601
  84.  25.6316891  0.00135023624  13.15166
  86.  25.6338291  0.00147164462  13.2568302
  88.  25.6359692  0.0016049149  13.3724403
  90.  25.6110191  0.00174702273  13.4984598
  92.  25.5471592  0.00189806253  13.7974596
  94.  25.4831009  0.00206389418  14.5676003
  96.  25.4188309  0.00224451814  15.7570601
  98.  25.3543491  0.00244191871  17.3038406
  100.  25.2896099  0.00265902653  19.1459408
  102.  25.2246094  0.00289494335  21.2213707
  104.  25.1592999  0.00315354462  23.4681492
  106.  25.0936394  0.00343681453  25.8242607
  108.  25.0275803  0.0037457468  28.2277298
  110.  24.9610691  0.0040833177  30.6165504
  112.  24.8940506  0.00445052003  33.0717812
  114.  24.8263302  0.00485222088  36.5104713
  116.  24.7575397  0.00528747635  40.5626717
  118.  24.6873398  0.00576129463  44.5613213
  120.  24.6154499  0.0062745274  47.8393593
  122.  24.5418301  0.00683317566  49.7297211
  124.  24.4668694  0.00744120963  49.9500008
  126.  24.3908291  0.00810642727  49.9500008
  128.  24.31353  0.00883388426  49.9500008
  130.  24.2347603  0.00963137951  49.9500008
  132.  24.1543293  0.0105118146  49.9500008
  134.  24.07201  0.0114791114  49.9500008
  136.  23.9875908  0.0125352079  49.9500008
  138.  23.9008904  0.0137196593  49.9500008
  140.  23.8117294  0.0150126657  49.9500008
  142.  23.7012405  0.0164241008  49.9500008
  144.  23.5795708  0.0179638937  49.9500008
  146.  23.4555092  0.0196517017  49.9500008
  148.  23.3291302  0.0214875713  49.9500008
  150.  23.2005005  0.0234912112  49.9500008
  152.  23.0697098  0.0256626662  49.9500008
  154.  22.9368896  0.028001938  49.9500008
  156.  22.8022709  0.0304991025  49.9500008
  158.  22.6661091  0.0331443287  49.9500008
  160.  22.5286694  0.0359277427  49.9500008
  162.  22.3902092  0.0388098769  49.9500008
  164.  22.2509003  0.0417906903  49.9500008
  166.  22.1108799  0.0448109657  49.9500008
  168.  21.9702091  0.0478707477  49.9500008
  170.  21.8289795  0.0509206541  49.9500008
  172.  21.6872597  0.0539310984  49.9500008
  174.  21.5451508  0.0568823256  49.9500008
  176.  21.40271  0.0597348325  49.9500008
  178.  21.2600498  0.0624688566  49.9500008
  180.  21.1172199  0.0650548637  49.9500008
  182.  20.9742699  0.067482993  49.9500008
  184.  20.8311501  0.0697432458  49.9500008
  186.  20.6875401  0.0718554854  49.9500008
  188.  20.5429192  0.0738690421  49.9500008
  190.  20.3965397  0.0758430958  49.9500008
  192.  20.24753  0.0778467357  49.9500008
  194.  20.0949802  0.0799491033  49.9500008
  196.  19.9380493  0.0822291598  49.9500008
  198.  19.7760792  0.0847263262  49.9500008
  200.  19.6087093  0.0874702707  49.9500008
  202.  19.43573  0.0904510841  49.9500008
  204.  19.2571106  0.0936786383  49.9500008
  206.  19.0729408  0.0971036404  49.9500008
  208.  18.8833904  0.100676656  49.9500008
  210.  18.6887493  0.104427367  49.9500008
  212.  18.4893398  0.108178072  49.9500008
  214.  18.2854996  0.111928731  49.9500008
  216.  18.0774899  0.11558076  49.9500008
  218.  17.86549  0.119232744  49.9500008
  220.  17.6496391  0.12278606  49.9500008
  222.  17.4299698  0.126339331  49.9500008
  224.  17.2065296  0.129695222  49.9500008
  226.  16.9792995  0.133051127  49.9500008
  228.  16.78195  0.136900514  49.9500008
  230.  16.5803795  0.140749916  49.9500008
  232.  16.3731194  0.144599333  49.9500008
  234.  16.16012  0.148547441  49.9500008
  236.  15.94135  0.152495518  49.9404411
  238.  15.7172298  0.156443626  49.8589706
  240.  15.4888401  0.159799531  49.7036896
  242.  15.2580099  0.161872253  49.4859314
  244.  15.0275698  0.161970973  49.2170601
  246.  14.8009501  0.159996927  48.9084282
  248.  14.5810604  0.156739742  48.5713997
  250.  14.3692703  0.153383836  48.2173309
  252.  14.1651402  0.151113704  47.8575592
  254.  13.9716501  0.150817588  47.5034409
  256.  13.7982597  0.15308775  47.1663017
  258.  13.6319704  0.15782547  46.8416405
  260.  13.4688597  0.165030763  46.5158501
  262.  13.3039799  0.174506187  46.1824188
  264.  13.1314201  0.185067385  45.8348694
  266.  12.9457397  0.195036337  45.4666901
  268.  12.7447796  0.202142909  45.0713882
  270.  12.5317297  0.205794945  44.6424789
  272.  12.3138905  0.206781954  44.1734695
  274.  12.0989304  0.207078069  43.6578484
  276.  11.8916597  0.208756  43.0891304
  278.  11.6929903  0.213098913  42.4605904
  280.  11.50214  0.220501602  41.5924301
  282.  11.3196001  0.231556296  40.3815689
  284.  11.1459398  0.246559128  38.8993492
  286.  10.9815903  0.265806079  37.2170715
  288.  10.8257303  0.289790839  35.4060593
  290.  10.67735  0.318612009  33.5376282
  292.  10.5357704  0.352368295  31.6830997
  294.  10.4012203  0.390862286  29.9137897
  296.  10.2715597  0.433600545  28.3010101
  298.  10.1429396  0.479398519  26.9160805
  300.  10.0120897  0.526775777  25.8303204
  302.  9.87481976  0.573758245  25.0845604
  304.  9.70715046  0.615706861  24.4241695
  306.  9.53141975  0.65341121  23.7819595
  308.  9.3544302  0.686772704  23.1655998
  310.  9.17776966  0.715495169  22.5827694
  312.  9.00269985  0.739479899  22.0411491
  314.  8.83016968  0.758924365  21.5484009
  316.  8.66069031  0.774321914  21.1122208
  318.  8.49427032  0.78596884  20.7402706
  320.  8.33055973  0.794555962  20.4402294
  322.  8.16893005  0.800182045  20.2197704
  324.  8.00852966  0.803241789  20.0865803
  326.  7.84827995  0.803636611  20.0478897
  328.  7.68723011  0.801366448  20.0701294
  330.  7.52513981  0.796135187  20.1269398
  332.  7.36194992  0.787449419  20.2161903
  334.  7.19776011  0.774716735  20.3357601
  336.  7.03281021  0.757246375  20.4835205
  338.  6.86745024  0.734643459  20.6573391
  340.  6.70215988  0.706513286  20.8550892
  342.  6.53782988  0.673250496  21.0746498
  344.  6.37552977  0.635348678  21.3138809
  346.  6.21636009  0.593794882  21.5706596
  348.  6.06141996  0.54987222  21.8428707
  350.  5.91146994  0.504863858  22.1283607
  352.  5.76640987  0.459954113  22.4349194
  354.  5.62572002  0.416327596  22.9652309
  356.  5.4870801  0.374280304  23.7606297
  358.  5.34896994  0.334404439  24.7685795
  360.  5.21084976  0.297094852  25.9365101
  362.  5.07287979  0.262647629  27.2118797
  364.  4.93567991  0.231260195  28.5421391
  366.  4.80011988  0.203129962  29.8747292
  368.  4.66810989  0.178059503  31.1570892
  370.  4.54203987  0.156147495  32.336689
  372.  4.4228301  0.1371966  33.3609581
  374.  4.3111701  0.120812006  34.1773491
  376.  4.20760012  0.106697492  34.7333107
  378.  4.10521984  0.094339937  34.9762993
  380.  4.01172018  0.0838478729  34.9760284
  382.  3.92655993  0.0749448985  34.9510307
  384.  3.84833002  0.0672954097  34.9104004
  386.  3.77612996  0.0606922507  34.8571701
  388.  3.70918989  0.0549378544  34.7943497
  390.  3.64685988  0.0498843081  34.7249718
  392.  3.58866  0.0454327986  34.65205
  394.  3.53422999  0.0415143184  34.5786018
  396.  3.48328996  0.0380597375  34.5076599
  398.  3.43565011  0.0350197069  34.4422417
  400.  3.39119005  0.0323744342  34.3853607
  402.  3.34977007  0.0300647914  34.3400497
  404.  3.31125998  0.0280611534  34.3093185
  406.  3.2755301  0.0263338368  34.2961998
  408.  3.24238992  0.0248434301  34.3229294
  410.  3.21162009  0.0235701799  34.4190292
  412.  3.18298006  0.0224646982  34.5742798
  414.  3.15627003  0.0215171557  34.7777405
  416.  3.13126993  0.0206979197  35.0184402
  418.  3.10782003  0.0199971646  35.28545
  420.  3.08578992  0.0193950851  35.5677986
  422.  3.06506991  0.0188818052  35.8545609
  424.  3.04561996  0.0184573736  36.1347694
  426.  3.02741003  0.0181020796  36.3974686
  428.  3.0104599  0.0178257097  36.6317215
  430.  2.99479008  0.0176381376  36.8265686
  432.  2.98042011  0.0175197069  36.9710617
  434.  2.96738005  0.0174901243  37.0542488
  436.  2.95567012  0.0175394602  37.0708694
  438.  2.94525003  0.0176677685  37.0724411
  440.  2.93604994  0.017884925  37.0739098
  442.  2.92796993  0.0181810018  37.0752716
  444.  2.92092991  0.0185659733  37.0765114
  446.  2.91481996  0.0190397445  37.0776405
  448.  2.90953994  0.0196023155  37.0786514
  450.  2.90497994  0.0202735346  37.0795593
  452.  2.90101004  0.0210433844  37.080349
  454.  2.89749002  0.021921834  37.0810204
  456.  2.89427996  0.022908885  37.0815887
  458.  2.89122009  0.0240241978  37.0820389
  460.  2.88812995  0.0252678636  37.0823784
  462.  2.88483  0.0266497172  37.0826111
  464.  2.88109994  0.0281697083  37.0827293
  466.  2.87674999  0.02982793  37.0672112
  468.  2.87161994  0.0316440463  36.9372482
  470.  2.86562991  0.0336082205  36.6909485
  472.  2.85868001  0.0357303396  36.3471413
  474.  2.8506999  0.0380005203  35.9246712
  476.  2.84157991  0.04042859  35.442379
  478.  2.83123994  0.0429948457  34.9191093
  480.  2.81954002  0.0456992909  34.3736801
  482.  2.80439997  0.0484531187  33.8249588
  484.  2.78469992  0.0511674397  33.2917786
  486.  2.76348996  0.0539310984  32.7929688
  488.  2.74063993  0.056714505  32.3473816
  490.  2.71601009  0.0594683364  31.9738407
  492.  2.68941998  0.0621530227  31.6912098
  494.  2.66069007  0.0647094548  31.5183105
  496.  2.62959003  0.0670980141  31.4720898
  498.  2.59596992  0.0692398846  31.4907894
  500.  2.55980992  0.0710856393  31.5348206
  502.  2.52120996  0.072585918  31.6013794
  504.  2.48029995  0.0737209842  31.6876793
  506.  2.43726993  0.0744414926  31.7909203
  508.  2.39236999  0.0747672021  31.9083099
  510.  2.34589005  0.0747079924  32.0370598
  512.  2.29816008  0.0742934272  32.1743698
  514.  2.24954009  0.0735729188  32.31744
  516.  2.20040011  0.0726155043  32.4634895
  518.  2.15109992  0.0714705586  32.6097183
  520.  2.10198998  0.0702268928  32.7533417
  522.  2.05339003  0.0689338893  32.8915405
  524.  2.00556993  0.0676606372  33.0215492
  526.  1.95872998  0.0664465949  33.1405602
  528.  1.91299999  0.065331243  33.2492714
  530.  1.86843002  0.0643343553  33.3612404
  532.  1.82492995  0.0634559095  33.4771385
  534.  1.78249002  0.0627156422  33.5958099
  536.  1.74112999  0.0621234402  33.7160683
  538.  1.70085001  0.0616792552  33.83675
  540.  1.66163003  0.0614028834  33.9566689
  542.  1.62345004  0.0612844527  34.0746613
  544.  1.58629  0.0613436699  34.1895485
  546.  1.55007994  0.061590407  34.3001709
  548.  1.51477003  0.0620148405  34.4053383
  550.  1.48029006  0.062626794  34.503891
  552.  1.44652998  0.0634262785  34.5946503
  554.  1.41340005  0.0644232035  34.6764412
  556.  1.38079  0.0656076074  34.7480888
  558.  1.34856999  0.0669598281  34.8084297
  560.  1.34856999  0.0669598281  34.8084297
""")

cbaero.input.Trajectory = ["hl20","hl20_2"]

# # Write enthalpy table instead of referencing the TPSSIZER example
# filename = os.path.join(cbaero.analysisDir, "air_enthalpy_298_15.inp")
# with open(filename, "w") as f:
#     f.write( \
# """
# C Temperature (Kelvin) vs Enthalpy (J/kg) array
# C for air
# C Reference temperature is 536.67 R = 298.15 K
# C Gordon - McBride
#           0.0, -2.9960E+05
#           55.56, -2.4448E+05
#           277.78, -2.1663E+04
#           555.56, 2.6093E+05
#           833.33, 5.5910E+05
#           1111.11, 8.7420E+05
#           1388.89, 1.2024E+06
#           1666.67, 1.5394E+06
#           1944.44, 1.8830E+06
#           2222.22, 2.2316E+06
# """)

cbaero.input.TPSin = {
"SoakOutTime": 1.0 * second,
"MarginOnHeatLoads_Multiplier": 1.0,
"MarginOnRecession": 1.0,
"InitialTemperature": 277.59 * kelvin,
"MarginOnInsulation": 1.0,
"BlowingFactor": 0.5,
"RunWhichRSSedEnvCase0123": 2,
"EnthalpyTableFileName": "air_enthalpy_298_15.inp",
"TimeToOption3": 1000.0 * second,
    }

tps_libs = TPSSIZER_LIB + os.sep

cbaero.input.TPS_Group = {"fuselage":{"StackUpListFileName":tps_libs + "Stacklists"+os.sep+"example_HL20",
                                      "StructuresStackUpListFileName":tps_libs+"Structures"+os.sep+"example_Al_HC",
                                      "SplitLineMarginsFileName":tps_libs+"Margins"+os.sep+"Example",
                                      "EnvironmentMarginsFileName":tps_libs+"Margins"+os.sep+"NoMargins",
                                      "BumpFactorFileName":tps_libs+"BumpModels"+os.sep+"None",
                                      "UseOnlyOneMaterial":1,
                                      "UseHeatLoadSubZones":1,
                                      "NumberOfHeatLoadSubZones":2},
                          "tps2":{"StackUpListFileName":tps_libs + "Stacklists"+os.sep+"example_HL20",
                                  "StructuresStackUpListFileName":tps_libs+"Structures"+os.sep+"example_GRBMI",
                                  "SplitLineMarginsFileName":tps_libs+"Margins"+os.sep+"NoMargins",
                                  "EnvironmentMarginsFileName":tps_libs+"Margins"+os.sep+"Example",
                                  "BumpFactorFileName":tps_libs+"BumpModels"+os.sep+"None",
                                  "UseOnlyOneMaterial":0,
                                  "UseHeatLoadSubZones":1,
                                  "NumberOfHeatLoadSubZones":10,
                                  "groupName":["horizontalCS", "verticalCS"]} #, "fuselage_base", "inlet", "hollowInlet"]}
                         }

# Specify the TPSSizer execiutable (required to trigger TPSSizer
cbaero.input.TPSSizer = "tpssizerv36_linux"

# FIAT Version for TPSSizer -FIAT argument
cbaero.input.FIAT = "FIATv3p2p0"

# Explicitly run analysis (optional)
cbaero.runAnalysis()

# Get all ouputs
for i in cbaero.output.keys():
    print(str(i) + " = " + str(cbaero.output[i].value))
