#------------------------------------------------------------------------------#
# Import pyCAPS module
import pyCAPS

# Import other modules
import os

# Import argparse module
import argparse

# Import SU2 Python interface module
from parallel_computation import parallel_computation as su2Run
#------------------------------------------------------------------------------#


# Setup and read command line options. Please note that this isn't required for pyCAPS
parser = argparse.ArgumentParser(description = 'SU2 w/ AFLR and refine Pytest Example',
                                 prog = 'su2_refine_PyTest.py',
                                 formatter_class = argparse.ArgumentDefaultsHelpFormatter)

#Setup the available commandline options
parser.add_argument('-workDir', default = "." + os.sep, nargs=1, type=str, help = 'Set working/run directory')
parser.add_argument('-numberProc', default = 1, nargs=1, type=float, help = 'Number of processors')
parser.add_argument("-outLevel", default = 1, type=int, choices=[0, 1, 2], help="Set output verbosity")
args = parser.parse_args()

#------------------------------------------------------------------------------#

# Working directory
workDir = os.path.join(str(args.workDir[0]), "SU2_refine_AFLR3")

# Load CAPS Problem
capsProblem = pyCAPS.Problem(problemName = workDir,
                             capsFile=os.path.join("..","csmData","cfdSymmetry.csm"),
                             outLevel = args.outLevel);

#------------------------------------------------------------------------------#

# Create aflr4 AIM
aflr4 = capsProblem.analysis.create(aim  = "aflr4AIM",
                                    name = "aflr4")

# Farfield growth factor
aflr4.input.ff_cdfr = 1.8

# Set maximum and minimum edge lengths relative to capsMeshLength
aflr4.input.max_scale = 0.2
aflr4.input.ideal_min_scale = 0.05

aflr4.input.Mesh_Length_Factor = 2.5
aflr4.input.curv_angle = 20

# Mesing boundary conditions
aflr4.input.Mesh_Sizing = {"Farfield": {"bcType":"Farfield"},
                           "Symmetry": {"bcType":"Symmetry"}}

#------------------------------------------------------------------------------#

# Create AFLR3 AIM to generate the volume mesh
aflr3 = capsProblem.analysis.create(aim  = "aflr3AIM",
                                    name = "aflr3")

# Link the aflr4 Surface_Mesh as input to aflr3
aflr3.input["Surface_Mesh"].link(aflr4.output["Surface_Mesh"])

# Write a tecplot mesh
aflr3.input.Mesh_Format = "Tecplot"

# Specify equi or right
aflr3.input.MetricSpace = "EquiAngle" # "RightAngle

#------------------------------------------------------------------------------#

# Create refine AIM
refine = capsProblem.analysis.create(aim = "refineAIM",
                                     name= "refine")

# Limit the number of passes to speed up this example (generally this should be the default)
refine.input.Passes = 2

# Define executeable string
#refine.input.ref = "mpiexec -n 4 refmpifull";

# Link the aflr3 mesh as the refine for the multi-scale calculation
refine.input["Mesh"].link(aflr3.output["Volume_Mesh"]);

#------------------------------------------------------------------------------#

# Create SU2 AIM
su2 = capsProblem.analysis.create(aim = "su2AIM",
                                  name = "su2")

# Link the aflr3 Mesh as input to su2
su2.input["Mesh"].link(aflr3.output["Volume_Mesh"])



speedofSound = 340.0 # m/s
refVelocity  = 0.7*speedofSound # m/s
refDensity   = 1.2   # kg/m^3

su2.input.Alpha = 10.0                     # AoA
su2.input.Mach  = refVelocity/speedofSound # Mach number
su2.input.Equation_Type = "compressible"   # Equation type
su2.input.Physical_Problem = "Euler"       # Set equation type
su2.input.Num_Iter = 5                     # Number of iterations

# Set boundary conditions via capsGroup
inviscidBC = {"bcType" : "Inviscid"}

bc1 = {"bcType" : "Inviscid", "wallTemperature" : 1}
su2.input.Boundary_Condition = {"Wing1"    : bc1,
                                "Wing2"    : inviscidBC,
                                "Symmetry" : "SymmetryY",
                                "Farfield" : "farfield"}

# Specifcy the boundares used to compute forces
su2.input.Surface_Monitor = ["Wing1", "Wing2"]

# Dump out an ASCII tecplot file that is converted into .sol file
su2.input.Output_Format = "Tecplot_ASCII"

# Specifcy the Mach number volume output for metric generation
su2.input.Volume_Output = "Mach"

#------------------------------------------------------------------------------#

# Specify number of adaptation iterations for each target complexity
nAdapt     = [   2,     2]
tar_Complex= [8000, 16000]
for ii,iComplex in enumerate(tar_Complex):
    for iadapt in range(0,nAdapt[ii]):

        aflr3.input.Proj_Name = "Adapt_"+str(iComplex)+"_"+str(iadapt)

        # Set the  mesh size
        refine.input["Complexity"].value = iComplex

        # Set project name. Files written to su2.analysisDir will have this name
        su2.input.Proj_Name = "refineTest_" + str(iComplex) + "_" + str(iadapt)

        # Run AIM pre-analysis
        su2.preAnalysis()

        ####### Run su2 ####################
        print ("\n\nRunning SU2......")
        currentDirectory = os.getcwd() # Get our current working directory
        
        os.chdir(su2.analysisDir) # Move into test directory
        
        su2Run(su2.input.Proj_Name + ".cfg", args.numberProc) # Run SU2
        
        os.chdir(currentDirectory) # Move back to top directory
        #######################################

        # Run AIM post-analysis
        su2.postAnalysis()

        # Specify the file name for the primitive variable field use to compute multiscale metric
        ScalarFieldFile = os.path.join(su2.analysisDir, "flow_" + su2.input.Proj_Name + ".sol")
        
        flowFile = os.path.join(su2.analysisDir, "flow_" + su2.input.Proj_Name + ".dat")

        # Get force results
        print ("\n==> Total Forces and Moments")
        # Get Lift and Drag coefficients
        print ("--> Cl = ", su2.output.CLtot,
                   "Cd = ", su2.output.CDtot)

        # Get Cmx, Cmy, and Cmz coefficients
        print ("--> Cmx = ", su2.output.CMXtot,
                   "Cmy = ", su2.output.CMYtot,
                   "Cmz = ", su2.output.CMZtot)

        # Get Cx, Cy, Cz coefficients
        print ("--> Cx = ", su2.output.CXtot,
                   "Cy = ", su2.output.CYtot,
                   "Cz = ", su2.output.CZtot)

        ####### Read Mach from SU2 Flow file ########
        flow = open(flowFile, "r")
        TITLE = flow.readline()
        VARIABLES = flow.readline().replace("\n", "").replace("\r", "").split(",")
        ZONE = flow.readline()

        nodes = int(ZONE.split("NODES=")[-1].split(",")[0])

        sol = open(ScalarFieldFile, "w")
        sol.write("MeshVersionFormatted 2\n")
        sol.write("Dimension\n")
        sol.write("3\n")
        sol.write("SolAtVertices\n")
        sol.write(str(nodes) + " 1 1\n")
        
        k = VARIABLES.index('"Mach"')
        
        # Copy over the Mach number
        for j in range(nodes):
            data = flow.readline().split("\t")
            sol.write(data[k] + "\n")

        flow.close()
        sol.close()
        #######################################

        # Set the new scalar field file
        refine.input.ScalarFieldFile = ScalarFieldFile

        # Set the metric field file for aflr3
        aflr3.input.MetricFieldFile = refine.output.MetricFieldFile
        aflr3.input.MetricMeshFile  = os.path.join(refine.analysisDir, "refine_out.meshb")

        # Unlink the refine mesh input
        #if iadapt == 0: refine.input["Mesh"].unlink()


capsProblem.closePhase()
