// This software has been cleared for public release on 05 Nov 2020, case number 88ABW-2020-3462.


#ifdef HAVE_PYTHON
#include "Python.h" // Bring in Python API
#endif

#include <string.h>
#include <math.h>

#include "aimUtil.h"

#include "nastranPyUtils.h" // Bring in nastran utility header

//#ifdef HAVE_PYTHON
//#include "nastranOP2Reader.h" // Bring in Cython generated header file
//
//#ifndef CYTHON_PEP489_MULTI_PHASE_INIT
//#define CYTHON_PEP489_MULTI_PHASE_INIT (PY_VERSION_HEX >= 0x03050000)
//#endif
//
//#if CYTHON_PEP489_MULTI_PHASE_INIT
//static int nastranOP2Reader_Initialized = (int)false;
//#endif
//
//#endif


// Open and read nastran OP2 file
int nastran_openOP2(void *aimInfo, const char *filename, PyOP2model *model)
{
  int status = CAPS_SUCCESS;
  char aimFile[PATH_MAX];

//#ifdef HAVE_PYTHON
//  PyObject* mobj = NULL;
//#if CYTHON_PEP489_MULTI_PHASE_INIT
//  PyModuleDef *mdef = NULL;
//  PyObject *modname = NULL;
//#endif
//#endif

  if (model == NULL) return CAPS_NULLOBJ;

  status = aim_file(aimInfo, filename, aimFile);
  AIM_STATUS(aimInfo, status);

#ifdef HAVE_PYTHON
  {
    // Initialize python
    if (Py_IsInitialized() == 0) {
      Py_InitializeEx(0);
      model->initPy = (int) true;
    } else {
      model->initPy = (int) false;
    }

    PyGILState_STATE gstate;
    gstate = PyGILState_Ensure();

    // Taken from "main" by running cython with --embed
//#if PY_MAJOR_VERSION < 3
//    initnastranOP2Reader();
//#elif CYTHON_PEP489_MULTI_PHASE_INIT
//    if (nastranOP2Reader_Initialized == (int)false || model->initPy == (int)true) {
//      nastranOP2Reader_Initialized = (int)true;
//      mobj = PyInit_nastranOP2Reader();
//      if (!PyModule_Check(mobj)) {
//        mdef = (PyModuleDef *) mobj;
//        modname = PyUnicode_FromString("nastranOP2reader");
//        mobj = NULL;
//        if (modname) {
//          mobj = PyModule_NewObject(modname);
//          Py_DECREF(modname);
//          if (mobj) PyModule_ExecDef(mobj, mdef);
//        }
//      }
//    }
//#else
//    mobj = PyInit_nastranOP2Reader();
//#endif

    if (PyErr_Occurred()) {
      PyErr_Print();
#if PY_MAJOR_VERSION < 3
      if (Py_FlushLine()) PyErr_Clear();
#endif
      /* Release the thread. No Python API allowed beyond this point. */
      PyGILState_Release(gstate);
      status = CAPS_BADOBJECT;
      goto cleanup;
    }

    PyObject *globals = PyDict_New();
    PyObject *pyfilename = PyUnicode_FromString(aimFile);
    PyDict_SetItemString(globals, "filename", pyfilename);
    PyObject *res = PyRun_String(
        "from pyNastran.op2.op2 import OP2\n"
        "model = OP2(debug=False)\n"
        "model.read_op2(filename, combine=False)\n",
        Py_file_input, globals, globals);
    Py_XDECREF(res);
    PyObject *pymodel = PyDict_GetItemString(globals, "model");
    Py_XINCREF(pymodel);
    Py_XDECREF(pyfilename);
    Py_XDECREF(globals);

    model->pymodel = pymodel; //nastran_modelOP2(aimFile);
    
    if (PyErr_Occurred()) {
      PyErr_Print();
      status = CAPS_IOERR;
#if PY_MAJOR_VERSION < 3
      if (Py_FlushLine()) PyErr_Clear();
#endif
    }

    /* Release the thread. No Python API allowed beyond this point. */
    PyGILState_Release(gstate);
  }
#else

  AIM_ERROR(aimInfo, "CAPS is not compiled with Python!");
  status = CAPS_NOTIMPLEMENT;
  goto cleanup;
#endif

cleanup:

  return status;
}

// Close the model
int nastran_closeOP2(PyOP2model *model)
{
  int status = CAPS_SUCCESS;

  if (model == NULL) return CAPS_NULLOBJ;
  if (model->pymodel == NULL) return CAPS_SUCCESS;

#ifdef HAVE_PYTHON
  {
    PyGILState_STATE gstate;
    gstate = PyGILState_Ensure();

    PyObject *pymodel = (PyObject *)model->pymodel;
    Py_XDECREF(pymodel);
    model->pymodel = NULL;

    if (PyErr_Occurred()) {
      PyErr_Print();
      status = CAPS_IOERR;
#if PY_MAJOR_VERSION < 3
      if (Py_FlushLine()) PyErr_Clear();
#endif
    }

    /* Release the thread. No Python API allowed beyond this point. */
    PyGILState_Release(gstate);

    // Cannot close down python due to numpy...
//    if (model->initPy == (int) true) {
//      Py_Finalize(); // Do not finalize if the AIM was called from Python
//    }
  }
#else

  //AIM_ERROR(aimInfo, "CAPS is not compiled with Python!");
  status = CAPS_NOTIMPLEMENT;
#endif

  return status;
}


// Read displacement values for a Nastran OP2 file and load it into a dataMatrix
int nastran_readOP2Displacement(void *aimInfo, PyOP2model *model,
                                /*@unused@*/int itime, /*@unused@*/int subcaseId,
                                /*@unused@*/int *numGridPoint, /*@unused@*/double ***dataMatrix)
{
  int status = CAPS_SUCCESS;

  if (model == NULL) return CAPS_NULLOBJ;
  if (model->pymodel == NULL) return CAPS_NULLOBJ;

  *numGridPoint = 0;
  *dataMatrix = NULL;

#ifdef HAVE_PYTHON
  const int numVariable = 8; // Grid Id, coord Id, T1, T2, T3, R1, R2, R3
  {
    PyGILState_STATE gstate;
    gstate = PyGILState_Ensure();

    PyObject *pymodel = (PyObject *)model->pymodel;
    //nastran_getOP2Displacements(pymodel, itime, subcaseId, numGridPoint, dataMatrix);

    PyObject *globals = PyDict_New(); // New Ref
    PyObject *pyitime = PyLong_FromLong(itime); // New Ref
    PyObject *pysubcaseId = PyLong_FromLong(subcaseId); // New Ref
    PyDict_SetItemString(globals, "model", pymodel);
    PyDict_SetItemString(globals, "itime", pyitime);
    PyDict_SetItemString(globals, "subcaseId", pysubcaseId);
    PyObject *res = PyRun_String(
        "key = list(model.displacements)[-1]\n"
        "disp = model.displacements[key]\n"
        "txyzr = disp.data[itime, :, :]\n"
        "node_gridtype = disp.node_gridtype\n",
        Py_file_input, globals, globals);
    Py_XDECREF(res);
    PyObject *txyzr         = PyDict_GetItemString(globals, "txyzr"); // Borrowed Ref
    PyObject *node_gridtype = PyDict_GetItemString(globals, "node_gridtype"); // Borrowed Ref

    Py_ssize_t i;
    Py_ssize_t numData = PySequence_Length(txyzr);
    AIM_ALLOC((*dataMatrix), numData, double*, aimInfo, status);
    for (i = 0; i < numData; i++) (*dataMatrix)[i] = NULL;
    *numGridPoint = numData;

    for (i = 0; i < numData; i++) {
      AIM_ALLOC((*dataMatrix)[i], numVariable, double, aimInfo, status);

      PyObject *nid_gt_i = PySequence_GetItem(node_gridtype, i); // New Ref
      PyObject *txyzr_i = PySequence_GetItem(txyzr, i); // New Ref

      PyObject *ii = NULL;

      ii = PySequence_GetItem(nid_gt_i, 0); (*dataMatrix)[i][0] = PyLong_AsLong(ii);    Py_XDECREF(ii);
      ii = PySequence_GetItem(nid_gt_i, 1); (*dataMatrix)[i][1] = PyLong_AsLong(ii);    Py_XDECREF(ii);
      ii = PySequence_GetItem(txyzr_i, 0);  (*dataMatrix)[i][2] = PyFloat_AsDouble(ii); Py_XDECREF(ii);
      ii = PySequence_GetItem(txyzr_i, 1);  (*dataMatrix)[i][3] = PyFloat_AsDouble(ii); Py_XDECREF(ii);
      ii = PySequence_GetItem(txyzr_i, 2);  (*dataMatrix)[i][4] = PyFloat_AsDouble(ii); Py_XDECREF(ii);
      ii = PySequence_GetItem(txyzr_i, 3);  (*dataMatrix)[i][5] = PyFloat_AsDouble(ii); Py_XDECREF(ii);
      ii = PySequence_GetItem(txyzr_i, 4);  (*dataMatrix)[i][6] = PyFloat_AsDouble(ii); Py_XDECREF(ii);
      ii = PySequence_GetItem(txyzr_i, 5);  (*dataMatrix)[i][7] = PyFloat_AsDouble(ii); Py_XDECREF(ii);

      Py_XDECREF(nid_gt_i);
      Py_XDECREF(txyzr_i);
    }

    if (PyErr_Occurred()) {
      PyErr_Print();
      AIM_ERROR(aimInfo, "Python error occurred while reading OP2 Displacement\n");
      status = CAPS_IOERR;
#if PY_MAJOR_VERSION < 3
      if (Py_FlushLine()) PyErr_Clear();
#endif
    }

cleanup:

    Py_XDECREF(pyitime);
    Py_XDECREF(pysubcaseId);
    Py_XDECREF(globals);

    /* Release the thread. No Python API allowed beyond this point. */
    PyGILState_Release(gstate);
  }
#else

  AIM_ERROR(aimInfo, "CAPS is not compiled with Python!");
  status = CAPS_NOTIMPLEMENT;
#endif

  return status;
}


// Read displacement values for a Nastran OP2 file and load it into a dataMatrix
int nastran_readOP2EigenValue(void *aimInfo, PyOP2model *model,
                              /*@unused@*/int subcaseId, int *numGridPoint, double ***dataMatrix)
{
  int status = CAPS_SUCCESS;

  if (model == NULL) return CAPS_NULLOBJ;
  if (model->pymodel == NULL) return CAPS_NULLOBJ;

  *numGridPoint = 0;
  *dataMatrix = NULL;

#ifdef HAVE_PYTHON
  const int numVariable = 5; // EigenValue, eigenValue(radians), eigenValue(cycles), generalized mass, and generalized stiffness

  {
    PyGILState_STATE gstate;
    gstate = PyGILState_Ensure();

    PyObject *pymodel = (PyObject *)model->pymodel;
    //nastran_getOP2EigenValue(pymodel, subcaseId, numGridPoint, dataMatrix);

    PyObject *globals = PyDict_New();
    PyDict_SetItemString(globals, "model", pymodel);
    PyObject *res = PyRun_String(
        "key = list(model.eigenvalues)[-1]\n"
        "eigenvalues = model.eigenvalues[key]\n"
        "values = eigenvalues.eigenvalues\n"
        "radians = eigenvalues.radians\n"
        "cycles = eigenvalues.cycles\n"
        "generalized_mass = eigenvalues.generalized_mass\n"
        "generalized_stiffness = eigenvalues.generalized_stiffness\n",
        Py_file_input, globals, globals);
    Py_XDECREF(res);
    PyObject *values  = PyDict_GetItemString(globals, "values" ); // Borrowed Ref
    PyObject *radians = PyDict_GetItemString(globals, "radians"); // Borrowed Ref
    PyObject *cycles  = PyDict_GetItemString(globals, "cycles" ); // Borrowed Ref
    PyObject *generalized_masses = PyDict_GetItemString(globals, "generalized_mass"); // Borrowed Ref
    PyObject *generalized_stiffnesses = PyDict_GetItemString(globals, "generalized_stiffness"); // Borrowed Ref

    Py_ssize_t i;
    Py_ssize_t numData = PySequence_Length(values);
    AIM_ALLOC((*dataMatrix), numData, double*, aimInfo, status);
    for (i = 0; i < numData; i++) (*dataMatrix)[i] = NULL;
    *numGridPoint = numData;

    for (i = 0; i < numData; i++) {
      AIM_ALLOC((*dataMatrix)[i], numVariable, double, aimInfo, status);

      PyObject *value     = PySequence_GetItem(values, i);
      PyObject *radian    = PySequence_GetItem(radians, i);
      PyObject *cycle     = PySequence_GetItem(cycles, i);
      PyObject *gen_mass  = PySequence_GetItem(generalized_masses, i);
      PyObject *gen_stiff = PySequence_GetItem(generalized_stiffnesses, i);

      (*dataMatrix)[i][0] = PyFloat_AsDouble(value    );
      (*dataMatrix)[i][1] = PyFloat_AsDouble(radian   );
      (*dataMatrix)[i][2] = PyFloat_AsDouble(cycle    );
      (*dataMatrix)[i][3] = PyFloat_AsDouble(gen_mass );
      (*dataMatrix)[i][4] = PyFloat_AsDouble(gen_stiff);

      Py_XDECREF(value);
      Py_XDECREF(radian);
      Py_XDECREF(cycle);
      Py_XDECREF(gen_mass);
      Py_XDECREF(gen_stiff);
    }

    if (PyErr_Occurred()) {
      PyErr_Print();
      AIM_ERROR(aimInfo, "Python error occurred while reading OP2 Eigen Value\n");
      status = CAPS_IOERR;
#if PY_MAJOR_VERSION < 3
      if (Py_FlushLine()) PyErr_Clear();
#endif
    }

cleanup:

    Py_XDECREF(globals);

    /* Release the thread. No Python API allowed beyond this point. */
    PyGILState_Release(gstate);
  }
#else

  AIM_ERROR(aimInfo, "CAPS is not compiled with Python!");
  status = CAPS_NOTIMPLEMENT;
#endif

  return status;
}


// Read weight value from a Nastran OP2 file
int nastran_readOP2GridPointWeightGeneratorOutput(/*@unused@*/void *aimInfo, PyOP2model *model,
                                                  /*@unused@*/double *mass, /*@unused@*/double cg[3],
                                                  /*@unused@*/double is[6], /*@unused@*/double iq[3], /*@unused@*/double q[9])
{
  int status = CAPS_SUCCESS;

  if (model == NULL) return CAPS_NULLOBJ;
  if (model->pymodel == NULL) return CAPS_NULLOBJ;

  status = CAPS_NOTIMPLEMENT;
#if 0
#ifdef HAVE_PYTHON
  {
    PyGILState_STATE gstate;
    gstate = PyGILState_Ensure();

    PyObject *pymodel = (PyObject *)model->pymodel;
    //status = nastran_getOP2GridPointWeight(pymodel, mass, cg, is, iq, q);

    if (PyErr_Occurred()) {
      PyErr_Print();
      AIM_ERROR(aimInfo, "Python error occurred while reading OP2 Grid Point Weight");
      status = CAPS_IOERR;
#if PY_MAJOR_VERSION < 3
      if (Py_FlushLine()) PyErr_Clear();
#endif
    }

    /* Release the thread. No Python API allowed beyond this point. */
    PyGILState_Release(gstate);
  }
#else

  AIM_ERROR(aimInfo, "CAPS is not compiled with Python!");
  status = CAPS_NOTIMPLEMENT;
#endif
#endif

  return status;
}


// Read objective values for a Nastran OP2 file and load it into a dataMatrix[numPoint]
int nastran_readOP2Objective(void *aimInfo, /*@unused@*/PyOP2model *model, int *numData,  double **dataMatrix)
{
  int status = CAPS_SUCCESS;

  if (model == NULL) return CAPS_NULLOBJ;
  if (model->pymodel == NULL) return CAPS_NULLOBJ;

  *numData = 0;
  *dataMatrix = NULL;

#ifdef HAVE_PYTHON
  {
    PyGILState_STATE gstate;
    gstate = PyGILState_Ensure();

    PyObject *pymodel = (PyObject *)model->pymodel;
    //nastran_getObjective(pymodel, numData, dataMatrix);

    PyObject *op2_results = PyObject_GetAttrString(pymodel, "op2_results");
    PyObject *responses = PyObject_GetAttrString(op2_results, "responses");
    PyObject *convergence_data = PyObject_GetAttrString(responses, "convergence_data");
    PyObject *objFunc = PyObject_GetAttrString(convergence_data, "obj_initial");

    Py_ssize_t i;
    Py_ssize_t numObj = PySequence_Length(objFunc);
    AIM_ALLOC((*dataMatrix), numObj, double, aimInfo, status);
    *numData = numObj;

    for (i = 0; i < numObj; i++) {
      PyObject *objFunc_i = PySequence_GetItem(objFunc, i);

      (*dataMatrix)[i] = PyFloat_AsDouble(objFunc_i);

      Py_XDECREF(objFunc_i);
    }

    if (PyErr_Occurred()) {
      PyErr_Print();
      AIM_ERROR(aimInfo, "Python error occurred while reading OP2 Objective");
      status = CAPS_IOERR;
#if PY_MAJOR_VERSION < 3
      if (Py_FlushLine()) PyErr_Clear();
#endif
    }

cleanup:
    Py_XDECREF(objFunc);
    Py_XDECREF(convergence_data);
    Py_XDECREF(responses);
    Py_XDECREF(op2_results);

    /* Release the thread. No Python API allowed beyond this point. */
    PyGILState_Release(gstate);
  }
#else

  AIM_ERROR(aimInfo, "CAPS is not compiled with Python!");
  status = CAPS_NOTIMPLEMENT;
#endif

  return status;
}
