/*
 *      CAPS: Computational Aircraft Prototype Syntheses
 *
 *      Copyright 2014-2025, Massachusetts Institute of Technology
 *      Licensed under The GNU Lesser General Public License, version 2.1
 *      See http://www.opensource.org/licenses/lgpl-2.1.php
 *
 */


/* Include miscUtils.c to test static functions can be tested */
#include "miscUtils.c"
#include "aimTest.h"

//---------------------------------------------------------------------------//
/* Used to test the WIN32 getline implementation */
void test_getline(void *aimInfo)
{
  FILE *stream;
  char *line = NULL;
  size_t len = 0;
  int nread;

  const char* filename = "test_getline_file.txt";

  const char* line0 = "A header\n";
  const char* line1 = "\n";
  const char* line2 = "More things\n";

  stream = fopen(filename, "w");
  TEST_ASSERT_(stream != NULL, "opening test file %s", filename);

  /* Write some text to the file */
  fprintf(stream, "%s", line0);
  fprintf(stream, "%s", line1); // empty line
  fprintf(stream, "%s", line2);

  if (stream != NULL) fclose(stream);
  stream = fopen(filename, "r");
  TEST_ASSERT(stream != NULL);

  nread = getline(&line, &len, stream);
  TEST_CHECK(nread == strlen(line0) && nread > 0);
  TEST_ASSERT(line != NULL);
  TEST_MSG("nread (%d) != %d", nread, (int)strlen(line0));
  TEST_MSG("expected \"%s\" != \"%s\"", line0, line);
  TEST_CHECK_STRING( line, line0 );

  nread = getline(&line, &len, stream);
  TEST_CHECK(nread == strlen(line1) && nread > 0);
  TEST_ASSERT(line != NULL);
  TEST_MSG("nread (%d) != %d", nread, (int)strlen(line1));
  TEST_MSG("expected \"%s\" != \"%s\"\n", line1, line);
  TEST_CHECK_STRING( line, line1 );

  nread = getline(&line, &len, stream);
  TEST_CHECK(nread == strlen(line2) && nread > 0);
  TEST_ASSERT(line != NULL);
  TEST_MSG("nread (%d) != %d", nread, (int)strlen(line2));
  TEST_MSG("expected \"%s\" != \"%s\"\n", line2, line);
  TEST_CHECK_STRING( line, line2 );

  nread = getline(&line, &len, stream);
  TEST_CHECK(nread == -1);
  TEST_MSG("nread (%d) != -1", nread);

  AIM_FREE(line);
  if (stream != NULL) fclose(stream);
  remove(filename);
}

//---------------------------------------------------------------------------//
/* Used to test the convert_doiubleToString implementation */
void test_convert_doubleToString(void *aimInfo)
{
  char stringVal[16];
  int fieldWidth = 7;
  double doubleVal;

  doubleVal = 0;
  convert_doubleToString(doubleVal, 1, 0, stringVal);
  TEST_CHECK_STRING( "NaN", stringVal);
  convert_doubleToString(doubleVal, 2, 0, stringVal);
  TEST_CHECK_STRING( "0.", stringVal);
  convert_doubleToString(doubleVal, 3, 0, stringVal);
  TEST_CHECK_STRING( "0.0", stringVal);
  convert_doubleToString(doubleVal, 4, 0, stringVal);
  TEST_CHECK_STRING( "0.00", stringVal);
  convert_doubleToString(doubleVal, 7, 0, stringVal);
  TEST_CHECK_STRING( "0.00000", stringVal);
  convert_doubleToString(doubleVal, 8, 0, stringVal);
  TEST_CHECK_STRING( "0.000000", stringVal);
  convert_doubleToString(doubleVal, 11, 0, stringVal);
  TEST_CHECK_STRING( "0.000000000", stringVal);


  /******* New field width ******/
  fieldWidth = 7;

  doubleVal = 0;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "0.00000", stringVal);

  /* Check positive numbers */
  doubleVal = 1.23e-10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.2E-10", stringVal);

  doubleVal = 1.26e-10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.3E-10", stringVal);

  doubleVal = 1.23e-3;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.23E-3", stringVal);

  doubleVal = 0.0123456;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.23E-2", stringVal);

  doubleVal = 0.123456;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "0.12346", stringVal);

  doubleVal = 1.23456;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.23456", stringVal);

  doubleVal = 12.3456;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "12.3456", stringVal);

  doubleVal = 123.456;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "123.456", stringVal);

  doubleVal = 1234.56;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1234.56", stringVal);

  doubleVal = 12345.6;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "12345.6", stringVal);

  doubleVal = 123456.;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "123456.", stringVal);

  doubleVal = 1.23456e5;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "123456.", stringVal);

  doubleVal = 1.23e10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.2E+10", stringVal);

  doubleVal = 1.2e20;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.2E+20", stringVal);

  /* Check positive numbers on multiples of 10*/
  doubleVal = 1.0e-20;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.0E-20", stringVal);

  doubleVal = 1.0e-10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.0E-10", stringVal);

  doubleVal = 1.00e-4;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.00E-4", stringVal);

  doubleVal = 1.00e-3;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.00E-3", stringVal);

  doubleVal = 0.01;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.00E-2", stringVal);

  doubleVal = 0.1;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "0.10000", stringVal);

  doubleVal = 1.00000;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.00000", stringVal);

  doubleVal = 10.0000;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "10.0000", stringVal);

  doubleVal = 100.000;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "100.000", stringVal);

  doubleVal = 1000.00;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1000.00", stringVal);

  doubleVal = 10000.0;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "10000.0", stringVal);

  doubleVal = 1.00e5;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "100000.", stringVal);

  doubleVal = 1.00e6;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.00E+6", stringVal);

  doubleVal = 1.00e7;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.00E+7", stringVal);

  doubleVal = 1.0e10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.0E+10", stringVal);

  doubleVal = 1.0e20;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.0E+20", stringVal);

  /* Check more positive numbers*/
  doubleVal = exp(2.6*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "398.107", stringVal);

  doubleVal = exp(2.4*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "251.189", stringVal);

  doubleVal = exp(1.6*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "39.8107", stringVal);

  doubleVal = exp(1.4*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "25.1189", stringVal);

  doubleVal = exp(0.6*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "3.98107", stringVal);

  doubleVal = exp(0.4*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "2.51189", stringVal);

  doubleVal = exp(-0.4*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "0.39811", stringVal);

  doubleVal = exp(-0.6*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "0.25119", stringVal);

  doubleVal = exp(-1.4*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "3.98E-2", stringVal);

  doubleVal = exp(-1.6*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "2.51E-2", stringVal);

  doubleVal = exp(-2.4*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "3.98E-3", stringVal);

  doubleVal = exp(-2.6*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "2.51E-3", stringVal);

  doubleVal = exp(-3.4*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "3.98E-4", stringVal);

  doubleVal = exp(-3.6*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "2.51E-4", stringVal);

  /* Check negative numbers */
  doubleVal = -1.23e-10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.E-10", stringVal);

  doubleVal = -1.23e-3;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.2E-3", stringVal);

  doubleVal = -0.0123456;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.2E-2", stringVal);

  doubleVal = -0.123456;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-0.1235", stringVal);

  doubleVal = -1.23456;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.2346", stringVal);

  doubleVal = -12.3456;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-12.346", stringVal);

  doubleVal = -123.456;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-123.46", stringVal);

  doubleVal = -1234.56;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1234.6", stringVal);

  doubleVal = -12345.6;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-12346.", stringVal);

  doubleVal = -1.234e5;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.2E+5", stringVal);

  doubleVal = -1.23e10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.E+10", stringVal);

  doubleVal = -1.2e20;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.E+20", stringVal);

  /* Check negative numbers on multiples of 10*/
  doubleVal = -1.0e-20;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.E-20", stringVal);

  doubleVal = -1.0e-10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.E-10", stringVal);

  doubleVal = -1.00e-4;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.0E-4", stringVal);

  doubleVal = -1.00e-3;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.0E-3", stringVal);

  doubleVal = -0.01;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.0E-2", stringVal);

  doubleVal = -0.1;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-0.1000", stringVal);

  doubleVal = -1.00000;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.0000", stringVal);

  doubleVal = -10.0000;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-10.000", stringVal);

  doubleVal = -100.000;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-100.00", stringVal);

  doubleVal = -1000.00;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1000.0", stringVal);

  doubleVal = -10000.0;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-10000.", stringVal);

  doubleVal = -1.00e5;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.0E+5", stringVal);

  doubleVal = -1.00e6;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.0E+6", stringVal);

  doubleVal = -1.0e10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.E+10", stringVal);

  doubleVal = -1.0e20;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.E+20", stringVal);

  /******* New field width ******/
  fieldWidth = 8;

  doubleVal = 0;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "0.000000", stringVal);

  /* Check positive numbers */
  doubleVal = 1.234e-10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.23E-10", stringVal);

  doubleVal = 1.236e-10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.24E-10", stringVal);

  doubleVal = 1.234e-3;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.234E-3", stringVal);

  doubleVal = 0.0123456;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.235E-2", stringVal);

  doubleVal = 0.1234567;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "0.123457", stringVal);

  doubleVal = 1.234567;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.234567", stringVal);

  doubleVal = 12.34567;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "12.34567", stringVal);

  doubleVal = 123.4567;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "123.4567", stringVal);

  doubleVal = 1234.567;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1234.567", stringVal);

  doubleVal = 12345.67;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "12345.67", stringVal);

  doubleVal = 1.234e5;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "123400.0", stringVal);

  doubleVal = 1.23e10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.23E+10", stringVal);

  doubleVal = 1.235e20;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.24E+20", stringVal);

  /* Check positive numbers on multiples of 10*/
  doubleVal = 1.0e-20;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.00E-20", stringVal);

  doubleVal = 1.0e-10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.00E-10", stringVal);

  doubleVal = 1.00e-4;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.000E-4", stringVal);

  doubleVal = 1.00e-3;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.000E-3", stringVal);

  doubleVal = 0.01;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.000E-2", stringVal);

  doubleVal = 0.1;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "0.100000", stringVal);

  doubleVal = 1.00000;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.000000", stringVal);

  doubleVal = 10.0000;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "10.00000", stringVal);

  doubleVal = 100.000;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "100.0000", stringVal);

  doubleVal = 1000.00;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1000.000", stringVal);

  doubleVal = 10000.0;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "10000.00", stringVal);

  doubleVal = 1.00e5;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "100000.0", stringVal);

  doubleVal = 1.00e6;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1000000.", stringVal);

  doubleVal = 1.00e7;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.000E+7", stringVal);

  doubleVal = 1.0e10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.00E+10", stringVal);

  doubleVal = 1.0e20;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.00E+20", stringVal);

  /* Check more positive numbers*/
  doubleVal = exp(2.6*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "398.1072", stringVal);

  doubleVal = exp(2.4*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "251.1886", stringVal);

  doubleVal = exp(1.6*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "39.81072", stringVal);

  doubleVal = exp(1.4*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "25.11886", stringVal);

  doubleVal = exp(0.6*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "3.981072", stringVal);

  doubleVal = exp(0.4*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "2.511886", stringVal);

  doubleVal = exp(-0.4*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "0.398107", stringVal);

  doubleVal = exp(-0.6*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "0.251189", stringVal);

  doubleVal = exp(-1.4*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "3.981E-2", stringVal);

  doubleVal = exp(-1.6*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "2.512E-2", stringVal);

  doubleVal = exp(-2.4*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "3.981E-3", stringVal);

  doubleVal = exp(-2.6*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "2.512E-3", stringVal);

  doubleVal = exp(-3.4*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "3.981E-4", stringVal);

  doubleVal = exp(-3.6*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "2.512E-4", stringVal);

  /* Check negative numbers */
  doubleVal = -1.23e-10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.2E-10", stringVal);

  doubleVal = -1.234e-3;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.23E-3", stringVal);

  doubleVal = -0.01234567;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.23E-2", stringVal);

  doubleVal = -0.1234567;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-0.12346", stringVal);

  doubleVal = -1.234567;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.23457", stringVal);

  doubleVal = -12.34567;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-12.3457", stringVal);

  doubleVal = -123.4567;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-123.457", stringVal);

  doubleVal = -1234.567;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1234.57", stringVal);

  doubleVal = -12345.67;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-12345.7", stringVal);

  doubleVal = -1.234e5;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-123400.", stringVal);

  doubleVal = -1.234e6;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.23E+6", stringVal);

  doubleVal = -1.23e10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.2E+10", stringVal);

  doubleVal = -1.2e20;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.2E+20", stringVal);

  /* Check negative numbers on multiples of 10*/
  doubleVal = -1.00e-20;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.0E-20", stringVal);

  doubleVal = -1.0e-10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.0E-10", stringVal);

  doubleVal = -1.00e-4;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.00E-4", stringVal);

  doubleVal = -1.00e-3;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.00E-3", stringVal);

  doubleVal = -0.01;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.00E-2", stringVal);

  doubleVal = -0.1;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-0.10000", stringVal);

  doubleVal = -1.00000;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.00000", stringVal);

  doubleVal = -10.0000;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-10.0000", stringVal);

  doubleVal = -100.000;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-100.000", stringVal);

  doubleVal = -1000.00;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1000.00", stringVal);

  doubleVal = -10000.0;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-10000.0", stringVal);

  doubleVal = -1.00e5;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-100000.", stringVal);

  doubleVal = -1.00e6;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.00E+6", stringVal);

  doubleVal = -1.00e7;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.00E+7", stringVal);

  doubleVal = -1.0e10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.0E+10", stringVal);

  doubleVal = -1.0e20;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.0E+20", stringVal);

  /******* New field width ******/
  fieldWidth = 15;

  doubleVal = 0;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "0.0000000000000", stringVal);

  /* Check positive numbers */
  doubleVal = 1.234e-100;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.23400000E-100", stringVal);

  doubleVal = 1.236e-15;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.236000000E-15", stringVal);

  doubleVal = 1.234693e-10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.234693000E-10", stringVal);

  doubleVal = 1.23456789123456e-3;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.2345678912E-3", stringVal);

  doubleVal = 0.0123456789123456;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.2345678912E-2", stringVal);

  doubleVal = 0.1234567;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "0.1234567000000", stringVal);

  doubleVal = 1.234567;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.2345670000000", stringVal);

  doubleVal = 12.34567;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "12.345670000000", stringVal);

  doubleVal = 123.4567;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "123.45670000000", stringVal);

  doubleVal = 1234.567;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1234.5670000000", stringVal);

  doubleVal = 12345.67;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "12345.670000000", stringVal);

  doubleVal = 1.23456789123456e10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "12345678912.346", stringVal);

  doubleVal = 1.23456789123456e12;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1234567891234.6", stringVal);

  doubleVal = 1.23456789123456e13;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "12345678912346.", stringVal);

  doubleVal = 1.23456789123456e14;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.234567891E+14", stringVal);

  doubleVal = 1.23456789123456e15;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.234567891E+15", stringVal);

  doubleVal = 1.23456789123456e20;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.234567891E+20", stringVal);

  doubleVal = 1.23456789123456e100;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.23456789E+100", stringVal);

  doubleVal = 1.23456789123456e200;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.23456789E+200", stringVal);

  /* Check positive numbers on multiples of 10*/
  doubleVal = 1.0e-100;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.00000000E-100", stringVal);

  doubleVal = 1.0e-20;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.000000000E-20", stringVal);

  doubleVal = 1.0e-10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.000000000E-10", stringVal);

  doubleVal = 1.00e-4;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.0000000000E-4", stringVal);

  doubleVal = 1.00e-3;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.0000000000E-3", stringVal);

  doubleVal = 0.01;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.0000000000E-2", stringVal);

  doubleVal = 0.1;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "0.1000000000000", stringVal);

  doubleVal = 1.00000;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.0000000000000", stringVal);

  doubleVal = 10.0000;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "10.000000000000", stringVal);

  doubleVal = 100.000;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "100.00000000000", stringVal);

  doubleVal = 1000.00;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1000.0000000000", stringVal);

  doubleVal = 10000.0;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "10000.000000000", stringVal);

  doubleVal = 1.00e5;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "100000.00000000", stringVal);

  doubleVal = 1.0e10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "10000000000.000", stringVal);

  doubleVal = 1.0e20;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.000000000E+20", stringVal);

  doubleVal = 1.0e100;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.00000000E+100", stringVal);

  /* Check more positive numbers*/
  doubleVal = exp(100.1*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "1.25892541E+100", stringVal);

  doubleVal = exp(5.1*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "125892.54117942", stringVal);

  doubleVal = exp(2.6*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "398.10717055350", stringVal);

  doubleVal = exp(2.4*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "251.18864315096", stringVal);

  doubleVal = exp(1.6*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "39.810717055350", stringVal);

  doubleVal = exp(1.4*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "25.118864315096", stringVal);

  doubleVal = exp(0.6*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "3.9810717055350", stringVal);

  doubleVal = exp(0.4*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "2.5118864315096", stringVal);

  doubleVal = exp(-0.4*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "0.3981071705535", stringVal);

  doubleVal = exp(-0.6*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "0.2511886431510", stringVal);

  doubleVal = exp(-1.4*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "3.9810717055E-2", stringVal);

  doubleVal = exp(-1.6*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "2.5118864315E-2", stringVal);

  doubleVal = exp(-2.4*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "3.9810717055E-3", stringVal);

  doubleVal = exp(-2.6*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "2.5118864315E-3", stringVal);

  doubleVal = exp(-3.4*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "3.9810717055E-4", stringVal);

  doubleVal = exp(-3.6*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "2.5118864315E-4", stringVal);

  doubleVal = exp(-13.6*log(10));
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "2.511886432E-14", stringVal);

  /* Check negative numbers */
  doubleVal = -1.23456789123456e-15;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.23456789E-15", stringVal);

  doubleVal = -1.23456789123456e-10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.23456789E-10", stringVal);

  doubleVal = -1.23456789123456e-3;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.234567891E-3", stringVal);

  doubleVal = -0.01234567891234;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.234567891E-2", stringVal);

  doubleVal = -0.0123456789128;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.234567891E-2", stringVal);

  doubleVal = -0.1234567;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-0.123456700000", stringVal);

  doubleVal = -1.234567;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.234567000000", stringVal);

  doubleVal = -12.34567;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-12.34567000000", stringVal);

  doubleVal = -123.4567;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-123.4567000000", stringVal);

  doubleVal = -1234.567;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1234.567000000", stringVal);

  doubleVal = -12345.67;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-12345.67000000", stringVal);

  doubleVal = -1.234e5;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-123400.0000000", stringVal);

  doubleVal = -1.23e10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-12300000000.00", stringVal);

  doubleVal = -1.2345678912344e12;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1234567891234.", stringVal);

  doubleVal = -1.2345678912345e13;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.23456789E+13", stringVal);

  doubleVal = -1.2345678912345e14;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.23456789E+14", stringVal);

  doubleVal = -1.2e20;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.20000000E+20", stringVal);

  /* Check negative numbers on multiples of 10*/
  doubleVal = -1.00e-20;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.00000000E-20", stringVal);

  doubleVal = -1.0e-10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.00000000E-10", stringVal);

  doubleVal = -1.00e-4;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.000000000E-4", stringVal);

  doubleVal = -1.00e-3;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.000000000E-3", stringVal);

  doubleVal = -0.01;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.000000000E-2", stringVal);

  doubleVal = -0.1;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-0.100000000000", stringVal);

  doubleVal = -1.00000;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.000000000000", stringVal);

  doubleVal = -10.0000;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-10.00000000000", stringVal);

  doubleVal = -100.000;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-100.0000000000", stringVal);

  doubleVal = -1000.00;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1000.000000000", stringVal);

  doubleVal = -10000.0;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-10000.00000000", stringVal);

  doubleVal = -1.00e5;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-100000.0000000", stringVal);

  doubleVal = -1.0e10;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-10000000000.00", stringVal);

  doubleVal = -1.0e12;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1000000000000.", stringVal);

  doubleVal = -1.0e13;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.00000000E+13", stringVal);

  doubleVal = -1.0e14;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.00000000E+14", stringVal);

  doubleVal = -1.0e20;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "-1.00000000E+20", stringVal);


  /* check rounding */
  doubleVal = 99.9999999999999999999999999;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "100.00000000000", stringVal );

  doubleVal = 100.000000000001;
  convert_doubleToString(doubleVal, fieldWidth, 0, stringVal);
  TEST_CHECK_STRING( "100.00000000000", stringVal );

  doubleVal = 9.999999999999e9;
  convert_doubleToString(doubleVal, 10, 0, stringVal);
  TEST_CHECK_STRING( "1.0000E+10", stringVal );

  /*** Test right justfied ***/
  fieldWidth = 9;

  doubleVal = 10000.0;
  convert_doubleToString(doubleVal, fieldWidth, 1, stringVal);
  TEST_CHECK_STRING( "10000.000", stringVal);

  doubleVal = 1.00e5;
  convert_doubleToString(doubleVal, fieldWidth, 1, stringVal);
  TEST_CHECK_STRING( "100000.00", stringVal);

  doubleVal = 1.0e10;
  convert_doubleToString(doubleVal, fieldWidth, 1, stringVal);
  TEST_CHECK_STRING( "1.000E+10", stringVal);

  doubleVal = 1.0e20;
  convert_doubleToString(doubleVal, fieldWidth, 1, stringVal);
  TEST_CHECK_STRING( "1.000E+20", stringVal);

  doubleVal = -10000.0;
  convert_doubleToString(doubleVal, fieldWidth, 1, stringVal);
  TEST_CHECK_STRING( "-10000.00", stringVal);

  doubleVal = -1.00e5;
  convert_doubleToString(doubleVal, fieldWidth, 1, stringVal);
  TEST_CHECK_STRING( "-100000.0", stringVal);

  doubleVal = -1.0e10;
  convert_doubleToString(doubleVal, fieldWidth, 1, stringVal);
  TEST_CHECK_STRING( "-1.00E+10", stringVal);

  doubleVal = -1.0e20;
  convert_doubleToString(doubleVal, fieldWidth, 1, stringVal);
  TEST_CHECK_STRING( "-1.00E+20", stringVal);

  /**** Rounding of exponent ****///
  convert_doubleToString(+9.9999e6, 6, 1, stringVal);
  TEST_CHECK_STRING( "1.0E+7", stringVal );

  convert_doubleToString(-9.9999e6, 6, 1, stringVal);
  TEST_CHECK_STRING( "-1.E+7", stringVal );

  doubleVal = -9.88e-10;
  convert_doubleToString(doubleVal, 7, 0, stringVal);
  TEST_CHECK_STRING( "-1.E-9 ", stringVal);
  convert_doubleToString(doubleVal, 7, 1, stringVal);
  TEST_CHECK_STRING( " -1.E-9", stringVal);

  convert_doubleToString(+9.9999e-8, 6, 1, stringVal);
  TEST_CHECK_STRING( "1.0E-7", stringVal );

  convert_doubleToString(-9.9999e-8, 6, 1, stringVal);
  TEST_CHECK_STRING( "-1.E-7", stringVal );

  /**** Expect NaN strings ****///
  convert_doubleToString(9, -2, 1, stringVal);
  TEST_CHECK_STRING( "NaN", stringVal );

  convert_doubleToString(9, 0, 1, stringVal);
  TEST_CHECK_STRING( "NaN", stringVal );

  convert_doubleToString(-1e100, 7, 1, stringVal);
  TEST_CHECK_STRING( "NaN", stringVal );

  // this will fit in 7 characters
  convert_doubleToString(-9.e99, 7, 1, stringVal);
  TEST_CHECK_STRING( "-9.E+99", stringVal );

  // this will not fit due to rounding
  convert_doubleToString(-9.9e99, 7, 1, stringVal);
  TEST_CHECK_STRING( "NaN", stringVal );

  convert_doubleToString(-1.0, 3, 1, stringVal);
  TEST_CHECK_STRING( "-1.", stringVal );

  convert_doubleToString(-1.0, 2, 1, stringVal);
  TEST_CHECK_STRING( "NaN", stringVal );

  convert_doubleToString(1.0, 2, 1, stringVal);
  TEST_CHECK_STRING( "1.", stringVal );

  convert_doubleToString(1.0, 1, 1, stringVal);
  TEST_CHECK_STRING( "NaN", stringVal );
}

//---------------------------------------------------------------------------//
/* Used to test the convert_integerToString implementation */
void test_convert_integerToString(void *aimInfo)
{
  char stringVal[16];

  convert_integerToString(0, 2, 0, stringVal);
  TEST_CHECK_STRING( "0 ", stringVal );


  convert_integerToString(0, 2, 1, stringVal);
  TEST_CHECK_STRING( " 0", stringVal );


  convert_integerToString(-1, 3, 0, stringVal);
  TEST_CHECK_STRING( "-1 ", stringVal );


  convert_integerToString(-1, 3, 1, stringVal);
  TEST_CHECK_STRING( " -1", stringVal );


  convert_integerToString(-11, 3, 0, stringVal);
  TEST_CHECK_STRING( "-11", stringVal );


  convert_integerToString(-11, 3, 1, stringVal);
  TEST_CHECK_STRING( "-11", stringVal );


  convert_integerToString(99, 5, 0, stringVal);
  TEST_CHECK_STRING( "99   ", stringVal );


  convert_integerToString(99, 5, 1, stringVal);
  TEST_CHECK_STRING( "   99", stringVal );


  convert_integerToString(-99, 5, 0, stringVal);
  TEST_CHECK_STRING( "-99  ", stringVal );


  convert_integerToString(-99, 5, 1, stringVal);
  TEST_CHECK_STRING( "  -99", stringVal );


  convert_integerToString(9999999, 7, 0, stringVal);
  TEST_CHECK_STRING( "9999999", stringVal );


  convert_integerToString(9999999, 7, 1, stringVal);
  TEST_CHECK_STRING( "9999999", stringVal );


  convert_integerToString(9999999, 8, 0, stringVal);
  TEST_CHECK_STRING( "9999999 ", stringVal );


  convert_integerToString(9999999, 8, 1, stringVal);
  TEST_CHECK_STRING( " 9999999", stringVal );


  // Expect NaN strings
  convert_integerToString(9999999, 6, 1, stringVal);
  TEST_CHECK_STRING( "NaN", stringVal );


  convert_integerToString(-999999, 6, 1, stringVal);
  TEST_CHECK_STRING( "NaN", stringVal );


  convert_integerToString(9999999, 16, 1, stringVal);
  TEST_CHECK_STRING( "NaN", stringVal );


  convert_integerToString(9, -2, 1, stringVal);
  TEST_CHECK_STRING( "NaN", stringVal );


  convert_integerToString(9, 0, 1, stringVal);
  TEST_CHECK_STRING( "NaN", stringVal );
}

/* Used to test the string_toInteger implementation */
void test_string_toInteger(void *aimInfo)
{
  int intVal = 0;

  TEST_CHECK_SUCCESS( string_toInteger("0", &intVal) );
  TEST_CHECK_EQUAL_INT( 0, intVal );

  TEST_CHECK_SUCCESS( string_toInteger("1", &intVal) );
  TEST_CHECK_EQUAL_INT( 1, intVal );

  TEST_CHECK_SUCCESS( string_toInteger("10", &intVal) );
  TEST_CHECK_EQUAL_INT( 10, intVal );

  TEST_CHECK_SUCCESS( string_toInteger("  20", &intVal) );
  TEST_CHECK_EQUAL_INT( 20, intVal );

  TEST_CHECK_SUCCESS( string_toInteger("-30  ", &intVal) );
  TEST_CHECK_EQUAL_INT( -30, intVal );

  TEST_CHECK_SUCCESS( string_toInteger("3e2", &intVal) );
  TEST_CHECK_EQUAL_INT( 300, intVal );

  TEST_CHECK_STATUS( CAPS_BADVALUE , string_toInteger("", &intVal) );
  TEST_CHECK_STATUS( CAPS_BADVALUE , string_toInteger("foo", &intVal) );
  TEST_CHECK_STATUS( CAPS_NULLVALUE, string_toInteger(NULL, &intVal) );
}

//---------------------------------------------------------------------------//
/* Used to test the string_toDouble implementation */
void test_string_toDouble(void *aimInfo)
{
  double doubleVal = 0;

  TEST_CHECK_SUCCESS( string_toDouble("0.0", &doubleVal) );
  TEST_CHECK_EQUAL_DBL( 0.0, doubleVal );

  TEST_CHECK_SUCCESS( string_toDouble("1.2", &doubleVal) );
  TEST_CHECK_EQUAL_DBL( 1.2, doubleVal );

  TEST_CHECK_SUCCESS( string_toDouble("-10.7", &doubleVal) );
  TEST_CHECK_EQUAL_DBL( -10.7, doubleVal );

  TEST_CHECK_SUCCESS( string_toDouble("  20.9", &doubleVal) );
  TEST_CHECK_EQUAL_DBL( 20.9, doubleVal );

  TEST_CHECK_SUCCESS( string_toDouble("30e2  ", &doubleVal) );
  TEST_CHECK_EQUAL_DBL( 30e2, doubleVal );

  TEST_CHECK_STATUS( CAPS_BADVALUE , string_toDouble("", &doubleVal) );
  TEST_CHECK_STATUS( CAPS_BADVALUE , string_toDouble("foo", &doubleVal) );
  TEST_CHECK_STATUS( CAPS_NULLVALUE, string_toDouble(NULL, &doubleVal) );
}

//---------------------------------------------------------------------------//
/* Used to test the solveLU implementation */
void test_solveLU(void *aimInfo)
{
  int n;
  double *A, *b, *x;

  n = 3;
  A = (double *) EG_alloc(n*n*sizeof(double));
  b = (double *) EG_alloc(  n*sizeof(double));
  x = (double *) EG_alloc(  n*sizeof(double));

  A[0*n+0]=4; A[0*n+1]=2; A[0*n+2]=2;
  A[1*n+0]=2; A[1*n+1]=4; A[1*n+2]=6;
  A[2*n+0]=2; A[2*n+1]=6; A[2*n+2]=6;

  b[0]=1; b[1]=3; b[2]=2;

  TEST_CHECK_SUCCESS( solveLU(n, A, b, x) );

  TEST_CHECK_CLOSE( x[0],   0.1, 1e-10 );
  TEST_CHECK_CLOSE( x[1],  -0.5, 1e-10 );
  TEST_CHECK_CLOSE( x[2], 4./5., 1e-10 );

  EG_free(A);
  EG_free(b);
  EG_free(x);
}

//---------------------------------------------------------------------------//
/* Used to test JSON parser */
void test_jsonParser(void *aimInfo) {

  const char *keyWord = NULL;
  char *keyValue = NULL; // Freeable

  int numString = 0;
  char **stringArray = NULL;

  keyWord = "CG";
  TEST_CHECK_SUCCESS(search_jsonDictionary( "{\"CG\":[[1,3,4],\"ft\"]}", keyWord, &keyValue));
  TEST_CHECK_STRING( keyValue, "[[1,3,4],\"ft\"]");
  EG_free(keyValue); keyValue = NULL;

  keyWord = "CG";
  TEST_CHECK_SUCCESS(search_jsonDictionary( "{\"CG\":{\"num\":[[1,3],[4,5]],\"ft\":10}}", keyWord, &keyValue));
  TEST_ASSERT(keyValue != NULL);
  TEST_CHECK_STRING( keyValue, "{\"num\":[[1,3],[4,5]],\"ft\":10}");
  AIM_FREE(keyValue);

  TEST_CHECK_SUCCESS(json_parseTuple("[1,3,4]", &numString, &stringArray));
  TEST_ASSERT(stringArray != NULL);
  //for (int i = 0; i< numString; i++) printf("%s\n", stringArray[i]);

  TEST_CHECK_EQUAL_INT( numString,  3);
  TEST_CHECK_STRING( stringArray[0], "1");
  TEST_CHECK_STRING( stringArray[1], "3");
  TEST_CHECK_STRING( stringArray[2], "4");

  TEST_CHECK_SUCCESS(string_freeArray(numString, &stringArray));

  TEST_CHECK_SUCCESS(json_parseTuple("[[1, 3, 4],\"inches\"]", &numString, &stringArray));
  TEST_ASSERT(stringArray != NULL);
  //for (int i = 0; i< numString; i++) printf("%s\n", stringArray[i]);
  TEST_CHECK_EQUAL_INT( numString,  2);
  TEST_CHECK_STRING( stringArray[0], "[1, 3, 4]");
  TEST_CHECK_STRING( stringArray[1], "inches");

  TEST_CHECK_SUCCESS(string_freeArray(numString, &stringArray));

  TEST_CHECK_SUCCESS(json_parseTuple("[[1, 3, 4], (\"a\",\"b\"), \"m\"]", &numString, &stringArray));
  TEST_ASSERT(stringArray != NULL);
  //for (int i = 0; i< numString; i++) printf("%s\n", stringArray[i]);
  TEST_CHECK_EQUAL_INT( numString,  3);
  TEST_CHECK_STRING( stringArray[0], "[1, 3, 4]");
  TEST_CHECK_STRING( stringArray[1], "(\"a\",\"b\")");
  TEST_CHECK_STRING( stringArray[2], "m");

  TEST_CHECK_SUCCESS(string_freeArray(numString, &stringArray));

  TEST_CHECK_SUCCESS(json_parseTuple("[[[1, 2], (3,4)], ({5:6, 7:8}, [9, 10])]", &numString, &stringArray));
  TEST_ASSERT(stringArray != NULL);
  //for (int i = 0; i< numString; i++) printf("%s\n", stringArray[i]);
  TEST_CHECK_EQUAL_INT( numString,  2);
  TEST_CHECK_STRING( stringArray[0], "[[1, 2], (3,4)]");
  TEST_CHECK_STRING( stringArray[1], "({5:6, 7:8}, [9, 10])");

  TEST_CHECK_SUCCESS(string_freeArray(numString, &stringArray));
}


TEST_LIST = {
    { "getline", test_getline },
    { "convert_doubleToString", test_convert_doubleToString },
    { "convert_integerToString", test_convert_integerToString },
    { "solveLU", test_solveLU },
    { "jsonParser", test_jsonParser },
    { NULL, NULL }
};
