/*
 ************************************************************************
 *                                                                      *
 * udfPatch -- make a patch to fill a loop of Edges                     *
 *                                                                      *
 *            Written by John Dannenhoffer @ Syracuse University        *
 *            Patterned after code written by Bob Haimes  @ MIT         *
 *                                                                      *
 ************************************************************************
 */

/*
 * Copyright (C) 2011/2025  John F. Dannenhoffer, III (Syracuse University)
 *
 * This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *     MA  02110-1301  USA
 */

#define NUMUDPARGS       2
#define NUMUDPINPUTBODYS 1
#include "udpUtilities.h"

/* shorthands for accessing argument values and velocities */
#define EDGELIST(    IUDP,I)  ((int    *) (udps[IUDP].arg[0].val))[I]
#define EDGELIST_SIZ(IUDP)    ( int     )  udps[IUDP].arg[0].size
#define NSMTH(       IUDP)    ((int    *) (udps[IUDP].arg[1].val))[0]
#define NSMTH_SIZ(   IUDP)    ( int     )  udps[IUDP].arg[1].size

/* data about possible arguments */
static char  *argNames[NUMUDPARGS] = {"edgelist", "nsmth", };
static int    argTypes[NUMUDPARGS] = {ATTRINT,    ATTRINT, };
static int    argIdefs[NUMUDPARGS] = {0,          0,       };
static double argDdefs[NUMUDPARGS] = {0.,         0.,      };

/* get utility routines: udpErrorStr, udpInitialize, udpReset, udpSet,
                         udpGet, udpVel, udpClean, udpMesh */
#include "udpUtilities.c"

/* prototype for function defined below */

#include "OpenCSM.h"


/*
 ************************************************************************
 *                                                                      *
 *   udpExecute - execute the primitive                                 *
 *                                                                      *
 ************************************************************************
 */

int
udpExecute(ego  emodel,                 /* (in)  Model containing Body */
           ego  *ebody,                 /* (out) Body pointer */
           int  *nMesh,                 /* (out) number of associated meshes */
           char *string[])              /* (out) error message */
{
    int     status = EGADS_SUCCESS;

    int     oclass, mtype, nchild, nchild2, *senses, *senses2, i, im1, j, foo;
    int     nedge, myNedge, ibeg, ncorn=0, one=1, ntemp, *stemps, attrType, attrLen;
    int     *header_s=NULL, *header_e=NULL, *header_n=NULL, *header_w=NULL, *header=NULL;
    int     periodic, imax, jmax, ismth;
    CINT    *tempIlist;
    double  trange[2], limits[4], xyz[18], *xyztemp=NULL, omega=1.0;
    double  data[4], dataim1[18], datai[18], *dotprod=NULL;
    double  *rdata_s=NULL, *rdata_e=NULL, *rdata_n=NULL, *rdata_w=NULL, *rdata=NULL;
    CDOUBLE *tempRlist;
    char    message[1024];
    CCHAR   *tempClist;
    ego     context, eref, *ebodys, *eedges=NULL, *myEdges=NULL, eloop=NULL;
    ego     *echilds, *echilds2, newSurface, newFace, eshell, *etemps;
    ego     bspline_s=NULL, bspline_e=NULL, bspline_n=NULL, bspline_w=NULL;
    ego     bspline_old, bspline_new, newEdges[4], ruleNodes[2], newLoop;
    udp_T   *udps = *Udps;

    ROUTINE(udpExecute);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    printf("udpExecute(emodel=%llx)\n", (long long)emodel);
    for (i = 0; i < EDGELIST_SIZ(0); i++) {
        printf("edgelist[%2d] = %d\n", i, EDGELIST(0,i));
    }
    printf("nsmth = %d\n", NSMTH(0));
#endif

    /* default return values */
    *ebody  = NULL;
    *nMesh  = 0;
    *string = NULL;

    message[0] = '\0';

    /* check that Model was input that contains one Body */
    status = EG_getTopology(emodel, &eref, &oclass, &mtype,
                            data, &nchild, &ebodys, &senses);
    CHECK_STATUS(EG_getTopology);

    if (oclass != MODEL) {
        snprintf(message, 1024, "expecting a Model\n");
        status = EGADS_NOTMODEL;
        goto cleanup;
    } else if (nchild != 1) {
        snprintf(message, 1024, "expecting Model to contain one Body (not %d)\n", nchild);
        status = EGADS_NOTBODY;
        goto cleanup;
    }

    status = EG_getContext(emodel, &context);
    CHECK_STATUS(EG_getContext);

    /* get the number of Edges */
    status = EG_getBodyTopos(ebodys[0], NULL, EDGE, &nedge, &eedges);
    CHECK_STATUS(EG_getBodyTopos);

    SPLINT_CHECK_FOR_NULL(eedges);

    echilds = eedges;                   // needed to avoid a clang warning

    /* check arguments */
    if (NSMTH_SIZ(0) > 1) {
        snprintf(message, 1024, "\"nsmth\" should be a scalar");
        status = EGADS_RANGERR;
        goto cleanup;
    }

    for (i = 0; i < EDGELIST_SIZ(0); i++) {
        if (EDGELIST(0,i) < 1 || EDGELIST(0,i) > nedge) {
            snprintf(message, 1024, "edgelist[%d]=%d is out of bounds", i, EDGELIST(0,i));
            status = EGADS_INDEXERR;
            goto cleanup;
        }
    }

    /* cache copy of arguments for future use */
    status = cacheUdp(NULL);
    CHECK_STATUS(cacheUdp);

#ifdef DEBUG
    for (i = 0; i < EDGELIST_SIZ(0); i++) {
        printf("edgelist[%2d,%2d] = %d\n", i, numUdp, EDGELIST(numUdp,i));
    }
    printf("nsmth[      %2d] = %d\n", numUdp, NSMTH(numUdp));
#endif

    /* make a Loop of the Edges in edgelist */
    MALLOC(myEdges, ego,    EDGELIST_SIZ(0));
    MALLOC(dotprod, double, EDGELIST_SIZ(0));

    /* start the loop at the beginning of edgelist unless we find
       a Node that has the attribute _patchCorn=2 */
    ibeg = 0;

    for (i = 0; i < EDGELIST_SIZ(0); i++) {
        status = EG_getTopology(eedges[EDGELIST(0,i)-1], &eref, &oclass, &mtype,
                                data, &nchild, &echilds, &senses);
        CHECK_STATUS(EG_getTopology);

        status = EG_attributeRet(echilds[0], "_patchCorn", &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status != EGADS_SUCCESS) {
            continue;
        } else if (attrType == ATTRINT && tempIlist[0] == 2) {
            ibeg = i;
            break;
        } else if (attrType == ATTRREAL && fabs(tempRlist[0]-2) < EPS06) {
            ibeg = i;
            break;
        }
    }

    /* we have to start the Loop at a corner */
    for (; ibeg < EDGELIST_SIZ(0); ibeg++) {

        myNedge = 0;
        for (i = 0; i < EDGELIST_SIZ(0); i++) {
            j = (i+ibeg) % EDGELIST_SIZ(0);
            myEdges[myNedge++] = eedges[EDGELIST(0,j)-1];
        }

        if (eloop != NULL) {
            status = EG_deleteObject(eloop);
            CHECK_STATUS(EG_deleteObject);
        }

        status = EG_makeLoop(myNedge, myEdges, NULL, 0, &eloop);
        CHECK_STATUS(EG_makeLoop);

        if (status > 0) {
            snprintf(message, 1024, "edgelist does not make a closed loop");
            status = EGADS_NOTTOPO;
            goto cleanup;
        }

        SPLINT_CHECK_FOR_NULL(eloop);

        /* find the normalized dot product at all the Nodes to find the corners */
        ncorn = 0;

        status = EG_getTopology(eloop, &eref, &oclass, &mtype,
                                data, &nchild, &echilds, &senses);
        CHECK_STATUS(EG_getTopology);

        /* dotprod[i] is dot product between i-1 and i */
        for (i = 0; i < nchild; i++) {
            im1 = (i-1+nchild) % nchild;

            status = EG_getTopology(echilds[im1], &eref, &oclass, &mtype,
                                    data, &nchild2, &echilds2, &senses2);
            CHECK_STATUS(EG_getTopology);

            if (senses[im1] == SFORWARD) {
                status = EG_evaluate(echilds[im1], &data[1], dataim1);
                CHECK_STATUS(EG_evaluate);
            } else {
                status = EG_evaluate(echilds[im1], &data[0], dataim1);
                CHECK_STATUS(EG_evaluate);

                dataim1[3] *= -1;
                dataim1[4] *= -1;
                dataim1[5] *= -1;
            }

            status = EG_getTopology(echilds[i], &eref, &oclass, &mtype,
                                    data, &nchild2, &echilds2, &senses2);
            CHECK_STATUS(EG_getTopology);

            if (senses[i] == SFORWARD) {
                status = EG_evaluate(echilds[i], &data[0], datai);
                CHECK_STATUS(EG_evaluate);
            } else {
                status = EG_evaluate(echilds[i], &data[1], datai);
                CHECK_STATUS(EG_evaluate);

                datai[3] *= -1;
                datai[4] *= -1;
                datai[5] *= -1;
            }

            /*@-evalorder@*/
            dotprod[i] = (   dataim1[3] * datai[  3] + dataim1[4] * datai[  4] + dataim1[5] * datai[  5])
                       /sqrt(dataim1[3] * dataim1[3] + dataim1[4] * dataim1[4] + dataim1[5] * dataim1[5])
                       /sqrt(datai[  3] * datai[  3] + datai[  4] * datai[  4] + datai[  5] * dataim1[5]);
            /*@+evalorder@*/

            if (dotprod[i] < 0.90) {
                ncorn++;
            } else if (i == 0) {
                break;        // start loop at another Edge
            }
        }

        /* this condition happens when the previous Loop started at a corner
           (which is a success) */
        if (i != 0) break;
    }

    /* add corners to all Nodes with the _patchCorn attribute */
    for (i = 0; i < nchild; i++) {
        if (dotprod[i] < 0.90) continue;

        status = EG_getTopology(echilds[i], &eref, &oclass, &mtype,
                                data, &ntemp, &etemps, &stemps);
        CHECK_STATUS(EG_getTopology);

        if (senses[i] == SFORWARD) {
            status = EG_attributeRet(etemps[0], "_patchCorn", &attrType, &attrLen,
                                     &tempIlist, &tempRlist, &tempClist);
            if (status == EGADS_SUCCESS) {
                dotprod[i] = 0;
                ncorn++;
            }
        } else {
            status = EG_attributeRet(etemps[1], "_patchCorn", &attrType, &attrLen,
                                     &tempIlist, &tempRlist, &tempClist);
            if (status == EGADS_SUCCESS) {
                dotprod[i] = 0;
                ncorn++;
            }
        }
    }

    if (ncorn != 3 && ncorn != 4) {
        snprintf(message, 1024, "there should be 3 or 4 corners but %d were found", ncorn);
        status = EGADS_CONSTERR;
        goto cleanup;
    }

#ifdef DEBUG
    for (i = 0; i < nchild; i++) {
        printf("dotprod[%d]=%10.5f\n", i, dotprod[i]);
    }

    /* print out Bspline info associated with the Edges in the Loop */
    for (i = 0; i < nchild; i++) {
        status = EG_convertToBSpline(echilds[i], &bspline_s);
        CHECK_STATUS(EG_convertToBSpline);

        SPLINT_CHECK_FOR_NULL(bspline_s);

        printf("Edge %d in original loop (with sense=%d)\n", i, senses[i]);
        ocsmPrintEgo(bspline_s);

        status = EG_deleteObject(bspline_s);
        CHECK_STATUS(EG_deleteObject);
    }
#endif

    /* create the Bsplines for the 4 sides of the patch */
    i = 0;

    /* south side */
    status = EG_convertToBSpline(echilds[i], &bspline_s);
    CHECK_STATUS(EG_convertToBSpline);
    i++;

    for (; i < nchild; i++) {
        if (dotprod[i] > 0.90) {
            bspline_old = bspline_s;

            status = EG_convertToBSpline(echilds[i], &bspline_new);
            CHECK_STATUS(EG_convertToBSpline);

            SPLINT_CHECK_FOR_NULL(bspline_old);
            SPLINT_CHECK_FOR_NULL(bspline_new);

            status = EG_mergeBSplineCurves(bspline_old, bspline_new, &bspline_s);
            CHECK_STATUS(EG_mergeBsplineCurves);

            status = EG_deleteObject(bspline_old);
            CHECK_STATUS(EG_deleteObject);

            status = EG_deleteObject(bspline_new);
            CHECK_STATUS(EG_deleteObject);
        } else {
            status = EG_convertToBSpline(echilds[i], &bspline_e);
            CHECK_STATUS(EG_convertToBSpline);
            i++;

            break;
        }
    }

    /* east side */
    for (; i < nchild; i++) {
        if (dotprod[i] > 0.90) {
            bspline_old = bspline_e;

            status = EG_convertToBSpline(echilds[i], &bspline_new);
            CHECK_STATUS(EG_convertToBSpline);

            SPLINT_CHECK_FOR_NULL(bspline_old);
            SPLINT_CHECK_FOR_NULL(bspline_new);

            status = EG_mergeBSplineCurves(bspline_old, bspline_new, &bspline_e);
            CHECK_STATUS(EG_mergeBsplineCurves);

            status = EG_deleteObject(bspline_old);
            CHECK_STATUS(EG_deleteObject);

            status = EG_deleteObject(bspline_new);
            CHECK_STATUS(EG_deleteObject);
        } else {
            status = EG_convertToBSpline(echilds[i], &bspline_n);
            CHECK_STATUS(EG_convertToBSpline);
            i++;

            break;
        }
    }

    /* north side */
    for (; i < nchild; i++) {
        if (dotprod[i] > 0.90) {
            bspline_old = bspline_n;

            status = EG_convertToBSpline(echilds[i], &bspline_new);
            CHECK_STATUS(EG_convertToBSpline);

            SPLINT_CHECK_FOR_NULL(bspline_old);
            SPLINT_CHECK_FOR_NULL(bspline_new);

            status = EG_mergeBSplineCurves(bspline_old, bspline_new, &bspline_n);
            CHECK_STATUS(EG_mergeBsplineCurves);

            status = EG_deleteObject(bspline_old);
            CHECK_STATUS(EG_deleteObject);

            status = EG_deleteObject(bspline_new);
            CHECK_STATUS(EG_deleteObject);
        } else {
            if (ncorn == 4) {
                status = EG_convertToBSpline(echilds[i], &bspline_w);
                CHECK_STATUS(EG_convertToBSpline);
                i++;
            }

            break;
        }
    }

    /* west side */
    for (; i < nchild; i++) {
        if (ncorn != 4) break;

        if (dotprod[i] > 0.90) {
            bspline_old = bspline_w;

            status = EG_convertToBSpline(echilds[i], &bspline_new);
            CHECK_STATUS(EG_convertToBSpline);

            SPLINT_CHECK_FOR_NULL(bspline_old);
            SPLINT_CHECK_FOR_NULL(bspline_new);

            status = EG_mergeBSplineCurves(bspline_old, bspline_new, &bspline_w);
            CHECK_STATUS(EG_mergeBsplineCurves);

            status = EG_deleteObject(bspline_old);
            CHECK_STATUS(EG_deleteObject);

            status = EG_deleteObject(bspline_new);
            CHECK_STATUS(EG_deleteObject);
        } else {
            break;
        }
    }

    /* get data associated with the bsplines */
    SPLINT_CHECK_FOR_NULL(bspline_s);
    status = EG_getGeometry(bspline_s, &oclass, &mtype, &eref,
                            &header_s, &rdata_s);
    CHECK_STATUS(EG_getGeometry);

#ifdef DEBUG
    printf("after merging: south\n");
    ocsmPrintEgo(bspline_s);
#endif

    SPLINT_CHECK_FOR_NULL(bspline_e);
    status = EG_getGeometry(bspline_e, &oclass, &mtype, &eref,
                            &header_e, &rdata_e);
    CHECK_STATUS(EG_getGeometry);

#ifdef DEBUG
    printf("after merging: west\n");
    ocsmPrintEgo(bspline_w);
#endif

    SPLINT_CHECK_FOR_NULL(bspline_n);
    status = EG_getGeometry(bspline_n, &oclass, &mtype, &eref,
                            &header_n, &rdata_n);
    CHECK_STATUS(EG_getGeometry);

#ifdef DEBUG
    printf("after merging: north\n");
    ocsmPrintEgo(bspline_n);
#endif

    if (ncorn == 4) {
        SPLINT_CHECK_FOR_NULL(bspline_w);
        status = EG_getGeometry(bspline_w, &oclass, &mtype, &eref,
                                &header_w, &rdata_w);
        CHECK_STATUS(EG_getGeometry);

#ifdef DEBUG
        printf("after merging: west\n");
        ocsmPrintEgo(bspline_w);
#endif
    }

    /* make a Loop from these bsplines */

    /* south */
    SPLINT_CHECK_FOR_NULL(header_s);
    SPLINT_CHECK_FOR_NULL(rdata_s );

    trange[0] = rdata_s[0];
    trange[1] = rdata_s[header_s[2]];

    status = EG_evaluate(bspline_s, &trange[0], xyz);
    CHECK_STATUS(EG_evaluate);

    status = EG_makeTopology(context, NULL, NODE, 0,
                             xyz, 0, NULL, NULL, &ruleNodes[0]);
    CHECK_STATUS(EG_makeTopology);

    status = EG_evaluate(bspline_s, &trange[1], xyz);
    CHECK_STATUS(EG_evaluate);

    status = EG_makeTopology(context, NULL, NODE, 0,
                             xyz, 0, NULL, NULL, &ruleNodes[1]);
    CHECK_STATUS(EG_makeTopology);

    foo = SFORWARD;
    status = EG_makeTopology(context, bspline_s, EDGE, TWONODE,
                             trange, 2, ruleNodes, &foo, &newEdges[0]);
    CHECK_STATUS(EG_makeTopology);

    /* east */
    SPLINT_CHECK_FOR_NULL(header_e);
    SPLINT_CHECK_FOR_NULL(rdata_e );

    trange[0] = rdata_e[0];
    trange[1] = rdata_e[header_e[2]];

    status = EG_evaluate(bspline_e, &trange[0], xyz);
    CHECK_STATUS(EG_evaluate);

    status = EG_makeTopology(context, NULL, NODE, 0,
                             xyz, 0, NULL, NULL, &ruleNodes[0]);
    CHECK_STATUS(EG_makeTopology);

    status = EG_evaluate(bspline_e, &trange[1], xyz);
    CHECK_STATUS(EG_evaluate);

    status = EG_makeTopology(context, NULL, NODE, 0,
                             xyz, 0, NULL, NULL, &ruleNodes[1]);
    CHECK_STATUS(EG_makeTopology);

    foo = SFORWARD;
    status = EG_makeTopology(context, bspline_e, EDGE, TWONODE,
                             trange, 2, ruleNodes, &foo, &newEdges[1]);
    CHECK_STATUS(EG_makeTopology);

    /* north */
    SPLINT_CHECK_FOR_NULL(header_n);
    SPLINT_CHECK_FOR_NULL(rdata_n );

    trange[0] = rdata_n[0];
    trange[1] = rdata_n[header_n[2]];

    status = EG_evaluate(bspline_n, &trange[0], xyz);
    CHECK_STATUS(EG_evaluate);

    status = EG_makeTopology(context, NULL, NODE, 0,
                             xyz, 0, NULL, NULL, &ruleNodes[0]);
    CHECK_STATUS(EG_makeTopology);

    status = EG_evaluate(bspline_n, &trange[1], xyz);
    CHECK_STATUS(EG_evaluate);

    status = EG_makeTopology(context, NULL, NODE, 0,
                             xyz, 0, NULL, NULL, &ruleNodes[1]);
    CHECK_STATUS(EG_makeTopology);

    foo = SFORWARD;
    status = EG_makeTopology(context, bspline_n, EDGE, TWONODE,
                             trange, 2, ruleNodes, &foo, &newEdges[2]);
    CHECK_STATUS(EG_makeTopology);

    /* west */
    if (ncorn == 4) {
        SPLINT_CHECK_FOR_NULL(bspline_w);

        SPLINT_CHECK_FOR_NULL(header_w );
        SPLINT_CHECK_FOR_NULL(rdata_w  );

        trange[0] = rdata_w[0];
        trange[1] = rdata_w[header_w[2]];

        status = EG_evaluate(bspline_w, &trange[0], xyz);
        CHECK_STATUS(EG_evaluate);

        status = EG_makeTopology(context, NULL, NODE, 0,
                                 xyz, 0, NULL, NULL, &ruleNodes[0]);
        CHECK_STATUS(EG_makeTopology);

        status = EG_evaluate(bspline_w, &trange[1], xyz);
        CHECK_STATUS(EG_evaluate);

        status = EG_makeTopology(context, NULL, NODE, 0,
                                 xyz, 0, NULL, NULL, &ruleNodes[1]);
        CHECK_STATUS(EG_makeTopology);

        foo = SFORWARD;
        status = EG_makeTopology(context, bspline_w, EDGE, TWONODE,
                                 trange, 2, ruleNodes, &foo, &newEdges[3]);
        CHECK_STATUS(EG_makeTopology);
    }

    /* make a Loop from these Edges and then make the secret call
       to EG_isoCline to make a Surface */
    status = EG_makeLoop(ncorn, newEdges, NULL, 0, &newLoop);
    CHECK_STATUS(EG_makeLoop);

    status = EG_isoCline(newLoop, 0, 0, &newSurface);
    CHECK_STATUS(EG_isoCline);

    /* smooth the control points if required */
    if (header != NULL) {EG_free(header);   header = NULL;}
    if (rdata  != NULL) {EG_free(rdata );   rdata  = NULL;}

    status = EG_getGeometry(newSurface, &oclass, &mtype, &eref,
                            &header, &rdata);
    CHECK_STATUS(EG_getGeometry);

    SPLINT_CHECK_FOR_NULL(header);
    SPLINT_CHECK_FOR_NULL(rdata );

    imax = header[2];
    jmax = header[5];

    MALLOC(xyztemp, double, header[3]+header[6]+4*imax*jmax);

#define XDATA(I,J)  rdata[header[3]+header[6]+3*((I)+(J)*imax)  ]
#define YDATA(I,J)  rdata[header[3]+header[6]+3*((I)+(J)*imax)+1]
#define ZDATA(I,J)  rdata[header[3]+header[6]+3*((I)+(J)*imax)+2]

#define XTEMP(I,J)  xyztemp[header[3]+header[6]+3*((I)+(J)*imax)  ]
#define YTEMP(I,J)  xyztemp[header[3]+header[6]+3*((I)+(J)*imax)+1]
#define ZTEMP(I,J)  xyztemp[header[3]+header[6]+3*((I)+(J)*imax)+2]

    for (ismth = 0; ismth < NSMTH(0); ismth++) {
        for (j = 1; j < jmax-1; j++) {
            for (i = 1; i < imax-1; i++) {
                XTEMP(i,j) = ( XDATA(i-1,j  ) + XDATA(i+1,j  )
                             + XDATA(i  ,j-1) + XDATA(i  ,j+1)) / 4;
                YTEMP(i,j) = ( YDATA(i-1,j  ) + YDATA(i+1,j  )
                             + YDATA(i  ,j-1) + YDATA(i  ,j+1)) / 4;
                ZTEMP(i,j) = ( ZDATA(i-1,j  ) + ZDATA(i+1,j  )
                             + ZDATA(i  ,j-1) + ZDATA(i  ,j+1)) / 4;
            }
        }

        for (j = 1; j < jmax-1; j++) {
            for (i = 1; i < imax-1; i++) {
                XDATA(i,j) = XDATA(i,j) + omega * (XTEMP(i,j) - XDATA(i,j));
                YDATA(i,j) = YDATA(i,j) + omega * (YTEMP(i,j) - YDATA(i,j));
                ZDATA(i,j) = ZDATA(i,j) + omega * (ZTEMP(i,j) - ZDATA(i,j));
            }
        }
    }

    FREE(xyztemp);

    status = EG_deleteObject(newSurface);
    CHECK_STATUS(EG_deleteObject);

    status = EG_makeGeometry(context, oclass, mtype, eref,
                             header, rdata, &newSurface);
    CHECK_STATUS(EG_makeGeometry);

    EG_free(header);   header = NULL;
    EG_free(rdata );   rdata  = NULL;

    /* make a Face and SheetBody */
    status = EG_getRange(newSurface, limits, &periodic);
    CHECK_STATUS(EG_getRange);

    status = EG_makeFace(newSurface, SFORWARD, limits, &newFace);
    CHECK_STATUS(EG_makeFace);

    status = EG_makeTopology(context, NULL, SHELL, OPEN,
                             NULL, 1, &newFace, NULL, &eshell);
    CHECK_STATUS(EG_makeTopology);

    status = EG_makeTopology(context, NULL, BODY, SHEETBODY,
                             NULL, 1, &eshell, NULL, ebody);
    CHECK_STATUS(EG_makeTopology);

    SPLINT_CHECK_FOR_NULL(*ebody);

#ifdef DEBUG
    printf("ebody:\n");
    ocsmPrintEgo(*ebody);
#endif

    /* tell OpenCSM to put _body, _brch, and Branch Attributes on the Faces */
    status = EG_attributeAdd(*ebody, "__markFaces__", ATTRINT, 1,
                             &one, NULL, NULL);
    CHECK_STATUS(EG_attributeAdd);

    /* set the output value */

    /* the copy of the Body that was annotated is returned */
    udps[numUdp].ebody = *ebody;

cleanup:
    if (eedges   != NULL) EG_free(eedges  );
    if (header   != NULL) EG_free(header  );
    if (header_s != NULL) EG_free(header_s);
    if (header_n != NULL) EG_free(header_n);
    if (header_w != NULL) EG_free(header_w);
    if (header_e != NULL) EG_free(header_e);
    if (rdata    != NULL) EG_free(rdata   );
    if (rdata_s  != NULL) EG_free(rdata_s );
    if (rdata_n  != NULL) EG_free(rdata_n );
    if (rdata_w  != NULL) EG_free(rdata_w );
    if (rdata_e  != NULL) EG_free(rdata_e );

    if (bspline_s != NULL) (void) EG_deleteObject(bspline_s);
    if (bspline_e != NULL) (void) EG_deleteObject(bspline_e);
    if (bspline_n != NULL) (void) EG_deleteObject(bspline_n);
    if (bspline_w != NULL) (void) EG_deleteObject(bspline_w);

    FREE(myEdges);
    FREE(dotprod);

    if (strlen(message) > 0) {
        MALLOC(*string, char, 1024);
        strncpy(*string, message, 1023);
    } else if (status != EGADS_SUCCESS) {
        *string = udpErrorStr(status);
    }

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   udpSensitivity - return sensitivity derivatives for the "real" argument *
 *                                                                      *
 ************************************************************************
 */

int
udpSensitivity(ego    ebody,            /* (in)  Body pointer */
   /*@unused@*/int    npnt,             /* (in)  number of points */
   /*@unused@*/int    entType,          /* (in)  OCSM entity type */
   /*@unused@*/int    entIndex,         /* (in)  OCSM entity index (bias-1) */
   /*@unused@*/double uvs[],            /* (in)  parametric coordinates for evaluation */
   /*@unused@*/double vels[])           /* (out) velocities */
{
    int iudp, judp;

    ROUTINE(udpSensitivity);

    /* --------------------------------------------------------------- */

    /* check that ebody matches one of the ebodys */
    iudp = 0;
    for (judp = 1; judp <= numUdp; judp++) {
        if (ebody == udps[judp].ebody) {
            iudp = judp;
            break;
        }
    }
    if (iudp <= 0) {
        return EGADS_NOTMODEL;
    }

    /* this routine is not written yet */
    return EGADS_NOLOAD;
}
