/*
 *      EGADS: Electronic Geometry Aircraft Design System
 *
 *             BSpline Fit and scribe Functions
 *
 *      Copyright 2011-2026, Massachusetts Institute of Technology
 *      Licensed under The GNU Lesser General Public License, version 2.1
 *      See http://www.opensource.org/licenses/lgpl-2.1.php
 *
 */


#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "egads.h"

#define CROSS(a,b,c)      a[0] = (b[1]*c[2]) - (b[2]*c[1]);\
                          a[1] = (b[2]*c[0]) - (b[0]*c[2]);\
                          a[2] = (b[0]*c[1]) - (b[1]*c[0])
#define DOT(a,b)         (a[0]*b[0] + a[1]*b[1] + a[2]*b[2])

typedef struct {
  double xyz[3];
  ego    node;
} newNode;

extern /*@kept@*/ /*@null@*/ egObject *EG_context( const ego object );
extern int EG_outLevel( const ego object );
extern int EG_sameThread( const ego object );
extern int EG_spline1dFit( int endx, int imaxx, const double *xyz,
                           const double *knots, double tol, int *header,
                           double **rvec );
extern int EG_spline1dPCrv( ego context, int endx, int imax, const double *xy,
                            const double *knots, double tol, ego *ecurv );
extern int EG_spline2dAppx( ego context,     int    endc,
                            /*@null@*/ const double *uknot,
                            /*@null@*/ const double *vknot,
                            /*@null@*/ const int    *vdata,
                            /*@null@*/ const double *wesT,
                            /*@null@*/ const double *easT,
                            /*@null@*/ const double *south,
                            /*@null@*/       double *snor,
                            /*@null@*/ const double *north,
                            /*@null@*/       double *nnor, int imax, int jmax,
                            const double *xyz, double tol,
                            ego *esurf );


int
EG_adjustCPs(const ego body, const ego face, double *CPs, ego *newBody,
             ego *newFace)
{
  int    i, j, k, stat, outLevel, oclass, mtype, nLoop, nEdge, iper, faceOr;
  int    *senses, *fsense, *header;
  double t, result[6], uvbox[4], trange[2], *rvec, *cntrl;
  ego    context, surf, ref, newSurf, loop, replace[2], *faces, *loops, *edges;

  *newBody = *newFace = NULL;
  if  (body == NULL)               return EGADS_NULLOBJ;
  if  (body->magicnumber != MAGIC) return EGADS_NOTOBJ;
  if  (body->oclass != BODY)       return EGADS_NOTBODY;
  if ((body->mtype != SHEETBODY) &&
      (body->mtype != SOLIDBODY))  return EGADS_NOTTOPO;
  if  (body->blind == NULL)        return EGADS_NODATA;
  if  (face == NULL)               return EGADS_NULLOBJ;
  if  (face->magicnumber != MAGIC) return EGADS_NOTOBJ;
  if  (face->oclass != FACE)       return EGADS_NOTTOPO;
  if  (face->blind == NULL)        return EGADS_NODATA;
  if  (CPs == NULL)                return EGADS_NODATA;
  outLevel = EG_outLevel(body);
  context  = EG_context(body);
  if  (context == NULL)            return EGADS_NOTCNTX;

  /* check the Topology */
  stat = EG_indexBodyTopo(body, face);
  if (stat < EGADS_SUCCESS) {
    if (outLevel > 0)
      printf(" EGADS Error: Face not in Body indexBodyTopo %d (EG_adjustCPs)!\n",
             stat);
    return stat;
  }
  stat = EG_getTopology(face, &surf, &oclass, &faceOr, uvbox, &nLoop, &loops,
                        &fsense);
  if (stat != EGADS_SUCCESS) {
    if (outLevel > 0)
      printf(" EGADS Error: getTopology on Face = %d (EG_adjustCPs)!\n", stat);
    return stat;
  }
  if (nLoop != 1) {
    if (outLevel > 0)
      printf(" EGADS Error: Face has %d Loops (EG_adjustCPs)!\n", nLoop);
    return EGADS_TOPOERR;
  }
  if (surf->mtype != BSPLINE) {
    if (outLevel > 0)
      printf(" EGADS Error: Surf Geometry not a BSpline (EG_adjustCPs)!\n");
    return EGADS_GEOMERR;
  }
  stat = EG_getTopology(loops[0], &ref, &oclass, &mtype, NULL, &nEdge, &edges,
                        &senses);
  if (stat != EGADS_SUCCESS) {
    if (outLevel > 0)
      printf(" EGADS Error: getTopology on Loop = %d (EG_adjustCPs)!\n", stat);
    return stat;
  }
  if (nEdge != 4) {
    if (outLevel > 0)
      printf(" EGADS Error: Loop has %d Edges (EG_adjustCPs)!\n", nEdge);
    return EGADS_TOPOERR;
  }
  /* is the center of the Edge (via the PCurve) on an IsoCline? */
  for (j = i = 0; i < 4; i++) {
    stat = EG_getRange(edges[i], trange, &iper);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" EGADS Error: getRange on Edge %d = %d (EG_adjustCPs)!\n",
               i+1, stat);
      return stat;
    }
    t    = trange[0] + (trange[1] - trange[0])/3.0;
    stat = EG_evaluate(edges[i+4], &t, result);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" EGADS Error: evaluate on PCurve 1/3 %d = %d (EG_adjustCPs)!\n",
               i+1, stat);
      return stat;
    }
    if ((fabs(result[2]) > 1.e-10) && (fabs(result[3]) > 1.e-10)) j++;
/*  printf(" EGADS Info:  PCurve %d 1 -- dUV/dt = %le %le (EG_adjustCPs)!\n",
           i+1, result[2], result[3]);  */
    t    = trange[0] + 2.0*(trange[1] - trange[0])/3.0;
    stat = EG_evaluate(edges[i+4], &t, result);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" EGADS Error: evaluate on PCurve 2/3 %d = %d (EG_adjustCPs)!\n",
               i+1, stat);
      return stat;
    }
    if ((fabs(result[2]) > 1.e-10) && (fabs(result[3]) > 1.e-10)) j++;
/*  printf(" EGADS Info:  PCurve %d 2 -- dUV/dt = %le %le (EG_adjustCPs)!\n",
           i+1, result[2], result[3]);  */
  }
  if (j != 0) {
    if (outLevel > 0)
      printf(" EGADS Error: PCurve(s) not on IsoCline(s) (EG_adjustCPs)!\n");
    return EGADS_NOTORTHO;
  }

  /* get the Geometry */
  stat = EG_getGeometry(surf, &oclass, &mtype, &ref, &header, &rvec);
  if (stat != EGADS_SUCCESS) {
    if (outLevel > 0)
      printf(" EGADS Error: getGeometry on Surf = %d (EG_adjustCPs)!\n", stat);
    return stat;
  }

  /* refill the control points */
  cntrl = &rvec[header[3]+header[6]];
  for (k = j = 0; j < header[5]; j++) {
    for (i = 0; i < header[2]; i++, k++) {
      if ((i == 0) || (i == header[2]-1) || (j == 0) || (j == header[5]-1))
        if ((CPs[3*k  ] != cntrl[3*k  ]) || (CPs[3*k+1] != cntrl[3*k+1]) ||
            (CPs[3*k+2] != cntrl[3*k+2])) {
          printf(" EGADS Error: CP[%d,%d] = %lf %lf %lf -- %lf %lf %lf (EG_adjustCPs)!\n",
                 i+1, j+1, CPs[3*k], cntrl[3*k], CPs[3*k+1], cntrl[3*k+1],
                 CPs[3*k+2], cntrl[3*k+2]);
          EG_free(header);
          EG_free(rvec);
          return EGADS_CONSTERR;
        }
      cntrl[3*k  ] = CPs[3*k  ];
      cntrl[3*k+1] = CPs[3*k+1];
      cntrl[3*k+2] = CPs[3*k+2];
    }
  }
  if ((header[0]&2) != 0) {
    k *= 3;
    for (j = 0; j < header[5]; j++) {
      for (i = 0; i < header[2]; i++, k++) {
        if ((i == 0) || (i == header[2]-1) || (j == 0) || (j == header[5]-1))
          if (CPs[k] != cntrl[k]) {
            printf(" EGADS Error: W[%d,%d] = %lf -- %lf (EG_adjustCPs)!\n",
                   i+1, j+1, CPs[k], cntrl[k]);
            EG_free(header);
            EG_free(rvec);
            return EGADS_CONSTERR;
          }
        cntrl[k] = CPs[k];
      }
    }
  }

  /* make the new surface, Loop & Face */
  stat = EG_makeGeometry(context, oclass, mtype, ref, header, rvec, &newSurf);
  EG_free(header);
  EG_free(rvec);
  if (stat != EGADS_SUCCESS) {
    if (outLevel > 0)
      printf(" EGADS Error: makeGeometry on newSurf = %d (EG_adjustCPs)!\n",
             stat);
    return stat;
  }
  stat = EG_makeTopology(context, newSurf, LOOP, CLOSED, NULL, 4, edges, senses,
                         &loop);
  if (stat != EGADS_SUCCESS) {
    if (outLevel > 0)
      printf(" EGADS Error: makeTopology on newLoop = %d (EG_adjustCPs)!\n",
             stat);
    EG_deleteObject(newSurf);
    return stat;
  }
  stat = EG_makeTopology(context, newSurf, FACE, faceOr, NULL, 1, &loop, fsense,
                         &replace[1]);
  if (stat != EGADS_SUCCESS) {
    if (outLevel > 0)
      printf(" EGADS Error: makeTopology on newFace = %d (EG_adjustCPs)!\n",
             stat);
    EG_deleteObject(loop);
    EG_deleteObject(newSurf);
    return stat;
  }

  /* replace the Face */
  replace[0] = face;
  stat = EG_replaceFaces(body, 1, replace, newBody);
  if ((stat != EGADS_SUCCESS) || (*newBody == NULL)) {
    if (outLevel > 0)
      printf(" EGADS Error: replaceFaces = %d (EG_adjustCPs)!\n", stat);
    EG_deleteObject(replace[1]);
    EG_deleteObject(loop);
    EG_deleteObject(newSurf);
    return stat;
  }

  /* get newFace from newBody */
  stat = EG_getBodyTopos(*newBody, NULL, FACE, &k, &faces);
  if (stat != EGADS_SUCCESS) {
    if (outLevel > 0)
      printf(" EGADS Error: getBodyTopos on newBody = %d (EG_adjustCPs)!\n",
             stat);
    EG_deleteObject(*newBody);
    EG_deleteObject(replace[1]);
    EG_deleteObject(loop);
    EG_deleteObject(newSurf);
    *newBody = NULL;
    return stat;
  }

  for (j = i = 0; i < k; i++) {
    stat = EG_isEquivalent(faces[i], replace[1]);
    if (stat < EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" EGADS Error: isEquivalent on Face %d = %d (EG_adjustCPs)!\n",
               i+1, stat);
      EG_free(faces);
      EG_deleteObject(*newBody);
      EG_deleteObject(replace[1]);
      EG_deleteObject(loop);
      EG_deleteObject(newSurf);
      *newBody = NULL;
      return stat;
    }
    if (stat == EGADS_SUCCESS) {
      if (*newFace != NULL)
        printf(" EGADS Warning: Face multi-match %d %d (EG_adjustCPs)!\n",
               i+1, j);
      *newFace = faces[i];
      j = i+1;
    }
  }
  if (*newFace == NULL) {
    printf(" EGADS Warning: newFace not found in newBody (EG_adjustCPs)!\n");
  } else {
    /* copy the attributes to the newFace */
    stat = EG_attributeDup(face, *newFace);
    if (stat < EGADS_SUCCESS)
      if (outLevel > 0)
        printf(" EGADS Warning: attributeDup = %d (EG_adjustCPs)!\n", stat);
  }

  EG_free(faces);
  EG_deleteObject(replace[1]);
  EG_deleteObject(loop);
  EG_deleteObject(newSurf);

  return EGADS_SUCCESS;
}


/*
 ************************************************************************
 *                                                                      *
 *   EG_spline2d - create 2d cubic spline from input data               *
 *                                                                      *
 ************************************************************************
 */

int EG_spline2d(ego context, int endc, /*@null@*/ const double **drnd,
                int imax, int jmax, const double *xyz, double tol, ego *esurf)
{
  int          i, j, k, stat, outLevel;
  double       dist, dy, *norT, *souT, *easT, *wesT;
  double       x0[3], x1[3], nnor[3], snor[3], enor[3], wnor[3], *rot;
  const double *north, *south, *east, *west;
  
  *esurf = NULL;
  if (context == NULL)               return EGADS_NULLOBJ;
  if (context->magicnumber != MAGIC) return EGADS_NOTOBJ;
  if (context->oclass != CONTXT)     return EGADS_NOTCNTX;
  if ((imax <  2) || (jmax < 2))     return EGADS_DEGEN;
  if ((endc < -3) || (endc > 2))     return EGADS_RANGERR;
  outLevel = EG_outLevel(context);
  
  /* check for degenerate sides */
  north = south = east = west = NULL;
  norT  = souT  = easT = wesT = NULL;
  i  = 0;
  dy = 0.0;
  for (j = 1; j < jmax; j++)
    dy += sqrt((xyz[3*((i)+(j)*imax)  ]-xyz[3*((i)+(j-1)*imax)  ])*
               (xyz[3*((i)+(j)*imax)  ]-xyz[3*((i)+(j-1)*imax)  ]) +
               (xyz[3*((i)+(j)*imax)+1]-xyz[3*((i)+(j-1)*imax)+1])*
               (xyz[3*((i)+(j)*imax)+1]-xyz[3*((i)+(j-1)*imax)+1]) +
               (xyz[3*((i)+(j)*imax)+2]-xyz[3*((i)+(j-1)*imax)+2])*
               (xyz[3*((i)+(j)*imax)+2]-xyz[3*((i)+(j-1)*imax)+2]));
  if (dy <= tol) {
#ifdef DEBUG
    printf("  spline2d: Imin (west) degenerate!\n");
#endif
    if (drnd != NULL)
      if (drnd[0] != NULL) {
        west  = drnd[0];
        x0[0] = drnd[0][1];
        x0[1] = drnd[0][2];
        x0[2] = drnd[0][3];
        x1[0] = drnd[0][5];
        x1[1] = drnd[0][6];
        x1[2] = drnd[0][7];
        CROSS(wnor, x0, x1);
        dist  = DOT(wnor, wnor);
        if ((dist == 0.0) || (DOT(x0, x1) > tol) ||
            (fabs(1.0-DOT(x0, x0)) > tol) ||
            (fabs(1.0-DOT(x1, x1)) > tol)) {
          if (outLevel > 0)
            printf(" EGADS Error: BAD West Axes (EG_spline2d)!\n");
          return EGADS_NOTORTHO;
        }
        dist   = 1.0/sqrt(dist);
        wnor[0] *= dist;
        wnor[1] *= dist;
        wnor[2] *= dist;
        wesT     = wnor;
#ifdef DEBUG
        printf("            with normal = %lf %lf %lf!\n",
               wnor[0], wnor[1], wnor[2]);
#endif
      }
  }
  j  = 0;
  dy = 0.0;
  for (i = 1; i < imax; i++)
    dy += sqrt((xyz[3*((i)+(j)*imax)  ]-xyz[3*((i-1)+(j)*imax)  ])*
               (xyz[3*((i)+(j)*imax)  ]-xyz[3*((i-1)+(j)*imax)  ]) +
               (xyz[3*((i)+(j)*imax)+1]-xyz[3*((i-1)+(j)*imax)+1])*
               (xyz[3*((i)+(j)*imax)+1]-xyz[3*((i-1)+(j)*imax)+1]) +
               (xyz[3*((i)+(j)*imax)+2]-xyz[3*((i-1)+(j)*imax)+2])*
               (xyz[3*((i)+(j)*imax)+2]-xyz[3*((i-1)+(j)*imax)+2]));
  if (dy <= tol) {
#ifdef DEBUG
    printf("  spline2d: Jmin (south) degenerate!\n");
#endif
    if (drnd != NULL)
      if (drnd[1] != NULL) {
        south = drnd[1];
        x0[0] = drnd[1][1];
        x0[1] = drnd[1][2];
        x0[2] = drnd[1][3];
        x1[0] = drnd[1][5];
        x1[1] = drnd[1][6];
        x1[2] = drnd[1][7];
        CROSS(snor, x0, x1);
        dist  = DOT(snor, snor);
        if ((dist == 0.0) || (DOT(x0, x1) > tol) ||
            (fabs(1.0-DOT(x0, x0)) > tol) ||
            (fabs(1.0-DOT(x1, x1)) > tol)) {
          if (outLevel > 0)
            printf(" EGADS Error: BAD South Axes (EG_spline2d)!\n");
          return EGADS_NOTORTHO;
        }
        dist   = 1.0/sqrt(dist);
        snor[0] *= dist;
        snor[1] *= dist;
        snor[2] *= dist;
        souT     = snor;
#ifdef DEBUG
        printf("            with normal = %lf %lf %lf!\n",
               snor[0], snor[1], snor[2]);
#endif
      }
  }
  i  = imax-1;
  dy = 0.0;
  for (j = 1; j < jmax; j++)
    dy += sqrt((xyz[3*((i)+(j)*imax)  ]-xyz[3*((i)+(j-1)*imax)  ])*
               (xyz[3*((i)+(j)*imax)  ]-xyz[3*((i)+(j-1)*imax)  ]) +
               (xyz[3*((i)+(j)*imax)+1]-xyz[3*((i)+(j-1)*imax)+1])*
               (xyz[3*((i)+(j)*imax)+1]-xyz[3*((i)+(j-1)*imax)+1]) +
               (xyz[3*((i)+(j)*imax)+2]-xyz[3*((i)+(j-1)*imax)+2])*
               (xyz[3*((i)+(j)*imax)+2]-xyz[3*((i)+(j-1)*imax)+2]));
  if (dy <= tol) {
#ifdef DEBUG
    printf("  spline2d: Imax (east) degenerate!\n");
#endif
    if (drnd != NULL)
      if (drnd[2] != NULL) {
        east  = drnd[2];
        x0[0] = drnd[2][1];
        x0[1] = drnd[2][2];
        x0[2] = drnd[2][3];
        x1[0] = drnd[2][5];
        x1[1] = drnd[2][6];
        x1[2] = drnd[2][7];
        CROSS(enor, x0, x1);
        dist  = DOT(enor, enor);
        if ((dist == 0.0) || (DOT(x0, x1) > tol)||
            (fabs(1.0-DOT(x0, x0)) > tol) ||
            (fabs(1.0-DOT(x1, x1)) > tol)) {
          if (outLevel > 0)
            printf(" EGADS Error: BAD East Axes (EG_spline2d)!\n");
          return EGADS_NOTORTHO;
        }
        dist   = 1.0/sqrt(dist);
        enor[0] *= dist;
        enor[1] *= dist;
        enor[2] *= dist;
        easT     = enor;
#ifdef DEBUG
        printf("            with normal = %lf %lf %lf!\n",
               enor[0], enor[1], enor[2]);
#endif
      }
  }
  j  = jmax-1;
  dy = 0.0;
  for (i = 1; i < imax; i++)
    dy += sqrt((xyz[3*((i)+(j)*imax)  ]-xyz[3*((i-1)+(j)*imax)  ])*
               (xyz[3*((i)+(j)*imax)  ]-xyz[3*((i-1)+(j)*imax)  ]) +
               (xyz[3*((i)+(j)*imax)+1]-xyz[3*((i-1)+(j)*imax)+1])*
               (xyz[3*((i)+(j)*imax)+1]-xyz[3*((i-1)+(j)*imax)+1]) +
               (xyz[3*((i)+(j)*imax)+2]-xyz[3*((i-1)+(j)*imax)+2])*
               (xyz[3*((i)+(j)*imax)+2]-xyz[3*((i-1)+(j)*imax)+2]));
  if (dy <= tol) {
#ifdef DEBUG
    printf("  spline2d: Jmax (north) degenerate!\n");
#endif
    if (drnd != NULL)
      if (drnd[3] != NULL) {
        north = drnd[3];
        x0[0] = drnd[3][1];
        x0[1] = drnd[3][2];
        x0[2] = drnd[3][3];
        x1[0] = drnd[3][5];
        x1[1] = drnd[3][6];
        x1[2] = drnd[3][7];
        CROSS(nnor, x0, x1);
        dist  = DOT(nnor, nnor);
        if ((dist == 0.0) || (DOT(x0, x1) > tol)||
            (fabs(1.0-DOT(x0, x0)) > tol) ||
            (fabs(1.0-DOT(x1, x1)) > tol)) {
          if (outLevel > 0)
            printf(" EGADS Error: BAD North Axes (EG_spline2d)!\n");
          return EGADS_NOTORTHO;
        }
        dist   = 1.0/sqrt(dist);
        nnor[0] *= dist;
        nnor[1] *= dist;
        nnor[2] *= dist;
        norT     = nnor;
#ifdef DEBUG
        printf("            with normal = %lf %lf %lf!\n",
               nnor[0], nnor[1], nnor[2]);
#endif
      }
  }
  if ((north != NULL) && ((east != NULL) || (west != NULL)))
    return EGADS_DEGEN;
  if ((south != NULL) && ((east != NULL) || (west != NULL)))
    return EGADS_DEGEN;
  
  /* call approx as is */
  if ((east == NULL) && (west == NULL))
    return EG_spline2dAppx(context, endc, NULL, NULL, NULL, NULL, NULL,
                           south, souT, north, norT, imax, jmax, xyz, tol,
                           esurf);
  
  if (endc < 0) {
    if (outLevel > 0)
      printf(" EGADS Error: Periodic U but required rotation (EG_spline2d)!\n");
    return EGADS_GEOMERR;
  }

  /* rotate to get special treatment as north/south */
  rot = (double *) EG_alloc(3*imax*jmax*sizeof(double));
  if (rot == NULL) return EGADS_MALLOC;
  
  for (k = i = 0; i < imax; i++)
    for (j = jmax-1; j >= 0; j--, k++) {
      rot[3*k  ] = xyz[3*(i+j*imax)  ];
      rot[3*k+1] = xyz[3*(i+j*imax)+1];
      rot[3*k+2] = xyz[3*(i+j*imax)+2];
    }
#ifdef DEBUG
  printf(" spline2d: rotate -- south=west, north=east!\n");
#endif
  stat = EG_spline2dAppx(context, endc, NULL, NULL, NULL, NULL, NULL,
                         west, wesT, east, easT, jmax, imax, rot, tol,
                         esurf);
  
  EG_free(rot);
  return stat;
}


static int
EG_addNewNode(ego context, int *nnewnode, newNode *newnodes, double *xyz,
              ego *node)
{
  int    i, stat;
  double toler, dist;
  
  *node = NULL;
  for (i = 0; i < *nnewnode; i++) {
    stat = EG_tolerance(newnodes[i].node, &toler);
    if (stat != EGADS_SUCCESS) return stat;
    dist = sqrt((xyz[0]-newnodes[i].xyz[0])*(xyz[0]-newnodes[i].xyz[0]) +
                (xyz[1]-newnodes[i].xyz[1])*(xyz[1]-newnodes[i].xyz[1]) +
                (xyz[2]-newnodes[i].xyz[2])*(xyz[2]-newnodes[i].xyz[2]));
    if (dist <= toler) {
      *node = newnodes[i].node;
      return EGADS_SUCCESS;
    }
  }
  
  /* make a new node */
  stat = EG_makeTopology(context, NULL, NODE, 0, xyz, 0, NULL, NULL, node);
  if (stat != EGADS_SUCCESS) return stat;
  newnodes[*nnewnode].xyz[0] = xyz[0];
  newnodes[*nnewnode].xyz[1] = xyz[1];
  newnodes[*nnewnode].xyz[2] = xyz[2];
  newnodes[*nnewnode].node   = *node;
  *nnewnode += 1;
  
  return EGADS_SUCCESS;
}


/*
 ************************************************************************
 *                                                                      *
 *   EG_scribeFaces - imprints Edges on a Body from discrete data       *
 *                                                                      *
 ************************************************************************
 */

int
EG_scribeFaces(const ego body,                /* input Body */
               /*@null@*/ const char *aName,  /* scribe Edge attribute name */
               int nscribe,                   /* num scribes (new Edges) */
               const int *iface,              /* Face index (per scribe) */
               const int *startEnd,           /* end-point info 2 per scribe */
               const int *npts,               /* number of points per scribe */
               const double *xyz,             /* all coordinates in order */
               ego *newbody)                  /* resultant updated Body */
{
  int     i, j, k, n, nn, stat, tpnts, ipnts, oclass, mtype, nloop, outLevel;
  int     nnewnode, iper, header[4], *senses;
  double  period, pos[3], coor[3], uvbox[4], data[6], *knots, *rvec, *uv;
  double  srange[4], trange[2];
  ego     context, surface, ref, nodes[2], *faces, *loops;
  ego     *objs = NULL, *pairs = NULL;
  newNode *newnodes = NULL;
  
  *newbody = NULL;
  if  (body == NULL)               return EGADS_NULLOBJ;
  if  (body->magicnumber != MAGIC) return EGADS_NOTOBJ;
  if  (body->oclass != BODY)       return EGADS_NOTBODY;
  if ((body->mtype != SOLIDBODY) &&
      (body->mtype != SHEETBODY))  return EGADS_TOPOERR;
  if  (body->blind == NULL)        return EGADS_NODATA;
  if  (EG_sameThread(body))        return EGADS_CNTXTHRD;
  if  (nscribe <= 0)               return EGADS_TOPOCNT;
  outLevel = EG_outLevel(body);
  context  = EG_context(body);
  if  (context == NULL)            return EGADS_NOTCNTX;
  
  /* get sizes and check inputs */
  stat = EG_getBodyTopos(body, NULL, NODE, &nn, NULL);
  if (stat != EGADS_SUCCESS) return stat;
  stat = EG_getBodyTopos(body, NULL, EDGE, &n,  NULL);
  if (stat != EGADS_SUCCESS) return stat;
  for (tpnts = i = 0; i < nscribe; i++) {
    if (startEnd[2*i ] == 0) {
      if (startEnd[2*i+1] != 0) return EGADS_NOTTOPO;
      /* first needs to equal the last */
      if ((xyz[3*tpnts  ] != xyz[3*(tpnts+npts[i])-3]) ||
          (xyz[3*tpnts+1] != xyz[3*(tpnts+npts[i])-2]) ||
          (xyz[3*tpnts+2] != xyz[3*(tpnts+npts[i])-1])) {
        if (outLevel > 0)
          printf(" EGADS Error: %d first != last (EG_scribeFaces)!\n", i+1);
        return EGADS_GEOMERR;
      }
    } else if (startEnd[2*i+1] == 0) {
      return EGADS_NOTTOPO;
    } else {
      if (startEnd[2*i  ] < 0) {
        if (-startEnd[2*i  ] > nn) return EGADS_NOTTOPO;
      } else {
        if ( startEnd[2*i  ] > n)  return EGADS_NOTTOPO;
      }
      if (startEnd[2*i+1] < 0) {
        if (-startEnd[2*i+1] > nn) return EGADS_NOTTOPO;
      } else {
        if ( startEnd[2*i+1] > n)  return EGADS_NOTTOPO;
      }
    }
    if (npts[i] <= 1) {
      if (outLevel > 0)
        printf(" EGADS Error: npts[%d] = %d (EG_scribeFaces)!\n", i, npts[i]);
      return EGADS_TOPOCNT;
    } else if (npts[i] == 2) {
      if ((xyz[3*tpnts  ] == xyz[3*tpnts+4]) &&
          (xyz[3*tpnts+1] == xyz[3*tpnts+5]) &&
          (xyz[3*tpnts+2] == xyz[3*tpnts+6])) {
        if (outLevel > 0)
          printf(" EGADS Error: %d first == last (EG_scribeFaces)!\n", i+1);
        return EGADS_GEOMERR;
      }
    }
    tpnts += npts[i];
  }
  if (outLevel > 1)
    printf(" scribeFaces: total pts = %d\n", tpnts);
  
  /* set up the 'pairs' for EG_imprintBody and 'objs' for temp storage */
  pairs = (ego *) EG_alloc(2*nscribe*sizeof(ego));
  if (pairs == NULL) {
    if (outLevel > 0)
      printf(" EGADS Error: Cannot allocate %d pairs (EG_scribeFaces)!\n",
             nscribe);
    stat = EGADS_MALLOC;
    goto cleanup;
  }
  for (i = 0; i < 2*nscribe; i++) pairs[i] = NULL;
  objs  = (ego *) EG_alloc(2*nscribe*sizeof(ego));
  if (objs == NULL) {
    if (outLevel > 0)
      printf(" EGADS Error: Cannot allocate %d objects (EG_scribeFaces)!\n",
             2*nscribe);
    stat = EGADS_MALLOC;
    goto cleanup;
  }
  for (i = 0; i < 2*nscribe; i++) objs[i] = NULL;
  nnewnode = 0;
  newnodes = (newNode *) EG_alloc(2*nscribe*sizeof(newNode));
  if (newnodes == NULL) {
    if (outLevel > 0)
      printf(" EGADS Error: Cannot allocate %d nodes (EG_scribeFaces)!\n",
             2*nscribe);
    stat = EGADS_MALLOC;
    goto cleanup;
  }

  /* set the Face objects in pairs */
  stat = EG_getBodyTopos(body, NULL, FACE, &n, &faces);
  if (stat != EGADS_SUCCESS) goto cleanup;
  for (i = 0; i < nscribe; i++) {
    if ((iface[i] <= 0) || (iface[i] > n)) {
      if (outLevel > 0)
        printf(" EGADS Error: Face index %d = %d [1,%d] (EG_scribeFaces)!\n",
               i+1, iface[i], n);
      stat = EGADS_INDEXERR;
      EG_free(faces);
      goto cleanup;
    }
    pairs[2*i  ] = faces[iface[i]-1];
  }
  EG_free(faces);
  
  /* make an Edge (with PCurve) for each scribe */
  for (ipnts = i = 0; i < nscribe; i++) {
    stat = EG_getTopology(pairs[2*i  ], &surface, &oclass, &mtype, uvbox,
                          &nloop, &loops, &senses);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" EGADS Error: getTopology Face %d = %d (EG_scribeFaces)!\n",
               iface[i], stat);
      goto cleanup;
    }
    while (surface->mtype == TRIMMED) {
      stat = EG_getGeometry(surface, &oclass, &mtype, &ref, NULL, NULL);
      if (stat != EGADS_SUCCESS) {
        printf(" EGADS Error: getGeometry %d = %d (EG_scribeFaces)!\n",
               i+1, stat);
      goto cleanup;
      }
      surface = ref;
    }
    stat = EG_getRange(surface, srange, &iper);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" EGADS Error: getRange Surface %d = %d (EG_scribeFaces)!\n",
               iface[i], stat);
      goto cleanup;
    }
    if ((iper == 1) || (iper == 3)) {
      if ((uvbox[0]+1.e-6 < srange[0]) || (uvbox[1]-1.e-6 > srange[1])) {
        printf(" EGADS Warning: Face %d Us cross Surf period (EG_scribeFaces)!\n",
               iface[i]);
        printf("                Face = %lf %lf   Surface = %lf %lf\n",
               uvbox[0], uvbox[1], srange[0], srange[1]);
      }
    }
    if ((iper == 2) || (iper == 3)) {
      if ((uvbox[2]+1.e-6 < srange[2]) || (uvbox[3]+1.e-6 > srange[3])) {
        printf(" EGADS Warning: Face %d Vs cross Surf period (EG_scribeFaces)!\n",
               iface[i]);
        printf("                Face = %lf %lf   Surface = %lf %lf\n",
               uvbox[2], uvbox[3], srange[2], srange[3]);
      }
    }
    
    knots     = NULL;
    trange[0] = 0.0;
    trange[1] = 1.0;
    if (npts[i] == 2) {
      /* straight line */
      data[0] = xyz[3*ipnts  ];
      data[1] = xyz[3*ipnts+1];
      data[2] = xyz[3*ipnts+2];
      data[3] = xyz[3*ipnts+3] - data[0];
      data[4] = xyz[3*ipnts+4] - data[1];
      data[5] = xyz[3*ipnts+5] - data[2];
      stat    = EG_makeGeometry(context, CURVE, LINE, NULL, NULL, data,
                                &objs[2*i  ]);
      if (stat != EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" EGADS Error: makeGeometry Line %d = %d (EG_scribeFaces)!\n",
                 i+1, stat);
        goto cleanup;
      }
      pos[0] = xyz[3*ipnts+3];
      pos[1] = xyz[3*ipnts+4];
      pos[2] = xyz[3*ipnts+5];
      stat   = EG_invEvaluate(objs[2*i  ], pos, &trange[1], coor);
      if (stat != EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" EGADS Error: invEvaluate Line %d = %d (EG_scribeFaces)!\n",
                 i+1, stat);
        goto cleanup;
      }
      if (outLevel > 1)
        printf(" scribeFaces: invEval end t = %lf   xyz = %lf %lf %lf\n",
               trange[1], pos[0], pos[1], pos[2]);
    } else {
      /* not a straight line */
      knots = (double *) EG_alloc(npts[i]*sizeof(double));
      if (knots == NULL) {
        if (outLevel > 0)
          printf(" EGADS Error: Cannot allocate %d/%d knots (EG_scribeFaces)!\n",
                 iface[i], npts[i]);
        stat = EGADS_MALLOC;
        goto cleanup;
      }
      knots[0] = 0.0;
      for (j = 1; j < npts[i]; j++) {
        k = ipnts + j;
        knots[j] = knots[j-1] +
                      sqrt((xyz[3*k  ]-xyz[3*k-3])*(xyz[3*k  ]-xyz[3*k-3]) +
                           (xyz[3*k+1]-xyz[3*k-2])*(xyz[3*k+1]-xyz[3*k-2]) +
                           (xyz[3*k+2]-xyz[3*k-1])*(xyz[3*k+2]-xyz[3*k-1]));
        if (knots[j] == knots[j-1]) {
          if (outLevel > 0)
            printf(" EGADS Error: Repeated knot %d/%d (EG_scribeFaces)!\n",
                   iface[i], j);
          EG_free(knots);
          stat = EGADS_DEGEN;
          goto cleanup;
        }
      }
      /* set the parameterization to go from 0.0 to 1.0 */
      for (j = 0; j < npts[i]; j++) knots[j] /= knots[npts[i]-1];
      stat = EG_spline1dFit(0, npts[i], &xyz[3*ipnts], knots, 1.e-8, header,
                            &rvec);
      if (stat != EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" EGADS Error: spline1dFit %d = %d (EG_scribeFaces)!\n",
                 iface[i], stat);
        EG_free(knots);
        goto cleanup;
      }
      stat = EG_makeGeometry(context, CURVE, BSPLINE, NULL, header, rvec,
                             &objs[2*i  ]);
      EG_free(rvec);
      if (stat != EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" EGADS Error: makeGeometry Curve %d = %d (EG_scribeFaces)!\n",
                 iface[i], stat);
        EG_free(knots);
        goto cleanup;
      }
    }
    
    /* deal with the end-points */
    if (startEnd[2*i  ] == 0) {
      if (npts[i] == 2) {
        if (outLevel > 0)
          printf(" EGADS Error: Closed Edge with npts = 2 (EG_scribeFaces)!\n");
        stat = EGADS_DEGEN;
        goto cleanup;
      }
      pos[0] = xyz[3*ipnts  ];
      pos[1] = xyz[3*ipnts+1];
      pos[2] = xyz[3*ipnts+2];
      stat   = EG_addNewNode(context, &nnewnode, newnodes, pos, &nodes[0]);
      if (stat != EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" EGADS Error: addNewNode OneNode %d = %d (EG_scribeFaces)!\n",
                 iface[i], stat);
        if (knots != NULL) EG_free(knots);
        goto cleanup;
      }
      nodes[1] = nodes[0];
    } else {
      if (startEnd[2*i  ] < 0) {
        stat = EG_objectBodyTopo(body, NODE, -startEnd[2*i  ], &nodes[0]);
        if (stat != EGADS_SUCCESS) {
          if (outLevel > 0)
            printf(" EGADS Error: objectBodyTopo0 %d = %d (EG_scribeFaces)!\n",
                   iface[i], stat);
          if (knots != NULL) EG_free(knots);
          goto cleanup;
        }
      } else {
        pos[0] = xyz[3*ipnts  ];
        pos[1] = xyz[3*ipnts+1];
        pos[2] = xyz[3*ipnts+2];
        stat   = EG_addNewNode(context, &nnewnode, newnodes, pos, &nodes[0]);
        if (stat != EGADS_SUCCESS) {
          if (outLevel > 0)
            printf(" EGADS Error: addNewNode Node0 %d = %d (EG_scribeFaces)!\n",
                   iface[i], stat);
          if (knots != NULL) EG_free(knots);
          goto cleanup;
        }
      }
      if (startEnd[2*i+1] < 0) {
        stat = EG_objectBodyTopo(body, NODE, -startEnd[2*i+1], &nodes[1]);
        if (stat != EGADS_SUCCESS) {
          if (outLevel > 0)
            printf(" EGADS Error: objectBodyTopo1 %d = %d (EG_scribeFaces)!\n",
                   i+1, stat);
          if (knots != NULL) EG_free(knots);
          goto cleanup;
        }
      } else {
        j = 3*(ipnts + npts[i] - 1);
        pos[0] = xyz[j  ];
        pos[1] = xyz[j+1];
        pos[2] = xyz[j+2];
        stat   = EG_addNewNode(context, &nnewnode, newnodes, pos, &nodes[1]);
        if (stat != EGADS_SUCCESS) {
          if (outLevel > 0)
            printf(" EGADS Error: makeTopology Node1 %d = %d (EG_scribeFaces)!\n",
                   i+1, stat);
          if (knots != NULL) EG_free(knots);
          goto cleanup;
        }
      }
    }
    
    /* make the Edge */
    nn    = 2;
    mtype = TWONODE;
    if (nodes[0] == nodes[1]) {
      nn    = 1;
      mtype = ONENODE;
    }
    stat = EG_makeTopology(context, objs[2*i  ], EDGE, mtype, trange, nn, nodes,
                           NULL, &pairs[2*i+1]);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" EGADS Error: makeTopology Edge %d = %d (EG_scribeFaces)!\n",
               i+1, stat);
      if (knots != NULL) EG_free(knots);
      goto cleanup;
    }
    /* add the attribute */
    if (aName != NULL) {
      j    = i+1;
      stat = EG_attributeAdd(pairs[2*i+1], aName, ATTRINT, 1, &j, NULL, NULL);
      if (stat != EGADS_SUCCESS)
        if (outLevel > 0)
          printf(" EGADS Warning: EG_attributeAdd %d = %d (EG_scribeFaces)!\n",
                 i+1, stat);
    }
    if (outLevel > 1) {
      double tol;
      (void) EG_tolerance(pairs[2*i+1], &tol);
      printf(" scribeFaces: Edge %2d -> toler = %le\n", i+1, tol);
    }

    /* make the PCurve and attach it to the Edge */
    if ((surface->mtype != PLANE) && (knots != NULL)) {
      /* fill up UVs */
      uv = EG_alloc(2*npts[i]*sizeof(double));
      if (uv == NULL) {
        if (outLevel > 0)
          printf(" EGADS Error: Cannot allocate %d/%d uvs (EG_scribeFaces)!\n",
                 i+1, npts[i]);
        stat = EGADS_MALLOC;
        EG_free(knots);
        goto cleanup;
      }
      for (j = 0; j < npts[i]; j++) {
        pos[0] = xyz[3*(ipnts+j)  ];
        pos[1] = xyz[3*(ipnts+j)+1];
        pos[2] = xyz[3*(ipnts+j)+2];
        stat = EG_invEvaluate(surface, pos, &uv[2*j], coor);
        if (stat != EGADS_SUCCESS) {
          if (outLevel > 0)
            printf(" EGADS Error: invEvaluate %d/%d = %d (EG_scribeFaces)!\n",
                   iface[i], j, stat);
          EG_free(uv);
          EG_free(knots);
          goto cleanup;
        }
      }
      /* correct uvs for crossing periodics */
      if ((iper != 0) && (startEnd[2*i  ] != 0)) {
        if ((iper == 1) || (iper == 3)) {
          period = srange[1] - srange[0];
          if (fabs(uv[0]-uv[2]) > 0.5*period)
            if (uv[2] > uv[0]) {
              uv[0] += period;
            } else {
              uv[0] -= period;
            }
          if (fabs(uv[2*npts[i]-2]-uv[2*npts[i]-4]) > 0.5*period)
            if (uv[2*npts[i]-4] > uv[2*npts[i]-2]) {
              uv[2*npts[i]-2] += period;
            } else {
              uv[2*npts[i]-2] -= period;
            }
        }
        if ((iper == 2) || (iper == 3)) {
          period = srange[3] - srange[2];
          if (fabs(uv[1]-uv[3]) > 0.5*period)
            if (uv[3] > uv[1]) {
              uv[1] += period;
            } else {
              uv[1] -= period;
            }
          if (fabs(uv[2*npts[i]-1]-uv[2*npts[i]-3]) > 0.5*period)
            if (uv[2*npts[i]-3] > uv[2*npts[i]-1]) {
              uv[2*npts[i]-1] += period;
            } else {
              uv[2*npts[i]-1] -= period;
            }
        }
      }
      /* use the same knot sequence as the Curve */
      stat = EG_spline1dPCrv(context, 0, npts[i], uv, knots, 1.e-6,
                             &objs[2*i+1]);
      EG_free(uv);
      if (stat != EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" EGADS Error: spline1dPCrv %d = %d (EG_scribeFaces)!\n",
                 i+1, stat);
        EG_free(knots);
        goto cleanup;
      }
      
      stat = EG_setPCurve(pairs[2*i  ], pairs[2*i+1], objs[2*i+1]);
      if (stat != EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" EGADS Error: setPCurve %d = %d (EG_scribeFaces)!\n",
                 i+1, stat);
        EG_free(knots);
        goto cleanup;
      }
      if (outLevel > 1) {
        double tol;
        (void) EG_tolerance(pairs[2*i+1], &tol);
        printf(" scribeFaces: Edge %2d -> PCtol = %le\n", i+1, tol);
      }
    }
    if (knots != NULL) EG_free(knots);
    
    ipnts += npts[i];
  }

  /* do the imprint -- use OpenCASCADE only */
  stat = EG_imprintBody(body, -nscribe, pairs, newbody);
  if (stat != EGADS_SUCCESS)
    if (outLevel > 0)
      printf(" EGADS Error: imprintBody = %d (EG_scribeFaces)!\n", stat);
  
cleanup:
  if (pairs != NULL) {
    for (i = 0; i < nscribe; i++)
      if (pairs[2*i+1] != NULL) EG_deleteObject(pairs[2*i+1]);
    EG_free(pairs);
  }
  if (objs != NULL) {
    for (i = 0; i < 2*nscribe; i++)
      if (objs[i] != NULL) EG_deleteObject(objs[i]);
    EG_free(objs);
  }
  if (newnodes != NULL) {
    for (i = 0; i < nnewnode; i++) EG_deleteObject(newnodes[i].node);
    EG_free(newnodes);
  }

  return stat;
}
