 /*
 ************************************************************************
 *                                                                      *
 * udfRemoveNodes -- remove spurious Nodes from a SolidBody or SheetBody*
 *                                                                      *
 *             Written by John Dannenhoffer @ Geocentric Technologies   *
 *                                                                      *
 ************************************************************************
 */

/*
 * Copyright (C) 2026  John F. Dannenhoffer, III (Geocentric Technologies))
 *
 * This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *     MA  02110-1301  USA
 */

/* uncomment the following to get DEBUG printouts */
//#define DEBUG 1

/* the number of "input" Bodys

   this only needs to be specified if this is a UDF (user-defined
   function) that consumes Bodys from OpenCSM's stack.  (the default
   value is 0).

   if NUMUDPINPUTBODYS>0  then exactly NUMUDPINPUTBODYS are in emodel
   if NUMUDPINPUTBODYS<0  then up to  -NUMUDPINPUTBODYS are in emodel
*/
#define NUMUDPINPUTBODYS 1

/* the number of arguments (specified below) */
#define NUMUDPARGS 1

/* set up the necessary structures (uses NUMUDPARGS) */
#include "udpUtilities.h"

/* shorthand macros for accessing argument values and velocities */
#define ANGLE(    IUDP)  ((double *) (udps[IUDP].arg[0].val))[0]
#define ANGLE_SIZ(IUDP)               udps[IUDP].arg[0].size

/* data about possible arguments
      argNames: argument name (must be all lower case)
      argTypes: argument type: +ATTRINT      integer input
                               -ATTRINT      integer output
                               +ATTRREAL     double input  (no sensitivities)
                               -ATTRREAL     double output (no sensitivities)
                               +ATTRREALSEN  double input  (with sensitivities)
                               -ATTRREALSEN  double output (with sensitivities)
                               +ATTRSTRING   string input
                               -ATTRSTRING   *** cannot be used ***
                               +ATTRFILE     input file
                               -ATTRFILE     *** cannot be used ***
                               +ATTRREBUILD  forces rebuild if any variable in
                                             semi-colon-separated list has been changed
                               -ATTRREBUILD  *** cannot be used ***
                               +ATTRRECYCLE  forces rebuild (always) by blocking recycling
                               -ATTRRECYCLE  *** cannot be used ***
      argIdefs: default value for ATTRINT
      argDdefs: default value for ATTRREAL or ATTRREALSEN */
static char  *argNames[NUMUDPARGS] = {"angle",  };
static int    argTypes[NUMUDPARGS] = {ATTRREAL, };
static int    argIdefs[NUMUDPARGS] = {0,        };
static double argDdefs[NUMUDPARGS] = {5.,       };

/* get utility routines: udpErrorStr, udpInitialize, udpReset, udpSet,
                         udpGet, udpVel, udpClean, udpMesh */
#include "udpUtilities.c"

static int computeDot(ego ebody, ego enode, double  *dot);


/*
 ************************************************************************
 *                                                                      *
 *   udpExecute - execute the primitive                                 *
 *                                                                      *
 ************************************************************************
 */

int
udpExecute(ego  emodel,                 /* (in)  model of input Bodys */
           ego  *ebody,                 /* (out) Body (or model) pointer */
           int  *nMesh,                 /* (out) number of associated meshes */
           char *string[])              /* (out) error message */
{
    int     status = EGADS_SUCCESS;

    int     oclass, mtype, nchild, *senses, nlist, nloop, iloop;
    int     nnode, inode, nface;
    double  data[18], dot;
    ego     eref, *ebodys, *enodes=NULL, *elist=NULL, etemp;
    char    *message=NULL;
    udp_T   *udps = *Udps;

    ROUTINE(udpExecute);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    /* debug printing of the input arguments */
    printf("udpExecute(emodel=%llx)\n", (long long)emodel);
    printf("angle(0) = %f\n", ANGLE(0));
#endif

    /* default return values */
    *ebody  = NULL;
    *nMesh  = 0;
    *string = NULL;

    /* the place where messages to the user are placed */
    MALLOC(message, char, 100);
    message[0] = '\0';

    /* check that Model was input that contains one Body */
    status = EG_getTopology(emodel, &eref, &oclass, &mtype,
                            data, &nchild, &ebodys, &senses);
    CHECK_STATUS(EG_getTopology);

    if (oclass != MODEL) {
        snprintf(message, 100, "expecting a Model, but found oclass=%d", oclass);
        status = EGADS_NOTMODEL;
        goto cleanup;
    } else if (nchild != 1) {
        snprintf(message, 100, "expecting Model to contain one Body (not %d)", nchild);
        status = EGADS_NOTBODY;
        goto cleanup;
    } else {
        status = EG_getInfo(ebodys[0], &oclass, &mtype, NULL, NULL, NULL);
        CHECK_STATUS(EG_getInfo);

        if (mtype != SOLIDBODY && mtype != SHEETBODY && mtype != FACEBODY) {
            snprintf(message, 100, "only SolidBodys, SheetBodys, and FaceBodys are supported\n");
            status = EGADS_NOTBODY;
            goto cleanup;
        }
    }

    /* check arguments */
    if (ANGLE_SIZ(0) > 1) {
        snprintf(message, 100, "\"angle\" should be a scalar");
        status = OCSM_ILLEGAL_VALUE;
        goto cleanup;

    } else if (ANGLE(0) <= 0) {
        snprintf(message, 100, "\"angle\" should be a positive");
        status  = OCSM_ILLEGAL_VALUE;
        goto cleanup;

    }

    /* cache copy of arguments for future use */
    status = cacheUdp(NULL);
    CHECK_STATUS(cacheUdp);

#ifdef DEBUG
    /* debug printing of cached input arguments */
    printf("angle[%d] = %f\n", numUdp, ANGLE(numUdp));
#endif

    /* make a list of Nodes to delete */
    status = EG_getBodyTopos(ebodys[0], NULL, NODE, &nnode, &enodes);
    CHECK_STATUS(EG_getBodyTopos);

    /* make a list of the Nodes to remove */
    MALLOC(elist, ego, nnode);
    nlist = 0;

    /* look at each Node to determine if it should be added to the list */
    for (inode = 0; inode < nnode; inode++) {
        SPLINT_CHECK_FOR_NULL(enodes);

        /* Node cannot be removed if there are more than 2 associated Faces */
        status = EG_getBodyTopos(ebodys[0], enodes[inode], FACE, &nface, NULL);
        CHECK_STATUS(EG_getBodyTopos);

        if (nface > 2) continue;

        /* compute the dot product between the Edges */
        status = computeDot(ebodys[0], enodes[inode], &dot);
        CHECK_STATUS(computeDot);

        /* if the dot product exceeds the dot product associated with ANGLE,
           add it to the list */
        if (fabs(dot) > cos(ANGLE(0) * PIo180)) {
            elist[nlist++] = enodes[inode];
        }
    }

    EG_free(enodes);   enodes = NULL;

    /* try to remove the Nodes */
    status = EG_removeNodes(ebodys[0], nlist, elist, ebody);

    /* if we succeeded, we are done */
    if (status == EGADS_SUCCESS) {

        /* set the output value(s) */

        /* remember this model (Body) */
        udps[numUdp].ebody = *ebody;

        goto cleanup;

    /* if the error was not EGADS_TOPOERR or EGADS_CONSTERR, then something
       went wrong, so return the error that was returned from EG_removeNodes */
    } else if (status != EGADS_TOPOERR && status != EGADS_CONSTERR) {

        goto cleanup;

    /* if we got an expected error, it is probably because the removal
       of some Node(s) caused a Loop with fewer than 2 Edges.  So we need
       instead to remove the Nodes one at a time */
    } else {

        status = EG_copyObject(ebodys[0], NULL, &etemp);
        CHECK_STATUS(EG_copyObject);

        nloop = nlist;
        for (iloop = 0; iloop < nloop; iloop++) {

            nlist = 0;

            status = EG_getBodyTopos(etemp, NULL, NODE, &nnode, &enodes);
            CHECK_STATUS(EG_getBodyTopos);

            for (inode = 0; inode < nnode; inode++) {
                SPLINT_CHECK_FOR_NULL(enodes);

                /* compute the dot product between the Edges */
                status = computeDot(etemp, enodes[inode], &dot);
                CHECK_STATUS(computeDot);

                /* if the dot product is less than the dot product associated with ANGLE,
                   this Node is not spurious */
                if (fabs(dot) <= cos(ANGLE(0) * PIo180)) continue;

                {
                    ego *echilds;
                    status = EG_getTopology(enodes[inode], &eref, &oclass, &mtype, data,
                                            &nchild, &echilds, &senses);
                    CHECK_STATUS(EG_getTopology);
                }

                /* try to remove the Node */
                status = EG_removeNodes(etemp, 1, &(enodes[inode]), ebody);

                /* if removing the Node was successful, break out of the
                   inode loop and repeat the whole process with a new etemp */
                if (status == EGADS_SUCCESS) {
                    status = EG_deleteObject(etemp);
                    CHECK_STATUS(EG_deleteObject);

                    etemp = *ebody;

                    SPLINT_CHECK_FOR_NULL(etemp);

                    nlist = 1;
                    EG_free(enodes);
                    break;
                } else {
                    status = EGADS_SUCCESS;
                }
            }

            /* if we got here and we did not successfully remove any Nodes on
               the last pass, we are done */
            if (nlist == 0) {
                *ebody = etemp;
                break;
            }
        }
    }

cleanup:
#ifdef DEBUG
    printf("udpExecute -> numUdp=%d, *ebody=%llx\n", numUdp, (long long)(*ebody));
#endif

    FREE(elist);

    if (enodes != NULL) EG_free(enodes);

    if (strlen(message) > 0) {
        *string = message;
        printf("%s\n", message);
    } else if (status != EGADS_SUCCESS) {
        FREE(message);
        *string = udpErrorStr(status);
    } else {
        FREE(message);
    }

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   udpSensitivity - return sensitivity derivatives for the "real" argument *
 *                                                                      *
 ************************************************************************
 */

int
udpSensitivity(ego    ebody,            /* (in)  Body pointer */
               /*@unused@*/int    npnt,             /* (in)  number of points */
               /*@unused@*/int    entType,          /* (in)  OCSM entity type */
               /*@unused@*/int    entIndex,         /* (in)  OCSM entity index (bias-1) */
               /*@unused@*/double uvs[],            /* (in)  parametric coordinates for evaluation */
               /*@unused@*/double vels[])           /* (out) velocities */
{
    int    status = EGADS_SUCCESS;

    int    iudp, judp;

    ROUTINE(udpSensitivity);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    if (uvs != NULL) {
        printf("udpSensitivity(ebody=%llx, npnt=%d, entType=%d, entIndex=%d, uvs=%f %f)\n",
               (long long)ebody, npnt, entType, entIndex, uvs[0], uvs[1]);
    } else {
        printf("udpSensitivity(ebody=%llx, npnt=%d, entType=%d, entIndex=%d, uvs=NULL)\n",
               (long long)ebody, npnt, entType, entIndex);
    }
#endif

    /* check that ebody matches one of the ebodys */
    iudp = 0;
    for (judp = 1; judp <= numUdp; judp++) {
        if (ebody == udps[judp].ebody) {
            iudp = judp;
            break;
        }
    }
    if (iudp <= 0) {
        status = EGADS_NOTMODEL;
        goto cleanup;
    }

    /* the following line should be included if sensitivities
       are not computed analytically */
    status = EGADS_NOLOAD;

cleanup:
#ifdef DEBUG
    printf("udpSensitivity -> vels=%f %f %f\n", vels[0], vels[1], vels[2]);
#endif
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   computeDot - compute normalized dot product at Node in given Body  *
 *                                                                      *
 ************************************************************************
 */

static int
computeDot(ego     ebody,               /* (in)  Body */
           ego     enode,               /* (in)  Node */
           double  *dot)                /* (out) dot product */
{
    int    status = EGADS_SUCCESS;

    int    nedge, oclass, mtype, nchild, *senses;
    double data[18], data0[18], data1[18], len0, len1;
    ego    *eedges=NULL, eref, *echilds;

    ROUTINE(computeDot);

    /* --------------------------------------------------------------- */

    status = EG_getBodyTopos(ebody, enode, EDGE, &nedge, &eedges);
    CHECK_STATUS(EG_getBodyTopos);

    /* Node cannot be removed if there are not 2 associated Edges,
     so return a zero dot product */
    if (nedge != 2) {
        *dot = 0;
        goto cleanup;
    }

    SPLINT_CHECK_FOR_NULL(eedges);

    /* cannot remove Node if either Edge is degenerate */
    status = EG_getInfo(eedges[0], &oclass, &mtype, NULL, NULL, NULL);
    CHECK_STATUS(EG_getInfo);

    if (mtype == DEGENERATE) {
        *dot = 0;
        goto cleanup;
    }

    status = EG_getInfo(eedges[1], &oclass, &mtype, NULL, NULL, NULL);
    CHECK_STATUS(EG_getInfo);

    if (mtype == DEGENERATE) {
        *dot = 0;
        goto cleanup;
    }

    /* find dot product between the ends of the 2 associate Edges */
    status = EG_getTopology(eedges[0], &eref, &oclass, &mtype,
                            data, &nchild, &echilds, &senses);
    CHECK_STATUS(EG_getTopology);

    if (echilds[0] == enode) {
        status = EG_evaluate(eedges[0], &(data[0]), data0);
        CHECK_STATUS(EG_evaluate);
    } else {
        status = EG_evaluate(eedges[0], &(data[1]), data0);
        CHECK_STATUS(EG_evaluate);
    }

    status = EG_getTopology(eedges[1], &eref, &oclass, &mtype,
                            data, &nchild, &echilds, &senses);
    CHECK_STATUS(EG_getTopology);

    if (echilds[0] == enode) {
        status = EG_evaluate(eedges[1], &(data[0]), data1);
        CHECK_STATUS(EG_evaluate);
    } else {
        status = EG_evaluate(eedges[1], &(data[1]), data1);
        CHECK_STATUS(EG_evaluate);
    }

    *dot =     (data0[3] * data1[3] + data0[4] * data1[4] + data0[5] * data1[5]);
    len0 = sqrt(data0[3] * data0[3] + data0[4] * data0[4] + data0[5] * data0[5]);
    len1 = sqrt(data1[3] * data1[3] + data1[4] * data1[4] + data1[5] * data1[5]);

    *dot = *dot / len0 / len1;

cleanup:
    if (eedges != NULL) EG_free(eedges);

    return status;
}
