#include <cstdio>
#include <cstdlib>
#include <cmath>
#include <unordered_map>
#include <vector>

#ifndef WIN32
#ifdef __arm64__
#include <sys/types.h>
#include <sys/sysctl.h>
#endif
#endif

#include "egads.h"
#include "tetgen.h"
#include "emp.h"


struct Interp {
  union {
    int     ind[2];   // ind[0] must be the smaller of the 2 vert indices
    int64_t key;
  } indices;
  double frac;        // the fractional distance between (1.0 all first,
                      //                                  0.0 all second)
};

struct Triad {
  int indices[3];
};

struct DistField {
  double dist;        // the nearest distance
  int    shell;       // the Shell associated with the nearest
};

// structure to pass data to the thread for a block
struct EMPdf {
  long      master;           // master thread ID
  int       index;            // index [0 - np-1]
  int       np;               // number of processors (threads)
  ego       body;             // the Body to be processed
  int       iShell;           // the Shell index
  int       *f2s;             // Face to Shell map
  int       npts;             // number of points in the surface tessellation
  tetgenio  *out;             // the output from TetGen
  double    *distf;           // the distance field
};


extern "C" /*@kept@*/ /*@null@*/ ego EG_context( const ego object );
extern "C" int  EG_outLevel( const ego object );
extern "C" int  EG_sameThread( const ego object );
extern     REAL orient3d(REAL *pa, REAL *pb, REAL *pc, REAL *pd);

extern "C" int  hollowMassProps( ego body, double tessfact, /*@null@*/ const char *stlname,
                                 double *props );

#define UINT16  unsigned short int
#define UINT32  unsigned int
#define REAL32  float


static void
fillSVert(int index, DistField *df, tetgenio &out, std::vector<Interp>& verts,
          int nface, ego *faces, int *f2s, double *off, REAL32 *vrt, int *shell)
{
  int    i, ir, il, is, stat;
  double d, dist, uv[2], v[3], result[3];
  
  ir     = verts[index].indices.ind[0]-1;
  il     = verts[index].indices.ind[1]-1;
  v[0]   = out.pointlist[3*il  ] +
           verts[index].frac*(out.pointlist[3*ir  ] - out.pointlist[3*il  ]);
  v[1]   = out.pointlist[3*il+1] +
           verts[index].frac*(out.pointlist[3*ir+1] - out.pointlist[3*il+1]);
  v[2]   = out.pointlist[3*il+2] +
           verts[index].frac*(out.pointlist[3*ir+2] - out.pointlist[3*il+2]);
  vrt[0] = v[0];
  vrt[1] = v[1];
  vrt[2] = v[2];
  if (df[ir].shell == df[il].shell) {
    *shell = df[ir].shell;
    return;
  }
  *shell = 0;
  dist   = 1.e200;
  for (i = 0; i < nface; i++) {
    is = f2s[i];
    if ((is != df[ir].shell) && (is != df[il].shell)) continue;
    stat = EG_invEvaluate(faces[i], v, uv, result);
    if (stat != EGADS_SUCCESS) {
      printf(" Warning: EG_invEvaluate = %d (fillSVert)!\n", stat);
      continue;
    }
    d = sqrt((v[0]-result[0])*(v[0]-result[0]) +
             (v[1]-result[1])*(v[1]-result[1]) +
             (v[2]-result[2])*(v[2]-result[2]));
    d = fabs(d-off[is-1]);
    if (d >= dist) continue;
    *shell = is;
    dist   = d;
  }
}


static void
writeIso(ego body, int *f2s, double *off, const char *filename, tetgenio &out,
         DistField *df, std::vector<Triad>& tris, std::vector<Interp>& verts)
{
  int  stat, nface;
  ego  *faces;
  char sheader[80] = "written by hollowMassProps                                                   ";

  stat = EG_getBodyTopos(body, NULL, FACE, &nface, &faces);
  if (stat != EGADS_SUCCESS) {
    printf(" Error: EG_getBodyTopos Face = %d (writeIso)!\n", stat);
    return;
  }
  
  FILE *fp = fopen(filename, "wb");
  if (fp == NULL) {
    printf(" Error: Cannot open %s for writing (writeIso)!\n", filename);
    EG_free(faces);
    return;
  }
  unsigned int totTri = tris.size();
  (void) fwrite(sheader, sizeof(char),  80, fp);
  (void) fwrite(&totTri, sizeof(UINT32), 1, fp);
  
  for (unsigned int i = 0; i < totTri; i++) {
    int    shell0,   shell1,   shell2;
    REAL32 vert0[3], vert1[3], vert2[3], norm[3];
    UINT16 col16 = 0;
    fillSVert(tris[i].indices[0], df, out, verts, nface, faces, f2s, off,
              vert0, &shell0);
    fillSVert(tris[i].indices[2], df, out, verts, nface, faces, f2s, off,
              vert1, &shell1);   /* flip */
    fillSVert(tris[i].indices[1], df, out, verts, nface, faces, f2s, off,
              vert2, &shell2);
    if ((shell0 == shell1) && (shell1 == shell2)) col16 = shell1;
    
    norm[0]  = (vert1[1] - vert0[1]) * (vert2[2] - vert0[2]) -
               (vert2[1] - vert0[1]) * (vert1[2] - vert0[2]);
    norm[1]  = (vert1[2] - vert0[2]) * (vert2[0] - vert0[0]) -
               (vert2[2] - vert0[2]) * (vert1[0] - vert0[0]);
    norm[2]  = (vert1[0] - vert0[0]) * (vert2[1] - vert0[1]) -
               (vert2[0] - vert0[0]) * (vert1[1] - vert0[1]);
    
    (void) fwrite(norm,   sizeof(REAL32), 3, fp);
    (void) fwrite(vert0,  sizeof(REAL32), 3, fp);
    (void) fwrite(vert1,  sizeof(REAL32), 3, fp);
    (void) fwrite(vert2,  sizeof(REAL32), 3, fp);
    (void) fwrite(&col16, sizeof(UINT16), 1, fp);
  }
  
  fclose(fp);
  EG_free(faces);
}


static int
makeDF(const ego body, int index, int iShell, int np, int *f2s, int npts,
       tetgenio *out, double *distf)
{
  int    i, j, stat, nFace, outLevel;
  double d, dist, uv[2], xyz[3];
  ego    *faces;

  outLevel = EG_outLevel(body);
  
  stat = EG_getBodyTopos(body, NULL, FACE, &nFace, &faces);
  if (stat != EGADS_SUCCESS) {
    if (outLevel > 0)
      printf(" Error: EG_getBodyTopos Face = %d (makeDF)!\n", stat);
    return stat;
  }

  for (i = npts+index; i < out->numberofpoints; i+=np) {
    dist = 1.e200;
    for (j = 0; j < nFace; j++) {
      if (f2s[j] != iShell+1) continue;
      stat = EG_invEvaluate(faces[j], &out->pointlist[3*i], uv, xyz);
      if (stat != EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" Error: EG_invEvaluate %d Face %d = %d (makeDF)!\n",
                 i+1, j+1, stat);
        EG_free(faces);
        return stat;
      }
      d = sqrt((out->pointlist[3*i  ]-xyz[0])*(out->pointlist[3*i  ]-xyz[0]) +
               (out->pointlist[3*i+1]-xyz[1])*(out->pointlist[3*i+1]-xyz[1]) +
               (out->pointlist[3*i+2]-xyz[2])*(out->pointlist[3*i+2]-xyz[2]));
      if (d > dist) continue;
      dist = d;
    }
    if (dist < 1.e199) {
      distf[i] = dist;
    } else {
      printf(" EGADS Warning: no dist for shell %d/point %d (makeDF)!\n",
             iShell+1, i+1);
    }
  }
  EG_free(faces);

  return EGADS_SUCCESS;
}


static void
dfThread(void *struc)
{
  int   outLevel, stat;
  long  ID;
  EMPdf *dfthread;
  
  dfthread = (EMPdf *) struc;
  outLevel = EG_outLevel(dfthread->body);
  
  /* get our identifier */
  ID = EMP_ThreadID();
  
  stat = makeDF(dfthread->body, dfthread->index, dfthread->iShell, dfthread->np,
                dfthread->f2s,  dfthread->npts,  dfthread->out,
                dfthread->distf);
  if (stat != EGADS_SUCCESS) {
    if (outLevel > 0)
      printf(" Error: makeDF = %d (hollowMassProps)!\n", stat);
  }
  
  /* exhausted all work -- exit */
  if (ID != dfthread->master) EMP_ThreadExit();
}


static void
marchingTets(tetgenio &out, double isoval, DistField *df,
             std::vector<Triad>& tris, std::vector<Interp>& verts)
{
  static int tabpts[16]  = { 0, 3, 3, 4, 3, 4, 4, 3, 3, 4, 4, 3, 4, 3, 3, 0 };
  static int left[16][4] = { {0,0,0,0}, {1,1,1,0}, {2,2,2,0}, {1,2,2,1},
                             {3,3,3,0}, {1,1,3,3}, {2,3,3,2}, {3,2,1,0},
                             {4,4,4,0}, {1,4,4,1}, {2,2,4,4}, {1,2,4,0},
                             {3,3,4,4}, {1,4,3,0}, {2,3,4,0}, {0,0,0,0} };
  static int rite[16][4] = { {0,0,0,0}, {2,4,3,0}, {1,3,4,0}, {3,3,4,4},
                             {2,1,4,0}, {2,4,4,2}, {1,1,4,4}, {4,4,4,0},
                             {3,1,2,0}, {2,2,3,3}, {1,3,3,1}, {3,3,3,0},
                             {2,1,1,2}, {2,2,2,0}, {1,1,1,0}, {0,0,0,0} };
  int    ind[4], cut[4], il, ir;
  double iso[4], frac;
  Triad  tri;
  Interp interp;

  std::unordered_map<int64_t, int> map;
  
  for (int itet = 0; itet < out.numberoftetrahedra; itet++) {
    for (int i = 0; i < 4; i++) {
      ind[i] = out.tetrahedronlist[4*itet+i];
      iso[i] = df[ind[i]-1].dist;
    }
    
    int tabindex = 0;
    for (int i = 0; i < 4; i++) if (iso[i] >= isoval) tabindex |= 1 << i;
    if (tabpts[tabindex] == 0) continue;
    
    for (int i = 0; i < tabpts[tabindex]; i++) {
      il   = left[tabindex][i] - 1;
      ir   = rite[tabindex][i] - 1;
      frac = (isoval - iso[il]) / (iso[ir] - iso[il]);
      if (iso[il] == isoval) {
        interp.indices.ind[0] = interp.indices.ind[1] = ind[il];
        interp.frac           = 1.0;
      } else if (iso[ir] == isoval) {
        interp.indices.ind[0] = interp.indices.ind[1] = ind[ir];
        interp.frac           = 1.0;
      } else if (ind[ir] > ind[il]) {
        interp.indices.ind[0] = ind[il];
        interp.indices.ind[1] = ind[ir];
        interp.frac           = 1.0 - frac;
      } else {
        interp.indices.ind[0] = ind[ir];
        interp.indices.ind[1] = ind[il];
        interp.frac           = frac;
      }
      auto it = map.find(interp.indices.key);
      if (it != map.end()) {
        cut[i] = it->second;
      } else {
/*      if (map.size() != verts.size())
          printf(" *** PROBLEM %zu %zu\n", map.size(), verts.size());  */
        int size                = map.size();
        map[interp.indices.key] = size;
        cut[i]                  = size;
        verts.push_back(interp);
      }
    }
    
    tri.indices[0] = cut[0];
    tri.indices[1] = cut[1];
    tri.indices[2] = cut[2];
    tris.push_back(tri);
    if (tabpts[tabindex] == 3) continue;
    tri.indices[0] = cut[0];
    tri.indices[1] = cut[2];
    tri.indices[2] = cut[3];
    tris.push_back(tri);
  }

}


static void
fillVert(int index, DistField *df, tetgenio &out, std::vector<Interp>& verts,
         double *vert)
{
  int ir  = verts[index].indices.ind[0]-1;
  int il  = verts[index].indices.ind[1]-1;
  vert[0] = out.pointlist[3*il  ] +
            verts[index].frac*(out.pointlist[3*ir  ] - out.pointlist[3*il  ]);
  vert[1] = out.pointlist[3*il+1] +
            verts[index].frac*(out.pointlist[3*ir+1] - out.pointlist[3*il+1]);
  vert[2] = out.pointlist[3*il+2] +
            verts[index].frac*(out.pointlist[3*ir+2] - out.pointlist[3*il+2]);
}


static void
massProps(tetgenio &out, DistField *df, std::vector<Triad>& tris,
          std::vector<Interp>& verts, double *props)
{
  double xa, ya, za, xb, yb, zb;
  double xbar, ybar, zbar, areax, areay, areaz;
  double area=0.0, vol=0.0, xcg=0.0, ycg=0.0, zcg=0.0;
  double Ixx=0.0,  Ixy=0.0, Ixz=0.0, Iyy=0.0, Iyz=0.0, Izz=0.0;

  for (unsigned int i = 0; i < tris.size(); i++) {
    double vert0[3], vert1[3], vert2[3];
    fillVert(tris[i].indices[0], df, out, verts, vert0);
    fillVert(tris[i].indices[2], df, out, verts, vert1);   /* flip */
    fillVert(tris[i].indices[1], df, out, verts, vert2);
    
    xa    = vert1[0] - vert0[0];
    ya    = vert1[1] - vert0[1];
    za    = vert1[2] - vert0[2];
    xb    = vert2[0] - vert0[0];
    yb    = vert2[1] - vert0[1];
    zb    = vert2[2] - vert0[2];
    xbar  = vert0[0] + vert1[0] + vert2[0];
    ybar  = vert0[1] + vert1[1] + vert2[1];
    zbar  = vert0[2] + vert1[2] + vert2[2];
    areax = ya * zb - za * yb;
    areay = za * xb - xa * zb;
    areaz = xa * yb - ya * xb;
    
    area += sqrt(areax*areax + areay*areay + areaz*areaz) / 2;
    vol  += (       xbar*areax +        ybar*areay +        zbar*areaz)/18;
    xcg  += (xbar/2*xbar*areax + xbar  *ybar*areay + xbar  *zbar*areaz)/54;
    ycg  += (ybar  *xbar*areax + ybar/2*ybar*areay + ybar  *zbar*areaz)/54;
    zcg  += (zbar  *xbar*areax + zbar  *ybar*areay + zbar/2*zbar*areaz)/54;
    Ixx  += (ybar*ybar*ybar*areay + zbar*zbar*zbar*areaz)/162;
    Iyy  += (xbar*xbar*xbar*areax + zbar*zbar*zbar*areaz)/162;
    Izz  += (xbar*xbar*xbar*areax + ybar*ybar*ybar*areay)/162;
    
    Ixy  -= (xbar*ybar*xbar*areax/2 + ybar*xbar*ybar*areay/2 +
             xbar*ybar*zbar*areaz  )/162;
    Ixz  -= (xbar*zbar*xbar*areax/2 + xbar*zbar*ybar*areay   +
             zbar*xbar*zbar*areaz/2)/162;
    Iyz  -= (ybar*zbar*xbar*areax   + ybar*zbar*ybar*areay/2 +
             zbar*ybar*zbar*areaz/2)/162;
  }
  
  xcg /= vol;
  ycg /= vol;
  zcg /= vol;
  
  /* parallel-axis theorem */
  Ixx -= vol * (            ycg * ycg + zcg * zcg);
  Iyy -= vol * (xcg * xcg             + zcg * zcg);
  Izz -= vol * (xcg * xcg + ycg * ycg            );
  
  Ixy += vol * (xcg * ycg      );
  Ixz += vol * (xcg *       zcg);
  Iyz += vol * (      ycg * zcg);
  
  /* return the properties */
  props[ 0] = vol;
  props[ 1] = area;
  props[ 2] = xcg;
  props[ 3] = ycg;
  props[ 4] = zcg;
  props[ 5] = Ixx;
  props[ 6] = Ixy;
  props[ 7] = Ixz;
  props[ 8] = Ixy;
  props[ 9] = Iyy;
  props[10] = Iyz;
  props[11] = Ixz;
  props[12] = Iyz;
  props[13] = Izz;
}


static int
seedHole(const ego body, const ego shell, tetgenio &in, tetgenio &empty,
         double *xyz)
{
  int i, k, iface, outLevel, stat, nFace, ntet, v0, n0;
  ego *faces;
  
  xyz[0]   = xyz[1] = xyz[2] = 0.0;
  outLevel = EG_outLevel(body);

  stat = EG_getBodyTopos(body, shell, FACE, &nFace, &faces);
  if (stat != EGADS_SUCCESS) {
    if (outLevel > 0)
      printf(" Error: EG_getBodyTopos Face = %d (seedHole)!\n", stat);
    return stat;
  }
  
  /* compute volume
  double vol = 0.0;
  for (k = 0; k < empty.numberoftetrahedra; k++) {
    int ind[4];
    for (int i = 0; i < 4; i++) ind[i] = empty.tetrahedronlist[4*k+i]-1;
    vol += orient3d(&empty.pointlist[3*ind[3]], &empty.pointlist[3*ind[0]],
                    &empty.pointlist[3*ind[1]], &empty.pointlist[3*ind[2]]);
  }
  printf("  full Volume = %lf\n", vol/6.0); */
  
  /* find first triangle of first Face in the Shell */
  iface = EG_indexBodyTopo(body, faces[0]);
  EG_free(faces);
  if (iface <= EGADS_SUCCESS) {
    if (outLevel > 0)
      printf(" Error: EG_indexBodyTopo = %d (seedHole)!\n", iface);
    return iface;
  }
  for (i = 0; i < in.numberoffacets; i++)
    if (in.facetmarkerlist[i] == iface) break;
  if (i == in.numberoffacets) {
    if (outLevel > 0)
      printf(" Error: Triangle for Face %d not found (seedHole)!\n", iface);
    return EGADS_NOTFOUND;
  }
  tetgenio::facet   *f = &in.facetlist[i];
  tetgenio::polygon *p = f->polygonlist;
  
  /* find the pair of tets that contain the triangle */
  for (ntet = k = 0; k < empty.numberoftetrahedra; k++) {
    bool match[3] = {false, false, false};
    bool other[4] = {true,  true,  true,  true};
    for (v0 = 0; v0 < 3; v0++) {
      for (n0 = 0; n0 < 4; n0++) {
        if (p->vertexlist[v0] == empty.tetrahedronlist[4*k+n0]) {
          match[v0] = true;
          other[n0] = false;
          break;
        }
      }
    }
    if (match[0] && match[1] && match[2]) {
      /* found a match -- does the volume have the correct sign? */
      int           indx = empty.tetrahedronlist[4*k  ];
      if (other[1]) indx = empty.tetrahedronlist[4*k+1];
      if (other[2]) indx = empty.tetrahedronlist[4*k+2];
      if (other[3]) indx = empty.tetrahedronlist[4*k+3];
      double vol = orient3d(&in.pointlist[3*p->vertexlist[0]-3],
                            &in.pointlist[3*p->vertexlist[1]-3],
                            &in.pointlist[3*p->vertexlist[2]-3],
                            &empty.pointlist[3*indx-3])/6.0;
/*    printf(" %d: vol = %le, others = %d %d %d %d\n",
             ntet, vol, other[0], other[1], other[2], other[3]);  */
      if (vol < 0.0) {
        xyz[0] = (in.pointlist[3*p->vertexlist[0]-3] +
                  in.pointlist[3*p->vertexlist[1]-3] +
                  in.pointlist[3*p->vertexlist[2]-3] +
                  empty.pointlist[3*indx-3])/4.0;
        xyz[1] = (in.pointlist[3*p->vertexlist[0]-2] +
                  in.pointlist[3*p->vertexlist[1]-2] +
                  in.pointlist[3*p->vertexlist[2]-2] +
                  empty.pointlist[3*indx-2])/4.0;
        xyz[2] = (in.pointlist[3*p->vertexlist[0]-1] +
                  in.pointlist[3*p->vertexlist[1]-1] +
                  in.pointlist[3*p->vertexlist[2]-1] +
                  empty.pointlist[3*indx-1])/4.0;
        return EGADS_SUCCESS;
      }
      ntet++;
    }
  }
  
  if (outLevel > 0)
    printf(" Error: No negative tet for Face %d  ntet = %d (seedHole)!\n",
           iface, ntet);

  return EGADS_NOTFOUND;
}


int
hollowMassProps(ego src, double tessfact, /*@null@*/ const char *stlname, double *massprops)
{
  int          i, j, k, np, stat, outLevel, oclass, mtype, nShell, nFace;
  int          npts, len, ntri, ntris, pt, pi, iface, nface, aType, aLen;
  int          *senses, *f2s = NULL;
  long         start;
  double       d, offset, size, tparms[3], box[6], *dist = NULL;
  const char   *str;
  const int    *ptype, *pindex, *tris, *tric, *ints;
  const double *xyzs, *uvs, *reals;
  ego          context, body, tess, geom, dum, *shells, *faces;
  tetgenio     in, empty, out;
  DistField    *df = NULL;
  void         *threads[23];
  EMPdf        dfmaster, dfthread[23];
#ifndef WIN32
#ifdef __arm64__
  size_t       lenp;
  int          nproc;
#endif
#endif
  
  for (i = 0; i < 14; i++) massprops[i] = 0.0;
  if  (src == NULL)                  return EGADS_NULLOBJ;
  if  (src->magicnumber != MAGIC)    return EGADS_NOTOBJ;
  if ((src->oclass != BODY) &&
      (src->oclass != TESSELLATION)) return EGADS_NOTBODY;
  if  (src->blind == NULL)           return EGADS_NODATA;
  if  (EG_sameThread(src))           return EGADS_CNTXTHRD;
  outLevel = EG_outLevel(src);
  context  = EG_context(src);
  if  (context == NULL)              return EGADS_NOTCNTX;
  
  if (src->oclass == BODY) {
    tess = NULL;
    body = src;
  } else {
    tess = src;
    stat = EG_statusTessBody(src, &body, &i, &npts);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" Error: EG_statusTessBody = %d (hollowMassProps)!\n", stat);
      return stat;
    }
  }
  if (body->mtype != SOLIDBODY) {
    if (outLevel > 0)
      printf(" Error: Body is not a Solid (hollowMassProps)!\n");
    return EGADS_TOPOERR;
  }
  
  /* get the shells & offsets */
  stat = EG_getTopology(body, &geom, &oclass, &mtype, NULL, &nShell, &shells,
                        &senses);
  if (stat != EGADS_SUCCESS) {
    if (outLevel > 0)
      printf(" Error: EG_getTopology = %d (hollowMassProps)!\n", stat);
    return stat;
  }
  std::vector<double> off(nShell);
  for (i = 0; i < nShell; i++) {
    stat = EG_attributeRet(shells[i], "__offset__", &aType, &aLen,
                           &ints, &reals, &str);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" Error: EG_attributeGet offset %d = %d (hollowMassProps)!\n",
               i+1, stat);
      return stat;
    }
    if (aType != ATTRREAL) {
      if (outLevel > 0)
        printf(" Error: Wrong Type -> Attribute offset %d (hollowMassProps)!\n",
               i+1);
      return EGADS_ATTRERR;
    }
    off[i] = reals[0];
  }
  offset = 0.0;
  for (i = 0; i < nShell; i++) offset += off[i];
  offset /= nShell;
  
  std::vector<Triad>  ctris;
  std::vector<Interp> verts;
  
  /* map Faces to Shells */
  stat = EG_getBodyTopos(body, NULL, FACE, &nFace, NULL);
  if (stat != EGADS_SUCCESS) {
    if (outLevel > 0)
      printf(" Error: EG_getBodyTopos Face = %d (hollowMassProps)!\n", stat);
    goto cleanup;
  }
  f2s = new int[nFace];
  for (i = 0; i < nFace;  i++) f2s[i] = 0;
  for (i = 0; i < nShell; i++) {
    stat = EG_getBodyTopos(body, shells[i], FACE, &nface, &faces);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" Error: EG_getBodyTopos Shell %d = %d (hollowMassProps)!\n",
               i+1, stat);
      goto cleanup;
    }
    for (j = 0; j < nface; j++) {
      iface = EG_indexBodyTopo(body, faces[j]);
      if (iface <= EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" Error: EG_indexBodyTopo %d/%d = %d (hollowMassProps)!\n",
                 i+1, j+1, iface);
        stat = iface;
        EG_free(faces);
        goto cleanup;
      }
      f2s[iface-1] = i+1;
    }
    EG_free(faces);
  }
  
  /* make a tessellation object if we do not already have one */
  if (tess == NULL) {
    stat = EG_getBoundingBox(body, box);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" Error: EG_getBoundingBox = %d (hollowMassProps)!\n", stat);
      goto cleanup;
    }
                              size = box[3]-box[0];
    if (size < box[4]-box[1]) size = box[4]-box[1];
    if (size < box[5]-box[2]) size = box[5]-box[2];
    tparms[0] =  tessfact*offset;
    tparms[1] =  0.0075*size;
    tparms[2] = 4.0;
    stat = EG_makeTessBody(body, tparms, &tess);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" Error: EG_makeTessBody = %d (hollowMassProps)!\n", stat);
      goto cleanup;
    }
    stat = EG_statusTessBody(tess, &dum, &i, &npts);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" Error: EG_statusTessBody = %d (hollowMassProps)!\n", stat);
      goto cleanup;
    }
  }
  
  /* is the tessellation complete? */
  ntris = 0;
  for (iface = 1; iface <= nFace; iface++) {
    stat = EG_getTessFace(tess, iface, &len, &xyzs, &uvs, &ptype, &pindex,
                          &ntri, &tris, &tric);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" Error: EG_getTessFace %d = %d (hollowMassProps)!\n",
               iface, stat);
      goto cleanup;
    }
    if ((len == 0) || (ntri == 0)) {
      if (outLevel > 0)
        printf(" Error: Face %d is empty (hollowMassProps)!\n", iface);
      stat = EGADS_TESSTATE;
      goto cleanup;
    }
    ntris += ntri;
  }

#ifdef DEBUG
  double props[14];
  stat = EG_tessMassProps(tess, props);
  if (stat != EGADS_SUCCESS) {
    if (outLevel > 0)
      printf(" Error: EG_tessMassProps = %d (hollowMassProps)!\n", stat);
    goto cleanup;
  }
  printf(" tessMassProps:  nFace = %d  nPts = %8d    nTris = %8d   vol = %lf\n",
         nFace, npts, ntris, props[0]);
#endif
  
  /* tetrahedralize using TetGen */
  in.firstnumber    = 1;
  in.numberofpoints = npts;
  in.numberoffacets = ntris;
  in.numberofholes  = 0;
  
  in.pointlist = new REAL[3*npts];
  for (i = 0; i < npts; i++) {
    stat = EG_getGlobal(tess, i+1, &pt, &pi, &in.pointlist[3*i]);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" Error: EG_getGlobal %d = %d (hollowMassProps)!\n", i+1, stat);
      goto cleanup;
    }
  }
  
  in.facetlist       = new tetgenio::facet[ntris];
  in.facetmarkerlist = new int[ntris];
  i = 0;
  for (iface = 1; iface <= nFace; iface++) {
    (void) EG_getTessFace(tess, iface, &len, &xyzs, &uvs, &ptype, &pindex,
                          &ntri, &tris, &tric);
    for (j = 0; j < ntri; j++, i++) {
      tetgenio::facet   *f;
      tetgenio::polygon *p;
      in.facetmarkerlist[i] = iface;
      f = &in.facetlist[i];
      f->numberofpolygons = 1;
      f->polygonlist = new tetgenio::polygon[1];
      f->numberofholes = 0;
      f->holelist = NULL;
      p = &f->polygonlist[0];
      p->numberofvertices = 3;
      p->vertexlist = new int[3];
      stat = EG_localToGlobal(tess, iface, tris[3*j  ], &p->vertexlist[0]);
      if (stat != EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" Error: EG_localToGlobal0 %d/%d = %d (hollowMassProps)!\n",
                 iface, j+1, stat);
        goto cleanup;
      }
      stat = EG_localToGlobal(tess, iface, tris[3*j+1], &p->vertexlist[1]);
      if (stat != EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" Error: EG_localToGlobal1 %d/%d = %d (hollowMassProps)!\n",
                 iface, j+1, stat);
        goto cleanup;
      }
      stat = EG_localToGlobal(tess, iface, tris[3*j+2], &p->vertexlist[2]);
      if (stat != EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" Error: EG_localToGlobal2 %d/%d = %d (hollowMassProps)!\n",
                 iface, j+1, stat);
        goto cleanup;
      }
    }
  }
  
  /* set holes for inner Shells */
  if (nShell > 1) {
    try {
      tetrahedralize((char *) "pYQ", &in, &empty);
    }
    catch (...) {
      printf(" Error: TetGen empty exception (hollowMassProps)!\n");
      stat = EGADS_OCSEGFLT;
      goto cleanup;
    }
    REAL *holelist = new REAL[3*(nShell-1)];
    for (j = i = 0; i < nShell; i++) {
      if (senses[i] == SOUTER) continue;
      stat = seedHole(body, shells[i], in, empty, &holelist[3*j]);
      if (stat != EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" Error: seedHole %d = %d (hollowMassProps)!\n", i+1, stat);
        stat = EGADS_TESSTATE;
        goto cleanup;
      }
      j++;
    }
    in.numberofholes = nShell-1;
    in.holelist      = holelist;
  }
  
  try {
    tetrahedralize((char *) "pYq1.2Q", &in, &out);
  }
  catch (...) {
    printf(" Error: TetGen exception (hollowMassProps)!\n");
    stat = EGADS_OCSEGFLT;
    goto cleanup;
  }
#ifdef DEBUG
  printf("                           nVert = %8d    nTets = %8d\n",
         out.numberofpoints, out.numberoftetrahedra);
  printf("                                              nTFace = %8d\n\n",
         out.numberoftrifaces);
#endif
  /* check to see if we have the same initial verts
   for (i = 0; i < npts; i++)
     if ((in.pointlist[3*i  ] != out.pointlist[3*i  ]) ||
         (in.pointlist[3*i+1] != out.pointlist[3*i+1]) ||
         (in.pointlist[3*i+2] != out.pointlist[3*i+2]))
       printf(" PointList mismatch at %d\n", i+1); */
  
  /* allocate the distance fields */
  df   = new DistField[out.numberofpoints];
  dist = new double[out.numberofpoints];
  
  /* setup threading */
  np = EMP_Init(&start);
#ifndef WIN32
#ifdef __arm64__
  if (np != 1) {
    /* np from EMP_Init may be too restrictive here for Apple Silicon */
    lenp = sizeof(nproc);
    sysctlbyname("hw.perflevel0.physicalcpu", &nproc, &lenp, NULL, 0);
    if ((nproc > 1) && (nproc > np)) np = nproc;
  }
#endif
#endif
  if (np > 24) np = 24;
  if (outLevel > 1) printf(" EMP NumProcs = %d!\n", np);
  dfmaster.master = EMP_ThreadID();
  dfmaster.np     = np;
  dfmaster.body   = body;
  dfmaster.index  = 0;
  dfmaster.f2s    = f2s;
  dfmaster.npts   = npts;
  dfmaster.out    = &out;
  dfmaster.distf  = dist;

  /* create the threads and get going -- one Shell at a time! */
  for (j = 0; j < nShell; j++) {
    dfmaster.iShell = j;
    for (k = 0; k < npts; k++) dist[k] = 0.0;
    for (i = 0; i < np-1; i++) {
      dfthread[i]       = dfmaster;
      dfthread[i].index = i+1;
      threads[i]        = EMP_ThreadCreate(dfThread, &dfthread[i]);
      if (threads[i] == NULL) {
        printf(" EMP Error Creating Thread #%d!\n", i+1);
        stat = EGADS_CNTXTHRD;
        goto cleanup;
      }
    }
    /* now run the thread block from the original thread */
    dfThread(&dfmaster);
    
    /* wait for all others to return */
    for (i = 0; i < np-1; i++)
      if (threads[i] != NULL) EMP_ThreadWait(threads[i]);
    for (i = 0; i < np-1; i++)
      if (threads[i] != NULL) EMP_ThreadDestroy(threads[i]);
    
    /* set or update the signed distance field */
    if (j == 0) {
      for (k = 0; k < out.numberofpoints; k++) {
        df[k].dist  = dist[k] - off[0];
        df[k].shell = j+1;
      }
    } else {
      for (k = 0; k < out.numberofpoints; k++) {
        d = dist[k] - off[j];
        if (d >= df[k].dist) continue;
        df[k].dist  = d;
        df[k].shell = j+1;
      }
    }
  }
  
  /* use "marching tets" to create our discrete surface(s) */
  marchingTets(out, 0.0, df, ctris, verts);
#ifdef DEBUG
  printf("                                             numTris = %zu  numVrts = %zu\n",
         ctris.size(), verts.size());
#endif

  massProps(out, df, ctris, verts, massprops);
  if (stlname != NULL)
    writeIso(body, f2s, off.data(), stlname, out, df, ctris, verts);
  stat = EGADS_SUCCESS;
  
cleanup:
  if (src != tess) EG_deleteObject(tess);
  delete [] f2s;
  delete [] dist;
  delete [] df;
  
  return stat;
}
