/*
 ************************************************************************
 *                                                                      *
 * udfHollowProps -- compute mass properties of a hollowed Body         *
 *                                                                      *
 *            Written by John Dannenhoffer @ Syracuse University        *
 *            Patterned after code written by Bob Haimes  @ MIT         *
 *                                                                      *
 ************************************************************************
 */

/*
 * Copyright (C) 2011/2026  John F. Dannenhoffer, III (Syracuse University)
 *
 * This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *     MA  02110-1301  USA
 */

//#define DEBUG

#define NUMUDPARGS 13
#define NUMUDPINPUTBODYS 1
#include "udpUtilities.h"

/* shorthands for accessing argument values and velocities */
#define TESSFACT(IUDP)  ((double *) (udps[IUDP].arg[ 0].val))[0]
#define VOLUME(  IUDP)  ((double *) (udps[IUDP].arg[ 1].val))[0]
#define AREA(    IUDP)  ((double *) (udps[IUDP].arg[ 2].val))[0]
#define XCG(     IUDP)  ((double *) (udps[IUDP].arg[ 3].val))[0]
#define YCG(     IUDP)  ((double *) (udps[IUDP].arg[ 4].val))[0]
#define ZCG(     IUDP)  ((double *) (udps[IUDP].arg[ 5].val))[0]
#define IXX(     IUDP)  ((double *) (udps[IUDP].arg[ 6].val))[0]
#define IXY(     IUDP)  ((double *) (udps[IUDP].arg[ 7].val))[0]
#define IXZ(     IUDP)  ((double *) (udps[IUDP].arg[ 8].val))[0]
#define IYY(     IUDP)  ((double *) (udps[IUDP].arg[ 9].val))[0]
#define IYZ(     IUDP)  ((double *) (udps[IUDP].arg[10].val))[0]
#define IZZ(     IUDP)  ((double *) (udps[IUDP].arg[11].val))[0]
#define STLFILE( IUDP)  ((char   *) (udps[IUDP].arg[12].val))

/* data about possible arguments */
static char  *argNames[NUMUDPARGS] = {"tessfact", "volume",  "area",    "xcg",     "ycg",     "zcg",
                                      "ixx",      "ixy",     "ixz",     "iyy",     "iyz",     "izz",     "stlfile",};
static int    argTypes[NUMUDPARGS] = { ATTRREAL,  -ATTRREAL, -ATTRREAL, -ATTRREAL, -ATTRREAL, -ATTRREAL,
                                      -ATTRREAL,  -ATTRREAL, -ATTRREAL, -ATTRREAL, -ATTRREAL, -ATTRREAL, ATTRFILE, };
static int    argIdefs[NUMUDPARGS] = {0,          0,          0,         0,         0,         0,
                                      0,          0,         0,         0,         0,         0,         0,        };
static double argDdefs[NUMUDPARGS] = {0.333,      0.,        0.,        0.,        0.,        0.,
                                      0.,         0.,        0.,        0.,        0.,        0.,        0.,       };

/* get utility routines: udpErrorStr, udpInitialize, udpReset, udpSet,
                         udpGet, udpVel, udpClean, udpMesh */
#include "udpUtilities.c"

#include "OpenCSM.h"
#include "hollowMassProps.h"

/* message (to be shared by all functions) */
static char message[1024];


/*
 ************************************************************************
 *                                                                      *
 *   udpExecute - execute the primitive                                 *
 *                                                                      *
 ************************************************************************
 */

int
udpExecute(ego  emodel,                 /* (in)  Model containing Body */
           ego  *ebody,                 /* (out) Body pointer */
           int  *nMesh,                 /* (out) number of associated meshes */
           char *string[])              /* (out) error message */
{
    int     status = EGADS_SUCCESS;

    int     oclass, mtype, nchild, *senses, nface, iface, nshell, attrType, attrLen;
    CINT    *tempIlist;
    double  data[18], props[14];
    CDOUBLE *tempRlist, *tempRlist2;
    CCHAR   *tempClist;
    ego     eref, *ebodys, *efaces=NULL, *eshells=NULL;
    udp_T   *udps = *Udps;

    ROUTINE(udpExecute);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    printf("udpExecute(emodel=%llx)\n", (long long)emodel);
#endif

    /* default return values */
    *ebody  = NULL;
    *nMesh  = 0;
    *string = NULL;

    message[0] = '\0';

    /* check that Model was input that contains one Body */
    status = EG_getTopology(emodel, &eref, &oclass, &mtype,
                            data, &nchild, &ebodys, &senses);
    CHECK_STATUS(EG_getTopology);

    if (oclass != MODEL) {
        snprintf(message, 1024, "expecting a Model\n");
        status = EGADS_NOTMODEL;
        goto cleanup;
    } else if (nchild != 1) {
        snprintf(message, 1024, "expecting Model to contain one Body (not %d)\n", nchild);
        status = EGADS_NOTBODY;
        goto cleanup;
    }

    /* check arguments */
    if        (udps[0].arg[0].size >= 2) {
        snprintf(message, 100, "\"tessfact\" should be a scalar");
        status  = EGADS_RANGERR;
        goto cleanup;

    } else if (TESSFACT(0) <= 0) {
        snprintf(message, 100, "\"tessfact\" should be a positive");
        status  = EGADS_RANGERR;
        goto cleanup;
    }

    /* make sure input is a Solid SheetBody */
    status = EG_getInfo(ebodys[0], &oclass, &mtype, NULL, NULL, NULL);
    CHECK_STATUS(EG_getInfo);

    /* if planar, make planar offset */
    if (mtype != SOLIDBODY) {
        snprintf(message, 1024, "expectign a SolidBody");
        status = EGADS_NOTBODY;
        goto cleanup;
    }

    /* cache copy of arguments for future use */
    status = cacheUdp(NULL);
    CHECK_STATUS(cacheUdp);

    /* make a copy of the Body (so that it does not get removed
     when OpenCSM deletes emodel) */
    status = EG_copyObject(ebodys[0], NULL, ebody);
    CHECK_STATUS(EG_copyObject);

    if (*ebody == NULL) goto cleanup;   // needed for splint

    /* transfer __offset__ from the Faces to their Shell */
    status = EG_getBodyTopos(*ebody, NULL, FACE, &nface, &efaces);
    CHECK_STATUS(EG_getBodyTopos);

    SPLINT_CHECK_FOR_NULL(efaces);

    for (iface = 0; iface < nface; iface++) {
        status = EG_attributeRet(efaces[iface], "__offset__", &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status != EGADS_SUCCESS) continue;

        if (attrType != ATTRREAL || attrLen != 1) {
            snprintf(message, 1024, "expecting __offset__ attribute on Face %d to be one real", iface+1);
            status = EGADS_ATTRERR;
            goto cleanup;
        }
        
        status = EG_getBodyTopos(*ebody, efaces[iface], SHELL, &nshell, &eshells);
        CHECK_STATUS(EG_getBodyTopos);

        SPLINT_CHECK_FOR_NULL(eshells);

        status = EG_attributeRet(eshells[0], "__offset__", &attrType, &attrLen,
                                 &tempIlist, &tempRlist2, &tempClist);
        if (status != EGADS_SUCCESS) {
            status = EG_attributeAdd(eshells[0], "__offset__", ATTRREAL, 1,
                                     NULL, tempRlist, NULL);
            CHECK_STATUS(EG_attributeAll);
        } else if (tempRlist[0] != tempRlist2[0]) {
            snprintf(message, 1024, "__offset__ on Face %d and its Shell disagree", iface+1);
            status = EGADS_ATTRERR;
            goto cleanup;
        }

        EG_free(eshells);   eshells = NULL;
    }

    EG_free(efaces);   efaces = NULL;

    /* compute the mass properties */
    if (strlen(STLFILE(0)) == 0) {
        status = hollowMassProps(*ebody, TESSFACT(0), NULL,       props);
    } else {
        status = hollowMassProps(*ebody, TESSFACT(0), STLFILE(0), props);
    }
    CHECK_STATUS(hollowMassProps);

    /* add a special Attribute to the Body to tell OpenCSM that there
       is no topological change and hence it should not adjust the
       Attributes on the Body in finishBody() */
    status = EG_attributeAdd(*ebody, "__noTopoChange__", ATTRSTRING,
                             0, NULL, NULL, "udfEditAttr");
    CHECK_STATUS(EG_attributeAdd);

    /* set the output value */
    VOLUME(numUdp) = props[ 0];
    AREA(  numUdp) = props[ 1];
    XCG(   numUdp) = props[ 2];
    YCG(   numUdp) = props[ 3];
    ZCG(   numUdp) = props[ 4];
    IXX(   numUdp) = props[ 5];
    IXY(   numUdp) = props[ 6];
    IXZ(   numUdp) = props[ 7];
//  IXY(   numUdp) = props[ 8];
    IYY(   numUdp) = props[ 9];
    IYZ(   numUdp) = props[10];
//  IXZ(   numUdp) = props[11];
//  IYZ(   numUdp) = props[12];
    IZZ(   numUdp) = props[13];

    /* return the new Body */
    udps[numUdp].ebody = *ebody;

cleanup:
    if (efaces  != NULL) EG_free(efaces );
    if (eshells != NULL) EG_free(eshells);
    
    if (strlen(message) > 0) {
        MALLOC(*string, char, 1024);
        strncpy(*string, message, 1023);
    } else if (status != EGADS_SUCCESS) {
        *string = udpErrorStr(status);
    }

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   udpSensitivity - return sensitivity derivatives for the "real" argument *
 *                                                                      *
 ************************************************************************
 */

int
udpSensitivity(ego    ebody,            /* (in)  Body pointer */
   /*@unused@*/int    npnt,             /* (in)  number of points */
   /*@unused@*/int    entType,          /* (in)  OCSM entity type */
   /*@unused@*/int    entIndex,         /* (in)  OCSM entity index (bias-1) */
   /*@unused@*/double uvs[],            /* (in)  parametric coordinates for evaluation */
   /*@unused@*/double vels[])           /* (out) velocities */
{
    int iudp, judp;

    ROUTINE(udpSensitivity);

    /* --------------------------------------------------------------- */

    /* check that ebody matches one of the ebodys */
    iudp = 0;
    for (judp = 1; judp <= numUdp; judp++) {
        if (ebody == udps[judp].ebody) {
            iudp = judp;
            break;
        }
    }
    if (iudp <= 0) {
        return EGADS_NOTMODEL;
    }

    /* this routine is not written yet */
    return EGADS_NOLOAD;
}
