# Import pyCAPS module
import pyCAPS

# Import os module
import os

# Import argparse module
import argparse

# Setup and read command line options. Please note that this isn't required for pyCAPS
parser = argparse.ArgumentParser(description = 'AFLR4 Symmetry PyTest Example',
                                 prog = 'aflr4_Symmetry_PyTest',
                                 formatter_class = argparse.ArgumentDefaultsHelpFormatter)

#Setup the available commandline options
parser.add_argument('-workDir', default = ["." + os.sep], nargs=1, type=str, help = 'Set working/run directory')
parser.add_argument("-outLevel", default = 1, type=int, choices=[0, 1, 2], help="Set output verbosity")
args = parser.parse_args()

# Working directory
workDir = os.path.join(str(args.workDir[0]), "AFLRSymmetryAnalysisTest")

# Load CSM file and build the geometry explicitly
geometryScript = os.path.join("..","csmData","cfdSymmetry.csm")
capsProblem = pyCAPS.Problem(problemName = workDir,
                             capsFile=geometryScript,
                             outLevel=args.outLevel)

capsProblem.geometry.cfgpmtr.sharpte = 1

# Load AFLR4 aim
aflr4 = capsProblem.analysis.create(aim = "aflr4AIM")

# Mesing boundary conditions
aflr4.input.Mesh_Sizing = {"Farfield": {"bcType":"Farfield"},
                           "Symmetry": {"bcType":"Symmetry"}}

# Set AIM verbosity
aflr4.input.Mesh_Quiet_Flag = True if args.outLevel == 0 else False

# Optional: Explicitly write mesh files
aflr4.input.Mesh_Format = "Tecplot"

# Farfield growth factor
aflr4.input.ff_cdfr = 1.3

# Generate quads and tris
# aflr4.input.Mesh_Gen_Input_String = "mquad=1 mpp=3"

# Make quad meshes
aflr4.input.AFLR4_Quad = True

# Set maximum and minimum edge lengths relative to capsMeshLength
aflr4.input.max_scale = 0.5
aflr4.input.ideal_min_scale = 0.05

# Set additional meshing inputs
aflr4.input.Mesh_Length_Factor = 5
aflr4.input.curv_angle = 10
aflr4.input.erw_all = 0.1
aflr4.input.mer_all = True


######################################
## AFRL4 executes automatically     ##
######################################

######################################
## Build volume mesh off of surface ##
##  mesh(es) using AFLR3            ##
######################################

# Load AFLR3 aim
aflr3 = capsProblem.analysis.create(aim = "aflr3AIM")

# Link the surface mesh
aflr3.input["Surface_Mesh"].link(aflr4.output["Surface_Mesh"])

# Set AIM verbosity
aflr3.input.Mesh_Quiet_Flag = True if args.outLevel == 0 else False

# Optional: Explicitly write mesh files
Mesh_Format = ["Tecplot"]
if os.path.isfile( os.path.join(os.environ["ESP_ROOT"],"lib","exodusWriter.so") ):
    Mesh_Format.append("EXODUS")
aflr3.input.Mesh_Format = Mesh_Format

# Set either global or local boundary layer thickness spacings
# These are coarse numbers just as an example, not a recommenation for good CFD solutions
useGlobal = False

if useGlobal:
    aflr3.input.BL_Initial_Spacing = 0.01
    aflr3.input.BL_Thickness = 0.1
    
    # Mesing boundary conditions
    aflr3.input.Mesh_Sizing = {"Farfield": {"bcType":"Farfield"},
                               "Symmetry": {"bcType":"Symmetry"}}
else:
    inviscidBC = {"boundaryLayerThickness" : 0.0, "boundaryLayerSpacing" : 0.0}
    viscousBC  = {"boundaryLayerThickness" : 0.1, "boundaryLayerSpacing" : 0.01}

    # Set mesh sizing parmeters
    aflr3.input.Mesh_Sizing = {"Wing1": viscousBC, "Wing2": inviscidBC,
                               "Farfield": {"bcType":"Farfield"},
                               "Symmetry": {"bcType":"Symmetry"}}

# Run AIM
aflr3.runAnalysis()
