/*
 ************************************************************************
 *                                                                      *
 * udfTrain2 -- training UDF                                            *
 *                                                                      *
 *             this modified attributes on a Body                       *
 *             sensitivities are not computed                           *
 *                                                                      *
 *             Written by John Dannenhoffer @ Geocentric Technologies   *
 *                                                                      *
 ************************************************************************
 */

/*
 * Copyright (C) 2025  John F. Dannenhoffer, III (Geocentric Technologies))
 *
 * This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *     MA  02110-1301  USA
 */

/* uncomment the following to get DEBUG printouts */
//#define DEBUG 1

/* the number of "input" Bodys

   this only needs to be specified if this is a UDF (user-defined
   function) that consumes Bodys from OpenCSM's stack.  (the default
   value is 0).

   if NUMUDPINPUTBODYS>0  then exactly NUMUDPINPUTBODYS are in emodel
   if NUMUDPINPUTBODYS<0  then up to  -NUMUDPINPUTBODYS are in emodel
*/
#define NUMUDPINPUTBODYS 1

/* the number of arguments (specified below) */
#define NUMUDPARGS 2

/* set up the necessary structures (uses NUMUDPARGS) */
#include "udpUtilities.h"

/* shorthand macros for accessing argument values and velocities */
#define FILENAME(IUDP)  ((char   *) (udps[IUDP].arg[0].val))
#define NSKIP(   IUDP)  ((int    *) (udps[IUDP].arg[1].val))[0]

/* data about possible arguments
      argNames: argument name (must be all lower case)
      argTypes: argument type: +ATTRINT      integer input
                               -ATTRINT      integer output
                               +ATTRREAL     double input  (no sensitivities)
                               -ATTRREAL     double output (no sensitivities)
                               +ATTRREALSEN  double input  (with sensitivities)
                               -ATTRREALSEN  double output (with sensitivities)
                               +ATTRSTRING   string input
                               -ATTRSTRING   *** cannot be used ***
                               +ATTRFILE     input file
                               -ATTRFILE     *** cannot be used ***
                               +ATTRREBUILD  forces rebuild if any variable in
                                             semi-colon-separated list has been changed
                               -ATTRREBUILD  *** cannot be used ***
                               +ATTRRECYCLE  forces rebuild (always) by blocking recycling
                               -ATTRRECYCLE  *** cannot be used ***
      argIdefs: default value for ATTRINT
      argDdefs: default value for ATTRREAL or ATTRREALSEN */
static char  *argNames[NUMUDPARGS] = {"filename", "nskip",  };
static int    argTypes[NUMUDPARGS] = {ATTRFILE,   -ATTRINT, };
static int    argIdefs[NUMUDPARGS] = {0,          0,        };
static double argDdefs[NUMUDPARGS] = {1.,         0.,       };

/* get utility routines: udpErrorStr, udpInitialize, udpReset, udpSet,
                         udpGet, udpVel, udpClean, udpMesh */
#include "udpUtilities.c"

static int addAttribute(ego eobj, char attrName[], char attrValue[], int index);
#ifdef DEBUG
static int printAttributes(ego eobj);
#endif
static int convertAtSign(char instring[], int number, char outstring[]);


/*
 ************************************************************************
 *                                                                      *
 *   udpExecute - execute the primitive                                 *
 *                                                                      *
 ************************************************************************
 */

int
udpExecute(ego  emodel,                 /* (in)  Model containing Body */
           ego  *ebody,                 /* (out) Body (or model) pointer */
           int  *nMesh,                 /* (out) number of associated meshes */
           char *string[])              /* (out) error message */
{
    int     status = EGADS_SUCCESS;

    char    *message=NULL;
    udp_T   *udps = *Udps;

    int     oclass, mtype, nchild, *senses, i, nskip;
    int     ieof, inext, nread;
    int     nnode, inode, nedge, iedge, nface, iface;
    double  data[18];
    char    templine[256], command[256], attrName[256], attrValue[256];
    ego     *ebodys, context, eref, modl;
    ego     *enodes=NULL, *eedges=NULL, *efaces=NULL;
    FILE    *fp=NULL;

    ROUTINE(udpExecute);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    /* debug printing of the input arguments */
    printf("udpExecute(emodel=%llx)\n", (long long)emodel);
    printf("filename(0) = %s\n", FILENAME(0));
    printf("nskip(   0) = %d\n", NSKIP(   0));
#endif

    /* default return values */
    *ebody  = NULL;
    *nMesh  = 0;
    *string = NULL;

    /* the place where messages to the user are placed */
    MALLOC(message, char, 100);
    message[0] = '\0';

    /* number of directives containing an error */
    nskip = 0;

    /* check that Model was input that contains one Body */
    status = EG_getTopology(emodel, &eref, &oclass, &mtype,
                            data, &nchild, &ebodys, &senses);
    CHECK_STATUS(EG_getTopology);

    if (oclass != MODEL) {
        snprintf(message, 100, "expecting a Model, but found oclass=%d", oclass);
        status = EGADS_NOTMODEL;
        goto cleanup;
    } else if (nchild != 1) {
        snprintf(message, 100, "expecting Model to contain one Body (not %d)", nchild);
        status = EGADS_NOTBODY;
        goto cleanup;
    }

    /* check arguments */
    if (STRLEN(FILENAME(0)) <= 0) {
        snprintf(message, 100, "\"filename\" must be specified");
        status = OCSM_ILLEGAL_VALUE;
        goto cleanup;
    }

    /* cache copy of arguments for future use */
    status = cacheUdp(NULL);
    CHECK_STATUS(cacheUdp);

#ifdef DEBUG
    /* debug printing of cached input arguments */
    printf("filename(%d) = %s\n", numUdp, FILENAME(numUdp));
    printf("nskip(   %d) = %d\n", numUdp, NSKIP(   numUdp));
#endif

    /* make a copy of the Body (so that it does not get removed
     when OpenCSM deletes emodel) */
    status = EG_copyObject(ebodys[0], NULL, ebody);
    CHECK_STATUS(EG_copyObject);

    SPLINT_CHECK_FOR_NULL(*ebody);

    /* get pointer to model */
    status = EG_getContext(emodel, &context);
    CHECK_STATUS(EG_getContext);

    status = EG_getUserPointer(context, (void**)(&(modl)));
    CHECK_STATUS(EG_getUserPointer);

    /* get arrays of the Nodes, Edges, and Faces in the Body */
    status = EG_getBodyTopos(*ebody, NULL, NODE, &nnode, &enodes);
    CHECK_STATUS(EG_getBodyTopos);

    status = EG_getBodyTopos(*ebody, NULL, EDGE, &nedge, &eedges);
    CHECK_STATUS(EG_getBodyTopos);

    status = EG_getBodyTopos(*ebody, NULL, FACE, &nface, &efaces);
    CHECK_STATUS(EG_getBodyTopos);

    /* debug print of attributes before processing */
#ifdef DEBUG
    printf("Attributes before processing\n");

    for (inode = 0; inode < nnode; inode++) {
        printf("Node %3d\n", inode+1);
        printAttributes(enodes[inode]);
    }

    for (iedge = 0; iedge < nedge; iedge++) {
        printf("Edge %3d\n", iedge+1);
        printAttributes(eedges[iedge]);
    }

    for (iface = 0; iface < nface; iface++) {
        printf("Face %3d\n", iface+1);
        printAttributes(efaces[iface]);
    }
#endif

    /* if FILENAME contains a real filename, open it now and remember
       if we are at the end of file */
    if (strncmp(FILENAME(0), "<<\n", 3) != 0) {
        fp = fopen(FILENAME(0), "rb");
        if (fp == NULL) {
            snprintf(message, 100, "could not open file \"%s\"", FILENAME(0));
            status = EGADS_NOTFOUND;
            goto cleanup;
        }

        ieof = feof(fp);

    /* otherwise we have an inline file, so set up the next character to read */
    } else {
        inext = 3;
        ieof  = 0;
    }

    /* read until end of file */
    while (ieof == 0) {

        /* read the next line */
        if (fp != NULL) {
            (void) fgets(templine, 255, fp);
            if (feof(fp) > 0) break;

            /* overwite the \n and \r at the end */
            if (STRLEN(templine) > 0 && templine[STRLEN(templine)-1] == '\n') {
                templine[STRLEN(templine)-1] = '\0';
            }
            if (STRLEN(templine) > 0 && templine[STRLEN(templine)-1] == '\r') {
                templine[STRLEN(templine)-1] = '\0';
            }
        } else {
            i = 0;
            for (; inext < STRLEN(FILENAME(0)); inext++) {
                if (FILENAME(0)[inext] == '\n' || FILENAME(0)[inext] == '\r') break;

                templine[i++] = FILENAME(0)[inext];
            }
            templine[i] = '\0';
            inext++;

            if (inext >= STRLEN(FILENAME(0))) {
                ieof = 1;
            }
        }

        printf("processing: %s\n", templine);

        /* remove inline comments */
        for (i = 0; i < strlen(templine); i++) {
            if (templine[i] == '#') {
                templine[i] =  '\0';
                break;
            }
        }

        /* read the fields out of templine */
        attrValue[0] = '\0';

        nread = sscanf(templine, "%s %s %s", command, attrName, attrValue);

        /* whole-line comment or blank line */
        if (nread < 0) {

        /* line with only command (error) */
        } else if (nread < 2) {
            printf("   error encountered when reading: %s\n", templine);
            nskip++;

        /* attributes on all Nodes */
        } else if (strcmp(command, "node") == 0 || strcmp(command, "NODE") == 0) {
            for (inode = 0; inode < nnode; inode++) {
                status = addAttribute(enodes[inode], attrName, attrValue, inode+1);
                CHECK_STATUS(addAttribute);
            }

        /* attributes on all Edges */
        } else if (strcmp(command, "edge") == 0 || strcmp(command, "EDGE") == 0) {
            for (iedge = 0; iedge < nedge; iedge++) {
                status = addAttribute(eedges[iedge], attrName, attrValue, iedge+1);
                CHECK_STATUS(addAttribute);
            }

        /* attributes on all Faces */
        } else if (strcmp(command, "face") == 0 || strcmp(command, "FACE") == 0) {
            for (iface = 0; iface < nface; iface++) {
                status = addAttribute(efaces[iface], attrName, attrValue, iface+1);
                CHECK_STATUS(addAttribute);
            }

        /* illegal command */
        } else {
            printf("   illegal command (%s) when reading: %s\n", command, templine);
            nskip++;
        }
    }

    /* debug print of attributes after processing */
#ifdef DEBUG
    printf("Attributes after processing\n");

    for (inode = 0; inode < nnode; inode++) {
        printf("Node %3d\n", inode+1);
        printAttributes(enodes[inode]);
    }

    for (iedge = 0; iedge < nedge; iedge++) {
        printf("Edge %3d\n", iedge+1);
        printAttributes(eedges[iedge]);
    }

    for (iface = 0; iface < nface; iface++) {
        printf("Face %3d\n", iface+1);
        printAttributes(efaces[iface]);
    }
#endif

    /* add a special Attribute to the Body to tell OpenCSM that there
       is no topological change and hence it should not adjust the
       Attributes on the Body in finishBody() */
    status = EG_attributeAdd(*ebody, "__noTopoChange__", ATTRSTRING,
                             0, NULL, NULL, "udfTrain2");
    CHECK_STATUS(EG_attributeAdd);

    /* set the output value(s) */
    NSKIP(numUdp) = nskip;

    /* remember this model (Body) */
    udps[numUdp].ebody = *ebody;

cleanup:
#ifdef DEBUG
    printf("udpExecute -> numUdp=%d, *ebody=%llx\n", numUdp, (long long)(*ebody));
#endif

    if (fp != NULL) fclose(fp);

    if (enodes != NULL) EG_free(enodes);
    if (eedges != NULL) EG_free(eedges);
    if (efaces != NULL) EG_free(efaces);

    if (strlen(message) > 0) {
        *string = message;
        printf("%s\n", message);
    } else if (status != EGADS_SUCCESS) {
        FREE(message);
        *string = udpErrorStr(status);
    } else {
        FREE(message);
    }

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   udpSensitivity - return sensitivity derivatives for the "real" argument *
 *                                                                      *
 ************************************************************************
 */

int
udpSensitivity(ego    ebody,            /* (in)  Body pointer */
               int    npnt,             /* (in)  number of points */
               int    entType,          /* (in)  OCSM entity type */
               int    entIndex,         /* (in)  OCSM entity index (bias-1) */
               double uvs[],            /* (in)  parametric coordinates for evaluation */
               double vels[])           /* (out) velocities */
{
    int    status = EGADS_SUCCESS;

    int    iudp, judp;

    ROUTINE(udpSensitivity);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    if (uvs != NULL) {
        printf("udpSensitivity(ebody=%llx, npnt=%d, entType=%d, entIndex=%d, uvs=%f %f)\n",
               (long long)ebody, npnt, entType, entIndex, uvs[0], uvs[1]);
    } else {
        printf("udpSensitivity(ebody=%llx, npnt=%d, entType=%d, entIndex=%d, uvs=NULL)\n",
               (long long)ebody, npnt, entType, entIndex);
    }
#endif

    /* check that ebody matches one of the ebodys */
    iudp = 0;
    for (judp = 1; judp <= numUdp; judp++) {
        if (ebody == udps[judp].ebody) {
            iudp = judp;
            break;
        }
    }
    if (iudp <= 0) {
        status = EGADS_NOTMODEL;
        goto cleanup;
    }

    /* the following line should be included if sensitivities
       are not computed analytically */
    status = EGADS_NOLOAD;

cleanup:
#ifdef DEBUG
    printf("udpSensitivity -> vels=%f %f %f\n", vels[0], vels[1], vels[2]);
#endif
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   addAttribute - add the specified attribute to the ego              *
 *                                                                      *
 ************************************************************************
 */

static int
addAttribute(ego    eobj,               /* (in)  ego to get attribute */
             char   attrName[],         /* (in)  attribute name */
             char   attrValue[],        /* (in)  attribute value (may contain @ */
             int    index)              /* (in)  index (such as bias-1 Node number) */
{
    int     status=EGADS_SUCCESS;       /* (out)  return status */

    int     attrType, attrLen;
    CINT    *tempIlist;
    CDOUBLE *tempRlist;
    char    tempValue[256];
    CCHAR   *tempClist;

    ROUTINE(printAttributes);

    /* --------------------------------------------------------------- */

    /* remove the attribute if it exists */
    status = EG_attributeRet(eobj, attrName, &attrType, &attrLen,
                             &tempIlist, &tempRlist, &tempClist);
    if (status == EGADS_SUCCESS) {
        status = EG_attributeDel(eobj, attrName);
        CHECK_STATUS(EG_attributeDel);
    }

    /* if attrValue was a percent-sign, add an integer attribute */
    if (strcmp(attrValue, "@") == 0) {
        status = EG_attributeAdd(eobj, attrName, ATTRINT, 1,
                                 &index, NULL, NULL);
        CHECK_STATUS(EG_attributeAdd);

    /* otherise add a string attribute */
    } else if (STRLEN(attrValue) > 0) {
        status = convertAtSign(attrValue, index, tempValue);
        CHECK_STATUS(convertAtSign);

        status = EG_attributeAdd(eobj, attrName, ATTRSTRING, 0,
                                 NULL, NULL, tempValue);
        CHECK_STATUS(EG_attributeAdd);
    }

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   printAttributes - print attributes associated with an ego          *
 *                                                                      *
 ************************************************************************
 */

#ifdef DEBUG
static int
printAttributes(ego eobj)               /* (in)  ego object */
{
    int     status=EGADS_SUCCESS;       /* (out)  return status */

    int     attrType, attrLen, nattr, iattr, i;
    CINT    *tempIlist;
    CDOUBLE *tempRlist;
    CCHAR   *tempClist, *attrName;

    ROUTINE(printAttributes);

    /* --------------------------------------------------------------- */

    status = EG_attributeNum(eobj, &nattr);
    CHECK_STATUS(EG_attributeNum);

    for (iattr = 1; iattr <= nattr; iattr++) {
        status = EG_attributeGet(eobj, iattr, &attrName, &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        CHECK_STATUS(EG_attributeGet);

        printf("   %-20s ", attrName);
        if (attrType == ATTRINT) {
            for (i = 0; i < attrLen; i++) {
                printf(" %6d", tempIlist[i]);
            }
            printf("\n");
        } else if (attrType == ATTRREAL) {
            for (i = 0; i < attrLen; i++) {
                printf(" %13.7f", tempRlist[i]);
            }
            printf("\n");
        } else if (attrType == ATTRSTRING) {
            printf(" %s\n", tempClist);
        }
    }

cleanup:
    return status;
}
#endif


/*
 ************************************************************************
 *                                                                      *
 *   convertAtSign - convert at-signs to number                         *
 *                                                                      *
 ************************************************************************
 */

static int
convertAtSign(char instring[],          /* (in)  input string */
              int  number,              /* (in)  number to use instead of at-sign */
              char outstring[])         /* (out) output string */
{
    int     status=EGADS_SUCCESS;       /* (out)  return status */

    int     i, j, natsign=0;
    char    format[512];

    ROUTINE(convertAtSign);

    /* --------------------------------------------------------------- */

    /* build a format string that replaces @ with %d */
    j = 0;
    for (i = 0; i < STRLEN(instring); i++) {
        if (instring[i] != '@') {
            format[j++] = instring[i];
        } else if(natsign >= 3) {
            format[j++] = instring[i];
        } else {
            natsign++;
            format[j++] = '%';
            format[j++] = 'd';
        }
    }
    format[j++] = '\0';

    /* if there are no at-signs, we can do a simple copy */
    if (natsign == 0) {
        strncpy(outstring, instring, 256);

    /* otherwise print to the outstring */
    } else {
        snprintf(outstring, 256, format, number, number, number);
    }

//cleanup:
    return status;
}
