/*
 *      CAPS: Computational Aircraft Prototype Syntheses
 *
 *             FRICTION AIM
 *
 *      Copyright 2014-2025, Massachusetts Institute of Technology
 *      Licensed under The GNU Lesser General Public License, version 2.1
 *      See http://www.opensource.org/licenses/lgpl-2.1.php
 *
 */

/*
 * Resources or possible additional form factors
 * https://www.sciencedirect.com/topics/engineering/form-drag
 */

#include <string.h>
#include <math.h>

#include "aimUtil.h"
#include "miscUtils.h"
#include "jsonUtils.h"
#include "vlmUtils.h"

#ifdef WIN32
#define strcasecmp stricmp
#define strncasecmp _strnicmp
#define strtok_r   strtok_s
#endif

#define PI        3.1415926535897931159979635
#define NINT(A)         (((A) < 0)   ? (int)(A-0.5) : (int)(A+0.5))
#define MAX(A,B)        (((A) < (B)) ? (B) : (A))

enum aimInputs
{
  inMach = 1,                    /* index is 1-based */
  inAltitude,
  inBL_Transition,
  inTwTaw,
  inLiftSurface,
  inRevolveSurface,
  NUMINPUT = inRevolveSurface    /* Total number of inputs */
};

enum aimOutputs
{
  outCDtotal = 1,              /* index is 1-based */
  outCDform,
  outCDfric,
  outSwet,
  NUMOUTCASE = outSwet             /* Total number of outputs */
};

//#define DEBUG


/*!\mainpage Introduction
 *
 * \tableofcontents
 *
 * \section overviewFRICTION FRICTION AIM Overview
 * FRICTION provides an estimate of laminar and turbulent skin friction and form
 * drag suitable for use in aircraft preliminary design \cite Friction. Taken from the
 * FRICTION manual:
 * "The program has its roots in a program by Ron Hendrickson at Grumman. It runs on any computer. The input
 * requires geometric information and either the Mach and altitude combination, or
 * the Mach and Reynolds number at which the results are desired. It uses standard
 * flat plate skin friction formulas. The compressibility effects on skin friction
 * are found using the Eckert Reference Temperature method for laminar flow and the
 * van Driest II formula for turbulent flow. The basic formulas are valid from
 * subsonic to hypersonic speeds, but the implementation makes assumptions that limit
 * the validity to moderate supersonic speeds (about Mach 3). The key assumption is
 * that the vehicle surface is at the adiabatic wall temperature (the user can easily
 * modify this assumption). Form factors are used to estimate the effect of thickness
 * on drag, and a composite formula is used to include the effect of a partial run of
 * laminar flow."
 *
 * An outline of the AIM's inputs, outputs and attributes are provided in \ref aimInputsFRICTION,
 * \ref aimOutputsFRICTION, and \ref attributeFRICTION, respectively.
 *
 * The AIM requires dimensional units which are described in \ref aimUnitsFRICTION.
 *
 * Friction drag is estimated based on laminar (White, Viscous Fluid Flow, 1974 ed., pg 589-590.) and
 * turbulent (an Driest II Method, NASA TN D-6945) flat plate skin friction. The reference length
 * for lifting surfaces is estimated as the average segment chord length between two airfoil sections, and the total
 * drag is an wetted area weighted sum of the segments. For bodies of revolution, the references length is the
 * total length of body (nose-to-tail). The form factor for lifting surfaces is based on the average thickness to chord
 * ratio for each segment, and the maximum diameter (estimated as \f$d = 2*\sqrt(A_{sec}/\pi)\f$) to length ratio.
 *
 *
 *
 * \section frictionModification FRICTION Modifications
 * While FRICTION is available from,
 * <a href="http://www.dept.aoe.vt.edu/~mason/Mason_f/MRsoft.html">FRICTION download</a>,
 * the AIM has re-implemented the calculations in memory. This has enabled the additional inputs
 * not available in the original Fortran software.
 *
 * The AIM previously used FRICTION source is still provided as part of CAPS, and contains a few modifications
 * from the original source code. These modificiations allows for longer input
 * and output file name lengths, as well as other I/O modifications. This modified version of
 * FRICTION, friction_eja_mod.f, is supplied and built with the AIM. During the compilation the source code is
 * compiled into an executable with the name \a friction (Linux and OSX) or \a friction.exe (Windows).
 *
 * \section exampleFRICTION Examples
 * An example problem using the FRICTION AIM may be found at
 * \ref frictionExample.
 *
 */

/*! \page attributeFRICTION AIM Attributes
 * The following list of attributes drives the FRICTION geometric definition. Aircraft components are defined as cross sections
 * in the low fidelity geometry definition. To be able to logically group the cross sections into wings, tails, fuselage, etc
 * they must be given a grouping attribute. This attribute defines a logical group along with identifying a set of cross sections
 * as a lifting surface or a body of revolution. The format is as follows.
 *
 *  - <b> capsGroup</b> This string attribute labels the <em> FaceBody</em> as to which type the section
 *  is assigned. This information is also used to logically group sections together by type to create wings, tails, stores, etc.
 *
 *  - <b> capsReferenceArea</b>  [Optional: Default 1.0] This attribute may exist on any <em> Body</em>.  Its
 * value will be used as the SREF entry in the FRICTION input.
 *
 *  - <b> capsLength</b> This attribute defines the length units that the *.csm file is generated in.  Friction input MUST be in
 *  units of feet.  The AIM handles all unit conversion internally based on this input.
 *
 */


typedef struct {
    double arcLength;
    double thickOverChord;
} fcnSectionStruct;

typedef struct {
    double Swet;
    double refLength;
    double thickOverChord;
    double formFactor;
} fcnSpanStruct;

typedef struct {
    double Swet;
    double refLength;
    double thickOverChord; // thickness to chord ratio
    double formFactor;

    double turbTrans; // 0.0 fully turbulent (default) 1.0 laminar
    double TwTaw;  // Tw/Taw, the wall temperature ratio

    int numSection;
    fcnSectionStruct *fcnSection;

    int numSpan;
    fcnSpanStruct *fcnSpan;

    int numCase;
    double* ReL;     // Reynolds number
    double* CDtotal; // Total (Form+Friction) Drag
    double* CDform;  // Form Drag
    double* CDfric;  // Friction Drag
} fcnSurfaceStruct;


typedef struct {

  char *length;
  char *area;

  int numLiftSurface;
  vlmSurfaceStruct *vlmLiftSurface;
  fcnSurfaceStruct *fcnLiftSurface;

  int numRevSurface;
  vlmSurfaceStruct *vlmRevSurface;
  fcnSurfaceStruct *fcnRevSurface;

  capsValue CDtotal; // Total (Form+Friction) Drag
  capsValue CDform;  // Form Drag
  capsValue CDfric;  // Friction Drag

} aimStorage;


/* ****************** FRICTION AIM Helper Functions ************************ */
static void
initiate_fcnSectionStruct(fcnSectionStruct *section) {

  section->arcLength      = 0.0;
  section->thickOverChord = 0.0;
}

static void
destroy_fcnSectionStruct(fcnSectionStruct *section) {

  section->arcLength      = 0.0;
  section->thickOverChord = 0.0;
}

static void
initiate_fcnSpanStruct(fcnSpanStruct *span) {

  span->Swet           = 0.0;
  span->refLength      = 0.0;
  span->thickOverChord = 0.0;
  span->formFactor     = 1.0;
}

static void
destroy_fcnSpanStruct(fcnSpanStruct *span) {

  span->Swet           = 0.0;
  span->refLength      = 0.0;
  span->thickOverChord = 0.0;
  span->formFactor     = 1.0;
}

static void
initiate_fcnSurfaceStruct(fcnSurfaceStruct *surface) {

  surface->Swet = 0.0;
  surface->refLength = 0.0;
  surface->thickOverChord = 0.0; // thickness to chord ratio
  surface->formFactor = 1.0;
  surface->turbTrans = 0; // 0.0 fully turbulent (default) 1.0 laminar
  surface->TwTaw = 1.0;

  surface->numSection = 0;
  surface->fcnSection = NULL;

  surface->numSpan = 0;
  surface->fcnSpan = NULL;

  surface->numCase = 0;
  surface->ReL     = NULL;
  surface->CDtotal = NULL;
  surface->CDform  = NULL;
  surface->CDfric  = NULL;
}

static void
destroy_fcnSurfaceStruct(fcnSurfaceStruct *surface) {

  int i;
  surface->Swet = 0.0;
  surface->refLength = 0.0;
  surface->thickOverChord = 0.0; // thickness to chord ratio
  surface->formFactor = 1.0;
  surface->turbTrans = 0; // 0.0 fully turbulent (default) 1.0 laminar
  surface->TwTaw = 1.0;

  for (i = 0; i < surface->numSection; i++)
    destroy_fcnSectionStruct(&surface->fcnSection[i]);

  AIM_FREE(surface->fcnSection);
  surface->numSection = 0;

  for (i = 0; i < surface->numSpan; i++)
    destroy_fcnSpanStruct(&surface->fcnSpan[i]);

  AIM_FREE(surface->fcnSpan);
  surface->numSpan = 0;

  surface->numCase = 0;
  AIM_FREE(surface->ReL);
  AIM_FREE(surface->CDtotal);
  AIM_FREE(surface->CDform);
  AIM_FREE(surface->CDfric);
}

static void
initiate_aimStorage(aimStorage *frictionInstance) {

  frictionInstance->length = NULL;
  frictionInstance->area   = NULL;

  frictionInstance->numLiftSurface = 0;
  frictionInstance->vlmLiftSurface = NULL;
  frictionInstance->fcnLiftSurface = NULL;

  frictionInstance->numRevSurface = 0;
  frictionInstance->vlmRevSurface = NULL;
  frictionInstance->fcnRevSurface = NULL;

  aim_initValue(&frictionInstance->CDtotal);
  aim_initValue(&frictionInstance->CDform);
  aim_initValue(&frictionInstance->CDfric);
}

static void
destroy_aimStorage(aimStorage *frictionInstance) {
  int i;

  AIM_FREE(frictionInstance->length);
  AIM_FREE(frictionInstance->area);

  for (i = 0; i < frictionInstance->numLiftSurface; i++) {
    destroy_vlmSurfaceStruct(&frictionInstance->vlmLiftSurface[i]);
    destroy_fcnSurfaceStruct(&frictionInstance->fcnLiftSurface[i]);
  }
  AIM_FREE(frictionInstance->vlmLiftSurface);
  AIM_FREE(frictionInstance->fcnLiftSurface);
  frictionInstance->numLiftSurface = 0;

  for (i = 0; i < frictionInstance->numRevSurface; i++) {
    destroy_vlmSurfaceStruct(&frictionInstance->vlmRevSurface[i]);
    destroy_fcnSurfaceStruct(&frictionInstance->fcnRevSurface[i]);
  }
  AIM_FREE(frictionInstance->vlmRevSurface);
  AIM_FREE(frictionInstance->fcnRevSurface);
  frictionInstance->numRevSurface = 0;

  aim_freeValue(&frictionInstance->CDtotal);
  aim_freeValue(&frictionInstance->CDform);
  aim_freeValue(&frictionInstance->CDfric);
}


// ********************** AIM Function Break *****************************
static void
calculate_distance(double *x1, double *x2, double *x0, double* D)
{
  // (x1)-------------(x2)
  //             |
  //             |D
  //             |
  //             (x0)

  double a[3], b[3], tmp[3], length;

  a[0] = x2[0] - x1[0];
  b[0] = x1[0] - x0[0];
  a[1] = x2[1] - x1[1];
  b[1] = x1[1] - x0[1];
  a[2] = x2[2] - x1[2];
  b[2] = x1[2] - x0[2];

  length = sqrt(a[0]*a[0] + a[1]*a[1] + a[2]*a[2]);
  if (length < 1.0e-8) {
    *D = sqrt(b[0]*b[0] + b[1]*b[1] + b[2]*b[2]);
    return;
  }

  cross_DoubleVal(a,b,tmp);
  *D = sqrt(tmp[0]*tmp[0] + tmp[1]*tmp[1] + tmp[2]*tmp[2]) / length;
}

// ********************** AIM Function Break *****************************
// Taken from Friction source
static int
stdatm(void *aimInfo, int KD, double Z,
       double *T, double *P, double *R, double *A,
       double *MU, double *TS, double *RR, double *PP,
       double *RM, double *QM)
{
/*
c   *********** 1976 STANDARD ATMOSPHERE SUBROUTINE **********
c
c     Mason's BASIC program, converted to FORTRAN - Sept. 1, 1989
c
c     kd -   = 0 - metric units
c           <> 0 - English units
c
c     z  - input altitude, in feet or meters (depending on kd)
c
c     output:
c
c     t  - temp.
c     p  - pressure
c     r  - density
c     a  - speed of sound
c     mu - viscosity
c
c     ts - t/t at sea level
c     rr - rho/rho at sea level
c     pp - p/p at sea level
c
c     rm - Reynolds number per Mach per unit of length
c     qm - dynamic pressure/Mach^2
*/
  int status = CAPS_SUCCESS;
  double TL, PL, RL, C1, AL, BT;
  double K, H;

  K  = 34.163195;
  C1 = 3.048E-04;
  if (KD == 0)  {
    TL = 288.15;
    PL = 101325;
    RL = 1.225;
    C1 = .001;
    AL = 340.294;
    BT = 1.458E-06;
  } else {
    TL = 518.67;
    PL = 2116.22;
    RL = .0023769;
    AL = 1116.45;
    BT = 3.0450963E-08;
  }

  H = C1 * Z / (1 + C1 * Z / 6356.766);
  if (H <= 11.0) {
    *T  = 288.15 - 6.5 * H;
    *PP = pow(288.15 / *T, - K / 6.5);
  } else if (H <= 20.0) {
    *T  = 216.65;
    *PP = .22336 *  exp( - K * (H - 11) / 216.65);
  } else if (H <= 32.0) {
    *T  = 216.65 + (H - 20);
    *PP = .054032 * pow(216.65 / (*T), K);
  } else if (H <= 47.0) {
    *T  = 228.65 + 2.8 * (H - 32);
    *PP = .0085666 * pow(228.65 / (*T), K / 2.8);
  } else if ( H <= 51.0) {
    *T  = 270.65;
    *PP = .0010945 *  exp ( - K * (H - 47) / 270.65);
  } else if (H <= 71.) {
    *T  = 270.65 - 2.8 * (H - 51);
    *PP = .00066063 * pow(270.65 / (*T), - K / 2.8);
  } else if (H <= 84.852) {
    *T  = 214.65 - 2 * (H - 71);
    *PP = 3.9046E-05 * pow(214.65 / (*T), - K / 2);
  } else {
    AIM_ERROR(aimInfo, " Out of Table in StdAtm. Too high H = %f12.3 > 84.852 km", H);
    status = CAPS_NOTIMPLEMENT;
    goto cleanup;
  }

  *RR = *PP / (*T / 288.15);
  *MU = BT * pow(*T, 1.5) / (*T + 110.4);
  *TS = *T / 288.15;
  *A  = AL *  sqrt(*TS);
  *T  = TL * (*TS);
  *R  = RL * (*RR);
  *P  = PL * (*PP);
  *RM = (*R) * (*A) / (*MU);
  *QM = .7 * (*P);

  status = CAPS_SUCCESS;
cleanup:
  return status;
}

// ********************** AIM Function Break *****************************
static int
lamcf(double Rex, double Xme, double TwTaw, double *CF)
{
/*
c     flat plate laminar skin friction routine
c
c     uses the Eckert Refereence Temperatrure Method
c     as described in White, Viscous Fluid Flow, 1974 ed., pg 589-590.
c
c     see Boundary Layer Analysis Methods in Aerocal Methods
c         Pak #4, pages 4-39 to 4-41
c
c     coded in FORTRAN, Feb. 24, 1992
c     by W.H. Mason
c
c     Input:
c                  Rex   - Reynolds number
c                  Xme   - Mach Number
c                  TwTaw - Tw/Taw, the wall temperature ratio
c
c     values of gamma, Pr, and Te are set in the program
c
c     Output:
c                CF    - total skin friction coefficient
*/
  double g, Pr, R, Te, Tk, TwTe, TstTe, Cstar;

  if (CF == NULL) return CAPS_NULLVALUE;

  g     = 1.4;
  Pr    = .72;
  R     = sqrt(Pr);
  Te    = 390.;
  Tk    = 200.;

  TwTe  = TwTaw*(1. + R*(g - 1.)/2.*pow(Xme,2));
  TstTe = 0.5 + 0.039*pow(Xme,2) + 0.5*TwTe;

  Cstar = sqrt(TstTe)*(1. + Tk/Te)/(TstTe + Tk/Te);

  (*CF)  = 2.*.664*sqrt(Cstar)/sqrt(Rex);

  return CAPS_SUCCESS;

}

// ********************** AIM Function Break *****************************
static int
turbcf(void *aimInfo, double Rex, double xme, double TwTaw, double *CF)
{
/*
c     flat plate turbulent skin friction routine
c
c     uses the Van Driest II Method, NASA TN D-6945
c
c     coded in FORTRAN, October 21, 1989, by W.H. Mason
c
c     Input:       Rex   - Reynolds number (based on length)
c                  Xme   - Mach Number
c                  TwTaw - Tw/Taw, the wall temperature ratio
c
c     values of gamma, recovery factor and Te are set in the program
c
c     Output:      CF    - total skin friction coefficient
*/
  double epsmax, eps;
  double G, r, Te, xm, TawTe;
  double F, Tw, A, B, denom, Alpha, Beta, Fc;
  double xnum, Ftheta, Fx, RexBar, Cfb, Cfo;
  int iter;

  epsmax = 0.2e-8;
  G      = 1.4;
  r      = 0.88;
  Te     = 222.0;

  xm    = (G - 1.)/2*pow(xme,2);
  TawTe = 1. + r*xm;
  F     = TwTaw*TawTe;
  Tw    = F * Te;
  A     = sqrt(r*xm/F);
  B     = (1. + r*xm - F)/F;
  denom = sqrt(4.*pow(A,2) + pow(B,2));
  Alpha = (2.*pow(A,2) - B)/denom;
  Beta  = B/denom;
  Fc    = pow((1.0 + sqrt(F))/2.0, 2);
  if ( xme > 0.1) Fc = r*xm/pow(asin(Alpha) + asin(Beta), 2);

  xnum   = (1. + 122./Tw*pow(10,-5/Tw));
  denom  = (1. + 122./Te*pow(10,-5/Te));
  Ftheta = sqrt(1./F)*(xnum/denom);
  Fx     = Ftheta/Fc;

  RexBar = Fx * Rex;

  Cfb    = 0.074/pow(RexBar, 0.20);

  iter   = 0;
  do {
    Cfo    = Cfb;
    xnum   = 0.242 - sqrt(Cfb)*log10(RexBar*Cfb);
    denom  = 0.121 + sqrt(Cfb)/log(10.);
    Cfb    = Cfb*(1.0 + xnum/denom);
    eps    = fabs(Cfb - Cfo);
    iter++;
    if (iter > 200) {
      AIM_ERROR(aimInfo, "did not converge in turbcf: eps = %15.7e, epsmax = %15.7e", eps, epsmax);
      return CAPS_EXECERR;
    }
  } while (eps > epsmax);

  *CF = Cfb/Fc;

  return CAPS_SUCCESS;
}


// ********************** AIM Function Break *****************************
static int
get_fcnSurfaceData(void *aimInfo,
                   const aimStorage *frictionInstance,
                   int revolution,
                   const char *lengthUnitsIn,
                   const char *areaUnitsIn,
                   int numSurface,
                   /*@null@*/vlmSurfaceStruct *vlmSurface,
                   /*@null@*/fcnSurfaceStruct *fcnSurface)
{
   //Compute arcLength and ToC for each section

  int    i, isurf, isec, status, nEdge;
  int    sectionIndex1, sectionIndex2;
  double dist, massData[14];
  double Swet, refLength, ToC, maxDiam = 0;
  ego    *edges=NULL;


  for (isurf = 0; isurf < numSurface; isurf++) {
    AIM_NOTNULL(vlmSurface, aimInfo, status);
    AIM_NOTNULL(fcnSurface, aimInfo, status);

    AIM_ALLOC(fcnSurface[isurf].fcnSection, vlmSurface[isurf].numSection, fcnSectionStruct, aimInfo, status);
    for (isec = 0; isec < vlmSurface[isurf].numSection; isec++)
      initiate_fcnSectionStruct(&fcnSurface[isurf].fcnSection[isec]);
    fcnSurface[isurf].numSection = vlmSurface[isurf].numSection;

    maxDiam = 0;
    for (isec = 0; isec < vlmSurface[isurf].numSection; isec++) {

      if (vlmSurface[isurf].vlmSection[isec].type == vlmSecNode) continue;

      if (revolution == (int)false) {
        // Get ToC
        status = vlm_getSectionToC(aimInfo, &vlmSurface[isurf].vlmSection[isec], &fcnSurface[isurf].fcnSection[isec].thickOverChord);
        AIM_STATUS(aimInfo, status);
      }

      // determine the arc length around the body (excluding the trailing edge)
      status = EG_getBodyTopos(vlmSurface[isurf].vlmSection[isec].ebody, NULL, EDGE, &nEdge, &edges);
      AIM_STATUS(aimInfo, status);
      AIM_NOTNULL(edges, aimInfo, status);

      for (i = 0; i < nEdge; i++) {
        if (edges[i] == vlmSurface[isurf].vlmSection[isec].teObj) continue;

        status = EG_getMassProperties(edges[i], massData);
        AIM_STATUS(aimInfo, status);

        // massData array population
        // volume, surface area (length), cg(3), inertia(9)
        fcnSurface[isurf].fcnSection[isec].arcLength += massData[1];
      }

      AIM_FREE(edges);

      if (revolution == (int)true) {
        // volume, surface area (length), cg(3), iniria(9)
        status = EG_getMassProperties(vlmSurface[isurf].vlmSection[isec].ebody, massData);
        AIM_STATUS(aimInfo, status);

        // ~diameter of cross section for the "chord" length
        vlmSurface[isurf].vlmSection[isec].chord = 2.0 * sqrt(massData[1] / PI);

        maxDiam = MAX(maxDiam, vlmSurface[isurf].vlmSection[isec].chord);
      }
    }

    // Set span properties between sections
    AIM_ALLOC(fcnSurface[isurf].fcnSpan, vlmSurface[isurf].numSection-1, fcnSpanStruct, aimInfo, status);
    for (i = 0; i < vlmSurface[isurf].numSection-1; i++)
      initiate_fcnSpanStruct(&fcnSurface[isurf].fcnSpan[i]);
    fcnSurface[isurf].numSpan = vlmSurface[isurf].numSection-1;

    for (i = 0; i < vlmSurface[isurf].numSection-1; i++) {

      // get the sorted section indices
      sectionIndex1 = vlmSurface[isurf].vlmSection[i  ].sectionIndex;
      sectionIndex2 = vlmSurface[isurf].vlmSection[i+1].sectionIndex;

      // compute the distance between sections
      calculate_distance(vlmSurface[isurf].vlmSection[sectionIndex1].xyzLE, vlmSurface[isurf].vlmSection[sectionIndex1].xyzTE,
                         vlmSurface[isurf].vlmSection[sectionIndex2].xyzLE, &dist);

      // compute span properties
      if (revolution == (int)true) {
        refLength = dist;
      } else {
        refLength = (vlmSurface[isurf].vlmSection[sectionIndex1].chord + vlmSurface[isurf].vlmSection[sectionIndex2].chord) / 2.0;
      }

      Swet = (dist * (fcnSurface[isurf].fcnSection[sectionIndex1].arcLength      + fcnSurface[isurf].fcnSection[sectionIndex2].arcLength)     ) / 2.0;
      ToC  = (       (fcnSurface[isurf].fcnSection[sectionIndex1].thickOverChord + fcnSurface[isurf].fcnSection[sectionIndex2].thickOverChord)) / 2.0;

      status = aim_convert(aimInfo, 1, lengthUnitsIn, &refLength, frictionInstance->length, &refLength);
      AIM_STATUS(aimInfo, status);

      status = aim_convert(aimInfo, 1, areaUnitsIn, &Swet, frictionInstance->area, &Swet);
      AIM_STATUS(aimInfo, status);

      fcnSurface[isurf].fcnSpan[i].Swet           = Swet;
      fcnSurface[isurf].fcnSpan[i].refLength      = refLength;
      fcnSurface[isurf].fcnSpan[i].thickOverChord = ToC;

      if (revolution == (int)false) {

        // Torenbeek [54, p. 499] FF for lifting surfaces airfoils t/c <= 0.21
        fcnSurface[isurf].fcnSpan[i].formFactor = 1.0 + 2.7*ToC + 100.*pow(ToC,4);

      }

      fcnSurface[isurf].Swet += Swet;

      if (revolution == (int)true) {
        // Sum for body of revolution
        fcnSurface[isurf].refLength      += refLength;
        fcnSurface[isurf].thickOverChord += ToC;
      } else {
        // Area average for lifting surface
        fcnSurface[isurf].refLength      += refLength*Swet;
        fcnSurface[isurf].thickOverChord += ToC*Swet;
      }
    }

    if (revolution == (int)true) {

      status = aim_convert(aimInfo, 1, lengthUnitsIn, &maxDiam, frictionInstance->length, &maxDiam);
      AIM_STATUS(aimInfo, status);

      // ToC based on maxChord and total refLength
      fcnSurface[isurf].thickOverChord = maxDiam / fcnSurface[isurf].refLength;

      ToC = fcnSurface[isurf].thickOverChord;

      // S.F. Hoerner, Fluid-Dynamic Drag, L. Hoerner, 1965
      // Hoerner [33, pp. 6–17] streamlined bodies Re > 10e5
      fcnSurface[isurf].formFactor = 1.0 + 1.5*pow(ToC,1.5) + 7.*pow(ToC,3);
    } else {

      // Area averaged length and ToC for lifting surfaces
      fcnSurface[isurf].refLength      /= fcnSurface[isurf].Swet;
      fcnSurface[isurf].thickOverChord /= fcnSurface[isurf].Swet;

    }
  }

  status = CAPS_SUCCESS;

cleanup:

  AIM_FREE(edges);

  return status;
}


static
int get_fcnSurface(void *aimInfo,
                   double BL_Transition,
                   double TwTaw,
                   int numTuple,
                   capsTuple surfaceTuple[],
                   mapAttrToIndexStruct *groupMap,
                   int *numSurface,
                   vlmSurfaceStruct *vlmSurface[],
                   fcnSurfaceStruct *fcnSurface[])
{
  /*! \page fcnSurface Friction Surface
   * Structure for the Friction Surface tuple  = ("Name of Surface", "Value").
   * "Name of surface defines the name of the surface in which the data should be applied.
   *  The "Value" must be be a JSON String dictionary.
   */

  int status; //Function return

  int i, groupIndex; // Indexing

  char *keyValue = NULL; // Key values from tuple searches
  const char *keyWord = NULL; // Key words to find in the tuples

  int numGroupName = 0;
  char **groupName = NULL;

  int numString = 0;
  char **stringArray = NULL; // Freeable

  int attrIndex;

  if (numTuple <= 0){
    return CAPS_SUCCESS;
  }

  printf("Getting friction surface data\n");

  AIM_ALLOC(*vlmSurface, numTuple, vlmSurfaceStruct, aimInfo, status);
  (*numSurface) = numTuple;

  // Initiate vlmSurfaces
  for (i = 0; i < (*numSurface); i++) {
    status = initiate_vlmSurfaceStruct(&(*vlmSurface)[i]);
    AIM_STATUS(aimInfo, status);
  }

  AIM_ALLOC(*fcnSurface, numTuple, fcnSurfaceStruct, aimInfo, status);

  // Initiate fcnSurface
  for (i = 0; i < (*numSurface); i++) {
    initiate_fcnSurfaceStruct(&(*fcnSurface)[i]);

    (*fcnSurface)[i].turbTrans = BL_Transition;
    (*fcnSurface)[i].TwTaw = TwTaw;
  }

  *numSurface = numTuple;

  for (i = 0; i < numTuple; i++) {

    printf("\tFriction surface name - %s\n", surfaceTuple[i].name);
    AIM_STRDUP((*vlmSurface)[i].name, surfaceTuple[i].name, aimInfo, status);

    // Do we have a json string?
    if (strncmp(surfaceTuple[i].value, "{", 1) == 0) {

      //printf("JSON String - %s\n",surfaceTuple[i].value);

      /*! \page fcnSurface
       * \section jsonStringSurface JSON String Dictionary
       *
       * If "Value" is a JSON string dictionary (eg. "Value" = {"BL_Transition": 0.1})
       * the following keywords ( = default values) may be used:
       *
       * <ul>
       *  <li> <B>groupName = "(no default)"</B> </li> <br>
       *  Single or list of <em>capsType</em> names used to define the surface (e.g. "Name1" or ["Name1","Name2",...].
       *  If no groupName variable is provided an attempted will be made to use the tuple name instead;
       * </ul>
       *
       */

      // Get surface variables
      keyWord = "groupName";
      status = search_jsonDictionary( surfaceTuple[i].value, keyWord, &keyValue);
      if (status == CAPS_SUCCESS) {

        status = string_toStringDynamicArray(keyValue, &numGroupName, &groupName);
        AIM_STATUS(aimInfo, status);
        AIM_NOTNULL(groupName, aimInfo, status);

        AIM_FREE(keyValue);

        // Determine how many capsTypes go into making the surface
        for (groupIndex = 0; groupIndex < numGroupName; groupIndex++) {

          status = get_mapAttrToIndexIndex(groupMap, groupName[groupIndex], &attrIndex);
          if (status == CAPS_NOTFOUND) {
            AIM_ERROR(aimInfo, "Surface '%s' groupName '%s' not found in '%s' attributes!", surfaceTuple[i].name, groupMap->mapName);
            print_mapAttrError(aimInfo, groupMap);
            goto cleanup;
          }
          AIM_STATUS(aimInfo, status);

          AIM_REALL((*vlmSurface)[i].attrIndex, (*vlmSurface)[i].numAttr+1, int, aimInfo, status);
          (*vlmSurface)[i].attrIndex[(*vlmSurface)[i].numAttr] = attrIndex;
          (*vlmSurface)[i].numAttr += 1;
        }

        status = string_freeArray(numGroupName, &groupName);
        AIM_STATUS(aimInfo, status);
        groupName = NULL;
        numGroupName = 0;

      } else {
        printf("\tNo \"groupName\" variable provided or no matches found, going to use tuple name\n");
      }

      if ((*vlmSurface)[i].numAttr == 0) {
        status = get_mapAttrToIndexIndex(groupMap, (*vlmSurface)[i].name, &attrIndex);
        if (status == CAPS_NOTFOUND) {
          AIM_ERROR(aimInfo, "VLM surface '%s' not found in %s attributes!", (*vlmSurface)[i].name, groupMap->mapName);
          print_mapAttrError(aimInfo, groupMap);
          goto cleanup;
        }

        AIM_REALL((*vlmSurface)[i].attrIndex, (*vlmSurface)[i].numAttr+1, int, aimInfo, status);
        (*vlmSurface)[i].attrIndex[(*vlmSurface)[i].numAttr] = attrIndex;
        (*vlmSurface)[i].numAttr += 1;
      }

      AIM_FREE(keyValue);

      /*! \page fcnSurface
       *
       * <ul>
       * <li> <B>BL_Transition = NULL</B> </li> <br>
       * Specify transition location in range [0-1] for the surface. Defaults to global BL_Transition input.
       * </ul>
       * \endif
       */
      keyWord = "BL_Transition";
      status = search_jsonDictionary( surfaceTuple[i].value, keyWord, &keyValue);
      if (status == CAPS_SUCCESS) {
        status = string_toDouble(keyValue, &(*fcnSurface)[i].turbTrans);
        AIM_STATUS(aimInfo, status);
        AIM_FREE(keyValue);
      }

      /*! \page fcnSurface
       *
       * <ul>
       * <li> <B>TwTaw = 1.0</B> </li> <br>
       * Tw/Taw, the wall temperature ratio.
       * </ul>
       * \endif
       */
      keyWord = "TwTaw";
      status = search_jsonDictionary( surfaceTuple[i].value, keyWord, &keyValue);
      if (status == CAPS_SUCCESS) {
        status = string_toDouble(keyValue, &(*fcnSurface)[i].TwTaw);
        AIM_STATUS(aimInfo, status);
        AIM_FREE(keyValue);
      }

    } else {

      /*! \page fcnSurface
       * \section keyStringVLMSurface Single Value String
       *
       * If "Value" is a single string the following options maybe used:
       * - (NONE Currently)
       *
       */
      AIM_ERROR(aimInfo, "Surface tuple value must be a JSON string");
      status = CAPS_BADVALUE;
      goto cleanup;
    }
  }

  printf("\tDone getting surface data\n");

  status = CAPS_SUCCESS;

cleanup:

  AIM_FREE(keyValue);

  if (numGroupName != 0 && groupName != NULL){
    (void) string_freeArray(numGroupName, &groupName);
  }

  (void) string_freeArray(numString, &stringArray);

  return status;
}



/* ********************** Exposed AIM Functions ***************************** */


// ********************** AIM Function Break *****************************
int aimInitialize(int inst, /*@unused@*/ const char *unitSys, /*@unused@*/ void *aimInfo,
                  /*@unused@*/ void **instStore, /*@unused@*/ int *major,
                  /*@unused@*/ int *minor, int *nIn, int *nOut,
                  int *nFields, char ***fnames, int **franks, int **fInOut)
{
  int status = CAPS_SUCCESS;
  aimStorage *frictionInstance=NULL;

#ifdef DEBUG
      printf("\n frictionAIM/aimInitialize   instance = %d!\n", inst);
#endif

  // specify the number of analysis input and out "parameters"
  *nIn     = NUMINPUT;    // Mach, Altitude
  *nOut    = NUMOUTCASE;      // CDtotal, CDform, CDfric
  if (inst == -1) return CAPS_SUCCESS;

  /* specify the field variables this analysis can generate and consume */
  *nFields = 0;
  *fnames  = NULL;
  *franks  = NULL;
  *fInOut  = NULL;

  // Allocate friction Instance
  AIM_ALLOC(frictionInstance, 1, aimStorage, aimInfo, status);
  *instStore = frictionInstance;

  // Initiate friction storage
  initiate_aimStorage(frictionInstance);


  /*! \page aimUnitsFRICTION AIM Units
   *  A unit system may be specified as "SI" or "US".
   *  For "SI", the internal calculations will be in meters, for "US" in feet.
   *  The default is "US".
   */
  if (unitSys != NULL) {

    if (strcasecmp(unitSys, "SI") == 0) {
      AIM_STRDUP(frictionInstance->length, "m"  , aimInfo, status);
      AIM_STRDUP(frictionInstance->area  , "m^2", aimInfo, status);
    } else if (strcasecmp(unitSys, "US") == 0) {
      AIM_STRDUP(frictionInstance->length, "ft"  , aimInfo, status);
      AIM_STRDUP(frictionInstance->area  , "ft^2", aimInfo, status);
    } else {
      AIM_ERROR(aimInfo, "unitSys ('%s') is expected to be 'SI' or 'US'", unitSys);
      status = CAPS_BADVALUE;
      goto cleanup;
    }
  } else {
    /* default for units specified */
    AIM_STRDUP(frictionInstance->length, "ft"  , aimInfo, status);
    AIM_STRDUP(frictionInstance->area  , "ft^2", aimInfo, status);
  }

  status = CAPS_SUCCESS;
cleanup:
  return status;
}


// ********************** AIM Function Break *****************************
int aimInputs(/*@unused@*/ void *instStore, /*@unused@*/ void *aimInfo,
              int index, char **ainame, capsValue *defval)
{
  int status = CAPS_SUCCESS;
  /*! \page aimInputsFRICTION AIM Inputs
   * The following list outlines the FRICTION inputs along with their default values available
   * through the AIM interface. All inputs to the FRICTION AIM are variable length arrays.
   * <B> All inputs must be the same length </B>.
   */

  aimStorage *frictionInstance;
  const char *length = NULL;

#ifdef DEBUG
  printf(" frictionAIM/aimInputs  index = %d!\n", index);
#endif

  frictionInstance = (aimStorage *) instStore;

  if (frictionInstance != NULL) length = frictionInstance->length;

  if (index == inMach) {
    *ainame               = EG_strdup("Mach");
    defval->type          = Double;
    defval->lfixed        = Change;
    defval->sfixed        = Change;
    defval->nullVal       = IsNull;
    defval->dim           = Vector;

    /*! \page aimInputsFRICTION
     * - <B> Mach = double </B> <br> OR
     * - <B> Mach = [double, ... , double] </B> <br>
     *  Mach number.<br>
     *  Must have the same number of entries as Altitude.
     */

  } else if (index == inAltitude) {
    *ainame               = EG_strdup("Altitude");
    defval->type          = Double;
    defval->lfixed        = Change;
    defval->sfixed        = Change;
    defval->nullVal       = IsNull;
    defval->dim           = Vector;
    if (length != NULL) {
      AIM_STRDUP(defval->units, length, aimInfo, status);
    }

    /*! \page aimInputsFRICTION
     * - <B> Altitude = double </B> <br> OR
     * - <B> Altitude = [double, ... , double] </B> <br>
     *  Altitude in units of ft for 'US' units and m for 'SI' units.
     *  Must have the same number of entries as Mach.
     */

  } else if (index == inBL_Transition) {
    *ainame               = EG_strdup("BL_Transition");
    defval->type          = Double;
    defval->dim           = Scalar;
    defval->vals.real     = 0.1;
    defval->limits.dlims[0] = 0.0; // Limit of accepted values
    defval->limits.dlims[1] = 1.0;

    /*! \page aimInputsFRICTION
     * - <B> BL_Transition = double [0.1 default] </B> <br>
     * Boundary layer laminar to turbulent transition percentage [0.0 turbulent to 1.0 laminar] location for all surfaces.
     */

  } else if (index == inTwTaw) {
    *ainame               = EG_strdup("TwTaw");
    defval->type          = Double;
    defval->dim           = Scalar;
    defval->vals.real     = 1.0;

    /*! \page aimInputsFRICTION
     * - <B> TwTaw = double [1.0 default] </B> <br>
     * Tw/Taw, the wall temperature ratio for all surfaces.
     */

  } else if (index == inLiftSurface) {
    *ainame              = EG_strdup("LiftSurface");
    defval->type         = Tuple;
    defval->nullVal      = IsNull;
    //defval->units        = NULL;
    defval->dim          = Vector;
    defval->lfixed       = Change;
    defval->vals.tuple   = NULL;

    /*! \page aimInputsFRICTION
     * - <B>LiftSurface = NULL </B> <br>
     * Defines lifting surfaces. See \ref fcnSurface for additional details.
     */

  } else if (index == inRevolveSurface) {
    *ainame              = EG_strdup("RevolveSurface");
    defval->type         = Tuple;
    defval->nullVal      = IsNull;
    //defval->units        = NULL;
    defval->dim          = Vector;
    defval->lfixed       = Change;
    defval->vals.tuple   = NULL;

    /*! \page aimInputsFRICTION
     * - <B>RevolveSurface = NULL </B> <br>
     * Defines revolution surfaces. See \ref fcnSurface for additional details.
     */

  } else {
    AIM_ERROR(aimInfo, "Unknown index %d", index);
    return CAPS_NOTIMPLEMENT;
  }

cleanup:
  return status;
}


// ********************** AIM Function Break *****************************
int aimUpdateState(/*@unused@*/ void *instStore, /*@unused@*/ void *aimInfo,
                   /*@unused@*/ capsValue *inputs)
{
  int status; // Function status return

  int i, isurf, ispan, icase; //Indexing

  // Bodies
  int numBody;
  ego *bodies = NULL;

  double      ReL, ReC;
  double      CfTurbL, CfTurbC, CfLam;
  double      Cf, CfSw, CfSwFF;

  int         foundSref = (int)false;

  double Sref = 1.0;

  int lunit;
  double xme;
  double alt, T, P, RHO, A, MU, TS, RR, PP, RM, QM;
  double RN;

  // Defined types
  int numLiftSurface = 0;
  fcnSurfaceStruct *fcnLiftSurface = NULL;

  int numRevSurface = 0;
  fcnSurfaceStruct *fcnRevSurface = NULL;

  mapAttrToIndexStruct groupMap; // capsGroup attribute to index map

  // EGADS function returns
  int atype, alen;
  const int       *ints;
  const double    *reals;
  const char      *intents, *string;
  const char      *lengthUnitsIn;
  char            *areaUnitsIn=NULL;

  aimStorage *frictionInstance = (aimStorage *)instStore;


#ifdef DEBUG
  printf(" frictionAIM/aimPreAnalysis\n");
#endif

  status = initiate_mapAttrToIndexStruct(&groupMap);
  AIM_STATUS(aimInfo, status);

  AIM_NOTNULL(inputs, aimInfo, status);

  // Get EGADS bodies
  status = aim_getBodies(aimInfo, &intents, &numBody, &bodies);
  AIM_STATUS(aimInfo, status);

  if ((numBody <= 0) || (bodies == NULL)) {
    AIM_ERROR(aimInfo,"No Bodies!");
    status = CAPS_SOURCEERR;
    goto cleanup;
  }

  if (inputs[inMach-1].nullVal == IsNull) {
    AIM_ERROR(aimInfo, "Mach input must be set!\n");
    status = CAPS_NULLVALUE;
    goto cleanup;
  }

  if (inputs[inAltitude-1].nullVal == IsNull) {
    AIM_ERROR(aimInfo, "Altitude input must be set!\n");
    status = CAPS_NULLVALUE;
    goto cleanup;
  }

  if (inputs[inMach-1].length != inputs[inAltitude-1].length) {
    AIM_ERROR(aimInfo, "Inputs Mach and Altitude must be the same length\n");
    status = CAPS_MISMATCH;
    goto cleanup;
  }

  // Get length units
  status = aim_capsLength(aimInfo, &lengthUnitsIn);
  if (status != CAPS_SUCCESS) {
    AIM_ERROR(aimInfo, "No units assigned *** capsLength is not set in *.csm file!");
    goto cleanup;
  }

  status = aim_unitRaise(aimInfo, lengthUnitsIn, 2, &areaUnitsIn);
  AIM_STATUS(aimInfo, status);
  AIM_NOTNULL(areaUnitsIn, aimInfo, status);

  // search for the "capsReferenceArea" attribute
  for (i = 0; i < numBody; i++) {
    status = EG_attributeRet(bodies[i], "capsReferenceArea", &atype, &alen,
                             &ints, &reals, &string);
    if (status == EGADS_SUCCESS) {

      if (atype == ATTRREAL && alen == 1) {
        Sref = reals[0];
        status = aim_convert(aimInfo, 1, areaUnitsIn, &Sref, frictionInstance->area, &Sref);
        AIM_STATUS(aimInfo, status);
        foundSref = (int)true;
      } else {
        AIM_ERROR(aimInfo, "capsReferenceArea should be followed by a single real value!");
        status = EGADS_ATTRERR;
        goto cleanup;
      }
    }
  }

  if (foundSref == (int)false) {
    AIM_ERROR(aimInfo, "capsReferenceArea is not set on any body!");
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  if (aim_newGeometry(aimInfo) == CAPS_SUCCESS) {

    // destroy previous surfaces
    for (i = 0; i < frictionInstance->numLiftSurface; i++) {
      destroy_vlmSurfaceStruct(&frictionInstance->vlmLiftSurface[i]);
      destroy_fcnSurfaceStruct(&frictionInstance->fcnLiftSurface[i]);
    }
    AIM_FREE(frictionInstance->vlmLiftSurface);
    AIM_FREE(frictionInstance->fcnLiftSurface);
    frictionInstance->numLiftSurface = 0;

    for (i = 0; i < frictionInstance->numRevSurface; i++) {
      destroy_vlmSurfaceStruct(&frictionInstance->vlmRevSurface[i]);
      destroy_fcnSurfaceStruct(&frictionInstance->fcnRevSurface[i]);
    }
    AIM_FREE(frictionInstance->vlmRevSurface);
    AIM_FREE(frictionInstance->fcnRevSurface);
    frictionInstance->numRevSurface = 0;

    // Get capsGroup name and index mapping to make sure all bodies have a capsGroup value
    status = create_CAPSGroupAttrToIndexMap(numBody,
                                            bodies,
                                            0, // Only search down to the body level of the EGADS body
                                            &groupMap);
    AIM_STATUS(aimInfo, status);


    // Get lifting surface inputs
    status = get_fcnSurface(aimInfo,
                            inputs[inBL_Transition-1].vals.real,
                            inputs[inTwTaw-1].vals.real,
                            inputs[inLiftSurface-1].length,
                            inputs[inLiftSurface-1].vals.tuple,
                            &groupMap,
                            &frictionInstance->numLiftSurface,
                            &frictionInstance->vlmLiftSurface,
                            &frictionInstance->fcnLiftSurface);
    AIM_STATUS(aimInfo, status);

    // Accumulate section data
    status = vlm_getSections(aimInfo, numBody, bodies, NULL, groupMap, vlmGENERIC, frictionInstance->numLiftSurface, &frictionInstance->vlmLiftSurface);
    AIM_STATUS(aimInfo, status);

    status = get_fcnSurfaceData(aimInfo, frictionInstance, (int)false, lengthUnitsIn, areaUnitsIn,
                                frictionInstance->numLiftSurface, frictionInstance->vlmLiftSurface, frictionInstance->fcnLiftSurface);
    AIM_STATUS(aimInfo, status);



    // Get revolution surface inputs
    status = get_fcnSurface(aimInfo,
                            inputs[inBL_Transition-1].vals.real,
                            inputs[inTwTaw-1].vals.real,
                            inputs[inRevolveSurface-1].length,
                            inputs[inRevolveSurface-1].vals.tuple,
                            &groupMap,
                            &frictionInstance->numRevSurface,
                            &frictionInstance->vlmRevSurface,
                            &frictionInstance->fcnRevSurface);
    AIM_STATUS(aimInfo, status);

    // Accumulate section data
    status = vlm_getSections(aimInfo, numBody, bodies, NULL, groupMap, vlmBODY, frictionInstance->numRevSurface, &frictionInstance->vlmRevSurface);
    AIM_STATUS(aimInfo, status);

    status = get_fcnSurfaceData(aimInfo, frictionInstance, (int)true, lengthUnitsIn, areaUnitsIn,
                                frictionInstance->numRevSurface, frictionInstance->vlmRevSurface, frictionInstance->fcnRevSurface);
    AIM_STATUS(aimInfo, status);
  }

  if (aim_newGeometry(aimInfo)               == CAPS_SUCCESS ||
      aim_newAnalysisIn(aimInfo, inMach    ) == CAPS_SUCCESS ||
      aim_newAnalysisIn(aimInfo, inAltitude) == CAPS_SUCCESS) {

    numLiftSurface = frictionInstance->numLiftSurface;
    fcnLiftSurface = frictionInstance->fcnLiftSurface;

    numRevSurface = frictionInstance->numRevSurface;
    fcnRevSurface = frictionInstance->fcnRevSurface;

    // allocate storage for drag calculations
    for (isurf = 0; isurf < numLiftSurface; isurf++) {
      AIM_NOTNULL(fcnLiftSurface, aimInfo, status);
      AIM_FREE(fcnLiftSurface[isurf].ReL    );
      AIM_FREE(fcnLiftSurface[isurf].CDtotal);
      AIM_FREE(fcnLiftSurface[isurf].CDform );
      AIM_FREE(fcnLiftSurface[isurf].CDfric );

      AIM_ALLOC(fcnLiftSurface[isurf].ReL    , inputs[inMach-1].length, double, aimInfo, status);
      AIM_ALLOC(fcnLiftSurface[isurf].CDtotal, inputs[inMach-1].length, double, aimInfo, status);
      AIM_ALLOC(fcnLiftSurface[isurf].CDform , inputs[inMach-1].length, double, aimInfo, status);
      AIM_ALLOC(fcnLiftSurface[isurf].CDfric , inputs[inMach-1].length, double, aimInfo, status);
      fcnLiftSurface[isurf].numCase = inputs[inMach-1].length;
    }

    for (isurf = 0; isurf < numRevSurface; isurf++) {
      AIM_NOTNULL(fcnRevSurface, aimInfo, status);
      AIM_FREE(fcnRevSurface[isurf].ReL    );
      AIM_FREE(fcnRevSurface[isurf].CDtotal);
      AIM_FREE(fcnRevSurface[isurf].CDform );
      AIM_FREE(fcnRevSurface[isurf].CDfric );

      AIM_ALLOC(fcnRevSurface[isurf].ReL    , inputs[inMach-1].length, double, aimInfo, status);
      AIM_ALLOC(fcnRevSurface[isurf].CDtotal, inputs[inMach-1].length, double, aimInfo, status);
      AIM_ALLOC(fcnRevSurface[isurf].CDform , inputs[inMach-1].length, double, aimInfo, status);
      AIM_ALLOC(fcnRevSurface[isurf].CDfric , inputs[inMach-1].length, double, aimInfo, status);
      fcnRevSurface[isurf].numCase = inputs[inMach-1].length;
    }


    aim_freeValue(&frictionInstance->CDtotal);
    frictionInstance->CDtotal.type   = Double;
    frictionInstance->CDtotal.dim    = inputs[inMach-1].length == 1 ? Scalar : Vector;
    frictionInstance->CDtotal.nrow   = inputs[inMach-1].length;
    frictionInstance->CDtotal.length = frictionInstance->CDtotal.nrow;
    if (frictionInstance->CDtotal.nrow > 1) {
      AIM_ALLOC(frictionInstance->CDtotal.vals.reals, frictionInstance->CDtotal.nrow, double, aimInfo, status);
      for (i = 0; i < frictionInstance->CDtotal.nrow; i++) frictionInstance->CDtotal.vals.reals[i] = 0;
    }

    aim_freeValue(&frictionInstance->CDform);
    frictionInstance->CDform.type   = Double;
    frictionInstance->CDform.dim    = inputs[inMach-1].length == 1 ? Scalar : Vector;
    frictionInstance->CDform.nrow   = inputs[inMach-1].length;
    frictionInstance->CDform.length = frictionInstance->CDform.nrow;
    if (frictionInstance->CDform.nrow > 1) {
      AIM_ALLOC(frictionInstance->CDform.vals.reals, frictionInstance->CDform.nrow, double, aimInfo, status);
      for (i = 0; i < frictionInstance->CDform.nrow; i++) frictionInstance->CDform.vals.reals[i] = 0;
    }

    aim_freeValue(&frictionInstance->CDfric);
    frictionInstance->CDfric.type   = Double;
    frictionInstance->CDfric.dim    = inputs[inMach-1].length == 1 ? Scalar : Vector;
    frictionInstance->CDfric.nrow   = inputs[inMach-1].length;
    frictionInstance->CDfric.length = frictionInstance->CDfric.nrow;
    if (frictionInstance->CDfric.nrow > 1) {
      AIM_ALLOC(frictionInstance->CDfric.vals.reals, frictionInstance->CDfric.nrow, double, aimInfo, status);
      for (i = 0; i < frictionInstance->CDfric.nrow; i++) frictionInstance->CDfric.vals.reals[i] = 0;
    }


    for (icase = 0; icase < inputs[inMach-1].length; icase++) {

      // get Mach and Altitude
      if (inputs[inMach-1].length == 1) {
        xme = inputs[inMach-1].vals.real;
        alt = inputs[inAltitude-1].vals.real;
      } else {
        xme = inputs[inMach-1].vals.reals[icase];
        alt = inputs[inAltitude-1].vals.reals[icase];
      }

      // Get the atmosphere conditions
      lunit = strcmp(frictionInstance->length,"m") == 0 ? 0 : 1;
      status = stdatm(aimInfo, lunit, alt,
                      &T, &P, &RHO, &A,
                      &MU, &TS, &RR, &PP,
                      &RM, &QM);
      AIM_STATUS(aimInfo, status);

      RN  = RM*xme;

      // Lifting surfaces
      for (isurf = 0; isurf < numLiftSurface; isurf++) {
        AIM_NOTNULL(fcnLiftSurface, aimInfo, status);

        CfSw   = 0;
        CfSwFF = 0;
        for (ispan = 0; ispan < fcnLiftSurface[isurf].numSpan; ispan++) {

          ReL = RN*fcnLiftSurface[isurf].fcnSpan[ispan].refLength;
          status = turbcf(aimInfo, ReL, xme, fcnLiftSurface[isurf].TwTaw, &CfTurbL);
          AIM_STATUS(aimInfo, status);

          if (fcnLiftSurface[isurf].turbTrans > 0) {
            ReC = ReL*fcnLiftSurface[isurf].turbTrans;
            status = turbcf(aimInfo, ReC, xme, fcnLiftSurface[isurf].TwTaw, &CfTurbC);
            AIM_STATUS(aimInfo, status);

            status = lamcf(ReC, xme, fcnLiftSurface[isurf].TwTaw, &CfLam);
            AIM_STATUS(aimInfo, status);
          } else{
            CfTurbC = 0;
            CfLam = 0;
          }

          Cf     = CfTurbL - fcnLiftSurface[isurf].turbTrans*(CfTurbC - CfLam);
          CfSw   += Cf*fcnLiftSurface[isurf].fcnSpan[ispan].Swet;
          CfSwFF += Cf*fcnLiftSurface[isurf].fcnSpan[ispan].Swet*fcnLiftSurface[isurf].fcnSpan[ispan].formFactor;
        }

        fcnLiftSurface[isurf].ReL[icase]     = RN*fcnLiftSurface[isurf].refLength;;
        fcnLiftSurface[isurf].CDtotal[icase] = CfSwFF/Sref;
        fcnLiftSurface[isurf].CDform[icase]  = (CfSwFF-CfSw)/Sref;
        fcnLiftSurface[isurf].CDfric[icase]  = CfSw/Sref;
        fcnLiftSurface[isurf].formFactor     = CfSwFF/CfSw;

        if (inputs[inMach-1].length == 1) {
          frictionInstance->CDtotal.vals.real += fcnLiftSurface[isurf].CDtotal[icase];
          frictionInstance->CDform.vals.real  += fcnLiftSurface[isurf].CDform[icase];
          frictionInstance->CDfric.vals.real  += fcnLiftSurface[isurf].CDfric[icase];
        } else {
          frictionInstance->CDtotal.vals.reals[icase] += fcnLiftSurface[isurf].CDtotal[icase];
          frictionInstance->CDform.vals.reals[icase]  += fcnLiftSurface[isurf].CDform[icase];
          frictionInstance->CDfric.vals.reals[icase]  += fcnLiftSurface[isurf].CDfric[icase];
        }
      }

      // Body of revolution surfaces
      for (isurf = 0; isurf < numRevSurface; isurf++) {
        AIM_NOTNULL(fcnRevSurface, aimInfo, status);

        ReL = RN*fcnRevSurface[isurf].refLength;
        status = turbcf(aimInfo, ReL, xme, fcnRevSurface[isurf].TwTaw, &CfTurbL);
        AIM_STATUS(aimInfo, status);

        if (fcnRevSurface[isurf].turbTrans > 0) {
          ReC = ReL*fcnRevSurface[isurf].turbTrans;
          status = turbcf(aimInfo, ReC, xme, fcnRevSurface[isurf].TwTaw, &CfTurbC);
          AIM_STATUS(aimInfo, status);

          status = lamcf(ReC, xme, fcnRevSurface[isurf].TwTaw, &CfLam);
          AIM_STATUS(aimInfo, status);
        } else{
          CfTurbC = 0;
          CfLam = 0;
        }

        Cf     = CfTurbL - fcnRevSurface[isurf].turbTrans*(CfTurbC - CfLam);
        CfSw   = Cf*fcnRevSurface[isurf].Swet;
        CfSwFF = Cf*fcnRevSurface[isurf].Swet*fcnRevSurface[isurf].formFactor;

        fcnRevSurface[isurf].ReL[icase]     = ReL;
        fcnRevSurface[isurf].CDtotal[icase] = CfSwFF/Sref;
        fcnRevSurface[isurf].CDform[icase]  = (CfSwFF-CfSw)/Sref;
        fcnRevSurface[isurf].CDfric[icase]  = CfSw/Sref;

        if (inputs[inMach-1].length == 1) {
          frictionInstance->CDtotal.vals.real += fcnRevSurface[isurf].CDtotal[icase];
          frictionInstance->CDform.vals.real  += fcnRevSurface[isurf].CDform[icase];
          frictionInstance->CDfric.vals.real  += fcnRevSurface[isurf].CDfric[icase];
        } else {
          frictionInstance->CDtotal.vals.reals[icase] += fcnRevSurface[isurf].CDtotal[icase];
          frictionInstance->CDform.vals.reals[icase]  += fcnRevSurface[isurf].CDform[icase];
          frictionInstance->CDfric.vals.reals[icase]  += fcnRevSurface[isurf].CDfric[icase];
        }
      }
    }

  }

  status = CAPS_SUCCESS;

cleanup:

  destroy_mapAttrToIndexStruct(&groupMap);

  AIM_FREE(areaUnitsIn);

  return status;
}


// ********************** AIM Function Break *****************************
int aimPreAnalysis(/*@unused@*/ const void *instStore, /*@unused@*/ void *aimInfo,
                   /*@unused@*/ /*@null@*/ capsValue *inputs)
{
  return CAPS_SUCCESS;
}


// ********************** AIM Function Break *****************************
int aimExecute(/*@unused@*/ const void *instStore, /*@unused@*/ void *aimInfo,
               int *state)
{
  *state = 0;
  return CAPS_SUCCESS;
}


// ********************** AIM Function Break *****************************
int aimPostAnalysis(/*@unused@*/ void *instStore, /*@unused@*/ void *aimInfo,
                    /*@unused@*/ int restart, /*@unused@*/ capsValue *inputs)
{
  int status = CAPS_SUCCESS;
  int i, k, isurf, icase;

#define TMPLEN 128
  char tmp[TMPLEN], *value=NULL;
  size_t len;
#define NUMOUTGEOM 4
#define NUMOUTCASE 4
  const char* outNames[NUMOUTCASE] = {"ReL","CDtotal", "CDform", "CDfric"};
  capsValue surfVal;

  int numCase = 0;
  int numSurface;
  vlmSurfaceStruct *vlmSurface;
  fcnSurfaceStruct *fcnSurface;

  aimStorage *frictionInstance = (aimStorage *)instStore;

  aim_initValue(&surfVal);

  for (k = 0; k < 2; k++) {

    if (k == 0) {
      numSurface = frictionInstance->numLiftSurface;
      fcnSurface = frictionInstance->fcnLiftSurface;
    } else {
      numSurface = frictionInstance->numRevSurface;
      fcnSurface = frictionInstance->fcnRevSurface;
    }

    for (isurf = 0; isurf < numSurface; isurf++)
      numCase = MAX(numCase, fcnSurface[isurf].numCase);
  }

  // get a size estimate for the value string
  snprintf(tmp, TMPLEN, "%+16.12e,", 1e100);
  len = numCase*strlen(tmp)+4;
  AIM_ALLOC(value, len, char, aimInfo, status);
  value[0] = '\0';


  for (k = 0; k < 2; k++) {

    if (k == 0) {
      numSurface = frictionInstance->numLiftSurface;
      vlmSurface = frictionInstance->vlmLiftSurface;
      fcnSurface = frictionInstance->fcnLiftSurface;
    } else {
      numSurface = frictionInstance->numRevSurface;
      vlmSurface = frictionInstance->vlmRevSurface;
      fcnSurface = frictionInstance->fcnRevSurface;
    }

    for (isurf = 0; isurf < numSurface; isurf++) {
      surfVal.type = Tuple;
      AIM_ALLOC(surfVal.vals.tuple, NUMOUTCASE+NUMOUTGEOM, capsTuple, aimInfo, status);
      for (i = 0; i < NUMOUTCASE+NUMOUTGEOM; i++) {
        surfVal.vals.tuple[i].name  = NULL;
        surfVal.vals.tuple[i].value = NULL;
      }
      surfVal.length = surfVal.nrow = NUMOUTCASE+NUMOUTGEOM;
      surfVal.dim = Vector;

      for (i = 0; i < NUMOUTCASE; i++) {
        AIM_STRDUP(surfVal.vals.tuple[i].name , outNames[i], aimInfo, status);

        value[0] = '\0';
        if (fcnSurface[isurf].numCase > 1) strcpy(value, "[");

        for (icase = 0; icase < fcnSurface[isurf].numCase; icase++) {
          if (i == 0) {
            snprintf(value+strlen(value), len-strlen(value), "%16.12e", fcnSurface[isurf].ReL[icase]);
          } else if (i == 1) {
            snprintf(value+strlen(value), len-strlen(value), "%16.12e", fcnSurface[isurf].CDtotal[icase]);
          } else if (i == 2) {
            snprintf(value+strlen(value), len-strlen(value), "%16.12e", fcnSurface[isurf].CDform[icase]);
          } else if (i == 3) {
            snprintf(value+strlen(value), len-strlen(value), "%16.12e", fcnSurface[isurf].CDfric[icase]);
          }

          if (icase < fcnSurface[isurf].numCase-1) strcat(value, ",");
        }
        if (fcnSurface[isurf].numCase > 1) strcat(value, "]");

        AIM_STRDUP(surfVal.vals.tuple[i].value, value, aimInfo, status);
      }

      i = NUMOUTCASE;
      AIM_STRDUP(surfVal.vals.tuple[i].name , "Swet", aimInfo, status);
      snprintf(tmp, TMPLEN, "[%16.12e, \"%s\"]", fcnSurface[isurf].Swet, frictionInstance->area);
      AIM_STRDUP(surfVal.vals.tuple[i].value, tmp, aimInfo, status);

      i++;
      AIM_STRDUP(surfVal.vals.tuple[i].name , "RefL", aimInfo, status);
      snprintf(tmp, TMPLEN, "[%16.12e, \"%s\"]", fcnSurface[isurf].refLength, frictionInstance->length);
      AIM_STRDUP(surfVal.vals.tuple[i].value, tmp, aimInfo, status);

      i++;
      AIM_STRDUP(surfVal.vals.tuple[i].name , "ToC", aimInfo, status);
      snprintf(tmp, TMPLEN, "%16.12e", fcnSurface[isurf].thickOverChord);
      AIM_STRDUP(surfVal.vals.tuple[i].value, tmp, aimInfo, status);

      i++;
      AIM_STRDUP(surfVal.vals.tuple[i].name , "FormFactor", aimInfo, status);
      snprintf(tmp, TMPLEN, "%16.12e", fcnSurface[isurf].formFactor);
      AIM_STRDUP(surfVal.vals.tuple[i].value, tmp, aimInfo, status);

      /* create the dynamic output */
      status = aim_makeDynamicOutput(aimInfo, vlmSurface[isurf].name, &surfVal);
      AIM_STATUS(aimInfo, status);
    }
  }

  status = CAPS_SUCCESS;
cleanup:
  aim_freeValue(&surfVal);
  AIM_FREE(value);
  return status;
}


// ********************** AIM Function Break *****************************
int aimOutputs(void *instStore, /*@unused@*/ void *aimInfo,
               int index, char **aoname, capsValue *form)
{
  int status = CAPS_SUCCESS;

  aimStorage *frictionInstance = (aimStorage *)instStore;
  AIM_NOTNULL(frictionInstance, aimInfo, status);

#ifdef DEBUG
  printf(" frictionAIM/aimOutputs  index = %d!\n", index);
#endif

  /*! \page aimOutputsFRICTION AIM Outputs
   * Total, Form, and Friction drag components:
   */

  form->type   = Double;
  form->lfixed = Change;
  form->sfixed = Fixed;
  form->dim    = Vector;
  form->nrow   = 1;
  form->ncol   = 1;
  form->vals.real = 0.0;
  form->vals.reals = NULL;

  if (index == outCDtotal) {

    *aoname = EG_strdup("CDtotal");

    /*! \page aimOutputsFRICTION
     * - <B> CDtotal = </B> Total Drag Coefficient [CDform + CDfric] for each analysis case.
     */

  } else if (index == outCDform) {

    *aoname = EG_strdup("CDform");

    /*! \page aimOutputsFRICTION
     * - <B> CDform = </B> Total Form Drag Coefficient for each analysis case.
     */

  } else if (index == outCDfric) {

    *aoname = EG_strdup("CDfric");

    /*! \page aimOutputsFRICTION
     * - <B> CDfric = </B> Total Friction Drag Coefficient for each analysis case.
     */

  } else if (index == outSwet) {

    *aoname = EG_strdup("Swet");
    form->type   = Double;
    form->lfixed = Fixed;
    form->sfixed = Fixed;
    form->dim    = Scalar;
    form->nrow   = 1;
    form->ncol   = 1;
    AIM_STRDUP(form->units, frictionInstance->area, aimInfo, status);

    /*! \page aimOutputsFRICTION
     * - <B> Swet = </B> Total wetted area.
     */

  } else {
    AIM_ERROR(aimInfo, "Unknown output index %d", index);
    status = CAPS_NOTIMPLEMENT;
    goto cleanup;
  }

  /*! \page aimOutputsFRICTION
   * Geometric component are also available as dynamic outputs. In addition to the above these include:
   * - <B> Swet = </B> Component wetted area
   * - <B> RefL = </B> Reference length (average chord for lifting surface, total length for body of revolution)
   * - <B> ReL = </B> Component Reynolds number based on RefL
   * - <B> ToC = </B> Wetted area averaged Thickness to Chord ratio for lifting surfaces (max Diameter to Length for body of revolution).
   * - <B> FormFactor = </B> The Form Factor for the surface (area averaged for lifting surfaces)
   */

cleanup:
  return status;
}


// ********************** AIM Function Break *****************************
int aimCalcOutput(/*@unused@*/ void *instStore, /*@unused@*/ void *aimInfo,
                  int index, capsValue *val)
{

  int status; // Function return status

  int k, isurf;

  int numSurface;
  fcnSurfaceStruct *fcnSurface;

  aimStorage *frictionInstance = (aimStorage *)instStore;

#ifdef DEBUG
  printf(" frictionAIM/aimCalcOutput  index = %d!\n", index);
#endif

  if (index == outCDtotal) {
    *val = frictionInstance->CDtotal;
    aim_initValue(&frictionInstance->CDtotal);
  }
  else if (index == outCDform) {
    *val = frictionInstance->CDform;
    aim_initValue(&frictionInstance->CDform);
  }
  else if (index == outCDfric) {
    *val = frictionInstance->CDfric;
    aim_initValue(&frictionInstance->CDfric);
  }
  else if (index == outSwet) {

    val->vals.real = 0.0;

    for (k = 0; k < 2; k++) {

      if (k == 0) {
        numSurface = frictionInstance->numLiftSurface;
        fcnSurface = frictionInstance->fcnLiftSurface;
      } else {
        numSurface = frictionInstance->numRevSurface;
        fcnSurface = frictionInstance->fcnRevSurface;
      }

      for (isurf = 0; isurf < numSurface; isurf++) {
        val->vals.real += fcnSurface[isurf].Swet;
      }
    }

  } else {
    AIM_ERROR(aimInfo, "Unknown output index %d", index);
    status = CAPS_NOTIMPLEMENT;
    goto cleanup;
  }

  status = CAPS_SUCCESS;

cleanup:

  return status;
}


// ********************** AIM Function Break *****************************
void
aimCleanup(/*@unused@*/ void *aimStore)
{
#ifdef DEBUG
  printf(" frictionAIM/aimCleanup!\n");
#endif
  aimStorage *frictionInstance = (aimStorage *)aimStore;

  destroy_aimStorage(frictionInstance);
  AIM_FREE(frictionInstance);
}
