/*
 *   DISTRIBUTION STATEMENT A. Approved for public release: distribution is unlimited. PA#: AFRL-2024-6115.
 *
 *     CAPS: Computational Aircraft Prototype Syntheses
 *
 *             Missile Datcom AIM
 *
 *      Copyright 2014-2025, Massachusetts Institute of Technology
 *      Licensed under The GNU Lesser General Public License, version 2.1
 *      See http://www.opensource.org/licenses/lgpl-2.1.php
 *
 *     Written by Dr. Ryan Durscher (AFRL/RQVC) and Marshall Galbraith (MIT)
 *
 *
 */

/*! \mainpage Introduction
 *
 * \section overviewMissileDatcom Missile Datcom AIM Overview
 * A module in the Computational Aircraft Prototype Syntheses (CAPS) has been developed to interact with
 * Missile Datcom.
 *
 * Details on the use of units are outlined in \ref aimUnitsMissileDatcom.
 *
 * An outline of the AIM's inputs, outputs and attributes are provided in \ref aimInputsMissileDatcom,
 * \ref aimOutputMissileDatcom and \ref attributesMissileDatcom, respectively.
 *
 *
 * \section assumptionsMissileDatcom Geometry Requirements and Assumptions
 * The Missile DATCOM coordinate system assumption (X -- downstream, Y -- out the right wing, Z -- up) needs to be followed.
 *
 * \subsection airfoilMissileDatcom Airfoils in ESP
 * Within <b>OpenCSM</b> there are a number of airfoil generation UDPs (User Defined Primitives). These include NACA 4
 * series, a more general NACA 4/5/6 series generator, Sobieczky's PARSEC parameterization and Kulfan's CST
 * parameterization. All of these UDPs generate <b>EGADS</b> <em>FaceBodies</em> where the <em>Face</em>'s underlying
 * <em>Surface</em>  is planar and the bounds of the <em>Face</em> is a closed set of <em>Edges</em> whose
 * underlying <em>Curves</em> contain the airfoil shape.
 *
 * <b>Important Airfoil Geometry Assumptions</b>
 * - There must be a <em>Node</em> that represents the <em>Leading Edge</em> point
 * - For a sharp trailing edge, there must be a <em>Nodes</em> at the <em>Trailing Edge</em>
 * - For a blunt trailing edge, the airfoil curve may be open, or closed by a single <em>Edge</em> connecting the upper/lower <em>Nodes</em>
 * - For a <em>FaceBody</em>, the airfoil coordinates traverse counter-clockwise around the <em>Face</em> normal. The <b>OpenCSM</b> <em>REORDER</em> operation may be used to flip the <em>Face</em> normal.
 * - For a <em>WireBody</em>, the airfoil coordinates traverse in the order of the loop
 *
 * <b>Note:</b> Additional spurious <em>Nodes</em> on the upper and lower <em>Edges</em> of the airfoil are acceptable.
 *
 * A <em>NodeBody</em> may be used for a sharp fin tip.
 *
 * Missile DATCOM also supports the the specific HEX, ARC, USER, and NACA fin sets. The fin set and
 * corresponding ESP airfoil section are listed in the table below.
 *
 * | missile DATCOM Airfoil | ESP Airfoil
 * |------------------------|-------------
 * |  HEX                   | misdathex
 * |  ARC                   | biconvex
 * |  USER                  | Any airfoil UDP
 * |  NACA                  | See FinSet SecNACA in \ref aimInputsMissileDatcom
 *
 * The misdathex UDP has the same arguments as the missile DATCOM HEX airfoil, i.e. zupper, zlower, lmaxu, lmaxl, lflatu,  and lflatl
 *
 * \section notesMissileDatacom Notes
 *  - Option 2 AIXBOD/ELLBOD support discontinuities by repeating sections 3 times (just like BLEND). Note that 2 repeated sections is not supported.
 *  - DEXIT is calculated based on any inner loop found on the last FaceBody of the missile's body using the following: \f$ D_{Exit} = \sqrt{A_{Loop}*4/\pi}\f$
 *
 * \section todosMissileDatacom ToDO
 *  - Add Base-Jet Plume Interaction Inputs
 *  - Add fin controls.
 *
 * \section distributionMissileDatcom Code Distribution
 *
 * DISTRIBUTION STATEMENT A. Approved for public release: distribution is unlimited. PA#: AFRL-2024-6115.
 */

/*!  \page attributesMissileDatcom AIM Attributes
 * The following list of attributes drives the Missile Datcom geometric definition. Attributes which
 * are required and those that are optional (note: Missile Datacom will provided default values inherently
 * for optional arguments) are marked so in the description:
 *
 * - <b> capsType</b> This string attribute labels the <em> FaceBody</em> as to which type the section
 *  is assigned. Can either be "Body" or "Fin".
 *
 * - <b> capsGroup</b> This string attribute labels the <em> FaceBody</em> as to which "Fin" sections need to be
 * grouped together.<br>
 *   Note: The capsGroup string for Fins should have a FinSet# prefix to group Fins into FinSets. For example: FinSet1_1, FinSet1_2, FinSet2_1, FinSet2_2<br>
 *   All Fins within a FinSet are assumed to have the same airfoil sections without checking.
 *
 * - <b> capsReferenceArea</b>  [Optional] This attribute may exist on any <em> Body</em>.  Its
 * value will be used as the SREF entry for reference area.
 *
 * - <b> capsReferenceChord</b>  [Optional] This attribute may exist on any <em> Body</em>.  Its
 * value will be used as the LREF entry for longitudinal reference length.
 *
 * - <b> capsReferenceSpan</b>  [Optional] This attribute may exist on any <em> Body</em>.  Its
 * value will be used as the LATREF entry for lateral reference length.
 *
 * - <b> capsReferenceX</b>  [Optional] This attribute may exist on any <em> Body</em>.  Its
 * value will be used as the XCG entry for longitudinal position of C.G.
 *
 * - <b> capsReferenceY</b>  [Optional] This attribute may exist on any <em> Body</em>.  Its
 * value will be used as the ZCG entry for vertical position of C.G.
 *
 */

#include <string.h>
#include <math.h>
#include "capsTypes.h"
#include "aimUtil.h"

#include "miscUtils.h"
#include "jsonUtils.h"
#include "vlmUtils.h"
#include "cfdUtils.h"

#ifdef WIN32
#define getcwd     _getcwd
#define snprintf   _snprintf
#define strcasecmp stricmp
#define strncasecmp _strnicmp
#define strtok_r   strtok_s
#define PATH_MAX   _MAX_PATH
#else
#include <unistd.h>
#include <limits.h>
#endif

#define MXCHAR  255
#define PI      3.1415926535897931159979635
#define MIN(A,B)         (((A) < (B)) ? (A) : (B))
#define MAX(A,B)         (((A) < (B)) ? (B) : (A))

static const char finSetKey[] = "FinSet";

enum aimInputs
{
  inMissileDATCOM = 1,           /* index is 1-based */
  inBODoption,
  inBODtnose,
  inBODpower,
  inBODbnose,
  inBODtrunc,
  inBODtaft,
  inFinSet,
  inMach,
  inRen,
  inVinf,
  inTinf,
  inPinf,
  inAlpha,
  inBeta,
  inAltitude,
  inBL_Type,
  inSurface_Roughness,
  inRoughness_Rating,
  NUMINPUT = inRoughness_Rating  /* Total number of inputs */
};


enum aimOutputs
{
  outCNtot = 1,                   /* index is 1-based */
  outCAtot,
  outCYtot,
  outCmtot,
  outCntot,
  outCltot,
  outCLtot,
  outCDtot,
  outXcp,
  outCNa,
  outCma,
  outCYb,
  outCnb,
  outClb,
  outAlpha,
  outMach,
  outAltitude,
  NUMOUTPUT = outAltitude       /* Total number of outputs */
};

enum misdat_BOD
{
  AXIBOD, // Axi-symmetric body
  ELLBOD  // Elliptic body
};

enum misdat_SECTYP
{
  mdHEX,
  NACA,
  mdARC,
  mdUSER,
  mdNODE, // NodeBody section (wildcard)
};

//#define DEBUG

typedef struct {
  int bodyIndex;
  ego body;      // reference to the body (do not delete)
  int discon;
  double xLoc;
  double width;  // Width or radius
  double height;
  double zOffset;
} mdBodySectionStruct;

typedef struct {

  char *sectyp; // type of section
  char *naca;   // naca airfoil control card (without there FinSet index)

  double *ler; // Leading Edge Radius

  int numPhif;
  double *phif;

  int numGam;
  double *gam;

  int nvor;

  int numSurf;
  vlmSurfaceStruct *surface;

} mdFinSetStruct;


/*===========================================================================*/
static void
initiate_mdBodySectionStruct(mdBodySectionStruct *section)
{
  section->bodyIndex = 0;
  section->body = NULL;
  section->discon = 0;
  section->xLoc = 0;
  section->width = 0;
  section->height = 0;
  section->zOffset = 0;
}

/*===========================================================================*/
static void
destroy_mdBodySectionStruct(mdBodySectionStruct *section)
{
  section->bodyIndex = 0;
  section->body = NULL;
  section->discon = 0;
  section->xLoc = 0;
  section->width = 0;
  section->height = 0;
  section->zOffset = 0;
}

/*===========================================================================*/
static void
copy_mdBodySectionStruct(mdBodySectionStruct *sectionIn,mdBodySectionStruct *sectionOut)
{
  sectionOut->bodyIndex = sectionIn->bodyIndex;
  sectionOut->body      = sectionIn->body;
  sectionOut->discon    = sectionIn->discon;
  sectionOut->xLoc      = sectionIn->xLoc;
  sectionOut->width     = sectionIn->width;
  sectionOut->height    = sectionIn->height;
  sectionOut->zOffset   = sectionIn->zOffset;
}

/*===========================================================================*/
static void
initiate_mdFinSetStruct(mdFinSetStruct *section)
{
  section->sectyp = NULL;
  section->naca = NULL;
  section->ler = NULL;
  section->numPhif = 0;
  section->phif = NULL;
  section->numGam = 0;
  section->gam = NULL;
  section->nvor = 1;

  section->numSurf = 0;
  section->surface = NULL;
}


/*===========================================================================*/
static void
destroy_mdFinSetStruct(mdFinSetStruct *section)
{
  int i;
  AIM_FREE(section->sectyp);
  AIM_FREE(section->naca);
  AIM_FREE(section->ler);
  section->numPhif = 0;
  AIM_FREE(section->phif);
  section->numGam = 0;
  AIM_FREE(section->gam);
  section->nvor = 1;

  if (section->surface != NULL) {
    for (i = 0; i < section->numSurf; i++)
      (void) destroy_vlmSurfaceStruct(&section->surface[i]);
    section->numSurf = 0;
    AIM_FREE(section->surface);
  }
}

typedef struct {

  char *dim;             // DIM input for MissileDatcom
  cfdUnitsStruct units;  // Units structure

  int BODtype; // 0 = AXIBOD, 1 = ELLBOD
  int BODoption;

  // nose parameters
  char *BODtnose;
  double BODpower;
  double BODbnose;
  int BODtrunc;

  // tail parameters
  char *BODtaft;
  double dexit;

  int numBodySection;
  mdBodySectionStruct *bodySection;

  mapAttrToIndexStruct finSetMap;

  int numFinSet;
  mdFinSetStruct *finSet;

} aimStorage;


/*===========================================================================*/
static int
initiate_aimStorage(aimStorage *mdInstance)
{
  int status = CAPS_SUCCESS;

  // Set initial values for mdInstance
  mdInstance->dim = NULL;
  initiate_cfdUnitsStruct(&mdInstance->units);

  mdInstance->BODtype = -1;
  mdInstance->BODoption = -1;

  mdInstance->BODtnose = NULL;
  mdInstance->BODpower = 0;
  mdInstance->BODbnose = 0;
  mdInstance->BODtrunc = (int)false;

  mdInstance->BODtaft = NULL;
  mdInstance->dexit = -1;

  mdInstance->numBodySection = 0;
  mdInstance->bodySection = NULL;

  (void)initiate_mapAttrToIndexStruct(&mdInstance->finSetMap);

  mdInstance->numFinSet = 0;
  mdInstance->finSet = NULL;

  return status;
}


/*===========================================================================*/
static int
destroy_aimStorage(aimStorage *mdInstance)
{
  int status = CAPS_SUCCESS;
  int i;

  AIM_FREE(mdInstance->dim);
  destroy_cfdUnitsStruct(&mdInstance->units);

  AIM_FREE(mdInstance->BODtnose);
  mdInstance->BODpower = 0;
  mdInstance->BODbnose = 0;
  mdInstance->BODtrunc = (int)false;

  AIM_FREE(mdInstance->BODtaft);
  mdInstance->dexit = -1;

  for (i = 0; i < mdInstance->numBodySection; i++) {
    destroy_mdBodySectionStruct(&mdInstance->bodySection[i]);
  }

  AIM_FREE(mdInstance->bodySection);
  mdInstance->bodySection = NULL;

  (void)destroy_mapAttrToIndexStruct(&mdInstance->finSetMap);

  for (i = 0; i < mdInstance->numFinSet; i++) {
    destroy_mdFinSetStruct(&mdInstance->finSet[i]);
  }

  AIM_FREE(mdInstance->finSet);

  return status;
}


/*===========================================================================*/
// Get the FinSet properties from a capsTuple
static int
missileDatcom_getFinSet(void *aimInfo,
                        int numFinSetTuple,
                        capsTuple finSetTuple[],
                        mapAttrToIndexStruct *finSetMap,
                        int numFinSet,
                        mdFinSetStruct finSets[]) {

  /*! \page mdFinSet Missile DATCOM FinSet
   * Structure for the constraint tuple  = ("FinSet Name", "Value").
   * "Constraint Name" defines the reference name for the constraint being specified.
   *  The "Value" can either be a JSON String dictionary (see Section \ref jsonStringFinSet) or a single string keyword
   *  (see Section \ref keyStringFinSet).
   */

  int status; //Function return

  int i, iFinSet; // Indexing

  char *keyValue = NULL; // Key values from tuple searches
  const char *keyWord = NULL; // Key words to find in the tuples

  // Nothing to do
  if (numFinSetTuple == 0) return CAPS_SUCCESS;

  printf("\nGetting FinSets.......\n");

  if (numFinSetTuple > numFinSet) {
    AIM_ERROR(aimInfo, "FinSet input exceeds the number of geometry FinSets");
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  printf("\tNumber of FinSets - %d\n", numFinSetTuple);

  // Loop through tuples and fill out the constraint structures
  for (i = 0; i < numFinSetTuple; i++) {

    printf("\tFinSet name - %s\n", finSetTuple[i].name );

    status = get_mapAttrToIndexIndex(finSetMap, finSetTuple[i].name, &iFinSet);
    if (status == CAPS_NOTFOUND) {
      AIM_ERROR(aimInfo, "Name %s not found in attribute map of FinSets!!!!\n", finSetTuple[i].name);
      goto cleanup;
    }

    // Do we have a json string?
    if (strncmp(finSetTuple[i].value, "{", 1) == 0) {
      //printf("JSON String - %s\n", constraintTuple[i].value);

      /*! \page mdFinSet
       * \section jsonStringFinSet JSON String Dictionary
       *
       * If "Value" is JSON string dictionary
       *  (eg. "Value" = {"Translation": "free", "Rotation":"free"})
       *  the following keywords ( = default values) may be used:
       */

      /*! \page mdFinSet
       *
       * <ul>
       *  <li> <B>SecNACA = "" </B> </li> <br>
       *  Airfoil section type for the FinSet.<br>
       *  For NACA, the full NACA control card should be specifed (without the FinSet index), e.g.:<br>
       *  SecTyp = "NACA-4-0012"<br>
       *  for a NACA 4-series airfoil. See the Missile Datcom NACA Control Card documentation for more info.
       * </ul>
       */
      keyWord = "SecNACA";
      status = search_jsonDictionary( finSetTuple[i].value, keyWord, &keyValue);
      if (status == CAPS_SUCCESS) {
        AIM_FREE(finSets[iFinSet].sectyp);
        finSets[iFinSet].sectyp = string_removeQuotation(keyValue);
        AIM_NOTNULL(finSets[iFinSet].sectyp, aimInfo, status);
        string_toUpperCase(finSets[iFinSet].sectyp);

        if (strncmp(finSets[iFinSet].sectyp, "NACA", 4) != 0) {
          AIM_ERROR(aimInfo, "SecNACA = '%s' should begin with 'NACA'");
          status = CAPS_BADVALUE;
          goto cleanup;
        }

        // Separate off the NACA Control card information
        AIM_STRDUP(finSets[iFinSet].naca, finSets[iFinSet].sectyp+4, aimInfo, status);
        finSets[iFinSet].sectyp[4] = '\0';
      }
      AIM_FREE(keyValue);


      /*! \page mdFinSet
       *
       * <ul>
       *  <li> <B>phif = <from geometry by default> </B> </li> <br>
       *  List of roll angle of each fin measured clockwise from top vertical center looking forward.<br>
       *  This overrides values otherwise computed from the geometry.<br>
       *  Note: If both phif and gam are specified they must be of the same length.
       * </ul>
       */
      keyWord = "phif";
      status = search_jsonDictionary( finSetTuple[i].value, keyWord, &keyValue);
      if (status == CAPS_SUCCESS) {

        status = string_toDoubleDynamicArray(keyValue, &finSets[iFinSet].numPhif, &finSets[iFinSet].phif);
        AIM_STATUS(aimInfo, status, "while parsing %s", keyValue);

      }
      AIM_FREE(keyValue);

      /*! \page mdFinSet
       *
       * <ul>
       *  <li> <B>gam = <from geometry by default> </B> </li> <br>
       *  List of roll angle of each fin measured clockwise from top vertical center looking forward
       *  This overrides values otherwise computed from the geometry.<br>
       *  Note: If both phif and gam are specified they must be of the same length.
       * </ul>
       */
      keyWord = "gam";
      status = search_jsonDictionary( finSetTuple[i].value, keyWord, &keyValue);
      if (status == CAPS_SUCCESS) {

        status = string_toDoubleDynamicArray(keyValue, &finSets[iFinSet].numGam, &finSets[iFinSet].gam);
        AIM_STATUS(aimInfo, status, "while parsing %s", keyValue);

      }
      AIM_FREE(keyValue);


      /*! \page mdFinSet
       *
       * <ul>
       *  <li> <B>nvor = 1 </B> </li> <br>
       *  Number of vortices shed from each panel, up to a maximum of 20
       * </ul>
       */
      keyWord = "nvor";
      status = search_jsonDictionary( finSetTuple[i].value, keyWord, &keyValue);
      if (status == CAPS_SUCCESS) {

        status = string_toInteger(keyValue, &finSets[iFinSet].nvor);
        AIM_STATUS(aimInfo, status, "while parsing %s", keyValue);

      }
      AIM_FREE(keyValue);

      if (finSets[iFinSet].numPhif > 0 && finSets[iFinSet].numGam > 0) {
        if (finSets[iFinSet].numPhif != finSets[iFinSet].numGam) {
          AIM_ERROR(aimInfo, "FinSet '%s': Length of 'phif' (%d) must equal length of 'gam' (%d)", finSetTuple[i].name, finSets[iFinSet].numPhif, finSets[iFinSet].numGam);
          status = CAPS_BADVALUE;
          goto cleanup;
        }
      }

    } else {

      // Call some look up table maybe?
      AIM_ERROR(aimInfo, "FinSet tuple value is expected to be a JSON string\n");
      return CAPS_BADVALUE;
    }
  }

  printf("\tDone getting FinSets\n");

  status = CAPS_SUCCESS;
cleanup:
  AIM_FREE(keyValue);

  return status;
}


/*===========================================================================*/
static int
missileDatcom_orderBody(int numSection, mdBodySectionStruct *section[])
{
  int status;

  int i, hit;

  mdBodySectionStruct temp;

  do {
    for (hit = i = 0; i < numSection-1; i++) {

      if ((*section)[i].xLoc <= (*section)[i+1].xLoc) continue;

      copy_mdBodySectionStruct(&(*section)[i], &temp);
      copy_mdBodySectionStruct(&(*section)[i+1], section[i]);
      copy_mdBodySectionStruct(&temp, &(*section)[i+1]);

      hit++;
    }
  } while (hit != 0);

  status = CAPS_SUCCESS;

//cleanup:
  return status;
}


/*===========================================================================*/
static int
missileDatcom_dexit( void *aimInfo, aimStorage *mdInstance)
{
  int status = CAPS_SUCCESS;

  int i;
  double data[5], area;
  int oclass, mtype, nchild;
  ego *echldrn, ref;
  int *sens, *loop_sens;

  int numLoop, numFace=0;
  ego *eloops=NULL, *efaces=NULL;

  mdBodySectionStruct *section = &mdInstance->bodySection[mdInstance->numBodySection-1];
  AIM_NOTNULL(section, aimInfo, status);
  AIM_NOTNULL(section->body, aimInfo, status);

  // Check for Dexit
  if (section->body->mtype == WIREBODY) {
    return CAPS_SUCCESS;
  } else if (section->body->mtype == FACEBODY) {

    // get the faces in the face body
    status = EG_getTopology(section->body, &ref, &oclass, &mtype, data, &numFace, &efaces, &sens);
    AIM_STATUS(aimInfo, status);

  } else if (section->body->mtype == SHEETBODY) {

    // get the shell
    status = EG_getTopology(section->body, &ref, &oclass, &mtype, data, &nchild, &echldrn, &sens);
    AIM_STATUS(aimInfo, status);

    if (nchild != 1) {
      AIM_ERROR(aimInfo, "Body is a SHEETBODY with more than 1 shell (nshell = %d)!", nchild);
      status = CAPS_BADTYPE;
      goto cleanup;
    }

    // get the faces in the shell
    status = EG_getTopology(echldrn[0], &ref, &oclass, &mtype, data, &numFace, &efaces, &sens);
    AIM_STATUS(aimInfo, status);

  } else if (section->body->mtype == SOLIDBODY) {
    AIM_ERROR(aimInfo, "Body is a SOLIDBODY which aren't supported (oclass = %d, mtype = %d)!", section->body->oclass, section->body->mtype);
    status = CAPS_BADTYPE;
    goto cleanup;
  }

  if (numFace != 1) {
    AIM_ERROR(aimInfo, "Body has more than one face (numFace = %d)!", numFace);
    status = CAPS_BADTYPE;
    goto cleanup;
  }
  AIM_NOTNULL(efaces, aimInfo, status);

  status = EG_getTopology(efaces[0], &ref, &oclass, &mtype, data, &numLoop, &eloops, &loop_sens);
  AIM_STATUS(aimInfo, status);
  AIM_NOTNULL(eloops, aimInfo, status);

  if (numLoop == 2 ) {
    for (i = 0; i < numLoop; i++) {

      // skip the outer loop
      if (loop_sens[i] == SOUTER) continue;

      // Get the area of the loop
      status = EG_getArea(eloops[i], NULL, &area);
      AIM_STATUS(aimInfo, status);

      // assume it's circular...
      mdInstance->dexit = sqrt(fabs(area)*4.0/PI);
      break;
    }
  }


  status = CAPS_SUCCESS;

cleanup:
  return status;
}


/*===========================================================================*/
static int
missileDatcom_classifySection(void *aimInfo, vlmSectionStruct *vlmSection, int *sectyp)
{
  int status;
  int i;
  int oclass, mtype, nchild, *sens;
  double data[4];

  int numEdge = 0;
  ego *eedges = NULL, ecurve, *echild=NULL;

  if (aim_isNodeBody(vlmSection->ebody, data) == CAPS_SUCCESS) {
    *sectyp = mdNODE;
    return CAPS_SUCCESS;
  }

  status = EG_getBodyTopos(vlmSection->ebody, NULL, EDGE, &numEdge, &eedges);
  AIM_STATUS(aimInfo, status);
  AIM_NOTNULL(eedges, aimInfo, status);

  if (numEdge == 2) {
    // ARC or USER?

    *sectyp = mdARC;

    // check the curves for the edges
    for (i = 0; i < numEdge; i++) {
      status = EG_getTopology(eedges[i], &ecurve, &oclass, &mtype, data, &nchild, &echild, &sens);
      AIM_STATUS(aimInfo, status);
      if (ecurve->mtype != CIRCLE) {
        *sectyp = mdUSER;
        break;
      }
    }

    status = CAPS_SUCCESS;
    goto cleanup;
  }

  if (numEdge == 4 || numEdge == 6) {
    // HEX or USER?

    *sectyp = mdHEX;

    // get the curves for the edges
    for (i = 0; i < numEdge; i++) {
      status = EG_getTopology(eedges[i], &ecurve, &oclass, &mtype, data, &nchild, &echild, &sens);
      AIM_STATUS(aimInfo, status);
      if (ecurve->mtype != LINE) {
        *sectyp = mdUSER;
        break;
      }
    }

    status = CAPS_SUCCESS;
    goto cleanup;
  }

  // Something else
  *sectyp = mdUSER;

  status = CAPS_SUCCESS;
cleanup:
  AIM_FREE(eedges);
  return status;
}

/*===========================================================================*/
static int
missileDatcom_parseBody(void *aimInfo, int numBody, ego *bodies,
                        aimStorage *mdInstance)
{
  int status;
  int i, j, k;
  int hit;

  int numBodySection = 0;
  mdBodySectionStruct *bodySection=NULL;

  int numFinSet = 0;
  mdFinSetStruct *finSet=NULL;

  double box[6], box2[6], xyz[3];

  // Attribute data
  int atype, alen;
  const int *ints;
  const double *reals;
  const char *string;

  int numLoop, numFace;

  int numFin;
  int numFinBody=0;
  ego *finBody=NULL, etmp;
  mapAttrToIndexStruct groupMap;
  int sectyp, sectyp0;

  const char *groupName;
  char finSetName[42];

  vlmSurfaceStruct **surface=NULL;

  // Container for attribute to index map
  status = initiate_mapAttrToIndexStruct(&groupMap);
  AIM_STATUS(aimInfo, status);

  for (i = 0; i < numBody; i++) {
    status = EG_attributeRet(bodies[i], "capsType", &atype, &alen, &ints, &reals, &string);
    if (status != EGADS_SUCCESS) {
      AIM_ERROR(aimInfo, "capsType not found on body %d!", i+1);
      print_AllAttr(aimInfo, bodies[i] );
      status = CAPS_NOTFOUND;
      goto cleanup;

    } else {
      if (atype != ATTRSTRING) {
        AIM_ERROR(aimInfo, "capsType should be followed by a single string!");
        print_AllAttr(aimInfo, bodies[i] );
        status = EGADS_ATTRERR;
        goto cleanup;
      }
    }

    if (strcasecmp(string, "Body") == 0) {
      numBodySection += 1;
      AIM_REALL(bodySection, numBodySection, mdBodySectionStruct, aimInfo, status);
      initiate_mdBodySectionStruct(&bodySection[numBodySection-1]);

      bodySection[numBodySection-1].bodyIndex = i;
      bodySection[numBodySection-1].body = bodies[i];

      status = EG_getBoundingBox(bodies[i], box);
      AIM_STATUS(aimInfo, status);

      if (fabs((box[3] - box[0])/max_DoubleVal(box[3],box[0])) > 1E-7) {
        AIM_ERROR(aimInfo, "'Body' cross-sections must be in the y-z plane (diff = %.12f)!", fabs((box[3] - box[0])/max_DoubleVal(box[3],box[0])));
        status = CAPS_BADVALUE;
        goto cleanup;
      }

      bodySection[numBodySection-1].xLoc = box[0];
      bodySection[numBodySection-1].width = (box[4]-box[1])/2;
      bodySection[numBodySection-1].height = (box[5]-box[2])/2;

      bodySection[numBodySection-1].zOffset = (box[5]+box[2])/2;

      // if it's a NodeBody -> continue
      if (aim_isNodeBody(bodies[i], xyz) == EGADS_SUCCESS) {
        continue;
      }

      // Check of the axilbody
      if (fabs((bodySection[numBodySection-1].width - bodySection[numBodySection-1].height)/max_DoubleVal(bodySection[numBodySection-1].width,bodySection[numBodySection-1].height)) < 1E-7) {
        mdInstance->BODtype = MAX(mdInstance->BODtype, AXIBOD);
      } else {
        mdInstance->BODtype = MAX(mdInstance->BODtype, ELLBOD);
      }

      if (bodies[i]->mtype == WIREBODY) {
        continue;
      } else if (bodies[i]->mtype == SOLIDBODY) {
        AIM_ERROR(aimInfo, "Body %d, is a SOLIDBODY which aren't supported (oclass = %d, mtype = %d)!",i+1, bodies[i]->oclass, bodies[i]->mtype);
        status = CAPS_BADTYPE;
        goto cleanup;
      }

      status = EG_getBodyTopos(bodies[i], NULL, FACE, &numFace, NULL);
      AIM_STATUS(aimInfo, status);

      if (numFace > 1) {
        AIM_ERROR(aimInfo, "Body %d, has more than one Face (numFace = %d)!",i+1, numFace);
        status = CAPS_BADTYPE;
        goto cleanup;
      }

      status = EG_getBodyTopos(bodies[i], NULL, LOOP, &numLoop, NULL);
      AIM_STATUS(aimInfo, status);

      if (numLoop > 2) {
        AIM_ERROR(aimInfo, "Body %d, has more than two Loops (numLoop = %d)!",i+1, numLoop);
        status = CAPS_BADTYPE;
        goto cleanup;
      }

      // check for repeated bodies to indicate discontinuity
      hit = 0;
      for (j = i+1; j < MIN(i+3,numBody); j++) {
        if (EG_isEquivalent(bodies[i], bodies[j]) == EGADS_SUCCESS) hit++;
      }
      if (hit == 1) {
        AIM_ERROR(aimInfo, "Body %d is repeated twice! Repeat 3-times for BOD Option 2 discontinues.",i+1);
        status = CAPS_BADTYPE;
        goto cleanup;
      }
      if (hit == 2) {
        i += 2; // skip repeated bodies
        bodySection[numBodySection-1].discon = 1;
      }

    } else if (strcasecmp(string, "Fin") == 0) {

      numFinBody += 1;
      AIM_REALL(finBody, numFinBody, ego, aimInfo, status);
      finBody[numFinBody-1] = bodies[i];

    } else {
      AIM_ERROR(aimInfo, "Body %d, unknown capsType - %s", i+1, string);
      AIM_ADDLINE(aimInfo, "capsType must be 'Body' or 'Fin'");
      status = EGADS_ATTRERR;
      goto cleanup;
    }
  }

  // Build up fins with just the fin bodies now
  if (numFinBody > 0) {
    AIM_NOTNULL(finBody, aimInfo, status);

    // sort the fins based on x
    do {
      for (hit = i = 0; i < numFinBody-1; i++) {

        status = EG_getBoundingBox(finBody[i  ], box);
        AIM_STATUS(aimInfo, status);
        status = EG_getBoundingBox(finBody[i+1], box2);
        AIM_STATUS(aimInfo, status);

        if (box[0] <= box2[0]) continue;

        etmp         = finBody[i  ];
        finBody[i  ] = finBody[i+1];
        finBody[i+1] = etmp;

        hit++;
      }
    } while (hit != 0);

    numFinSet = 0;
    for (i = 0; i < numFinBody; i++) {

      status = retrieve_CAPSGroupAttr(finBody[i], &groupName);
      AIM_STATUS(aimInfo, status);

      if (strncasecmp(groupName, finSetKey, strlen(finSetKey)) != 0) continue;

      // Check that the FinSet name is correct
      if (groupName[strlen(finSetKey)] < '1' || groupName[strlen(finSetKey)] > '9') {
        AIM_ERROR(aimInfo, "'%s' must be followed by an integer", finSetKey);
        status = CAPS_BADVALUE;
        goto cleanup;
      }

      strncpy(finSetName, groupName, strlen(finSetKey)+1);
      finSetName[strlen(finSetKey)+1] = '\0';

      status = increment_mapAttrToIndexStruct(&mdInstance->finSetMap, finSetName);
      if (status != CAPS_SUCCESS && status != EGADS_EXISTS) AIM_STATUS(aimInfo, status);
    }


    // Get capsGroup name and index mapping to make sure all bodies have a capsGroup value
    status = create_CAPSGroupAttrToIndexMap(numFinBody,
                                            finBody,
                                            0, // Only search down to the body level of the EGADS body
                                            &groupMap);
    AIM_STATUS(aimInfo, status);

    numFinSet = mdInstance->finSetMap.numAttribute;
    AIM_ALLOC(finSet, numFinSet, mdFinSetStruct, aimInfo, status);

    for (i = 0; i < numFinSet; i++) initiate_mdFinSetStruct(&finSet[i]);


    for (i = 0; i < numFinSet; i++) {

      numFin = 0;
      for (j = 0; j < groupMap.numAttribute; j++) {
        if (strncasecmp(mdInstance->finSetMap.attributeName[i], groupMap.attributeName[j], strlen(mdInstance->finSetMap.attributeName[i])) != 0) continue;
        numFin++;
      }

      AIM_FREE(surface);
      AIM_ALLOC(surface          , numFin, vlmSurfaceStruct*, aimInfo, status);
      AIM_ALLOC(finSet[i].surface, numFin, vlmSurfaceStruct , aimInfo, status);
      for (j = 0; j < numFin; j++) {
        surface[j] = NULL;
        initiate_vlmSurfaceStruct(&finSet[i].surface[j]);
      }
      finSet[i].numSurf = numFin;

      numFin = 0;
      for (j = 0; j < groupMap.numAttribute; j++) {

        if (strncasecmp(mdInstance->finSetMap.attributeName[i], groupMap.attributeName[j], strlen(mdInstance->finSetMap.attributeName[i])) != 0) continue;

        AIM_STRDUP(finSet[i].surface[numFin].name, groupMap.attributeName[j], aimInfo, status);

        finSet[i].surface[numFin].numAttr = 1;    // Number of capsGroup/attributes used to define a given surface
        AIM_ALLOC(finSet[i].surface[numFin].attrIndex, 1, int, aimInfo, status);

        finSet[i].surface[numFin].attrIndex[0] = groupMap.attributeIndex[j];
        surface[numFin] = &finSet[i].surface[numFin];
        numFin++;
      }

      // Accumulate section data
      status = vlm_getSections(aimInfo, numFinBody, finBody, NULL, groupMap, vlmRADIAL, numFin, surface);
      AIM_STATUS(aimInfo, status);

      sectyp0 = mdNODE;
      for (j = 0; j < finSet[i].numSurf; j++) {

        if (finSet[i].surface[j].numSection < 2) {
          AIM_ERROR(aimInfo, "Fins '%s' must have at least 2 sections!", finSet[i].surface[j].name);
          status = CAPS_BADVALUE;
          goto cleanup;
        }

        for (k = 0; k < finSet[i].surface[j].numSection; k++) {
          status = missileDatcom_classifySection(aimInfo, &finSet[i].surface[j].vlmSection[k], &sectyp);
          AIM_STATUS(aimInfo, status);

          if (sectyp0 == mdNODE && sectyp != mdNODE)
            sectyp0 = sectyp;

          if (sectyp0 != sectyp && sectyp != mdNODE) {
            AIM_ERROR(aimInfo, "Fins '%s' has inconsistent sections!", finSet[i].surface[j].name);
            status = CAPS_BADVALUE;
            goto cleanup;
          }
        }
      }

      // set sectyp based on geometry
      if (sectyp0 == mdHEX) {
        AIM_STRDUP(finSet[i].sectyp, "HEX", aimInfo, status);
      } else if (sectyp0 == mdARC) {
        AIM_STRDUP(finSet[i].sectyp, "ARC", aimInfo, status);
      } else if (sectyp0 == mdUSER) {
        AIM_STRDUP(finSet[i].sectyp, "USER", aimInfo, status);
      }

    }
  }

  status = missileDatcom_orderBody(numBodySection, &bodySection);
  AIM_STATUS(aimInfo, status);

  mdInstance->numBodySection = numBodySection;
  mdInstance->bodySection = bodySection;

  mdInstance->numFinSet = numFinSet;
  mdInstance->finSet = finSet;

  status = missileDatcom_dexit( aimInfo, mdInstance );
  AIM_STATUS(aimInfo, status);

  status = CAPS_SUCCESS;

cleanup:
  if (status != CAPS_SUCCESS)  {

    mdInstance->numBodySection = 0;
    mdInstance->bodySection = NULL;

    mdInstance->numFinSet = 0;
    mdInstance->finSet = NULL;

    if (bodySection != NULL) {
      for (i = 0; i < numBodySection; i++) destroy_mdBodySectionStruct(&bodySection[i]);
      /*@-kepttrans@*/
      AIM_FREE(bodySection);
      /*@+kepttrans@*/
    }

    if (finSet != NULL) {
      for (i = 0; i < numFinSet; i++) destroy_mdFinSetStruct(&finSet[i]);
      /*@-kepttrans@*/
      AIM_FREE(finSet);
      /*@-kepttrans@*/
    }
  }

  AIM_FREE(finBody);
  AIM_FREE(surface);

  // Attribute to index map
  (void ) destroy_mapAttrToIndexStruct(&groupMap);

  return status;
}


/*===========================================================================*/
static int
missileDatcom_writeReference(void *aimInfo, FILE *fp,
                             int numBody, ego bodies[],
                             double scaleFactor,
                             capsValue *aimInputs)
{
  int status;
  int i;

  // Attribute data
  int atype, alen;
  const int *ints;
  const double *reals;
  const char *string;

  capsValue *input;

  double Sref=CAPSMAGIC, Cref=CAPSMAGIC, Bref=CAPSMAGIC, Xref=0, Zref=0;

  // Loop over bodies and look for reference quantity attributes
  for (i=0; i < numBody; i++) {
    status = EG_attributeRet(bodies[i], "capsReferenceArea", &atype, &alen, &ints, &reals, &string);
    if (status == EGADS_SUCCESS) {

      if (atype == ATTRREAL && alen == 1) {
        Sref = (double) reals[0];
      } else {
        printf("capsReferenceArea should be followed by a single real value!\n");
        status = CAPS_BADVALUE;
        goto cleanup;
      }
    }

    status = EG_attributeRet(bodies[i], "capsReferenceChord", &atype, &alen, &ints, &reals, &string);
    if (status == EGADS_SUCCESS) {

      if (atype == ATTRREAL && alen == 1) {
        Cref = (double) reals[0];
      } else {
        printf("capsReferenceChord should be followed by a single real value!\n");
        status = CAPS_BADVALUE;
        goto cleanup;
      }
    }

    status = EG_attributeRet(bodies[i], "capsReferenceSpan", &atype, &alen, &ints, &reals, &string);
    if (status == EGADS_SUCCESS) {

      if (atype == ATTRREAL && alen == 1) {
        Bref = (double) reals[0];
      } else {
        printf("capsReferenceSpan should be followed by a single real value!\n");
        status = CAPS_BADVALUE;
        goto cleanup;
      }
    }

    status = EG_attributeRet(bodies[i], "capsReferenceX", &atype, &alen, &ints, &reals, &string);
    if (status == EGADS_SUCCESS) {

      if (atype == ATTRREAL && alen == 1) {
        Xref = (double) reals[0];
      } else {
        printf("capsReferenceX should be followed by a single real value!\n");
        status = CAPS_BADVALUE;
        goto cleanup;
      }
    }

#if 0
    status = EG_attributeRet(bodies[i], "capsReferenceY", &atype, &alen, &ints, &reals, &string);
    if (status == EGADS_SUCCESS) {

      if (atype == ATTRREAL && alen == 1) {
        Yref = (double) reals[0];
      } else {
        printf("capsReferenceY should be followed by a single real value!\n");
        status = CAPS_BADVALUE;
        goto cleanup;
      }
    }
#endif

    status = EG_attributeRet(bodies[i], "capsReferenceZ", &atype, &alen, &ints, &reals, &string);
    if (status == EGADS_SUCCESS){

      if (atype == ATTRREAL && alen == 1) {
        Zref = (double) reals[0];
      } else {
        printf("capsReferenceZ should be followed by a single real value!\n");
        status = CAPS_BADVALUE;
        goto cleanup;
      }
    }
  }

  fprintf(fp, "$REFQ\n");

  if (Sref != CAPSMAGIC) fprintf(fp, " SREF=%.16f,\n", Sref*scaleFactor*scaleFactor); //Reference area

  if (Sref != CAPSMAGIC) fprintf(fp, " LREF=%.16f,\n", Cref*scaleFactor); //Longitudinal reference length

  if (Sref != CAPSMAGIC) fprintf(fp, " LATREF=%.16f,\n", Bref*scaleFactor); //Lateral reference length
  if (Sref != CAPSMAGIC) fprintf(fp, " XCG=%.16f,\n", Xref*scaleFactor);
  if (Sref != CAPSMAGIC) fprintf(fp, " ZCG=%.16f,\n", Zref*scaleFactor);

  input = &aimInputs[aim_getIndex(aimInfo, "BL_Type", ANALYSISIN)-1];
  if (input->nullVal == NotNull) {
    fprintf(fp, " BLAYER=%s,\n", input->vals.string);
  }

  input = &aimInputs[inSurface_Roughness-1];
  if (input->nullVal == NotNull) {

    fprintf(fp, " ROUGH=%.16f,\n", input->vals.real);

  } else {

    input = &aimInputs[inRoughness_Rating-1];
    if (input->nullVal == NotNull) {
      fprintf(fp, " RHR=%.16f,\n", input->vals.real);
    }
  }

  fprintf(fp, " SCALE=%.16f,\n", 1.0);

  fprintf(fp, "$END\n");
  status = CAPS_SUCCESS;

cleanup:

  return status;
}


/*===========================================================================*/
static int
missileDatcom_hexSection(void *aimInfo, vlmSectionStruct *vlmSection,
                         double *zupper, double *zlower,
                         double *lmaxu, double *lmaxl,
                         double *lflatu, double *lflatl)
{
  int status;
  int i, d;
  int oclass, mtype, nchild, *sens;
  double data[4];
  double X[3];
  double xdot[3], ydot[3];
  double XY[7][2];
  double trange[2];

  int numNode=0, numEdge = 0, numChildren = 0;
  ego *enodes = NULL, *eedges = NULL, ecurve, *echild=NULL, *echildren, ref, nodeTE = NULL;
  ego nodeLoopOrder[7];

  int edgeIndex, *edgeLoopOrder=NULL, *edgeLoopSense = NULL;

  // This assumes the airfoil is generated with biconvex

  *zupper = *zlower = 0;
  *lmaxu  = *lmaxl  = 0;
  *lflatu = *lflatl = 0;

  if (aim_isNodeBody(vlmSection->ebody, X) == CAPS_SUCCESS) {
    return CAPS_SUCCESS;
  }

  status = EG_getBodyTopos(vlmSection->ebody, NULL, NODE, &numNode, &enodes);
  AIM_STATUS(aimInfo, status);

  status = EG_getBodyTopos(vlmSection->ebody, NULL, EDGE, &numEdge, &eedges);
  AIM_STATUS(aimInfo, status);

  if (numEdge != 4 && numEdge != 6) {
    AIM_ERROR(aimInfo, "HEX section with %d Edges!", numEdge);
    status = EGADS_TOPOERR;
    goto cleanup;
  }
  AIM_NOTNULL(enodes, aimInfo, status);
  AIM_NOTNULL(eedges, aimInfo, status);

  for (i = 0; i < numEdge; i++) {
    status = EG_getTopology(eedges[i], &ecurve, &oclass, &mtype, data, &nchild, &echild, &sens);
    AIM_STATUS(aimInfo, status);
    if (ecurve->mtype != LINE) {
      AIM_ERROR(aimInfo, "HEX section is not Line!");
      status = EGADS_GEOMERR;
      goto cleanup;
    }
  }

  xdot[0] = vlmSection->xyzTE[0] - vlmSection->xyzLE[0];
  xdot[1] = vlmSection->xyzTE[1] - vlmSection->xyzLE[1];
  xdot[2] = vlmSection->xyzTE[2] - vlmSection->xyzLE[2];

  vlmSection->chord = sqrt(dot_DoubleVal(xdot,xdot));

  xdot[0] /= vlmSection->chord;
  xdot[1] /= vlmSection->chord;
  xdot[2] /= vlmSection->chord;

  // cross with section PLANE normal to get perpendicular vector in the PLANE
  cross_DoubleVal(vlmSection->normal, xdot, ydot);

  // Get the loop edge ordering so it starts at the trailing edge NODE
  status = vlm_secOrderEdges(aimInfo,
                             numNode, enodes,
                             numEdge, eedges,
                             vlmSection->ebody, vlmSection->teObj,
                             &edgeLoopOrder, &edgeLoopSense, &nodeTE);
  AIM_STATUS(aimInfo, status);
  AIM_NOTNULL(edgeLoopOrder, aimInfo, status);
  AIM_NOTNULL(edgeLoopSense, aimInfo, status);

  for (i = 0; i < numEdge; i++) {
    edgeIndex = edgeLoopOrder[i] - 1; // -1 indexing

    // Get t- range for edge
    status = EG_getTopology(eedges[edgeIndex], &ref, &oclass, &mtype, trange, &numChildren, &echildren, &sens);
    if (mtype == DEGENERATE) continue;
    AIM_STATUS(aimInfo, status);

    if (edgeLoopSense[i] == SFORWARD) {
      nodeLoopOrder[i  ] = echildren[0];
      nodeLoopOrder[i+1] = echildren[1];
    } else {
      nodeLoopOrder[i  ] = echildren[1];
      nodeLoopOrder[i+1] = echildren[0];
    }
  }

  for (i = 0; i < numNode; i++) {
    status = EG_evaluate(nodeLoopOrder[i], NULL, X);
    AIM_STATUS(aimInfo, status);

    for (d = 0; d < 3; d++)
      X[d] -= vlmSection->xyzLE[d];

    XY[i][0] = dot_DoubleVal(xdot,X)/vlmSection->chord;
    XY[i][1] = dot_DoubleVal(ydot,X)/vlmSection->chord;

    *zupper = MAX(*zupper,  XY[i][1]);
    *zlower = MAX(*zlower, -XY[i][1]);
  }

  if (numNode == 4) {
    *lmaxu  = XY[1][0];
    *lmaxl  = XY[3][0];
  } else {
    *lmaxu  = XY[2][0];
    *lmaxl  = XY[4][0];

    *lflatu = XY[1][0] - *lmaxu;
    *lflatl = XY[5][0] - *lmaxl;
  }

  status = CAPS_SUCCESS;
cleanup:
  AIM_FREE(enodes);
  AIM_FREE(eedges);
  AIM_FREE(edgeLoopOrder);
  AIM_FREE(edgeLoopSense);
  return status;
}


/*===========================================================================*/
static int
missileDatcom_arcSection(void *aimInfo, vlmSectionStruct *vlmSection, double *zupper, double *zlower)
{
  int status;
  int i, d;
  int oclass, mtype, nchild, *sens;
  double data[4];
  double mid[3], z[4];

  int *ivec=NULL;
  double *rvec=NULL;

  int numEdge = 0;
  ego *eedges = NULL, ecurve[2], *echild=NULL, ref;

  // This assumes the airfoil is generated with biconvex

  if (aim_isNodeBody(vlmSection->ebody, data) == CAPS_SUCCESS) {
    *zupper = 0;
    *zlower = 0;
    return CAPS_SUCCESS;
  }

  status = EG_getBodyTopos(vlmSection->ebody, NULL, EDGE, &numEdge, &eedges);
  AIM_STATUS(aimInfo, status);

  if (numEdge != 2) {
    AIM_ERROR(aimInfo, "ARC section with %d Edges!", numEdge);
    status = EGADS_TOPOERR;
    goto cleanup;
  }
  AIM_NOTNULL(eedges, aimInfo, status);

  for (i = 0; i < numEdge; i++) {
    status = EG_getTopology(eedges[i], &ecurve[i], &oclass, &mtype, data, &nchild, &echild, &sens);
    AIM_STATUS(aimInfo, status);
    if (ecurve[i]->mtype != CIRCLE) {
      AIM_ERROR(aimInfo, "ARC section is not Circular!");
      status = EGADS_GEOMERR;
      goto cleanup;
    }
  }

  for (d = 0; d < 3; d++)
    mid[d] = (vlmSection->xyzLE[d] + vlmSection->xyzTE[d])/2;

  status = EG_getGeometry(ecurve[0], &oclass, &mtype, &ref, &ivec, &rvec);
  AIM_STATUS(aimInfo, status);
  AIM_NOTNULL(rvec, aimInfo, status);

  z[0] = sqrt(pow( rvec[3]*rvec[9]+rvec[0]-mid[0],2) +
              pow( rvec[4]*rvec[9]+rvec[1]-mid[1],2) +
              pow( rvec[5]*rvec[9]+rvec[2]-mid[2],2))/vlmSection->chord;

  z[1] = sqrt(pow(-rvec[3]*rvec[9]+rvec[0]-mid[0],2) +
              pow(-rvec[4]*rvec[9]+rvec[1]-mid[1],2) +
              pow(-rvec[5]*rvec[9]+rvec[2]-mid[2],2))/vlmSection->chord;

  z[2] = sqrt(pow( rvec[6]*rvec[9]+rvec[0]-mid[0],2) +
              pow( rvec[7]*rvec[9]+rvec[1]-mid[1],2) +
              pow( rvec[8]*rvec[9]+rvec[2]-mid[2],2))/vlmSection->chord;

  z[3] = sqrt(pow(-rvec[6]*rvec[9]+rvec[0]-mid[0],2) +
              pow(-rvec[7]*rvec[9]+rvec[1]-mid[1],2) +
              pow(-rvec[8]*rvec[9]+rvec[2]-mid[2],2))/vlmSection->chord;

  *zupper = MIN(MIN(MIN(z[0], z[1]), z[2]), z[3]);

  AIM_FREE(ivec);
  AIM_FREE(rvec);
  status = EG_getGeometry(ecurve[1], &oclass, &mtype, &ref, &ivec, &rvec);
  AIM_STATUS(aimInfo, status);
  AIM_NOTNULL(rvec, aimInfo, status);

  z[0] = sqrt(pow( rvec[3]*rvec[9]+rvec[0]-mid[0],2) +
              pow( rvec[4]*rvec[9]+rvec[1]-mid[1],2) +
              pow( rvec[5]*rvec[9]+rvec[2]-mid[2],2))/vlmSection->chord;

  z[1] = sqrt(pow(-rvec[3]*rvec[9]+rvec[0]-mid[0],2) +
              pow(-rvec[4]*rvec[9]+rvec[1]-mid[1],2) +
              pow(-rvec[5]*rvec[9]+rvec[2]-mid[2],2))/vlmSection->chord;

  z[2] = sqrt(pow( rvec[6]*rvec[9]+rvec[0]-mid[0],2) +
              pow( rvec[7]*rvec[9]+rvec[1]-mid[1],2) +
              pow( rvec[8]*rvec[9]+rvec[2]-mid[2],2))/vlmSection->chord;

  z[3] = sqrt(pow(-rvec[6]*rvec[9]+rvec[0]-mid[0],2) +
              pow(-rvec[7]*rvec[9]+rvec[1]-mid[1],2) +
              pow(-rvec[8]*rvec[9]+rvec[2]-mid[2],2))/vlmSection->chord;

  *zlower = MIN(MIN(MIN(z[0], z[1]), z[2]), z[3]);

  status = CAPS_SUCCESS;
cleanup:
  AIM_FREE(ivec);
  AIM_FREE(rvec);
  AIM_FREE(eedges);
  return status;
}


/*===========================================================================*/
static int
missileDatcom_writeFinSet(void *aimInfo,
                          FILE *fp,
                          double scaleFactor,
                          int numPointAirfoil,
                          int iFinSet,
                          const mdFinSetStruct *finSet)
{
  int status;
  int i, j, k;
  int zeroIndex, sectionIndex;
  double *xCoord=NULL, *yUpperCoord=NULL, *yLowerCoord=NULL;

  double span, phif, gam;
  double unit1[3]={0.0, 0.0, 0.0}, unit2[3]={0.0, 0.0, 0.0}, mag;
  double *xyzLE1, *xyzLE0;
  int npanel = 0;

  double *zupper=NULL, *zlower=NULL;
  double *lmaxu=NULL , *lmaxl=NULL;
  double *lflatu=NULL, *lflatl=NULL;


  fprintf(fp, "$FINSET%d\n", iFinSet+1);

  zeroIndex = 0;
  for (k = 0; k < finSet->surface[0].numSection; k++) {
    if (finSet->surface[0].vlmSection[k].sectionIndex == 0) {
      zeroIndex = k;
      break;
    }
  }

  for (j = 0; j < finSet->surface[0].numSection; j++) {

    sectionIndex = 0;
    for (k = 0; k < finSet->surface[0].numSection; k++) {
      if (finSet->surface[0].vlmSection[k].sectionIndex == j) {
        sectionIndex = k;
        break;
      }
    }

    if (j == 0) {
      fprintf(fp, " SSPAN=%.16f,\n", 0.0);
    } else {
      xyzLE0 = finSet->surface[0].vlmSection[zeroIndex].xyzLE;
      xyzLE1 = finSet->surface[0].vlmSection[sectionIndex].xyzLE;

      span = sqrt(pow(xyzLE1[1]-xyzLE0[1], 2) +
                  pow(xyzLE1[2]-xyzLE0[2], 2));
      fprintf(fp, "  %.16f,\n", span*scaleFactor);
    }
  }

  for (j = 0; j < finSet->surface[0].numSection; j++) {

    sectionIndex = 0;
    for (k = 0; k < finSet->surface[0].numSection; k++) {
      if (finSet->surface[0].vlmSection[k].sectionIndex == j) {
        sectionIndex = k;
        break;
      }
    }

    if (j == 0) {
      fprintf(fp, " CHORD=%.16f,\n", finSet->surface[0].vlmSection[sectionIndex].chord*scaleFactor);
    } else {
      fprintf(fp, "  %.16f,\n", finSet->surface[0].vlmSection[sectionIndex].chord*scaleFactor);
    }
  }

  for (j = 0; j < finSet->surface[0].numSection; j++) {

    sectionIndex = -1;
    for (k = 0; k < finSet->surface[0].numSection; k++) {
      if (finSet->surface[0].vlmSection[k].sectionIndex == j) {
        sectionIndex = k;
        break;
      }
    }
    xyzLE1 = finSet->surface[0].vlmSection[sectionIndex].xyzLE;

    if (j == 0) {
      fprintf(fp, " XLE=%.16f,\n", xyzLE1[0]*scaleFactor);
    } else {
      fprintf(fp, "  %.16f,\n", xyzLE1[0]*scaleFactor);
    }
  }


  fprintf(fp, " SECTYP=%s,\n",finSet->sectyp);

  if (strcmp(finSet->sectyp, "USER") == 0) {

    // User specified only supports the single root airfoil
    status = vlm_getSectionCoordX(aimInfo,
                                  &finSet->surface[0].vlmSection[zeroIndex],
                                  1,               // cosine spaceing
                                  (int) true,      // Normalize by chord (true/false)
                                  (int) false,     // Leave airfoil rotated (true/false)
                                  numPointAirfoil, // Max number of points in airfoil
                                  &xCoord,
                                  &yUpperCoord,
                                  &yLowerCoord);
    AIM_STATUS(aimInfo, status);
    AIM_NOTNULL(xCoord, aimInfo, status);
    AIM_NOTNULL(yUpperCoord, aimInfo, status);
    AIM_NOTNULL(yLowerCoord, aimInfo, status);


    for (j = 0; j < numPointAirfoil; j++) {

      if (j == 0) {
        fprintf(fp, " XCORD=%.16f,\n", xCoord[j]);
      } else {
        fprintf(fp, "  %.16f,\n", xCoord[j]);
      }

    }

    for (j = 0; j < numPointAirfoil; j++) {

      if (j == 0) {
        fprintf(fp, " YUPPER=%.16f,\n", yUpperCoord[j]);
      } else {
        fprintf(fp, "  %.16f,\n", yUpperCoord[j]);
      }
    }

    for (j = 0; j < numPointAirfoil; j++) {

      if (j == 0) {
        fprintf(fp, " YLOWER=%.16f,\n", yLowerCoord[j]);
      } else {
        fprintf(fp, "  %.16f,\n", yLowerCoord[j]);
      }
    }

  } else if (strcmp(finSet->sectyp, "ARC") == 0) {

    AIM_ALLOC(zupper, finSet->surface[0].numSection, double, aimInfo, status);
    AIM_ALLOC(zlower, finSet->surface[0].numSection, double, aimInfo, status);

    for (j = 0; j < finSet->surface[0].numSection; j++) {

      sectionIndex = 0;
      for (k = 0; k < finSet->surface[0].numSection; k++) {
        if (finSet->surface[0].vlmSection[k].sectionIndex == j) {
          sectionIndex = k;
          break;
        }
      }
      status = missileDatcom_arcSection(aimInfo, &finSet->surface[0].vlmSection[sectionIndex], &zupper[j], &zlower[j]);
      AIM_STATUS(aimInfo, status);
    }

    for (j = 0; j < finSet->surface[0].numSection; j++) {
      if (j == 0) {
        fprintf(fp, " ZUPPER=%.16f,\n", zupper[j]);
      } else {
        fprintf(fp, "  %.16f,\n", zupper[j]);
      }
    }

    for (j = 0; j < finSet->surface[0].numSection; j++) {
      if (j == 0) {
        fprintf(fp, " ZLOWER=%.16f,\n", zlower[j]);
      } else {
        fprintf(fp, "  %.16f,\n", zlower[j]);
      }
    }

  } else if (strcmp(finSet->sectyp, "HEX") == 0) {

    AIM_ALLOC(zupper, finSet->surface[0].numSection, double, aimInfo, status);
    AIM_ALLOC(zlower, finSet->surface[0].numSection, double, aimInfo, status);
    AIM_ALLOC(lmaxu , finSet->surface[0].numSection, double, aimInfo, status);
    AIM_ALLOC(lmaxl , finSet->surface[0].numSection, double, aimInfo, status);
    AIM_ALLOC(lflatu, finSet->surface[0].numSection, double, aimInfo, status);
    AIM_ALLOC(lflatl, finSet->surface[0].numSection, double, aimInfo, status);

    for (j = 0; j < finSet->surface[0].numSection; j++) {

      sectionIndex = 0;
      for (k = 0; k < finSet->surface[0].numSection; k++) {
        if (finSet->surface[0].vlmSection[k].sectionIndex == j) {
          sectionIndex = k;
          break;
        }
      }
      status = missileDatcom_hexSection(aimInfo, &finSet->surface[0].vlmSection[sectionIndex],
                                        &zupper[j], &zlower[j],
                                        &lmaxu[j] , &lmaxl[j] ,
                                        &lflatu[j], &lflatl[j]);
      AIM_STATUS(aimInfo, status);
    }

    for (j = 0; j < finSet->surface[0].numSection; j++) {
      if (j == 0) {
        fprintf(fp, " ZUPPER=%.16f,\n", zupper[j]*scaleFactor);
      } else {
        fprintf(fp, "  %.16f,\n", zupper[j]*scaleFactor);
      }
    }

    for (j = 0; j < finSet->surface[0].numSection; j++) {
      if (j == 0) {
        fprintf(fp, " ZLOWER=%.16f,\n", zlower[j]*scaleFactor);
      } else {
        fprintf(fp, "  %.16f,\n", zlower[j]*scaleFactor);
      }
    }

    for (j = 0; j < finSet->surface[0].numSection; j++) {
      if (j == 0) {
        fprintf(fp, " LMAXU=%.16f,\n", lmaxu[j]);
      } else {
        fprintf(fp, "  %.16f,\n", lmaxu[j]);
      }
    }

    for (j = 0; j < finSet->surface[0].numSection; j++) {
      if (j == 0) {
        fprintf(fp, " LMAXL=%.16f,\n", lmaxl[j]);
      } else {
        fprintf(fp, "  %.16f,\n", lmaxl[j]);
      }
    }

    for (j = 0; j < finSet->surface[0].numSection; j++) {
      if (j == 0) {
        fprintf(fp, " LFLATU=%.16f,\n", lflatu[j]);
      } else {
        fprintf(fp, "  %.16f,\n", lflatu[j]);
      }
    }

    for (j = 0; j < finSet->surface[0].numSection; j++) {
      if (j == 0) {
        fprintf(fp, " LFLATL=%.16f,\n", lflatl[j]);
      } else {
        fprintf(fp, "  %.16f,\n", lflatl[j]);
      }
    }
  }



  if (finSet->numPhif > 0 || finSet->numGam > 0) {
    npanel = MAX(finSet->numPhif, finSet->numGam);
  } else {
    npanel = finSet->numSurf;
  }
  fprintf(fp, " NPANEL=%.1f,\n", (double)npanel);

  if (finSet->numPhif == 0 && finSet->numGam == 0) {
    for (i = 0; i < finSet->numSurf; i++) {

      zeroIndex = 0;
      for (k = 0; k < finSet->surface[i].numSection; k++) {
        if (finSet->surface[i].vlmSection[k].sectionIndex == 0) {
          zeroIndex = k;
          break;
        }
      }
      xyzLE0 = finSet->surface[i].vlmSection[zeroIndex].xyzLE;

      unit1[0] = xyzLE0[1] - 0;
      unit1[1] = xyzLE0[2] - 0;
      unit1[2] = 0;

      mag = sqrt(dot_DoubleVal(unit1, unit1));

      unit1[0] /= mag;
      unit1[1] /= mag;
      unit1[2] /= mag;

      // Z-up y - to the right
      phif = atan2(unit1[0], unit1[1]) * 180/PI;
      if (phif < 0) phif += 360;

      if (i == 0) {
        fprintf(fp, " PHIF=%.16f,\n", phif);
      } else {
        fprintf(fp, "  %.16f,\n", phif);
      }
    }
  } else {
    for (i = 0; i < finSet->numPhif; i++) {
      if (i == 0) {
        fprintf(fp, " PHIF=%.16f,\n", finSet->phif[i]);
      } else {
        fprintf(fp, "  %.16f,\n", finSet->phif[i]);
      }
    }
  }


  if (finSet->numPhif == 0 && finSet->numGam == 0) {
    for (i = 0; i < finSet->numSurf; i++) {

      zeroIndex = 0;
      for (k = 0; k < finSet->surface[i].numSection; k++) {
        if (finSet->surface[i].vlmSection[k].sectionIndex == 0) {
          zeroIndex = k;
          break;
        }
      }

      sectionIndex = 0;
      for (k = 0; k < finSet->surface[i].numSection; k++) {
        if (finSet->surface[i].vlmSection[k].sectionIndex == 1) {
          sectionIndex = k;
          break;
        }
      }
      xyzLE0 = finSet->surface[i].vlmSection[zeroIndex].xyzLE;
      xyzLE1 = finSet->surface[i].vlmSection[sectionIndex].xyzLE;

      unit1[0] = xyzLE0[1] - 0;
      unit1[1] = xyzLE0[2] - 0;
      unit1[2] = 0;

      mag = sqrt(dot_DoubleVal(unit1, unit1));

      unit1[0] /= mag;
      unit1[1] /= mag;
      unit1[2] /= mag;


      unit2[0] = xyzLE1[1]-xyzLE0[1];
      unit2[1] = xyzLE1[2]-xyzLE0[2];
      unit2[2] = 0;

      mag = sqrt(dot_DoubleVal(unit2, unit2));

      unit2[0] /= mag;
      unit2[1] /= mag;
      unit2[2] /= mag;

      mag = dot_DoubleVal(unit1, unit2);
      mag = MAX(-1.0,MIN(1.0,mag));
      gam = acos(mag)*180/PI;

      if (i == 0) {
        fprintf(fp, " GAM=%.16f,\n", gam);
      } else {
        fprintf(fp, "  %.16f,\n", gam);
      }
    }
  } else {
    for (i = 0; i < finSet->numGam; i++) {
      if (i == 0) {
        fprintf(fp, " GAM=%.16f,\n", finSet->gam[i]);
      } else {
        fprintf(fp, "  %.16f,\n", finSet->gam[i]);
      }
    }
  }

  fprintf(fp, "$END\n");

  // Write the NACA control card
  if (strcmp(finSet->sectyp, "NACA") == 0) {
    fprintf(fp, "NACA-%d%s\n", iFinSet+1, finSet->naca);
  }

  status = CAPS_SUCCESS;

cleanup:

  AIM_FREE(xCoord);
  AIM_FREE(yUpperCoord);
  AIM_FREE(yLowerCoord);


  AIM_FREE(zupper);
  AIM_FREE(zlower);
  AIM_FREE(lmaxu);
  AIM_FREE(lmaxl);
  AIM_FREE(lflatu);
  AIM_FREE(lflatl);

  return status;

}


/*===========================================================================*/
static int
missileDatcom_writeAxiBod1( void *aimInfo, FILE *fp,
                            double scaleFactor,
                            const aimStorage *mdInstance)
{
  int status;
  double lnose, dnose;
  double lcentr, dcentr;
  double laft, daft;

  const int numSection = mdInstance->numBodySection;
  const mdBodySectionStruct *section = mdInstance->bodySection;

  if (numSection < 3 || numSection > 4) {
    AIM_ERROR(aimInfo, "AXIBOD Option 1 must have 3 or 4 sections! numSection = %d", numSection);
    AIM_ADDLINE(aimInfo, "Use AXIBOD Option 2 for more general bodies");
    return CAPS_BADTYPE;
  }

  lnose = (section[1].xLoc - section[0].xLoc)*scaleFactor;
  dnose =  section[1].width*scaleFactor;

  lcentr = (section[2].xLoc - section[1].xLoc)*scaleFactor;
  dcentr =  section[2].width*scaleFactor;

  if (numSection == 4) {
    laft = (section[3].xLoc - section[2].xLoc)*scaleFactor;
    daft =  section[3].width*scaleFactor;
  }

  fprintf(fp, "$AXIBOD\n");

  fprintf(fp, " X0=%.16f,\n", section[0].xLoc);
  fprintf(fp, " TNOSE=%s,\n", mdInstance->BODtnose);
  fprintf(fp, " POWER=%.16f,\n", mdInstance->BODpower);
  fprintf(fp, " LNOSE=%.16f,\n", lnose);
  fprintf(fp, " DNOSE=%.16f,\n", dnose);
  fprintf(fp, " BNOSE=%.16f,\n", mdInstance->BODbnose);
  fprintf(fp, " TRUNC=%s,\n", mdInstance->BODtrunc == (int)true ? ".TRUE." : ".FALSE.");
  fprintf(fp, " LCENTR=%.16f,\n", lcentr);
  fprintf(fp, " DCENTR=%.16f,\n", dcentr);

  if (numSection == 4) {
    fprintf(fp, " TAFT=%s,\n", mdInstance->BODtaft);
    fprintf(fp, " LAFT=%.16f,\n", laft);
    fprintf(fp, " DAFT=%.16f,\n", daft);
  }

  if (mdInstance->dexit >= 0) {
    fprintf(fp, " DEXIT=%.16f,\n", mdInstance->dexit);
  }

  fprintf(fp, "$END\n");


  status = CAPS_SUCCESS;

//cleanup:
  return status;
}


/*===========================================================================*/
static int
missileDatcom_writeAxiBod2( /*@unused@*/ void *aimInfo, FILE *fp,
                            double scaleFactor,
                            const aimStorage *mdInstance)
{
  int status;
  int i, discon = 0;

  const int numSection = mdInstance->numBodySection;
  const mdBodySectionStruct *section = mdInstance->bodySection;

  fprintf(fp, "$AXIBOD\n");

  fprintf(fp, " X0=%.16f,\n", section[0].xLoc*scaleFactor);
  fprintf(fp, " NX=%.1f,\n", (double)(numSection-1));
  for (i = 1; i < numSection; i++) {
    if (i == 1) {
      fprintf(fp, " X=%.16f,\n", section[i].xLoc*scaleFactor);
    } else {
      fprintf(fp, "  %.16f,\n", section[i].xLoc*scaleFactor);
    }
  }

  for (i = 1; i < numSection; i++) {
    if (i == 1) {
      fprintf(fp, " R=%.16f,\n", section[i].width*scaleFactor);
    } else {
      fprintf(fp, "  %.16f,\n", section[i].width*scaleFactor);
    }
  }

  for (i = 1; i < numSection; i++) {
    if (i == 1) {
      fprintf(fp, " Z=%.16f,\n", section[i].zOffset*scaleFactor);
    } else {
      fprintf(fp, "  %.16f,\n", section[i].zOffset*scaleFactor);
    }
  }

  for (i = 0; i < numSection; i++) {
    if (section[i].discon == 1) {
      if (discon == 0) {
        fprintf(fp, " DISCON=%d.,\n", i+1);
        discon = 1;
      } else {
        fprintf(fp, "        %d.,\n", i+1);
      }
    }
  }

  if (mdInstance->dexit >= 0) {
    fprintf(fp, " DEXIT=%.16f,\n", mdInstance->dexit*scaleFactor);
  }

  fprintf(fp, "$END\n");


  status = CAPS_SUCCESS;

//cleanup:
  return status;
}


/*===========================================================================*/
static int
missileDatcom_writeEllBod1( /*@unused@*/ void *aimInfo, FILE *fp,
                            double scaleFactor,
                            const aimStorage *mdInstance)
{
  int status;
  double lnose, wnose, enose;
  double lcentr, wcentr, ecentr;
  double laft, waft, eaft;

  const int numSection = mdInstance->numBodySection;
  const mdBodySectionStruct *section = mdInstance->bodySection;

  if (numSection < 3 || numSection > 4) {
    AIM_ERROR(aimInfo, "ELLBOD Option 1 must have 3 or 4 sections! numSection = %d", numSection);
    AIM_ADDLINE(aimInfo, "Use ELLBOD Option 2 for more general bodies");
    return CAPS_BADTYPE;
  }

  lnose = (section[1].xLoc - section[0].xLoc)*scaleFactor;
  wnose =  section[1].width*scaleFactor;
  enose =  section[1].height/section[1].width;

  lcentr = (section[2].xLoc - section[1].xLoc)*scaleFactor;
  wcentr =  section[2].width*scaleFactor;
  ecentr =  section[2].height/section[2].width;

  if (numSection == 4) {
    laft = (section[3].xLoc - section[2].xLoc)*scaleFactor;
    waft =  section[3].width*scaleFactor;
    eaft =  section[3].height/section[3].width;
  }

  fprintf(fp, "$ELLBOD\n");

  fprintf(fp, " X0=%.16f,\n", section[0].xLoc);
  fprintf(fp, " TNOSE=%s,\n", mdInstance->BODtnose);
  fprintf(fp, " POWER=%.16f,\n", mdInstance->BODpower);
  fprintf(fp, " LNOSE=%.16f,\n", lnose);
  fprintf(fp, " WNOSE=%.16f,\n", wnose);
  fprintf(fp, " ENOSE=%.16f,\n", enose);
  fprintf(fp, " BNOSE=%.16f,\n", mdInstance->BODbnose);
  fprintf(fp, " TRUNC=%s,\n", mdInstance->BODtrunc == (int)true ? ".TRUE." : ".FALSE.");
  fprintf(fp, " LCENTR=%.16f,\n", lcentr);
  fprintf(fp, " WCENTR=%.16f,\n", wcentr);
  fprintf(fp, " ECENTR=%.16f,\n", ecentr);

  if (numSection == 4) {
    fprintf(fp, " TAFT=%s,\n", mdInstance->BODtaft);
    fprintf(fp, " LAFT=%.16f,\n", laft);
    fprintf(fp, " WAFT=%.16f,\n", waft);
    fprintf(fp, " EAFT=%.16f,\n", eaft);
  }

  if (mdInstance->dexit >= 0) {
    fprintf(fp, " DEXIT=%.16f,\n", mdInstance->dexit);
  }

  fprintf(fp, "$END\n");

  status = CAPS_SUCCESS;

//cleanup:
  return status;
}


/*===========================================================================*/
static int
missileDatcom_writeEllBod2( /*@unused@*/ void *aimInfo, FILE *fp,
                            double scaleFactor,
                            const aimStorage *mdInstance)
{
  int status;
  int i, discon = 0;

  const int numSection = mdInstance->numBodySection;
  const mdBodySectionStruct *section = mdInstance->bodySection;

  fprintf(fp, "$ELLBOD\n");

  fprintf(fp, " X0=%.16f,\n", section[0].xLoc*scaleFactor);
  fprintf(fp, " NX=%.1f,\n", (double)(numSection-1));
  for (i = 1; i < numSection; i++) {
    if (i == 1) {
      fprintf(fp, " X=%.16f,\n", section[i].xLoc*scaleFactor);
    } else {
      fprintf(fp, "  %.16f,\n", section[i].xLoc*scaleFactor);
    }
  }

  for (i = 1; i < numSection; i++) {
    if (i == 1) {
      fprintf(fp, " W=%.16f,\n", section[i].width*scaleFactor);
    } else {
      fprintf(fp, "  %.16f,\n", section[i].width*scaleFactor);
    }
  }

  for (i = 1; i < numSection; i++) {
    if (i == 1) {
      fprintf(fp, " H=%.16f,\n", section[i].height*scaleFactor);
    } else {
      fprintf(fp, "  %.16f,\n", section[i].height*scaleFactor);
    }
  }

  for (i = 0; i < numSection; i++) {
    if (section[i].discon == 1) {
      if (discon == 0) {
        fprintf(fp, " DISCON=%d.,\n", i+1);
        discon = 1;
      } else {
        fprintf(fp, "        %d.,\n", i+1);
      }
    }
  }

  if (mdInstance->dexit >= 0) {
    fprintf(fp, " DEXIT=%.16f,\n", mdInstance->dexit*scaleFactor);
  }

  fprintf(fp, "$END\n");

  status = CAPS_SUCCESS;

//cleanup:
  return status;
}


/*===========================================================================*/
static int
missileDatcom_writeFlightCon(void *aimInfo, FILE *fp, capsValue *aimInputs)
{
  int status=0;
  int i;
  int nmach=0;
  capsValue *input;

  fprintf(fp, "$FLTCON\n");

  input = &aimInputs[aim_getIndex(aimInfo, "Alpha", ANALYSISIN)-1];
  if (input->nullVal == NotNull) {
    fprintf(fp, " NALPHA=%.1f,\n", (double)input->length);

    if  (input->length > 1) {
      fprintf(fp, " ALPHA=%.16f,\n", input->vals.reals[0]);
      for (i = 1; i <  input->length; i++){
        fprintf(fp, "  %.16f,\n",  input->vals.reals[i]);
      }
    } else {
      fprintf(fp, " ALPHA=%.16f,\n", input->vals.real);
    }
  }


  input = &aimInputs[aim_getIndex(aimInfo, "Beta", ANALYSISIN)-1];
  fprintf(fp, " BETA=%.16f,\n", input->vals.real);


  nmach = MAX(nmach, aimInputs[inMach-1    ].nullVal == NotNull ? aimInputs[inMach-1    ].length : 0);
  nmach = MAX(nmach, aimInputs[inAltitude-1].nullVal == NotNull ? aimInputs[inAltitude-1].length : 0);
  nmach = MAX(nmach, aimInputs[inRen-1     ].nullVal == NotNull ? aimInputs[inRen-1     ].length : 0);
  nmach = MAX(nmach, aimInputs[inVinf-1    ].nullVal == NotNull ? aimInputs[inVinf-1    ].length : 0);
  nmach = MAX(nmach, aimInputs[inTinf-1    ].nullVal == NotNull ? aimInputs[inTinf-1    ].length : 0);
  nmach = MAX(nmach, aimInputs[inPinf-1    ].nullVal == NotNull ? aimInputs[inPinf-1    ].length : 0);

  fprintf(fp, " NMACH=%.1f,\n", (double)nmach);

  input = &aimInputs[inMach-1];
  if (input->nullVal == NotNull) {
    if  (input->length > 1) {
      fprintf(fp, " MACH=%.16f,\n", input->vals.reals[0]);
      for (i = 1; i <  input->length; i++){
        fprintf(fp, "  %.16f,\n",  input->vals.reals[i]);
      }
    } else {
      fprintf(fp, " MACH=%.16f,\n", input->vals.real);
    }
  }

  input = &aimInputs[inAltitude-1];
  if (input->nullVal == NotNull) {
    if  (input->length > 1) {
      fprintf(fp, " ALT=%.16f,\n", input->vals.reals[0]);
      for (i = 1; i <  input->length; i++){
        fprintf(fp, "  %.16f,\n",  input->vals.reals[i]);
      }
    } else {
      fprintf(fp, " ALT=%.16f,\n", input->vals.real);
    }
  }

  input = &aimInputs[inRen-1];
  if (input->nullVal == NotNull) {
    if  (input->length > 1) {
      fprintf(fp, " REN=%.16f,\n", input->vals.reals[0]);
      for (i = 1; i <  input->length; i++){
        fprintf(fp, "  %.16f,\n",  input->vals.reals[i]);
      }
    } else {
      fprintf(fp, " REN=%.16f,\n", input->vals.real);
    }
  }

  input = &aimInputs[inVinf-1];
  if (input->nullVal == NotNull) {
    if  (input->length > 1) {
      fprintf(fp, " VINF=%.16f,\n", input->vals.reals[0]);
      for (i = 1; i <  input->length; i++){
        fprintf(fp, "  %.16f,\n",  input->vals.reals[i]);
      }
    } else {
      fprintf(fp, " VINF=%.16f,\n", input->vals.real);
    }
  }

  input = &aimInputs[inTinf-1];
  if (input->nullVal == NotNull) {
    if  (input->length > 1) {
      fprintf(fp, " TINF=%.16f,\n", input->vals.reals[0]);
      for (i = 1; i <  input->length; i++){
        fprintf(fp, "  %.16f,\n",  input->vals.reals[i]);
      }
    } else {
      fprintf(fp, " TINF=%.16f,\n", input->vals.real);
    }
  }

  input = &aimInputs[inPinf-1];
  if (input->nullVal == NotNull) {
    if  (input->length > 1) {
      fprintf(fp, " PINF=%.16f,\n", input->vals.reals[0]);
      for (i = 1; i <  input->length; i++){
        fprintf(fp, "  %.16f,\n",  input->vals.reals[i]);
      }
    } else {
      fprintf(fp, " PINF=%.16f,\n", input->vals.real);
    }
  }

  fprintf(fp, "$END\n");

  status = CAPS_SUCCESS;

//cleanup:

  return status;
}


/*===========================================================================*/
static int
missileDatcom_writeBod(void *aimInfo, FILE *fp, double scaleFactor, const aimStorage *mdInstance)
{
  int status;

  if (mdInstance->BODtype == AXIBOD) { // AxiBod

    if (mdInstance->BODoption == 1) {

      status =  missileDatcom_writeAxiBod1(aimInfo, fp, scaleFactor, mdInstance);
      AIM_STATUS(aimInfo, status);

    } else if (mdInstance->BODoption == 2) {

      status =  missileDatcom_writeAxiBod2(aimInfo, fp, scaleFactor, mdInstance);
      AIM_STATUS(aimInfo, status);

    } else {
      AIM_ERROR(aimInfo, "Developer error unknown BODoption = %d", mdInstance->BODoption);
      status = CAPS_NOTIMPLEMENT;
      goto cleanup;
    }

  } else if (mdInstance->BODtype == ELLBOD) { // Ellipse

    if (mdInstance->BODoption == 1) {

      status =  missileDatcom_writeEllBod1(aimInfo, fp, scaleFactor, mdInstance);
      AIM_STATUS(aimInfo, status);

    } else if (mdInstance->BODoption == 2) {

      status =  missileDatcom_writeEllBod2(aimInfo, fp, scaleFactor, mdInstance);
      AIM_STATUS(aimInfo, status);
    } else {
      AIM_ERROR(aimInfo, "Developer error unknown BODoption = %d", mdInstance->BODoption);
      status = CAPS_NOTIMPLEMENT;
      goto cleanup;
    }

  } else {
    AIM_ERROR(aimInfo, "Developer error unknown BODtype = %d", mdInstance->BODtype);
    status = CAPS_NOTIMPLEMENT;
    goto cleanup;
  }

  status = CAPS_SUCCESS;

cleanup:

  return status;
}


/* ********************** Exposed AIM Functions ***************************** */

int aimInitialize(int inst, /*@unused@*/ const char *unitSys, void *aimInfo,
                  /*@unused@*/ void **instStore, /*@unused@*/ int *major,
                  /*@unused@*/ int *minor, int *nIn, int *nOut,
                  int *nFields, char ***fnames, int **franks, int **fInOut)

{
  int status; // Function status return
  const char *keyWord;
  char *keyValue = NULL;
  cfdUnitsStruct *units=NULL;

  aimStorage *mdInstance=NULL;

#ifdef DEBUG
  printf("\n missileDatcomAIM/aimInitialize   ngIn = %d!\n", ngIn);
#endif

  /* specify the number of analysis input and out "parameters" */
  *nIn     = NUMINPUT;
  *nOut    = NUMOUTPUT;
  if (inst == -1) return CAPS_SUCCESS;

  /* specify the field variables this analysis can generate and consume */
  *nFields = 0;
  *fnames  = NULL;
  *franks  = NULL;
  *fInOut  = NULL;

  // Allocate mdInstance
  AIM_ALLOC(mdInstance, 1, aimStorage, aimInfo, status);
  *instStore = mdInstance;

  // Initiate storage
  status = initiate_aimStorage(mdInstance);
  AIM_STATUS(aimInfo, status);


  /*! \page aimUnitsMissileDatcom AIM Units
   *  A unit system may be optionally specified during AIM instance initiation. If
   *  a unit system is provided, all AIM  input values which have associated units must be specified as well.
   *  If no unit system is used, AIM inputs, which otherwise would require units, will be assumed
   *  unit consistent. Missile Datcom only supports modifying the length unit, which may be set with a JSON string:
   *  unitSys = "{"length": "m"}"
   */
  if (unitSys != NULL) {
    units = &mdInstance->units;

    // Do we have a json string?
    if (strncmp( unitSys, "{", 1) != 0) {
      AIM_ERROR(aimInfo, "unitSys ('%s') is expected to be a JSON string dictionary", unitSys);
      return CAPS_BADVALUE;
    }

    /*! \page aimUnitsMissileDatcom
     *  <ul>
     *  <li> <B>length = "None"</B> </li> <br>
     *  Length units - must be one of "in", "ft", "cm", "m"
     *  </ul>
     */
    keyWord = "length";
    status  = search_jsonDictionary(unitSys, keyWord, &keyValue);
    if (status == CAPS_SUCCESS) {
      AIM_NOTNULL(keyValue, aimInfo, status);
      mdInstance->dim = string_removeQuotation(keyValue);
      AIM_NOTNULL(mdInstance->dim, aimInfo, status);
      string_toUpperCase(mdInstance->dim);

      if (strcmp(mdInstance->dim, "IN") == 0 ||
          strcmp(mdInstance->dim, "FT") == 0) {
        AIM_STRDUP(units->length, "ft", aimInfo, status);
      } else if (strcmp(mdInstance->dim, "CM") == 0 ||
                 strcmp(mdInstance->dim, "M" ) == 0) {
        AIM_STRDUP(units->length, "m", aimInfo, status);
      } else {
        AIM_ERROR(aimInfo, "'length' = %s must be one of: 'in', 'ft', 'cm', 'm'", keyValue);
        status = CAPS_BADVALUE;
        goto cleanup;
      }

      AIM_FREE(keyValue);
    } else {
      AIM_ERROR(aimInfo, "unitSys ('%s') does not contain '%s'", unitSys, keyWord);
      status = CAPS_BADVALUE;
      goto cleanup;
    }

    // set remaining units
    AIM_STRDUP(units->time, "s", aimInfo, status);
    if (strcmp(mdInstance->dim, "IN") == 0 ||
        strcmp(mdInstance->dim, "FT") == 0) {
      AIM_STRDUP(units->mass, "lb", aimInfo, status);
      AIM_STRDUP(units->temperature, "Rankine", aimInfo, status);
    } else if (strcmp(mdInstance->dim, "CM") == 0 ||
               strcmp(mdInstance->dim, "M" ) == 0) {
      AIM_STRDUP(units->mass, "kg", aimInfo, status);
      AIM_STRDUP(units->temperature, "Kelvin", aimInfo, status);
    }

    status = cfd_cfdDerivedUnits(aimInfo, units);
    AIM_STATUS(aimInfo, status);
  }

  status = CAPS_SUCCESS;
cleanup:
  AIM_FREE(keyValue);

  return status;
}

// ********************** AIM Function Break *****************************
int aimInputs(/*@unused@*/ void *instStore, /*@unused@*/ void *aimInfo,
              int index, char **ainame, capsValue *defval)
{
  int status = CAPS_SUCCESS;
  /*! \page aimInputsMissileDatcom AIM Inputs
   * The following list outlines the Missile Datcom inputs along with their default values available
   * through the AIM interface.
   */

  aimStorage *mdInstance;
  cfdUnitsStruct *units=NULL;

#ifdef DEBUG
  printf(" missileDatcomAIM/aimInputs instance = %d  index = %d!\n", iIndex, index);
#endif

  *ainame = NULL;


  mdInstance = (aimStorage *) instStore;

  if (mdInstance != NULL) units = &mdInstance->units;

  // Missile Datcom Inputs
  if (index == inMissileDATCOM) {

    *ainame               = EG_strdup("missileDatcom");
    defval->type          = String;
    AIM_STRDUP(defval->vals.string, "misdat", aimInfo, status);

    /*! \page aimInputsMissileDatcom
     * - <B> missileDatcom = "misdat" </B><br>
     *  File name of missileDATCOM executable
     */

  } else if (index == inBODoption) {

    *ainame               = EG_strdup("BODoption");
    defval->type          = Integer;
    defval->nullVal       = IsNull;
    defval->vals.integer  = 0;

    /*! \page aimInputsMissileDatcom
     * - <B> BODoption = int (1 or 2)</B><br>
     * Generate AXIBOD/ELLBOD Option 1 or Option 2 geometry
     */

  } else if (index == inBODtnose) {

    *ainame               = EG_strdup("BODtnose");
    defval->type          = String;
    AIM_STRDUP(defval->vals.string, "OGIVE", aimInfo, status);

    /*! \page aimInputsMissileDatcom
     * - <B> BODtnose = "OGIVE"</B><br>
     * Nose shape:<br>
     *  "CONICAL" or "CONE" (cone)<br>
     *  "OGIVE" (tangent ogive)
     *  "POWER" (power law)
     *  "HAACK" (L-V constrained)
     *  "KARMAN" (L-D constrained)
     */

  } else if (index == inBODpower) {

    *ainame               = EG_strdup("BODpower");
    defval->type          = Double;
    defval->vals.real     = 0;

    /*! \page aimInputsMissileDatcom
     * - <B> BODpower = 0</B><br>
     * Exponent, n, for nose power law shape: $(r/R)=(x/L)^n$
     */

  } else if (index == inBODbnose) {

    *ainame               = EG_strdup("BODbnose");
    defval->type          = Double;
    defval->vals.real     = 0;

    /*! \page aimInputsMissileDatcom
     * - <B> BODbnose = 0</B><br>
     * Nose bluntness radius or radius of truncation
     */

  } else if (index == inBODtrunc) {

    *ainame               = EG_strdup("BODtrunc");
    defval->type          = Boolean;
    defval->vals.integer  = (int)false;

    /*! \page aimInputsMissileDatcom
     * - <B> BODtrunc = False</B><br>
     * Truncation flag: True if nose if truncated, False is nose is not truncated
     */

  } else if (index == inBODtaft) {

    *ainame               = EG_strdup("BODtaft");
    defval->type          = String;
    AIM_STRDUP(defval->vals.string, "CONICAL", aimInfo, status);

    /*! \page aimInputsMissileDatcom
     * - <B> BODtaft = "CONICAL"</B><br>
     * Afterbody shape:<br>
     *  "CONICAL" or "CONE" (cone)<br>
     *  "OGIVE" (tangent ogive)
     */

  } else if (index == inFinSet) {

    *ainame               = EG_strdup("FinSet");
    defval->type          = Tuple;
    defval->dim           = Vector;
    defval->lfixed        = Change;
    defval->nullVal       = IsNull;
    defval->vals.tuple    = NULL;

    /*! \page aimInputsMissileDatcom
     * - <B> FinSet = NULL</B><br>
     * Tuple used to input FinSet information, see \ref mdFinSet for additional details.
     */

  } else if (index == inMach) {

    *ainame               = EG_strdup("Mach");
    defval->type          = Double;
    defval->lfixed        = Change;
    defval->sfixed        = Fixed;
    defval->nullVal       = IsNull;
    defval->dim           = Vector;

    /*! \page aimInputsMissileDatcom
     * - <B> Mach = double </B> <br> OR
     * - <B> Mach = [double, ... , double] </B> <br>
     *  Mach number.
     */

  } else if (index == inRen) {

    *ainame               = EG_strdup("Ren");
    defval->type          = Double;
    defval->lfixed        = Change;
    defval->sfixed        = Fixed;
    defval->nullVal       = IsNull;
    defval->dim           = Vector;

    /*! \page aimInputsMissileDatcom
     * - <B> Ren = double </B> <br> OR
     * - <B> Ren = [double, ... , double] </B> <br>
     *  Reynolds numbers per unit length
     */

    if (units != NULL && units->length != NULL) {
      status = aim_unitInvert(aimInfo, units->length, &defval->units);
      AIM_STATUS(aimInfo, status);
    }

  } else if (index == inVinf) {

    *ainame               = EG_strdup("Vinf");
    defval->type          = Double;
    defval->lfixed        = Change;
    defval->sfixed        = Fixed;
    defval->nullVal       = IsNull;
    defval->dim           = Vector;

    /*! \page aimInputsMissileDatcom
     * - <B> Vinf = double </B> <br> OR
     * - <B> Vinf = [double, ... , double] </B> <br>
     *  Freestream velocities
     */

    if (units != NULL && units->length != NULL) {
      AIM_STRDUP(defval->units, units->speed, aimInfo, status);
    }

  } else if (index == inTinf) {

    *ainame               = EG_strdup("Tinf");
    defval->type          = Double;
    defval->lfixed        = Change;
    defval->sfixed        = Fixed;
    defval->nullVal       = IsNull;
    defval->dim           = Vector;

    /*! \page aimInputsMissileDatcom
     * - <B> Tinf = double </B> <br> OR
     * - <B> Tinf = [double, ... , double] </B> <br>
     *  Freestream static temperatures
     */

    if (units != NULL && units->length != NULL) {
      AIM_STRDUP(defval->units, units->temperature, aimInfo, status);
    }

  } else if (index == inPinf) {

    *ainame               = EG_strdup("Pinf");
    defval->type          = Double;
    defval->lfixed        = Change;
    defval->sfixed        = Fixed;
    defval->nullVal       = IsNull;
    defval->dim           = Vector;

    /*! \page aimInputsMissileDatcom
     * - <B> Pinf = double </B> <br> OR
     * - <B> Pinf = [double, ... , double] </B> <br>
     *  Freestream static pressures
     */

    if (units != NULL && units->length != NULL) {
      AIM_STRDUP(defval->units, units->pressure, aimInfo, status);
    }

  } else if (index == inAlpha) {

    *ainame           = EG_strdup("Alpha");
    defval->type      = Double;
    defval->lfixed    = Change;
    defval->sfixed    = Fixed;
    defval->nullVal   = IsNull;
    defval->dim       = Vector;
    if (units != NULL && units->length != NULL) {
      AIM_STRDUP(defval->units, "degree", aimInfo, status);
    }

    /*! \page aimInputsMissileDatcom
     * - <B> Alpha = double </B> <br> OR
     * - <B> Alpha = [double, ... , double] </B> <br>
     *
     *  Angle of attack [degree].
     */

  } else if (index == inBeta) {

    *ainame           = EG_strdup("Beta");
    defval->type      = Double;
    defval->vals.real = 0.0;
    if (units != NULL && units->length != NULL) {
      AIM_STRDUP(defval->units, "degree", aimInfo, status);
    }

    /*! \page aimInputsMissileDatcom
     * - <B> Beta = 0.0 </B> <br>
     *  Sideslip angle [degree].
     */

  } else if (index == inAltitude) {

    *ainame               = EG_strdup("Altitude");
    defval->type          = Double;
    defval->lfixed        = Change;
    defval->sfixed        = Fixed;
    defval->nullVal       = IsNull;
    defval->dim           = Vector;
    if (units != NULL && units->length != NULL) {
      AIM_STRDUP(defval->units, units->length, aimInfo, status);
    }

    /*! \page aimInputsMissileDatcom
     * - <B> Altitude = double </B> <br> OR
     * - <B> Altitude = [double, ... , double] </B> <br>
     *  Flight altitude.
     */

  } else if (index == inBL_Type) {

    *ainame           = EG_strdup("BL_Type");
    defval->type         = String;
    defval->nullVal      = NotNull;
    defval->vals.string  = EG_strdup("TURB");
    defval->lfixed       = Change;

    /*! \page aimInputsMissileDatcom
     * - <B> BL_Type = "TURB" </B> <br>
     *  Boundary layer type: "TURB" for fully turbulent and "NATURAL" for natural transition
     */

  } else if (index == inSurface_Roughness) {

    *ainame           = EG_strdup("Surface_Roughness");
    defval->type      = Double;
    defval->vals.real = 0.0;
    defval->nullVal   = IsNull;
    if (mdInstance != NULL && mdInstance->dim != NULL) {

      if (strcmp(mdInstance->dim, "IN") == 0 ||
          strcmp(mdInstance->dim, "FT") == 0) {
        AIM_STRDUP(defval->units, "in", aimInfo, status);
      } else if (strcmp(mdInstance->dim, "CM") == 0 ||
                 strcmp(mdInstance->dim, "M" ) == 0) {
        AIM_STRDUP(defval->units, "cm", aimInfo, status);
      }
    }

    /*! \page aimInputsMissileDatcom
     * - <B> Surface_Roughness = None </B> <br>
     *  Surface roughness height. Only use one of "Surface_Roughness" (will take precedence) or "Roughness_Rating".
     */

  } else if (index == inRoughness_Rating) {

    *ainame           = EG_strdup("Roughness_Rating");
    defval->type      = Double;
    defval->vals.real = 0.0;
    defval->nullVal   = IsNull;
    //defval->units     = EG_strdup("degree");

    /*! \page aimInputsMissileDatcom
     * - <B> Roughness_Rating = None </B> <br>
     *  Roughness height rating. Only use one of "Surface_Roughness" (will take precedence) or "Roughness_Rating".
     */
  } else {
    AIM_ERROR(aimInfo, "Developer error: unknown index = %d", index);
    status = CAPS_NOTIMPLEMENT;
    goto cleanup;
  }


  /*! \page aimInputsMissileDatcom
  *
  * Note: Any of the following combinations satisfy the minimum requirements for calculating
  * atmospheric conditions (Mach and Reynolds number):<br>
  * 1. MACH and REN<br>
  * 2. MACH and ALT<br>
  * 3. MACH and TINF and PINF<br>
  * 4. VINF and ALT<br>
  * 5. VINF and TINF and PINF<br>
  * 6. VINF and TINF and REN<br>
  * <br>
  */

  status = CAPS_SUCCESS;

cleanup:

  return status;
}


// ********************** AIM Function Break *****************************
int aimUpdateState(void *instStore, void *aimInfo,
                   capsValue *aimInputs)
{
  // Function return flag
  int status = CAPS_SUCCESS;

  // AIM input bodies
  const char *intent;
  int  numBody;
  ego *bodies = NULL;

  aimStorage *mdInstance = (aimStorage *)instStore;
  AIM_STATUS(aimInfo, status);

  AIM_NOTNULL(aimInputs, aimInfo, status);

  // reset the instance
  destroy_aimStorage(mdInstance);

  // Get AIM bodies
  status = aim_getBodies(aimInfo, &intent, &numBody, &bodies);
  AIM_STATUS(aimInfo, status);

#ifdef DEBUG
  printf(" missileDatcomAIM/aimUpdateState numBody = %d!\n", numBody);
#endif

  if ((numBody <= 0) || (bodies == NULL)) {
    AIM_ERROR(aimInfo, " No Bodies!\n");
    status = CAPS_SOURCEERR;
    goto cleanup;
  }

  // Check for valid FligthCon combinations
  if (!((aimInputs[inMach-1].nullVal == NotNull && aimInputs[inRen-1].nullVal == NotNull && aimInputs[inAltitude-1].nullVal == IsNull  && aimInputs[inVinf-1].nullVal == IsNull  && aimInputs[inTinf-1].nullVal == IsNull  && aimInputs[inPinf-1].nullVal == IsNull ) || // 1. Mach + Ren
        (aimInputs[inMach-1].nullVal == NotNull && aimInputs[inRen-1].nullVal == IsNull  && aimInputs[inAltitude-1].nullVal == NotNull && aimInputs[inVinf-1].nullVal == IsNull  && aimInputs[inTinf-1].nullVal == IsNull  && aimInputs[inPinf-1].nullVal == IsNull ) || // 2. Mach + Alt
        (aimInputs[inMach-1].nullVal == NotNull && aimInputs[inRen-1].nullVal == IsNull  && aimInputs[inAltitude-1].nullVal == IsNull  && aimInputs[inVinf-1].nullVal == IsNull  && aimInputs[inTinf-1].nullVal == NotNull && aimInputs[inPinf-1].nullVal == NotNull) || // 3. Mach + Tinf + Pinf
        (aimInputs[inMach-1].nullVal == IsNull  && aimInputs[inRen-1].nullVal == IsNull  && aimInputs[inAltitude-1].nullVal == NotNull && aimInputs[inVinf-1].nullVal == NotNull && aimInputs[inTinf-1].nullVal == IsNull  && aimInputs[inPinf-1].nullVal == IsNull ) || // 4. Vinf + Alt
        (aimInputs[inMach-1].nullVal == IsNull  && aimInputs[inRen-1].nullVal == IsNull  && aimInputs[inAltitude-1].nullVal == IsNull  && aimInputs[inVinf-1].nullVal == NotNull && aimInputs[inTinf-1].nullVal == NotNull && aimInputs[inPinf-1].nullVal == NotNull) || // 5. Vinf + Tinf + Pinf
        (aimInputs[inMach-1].nullVal == IsNull  && aimInputs[inRen-1].nullVal == NotNull && aimInputs[inAltitude-1].nullVal == IsNull  && aimInputs[inVinf-1].nullVal == NotNull && aimInputs[inTinf-1].nullVal == NotNull && aimInputs[inPinf-1].nullVal == IsNull )    // 6. Vinf + Tinf + Ren
       ) ) {
    AIM_ERROR(aimInfo, "Invalid combination of Mach (%s), Ren (%s), Altitude (%s), Vinf (%s), Pinf (%s), Tinf (%s)",
              aimInputs[inMach-1     ].nullVal == NotNull ? "*" : "",
              aimInputs[inRen-1      ].nullVal == NotNull ? "*" : "",
              aimInputs[inAltitude-1 ].nullVal == NotNull ? "*" : "",
              aimInputs[inVinf-1     ].nullVal == NotNull ? "*" : "",
              aimInputs[inPinf-1     ].nullVal == NotNull ? "*" : "",
              aimInputs[inTinf-1     ].nullVal == NotNull ? "*" : "");
    AIM_ADDLINE(aimInfo, "Valid combinations:");
    AIM_ADDLINE(aimInfo, "\t1. Mach + Ren");
    AIM_ADDLINE(aimInfo, "\t2. Mach + Altitude");
    AIM_ADDLINE(aimInfo, "\t3. Mach + Tinf + Pinf");
    AIM_ADDLINE(aimInfo, "\t4. Vinf + Altitude");
    AIM_ADDLINE(aimInfo, "\t5. Vinf + Tinf + Pinf");
    AIM_ADDLINE(aimInfo, "\t6. Vinf + Tinf + Ren");
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  // 1. Mach + Ren
  if (aimInputs[inMach-1].nullVal == NotNull && aimInputs[inRen-1].nullVal == NotNull &&
      aimInputs[inMach-1].length != aimInputs[inRen-1].length) {
    AIM_ERROR(aimInfo, "Mach (%d) and Ren (%d) must be the same length!",
              aimInputs[inMach-1].length, aimInputs[inRen-1].length);
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  // 2. Mach + Altitude
  if (aimInputs[inMach-1].nullVal == NotNull && aimInputs[inAltitude-1].nullVal == NotNull &&
      aimInputs[inMach-1].length != aimInputs[inAltitude-1].length) {
    AIM_ERROR(aimInfo, "Mach (%d) and Altitude (%d) must be the same length!",
              aimInputs[inMach-1].length, aimInputs[inAltitude-1].length);
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  // 3. Mach + Tinf + Pinf
  if (aimInputs[inMach-1].nullVal == NotNull && aimInputs[inTinf-1].nullVal == NotNull && aimInputs[inPinf-1].nullVal == NotNull &&
      (aimInputs[inMach-1].length != aimInputs[inTinf-1].length ||
       aimInputs[inMach-1].length != aimInputs[inPinf-1].length)) {
    AIM_ERROR(aimInfo, "Mach (%d), Ting (%d), and Pinf(%d) must be the same length!",
              aimInputs[inMach-1].length, aimInputs[inTinf-1].length, aimInputs[inPinf-1].length);
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  // 4. Vinf + Altitude
  if (aimInputs[inVinf-1].nullVal == NotNull && aimInputs[inAltitude-1].nullVal == NotNull &&
      aimInputs[inVinf-1].length != aimInputs[inAltitude-1].length) {
    AIM_ERROR(aimInfo, "Vinf (%d) and Altitude (%d) must be the same length!",
              aimInputs[inVinf-1].length, aimInputs[inAltitude-1].length);
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  // 5. Mach + Tinf + Pinf
  if (aimInputs[inVinf-1].nullVal == NotNull && aimInputs[inTinf-1].nullVal == NotNull && aimInputs[inPinf-1].nullVal == NotNull &&
      (aimInputs[inVinf-1].length != aimInputs[inTinf-1].length ||
       aimInputs[inVinf-1].length != aimInputs[inPinf-1].length)) {
    AIM_ERROR(aimInfo, "Vinf (%d), Ting (%d), and Pinf(%d) must be the same length!",
              aimInputs[inVinf-1].length, aimInputs[inTinf-1].length, aimInputs[inPinf-1].length);
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  // 6. Mach + Tinf + Ren
  if (aimInputs[inVinf-1].nullVal == NotNull && aimInputs[inTinf-1].nullVal == NotNull && aimInputs[inRen-1].nullVal == NotNull &&
      (aimInputs[inVinf-1].length != aimInputs[inTinf-1].length ||
       aimInputs[inVinf-1].length != aimInputs[inRen-1].length)) {
    AIM_ERROR(aimInfo, "Vinf (%d), Ting (%d), and Ren(%d) must be the same length!",
              aimInputs[inVinf-1].length, aimInputs[inTinf-1].length, aimInputs[inRen-1].length);
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  if (aimInputs[inBODoption-1].nullVal == IsNull ||
      !(aimInputs[inBODoption-1].vals.integer == 1 ||
        aimInputs[inBODoption-1].vals.integer == 2)) {
    AIM_ANALYSISIN_ERROR(aimInfo, inBODoption, "Must be specified as 1 or 2");
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  // store away inputs
  mdInstance->BODoption = aimInputs[inBODoption-1].vals.integer;
  AIM_STRDUP(mdInstance->BODtnose, aimInputs[inBODtnose-1].vals.string, aimInfo, status);
  mdInstance->BODpower  = aimInputs[inBODpower-1].vals.real;
  mdInstance->BODbnose  = aimInputs[inBODbnose-1].vals.real;
  mdInstance->BODtrunc  = aimInputs[inBODtrunc-1].vals.integer;
  AIM_STRDUP(mdInstance->BODtaft, aimInputs[inBODtaft-1].vals.string, aimInfo, status);

  string_toUpperCase(mdInstance->BODtnose);
  string_toUpperCase(mdInstance->BODtaft);

  if (!(strcmp(mdInstance->BODtnose, "CONICAL") == 0 ||
        strcmp(mdInstance->BODtnose, "CONE") == 0 ||
        strcmp(mdInstance->BODtnose, "OGIVE") == 0 ||
        strcmp(mdInstance->BODtnose, "POWER") == 0 ||
        strcmp(mdInstance->BODtnose, "HAACK") == 0 ||
        strcmp(mdInstance->BODtnose, "KARMAN") == 0)) {
    AIM_ERROR(aimInfo, "BODtnose = '%s' must be one of:", mdInstance->BODtnose);
    AIM_ADDLINE(aimInfo, "CONICAL");
    AIM_ADDLINE(aimInfo, "CONE");
    AIM_ADDLINE(aimInfo, "OGIVE");
    AIM_ADDLINE(aimInfo, "POWER");
    AIM_ADDLINE(aimInfo, "HAACK");
    AIM_ADDLINE(aimInfo, "KARMAN");
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  if (!(strcmp(mdInstance->BODtaft, "CONICAL") == 0 ||
        strcmp(mdInstance->BODtaft, "CONE") == 0 ||
        strcmp(mdInstance->BODtaft, "OGIVE") == 0)) {
    AIM_ERROR(aimInfo, "BODtaft = '%s' must be one of:", mdInstance->BODtaft);
    AIM_ADDLINE(aimInfo, "CONICAL");
    AIM_ADDLINE(aimInfo, "CONE");
    AIM_ADDLINE(aimInfo, "OGIVE");
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  // Parse bodies
  status = missileDatcom_parseBody(aimInfo, numBody, bodies, mdInstance);
  AIM_STATUS(aimInfo, status);

  // Update FinSets after parsing the bodies
  status = missileDatcom_getFinSet(aimInfo,
                                   aimInputs[inFinSet-1].length,
                                   aimInputs[inFinSet-1].vals.tuple,
                                   &mdInstance->finSetMap,
                                   mdInstance->numFinSet,
                                   mdInstance->finSet);
  AIM_STATUS(aimInfo, status);

  status = CAPS_SUCCESS;
cleanup:
  return status;
}


// ********************** AIM Function Break *****************************
int aimPreAnalysis(const void *instStore, void *aimInfo, capsValue *aimInputs)
{

  // Function return flag
  int status;

  // Data transfer
  //   	int  nrow, ncol, rank;
  //    void *dataTransfer = NULL;
  //    enum capsvType vtype;
  //    char *units = NULL;
  //

  // EGADS return values
  //    int          atype, alen;
  //    const int    *ints;
  //    const char   *string;
  //    const double *reals;

  const char *bodyLunits=NULL;
  double scaleFactor=1;

  const aimStorage *mdInstance = (const aimStorage *)instStore;

  int i;

  // AIM input bodies
  const char *intent;
  int  numBody;
  ego *bodies = NULL;

  //    double scaleFactor = 1.0;

  FILE *fp=NULL;
  const char *inputFile = "for005.dat";

  AIM_NOTNULL(aimInputs, aimInfo, status);

  // Get AIM bodies
  status = aim_getBodies(aimInfo, &intent, &numBody, &bodies);
  AIM_STATUS(aimInfo, status);

#ifdef DEBUG
  printf(" missileDatcomAIM/aimPreAnalysis instance = %d  numBody = %d!\n", iIndex, numBody);
#endif

  if ((numBody <= 0) || (bodies == NULL)) {
    printf(" missileDatcomAIM/aimPreAnalysis No Bodies!\n");
    status =  CAPS_SOURCEERR;
    goto cleanup;
  }

  if (mdInstance->dim != NULL) {
    // Check consistency on body length and get length unit
    status = aim_capsLength(aimInfo, &bodyLunits);
    if (status == CAPS_SUCCESS && bodyLunits != NULL) {
      status = aim_convert(aimInfo, 1, bodyLunits, &scaleFactor, mdInstance->dim, &scaleFactor);
      if (status != CAPS_SUCCESS) {
        AIM_ERROR("Cannot convert capsLength %s to ''!\n", bodyLunits, mdInstance->dim);
        goto cleanup;
      }
    }
  }

  // Write input file
  fp = aim_fopen(aimInfo, inputFile, "w");
  if (fp == NULL) {
    status = CAPS_IOERR;
    goto cleanup;
  }


  fprintf(fp, "CASEID %s\n", "CAPS Case");
  if (mdInstance->dim != NULL)
    fprintf(fp, "DIM %s\n", mdInstance->dim);

  // Write flight conditions
  status = missileDatcom_writeFlightCon(aimInfo, fp, aimInputs);
  AIM_STATUS(aimInfo, status);

  // Write reference conditions
  status = missileDatcom_writeReference(aimInfo, fp, numBody, bodies, scaleFactor, aimInputs);
  AIM_STATUS(aimInfo, status);

  // Write body
  status = missileDatcom_writeBod(aimInfo, fp, scaleFactor, mdInstance);
  AIM_STATUS(aimInfo, status);

  // Write fins
  for (i = 0; i < mdInstance->numFinSet; i++) {
    status = missileDatcom_writeFinSet(aimInfo, fp, scaleFactor,
                                       50, i, &mdInstance->finSet[i]);
    AIM_STATUS(aimInfo, status);
  }

  status = CAPS_SUCCESS;

cleanup:

  if (fp != NULL) fclose(fp);

  return status;
}

// ********************** AIM Function Break *****************************
int aimExecute(/*@unused@*/ const void *instStore, /*@unused@*/ void *aimInfo,
               int *state)
{
  /*! \page aimExecuteMissileDatcom AIM Execution
   *
   * If auto execution is enabled when creating an missileDATCOM AIM,
   * the AIM will execute missileDATCOM just-in-time with the command line:
   *
   * \code{.sh}
   * misdat > missileDatcomOutput.txt
   * \endcode
   *
   * where preAnalysis generated the file for005.dat.
   *
   * The analysis can be also be explicitly executed with caps_execute in the C-API
   * or via Analysis.runAnalysis in the pyCAPS API.
   *
   * Calling preAnalysis and postAnalysis is NOT allowed when auto execution is enabled.
   *
   * Auto execution can also be disabled when creating an missileDATCOM AIM object.
   * In this mode, caps_execute and Analysis.runAnalysis can be used to run the analysis,
   * or missileDatcom can be executed by calling preAnalysis, system call, and posAnalysis as demonstrated
   * below with a pyCAPS example:
   *
   * \code{.py}
   * print ("\n\preAnalysis......")
   * missileDatcom.preAnalysis()
   *
   * print ("\n\nRunning......")
   * missileDatcom.system("missileDatcom > missileDatcomOutput.txt"); # Run via system call
   *
   * print ("\n\postAnalysis......")
   * missileDatcom.postAnalysis()
   * \endcode
   */

  int status;
  capsValue *misdat;
  char command[PATH_MAX];

  status = aim_getValue(aimInfo, inMissileDATCOM, ANALYSISIN, &misdat);
  AIM_STATUS(aimInfo, status);

  snprintf(command, PATH_MAX, "%s > missileDatcomOutput.txt", misdat->vals.string);

  status = aim_system(aimInfo, NULL, command);
  AIM_STATUS(aimInfo, status, "Failed to execute: %s", command);

  status = CAPS_SUCCESS;
  *state = 0;

cleanup:
  return status;
}


// ********************** AIM Function Break *****************************
int aimPostAnalysis(/*@unused@*/ void *instStore, /*@unused@*/ void *aimInfo,
                    /*@unused@*/ int restart, /*@unused@*/ capsValue *inputs)
{
    int status = CAPS_SUCCESS;

//    char aimFile[PATH_MAX];
//
//    aimStorage *mdInstance;
//
//    mdInstance = (aimStorage *) instStore;

    status = CAPS_SUCCESS;

//cleanup:

    return status;
}


// ********************** AIM Function Break *****************************
int aimOutputs(/*@unused@*/ void *instStore, /*@unused@*/ void *aimInfo,
               int index, char **aoname, capsValue *form)
{

  /*! \page aimOutputMissileDatcom AIM Outputs
   * The following list outlines the Missile Datcom outputs available through the AIM interface.
   */

  //     CN Normal Force Coefficient (body axis)
  //     CL Lift Coefficient (wind axis)
  //     CM Pitching Moment Coefficient (body axis)
  //     Xcp Center of Pressure in calibers from the moment reference center
  //     CA Axial Force Coefficient (body axis)
  //     CD Drag Coefficient (wind axis)
  //     CY Side Force Coefficient (body axis)
  //     Cn Yawing Moment Coefficient (body axis)
  //     Cl Rolling Moment Coefficient (body axis)

  // Body control derivatives
  if (index == outCNtot) {
    *aoname = EG_strdup("CNtot");

    /*! \page aimOutputMissileDatcom
     * Forces and moments:
     * - <B> CNtot </B> = Normal Force Coefficient (body axis).
     */
  } else if (index == outCAtot) {
    *aoname = EG_strdup("CAtot");

    /*! \page aimOutputMissileDatcom
     * - <B> CAtot </B> = Axial Force Coefficient (body axis).
     */
  } else if (index == outCYtot) {
    *aoname = EG_strdup("CYtot");

    /*! \page aimOutputMissileDatcom
     * - <B> CYtot </B> = Side Force Coefficient (body axis).
     */
  } else if (index == outCmtot) {
    *aoname = EG_strdup("Cmtot");

    /*! \page aimOutputMissileDatcom
     * - <B> Cmtot </B> = Pitching Moment Coefficient (body axis).
     */
  } else if (index == outCntot) {
    *aoname = EG_strdup("Cntot");

    /*! \page aimOutputMissileDatcom
     * - <B> Cntot </B> = Yawing Moment Coefficient (body axis).
     */
  } else if (index == outCltot) {
    *aoname = EG_strdup("Cltot");

    /*! \page aimOutputMissileDatcom
     * - <B> Cltot </B> = Rolling Moment Coefficient (body axis).
     */
  } else if (index == outCLtot) {
    *aoname = EG_strdup("CLtot");

    /*! \page aimOutputMissileDatcom
     * - <B> CLtot </B> = Lift Coefficient (wind axis).
     */
  } else if (index == outCDtot) {
    *aoname = EG_strdup("CDtot");

    /*! \page aimOutputMissileDatcom
     * - <B> CDtot </B> = Drag Coefficient (wind axis).
     */
  } else if (index == outXcp) {
    *aoname = EG_strdup("Xcp");

    /*! \page aimOutputMissileDatcom
     * - <B> Xcp </B> =  Center of Pressure in calibers from the moment reference center.
     */
  } else if (index == outCNa) { // Alpha stability derivatives
    *aoname = EG_strdup("CNa");

  } else if (index == outCma) {
    *aoname = EG_strdup("Cma");

    /*! \page aimOutputMissileDatcom
     * Derivatives - Alpha:
     * - <B>CNa</B> = Normal force coefficient derivative with angle of attack.
     * - <B>Cma</B> =  Pitching moment coefficient derivative with angle of attack.
     */

  } else if (index == outCYb) { // Beta stability derivatives
    *aoname = EG_strdup("CYb");

  } else if (index == outCnb) {
    *aoname = EG_strdup("Cnb");

  } else if (index == outClb) {
    *aoname = EG_strdup("Clb");

    /*! \page aimOutputMissileDatcom
     * Derivatives - Beta:
     * - <B>CYb</B> = Side force coefficient derivative with sideslip angle.
     * - <B>Cnb</B> = Yawing moment coefficient derivative with sideslip angle (body axis).
     * - <B>Clb</B> = Rolling moment coefficient derivative with sideslip angle (body axis).
     */
  } else if (index == outAlpha) {
    *aoname = EG_strdup("Alpha");

    /*! \page aimOutputMissileDatcom
     * Echo inputs for case specific information:
     * - <B> Alpha </B> =  Angle of attack from output file.
     */
  } else if (index == outMach) {
    *aoname = EG_strdup("Mach");

    /*! \page aimOutputMissileDatcom
     * - <B> Mach </B> =  Mach number from output file.
     */

  } else if (index == outAltitude) {
    *aoname = EG_strdup("Altitude");

    /*! \page aimOutputMissileDatcom
     * - <B> Altitude </B> =  Altitude from output file.
     */
  } else {
    return CAPS_NOTFOUND;
  }

  form->type = Double;
  form->units = NULL;
  form->vals.reals = NULL;
  form->vals.real = 0;

  return CAPS_SUCCESS;
}

static int _csv_getSize(FILE *fp, int *numVar, int *numAlpha, int  *numMach) {
  int status = CAPS_NOTFOUND;

  size_t linecap = 0;
  char *line = NULL, *valstr = NULL;
  char size[12], nvar[12],nalpha[12],nmach[12];

  while (getline(&line, &linecap, fp) >= 0) {
    if (line == NULL) continue;

    valstr = strstr(line, "'SIZES'");

    if (valstr != NULL) {

      sscanf( line, "%s%s%d,%s%d,%s%d,", size,
                                         nvar, numVar,
                                         nalpha, numAlpha,
                                         nmach, numMach);

      //             printf("%s %s %d %s %d %s %d\n",size,
      //                                             nvar, *numVar,
      //                                             nalpha, *numAlpha,
      //                                             nmach, *numMach);
      status = CAPS_SUCCESS;
      break;
    }
  }

  AIM_FREE(line);

  return status;
}

static int _csv_getIndex(FILE *fp, const char *key, int *index) {

  int status = CAPS_NOTFOUND;
  int i;

  size_t linecap = 0;
  char *line = NULL, *valstr = NULL;
  int comma=0;

  while (getline(&line, &linecap, fp) >= 0) {

    if (line == NULL) continue;

    valstr = strstr(line, key);

    if (valstr != NULL) {

      for (i = 0; i <valstr - line;i++ ) {
        if (line[i] == ',') comma +=1;
      }
      *index = comma+1;
      status = CAPS_SUCCESS;
      break;
    }
  }

  AIM_FREE(line);

  return status;
}

static int _read_Data(void *aimInfo,
                      const char *file, const char *key, int *length, double *data[])
{
  int status;
  int i, j, k; //Indexing

  size_t     linecap = 0;
  char       *line = NULL;
  FILE       *fp = NULL;

  int numVar, numAlpha, numMach, index;
  double temp;

  *data = NULL;

  if (file == NULL) return CAPS_NULLVALUE;
  if (key == NULL) return CAPS_NULLVALUE;

  fp = aim_fopen(aimInfo, file, "r");
  if (fp == NULL) {
    return CAPS_IOERR;
  }

  // Only getting the first case - NOT FIT FOR TRIM
  status = _csv_getSize(fp, &numVar, &numAlpha, &numMach);
  if (status == CAPS_NOTFOUND) {
    AIM_ERROR(aimInfo, "Failed to find 'SIZES' in '%s'!", file);
    goto cleanup;
  }
  AIM_STATUS(aimInfo, status);

  if (numVar == 0 || numAlpha == 0 || numMach == 0) {
    AIM_ERROR(aimInfo, "None of (NumVar = %d, NumAlpha = %d, and NumMach = %d) may be ZERO!\n", numVar, numAlpha, numMach);
    status = CAPS_NOTFOUND;
    goto cleanup;
  }
  status = _csv_getIndex(fp, key, &index);
  if (status == CAPS_NOTFOUND) {
    AIM_ERROR(aimInfo, "Failed to find '%s' in '%s'!", key, file);
    goto cleanup;
  }
  AIM_STATUS(aimInfo, status);

  //    printf("%d %d %d %d\n", numVar, numAlpha, numMach, index);

  AIM_ALLOC(*(data), numAlpha*numMach, double, aimInfo, status);

  *length = numAlpha*numMach;
  //    printf("LENGTH = %d\n", *length);

  // Should be on the data line
  i = 0;
  while (getline(&line, &linecap, fp) >= 0) {
    AIM_NOTNULL(line, aimInfo, status);
    j = 1;
    for (k = 0; k < strlen(line); k++) {
      if (line[k] == ',') {

        if (j == index-1) {
          break;
        } else {
          j+=1;
          continue;
        }
      } else {
        continue;
      }
    }
    sscanf( &line[k+1], "%lf", &temp);
    //sscanf( &line[(index-1)*13 + 1], "%lf", &temp); // Only works on 13 character formating

    (*data)[i] = temp;
    i +=1;
    if (i >= *length) break;
  }

  status = CAPS_SUCCESS;

cleanup:

  // Restore the path we came in with
  if (fp != NULL) fclose(fp);

  if (line != NULL) free(line);
  return status;
}

// ********************** AIM Function Break *****************************
// Calculate Missile Datcom output
int aimCalcOutput(/*@unused@*/ void *instStore, void *aimInfo, int index,
                  capsValue *val)
{

  int status = CAPS_NOTFOUND;
  int i;

  const char *fileToOpen = "for042.csv";
  int length=0;
  double *data=NULL;
  const char *key = NULL;

  switch (index) {
  case outCNtot:
    key = "'CN'"; break;
  case outCAtot:
    key = "'CA'"; break;
  case outCYtot:
    key = "'CY'"; break;
  case outCmtot:
    key = "'CM'"; break;
  case outCntot:
    key = "'CLN'"; break;
  case outCltot:
    key = "'CLL'"; break;
  case outCLtot:
    key = "'CL'"; break;
  case outCDtot:
    key = "'CD'"; break;
  case outXcp:
    key = "'X-C.P.'"; break;
  case outCNa:
    key = "'CNA'"; break;
  case outCma:
    key = "'CMA'"; break;
  case outCYb:
    key = "'CYB'"; break;
  case outCnb:
    key = "'CLNB'"; break;
  case outClb:
    key = "'CLLB'"; break;
  case outAlpha:
    key = "'ALPHA'"; break;
  case outMach:
    key = "'MACH'"; break;
  case outAltitude:
    key = "'ALT'"; break;
  default:
    status = CAPS_NOTFOUND;
    AIM_STATUS(aimInfo, status);
  }
  AIM_NOTNULL(key, aimInfo, status);

  status = _read_Data(aimInfo, fileToOpen, key, &length, &data);
  AIM_STATUS(aimInfo, status);
  AIM_NOTNULL(data, aimInfo, status);

  val->length = length;
  val->ncol = 1;
  val->nrow = length;

  if (val->length == 1) {
    val->dim = 0;
    val->vals.real = data[0];
  } else {
    val->dim = 1;
    AIM_ALLOC(val->vals.reals, val->length, double, aimInfo, status);
    for (i = 0; i < length; i++) val->vals.reals[i] = data[i];
  }

  status = CAPS_SUCCESS;

cleanup:

  AIM_FREE(data);

  return status;
}

// ********************** AIM Function Break *****************************
void aimCleanup(void *instStore)
{
  aimStorage *mdInstance = (aimStorage *)instStore;

#ifdef DEBUG
  printf(" missileDatcomAIM/aimClenup!\n");
#endif

  // Clean up mdInstance data
  destroy_aimStorage(mdInstance);
  AIM_FREE(mdInstance);
}

