# Import other need modules

import unittest
import os
import glob
import shutil

# Import pyCAPS class file
import pyCAPS
from operator import __truediv__

# Name of the FlightStream executable
FlightStream_exe = "FlightStream"

# Helper function to check if an executable exists
def which(program):
    import os
    def is_exe(fpath):
        return os.path.isfile(fpath) and os.access(fpath, os.X_OK)

    fpath, fname = os.path.split(program)
    if fpath:
        if is_exe(program):
            return program
    else:
        for path in os.environ["PATH"].split(os.pathsep):
            exe_file = os.path.join(path, program)
            if is_exe(exe_file):
                return exe_file

    return None


class TestFlightStream(unittest.TestCase):

    @classmethod
    def setUpClass(cls):
        if which(FlightStream_exe) == None:
            return

        # Create working directory variable
        cls.problemName = "workDir_fsAnalysisTest"

        cls.cleanUp()

        # Initialize Problem object
        cls.capsProblem = pyCAPS.Problem(cls.problemName,
                                       capsFile=os.path.join("..","csmData","avlSections.csm"),
                                       outLevel=0)

    @classmethod
    def tearDownClass(cls):
        if which(FlightStream_exe) == None:
            return

        del cls.capsProblem
        cls.cleanUp()

    @classmethod
    def cleanUp(cls):

        # Remove problemName directories
        dirs = glob.glob( cls.problemName + '*')
        for dir in dirs:
            if os.path.isdir(dir):
                shutil.rmtree(dir)

    # This is executed prior to each test
    def setUp(self):
        if which(FlightStream_exe) == None:
            self.skipTest("No FlightStream executable")

#==============================================================================
    def test_inputs(self):
        # Define units
        deg  = pyCAPS.Unit("degree")
        Pa   = pyCAPS.Unit("Pa")
        ft   = pyCAPS.Unit("feet")
        m    = pyCAPS.Unit("m")
        kg   = pyCAPS.Unit("kg")
        K    = pyCAPS.Unit("Kelvin")
        s    = pyCAPS.Unit("s")

        # Load flightstream aim
        flightstream = self.capsProblem.analysis.create(aim = "flightstreamAIM")

        # Set inputs
        flightstream.input.FlightStream  = FlightStream_exe
        flightstream.input.Mach  = 0.5
        flightstream.input.Alpha = 1.0 * deg
        flightstream.input.Beta  = 0.0 * deg

        flightstream.input.Mach  = None
        flightstream.input.Alpha = None
        flightstream.input.Beta  = None

        flightstream.input.SweepRangeMach = [0.5, 0.7, 0.1] #Start, end, delta
        flightstream.input.SweepRangeAlpha = [1.0, 3.0, 1.0] * deg
        flightstream.input.SweepRangeBeta = [-1.0, -3.0, -1.0] * deg
        flightstream.input.SweepClear_Solution_After_Each_Run = True
        flightstream.input.SweepReference_Velocity_Equals_Freestream = True
        flightstream.input.SweepExport_Surface_Data = "VTK"

        flightstream.input.SweepRangeMach = None
        flightstream.input.SweepRangeAlpha = None
        flightstream.input.SweepRangeBeta = None

        flightstream.input.Altitude = 1000 * ft
        flightstream.input.Altitude = None
        flightstream.input.Fluid_Properties = {"density": 0.072 * kg/m**3,
                                               "pressure": 5692.95 * Pa,
                                               "sonic_velocity": 295.06 * m/s,
                                               "temperature": 216.67 * K,
                                               "viscosity": 0.0000142 * Pa*s}
        flightstream.input.Solver_Model = "SUBSONIC_PRANDTL_GLAUERT"
        flightstream.input.Stabilization = 2.0
        flightstream.input.Wall_Collision_Avoidance = True

        flightstream.input.CCS_GeneralSurface = {"Fuselage": {}}
        flightstream.input.CCS_LiftingSurface = {"Wing": {}, "Htail": {}, "Vtail":{}}
        AileronLeft = {"deflectionAngle" : 25.0 * deg, "slotGap" : 0.005} # Large angle: no-wake termination nodes
        AileronRite = {"deflectionAngle" : -5.0 * deg, "slotGap" : 0.005} # Small angle: wake termination nodes
        flightstream.input.CCS_Control = {"AileronLeft": AileronLeft,
                                          "AileronRite": AileronRite,
                                          "Stabilizer" : {"deflectionAngle":15*deg},
                                          "Rudder"     : {"deflectionAngle":-10*deg}}
        flightstream.input.CCS_Revolution = {"NacelleR": {}}
        flightstream.input.CCS_Annular = {"NacelleL": {}}

        flightstream.input.ReferenceChord = 1*m
        flightstream.input.ReferenceSpan = 2*m
        flightstream.input.ReferenceArea = 3*m**2
        flightstream.input.ReferenceX = 0.5*m
        flightstream.input.ReferenceY = 0.1*m
        flightstream.input.ReferenceZ = 0.2*m
        flightstream.input.ReferenceVelocity = 0.9*m/s
        flightstream.input.Pressure_Scale_Factor = 2
        flightstream.input.ConvergenceTol = 1e-6
        flightstream.input.Export_Solver_Analysis = ["Tecplot", "VTK"]
        flightstream.input.FlightScriptPre = ["# One comment", "# Two!"]
        flightstream.input.FlightScriptPost = ["# Three comment", "# Four!"]
        flightstream.input.Hidden = True
        flightstream.input.Max_Parallel_Threads = 16
        flightstream.input.SaveFSM = True
        flightstream.input.Mesh_Morph = True


#==============================================================================
    def test_alpha_custom_increment(self):
        # Define units
        deg  = pyCAPS.Unit("degree")
        ft   = pyCAPS.Unit("ft")
        m    = pyCAPS.Unit("m")
        s    = pyCAPS.Unit("s")

        # Load flightstream aim
        flightstream = self.capsProblem.analysis.create(aim = "flightstreamAIM")

        # Set new Mach/Alt parameters
        flightstream.input.FlightStream  = FlightStream_exe
        flightstream.input.Mach  = 0.5
        flightstream.input.Alpha = 1.0 * deg
        flightstream.input.Beta  = 0.0 * deg
        flightstream.input.Altitude = 1000 * ft
        flightstream.input.Solver_Model = "SUBSONIC_PRANDTL_GLAUERT"
        flightstream.input.ReferenceVelocity = 295.06 * m/s
        flightstream.input.Export_Solver_Analysis = ["Tecplot", "VTK"]

        wing = {"groupName"         : "Wing",
                "numChord"          : 48,
                "growthTypeChord"   : 2,
                "growthRateChord"   : 1.2,
                "periodicityChord"  : 3,
                "numSpanPerSection" : 22,
                "growthTypeSpan"   : 4,
                "growthRateSpan"   : 1.3,
                "periodicitySpan"  : 1}

        flightstream.input.CCS_LiftingSurface = {"Wing": wing}

        Alpha   = [0.0, 3.0, 9.0] * deg
        CxTrue  = [ 0.0125,  0.0088, -0.0043]
        CyTrue  = [ 0.0069,  0.0089,  0.0129]
        CzTrue  = [ 0.0893,  0.1518,  0.2739]
        CLTrue  = [ 0.0893,  0.1513,  0.2716]
        CDiTrue = [ 0.0108,  0.0150,  0.0369]
        CDoTrue = [ 0.0017,  0.0017,  0.0017]
        CMxTrue = [ 0.0462,  0.0456,  0.0430]
        CMyTrue = [-0.0315, -0.0311, -0.0304]
        CMzTrue = [ 0.0002, -0.0019, -0.0060]

        for i in range(0,len(Alpha)):
            # Set custom AoA
            #print("Alpha = ", Alpha[i])
            flightstream.input.Alpha = Alpha[i]

            # Retrieve results
            Cx  = flightstream.output.Cx
            Cy  = flightstream.output.Cy
            Cz  = flightstream.output.Cz
            CL  = flightstream.output.CL
            CDi = flightstream.output.CDi
            CDo = flightstream.output.CDo
            CMx = flightstream.output.CMx
            CMy = flightstream.output.CMy
            CMz = flightstream.output.CMz

            #print("Cx  = ", Cx )
            #print("Cy  = ", Cy )
            #print("Cz  = ", Cz )
            #print("CL  = ", CL )
            #print("CDi = ", CDi)
            #print("CDo = ", CDo)
            #print("CMx = ", CMx)
            #print("CMy = ", CMy)
            #print("CMz = ", CMz)
            ## Get all dynamic ouputs
            #for key in flightstream.dynout.keys():
            #    print(str(key) + " = " + str(flightstream.dynout[key].value))

            self.assertAlmostEqual(CxTrue[i] , Cx , 4)
            self.assertAlmostEqual(CyTrue[i] , Cy , 4)
            self.assertAlmostEqual(CzTrue[i] , Cz , 4)
            self.assertAlmostEqual(CLTrue[i] , CL , 4)
            self.assertAlmostEqual(CDiTrue[i], CDi, 4)
            self.assertAlmostEqual(CDoTrue[i], CDo, 4)
            self.assertAlmostEqual(CMxTrue[i], CMx, 4)
            self.assertAlmostEqual(CMyTrue[i], CMy, 4)
            self.assertAlmostEqual(CMzTrue[i], CMz, 4)


#==============================================================================
    def test_sweep(self):
        # Define units
        deg  = pyCAPS.Unit("degree")
        ft   = pyCAPS.Unit("ft")
        m    = pyCAPS.Unit("m")
        s    = pyCAPS.Unit("s")

        # Load flightstream aim
        flightstream = self.capsProblem.analysis.create(aim = "flightstreamAIM")

        # Set new Mach/Alt parameters
        flightstream.input.FlightStream  = FlightStream_exe
        Mach  = 0.5
        Alpha = 1.0 * deg
        Beta  = 0.0 * deg
        flightstream.input.Altitude = 1000 * ft
        flightstream.input.Solver_Model = "SUBSONIC_PRANDTL_GLAUERT"
        flightstream.input.ReferenceVelocity = 295.06 * m/s
        flightstream.input.Export_Solver_Analysis = ["Tecplot", "VTK"]

        wing = {"groupName"         : "Wing",
                "numChord"          : 48,
                "growthTypeChord"   : 2,
                "growthRateChord"   : 1.2,
                "periodicityChord"  : 3,
                "numSpanPerSection" : 22,
                "growthTypeSpan"   : 4,
                "growthRateSpan"   : 1.3,
                "periodicitySpan"  : 1}

        flightstream.input.CCS_LiftingSurface = {"Wing": wing}

        flightstream.input.Mach               = None
        flightstream.input.SweepRangeMach     = [0.1, 0.3, 0.1]
        flightstream.input.SweepRangeVelocity = None
        flightstream.input.Alpha              = 1.0 * deg
        flightstream.input.SweepRangeAlpha    = None
        flightstream.input.Beta               = 0.0 * deg
        flightstream.input.SweepRangeBeta     = None

        # Retrieve results
        Cx  = flightstream.output.Cx
        Cy  = flightstream.output.Cy
        Cz  = flightstream.output.Cz
        CL  = flightstream.output.CL
        CDi = flightstream.output.CDi
        CDo = flightstream.output.CDo
        CMx = flightstream.output.CMx
        CMy = flightstream.output.CMy
        CMz = flightstream.output.CMz

        #print("        CxTrue  = ", Cx )
        #print("        CyTrue  = ", Cy )
        #print("        CzTrue  = ", Cz )
        #print("        CLTrue  = ", CL )
        #print("        CDiTrue = ", CDi)
        #print("        CDoTrue = ", CDo)
        #print("        CMxTrue = ", CMx)
        #print("        CMyTrue = ", CMy)
        #print("        CMzTrue = ", CMz)

        CxTrue  =  [0.0007, 0.0019, 0.004]
        CyTrue  =  [0.0003, 0.0011, 0.0025]
        CzTrue  =  [0.0038, 0.0158, 0.0364]
        CLTrue  =  [0.0038, 0.0157, 0.0363]
        CDiTrue =  [0.0007, 0.0019, 0.004]
        CDoTrue =  [0.0001, 0.0003, 0.0006]
        CMxTrue =  [0.0016, 0.0066, 0.0153]
        CMyTrue =  [-0.0011, -0.0044, -0.0103]
        CMzTrue =  [-0.0, -0.0001, -0.0002]

        for i in range(len(CxTrue)):
            self.assertAlmostEqual(CxTrue[i] , Cx[i] , 4)
            self.assertAlmostEqual(CyTrue[i] , Cy[i] , 4)
            self.assertAlmostEqual(CzTrue[i] , Cz[i] , 4)
            self.assertAlmostEqual(CLTrue[i] , CL[i] , 4)
            self.assertAlmostEqual(CDiTrue[i], CDi[i], 4)
            self.assertAlmostEqual(CDoTrue[i], CDo[i], 4)
            self.assertAlmostEqual(CMxTrue[i], CMx[i], 4)
            self.assertAlmostEqual(CMyTrue[i], CMy[i], 4)
            self.assertAlmostEqual(CMzTrue[i], CMz[i], 4)

        flightstream.input.Mach               = None
        flightstream.input.SweepRangeMach     = None
        flightstream.input.SweepRangeVelocity = [100, 200, 50]*m/s
        flightstream.input.Alpha              = 1.0 * deg
        flightstream.input.SweepRangeAlpha    = None
        flightstream.input.Beta               = 0.0 * deg
        flightstream.input.SweepRangeBeta     = None

        # Retrieve results
        Cx  = flightstream.output.Cx
        Cy  = flightstream.output.Cy
        Cz  = flightstream.output.Cz
        CL  = flightstream.output.CL
        CDi = flightstream.output.CDi
        CDo = flightstream.output.CDo
        CMx = flightstream.output.CMx
        CMy = flightstream.output.CMy
        CMz = flightstream.output.CMz

        #print("        CxTrue  = ", Cx )
        #print("        CyTrue  = ", Cy )
        #print("        CzTrue  = ", Cz )
        #print("        CLTrue  = ", CL )
        #print("        CDiTrue = ", CDi)
        #print("        CDoTrue = ", CDo)
        #print("        CMxTrue = ", CMx)
        #print("        CMyTrue = ", CMy)
        #print("        CMzTrue = ", CMz)

        CxTrue  =  [0.0039, 0.0088, 0.0168]
        CyTrue  =  [0.0024, 0.0058, 0.0113]
        CzTrue  =  [0.0351, 0.0836, 0.1631]
        CLTrue  =  [0.0351, 0.0835, 0.1629]
        CDiTrue =  [0.0039, 0.0089, 0.0173]
        CDoTrue =  [0.0006, 0.0014, 0.0024]
        CMxTrue =  [0.0147, 0.035, 0.0679]
        CMyTrue =  [-0.0099, -0.0237, -0.0466]
        CMzTrue =  [-0.0002, -0.0004, -0.0007]

        for i in range(len(CxTrue)):
            self.assertAlmostEqual(CxTrue[i] , Cx[i] , 4)
            self.assertAlmostEqual(CyTrue[i] , Cy[i] , 4)
            self.assertAlmostEqual(CzTrue[i] , Cz[i] , 4)
            self.assertAlmostEqual(CLTrue[i] , CL[i] , 4)
            self.assertAlmostEqual(CDiTrue[i], CDi[i], 4)
            self.assertAlmostEqual(CDoTrue[i], CDo[i], 4)
            self.assertAlmostEqual(CMxTrue[i], CMx[i], 4)
            self.assertAlmostEqual(CMyTrue[i], CMy[i], 4)
            self.assertAlmostEqual(CMzTrue[i], CMz[i], 4)

        flightstream.input.Mach               = 0.5
        flightstream.input.SweepRangeMach     = None
        flightstream.input.SweepRangeVelocity = None
        flightstream.input.Alpha              = None
        flightstream.input.SweepRangeAlpha    = [1, 3, 1] * deg
        flightstream.input.Beta               = 0.0 * deg
        flightstream.input.SweepRangeBeta     = None

        # Retrieve results
        Cx  = flightstream.output.Cx
        Cy  = flightstream.output.Cy
        Cz  = flightstream.output.Cz
        CL  = flightstream.output.CL
        CDi = flightstream.output.CDi
        CDo = flightstream.output.CDo
        CMx = flightstream.output.CMx
        CMy = flightstream.output.CMy
        CMz = flightstream.output.CMz

        #print("        CxTrue  = ", Cx )
        #print("        CyTrue  = ", Cy )
        #print("        CzTrue  = ", Cz )
        #print("        CLTrue  = ", CL )
        #print("        CDiTrue = ", CDi)
        #print("        CDoTrue = ", CDo)
        #print("        CMxTrue = ", CMx)
        #print("        CMyTrue = ", CMy)
        #print("        CMzTrue = ", CMz)

        CxTrue  =  [0.0115, 0.0103, 0.0088]
        CyTrue  =  [0.0076, 0.0083, 0.0089]
        CzTrue  =  [0.1102, 0.131, 0.1518]
        CLTrue  =  [0.1101, 0.1307, 0.1513]
        CDiTrue =  [0.0117, 0.0131, 0.015]
        CDoTrue =  [0.0017, 0.0017, 0.0017]
        CMxTrue =  [0.046, 0.0458, 0.0455]
        CMyTrue =  [-0.0314, -0.0312, -0.0311]
        CMzTrue =  [-0.0005, -0.0012, -0.0019]

        for i in range(len(CxTrue)):
            self.assertAlmostEqual(CxTrue[i] , Cx[i] , 4)
            self.assertAlmostEqual(CyTrue[i] , Cy[i] , 4)
            self.assertAlmostEqual(CzTrue[i] , Cz[i] , 4)
            self.assertAlmostEqual(CLTrue[i] , CL[i] , 4)
            self.assertAlmostEqual(CDiTrue[i], CDi[i], 4)
            self.assertAlmostEqual(CDoTrue[i], CDo[i], 4)
            self.assertAlmostEqual(CMxTrue[i], CMx[i], 4)
            self.assertAlmostEqual(CMyTrue[i], CMy[i], 4)
            self.assertAlmostEqual(CMzTrue[i], CMz[i], 4)

        flightstream.input.Mach               = 0.5
        flightstream.input.SweepRangeMach     = None
        flightstream.input.SweepRangeVelocity = None
        flightstream.input.Alpha              = 1.0 * deg
        flightstream.input.SweepRangeAlpha    = None
        flightstream.input.Beta               = None
        flightstream.input.SweepRangeBeta     = [-1, 1, 1] * deg

        # Retrieve results
        Cx  = flightstream.output.Cx
        Cy  = flightstream.output.Cy
        Cz  = flightstream.output.Cz
        CL  = flightstream.output.CL
        CDi = flightstream.output.CDi
        CDo = flightstream.output.CDo
        CMx = flightstream.output.CMx
        CMy = flightstream.output.CMy
        CMz = flightstream.output.CMz

        #print("        CxTrue  = ", Cx )
        #print("        CyTrue  = ", Cy )
        #print("        CzTrue  = ", Cz )
        #print("        CLTrue  = ", CL )
        #print("        CDiTrue = ", CDi)
        #print("        CDoTrue = ", CDo)
        #print("        CMxTrue = ", CMx)
        #print("        CMyTrue = ", CMy)
        #print("        CMzTrue = ", CMz)

        CxTrue  =  [0.0114, 0.0116, 0.0117]
        CyTrue  =  [0.0078, 0.0076, 0.0074]
        CzTrue  =  [0.1123, 0.1102, 0.108]
        CLTrue  =  [0.1122, 0.1101, 0.1079]
        CDiTrue =  [0.0118, 0.0117, 0.0117]
        CDoTrue =  [0.0017, 0.0017, 0.0017]
        CMxTrue =  [0.0453, 0.046, 0.0467]
        CMyTrue =  [-0.0316, -0.0313, -0.031]
        CMzTrue =  [-0.0004, -0.0005, -0.0006]

        for i in range(len(CxTrue)):
            self.assertAlmostEqual(CxTrue[i] , Cx[i] , 4)
            self.assertAlmostEqual(CyTrue[i] , Cy[i] , 4)
            self.assertAlmostEqual(CzTrue[i] , Cz[i] , 4)
            self.assertAlmostEqual(CLTrue[i] , CL[i] , 4)
            self.assertAlmostEqual(CDiTrue[i], CDi[i], 4)
            self.assertAlmostEqual(CDoTrue[i], CDo[i], 4)
            self.assertAlmostEqual(CMxTrue[i], CMx[i], 4)
            self.assertAlmostEqual(CMyTrue[i], CMy[i], 4)
            self.assertAlmostEqual(CMzTrue[i], CMz[i], 4)

#==============================================================================
    def test_wing_tail(self):
        # Define units
        deg  = pyCAPS.Unit("degree")
        ft   = pyCAPS.Unit("ft")
        m    = pyCAPS.Unit("m")
        s    = pyCAPS.Unit("s")

        # Load flightstream aim
        flightstream = self.capsProblem.analysis.create(aim = "flightstreamAIM")

        # Set new Mach/Alt parameters
        flightstream.input.FlightStream  = FlightStream_exe
        flightstream.input.Mach  = 0.5
        flightstream.input.Alpha = 1.0 * deg
        flightstream.input.Beta  = 0.0 * deg
        flightstream.input.Altitude = 1000 * ft
        flightstream.input.Solver_Model = "SUBSONIC_PRANDTL_GLAUERT"
        flightstream.input.ReferenceVelocity = 295.06 * m/s
        flightstream.input.Export_Solver_Analysis = ["Tecplot", "VTK"]

        wing = {"groupName"         : "Wing",
                "numChord"          : 48,
                "numSpanPerSection" : 22}

        htail = {"groupName"         : "hTail",
                 "numChord"          : 48,
                 "numSpanTotal"      : 20}

        vtail = {"numChord"          : 48,
                 "numSpanTotal"      : 18}

        flightstream.input.CCS_LiftingSurface = {"Wing" : wing,
                                                 "hTail": htail,
                                                 "vTail": vtail}

        #flightstream.preAnalysis()
        #shutil.copytree("fsAnalysisTest_ref/Scratch/flightstreamAIM0", flightstream.analysisDir, dirs_exist_ok=True)
        #flightstream.postAnalysis()

        Cx  = flightstream.output.Cx
        Cy  = flightstream.output.Cy
        Cz  = flightstream.output.Cz
        CL  = flightstream.output.CL
        CDi = flightstream.output.CDi
        CDo = flightstream.output.CDo
        CMx = flightstream.output.CMx
        CMy = flightstream.output.CMy
        CMz = flightstream.output.CMz

        #print("        CxTrue  = ", Cx )
        #print("        CyTrue  = ", Cy )
        #print("        CzTrue  = ", Cz )
        #print("        CLTrue  = ", CL )
        #print("        CDiTrue = ", CDi)
        #print("        CDoTrue = ", CDo)
        #print("        CMxTrue = ", CMx)
        #print("        CMyTrue = ", CMy)
        #print("        CMzTrue = ", CMz)
        ## Get all dynamic ouputs
        #for i in flightstream.dynout.keys():
        #    print("            '" + str(i) + "' : " + str(flightstream.dynout[i].value) + ",")

        CxTrue  =  0.0699
        CyTrue  =  0.4019
        CzTrue  =  0.0541
        CLTrue  =  0.053
        CDiTrue =  0.0683
        CDoTrue =  0.0025
        CMxTrue =  -0.2623
        CMyTrue =  0.1593
        CMzTrue =  0.9049

        dynTrue = {
            'Wing' : {'Cx': 0.0076, 'Cy': 0.0069, 'Cz': 0.1167, 'CL': 0.1166, 'CDi': 0.0082, 'CDo': 0.0015, 'CMx': 0.0179, 'CMy': -0.0268, 'CMz': -0.0027},
            'LeftAileron' : {'Cx': -0.0003, 'Cy': 0.0013, 'Cz': 0.0055, 'CL': 0.0055, 'CDi': -0.0003, 'CDo': 0.0001, 'CMx': -0.0058, 'CMy': -0.0035, 'CMz': 0.0004},
            'RightAileron' : {'Cx': 0.0009, 'Cy': 0.0006, 'Cz': 0.0036, 'CL': 0.0037, 'CDi': 0.0008, 'CDo': 0.0001, 'CMx': 0.0036, 'CMy': -0.0024, 'CMz': -0.0006},
            'Wing_base' : {'Cx': 0.0001, 'Cy': -0.0, 'Cz': -0.0, 'CL': -0.0, 'CDi': 0.0001, 'CDo': 0.0, 'CMx': 0.0, 'CMy': 0.0, 'CMz': 0.0},
            'LeftAileron_base' : {'Cx': 0.0001, 'Cy': -0.0, 'Cz': -0.0, 'CL': -0.0, 'CDi': 0.0001, 'CDo': 0.0, 'CMx': 0.0, 'CMy': 0.0, 'CMz': 0.0001},
            'RightAileron_base' : {'Cx': 0.0001, 'Cy': -0.0, 'Cz': -0.0, 'CL': -0.0, 'CDi': 0.0001, 'CDo': 0.0, 'CMx': -0.0, 'CMy': 0.0, 'CMz': -0.0001},
            'LeftAileron_side' : {'Cx': -0.0, 'Cy': 0.0, 'Cz': 0.0, 'CL': 0.0, 'CDi': -0.0, 'CDo': 0.0, 'CMx': -0.0, 'CMy': -0.0, 'CMz': 0.0},
            'RightAileron_side' : {'Cx': -0.0, 'Cy': -0.0, 'Cz': -0.0, 'CL': -0.0, 'CDi': -0.0, 'CDo': 0.0, 'CMx': 0.0, 'CMy': 0.0, 'CMz': -0.0},
            'hTail' : {'Cx': 0.0455, 'Cy': 0.1373, 'Cz': -0.0761, 'CL': -0.0769, 'CDi': 0.0438, 'CDo': 0.0003, 'CMx': -0.1497, 'CMy': 0.1919, 'CMz': 0.3055},
            'Elevator' : {'Cx': 0.0186, 'Cy': 0.0307, 'Cz': -0.0358, 'CL': -0.0361, 'CDi': 0.018, 'CDo': 0.0, 'CMx': -0.0241, 'CMy': 0.0999, 'CMz': 0.0766},
            'Elevator_base' : {'Cx': -0.0012, 'Cy': -0.0016, 'Cz': 0.0108, 'CL': 0.0109, 'CDi': -0.001, 'CDo': 0.0, 'CMx': -0.0009, 'CMy': -0.0294, 'CMz': -0.004},
            'vTail' : {'Cx': -0.0024, 'Cy': 0.1424, 'Cz': 0.0071, 'CL': 0.0072, 'CDi': -0.0026, 'CDo': 0.0003, 'CMx': -0.0702, 'CMy': -0.0145, 'CMz': 0.3105},
            'Rudder' : {'Cx': 0.0007, 'Cy': 0.0844, 'Cz': 0.0222, 'CL': 0.0222, 'CDi': 0.001, 'CDo': 0.0002, 'CMx': -0.033, 'CMy': -0.0559, 'CMz': 0.2192},
            'Rudder_base' : {'Cx': 0.0002, 'Cy': -0.0, 'Cz': -0.0, 'CL': -0.0, 'CDi': 0.0002, 'CDo': 0.0, 'CMx': 0.0, 'CMy': 0.0001, 'CMz': -0.0},
        }

        self.assertAlmostEqual(CxTrue , Cx , 4)
        self.assertAlmostEqual(CyTrue , Cy , 4)
        self.assertAlmostEqual(CzTrue , Cz , 4)
        self.assertAlmostEqual(CLTrue , CL , 4)
        self.assertAlmostEqual(CDiTrue, CDi, 4)
        self.assertAlmostEqual(CDoTrue, CDo, 4)
        self.assertAlmostEqual(CMxTrue, CMx, 4)
        self.assertAlmostEqual(CMyTrue, CMy, 4)
        self.assertAlmostEqual(CMzTrue, CMz, 4)

        for i in dynTrue.keys():
            dyn = flightstream.dynout[i].value
            for j in dynTrue[i].keys():
                self.assertAlmostEqual(dynTrue[i][j], dyn[j], 4)


#==============================================================================
    # Dissabled due to Vtail intersection causing problems
    def off_test_wing_Vtail(self):
        # Define units
        deg  = pyCAPS.Unit("degree")
        ft   = pyCAPS.Unit("ft")
        m    = pyCAPS.Unit("m")
        s    = pyCAPS.Unit("s")

        # Load flightstream aim
        flightstream = self.capsProblem.analysis.create(aim = "flightstreamAIM")

        # Set new Mach/Alt parameters
        flightstream.input.FlightStream  = FlightStream_exe
        flightstream.input.Mach  = 0.5
        flightstream.input.Alpha = 1.0 * deg
        flightstream.input.Beta  = 0.0 * deg
        flightstream.input.Altitude = 1000 * ft
        flightstream.input.Solver_Model = "SUBSONIC_PRANDTL_GLAUERT"
        flightstream.input.ReferenceVelocity = 295.06 * m/s
        flightstream.input.Export_Solver_Analysis = ["Tecplot", "VTK"]

        wing = {"groupName"         : "Wing",
                "numChord"          : 48,
                "numSpanPerSection" : 22}

        vtail = {"numChord"          : 48,
                 "numSpanTotal"      : 18}

        flightstream.input.CCS_LiftingSurface = {"Wing"  : wing,
                                                 "VTail1": vtail,
                                                 "VTail2": vtail}

        # Retrieve results
        Cx  = flightstream.output.Cx
        Cy  = flightstream.output.Cy
        Cz  = flightstream.output.Cz
        CL  = flightstream.output.CL
        CDi = flightstream.output.CDi
        CDo = flightstream.output.CDo
        CMx = flightstream.output.CMx
        CMy = flightstream.output.CMy
        CMz = flightstream.output.CMz

        #print("        CxTrue  = ", Cx )
        #print("        CyTrue  = ", Cy )
        #print("        CzTrue  = ", Cz )
        #print("        CLTrue  = ", CL )
        #print("        CDiTrue = ", CDi)
        #print("        CDoTrue = ", CDo)
        #print("        CMxTrue = ", CMx)
        #print("        CMyTrue = ", CMy)
        #print("        CMzTrue = ", CMz)
        # Get all dynamic ouputs
        #for i in flightstream.dynout.keys():
        #    print("'" + i + "' : " + str(flightstream.dynout[i].value) + ",")

        CxTrue  = 0.3045
        CyTrue  = 0.0278
        CzTrue  = 0.1835
        CLTrue  = 0.1782
        CDiTrue = 0.2728
        CDoTrue = 0.0348
        CMxTrue = 0.788
        CMyTrue = 0.5104
        CMzTrue = 0.0606

        dynTrue = {
            'VTail1' : {'Cx': 0.0455, 'Cy': 0.0112, 'Cz': -0.0449, 'CL': -0.0457, 'CDi': 0.0367, 'CDo': 0.008, 'CMx': -0.0836, 'CMy': 0.5082, 'CMz': 0.0981},
            'VTail1_base' : {'Cx': 0.0027, 'Cy': -0.0003, 'Cz': 0.0003, 'CL': 0.0002, 'CDi': 0.0027, 'CDo': 0.0, 'CMx': 0.0008, 'CMy': 0.002, 'CMz': -0.007},
            'VTail2' : {'Cx': 0.0443, 'Cy': -0.0131, 'Cz': -0.0471, 'CL': -0.0478, 'CDi': 0.0353, 'CDo': 0.0081, 'CMx': 0.0861, 'CMy': 0.5225, 'CMz': -0.1132},
            'VTail2_base' : {'Cx': 0.0027, 'Cy': 0.0003, 'Cz': 0.0003, 'CL': 0.0002, 'CDi': 0.0027, 'CDo': 0.0, 'CMx': -0.0008, 'CMy': 0.002, 'CMz': 0.007},
            'Wing' : {'Cx': 0.1295, 'Cy': 0.0321, 'Cz': 0.282, 'CL': 0.2797, 'CDi': 0.1157, 'CDo': 0.0187, 'CMx': 0.8231, 'CMy': -0.5533, 'CMz': 0.0768},
            'LeftAileron' : {'Cx': 0.0391, 'Cy': 0.0031, 'Cz': 0.0009, 'CL': 0.0002, 'CDi': 0.0391, 'CDo': 0.0, 'CMx': -0.0088, 'CMy': 0.0138, 'CMz': 0.1375},
            'RightAileron' : {'Cx': 0.0387, 'Cy': -0.0053, 'Cz': -0.0077, 'CL': -0.0084, 'CDi': 0.0385, 'CDo': 0.0, 'CMx': -0.0292, 'CMy': 0.0142, 'CMz': -0.1407},
            'Wing_base' : {'Cx': 0.001, 'Cy': -0.0, 'Cz': -0.0001, 'CL': -0.0001, 'CDi': 0.001, 'CDo': 0.0, 'CMx': 0.0001, 'CMy': 0.0004, 'CMz': 0.0005},
            'LeftAileron_base' : {'Cx': 0.001, 'Cy': -0.0001, 'Cz': -0.0001, 'CL': -0.0001, 'CDi': 0.001, 'CDo': 0.0, 'CMx': 0.0004, 'CMy': 0.0007, 'CMz': 0.0024},
            'RightAileron_base' : {'Cx': 0.0, 'Cy': 0.0, 'Cz': 0.0, 'CL': 0.0, 'CDi': 0.0, 'CDo': 0.0, 'CMx': 0.0, 'CMy': 0.0, 'CMz': 0.0},
            'LeftAileron_side' : {'Cx': 0.0, 'Cy': -0.0003, 'Cz': 0.0, 'CL': 0.0, 'CDi': 0.0, 'CDo': 0.0, 'CMx': 0.0, 'CMy': -0.0, 'CMz': -0.001},
            'RightAileron_side' : {'Cx': 0.0, 'Cy': 0.0001, 'Cz': 0.0, 'CL': -0.0, 'CDi': -0.0, 'CDo': 0.0, 'CMx': -0.0, 'CMy': -0.0, 'CMz': 0.0002},
        }

        self.assertAlmostEqual(CxTrue , Cx , 4)
        self.assertAlmostEqual(CyTrue , Cy , 4)
        self.assertAlmostEqual(CzTrue , Cz , 4)
        self.assertAlmostEqual(CLTrue , CL , 4)
        self.assertAlmostEqual(CDiTrue, CDi, 4)
        self.assertAlmostEqual(CDoTrue, CDo, 4)
        self.assertAlmostEqual(CMxTrue, CMx, 4)
        self.assertAlmostEqual(CMyTrue, CMy, 4)
        self.assertAlmostEqual(CMzTrue, CMz, 4)

        for i in dynTrue.keys():
            dyn = flightstream.dynout[i].value
            for j in dynTrue[i].keys():
                self.assertAlmostEqual(dynTrue[i][j], dyn[j], 4)

#==============================================================================
    def test_wing_control(self):
        # Define units
        deg  = pyCAPS.Unit("degree")
        ft   = pyCAPS.Unit("ft")
        m    = pyCAPS.Unit("m")
        s    = pyCAPS.Unit("s")

        # Load flightstream aim
        flightstream = self.capsProblem.analysis.create(aim = "flightstreamAIM")

        # Set new Mach/Alt parameters
        flightstream.input.FlightStream  = FlightStream_exe
        flightstream.input.Mach  = 0.5
        flightstream.input.Alpha = 1.0 * deg
        flightstream.input.Beta  = 0.0 * deg
        flightstream.input.Altitude = 1000 * ft
        flightstream.input.Solver_Model = "SUBSONIC_PRANDTL_GLAUERT"
        flightstream.input.ReferenceVelocity = 295.06 * m/s
        flightstream.input.Export_Solver_Analysis = ["Tecplot", "VTK"]

        wing = {"groupName"         : "Wing",
                "numChord"          : 48,
                "numSpanPerSection" : 22}

        flightstream.input.CCS_LiftingSurface = {"Wing" : wing}


        CxTrue  = {10 :  0.0195, 25 :  0.0361}
        CyTrue  = {10 :  0.0052, 25 :  0.0020}
        CzTrue  = {10 :  0.0961, 25 :  0.0734}
        CLTrue  = {10 :  0.0958, 25 :  0.0729}
        CDiTrue = {10 :  0.0194, 25 :  0.0354}
        CDoTrue = {10 :  0.0018, 25 :  0.0019}
        CMxTrue = {10 :  0.1009, 25 :  0.1480}
        CMyTrue = {10 : -0.0268, 25 : -0.0206}
        CMzTrue = {10 : -0.0036, 25 : -0.0062}

        for angle in [10, 25]:

            # Set control surface parameters
            aileronLeft  = {"deflectionAngle" : -angle * deg}
            aileronRight = {"deflectionAngle" :  angle * deg}

            flightstream.input.CCS_Control = {"LeftAileron" : aileronLeft ,
                                              "RightAileron": aileronRight}

            # Retrieve results
            Cx  = flightstream.output.Cx
            Cy  = flightstream.output.Cy
            Cz  = flightstream.output.Cz
            CL  = flightstream.output.CL
            CDi = flightstream.output.CDi
            CDo = flightstream.output.CDo
            CMx = flightstream.output.CMx
            CMy = flightstream.output.CMy
            CMz = flightstream.output.CMz

            #print("Angle = ", angle )
            #print("Cx  = ", Cx )
            #print("Cy  = ", Cy )
            #print("Cz  = ", Cz )
            #print("CL  = ", CL)
            #print("CDi = ", CDi)
            #print("CDo = ", CDo)
            #print("CMx = ", CMx)
            #print("CMy = ", CMy)
            #print("CMz = ", CMz)

            self.assertAlmostEqual(CxTrue[angle] , Cx , 4)
            self.assertAlmostEqual(CyTrue[angle] , Cy , 4)
            self.assertAlmostEqual(CzTrue[angle] , Cz , 4)
            self.assertAlmostEqual(CLTrue[angle] , CL , 4)
            self.assertAlmostEqual(CDiTrue[angle], CDi, 4)
            self.assertAlmostEqual(CDoTrue[angle], CDo, 4)
            self.assertAlmostEqual(CMxTrue[angle], CMx, 4)
            self.assertAlmostEqual(CMyTrue[angle], CMy, 4)
            self.assertAlmostEqual(CMzTrue[angle], CMz, 4)

#==============================================================================
    def test_geom_change(self):
        # Define units
        deg  = pyCAPS.Unit("degree")
        ft   = pyCAPS.Unit("ft")
        m    = pyCAPS.Unit("m")
        s    = pyCAPS.Unit("s")

        # Load flightstream aim
        flightstream = self.capsProblem.analysis.create(aim = "flightstreamAIM")

        # Set new Mach/Alt parameters
        flightstream.input.FlightStream  = FlightStream_exe
        flightstream.input.Mach  = 0.5
        flightstream.input.Alpha = 1.0 * deg
        flightstream.input.Beta  = 0.0 * deg
        flightstream.input.Altitude = 1000 * ft
        flightstream.input.Solver_Model = "SUBSONIC_PRANDTL_GLAUERT"
        flightstream.input.ReferenceVelocity = 295.06 * m/s
        flightstream.input.Export_Solver_Analysis = ["Tecplot", "VTK"]

        wing = {"groupName"         : "Wing",
                "numChord"          : 48,
                "numSpanPerSection" : 22}

        flightstream.input.CCS_LiftingSurface = {"Wing": wing}

        # Retrieve results
        Cx  = flightstream.output.Cx
        Cy  = flightstream.output.Cy
        Cz  = flightstream.output.Cz
        CL  = flightstream.output.CL
        CDi = flightstream.output.CDi
        CDo = flightstream.output.CDo
        CMx = flightstream.output.CMx
        CMy = flightstream.output.CMy
        CMz = flightstream.output.CMz

        #print("        CxTrue  = ", Cx )
        #print("        CyTrue  = ", Cy )
        #print("        CzTrue  = ", Cz )
        #print("        CLTrue  = ", CL )
        #print("        CDiTrue = ", CDi)
        #print("        CDoTrue = ", CDo)
        #print("        CMxTrue = ", CMx)
        #print("        CMyTrue = ", CMy)
        #print("        CMzTrue = ", CMz)

        CxTrue  =  0.0115
        CyTrue  =  0.0076
        CzTrue  =  0.1102
        CLTrue  =  0.1101
        CDiTrue =  0.0117
        CDoTrue =  0.0017
        CMxTrue =  0.046
        CMyTrue =  -0.0314
        CMzTrue =  -0.0005

        self.assertAlmostEqual(CxTrue , Cx , 4)
        self.assertAlmostEqual(CyTrue , Cy , 4)
        self.assertAlmostEqual(CzTrue , Cz , 4)
        self.assertAlmostEqual(CLTrue , CL , 4)
        self.assertAlmostEqual(CDiTrue, CDi, 4)
        self.assertAlmostEqual(CDoTrue, CDo, 4)
        self.assertAlmostEqual(CMxTrue, CMx, 4)
        self.assertAlmostEqual(CMyTrue, CMy, 4)
        self.assertAlmostEqual(CMzTrue, CMz, 4)

        # change the geometry and rerun
        camber = self.capsProblem.geometry.despmtr.camber
        self.capsProblem.geometry.despmtr.camber = 1.1*camber

        # Retrieve results
        Cx  = flightstream.output.Cx
        Cy  = flightstream.output.Cy
        Cz  = flightstream.output.Cz
        CL  = flightstream.output.CL
        CDi = flightstream.output.CDi
        CDo = flightstream.output.CDo
        CMx = flightstream.output.CMx
        CMy = flightstream.output.CMy
        CMz = flightstream.output.CMz

        #print("        CxTrue  = ", Cx )
        #print("        CyTrue  = ", Cy )
        #print("        CzTrue  = ", Cz )
        #print("        CLTrue  = ", CL )
        #print("        CDiTrue = ", CDi)
        #print("        CDoTrue = ", CDo)
        #print("        CMxTrue = ", CMx)
        #print("        CMyTrue = ", CMy)
        #print("        CMzTrue = ", CMz)

        CxTrue  =  0.0122
        CyTrue  =  0.0084
        CzTrue  =  0.1205
        CLTrue  =  0.1203
        CDiTrue =  0.0126
        CDoTrue =  0.0017
        CMxTrue =  0.0458
        CMyTrue =  -0.0345
        CMzTrue =  -0.0006

        self.assertAlmostEqual(CxTrue , Cx , 4)
        self.assertAlmostEqual(CyTrue , Cy , 4)
        self.assertAlmostEqual(CzTrue , Cz , 4)
        self.assertAlmostEqual(CLTrue , CL , 4)
        self.assertAlmostEqual(CDiTrue, CDi, 4)
        self.assertAlmostEqual(CDoTrue, CDo, 4)
        self.assertAlmostEqual(CMxTrue, CMx, 4)
        self.assertAlmostEqual(CMyTrue, CMy, 4)
        self.assertAlmostEqual(CMzTrue, CMz, 4)

        # reset the geometry
        self.capsProblem.geometry.despmtr.camber = camber


#==============================================================================
    def test_mesh_change_MachSweep(self):
        # Define units
        deg  = pyCAPS.Unit("degree")
        ft   = pyCAPS.Unit("ft")
        m    = pyCAPS.Unit("m")
        s    = pyCAPS.Unit("s")

        # Load flightstream aim
        flightstream = self.capsProblem.analysis.create(aim = "flightstreamAIM")

        # Set new Mach/Alt parameters
        flightstream.input.FlightStream  = FlightStream_exe
        flightstream.input.SweepRangeMach = [0.5, 0.6, 0.1]
        flightstream.input.Alpha = 1.0 * deg
        flightstream.input.Beta  = 0.0 * deg
        flightstream.input.Altitude = 1000 * ft
        flightstream.input.Solver_Model = "SUBSONIC_PRANDTL_GLAUERT"
        flightstream.input.ReferenceVelocity = 295.06 * m/s
        flightstream.input.Export_Solver_Analysis = ["Tecplot", "VTK"]

        wing = {"groupName"         : "Wing",
                "numChord"          : 48,
                "numSpanPerSection" : 22}

        flightstream.input.CCS_LiftingSurface = {"Wing": wing}

        # Retrieve results
        Cx  = flightstream.output.Cx
        Cy  = flightstream.output.Cy
        Cz  = flightstream.output.Cz
        CL  = flightstream.output.CL
        CDi = flightstream.output.CDi
        CDo = flightstream.output.CDo
        CMx = flightstream.output.CMx
        CMy = flightstream.output.CMy
        CMz = flightstream.output.CMz

        # print("        CxTrue  = ", Cx )
        # print("        CyTrue  = ", Cy )
        # print("        CzTrue  = ", Cz )
        # print("        CLTrue  = ", CL )
        # print("        CDiTrue = ", CDi)
        # print("        CDoTrue = ", CDo)
        # print("        CMxTrue = ", CMx)
        # print("        CMyTrue = ", CMy)
        # print("        CMzTrue = ", CMz)

        CxTrue  =  [0.0115, 0.0176]
        CyTrue  =  [0.0076, 0.0118]
        CzTrue  =  [0.1102, 0.1702]
        CLTrue  =  [0.1101, 0.17]
        CDiTrue =  [0.0117, 0.0181]
        CDoTrue =  [0.0017, 0.0025]
        CMxTrue =  [0.046, 0.0708]
        CMyTrue =  [-0.0314, -0.0487]
        CMzTrue =  [-0.0005, -0.0007]

        for i in range(len(CxTrue)):
            self.assertAlmostEqual(CxTrue[i] , Cx[i] , 4)
            self.assertAlmostEqual(CyTrue[i] , Cy[i] , 4)
            self.assertAlmostEqual(CzTrue[i] , Cz[i] , 4)
            self.assertAlmostEqual(CLTrue[i] , CL[i] , 4)
            self.assertAlmostEqual(CDiTrue[i], CDi[i], 4)
            self.assertAlmostEqual(CDoTrue[i], CDo[i], 4)
            self.assertAlmostEqual(CMxTrue[i], CMx[i], 4)
            self.assertAlmostEqual(CMyTrue[i], CMy[i], 4)
            self.assertAlmostEqual(CMzTrue[i], CMz[i], 4)

        # change the mesh resolution and rerun
        wing = {"groupName"         : "Wing",
                "numChord"          : 64,
                "numSpanPerSection" : 28}

        flightstream.input.CCS_LiftingSurface = {"Wing": wing}


        flightstream.input.SweepRangeMach = [0.4, 0.6, 0.1]

        # Retrieve results
        Cx  = flightstream.output.Cx
        Cy  = flightstream.output.Cy
        Cz  = flightstream.output.Cz
        CL  = flightstream.output.CL
        CDi = flightstream.output.CDi
        CDo = flightstream.output.CDo
        CMx = flightstream.output.CMx
        CMy = flightstream.output.CMy
        CMz = flightstream.output.CMz

        # print("        CxTrue  = ", Cx )
        # print("        CyTrue  = ", Cy )
        # print("        CzTrue  = ", Cz )
        # print("        CLTrue  = ", CL )
        # print("        CDiTrue = ", CDi)
        # print("        CDoTrue = ", CDo)
        # print("        CMxTrue = ", CMx)
        # print("        CMyTrue = ", CMy)
        # print("        CMzTrue = ", CMz)

        CxTrue  =  [0.0069, 0.0111, 0.0169]
        CyTrue  =  [0.0047, 0.0077, 0.0119]
        CzTrue  =  [0.0665, 0.1092, 0.1685]
        CLTrue  =  [0.0665, 0.1091, 0.1684]
        CDiTrue =  [0.0068, 0.0111, 0.0171]
        CDoTrue =  [0.0012, 0.0019, 0.0027]
        CMxTrue =  [0.028, 0.0459, 0.0704]
        CMyTrue =  [-0.0192, -0.0318, -0.0493]
        CMzTrue =  [-0.0002, -0.0004, -0.0005]

        for i in range(len(CxTrue)):
            self.assertAlmostEqual(CxTrue[i] , Cx[i] , 4)
            self.assertAlmostEqual(CyTrue[i] , Cy[i] , 4)
            self.assertAlmostEqual(CzTrue[i] , Cz[i] , 4)
            self.assertAlmostEqual(CLTrue[i] , CL[i] , 4)
            self.assertAlmostEqual(CDiTrue[i], CDi[i], 4)
            self.assertAlmostEqual(CDoTrue[i], CDo[i], 4)
            self.assertAlmostEqual(CMxTrue[i], CMx[i], 4)
            self.assertAlmostEqual(CMyTrue[i], CMy[i], 4)
            self.assertAlmostEqual(CMzTrue[i], CMz[i], 4)


#==============================================================================
    def test_phase(self):
        # Define units
        deg  = pyCAPS.Unit("degree")
        ft   = pyCAPS.Unit("ft")
        m    = pyCAPS.Unit("m")
        s    = pyCAPS.Unit("s")

        # Initialize Problem object
        problemName = self.problemName + "_Phase"
        capsProblem = pyCAPS.Problem(problemName, phaseName="Phase0",
                                   capsFile=os.path.join("..","csmData","avlSections.csm"),
                                   outLevel=0)

        # Load flightstream aim
        flightstream = capsProblem.analysis.create(aim = "flightstreamAIM", name = "flightstream")

        # Set new Mach/Alt parameters
        flightstream.input.FlightStream  = FlightStream_exe
        flightstream.input.Mach  = 0.5
        flightstream.input.Alpha = 1.0 * deg
        flightstream.input.Beta  = 0.0 * deg
        flightstream.input.Altitude = 1000 * ft
        flightstream.input.Solver_Model = "SUBSONIC_PRANDTL_GLAUERT"
        flightstream.input.ReferenceVelocity = 295.06 * m/s
        flightstream.input.Export_Solver_Analysis = ["Tecplot", "VTK"]

        wing = {"groupName"         : "Wing", # Notice Wing is the value for the capsGroup attribute
                "numChord"          : 48,
                "numSpanPerSection" : 22}

        flightstream.input.CCS_LiftingSurface = {"Wing": wing}

        #flightstream.preAnalysis()
        #shutil.copytree("fsAnalysisTest_Phase/Phase0/flightstream/", flightstream.analysisDir, dirs_exist_ok=True)
        #flightstream.postAnalysis()

        # Retrieve results
        Cx  = flightstream.output.Cx
        Cy  = flightstream.output.Cy
        Cz  = flightstream.output.Cz
        CL  = flightstream.output.CL
        CDi = flightstream.output.CDi
        CDo = flightstream.output.CDo
        CMx = flightstream.output.CMx
        CMy = flightstream.output.CMy
        CMz = flightstream.output.CMz

        capsProblem.closePhase()

        # Initialize Problem from the last phase and make a new phase
        capsProblem = pyCAPS.Problem(problemName, phaseName="Phase1", phaseStart="Phase0", outLevel=0)

        flightstream = capsProblem.analysis["flightstream"]

        # Retrieve results
        self.assertAlmostEqual(Cx , flightstream.output.Cx )
        self.assertAlmostEqual(Cy , flightstream.output.Cy )
        self.assertAlmostEqual(Cz , flightstream.output.Cz )
        self.assertAlmostEqual(CL , flightstream.output.CL )
        self.assertAlmostEqual(CDi, flightstream.output.CDi)
        self.assertAlmostEqual(CDo, flightstream.output.CDo)
        self.assertAlmostEqual(CMx, flightstream.output.CMx)
        self.assertAlmostEqual(CMy, flightstream.output.CMy)
        self.assertAlmostEqual(CMz, flightstream.output.CMz)

        # Change angle of attack
        flightstream.input.Alpha  = 3.0 * deg

        Cx  = flightstream.output.Cx
        Cy  = flightstream.output.Cy
        Cz  = flightstream.output.Cz
        CL  = flightstream.output.CL
        CDi = flightstream.output.CDi
        CDo = flightstream.output.CDo
        CMx = flightstream.output.CMx
        CMy = flightstream.output.CMy
        CMz = flightstream.output.CMz

        #print("        CxTrue  = ", Cx )
        #print("        CyTrue  = ", Cy )
        #print("        CzTrue  = ", Cz )
        #print("        CLTrue  = ", CL )
        #print("        CDiTrue = ", CDi)
        #print("        CDoTrue = ", CDo)
        #print("        CMxTrue = ", CMx)
        #print("        CMyTrue = ", CMy)
        #print("        CMzTrue = ", CMz)

        CxTrue  =  0.0088
        CyTrue  =  0.0089
        CzTrue  =  0.1518
        CLTrue  =  0.1513
        CDiTrue =  0.015
        CDoTrue =  0.0017
        CMxTrue =  0.0456
        CMyTrue =  -0.0311
        CMzTrue =  -0.0019

        self.assertAlmostEqual(CxTrue , Cx , 4)
        self.assertAlmostEqual(CyTrue , Cy , 4)
        self.assertAlmostEqual(CzTrue , Cz , 4)
        self.assertAlmostEqual(CLTrue , CL , 4)
        self.assertAlmostEqual(CDiTrue, CDi, 4)
        self.assertAlmostEqual(CDoTrue, CDo, 4)
        self.assertAlmostEqual(CMxTrue, CMx, 4)
        self.assertAlmostEqual(CMyTrue, CMy, 4)
        self.assertAlmostEqual(CMzTrue, CMz, 4)


#==============================================================================
    def run_journal(self, capsProblem, line_exit):

        verbose = False

        # Define units
        deg  = pyCAPS.Unit("degree")
        ft   = pyCAPS.Unit("ft")
        m    = pyCAPS.Unit("m")
        s    = pyCAPS.Unit("s")

        line = 0
        if line == line_exit: return line
        if line_exit > 0: self.assertTrue(capsProblem.journaling())

        # Load flightstream aim
        if verbose: print(6*"-", line, line,"Load flightstreamAIM")
        flightstream = self.capsProblem.analysis.create(aim = "flightstreamAIM"); line += 1
        if line == line_exit: return line
        if line_exit > 0: self.assertTrue(capsProblem.journaling())

        # Set new Mach/Alt parameters
        if verbose: print(6*"-", line, line,"Set FlightStream")
        flightstream.input.FlightStream  = FlightStream_exe
        if line == line_exit: return line
        if line_exit > 0: self.assertTrue(capsProblem.journaling())

        if verbose: print(6*"-", line, line,"Set Mach")
        flightstream.input.Mach  = 0.5; line += 1
        if line == line_exit: return line
        if line_exit > 0: self.assertTrue(capsProblem.journaling())

        if verbose: print(6*"-", line, line,"Set Alpha")
        flightstream.input.Alpha = 1.0 * deg; line += 1
        if line == line_exit: return line
        if line_exit > 0: self.assertTrue(capsProblem.journaling())

        if verbose: print(6*"-", line, line,"Set Beta")
        flightstream.input.Beta  = 0.0 * deg; line += 1
        if line == line_exit: return line
        if line_exit > 0: self.assertTrue(capsProblem.journaling())

        if verbose: print(6*"-", line, line,"Set Altitude")
        flightstream.input.Altitude = 1000 * ft; line += 1
        if line == line_exit: return line
        if line_exit > 0: self.assertTrue(capsProblem.journaling())

        if verbose: print(6*"-", line, line,"Set Solver_Model")
        flightstream.input.Solver_Model = "SUBSONIC_PRANDTL_GLAUERT"; line += 1
        if line == line_exit: return line
        if line_exit > 0: self.assertTrue(capsProblem.journaling())

        if verbose: print(6*"-", line, line,"Set ReferenceVelocity")
        flightstream.input.ReferenceVelocity = 295.06 * m/s; line += 1
        if line == line_exit: return line
        if line_exit > 0: self.assertTrue(capsProblem.journaling())

        wing = {"groupName"         : "Wing",
                "numChord"          : 48,
                "numSpanPerSection" : 22}

        if verbose: print(6*"-", line, line,"Set CCS_LiftingSurface")
        flightstream.input.CCS_LiftingSurface = {"Wing": wing}; line += 1
        if line == line_exit: return line
        if line_exit > 0: self.assertTrue(capsProblem.journaling())

        # Retrieve results
        if verbose: print(6*"-", line, line,"Get Cx")
        Cx  = flightstream.output.Cx; line += 1
        if line == line_exit: return line
        if line_exit > 0: self.assertTrue(capsProblem.journaling())

        if verbose: print(6*"-", line, line,"Get Cy")
        Cy  = flightstream.output.Cy; line += 1
        if line == line_exit: return line
        if line_exit > 0: self.assertTrue(capsProblem.journaling())

        if verbose: print(6*"-", line, line,"Get Cz")
        Cz  = flightstream.output.Cz; line += 1
        if line == line_exit: return line
        if line_exit > 0: self.assertTrue(capsProblem.journaling())

        CxTrue  = 0.0115
        CyTrue  = 0.0076
        CzTrue  = 0.1102

        self.assertAlmostEqual(CxTrue , Cx , 4)
        self.assertAlmostEqual(CyTrue , Cy , 4)
        self.assertAlmostEqual(CzTrue , Cz , 4)

        # make sure the last call journals everything
        return line+2

#==============================================================================
    def test_journal(self):

        capsFile = capsFile=os.path.join("..","csmData","avlSections.csm")
        problemName = self.problemName + "_Journal"

        capsProblem = pyCAPS.Problem(problemName, capsFile=capsFile, outLevel=0)

        # Run once to get the total line count
        line_total = self.run_journal(capsProblem, -1)

        capsProblem.close()
        shutil.rmtree(problemName)

        #print(80*"=")
        #print(80*"=")
        # Create the problem to start journaling
        capsProblem = pyCAPS.Problem(problemName, capsFile=capsFile, outLevel=0)
        capsProblem.close()

        for line_exit in range(line_total):
            #print(80*"=")
            capsProblem = pyCAPS.Problem(problemName, phaseName="Scratch", capsFile=capsFile, outLevel=0)
            self.run_journal(capsProblem, line_exit)
            capsProblem.close()

if __name__ == '__main__':
    unittest.main()
