#include <math.h>
#include <stdbool.h>
#include "egads.h"
#include "OpenCSM.h"


#define PARMTOL 1.e-8                  /* values in parameter space are same */
#define RELDTOL 0.10                   /* relative side distance intersection
                                          criteria to remove the vertex */
#define AREARAT 0.05                   /* minimum area ratio to accept swap */


#define PI              3.1415926535897931159979635
#define AREA2D(a,b,c)   ((a[0]-c[0])*(b[1]-c[1]) - (a[1]-c[1])*(b[0]-c[0]))
#define CROSS(a,b,c)      a[0] = (b[1]*c[2]) - (b[2]*c[1]);\
                          a[1] = (b[2]*c[0]) - (b[0]*c[2]);\
                          a[2] = (b[0]*c[1]) - (b[1]*c[0])
#define DOT(a,b)         (a[0]*b[0] + a[1]*b[1] + a[2]*b[2])
#define MIN(A,B)        (((A) < (B)) ? (A) : (B))
#define MAX(A,B)        (((A) < (B)) ? (B) : (A))


/* Quad:
      3---2
      | / |
      0---1
 */

static int sides[3][2] = {{1,2}, {2,0}, {0,1}};


/* undocumented EGADS function */
extern int EG_relPosTs( ego geom, int n, /*@null@*/ const double *rel,
                        double *ts, double *xyzs );



/* Given three collinear points p, q, r, does point q lies on line segment pr */
static bool onSegment(const double *p, const double *q, const double *r)
{
  if (q[0] <= MAX(p[0], r[0]) && q[0] >= MIN(p[0], r[0]) &&
      q[1] <= MAX(p[1], r[1]) && q[1] >= MIN(p[1], r[1])) return true;

  return false;
}


/* Finds orientation of ordered triplet (p, q, r).
    Returns: 0 -> p q & r are collinear, 1 -> Clockwise, 2 -> Counterclockwise
 */
static int orientLine(const double *p, const double *q, const double *r)
{
  double val = (q[1] - p[1])*(r[0] - q[0]) - (q[0] - p[0])*(r[1] - q[1]);

  if (val == 0.0) return 0; /* collinear */

  return (val > 0.0)? 1: 2; /* clock or counterclockwise */
}


/* Function that returns true if line segment 'p1q1' and 'p2q2' intersect */
static bool llIntersect(const double *p1, const double *q1,
                        const double *p2, const double *q2,
                        double *x, bool *overlap)
{
  int    o1, o2, o3, o4;
  double a1, b1, c1, a2, b2, c2, determinant;
  
  *overlap = true;
  
  /* Find the four orientations needed for all cases */
  o1 = orientLine(p1, q1, p2);
  o2 = orientLine(p1, q1, q2);
  o3 = orientLine(p2, q2, p1);
  o4 = orientLine(p2, q2, q1);

  /* General case */
  if (o1 != o2 && o3 != o4) {
    a1 = q1[1] - p1[1];
    b1 = p1[0] - q1[0];
    c1 = a1 * p1[0] + b1 * p1[1];

    a2 = q2[1] - p2[1];
    b2 = p2[0] - q2[0];
    c2 = a2 * p2[0] + b2 * p2[1];

    determinant = a1 * b2 - a2 * b1;
    if (determinant != 0.0) {
      x[0] = (c1 * b2 - c2 * b1) / determinant;
      x[1] = (a1 * c2 - a2 * c1) / determinant;
      *overlap = false;
    }
    return true;
  }

  /* Special cases
     p1, q1 and p2 are collinear and p2 lies on segment p1q1 */
  if (o1 == 0 && onSegment(p1, p2, q1)) return true;

  /* p1, q1 and q2 are collinear and q2 lies on segment p1q1 */
  if (o2 == 0 && onSegment(p1, q2, q1)) return true;

  /* p2, q2 and p1 are collinear and p1 lies on segment p2q2 */
  if (o3 == 0 && onSegment(p2, p1, q2)) return true;

  /* p2, q2 and q1 are collinear and q1 lies on segment p2q2 */
  if (o4 == 0 && onSegment(p2, q1, q2)) return true;

  return false; /* Doesn't fall in any of the above cases */
}


/* g1/g2 is the grid segment; e1/e2 is the edge segmnent
   returns -1 for no intersection, or relative distance along g1/g2 */
static double lineGridIntersect(const double *g1, const double *g2,
                                const double *e1, const double *e2)
{
  bool   overlap;
  double xi[2], disx, dist;
  
  if (!llIntersect(g1, g2, e1, e2, xi, &overlap)) return -1.0;
  if (overlap) {
    dist = sqrt((g1[0]-e1[0])*(g1[0]-e1[0]) + (g1[1]-e1[1])*(g1[1]-e1[1]));
    if (dist < PARMTOL) return 0.0;
    dist = sqrt((g1[0]-e2[0])*(g1[0]-e2[0]) + (g1[1]-e2[1])*(g1[1]-e2[1]));
    if (dist < PARMTOL) return 0.0;
    dist = sqrt((g2[0]-e1[0])*(g2[0]-e1[0]) + (g2[1]-e1[1])*(g2[1]-e1[1]));
    if (dist < PARMTOL) return 1.0;
    dist = sqrt((g2[0]-e2[0])*(g2[0]-e2[0]) + (g2[1]-e2[1])*(g2[1]-e2[1]));
    if (dist < PARMTOL) return 1.0;
    printf(" Info: full overlap (lineGridIntersect)!\n");
    return -1.0;
  }

  dist = sqrt((g1[0]-g2[0])*(g1[0]-g2[0]) + (g1[1]-g2[1])*(g1[1]-g2[1]));
  disx = sqrt((g1[0]-xi[0])*(g1[0]-xi[0]) + (g1[1]-xi[1])*(g1[1]-xi[1]));
  return disx/dist;
}


static void gridEdgeIntersect(ego face, ego edge, int nt, const double *ts,
                              int nu, int nv, const double *uvs, bool *grid)
{
  int    i, j, k, i0, status;
  double *eUVs, uv0[2], uv1[2], uv2[2], uv3[2], frac;
  
  eUVs = (double *) EG_alloc(2*nt*sizeof(double));
  if (eUVs == NULL) {
    printf(" Error: Allocating %d doubles (gridEdgeIntersect)!\n", 2*nt);
    return;
  }
  for (k = 0; k < nt; k++) {
    status = EG_getEdgeUV(face, edge, 0, ts[k], &eUVs[2*k]);
    if (status != EGADS_SUCCESS)
      (void) EG_getEdgeUV(face, edge, 1, ts[k], &eUVs[2*k]);
  }
  
  for (k = 0; k < nt-1; k++)
    for (j = 1; j < nv-2; j++)
      for (i = 1; i < nu-2; i++) {
        i0     = j*nu + i;
        uv0[0] = uvs[2* i0        ];
        uv0[1] = uvs[2* i0      +1];
        uv1[0] = uvs[2*(i0   +1)  ];
        uv1[1] = uvs[2*(i0   +1)+1];
        uv2[0] = uvs[2*(i0+nu+1)  ];
        uv2[1] = uvs[2*(i0+nu+1)+1];
        uv3[0] = uvs[2*(i0+nu  )  ];
        uv3[1] = uvs[2*(i0+nu  )+1];
        if (grid[i0] || grid[i0+1]) {
          frac = lineGridIntersect(uv0, uv1, &eUVs[2*k], &eUVs[2*k+2]);
          if (frac >= 0.0) {
            if (frac < RELDTOL) {
              if (grid[i0]) {
                grid[i0] = false;
#ifdef DEBUG
                printf("           01 %3d %3d  %lf\n", i,   j, frac);
#endif
              }
            } else if (frac > 1.0-RELDTOL) {
              if (grid[i0+1]) {
                grid[i0+1] = false;
#ifdef DEBUG
                printf("           01 %3d %3d  %lf\n", i+1, j, frac);
#endif
              }
            }
          }
        }
        if (grid[i0+1] || grid[i0+nu+1]) {
          frac = lineGridIntersect(uv1, uv2, &eUVs[2*k], &eUVs[2*k+2]);
          if (frac >= 0.0) {
            if (frac < RELDTOL) {
              if (grid[i0+1]) {
                grid[i0+1] = false;
#ifdef DEBUG
                printf("           12 %3d %3d  %lf\n", i+1, j,   frac);
#endif
              }
            } else if (frac > 1.0-RELDTOL) {
              if (grid[i0+nu+1]) {
                grid[i0+nu+1] = false;
#ifdef DEBUG
                printf("           12 %3d %3d  %lf\n", i+1, j+1, frac);
#endif
              }
            }
          }
        }
        if (grid[i0+nu+1] || grid[i0+nu]) {
          frac = lineGridIntersect(uv2, uv3, &eUVs[2*k], &eUVs[2*k+2]);
          if (frac >= 0.0) {
            if (frac < RELDTOL) {
              if (grid[i0+nu+1]) {
                grid[i0+nu+1] = false;
#ifdef DEBUG
                printf("           23 %3d %3d  %lf\n", i, j+1, frac);
#endif
              }
            } else if (frac > 1.0-RELDTOL) {
              if (grid[i0+nu]) {
                grid[i0+nu] = false;
#ifdef DEBUG
                printf("           23 %3d %3d  %lf\n", i, j+1, frac);
#endif
              }
            }
          }
        }
        if (grid[i0+nu] || grid[i0]) {
          frac = lineGridIntersect(uv3, uv0, &eUVs[2*k], &eUVs[2*k+2]);
          if (frac >= 0.0) {
            if (frac < RELDTOL) {
              if (grid[i0+nu]) {
                grid[i0+nu] = false;
#ifdef DEBUG
                printf("           30 %3d %3d  %lf\n", i, j+1, frac);
#endif
              }
            } else if (frac > 1.0-RELDTOL) {
              if (grid[i0]) {
                grid[i0] = false;
#ifdef DEBUG
                printf("           30 %3d %3d  %lf\n", i, j,   frac);
#endif
              }
            }
          }
        }
      }
  
  EG_free(eUVs);
}


static void closeGridVerts(ego body, ego tess, ego face, int nu, int nv,
                           const double *uvs, bool *grid)
{
  int          status, oclass, mtype, nLoop, iLoop, nEdge, iEdge, eIndex, nt;
  int          atype, alen, *senses;
  double       frange[4];
  ego          geom, *loops, *edges;
  const int    *ints;
  const char   *string;
  const double *reals, *xyzs, *ts;
  
  status = EG_getTopology(face, &geom, &oclass, &mtype, frange, &nLoop,
                          &loops, &senses);
  if (status != EGADS_SUCCESS) {
    printf(" Error: EG_getTopology Face = %d (closeGridVerts)!\n", status);
    return;
  }
  
  for (iLoop = 0; iLoop < nLoop; iLoop++) {
    status = EG_getTopology(loops[iLoop], &geom, &oclass, &mtype, NULL,
                           &nEdge, &edges, &senses);
    if (status != EGADS_SUCCESS) {
      printf(" Warning: EG_getTopology Loop = %d (closeGridVerts)!\n",
             status);
      continue;
    }
    for (iEdge = 0; iEdge < nEdge; iEdge++) {
      if (edges[iEdge]->mtype == DEGENERATE) continue;
      status = EG_attributeRet(edges[iEdge], "meshT", &atype, &alen,
                               &ints, &reals, &string);
      if (status == EGADS_SUCCESS) continue;
      eIndex = EG_indexBodyTopo(body, edges[iEdge]);
      status = EG_getTessEdge(tess, eIndex, &nt, &xyzs, &ts);
      if (status == EGADS_SUCCESS)
        gridEdgeIntersect(face, edges[iEdge], nt, ts, nu, nv, uvs, grid);
    }
  }
}


/* find the quad that contains the specified uv */
static void inQuad(double *uv, int nu, int nv, const double *uvs, int *ij)
{
  int    i, j, status, i0, nij[2];
  double w[3], uv0[2], uv1[2], uv2[2], uv3[2], wneg, wtneg;
  
  wtneg  = -1.e300;
  nij[0] = nij[1] = -1;
  ij[0]  = ij[1]  = -1;
  for (j = 0; j < nv-1; j++) {
    for (i = 0; i < nu-1; i++) {
      i0     = j *nu + i;
      uv0[0] = uvs[2* i0        ];
      uv0[1] = uvs[2* i0      +1];
      uv1[0] = uvs[2*(i0   +1)  ];
      uv1[1] = uvs[2*(i0   +1)+1];
      uv2[0] = uvs[2*(i0+nu+1)  ];
      uv2[1] = uvs[2*(i0+nu+1)+1];
      uv3[0] = uvs[2*(i0+nu  )  ];
      uv3[1] = uvs[2*(i0+nu  )+1];
      status = EG_inTriExact(uv0, uv1, uv2, uv, w);
      if (status == EGADS_SUCCESS) {
        ij[0] = i;
        ij[1] = j;
        return;
      } else if (status == EGADS_OUTSIDE) {
        wneg = w[0];
        if (wneg > w[1]) wneg = w[1];
        if (wneg > w[2]) wneg = w[2];
        if (wneg > wtneg) {
          wtneg  = wneg;
          nij[0] = i;
          nij[1] = j;
        }
      }
      status = EG_inTriExact(uv0, uv2, uv3, uv, w);
      if (status == EGADS_SUCCESS) {
        ij[0] = i;
        ij[1] = j;
        return;
      } else if (status == EGADS_OUTSIDE) {
        wneg = w[0];
        if (wneg > w[1]) wneg = w[1];
        if (wneg > w[2]) wneg = w[2];
        if (wneg > wtneg) {
          wtneg  = wneg;
          nij[0] = i;
          nij[1] = j;
        }
      }
    }
  }
  /* negative */
  ij[0] = nij[0];
  ij[1] = nij[1];
  printf(" Info: weight = %le, %d %d of %dx%d (inQuad)!\n",
         wtneg, ij[0], ij[1], nu, nv);
}


/* tessellate an Edge that bounds and is complete */
static int fullEdge(ego tess, int eIndex, ego edge, int nt, const double *ts)
{
  int    i, status, oclass, mtype, nnodes, *senses;
  double *xyzs, trange[2], results[9];
  ego    geom, *nodes;
  
  if (edge->mtype == DEGENERATE) return EGADS_SUCCESS;
  xyzs = (double *) EG_alloc(3*nt*sizeof(double));
  if (xyzs == NULL) {
    printf(" Error: Allocating %d coordinates (fullEdge)!\n", nt);
    return EGADS_MALLOC;
  }
  
  status = EG_getTopology(edge, &geom, &oclass, &mtype, trange, &nnodes,
                          &nodes, &senses);
  if (status != EGADS_SUCCESS) {
    printf(" Error: EG_getTopology = %d (fullEdge)!\n", status);
    EG_free(xyzs);
    return status;
  }
  status = EG_evaluate(nodes[0], NULL, xyzs);
  if (status != EGADS_SUCCESS) {
    printf(" Error: EG_evaluate Node 0 = %d (fullEdge)!\n", status);
    EG_free(xyzs);
    return status;
  }
  i = 0;
  if (nnodes == 2) i = 1;
  status = EG_evaluate(nodes[i], NULL, &xyzs[3*nt-3]);
  if (status != EGADS_SUCCESS) {
    printf(" Error: EG_evaluate Node %d = %d (fullEdge)!\n", i, status);
    EG_free(xyzs);
    return status;
  }
  
  for (i = 1; i < nt-1; i++) {
    status = EG_evaluate(edge, &ts[i], results);
    if (status != EGADS_SUCCESS) {
      printf(" Error: EG_evaluate Edge %d = %d (fullEdge)!\n", i, status);
      EG_free(xyzs);
      return status;
    }
    xyzs[3*i  ] = results[0];
    xyzs[3*i+1] = results[1];
    xyzs[3*i+2] = results[2];
  }

  status = EG_setTessEdge(tess, eIndex, nt, xyzs, ts);
  if (status != EGADS_SUCCESS)
    printf(" Error: EG_setTessEdge = %d (fullEdge)!\n", status);
  
  EG_free(xyzs);
  return status;
}


/* tessellate an Edge that bounds and is clipped */
static int partialEdge(ego tess, int eIndex, ego edge, int nt, const double *ts)
{
  int    i, status, oclass, mtype, nnodes, ne, beg, end, *senses;
  double *xyzs, *tes, ratio, trange[2], results[9];
  ego    geom, *nodes;
  
  if (edge->mtype == DEGENERATE) return EGADS_SUCCESS;
  
  /* find the count */
  status = EG_getTopology(edge, &geom, &oclass, &mtype, trange, &nnodes,
                          &nodes, &senses);
  if (status != EGADS_SUCCESS) {
    printf(" Error: EG_getTopology = %d (partialEdge)!\n", status);
    return status;
  }
  beg = 0;
  if (fabs(ts[beg]-trange[0]) > PARMTOL)
    for (beg = 1; beg < nt; beg++)
      if (trange[0] < ts[beg]) break;
  if (beg != 0) {
    ratio = fabs(ts[beg]-trange[0])/(ts[beg]-ts[beg-1]);
    if (ratio < RELDTOL) {
#ifdef DEBUG
      printf(" Edge %3d: beg  %lf %lf   ratio = %lf\n",
             eIndex, trange[0], ts[beg], ratio);
#endif
      beg++;
    }
  }
  if (fabs(ts[beg]-trange[0]) <= PARMTOL) beg++;
  end = nt-1;
  if (fabs(ts[end]-trange[1]) > PARMTOL)
    for (end = nt-2; end > beg; end--)
      if (trange[1] > ts[end]) break;
  if (end != nt-1) {
    ratio = fabs(ts[end]-trange[1])/(ts[end+1]-ts[end]);
    if (ratio < RELDTOL) {
#ifdef DEBUG
      printf(" Edge %3d: end  %lf %lf   ratio = %lf\n",
             eIndex, ts[end], trange[1], ratio);
#endif
      end--;
    }
  }
  if (fabs(ts[end]-trange[1]) <= PARMTOL) end--;
  
  xyzs = (double *) EG_alloc(3*nt*sizeof(double));
  if (xyzs == NULL) {
    printf(" Error: Allocating %d coordinates (partialEdge)!\n", nt);
    return EGADS_MALLOC;
  }
  tes = (double *) EG_alloc(nt*sizeof(double));
  if (tes == NULL) {
    EG_free(xyzs);
    printf(" Error: Allocating %d coordinates (partialEdge)!\n", nt);
    return EGADS_MALLOC;
  }

  status = EG_evaluate(nodes[0], NULL, xyzs);
  if (status != EGADS_SUCCESS) {
    printf(" Error: EG_evaluate Node 0 = %d (partialEdge)!\n", status);
    EG_free(tes);
    EG_free(xyzs);
    return status;
  }
  tes[0] = trange[0];
  
  ne = 1;
  for (i = beg; i <= end; i++, ne++) {
    tes[ne] = ts[i];
    status = EG_evaluate(edge, &ts[i], results);
    if (status != EGADS_SUCCESS) {
      printf(" Error: EG_evaluate Edge %d = %d (partialEdge)!\n", i, status);
      EG_free(tes);
      EG_free(xyzs);
      return status;
    }
    xyzs[3*ne  ] = results[0];
    xyzs[3*ne+1] = results[1];
    xyzs[3*ne+2] = results[2];
  }

  status = EG_evaluate(nodes[1], NULL, &xyzs[3*ne]);
  if (status != EGADS_SUCCESS) {
    printf(" Error: EG_evaluate Node 1 = %d (partialEdge)!\n", status);
    EG_free(tes);
    EG_free(xyzs);
    return status;
  }
  tes[ne] = trange[1];
  ne++;

  status = EG_setTessEdge(tess, eIndex, ne, xyzs, tes);
  if (status != EGADS_SUCCESS)
    printf(" Error: EG_setTessEdge = %d (partialEdge)!\n", status);
  
  EG_free(tes);
  EG_free(xyzs);
  return status;
}


/* tessellate a trimming Edge (meshT not specified) */
static int trimmingEdge(ego body, ego tess, int eIndex, ego edge)
{
  int          i, k, nface, ij0[2], ij1[2], nt, status, atype, alen, nu, nv;
  int          oclass, mtype, nnodes, iper, *senses;
  double       trange[2], uvf[2], uvl[2], results[9], *ts, *xyzs;
  ego          geom, *nodes, *faces;
  const int    *ints;
  const char   *string;
  const double *reals;
  
  status = EG_getTopology(edge, &geom, &oclass, &mtype, trange, &nnodes,
                          &nodes, &senses);
  if (status != EGADS_SUCCESS) {
    printf(" Error: EG_getTopology = %d (trimmingEdge)!\n", status);
    return status;
  }

  /* is the spacing prescribed (meshTint) ? */
  status = EG_attributeRet(edge, "meshTint", &atype, &nt, &ints, &reals,
                           &string);
  if (status == EGADS_SUCCESS) {
    /* yes -- use it */
    if ((fabs(trange[0]-reals[0])    < PARMTOL) &&
        (fabs(trange[1]-reals[nt-1]) < PARMTOL)) {
      xyzs = (double *) EG_alloc(3*nt*sizeof(double));
      if (xyzs == NULL) {
        printf(" Error: Allocating %d coordinates (trimmingEdge)!\n", nt);
        return EGADS_MALLOC;
      }
      
      status = EG_evaluate(nodes[0], NULL, xyzs);
      if (status != EGADS_SUCCESS) {
        printf(" Error: EG_evaluate Node 0 = %d (trimmingEdge)!\n", status);
        EG_free(xyzs);
        return status;
      }
      i = 0;
      if (nnodes == 2) i = 1;
      status = EG_evaluate(nodes[i], NULL, &xyzs[3*nt-3]);
      if (status != EGADS_SUCCESS) {
        printf(" Error: EG_evaluate Node %d = %d (trimmingEdge)!\n", i, status);
        EG_free(xyzs);
        return status;
      }
      
      for (i = 1; i < nt-1; i++) {
        status = EG_evaluate(edge, &reals[i], results);
        if (status != EGADS_SUCCESS) {
          printf(" Error: EG_evaluate Edge %d = %d (trimmingEdge)!\n",
                 i, status);
          EG_free(xyzs);
          return status;
        }
        xyzs[3*i  ] = results[0];
        xyzs[3*i+1] = results[1];
        xyzs[3*i+2] = results[2];
      }

      status = EG_setTessEdge(tess, eIndex, nt, xyzs, reals);
      if (status != EGADS_SUCCESS)
        printf(" Error: EG_setTessEdge = %d (trimmingEdge)!\n", status);
      
      EG_free(xyzs);
      return status;
    }
    printf(" Warning: Edge %3d - trange = %lf %lf  attr = %lf %lf (trimmingEdge)\n",
           eIndex, trange[0], trange[1], reals[0], reals[nt-1]);
  }
  
  /* no - count the intersected quads for each Face touching the Edge */
  status = EG_getBodyTopos(body, edge, FACE, &nface, &faces);
  if (status != EGADS_SUCCESS) {
    printf(" Error: EG_getBodyTopos Edge %d = %d (trimmingEdge)!\n",
           eIndex, status);
    return status;
  }
  
  /* get number of mesh elements traversed */
  for (nt = k = 0; k < nface; k++) {
    status = EG_attributeRet(faces[k], "meshSize", &atype, &alen,
                             &ints, &reals, &string);
    if (status != EGADS_SUCCESS) continue;
    if ((alen != 2) || (atype != ATTRINT)) {
      printf(" Error: meshSize attrType = %d, attrLen = %d (trimmingEdge)!\n",
             atype, alen);
      EG_free(faces);
      return EGADS_ATTRERR;
    }
    nu = ints[0];
    nv = ints[1];
    status = EG_attributeRet(faces[k], "meshUV", &atype, &alen,
                             &ints, &reals, &string);
    if ((status != EGADS_SUCCESS) || (alen != 2*nu*nv) || (atype != ATTRREAL)) {
      printf(" Error: meshUV = %d  atype = %d  alen = %d (trimmingEdge)!\n",
             status, atype, alen);
      EG_free(faces);
      return EGADS_ATTRERR;
    }
    /* interior Edge in Face 2x and NOT exterior */
    status = EG_getEdgeUV(faces[k], edge, 0, trange[0], uvf);
    if (status == EGADS_TOPOERR) {
      status = EG_getEdgeUV(faces[k], edge, 1, trange[0], uvf);
      if (status == EGADS_SUCCESS) {
        status = EG_getEdgeUV(faces[k], edge, -1, trange[0], uvl);
        if (status == EGADS_SUCCESS) {
          if ((fabs(uvf[0]-uvl[0]) > PARMTOL) ||
              (fabs(uvf[1]-uvl[1]) > PARMTOL)) {
            printf(" Error: Called with periodic Edge (trimmingEdge)!\n");
            EG_free(faces);
            return EGADS_TOPOERR;
          }
        }
      }
    }
    if (status != EGADS_SUCCESS) {
      printf(" Error: EG_getEdgeUV first = %d (trimmingEdge)!\n", status);
      return status;
    }
    status = EG_getEdgeUV(faces[k], edge, 0, trange[1], uvl);
    if (status == EGADS_TOPOERR) {
      status = EG_getEdgeUV(faces[k], edge, 1, trange[1], uvl);
    }
    if (status != EGADS_SUCCESS) {
      printf(" Error: EG_getEdgeUV last = %d (trimmingEdge)!\n", status);
      return status;
    }
    
    inQuad(uvf, nu, nv, reals, ij0);
    inQuad(uvl, nu, nv, reals, ij1);
    iper = abs(ij0[0]-ij1[0]) + abs(ij0[1]-ij1[1]) + 2;
    if (iper > nt) nt = iper;
#ifdef DEBUG
    printf("  %3d: %2d %2d    %2d %2d    nt = %2d   (%d)\n",
           EG_indexBodyTopo(body, faces[k]), ij0[0], ij0[1], ij1[0], ij1[1],
           nt, k);
#endif
  }
  EG_free(faces);
  
  if (nt == 0) {
    printf(" Warning: Edge %d No Meshes found (trimmingEdge)!\n", eIndex);
    return EGADS_SUCCESS;
  }

  xyzs = (double *) EG_alloc(3*nt*sizeof(double));
  if (xyzs == NULL) {
    printf(" Error: Allocating %d coordinates (trimmingEdge)!\n", nt);
    return EGADS_MALLOC;
  }
  ts = (double *) EG_alloc(nt*sizeof(double));
  if (ts == NULL) {
    EG_free(xyzs);
    printf(" Error: Allocating %d coordinates (trimmingEdge)!\n", nt);
    return EGADS_MALLOC;
  }
  ts[0]    = trange[0];
  ts[nt-1] = trange[1];
  status   = EG_relPosTs(edge, nt, NULL, ts, xyzs);
  if (status != EGADS_SUCCESS) {
    printf(" Error: EG_relPosTs = %d (trimmingEdge)!\n", status);
    EG_free(ts);
    EG_free(xyzs);
    return status;
  }
  geom = nodes[0];
  status = EG_evaluate(geom, NULL, xyzs);
  if (status != EGADS_SUCCESS) {
    printf(" Error: EG_evaluate Node 0 = %d (trimmingEdge)!\n", status);
    EG_free(ts);
    EG_free(xyzs);
    return status;
  }
  if (nnodes != 1) geom = nodes[1];
  status = EG_evaluate(geom, NULL, &xyzs[3*nt-3]);
  if (status != EGADS_SUCCESS) {
    printf(" Error: EG_evaluate Node 1 = %d (trimmingEdge)!\n", status);
    EG_free(ts);
    EG_free(xyzs);
    return status;
  }

  status = EG_setTessEdge(tess, eIndex, nt, xyzs, ts);
  if (status != EGADS_SUCCESS)
    printf(" Error: EG_setTessEdge = %d (trimmingEdge)!\n", status);
    
  EG_free(ts);
  EG_free(xyzs);
  return status;
}


/* take the valid grid points and tell the EGADS tessellator to insert */
static int addFaceInserts(ego body, ego tess, ego face, int fIndex,
                          int nu, int nv, const double *uvs)
{
  int    i, j, len, status;
  double *uvx;
  bool   *grid;
  
  len = (nu-1)*(nv-1);
  uvx = (double *) EG_alloc(2*len*sizeof(double));
  if (uvx == NULL) {
    printf(" Warning: Allocate of %d UVs (addFaceInserts)!\n", len);
    return 0;
  }
  grid = (bool *) EG_alloc(nu*nv*sizeof(bool));
  if (grid == NULL) {
    printf(" Warning: Allocate of %d grid (addFaceInserts)!\n", nu*nv);
    EG_free(uvx);
    return 0;
  }
  for (i = 0; i < nu*nv; i++) grid[i] = true;
  
  /* check interior vertices if inFace */
  for (j = 1; j < nv-1; j++)
    for (i = 1; i < nu-1; i++) {
      status = EG_inFace(face, &uvs[2*(j*nu+i)]);
      if (status == EGADS_OUTSIDE) grid[j*nu+i] = false;
      if (status <  EGADS_SUCCESS)
        printf(" Warning: EG_inFace Face %d [%d,%d] = %d (addFaceInserts)!\n",
               fIndex, i, j, status);
    }
  
  /* remove vertices that are close to trimming */
  closeGridVerts(body, tess, face, nu, nv, uvs, grid);
  
  /* fill in the active vertices */
  len = 0;
  for (j = 1; j < nv-1; j++)
    for (i = 1; i < nu-1; i++) {
      if (!grid[j*nu+i]) continue;
      uvx[2*len  ] = uvs[2*(j*nu+i)  ];
      uvx[2*len+1] = uvs[2*(j*nu+i)+1];
      len++;
    }
  
  status = EG_attributeAdd(face, ".insert!", ATTRREAL, 2*len, NULL, uvx,
                           NULL);
  if (status != EGADS_SUCCESS)
    printf(" Warning: EG_attributeAdd %d .insert! = %d (addFaceInserts)!\n",
           fIndex, status);
  
  EG_free(grid);
  EG_free(uvx);
  return len;
}


/* compute the max angle in XYZ for a triangle */
static double maxXYZangle(const double *xyz1, const double *xyz2,
                          const double *xyz3)
{
  double cosa, sina, ang0, ang1, ang2, vec1[3], vec2[3], n[3];

  vec1[0] = xyz2[0] - xyz1[0];
  vec1[1] = xyz2[1] - xyz1[1];
  vec1[2] = xyz2[2] - xyz1[2];
  vec2[0] = xyz3[0] - xyz1[0];
  vec2[1] = xyz3[1] - xyz1[1];
  vec2[2] = xyz3[2] - xyz1[2];
  cosa    = DOT(vec1, vec2);
  CROSS(n, vec1, vec2);
  sina = sqrt(DOT(n,n));
  ang0 = atan2(sina, cosa);

  vec1[0] = -vec1[0];
  vec1[1] = -vec1[1];
  vec1[2] = -vec1[2];
  vec2[0] = xyz3[0] - xyz2[0];
  vec2[1] = xyz3[1] - xyz2[1];
  vec2[2] = xyz3[2] - xyz2[2];
  cosa    = DOT(vec1, vec2);
  CROSS(n, vec1, vec2);
  sina = sqrt(DOT(n,n));
  ang1 = atan2(sina, cosa);

  ang2 = PI - ang1 - ang0;
  ang0 = MAX(ang0, ang1);
  return MAX(ang0, ang2);
}


/* return a flag indicating a potential swap due to triangle dots */
static bool dotTest(const double *xyz0, const double *xyz1,
                    const double *xyz2, const double *xyz3)
{
  double x1[3], x2[3], n1[3], n2[3], dist, newd, old = -2.0;
  
  /* compare dot of normals -- pick the maximum */
  x1[0] = xyz1[0] - xyz0[0];
  x2[0] = xyz2[0] - xyz0[0];
  x1[1] = xyz1[1] - xyz0[1];
  x2[1] = xyz2[1] - xyz0[1];
  x1[2] = xyz1[2] - xyz0[2];
  x2[2] = xyz2[2] - xyz0[2];
  CROSS(n1, x1, x2);
  dist = DOT(n1, n1);
  if (dist != 0.0) {
    dist   = 1.0/sqrt(dist);
    n1[0] *= dist;
    n1[1] *= dist;
    n1[2] *= dist;

    x1[0] = xyz2[0] - xyz3[0];
    x2[0] = xyz1[0] - xyz3[0];
    x1[1] = xyz2[1] - xyz3[1];
    x2[1] = xyz1[1] - xyz3[1];
    x1[2] = xyz2[2] - xyz3[2];
    x2[2] = xyz1[2] - xyz3[2];
    CROSS(n2, x1, x2);
    dist = DOT(n2, n2);
    if (dist != 0.0) {
      dist   = 1.0/sqrt(dist);
      n2[0] *= dist;
      n2[1] *= dist;
      n2[2] *= dist;

      old = DOT(n1, n2);
    }
  }

  x1[0] = xyz3[0] - xyz1[0];
  x2[0] = xyz0[0] - xyz1[0];
  x1[1] = xyz3[1] - xyz1[1];
  x2[1] = xyz0[1] - xyz1[1];
  x1[2] = xyz3[2] - xyz1[2];
  x2[2] = xyz0[2] - xyz1[2];
  CROSS(n1, x1, x2);
  dist = DOT(n1, n1);
  if (dist == 0.0) return false;
  dist   = 1.0/sqrt(dist);
  n1[0] *= dist;
  n1[1] *= dist;
  n1[2] *= dist;

  x1[0] = xyz0[0] - xyz2[0];
  x2[0] = xyz3[0] - xyz2[0];
  x1[1] = xyz0[1] - xyz2[1];
  x2[1] = xyz3[1] - xyz2[1];
  x1[2] = xyz0[2] - xyz2[2];
  x2[2] = xyz3[2] - xyz2[2];
  CROSS(n2, x1, x2);
  dist = DOT(n2, n2);
  if (dist == 0.0) return false;
  dist   = 1.0/sqrt(dist);
  n2[0] *= dist;
  n2[1] *= dist;
  n2[2] *= dist;

  newd = DOT(n1, n2);
  if (newd > old + 1.e-6) return true;

  return false;
}


/* swap the 2 triangles and fix up the storage */
static void swapTriangles(int t1, int side, int t2, int *tris, int *tric)
{
  int i, i0, i1, i2, i3, n11, n12, n21, n22, os;
  
                             os = 0;
  if  (tric[3*t2+1] == t1+1) os = 1;
  if  (tric[3*t2+2] == t1+1) os = 2;
  i0  = tris[3*t1+side];
  i1  = tris[3*t1+sides[side][0]];
  i2  = tris[3*t1+sides[side][1]];
  i3  = tris[3*t2+os];
  
  n11 = tric[3*t1+sides[side][0]];
  n12 = tric[3*t1+sides[side][1]];
  if (tris[3*t2+sides[os][0]] == i1) {
    n21 = tric[3*t2+sides[os][0]];
    n22 = tric[3*t2+sides[os][1]];
  } else {
    n22 = tric[3*t2+sides[os][0]];
    n21 = tric[3*t2+sides[os][1]];
  }
  
  tris[3*t1  ] = i1;
  tris[3*t1+1] = i3;
  tris[3*t1+2] = i0;
  tric[3*t1  ] = t2+1;
  tric[3*t1+1] = n12;
  tric[3*t1+2] = n22;
  if (n22 > 0)
    for (i = 0; i < 3; i++)
      if (tric[3*(n22-1)+i] == t2+1) tric[3*(n22-1)+i] = t1+1;
 
  tris[3*t2  ] = i2;
  tris[3*t2+1] = i0;
  tris[3*t2+2] = i3;
  tric[3*t2  ] = t1+1;
  tric[3*t2+1] = n21;
  tric[3*t2+2] = n11;
  if (n11 > 0)
    for (i = 0; i < 3; i++)
      if (tric[3*(n11-1)+i] == t1+1) tric[3*(n11-1)+i] = t2+1;

}


static bool checkSwapOr(int t1, int side, int t2, int *tris, const double *uv)
{
  int    i0, i1, i2, i3;
  double a1, a2, rArea;

  /* check if the orientation for the swapped pair of triangles is OK */
  i0 = tris[3*t1+side];
  i1 = tris[3*t1+sides[side][0]];
  i2 = tris[3*t1+sides[side][1]];
  i3 = tris[3*t2  ] + tris[3*t2+1] + tris[3*t2+2] - i1 - i2;
  i0--; i1--; i2--; i3--;
 
  a1 = ((uv[2*i0  ]-uv[2*i3  ])*(uv[2*i1+1]-uv[2*i3+1]) -
        (uv[2*i0+1]-uv[2*i3+1])*(uv[2*i1  ]-uv[2*i3  ]));
  a2 = ((uv[2*i0  ]-uv[2*i2  ])*(uv[2*i3+1]-uv[2*i2+1]) -
        (uv[2*i0+1]-uv[2*i2+1])*(uv[2*i3  ]-uv[2*i2  ]));

  if (a1*a2 <= 0.0) return false;
  rArea = MIN(fabs(a1),fabs(a2))/MAX(fabs(a1),fabs(a2));
  if (rArea < AREARAT) {
#ifdef DEBUG
    printf(" Info: Check fail -- small relative area = %le (checkSwapOr)\n",
           rArea);
#endif
    return false;
  }
  return true;
}


/* set the triangle in the appropriate quad */
static void markQuad(int tri, int *tris, bool *tmap, int nu, int nv,
                     int *grid, int *quads)
{
  int i, j, m, n, indx, v[4];
  
  /* remove any storage of this tri */
  for (j = 0; j < nv-1; j++)
    for (i = 0; i < nu-1; i++) {
      indx = 2*(j*(nu-1) + i);
      if ((quads[indx] == tri) && (quads[indx+1] == -1)) {
        quads[indx] = -1;
        break;
      }
    }

  /* add this triangle into quad list (if found) */
  for (j = 0; j < nv-1; j++)
    for (i = 0; i < nu-1; i++) {
      indx = 2*(j*(nu-1) + i);
      if (quads[indx]   == -2) continue;
      if (quads[indx+1] != -1) continue;
      v[0] = grid[ j   *nu + i];
      v[1] = grid[ j   *nu + i + 1];
      v[2] = grid[(j+1)*nu + i + 1];
      v[3] = grid[(j+1)*nu + i];
      if ((v[0] == -1) || (v[1] == -1) || (v[2] == -1) || (v[3] == -1)) continue;
      for (n = m = 0; m < 4; m++) {
        if (tris[3*tri  ] == v[m]) n++;
        if (tris[3*tri+1] == v[m]) n++;
        if (tris[3*tri+2] == v[m]) n++;
      }
      if (n != 3) continue;
      if (quads[indx] == -1) {
        quads[indx] = tri;
        return;
      } else if (quads[indx+1] == -1) {
        quads[indx+1]     = tri;
        tmap[tri]         = false;
        tmap[quads[indx]] = false;
        return;
      }
    }
}


/* find the triangle on a quad side closest to the target uv */
static int findTriangle(int v0, int v1, double *uvt, int ntri, int *tris,
                        int *tric, const double *uv, bool *tmap)
{
  int    i, j, n, i0, i1, i2;
  double d0, d1, uvx[2];
  
  for (i = 0; i < ntri; i++) {
    if (!tmap[i]) continue;
    for (n = j = 0; j < 3; j++) {
      if (tris[3*i+j] == v0) n++;
      if (tris[3*i+j] == v1) n++;
    }
    if (n != 2) continue;
    j = 0;
    if ((tris[3*i+1] != v0) && (tris[3*i+1] != v1)) j = 1;
    if ((tris[3*i+2] != v0) && (tris[3*i+2] != v1)) j = 2;
    j = tric[3*i+j]-1;
    if (j < 0)    return i;
    if (!tmap[j]) return i;
    
    i0     = tris[3*i  ]-1;
    i1     = tris[3*i+1]-1;
    i2     = tris[3*i+2]-1;
    uvx[0] = (uv[2*i0  ] + uv[2*i1  ] + uv[2*i2  ])/3.0;
    uvx[1] = (uv[2*i0+1] + uv[2*i1+1] + uv[2*i2+1])/3.0;
    d0     = sqrt((uvt[0]-uvx[0])*(uvt[0]-uvx[0]) +
                  (uvt[1]-uvx[1])*(uvt[1]-uvx[1]));
    i0     = tris[3*j  ]-1;
    i1     = tris[3*j+1]-1;
    i2     = tris[3*j+2]-1;
    uvx[0] = (uv[2*i0  ] + uv[2*i1  ] + uv[2*i2  ])/3.0;
    uvx[1] = (uv[2*i0+1] + uv[2*i1+1] + uv[2*i2+1])/3.0;
    d1     = sqrt((uvt[0]-uvx[0])*(uvt[0]-uvx[0]) +
                  (uvt[1]-uvx[1])*(uvt[1]-uvx[1]));
    if (d0 < d1) return i;
    return j;
  }
  
  return -1;
}

/* swap pairs of triangles back */
static void swapBack(int type, int nu, int nv, int ntri, int *tris, int *tric,
                     const double *xyz, const double *uv, int *grid, int *quads,
                     bool *tmap)
{
  int    i, i0, i1, i2, i3, iter, side, swap, nei;
  double angle, angle_now, angle_swap;
  
  /* do the swap-back passes */
  iter = 0;
  do {
    iter++;
    swap = 0;
    for (i = 0; i < ntri; i++) {
      if (!tmap[i]) continue;
      for (side = 0; side < 3; side++) {
        nei = tric[3*i+side] - 1;
        if (nei < i)  continue;
        if (!tmap[nei]) continue;
        if (!checkSwapOr(i, side, nei, tris, uv)) continue;
        i0 = tris[3*i+side];
        i1 = tris[3*i+sides[side][0]];
        i2 = tris[3*i+sides[side][1]];
        i3 = tris[3*nei] + tris[3*nei+1] + tris[3*nei+2] - i1 - i2;
        if (type == 0) {
          angle_now  = maxXYZangle(&xyz[3*i0-3], &xyz[3*i1-3], &xyz[3*i2-3]);
          angle      = maxXYZangle(&xyz[3*i1-3], &xyz[3*i3-3], &xyz[3*i2-3]);
          angle_now  = MAX(angle, angle_now);
          angle_swap = maxXYZangle(&xyz[3*i0-3], &xyz[3*i1-3], &xyz[3*i3-3]);
          angle      = maxXYZangle(&xyz[3*i0-3], &xyz[3*i3-3], &xyz[3*i2-3]);
          angle_swap = MAX(angle, angle_swap);
          if (angle_swap + 1.e-6 >=  angle_now) continue;
        } else {
          if (!dotTest(&xyz[3*i0-3], &xyz[3*i1-3], &xyz[3*i2-3], &xyz[3*i3-3]))
            continue;
        }
        swapTriangles(i, side, nei, tris, tric);
        markQuad(i,   tris, tmap, nu, nv, grid, quads);
        markQuad(nei, tris, tmap, nu, nv, grid, quads);
        swap++;
      }
    }
    printf("       Pass %d: %d nSwaps =%3d\n", iter, type, swap);
  } while (swap != 0);

}


/* swap pairs of triangles to get better grid/quad coherence */
static void swapRecover(int ntri, const int *trisx, const int *tricx,
                        const double *xyz, const double *uv, int *grid,
                        int *quads, bool *tmap, int nu, int nv,
                        const double *uvs, int **stri)
{
  int    i, j, k, l, m, n, i0, i1, i2, i3, indx, nei, swap, iter, next;
  int    v[4], tv[3], q[4], *tris, *tric, *last;
  double d, dist, uvt[2], uvx[2];
  
  tric = (int *) EG_alloc(3*ntri*sizeof(int));
  if (tric == NULL) {
    printf(" Warning: Allocate of %d tric (swapRecover)!\n", ntri);
    return;
  }
  tris = (int *) EG_alloc(3*ntri*sizeof(int));
  if (tris == NULL) {
    printf(" Warning: Allocate of %d tris (swapRecover)!\n", ntri);
    EG_free(tric);
    return;
  }
  for (i = 0; i < 3*ntri; i++) {
    tris[i] = trisx[i];
    tric[i] = tricx[i];
  }
  
  /* locate quads with a single tri & single swap */
  for (swap = j = 0; j < nv-1; j++)
    for (i = 0; i < nu-1; i++) {
      indx = 2*(j*(nu-1) + i);
      if  (quads[indx] == -2) continue;
      if ((quads[indx] == -1) || (quads[indx+1] != -1)) continue;
      v[0] = grid[ j   *nu + i];
      v[1] = grid[ j   *nu + i + 1];
      v[2] = grid[(j+1)*nu + i + 1];
      v[3] = grid[(j+1)*nu + i];
      if ((v[0] == -1) || (v[1] == -1) || (v[2] == -1) || (v[3] == -1))
        continue;
      /* find pair to swap */
#ifdef DEBUG
      printf("   quad %3d %3d -- tri = %3d\n", i, j, quads[indx]);
      printf("        %3d %3d %3d %3d\n", v[0], v[1], v[2], v[3]);
#endif
      k    = quads[indx];
      tv[1] = v[0]+v[1]+v[2]+v[3] - tris[3*k]-tris[3*k+1]-tris[3*k+2];
#ifdef DEBUG
      printf("        %3d %3d %3d ---> %3d", tris[3*k], tris[3*k+1],
             tris[3*k+2], tv[1]);
#endif
      for (nei = 0; nei < 4; nei++)
        if (tv[1] == v[nei]) break;
      m = nei - 1;
      if (m < 0) m = 3;
      tv[0] = v[m];
      m = nei + 1;
      if (m > 3) m = 0;
      tv[2] = v[m];
#ifdef DEBUG
      printf(" tvs = %3d %3d %3d\n", tv[0], tv[1], tv[2]);
#endif
      for (k = 0; k < ntri; k++) {
        if (!tmap[k]) continue;
        q[0] = tris[3*k  ];
        q[1] = tris[3*k+1];
        q[2] = tris[3*k+2];
        for (n = m = 0; m < 3; m++) {
          nei = tric[3*k+m] - 1;
          if (nei < k)  continue;
          if (!tmap[nei]) continue;
          if ((tris[3*nei  ] != q[0]) && (tris[3*nei  ] != q[1]) &&
              (tris[3*nei  ] != q[2])) q[3] = tris[3*nei  ];
          if ((tris[3*nei+1] != q[0]) && (tris[3*nei+1] != q[1]) &&
              (tris[3*nei+1] != q[2])) q[3] = tris[3*nei+1];
          if ((tris[3*nei+2] != q[0]) && (tris[3*nei+2] != q[1]) &&
              (tris[3*nei+2] != q[2])) q[3] = tris[3*nei+2];
          for (n = l = 0; l < 4; l++) {
            if (q[l] == tv[0]) n++;
            if (q[l] == tv[1]) n++;
            if (q[l] == tv[2]) n++;
          }
          if (n == 3) {
#ifdef DEBUG
            printf("        pair = %3d (side %d)  %3d", k, m, nei);
#endif
            swapTriangles(k, m, nei, tris, tric);
            for (n = l = 0; l < 3; l++) {
              if (tris[3*k  ] == tv[l]) n++;
              if (tris[3*k+1] == tv[l]) n++;
              if (tris[3*k+2] == tv[l]) n++;
            }
#ifdef DEBUG
            printf("  newCnt = %d", n);
#endif
            if (n == 3) {
#ifdef DEBUG
              printf("\n");
#endif
            } else {
              n   = k;
              k   = nei;
              nei = n;
              for (n = l = 0; l < 3; l++) {
                if (tris[3*k  ] == tv[l]) n++;
                if (tris[3*k+1] == tv[l]) n++;
                if (tris[3*k+2] == tv[l]) n++;
              }
#ifdef DEBUG
              printf(" %d\n", n);
#endif
            }
            quads[indx+1]     = k;
            tmap[k]           = false;
            tmap[quads[indx]] = false;
            markQuad(nei, tris, tmap, nu, nv, grid, quads);
            swap++;
            break;
          }
        }
        if (n == 3) break;
      }
    }
  printf("       Pass 0:   nSwaps =%3d\n", swap);

  /* locate quads with more complex swaps */
  last = (int *) EG_alloc(ntri*sizeof(int));
  if (last != NULL) for (i = 0; i < ntri; i++) last[i] = -1;
  for (iter = 0; iter < 9; iter++) {
    swap = 0;
    for (j = 0; j < nv-1; j++)
      for (i = 0; i < nu-1; i++) {
        indx = 2*(j*(nu-1) + i);
        if (quads[indx]   == -2) continue;
        if (quads[indx+1] != -1) continue;
        v[0] = grid[ j   *nu + i];
        v[1] = grid[ j   *nu + i + 1];
        v[2] = grid[(j+1)*nu + i + 1];
        v[3] = grid[(j+1)*nu + i];
        if ((v[0] == -1) || (v[1] == -1) || (v[2] == -1) || (v[3] == -1))
          continue;
        
        /* find pair to swap */
        if (quads[indx] != -1) {
          /* we have a single tri */
          k   = quads[indx];
          m   = v[0]+v[1]+v[2]+v[3] - tris[3*k]-tris[3*k+1]-tris[3*k+2];
          nei = l = -1;
          if (m == v[0]) {
            l   =  j   *nu + i;
            nei = tric[3*k+1]-1;
          }
          if (m == v[1]) {
            l   =  j   *nu + i + 1;
            nei = tric[3*k+2]-1;
          }
          if (m == v[2]) {
            l   = (j+1)*nu + i + 1;
            nei = tric[3*k  ]-1;
          }
          if (m == v[3]) {
            l   = (j+1)*nu + i;
            nei = tric[3*k+1]-1;
          }
          if (l == -1) {
            printf(" Info: Can't find vert %d in %d %d %d %d (swapRecover)!\n",
                   m, v[0], v[1], v[2], v[3]);
            continue;
          }
          if (nei < 0) continue;
          if (!tmap[nei]) continue;
          uvt[0] = uvs[2*l  ];
          uvt[1] = uvs[2*l+1];
          next   = m = -1;
          dist   = 1e200;
          for (l = 0; l < 3; l++) {
            n = tric[3*nei+l]-1;
            if (n < 0) continue;
            if (last != NULL)
              if (last[nei] == n) continue;
            if ((n == k) || !tmap[n]) continue;
            if (!checkSwapOr(nei, l, n, tris, uv)) continue;
            i0     = tris[3*n  ]-1;
            i1     = tris[3*n+1]-1;
            i2     = tris[3*n+2]-1;
            uvx[0] = (uv[2*i0  ] + uv[2*i1  ] + uv[2*i2  ])/3.0;
            uvx[1] = (uv[2*i0+1] + uv[2*i1+1] + uv[2*i2+1])/3.0;
            d      = sqrt((uvt[0]-uvx[0])*(uvt[0]-uvx[0]) +
                          (uvt[1]-uvx[1])*(uvt[1]-uvx[1]));
            if (d < dist) {
              dist = d;
              next = n;
              m    = l;
            }
          }
          if (m != -1) {
            swapTriangles(nei, m, next, tris, tric);
            markQuad(nei,  tris, tmap, nu, nv, grid, quads);
            markQuad(next, tris, tmap, nu, nv, grid, quads);
            if (last != NULL) {
              last[nei]  = next;
              last[next] = nei;
            }
            swap++;
            continue;
          }
        } else {
          /* we are empty -- swap tri into corner */
#ifdef DEBUG
          printf("        Tris = ");
#endif
          n      = 0;
          i0     =  j   *nu + i;
          i1     =  j   *nu + i + 1;
          i2     = (j+1)*nu + i + 1;
          i3     = (j+1)*nu + i;
          uvt[0] = 0.5*(uvs[2*i2  ] + uvs[2*i3  ]);
          uvt[1] = 0.5*(uvs[2*i2+1] + uvs[2*i3+1]);
          nei = findTriangle(v[0], v[1], uvt, ntri, tris, tric, uv, tmap);
          if (nei >= 0) {
#ifdef DEBUG
            printf("   %3d (01)", nei);
#endif
            q[n] = nei;
            n++;
          }
          uvt[0] = 0.5*(uvs[2*i0  ] + uvs[2*i3  ]);
          uvt[1] = 0.5*(uvs[2*i0+1] + uvs[2*i3+1]);
          nei = findTriangle(v[1], v[2], uvt, ntri, tris, tric, uv, tmap);
          if (nei >= 0) {
#ifdef DEBUG
            printf("   %3d (12)", nei);
#endif
            q[n] = nei;
            n++;
          }
          uvt[0] = 0.5*(uvs[2*i0  ] + uvs[2*i1  ]);
          uvt[1] = 0.5*(uvs[2*i0+1] + uvs[2*i1+1]);
          nei = findTriangle(v[2], v[3], uvt, ntri, tris, tric, uv, tmap);
          if (nei >= 0) {
#ifdef DEBUG
            printf("   %3d (23)", nei);
#endif
            q[n] = nei;
            n++;
          }
          uvt[0] = 0.5*(uvs[2*i2  ] + uvs[2*i1  ]);
          uvt[1] = 0.5*(uvs[2*i2+1] + uvs[2*i1+1]);
          nei = findTriangle(v[3], v[0], uvt, ntri, tris, tric, uv, tmap);
          if (nei >= 0) {
#ifdef DEBUG
            printf("   %3d (30)", nei);
#endif
            q[n] = nei;
            n++;
          }
#ifdef DEBUG
          printf("   n = %d\n", n);
#endif
          for (k = 0; k < n-1; k++) {
            nei = q[k];
            for (l = k+1; l < n; l++) {
              next = q[l];
              m    = -1;
              if (tric[3*nei  ]-1 == next) m = 0;
              if (tric[3*nei+1]-1 == next) m = 1;
              if (tric[3*nei+2]-1 == next) m = 2;
              if (m != -1)
                if (!checkSwapOr(nei, m, next, tris, uv)) m = -1;
              if (m != -1) {
                swapTriangles(nei, m, next, tris, tric);
                markQuad(nei,  tris, tmap, nu, nv, grid, quads);
                markQuad(next, tris, tmap, nu, nv, grid, quads);
                if (last != NULL) {
                  last[nei]  = next;
                  last[next] = nei;
                }
                swap++;
#ifdef DEBUG
                printf("        Swap: %3d (%d)  %3d -- %3d\n",
                       nei, m, next, quads[indx]);
#endif
                break;
              }
            }
            if (quads[indx] != -1) break;
          }
        }
      }
    printf("       Pass %d:   nSwaps =%3d\n", iter+1, swap);
    if (swap == 0) break;
  }

  if (last != NULL) EG_free(last);
  
  /* swap around the unused triangles based on angles and normals */
  swapBack(0, nu, nv, ntri, tris, tric, xyz, uv, grid, quads, tmap);
  swapBack(1, nu, nv, ntri, tris, tric, xyz, uv, grid, quads, tmap);
  
  EG_free(tric);
  *stri = tris;
}


/* make sure we know which quads are intersected by internal Edges */
static void markInternalEdges(ego body, ego tess, ego face, int nu, int nv,
                              const double *uvs, int *quads)
{
  int          i, j, k, oclass, mtype, nLoop, iLoop, nEdge, iEdge, jEdge;
  int          status, iper, eIndex, nt, indx, i0, *senses;
  double       frange[4], trange[2];
  double       uv0[2], uv1[2], uv2[2], uv3[2], uvp[2], uv[2];
  ego          geom, *loops, *edges;
  const double *xyzs, *ts;
  
  status = EG_getTopology(face, &geom, &oclass, &mtype, frange, &nLoop,
                          &loops, &senses);
  if (status != EGADS_SUCCESS) {
    printf(" Error: EG_getTopology Face = %d (markInternalEdges)!\n", status);
    return;
  }
  
  for (iLoop = 0; iLoop < nLoop; iLoop++) {
    status = EG_getTopology(loops[iLoop], &geom, &oclass, &mtype, NULL,
                           &nEdge, &edges, &senses);
    if (status != EGADS_SUCCESS) {
      printf(" Warning: EG_getTopology Loop = %d (markInternalEdges)!\n",
             status);
      continue;
    }
    for (iEdge = 0; iEdge < nEdge-1; iEdge++) {
      if (edges[iEdge]->mtype == DEGENERATE) continue;
      for (jEdge = iEdge+1; jEdge < nEdge; jEdge++) {
        if (edges[iEdge] != edges[jEdge]) continue;
        (void) EG_getRange(edges[iEdge], trange, &iper);
        status = EG_getEdgeUV(face, edges[iEdge], 0, trange[0], uv0);
        if (status != EGADS_TOPOERR) continue;
        uv0[0] = uv1[0] = uv0[1] = uv1[1] = 0.0;
        (void) EG_getEdgeUV(face, edges[iEdge], -1, trange[0], uv0);
        (void) EG_getEdgeUV(face, edges[iEdge], +1, trange[0], uv1);
        /* internal Edge? */
        if ((fabs(uv0[0]-uv1[0]) > PARMTOL) || (fabs(uv0[1]-uv1[1]) > PARMTOL))
          continue;
        /* mark intersected quads */
        eIndex = EG_indexBodyTopo(body, edges[iEdge]);
        status = EG_getTessEdge(tess, eIndex, &nt, &xyzs, &ts);
        if (status != EGADS_SUCCESS) {
          printf(" Warning: EG_getTessEdge %d = %d (markInternalEdges)!\n",
                 eIndex, status);
          continue;
        }
/*      printf("  *** Edge %d in Face twice! ***\n", eIndex);  */
        uvp[0] = uv0[0];
        uvp[1] = uv0[1];
        for (k = 0; k < nt-1; k++) {
          status = EG_getEdgeUV(face, edges[iEdge], 1, ts[k+1], uv);
          if (status != EGADS_SUCCESS) {
            printf(" Warning: EG_getEdgeUV %d %d = %d (markInternalEdges)!\n",
                   eIndex, k, status);
            continue;
          }
          for (j = 0; j < nv-1; j++)
            for (i = 0; i < nu-1; i++) {
              indx   = 2*(j*(nu-1) + i);
              i0     = j*nu + i;
              uv0[0] = uvs[2* i0        ];
              uv0[1] = uvs[2* i0      +1];
              uv1[0] = uvs[2*(i0   +1)  ];
              uv1[1] = uvs[2*(i0   +1)+1];
              uv2[0] = uvs[2*(i0+nu+1)  ];
              uv2[1] = uvs[2*(i0+nu+1)+1];
              uv3[0] = uvs[2*(i0+nu  )  ];
              uv3[1] = uvs[2*(i0+nu  )+1];
              if (lineGridIntersect(uv0, uv1, uvp, uv) >= 0.0) {
                quads[indx] = quads[indx+1] = -2;
                continue;
              }
              if (lineGridIntersect(uv1, uv2, uvp, uv) >= 0.0) {
                quads[indx] = quads[indx+1] = -2;
                continue;
              }
              if (lineGridIntersect(uv2, uv3, uvp, uv) >= 0.0) {
                quads[indx] = quads[indx+1] = -2;
                continue;
              }
              if (lineGridIntersect(uv3, uv0, uvp, uv) >= 0.0) {
                quads[indx] = quads[indx+1] = -2;
                continue;
              }
            }
          uvp[0] = uv[0];
          uvp[1] = uv[1];
        }
#ifdef DEBUG
        for (j = nv-2; j >= 0; j--) {
          printf("    ");
          for (i = 0; i < nu-1; i++)
            if (quads[2*(j*(nu-1)+i)] == -2) {
              printf(".");
            } else {
              printf("X");
          }
          printf("\n");
        }
#endif
        break;
      }
    }
  }
  
}


/* find isolated groups of quads where the triangles can be removed */
static void fillIslands(int npts, int ntri, int *tris, int *grid, int *quads,
                        bool *tmap, int nu, int nv)
{
  int  i, j, k, n, nqij, indx, indi, indj, dir, v[4], *qij;
  bool *verts;
  
  for (n = k = 0; k < ntri; k++)
    if (tmap[k]) n++;
  if (n == 0) return;
  n++;
  
  qij = (int *) EG_alloc(n*sizeof(int));
  if (qij == NULL) {
    printf(" Warning: Allocate of %d tmp ints (fillIslands)!\n", n);
    return;
  }
  verts = (bool *) EG_alloc(npts*sizeof(bool));
  if (verts == NULL) {
    printf(" Warning: Allocate of %d tmp verts (fillIslands)!\n", npts);
    EG_free(qij);
    return;
  }
  
  /* find stacks of unfilled quads */
  for (dir = 0; dir < 2; dir++) {
    for (j = 0; j < nv-1; j++) {
      for (i = 0; i < nu-1; i++) {
        indx = 2*(j*(nu-1) + i);
        if (quads[indx] == -2) continue;
        v[0] = grid[ j   *nu + i  ];
        v[1] = grid[ j   *nu + i+1];
        v[2] = grid[(j+1)*nu + i+1];
        v[3] = grid[(j+1)*nu + i  ];
        if ((v[0] == -1) || (v[1] == -1) || (v[2] == -1) || (v[3] == -1))
          continue;
        if (quads[indx+1] != -1) continue;
        for (n = 0; n < npts; n++) verts[  n   ] = false;
        for (n = 0; n <    4; n++) verts[v[n]-1] = true;
        nqij   = 1;
        qij[0] = i;
        qij[1] = j;
        if (dir == 0) {
          for (k = i+1; k < nu-1; k++) {
            indi = 2*(j*(nu-1) + k);
            if (quads[indi] == -2) continue;
            v[0] = grid[ j   *nu + k  ];
            v[1] = grid[ j   *nu + k+1];
            v[2] = grid[(j+1)*nu + k+1];
            v[3] = grid[(j+1)*nu + k  ];
            if ((v[0] == -1) || (v[1] == -1) || (v[2] == -1) || (v[3] == -1))
              break;
            if (quads[indi+1] != -1) break;
            for (n = 0; n < 4; n++) verts[v[n]-1] = true;
            qij[2*nqij  ] = k;
            qij[2*nqij+1] = j;
            nqij++;
          }
        } else {
          for (k = j+1; k < nv-1; k++) {
            indj = 2*(k*(nu-1) + i);
            if (quads[indj] == -2) continue;
            v[0] = grid[ k   *nu + i  ];
            v[1] = grid[ k   *nu + i+1];
            v[2] = grid[(k+1)*nu + i+1];
            v[3] = grid[(k+1)*nu + i  ];
            if ((v[0] == -1) || (v[1] == -1) || (v[2] == -1) || (v[3] == -1))
              break;
            if (quads[indj+1] != -1) break;
            for (n = 0; n < 4; n++) verts[v[n]-1] = true;
            qij[2*nqij  ] = i;
            qij[2*nqij+1] = k;
            nqij++;
          }
        }
        if (nqij < 2) continue;
        /* check the verts */
        for (n = k = 0; k < ntri; k++) {
          if (!tmap[k]) continue;
          if (!verts[tris[3*k  ]-1]) continue;
          if (!verts[tris[3*k+1]-1]) continue;
          if (!verts[tris[3*k+2]-1]) continue;
          n++;
        }
        if (2*nqij != n) continue;
#ifdef DEBUG
        printf("       *** stack len  %d = %d   ntris = %d ***\n",
               dir, nqij, n);
#endif
        /* we have a stack -- adjust the quad & tri markers */
        for (k = 0; k < nqij; k++) {
          indi = 2*(qij[2*k+1]*(nu-1) + qij[2*k]);
          /* just set them to anything! */
          quads[indi  ] = 0;
          quads[indi+1] = 0;
        }
        for (k = 0; k < ntri; k++) {
          if (!tmap[k]) continue;
          if (!verts[tris[3*k  ]-1]) continue;
          if (!verts[tris[3*k+1]-1]) continue;
          if (!verts[tris[3*k+2]-1]) continue;
          tmap[k] = false;
        }
      }
    }
  }
  
  EG_free(verts);
  EG_free(qij);
}


/* do we have a triangle with this side? */
static int triSide(int ntri, int *tris, bool *tmap, int v0, int v1)
{
  int k;
  
  for (k = 0; k < ntri; k++) {
    if (!tmap[k]) continue;
    if ((tris[3*k  ] == v0) || (tris[3*k+1] == v0) || (tris[3*k+2] == v0))
      if ((tris[3*k  ] == v1) || (tris[3*k+1] == v1) || (tris[3*k+2] == v1))
        return k;
  }
  
  return -1;
}


/* find quad stacks where the triangles can be compressed to make more quads */
static int xtendStacks(int frame, int ntri, int *tris, const double *uv,
                       int *grid, int *quads, bool *tmap, int nu, int nv,
                       const double *uvs)
{
  int    i, is, ie, j, js, je, k, m, n, nqij, indx, indi, indj, dir, nx = 0;
  int    i0, i1, i2, t0, t1, cnt, ij[2], v[4], qij[6], srcVert[2], tgtVert[2];
  bool   *sTris;
  double uvc[2], uv0[2], uv1[2], uv2[2], area, newArea;
  
  for (n = k = 0; k < ntri; k++)
    if (tmap[k]) n++;
  if (n == 0) return nx;

  sTris = (bool *) EG_alloc(ntri*sizeof(bool));
  if (sTris == NULL) {
    printf(" Warning: Allocate of %d tri flags (xtendStacks)!\n", ntri);
    return nx;
  }
  
  /* find unfilled quad */
  for (dir = 0; dir < 2; dir++) {
    for (j = 0; j < nv-1; j++) {
      for (i = 0; i < nu-1; i++) {
        indx = 2*(j*(nu-1) + i);
        if (quads[indx] == -2) continue;
        v[0] = grid[ j   *nu + i  ];
        v[1] = grid[ j   *nu + i+1];
        v[2] = grid[(j+1)*nu + i+1];
        v[3] = grid[(j+1)*nu + i  ];
        if ((v[0] == -1) || (v[1] == -1) || (v[2] == -1) || (v[3] == -1))
          continue;
        if (quads[indx+1] != -1) continue;
        
        /* unfilled quad - neighbor in our direction must be a quad */
        nqij    = 1;
        qij[0]  = i;
        qij[1]  = j;
        is = js = -1;
        ie = je =  1;
        if (i ==    0) is =  1;
        if (i == nu-2) ie = -1;
        if (j ==    0) js =  1;
        if (j == nv-2) je = -1;
        if (dir == 0) {
          for (m = is; m <= ie; m+=2) {
            k    = i + m;
            indi = 2*(j*(nu-1) + k);
            if (quads[indi] == -2) continue;
            v[0] = grid[ j   *nu + k  ];
            v[1] = grid[ j   *nu + k+1];
            v[2] = grid[(j+1)*nu + k+1];
            v[3] = grid[(j+1)*nu + k  ];
            if ((v[0] == -1) || (v[1] == -1) || (v[2] == -1) || (v[3] == -1))
              continue;
            if (quads[indi+1] == -1) continue;
            /* do we have the sides lined up? */
            t0 = triSide(ntri, tris, tmap, grid[ j   *nu + i  ],
                                           grid[ j   *nu + i+1]);
            t1 = triSide(ntri, tris, tmap, grid[(j+1)*nu + i+1],
                                           grid[(j+1)*nu + i  ]);
            if ((t0 == -1) || (t1 == -1)) continue;
            qij[2*nqij  ] = k;
            qij[2*nqij+1] = j;
            nqij++;
          }
        } else {
          for (m = js; m <= je; m+=2) {
            k    = j + m;
            indj = 2*(k*(nu-1) + i);
            if (quads[indj] == -2) continue;
            v[0] = grid[ k   *nu + i  ];
            v[1] = grid[ k   *nu + i+1];
            v[2] = grid[(k+1)*nu + i+1];
            v[3] = grid[(k+1)*nu + i  ];
            if ((v[0] == -1) || (v[1] == -1) || (v[2] == -1) || (v[3] == -1))
              continue;
            if (quads[indj+1] == -1) continue;
            /* do we have the sides lined up? */
            t0 = triSide(ntri, tris, tmap, grid[ j   *nu + i  ],
                                           grid[(j+1)*nu + i  ]);
            t1 = triSide(ntri, tris, tmap, grid[ j   *nu + i+1],
                                           grid[(j+1)*nu + i+1]);
            if ((t0 == -1) || (t1 == -1)) continue;
            qij[2*nqij  ] = i;
            qij[2*nqij+1] = k;
            nqij++;
          }
        }
        if (nqij != 2) continue;
        
        /* make a list of triangles in our quad stack */
        if (dir == 0) {
          for (k = 0; k < ntri; k++) {
            sTris[k] = false;
            if (!tmap[k]) continue;
            i0     = tris[3*k  ]-1;
            i1     = tris[3*k+1]-1;
            i2     = tris[3*k+2]-1;
            uvc[0] = (uv[2*i0  ] + uv[2*i1  ] + uv[2*i2  ])/3.0;
            uvc[1] = (uv[2*i0+1] + uv[2*i1+1] + uv[2*i2+1])/3.0;
            inQuad(uvc, nu, nv, uvs, ij);
            if (ij[1] != j) continue;
            sTris[k] = true;
/*          printf(" tri %3d -- %3d %3d %3d    %3d %3d\n",
                   k+1, i0+1, i1+1, i2+1, ij[0], ij[1]);  */
          }
          if (qij[2] == i-1) {
            /* quad is behind - look positive i */
            srcVert[0] = grid[ j   *nu + i  ];
            tgtVert[0] = grid[ j   *nu + i+1];
            srcVert[1] = grid[(j+1)*nu + i  ];
            tgtVert[1] = grid[(j+1)*nu + i+1];
          } else {
            /* quad is infront - look negative i */
            srcVert[0] = grid[ j   *nu + i+1];
            tgtVert[0] = grid[ j   *nu + i  ];
            srcVert[1] = grid[(j+1)*nu + i+1];
            tgtVert[1] = grid[(j+1)*nu + i  ];
          }
        } else {
          for (k = 0; k < ntri; k++) {
            sTris[k] = false;
            if (!tmap[k]) continue;
            i0     = tris[3*k  ]-1;
            i1     = tris[3*k+1]-1;
            i2     = tris[3*k+2]-1;
            uvc[0] = (uv[2*i0  ] + uv[2*i1  ] + uv[2*i2  ])/3.0;
            uvc[1] = (uv[2*i0+1] + uv[2*i1+1] + uv[2*i2+1])/3.0;
            inQuad(uvc, nu, nv, uvs, ij);
            if (ij[0] != i) continue;
            sTris[k] = true;
 /*         printf(" tri %3d -- %3d %3d %3d    %3d %3d\n",
                   k+1, i0+1, i1+1, i2+1, ij[0], ij[1]);  */
          }
          if (qij[3] == j-1) {
            /* quad is behind - look positive j */
            srcVert[0] = grid[ j   *nu + i  ];
            tgtVert[0] = grid[(j+1)*nu + i  ];
            srcVert[1] = grid[ j   *nu + i+1];
            tgtVert[1] = grid[(j+1)*nu + i+1];
          } else {
            /* quad is infront - look negative j */
            srcVert[0] = grid[(j+1)*nu + i  ];
            tgtVert[0] = grid[ j   *nu + i  ];
            srcVert[1] = grid[(j+1)*nu + i+1];
            tgtVert[1] = grid[ j   *nu + i+1];
          }
        }
#ifdef DEBUG
        printf("  **** dir = %d ", dir);
        for (k = 0; k < nqij; k++) printf(" [%2d,%2d]", qij[2*k  ], qij[2*k+1]);
        printf("  srcVert = %3d %3d   tgtVert = %3d %3d ****\n", srcVert[0],
               srcVert[1], tgtVert[0], tgtVert[1]);
#endif
        if ((tgtVert[0] < frame) || (tgtVert[1] < frame)) continue;
        
        /* check stack triangles -- find collapsed & insure areas do not filp */
        for (cnt = k = 0; k < ntri; k++) {
          if (!sTris[k]) continue;
          if ((tris[3*k  ] == srcVert[0]) || (tris[3*k+1] == srcVert[0]) ||
              (tris[3*k+2] == srcVert[0]))
            if ((tris[3*k  ] == tgtVert[0]) || (tris[3*k+1] == tgtVert[0]) ||
                (tris[3*k+2] == tgtVert[0])) {
/*            printf(" tri %3d = %3d %3d %3d\n",
                     k+1, tris[3*k], tris[3*k+1], tris[3*k+2]);  */
              cnt++;
            } else {
              i0     = tris[3*k  ]-1;
              i1     = tris[3*k+1]-1;
              i2     = tris[3*k+2]-1;
              uv0[0] = uv[2*i0  ];
              uv0[1] = uv[2*i0+1];
              uv1[0] = uv[2*i1  ];
              uv1[1] = uv[2*i1+1];
              uv2[0] = uv[2*i2  ];
              uv2[1] = uv[2*i2+1];
              area   = AREA2D(uv0, uv1, uv2);
              if (tris[3*k  ] == srcVert[0]) {
                i0      = tgtVert[0]-1;
                uv0[0]  = uv[2*i0  ];
                uv0[1]  = uv[2*i0+1];
              }
              if (tris[3*k+1] == srcVert[0]) {
                i1     = tgtVert[0]-1;
                uv1[0] = uv[2*i1  ];
                uv1[1] = uv[2*i1+1];
              }
              if (tris[3*k+2] == srcVert[0]) {
                i2     = tgtVert[0]-1;
                uv2[0] = uv[2*i2  ];
                uv2[1] = uv[2*i2+1];
              }
              newArea = AREA2D(uv0, uv1, uv2);
              if (area*newArea <= 0.0) {
/*              printf(" Area flip0 @ %d %d  %lf %lf\n", i, j, area, newArea);*/
                cnt = -1;
                break;
              }
            }
          if ((tris[3*k  ] == srcVert[1]) || (tris[3*k+1] == srcVert[1]) ||
              (tris[3*k+2] == srcVert[1]))
            if ((tris[3*k  ] == tgtVert[1]) || (tris[3*k+1] == tgtVert[1]) ||
                (tris[3*k+2] == tgtVert[1])) {
/*            printf(" tri %3d = %3d %3d %3d\n",
                     k+1, tris[3*k], tris[3*k+1], tris[3*k+2]);  */
              cnt++;
            } else {
              i0     = tris[3*k  ]-1;
              i1     = tris[3*k+1]-1;
              i2     = tris[3*k+2]-1;
              uv0[0] = uv[2*i0  ];
              uv0[1] = uv[2*i0+1];
              uv1[0] = uv[2*i1  ];
              uv1[1] = uv[2*i1+1];
              uv2[0] = uv[2*i2  ];
              uv2[1] = uv[2*i2+1];
              area   = AREA2D(uv0, uv1, uv2);
              if (tris[3*k  ] == srcVert[1]) {
                i0      = tgtVert[1]-1;
                uv0[0]  = uv[2*i0  ];
                uv0[1]  = uv[2*i0+1];
              }
              if (tris[3*k+1] == srcVert[1]) {
                i1     = tgtVert[1]-1;
                uv1[0] = uv[2*i1  ];
                uv1[1] = uv[2*i1+1];
              }
              if (tris[3*k+2] == srcVert[1]) {
                i2     = tgtVert[1]-1;
                uv2[0] = uv[2*i2  ];
                uv2[1] = uv[2*i2+1];
              }
              newArea = AREA2D(uv0, uv1, uv2);
              if (area*newArea <= 0.0) {
/*              printf(" Area flip1 @ %d %d  %lf %lf\n", i, j, area, newArea);*/
                cnt = -1;
                break;
              }
            }
        }
        if (cnt == -1) continue;
        if (cnt !=  2) {
          printf(" Warning: nTri collapse = %d (xtendStacks)!\n", cnt);
          continue;
        }
        
        /* fix up triangles & set quad */
        for (k = 0; k < ntri; k++) {
          if (!sTris[k]) continue;
          if ((tris[3*k  ] == srcVert[0]) || (tris[3*k+1] == srcVert[0]) ||
              (tris[3*k+2] == srcVert[0]))
            if ((tris[3*k  ] == tgtVert[0]) || (tris[3*k+1] == tgtVert[0]) ||
                (tris[3*k+2] == tgtVert[0])) tmap[k] = false;
          if (tmap[k]) {
            if (tris[3*k  ] == srcVert[0]) tris[3*k  ] = tgtVert[0];
            if (tris[3*k+1] == srcVert[0]) tris[3*k+1] = tgtVert[0];
            if (tris[3*k+2] == srcVert[0]) tris[3*k+2] = tgtVert[0];
          }
          if ((tris[3*k  ] == srcVert[1]) || (tris[3*k+1] == srcVert[1]) ||
              (tris[3*k+2] == srcVert[1]))
            if ((tris[3*k  ] == tgtVert[1]) || (tris[3*k+1] == tgtVert[1]) ||
                (tris[3*k+2] == tgtVert[1])) tmap[k] = false;
          if (tmap[k]) {
            if (tris[3*k  ] == srcVert[1]) tris[3*k  ] = tgtVert[1];
            if (tris[3*k+1] == srcVert[1]) tris[3*k+1] = tgtVert[1];
            if (tris[3*k+2] == srcVert[1]) tris[3*k+2] = tgtVert[1];
          }
        }
        quads[indx  ] = 0;
        quads[indx+1] = 0;
        nx++;
      }
    }
  }

  EG_free(sTris);
  if (nx != 0) printf("       Xtend:    nQuads = %2d\n", nx);
  return nx;
}


static int recoverQuads(ego body, ego tess, ego face, int fIndex,
                        int nu, int nv, const double *uvs)
{
  int          i, j, k, m, n, npts, ntri, status, nquad, indx, nswap, frame;
  int          v[4], *grid, *quads, *trix, *stri = NULL;
  bool         *tmap;
  const double *xyz, *uv;
  const int    *ptype, *pindx, *tris, *tric;
  
  status = EG_getTessFace(tess, fIndex, &npts, &xyz, &uv, &ptype, &pindx,
                          &ntri, &tris, &tric);
  if (status != EGADS_SUCCESS) {
    printf(" Warning: EG_getTessFace %d = %d (recoverQuads)!\n",
           fIndex, status);
    return 0;
  }
  for (frame = 0; frame < npts; frame++)
    if (ptype[frame] < 0) break;

  grid = (int *) EG_alloc(nu*nv*sizeof(int));
  if (grid == NULL) {
    printf(" Warning: Allocate of %d grid (recoverQuads)!\n", nu*nv);
    return 0;
  }
  for (i = 0; i < nu*nv; i++) grid[i] = -1;
  
  /* find all vertices */
  for (n = i = 0; i < nu*nv; i++) {
    for (j = 0; j < npts; j++) {
      if ((fabs(uvs[2*i  ] - uv[2*j  ]) < PARMTOL) &&
          (fabs(uvs[2*i+1] - uv[2*j+1]) < PARMTOL)) {
        grid[i] = j + 1;
        break;
      }
    }
    if (j == npts) n++;
  }
#ifdef DEBUG
  printf(" Face %3d:  notFilled = %d\n", fIndex, n);
  for (j = nv-1; j >= 0; j--) {
    printf("    ");
    for (i = 0; i < nu; i++)
      if (grid[j*nu+i] == -1) {
        printf(".");
      } else {
        printf("X");
    }
    printf("\n");
  }
#endif

  /* full coverage -- rebuild the entire Face tessellation */
  if ((n == 0) && (ntri == 2*(nu-1)*(nv-1))) {
    nquad = (nu-1)*(nv-1);
    trix  = (int *) EG_alloc(3*2*nquad*sizeof(int));
    if (trix == NULL) {
      printf(" Warning: Allocate of %d tris (recoverQuads)!\n", 2*nquad);
      EG_free(grid);
      return 0;
    }
    printf(" Info: Face %3d -- nQuads = %3d (Full)\n", fIndex, nquad);
    for (n = j = 0; j < nv-1; j++)
      for (i = 0; i < nu-1; i++) {
        trix[3*n  ] = grid[ j   *nu + i];
        trix[3*n+1] = grid[ j   *nu + i + 1];
        trix[3*n+2] = grid[(j+1)*nu + i + 1];
        n++;
        trix[3*n  ] = grid[ j   *nu + i];
        trix[3*n+1] = grid[(j+1)*nu + i + 1];
        trix[3*n+2] = grid[(j+1)*nu + i];
        n++;
      }
    EG_free(grid);
    status = EG_setTessFace(tess, fIndex, npts, xyz, uv, ntri, trix);
    EG_free(trix);
    if (status != EGADS_SUCCESS) {
      printf(" Warning: EG_setTessFace Full %d = %d (recoverQuads)!\n",
             fIndex, status);
      return 0;
    }
    
    return nquad;
  }

  /* possbile quads */
  quads = (int *) EG_alloc(2*(nu-1)*(nv-1)*sizeof(int));
  if (quads == NULL) {
    printf(" Warning: Allocate of %d quads (recoverQuads)!\n", 2*(nu-1)*(nv-1));
    EG_free(grid);
    return 0;
  }
  for (i = 0; i < 2*(nu-1)*(nv-1); i++) quads[i] = -1;
  markInternalEdges(body, tess, face, nu, nv, uvs, quads);
  
  /* free triangle flags */
  tmap = (bool *) EG_alloc(ntri*sizeof(bool));
  if (tmap == NULL) {
    printf(" Warning: Allocate of %d tris (recoverQuads)!\n", ntri);
    EG_free(quads);
    EG_free(grid);
    return 0;
  }
  for (i = 0; i < ntri; i++) tmap[i] = true;
  
  /* find the quads we have */
  for (nquad = j = 0; j < nv-1; j++)
    for (i = 0; i < nu-1; i++) {
      indx = 2*(j*(nu-1) + i);
      if (quads[indx] == -2) continue;
      v[0] = grid[ j   *nu + i];
      v[1] = grid[ j   *nu + i + 1];
      v[2] = grid[(j+1)*nu + i + 1];
      v[3] = grid[(j+1)*nu + i];
      if ((v[0] == -1) || (v[1] == -1) || (v[2] == -1) || (v[3] == -1)) continue;
      for (k = 0; k < ntri; k++) {
        for (n = m = 0; m < 4; m++) {
          if (tris[3*k  ] == v[m]) n++;
          if (tris[3*k+1] == v[m]) n++;
          if (tris[3*k+2] == v[m]) n++;
        }
        if (n > 3)
          printf(" Info: Face %3d -- Triangle %d hit = %d (recoverQuads)!\n",
                 fIndex, k+1, n);
        if (n != 3) continue;
        if (quads[indx] == -1) {
          quads[indx] = k;
        } else if (quads[indx+1] == -1) {
          quads[indx+1] = k;
          tmap[k]           = false;
          tmap[quads[indx]] = false;
          nquad++;
        } else {
          printf(" Info: Face %3d -- Triangle %d 2+ on quad %d (recoverQuads)!\n",
                 fIndex, k+1, indx/2);
        }
      }
    }
  for (nswap = j = 0; j < nv-1; j++)
    for (i = 0; i < nu-1; i++) {
      indx = 2*(j*(nu-1) + i);
      if (quads[indx] == -2) continue;
      v[0] = grid[ j   *nu + i];
      v[1] = grid[ j   *nu + i + 1];
      v[2] = grid[(j+1)*nu + i + 1];
      v[3] = grid[(j+1)*nu + i];
      if ((v[0] == -1) || (v[1] == -1) || (v[2] == -1) || (v[3] == -1)) continue;
      if (quads[indx+1] == -1) nswap++;
    }
  
  /* swap to recover */
  if (nswap != 0) {
    swapRecover(ntri, tris, tric, xyz, uv, grid, quads, tmap, nu, nv, uvs,
                &stri);
    if (stri == NULL) {
      printf(" Warning: Face %3d -- swapRecover failure (recoverQuads)!\n",
             fIndex);
      EG_free(tmap);
      EG_free(quads);
      EG_free(grid);
      return 0;
    }
    /* fill in quad islands/stacks that we cannot recover by swaps */
    fillIslands(npts, ntri, stri, grid, quads, tmap, nu, nv);
    while (xtendStacks(frame, ntri, stri, uv, grid, quads, tmap,
                       nu, nv, uvs) != 0);
    
    /* count */
    nswap = nquad;
    for (nquad = j = 0; j < nv-1; j++)
      for (i = 0; i < nu-1; i++) {
        indx = 2*(j*(nu-1) + i);
        if (quads[indx]   == -2) continue;
        if (quads[indx+1] != -1) nquad++;
      }
    for (n = k = 0; k < ntri; k++) if (tmap[k]) n++;
    printf(" Info: Face %3d -- nQuads = %3d -> %3d   nTris = %3d\n",
           fIndex, nswap, nquad, n);
  } else {
    for (n = k = 0; k < ntri; k++) if (tmap[k]) n++;
    printf(" Info: Face %3d -- nQuads = %3d          nTris = %3d\n",
           fIndex, nquad, n);
  }
  
  /* look at the counts */
  if (ntri != n+2*nquad) {
    printf(" Warning: Face %3d -- nQuads = %d  nTris = %d   counts = %d %d\n",
           fIndex, nquad, n, ntri, n+2*nquad);
    if (stri != NULL) EG_free(stri);
    EG_free(tmap);
    EG_free(quads);
    EG_free(grid);
    return 0;
  }
  trix = (int *) EG_alloc(3*ntri*sizeof(int));
  if (trix == NULL) {
    printf(" Warning: Allocate of %d tris (recoverQuads)!\n", ntri);
    if (stri != NULL) EG_free(stri);
    EG_free(tmap);
    EG_free(quads);
    EG_free(grid);
    return 0;
  }
  
  /* fill the triangles */
  if (stri == NULL) {
    for (n = k = 0; k < ntri; k++)
      if (tmap[k]) {
        trix[3*n  ] = tris[3*k  ];
        trix[3*n+1] = tris[3*k+1];
        trix[3*n+2] = tris[3*k+2];
        n++;
      }
  } else {
    for (n = k = 0; k < ntri; k++)
      if (tmap[k]) {
        trix[3*n  ] = stri[3*k  ];
        trix[3*n+1] = stri[3*k+1];
        trix[3*n+2] = stri[3*k+2];
        n++;
      }
  }
  
  /* fill the quads */
  for (j = 0; j < nv-1; j++)
    for (i = 0; i < nu-1; i++) {
      indx = 2*(j*(nu-1) + i);
      if (quads[indx]   == -2) continue;
      if (quads[indx+1] == -1) continue;
      trix[3*n  ] = grid[ j   *nu + i];
      trix[3*n+1] = grid[ j   *nu + i + 1];
      trix[3*n+2] = grid[(j+1)*nu + i + 1];
      n++;
      trix[3*n  ] = grid[ j   *nu + i];
      trix[3*n+1] = grid[(j+1)*nu + i + 1];
      trix[3*n+2] = grid[(j+1)*nu + i];
      n++;
    }
  if (stri != NULL) EG_free(stri);
  EG_free(tmap);
  EG_free(quads);
  EG_free(grid);
  
  status = EG_setTessFace(tess, fIndex, npts, xyz, uv, ntri, trix);
  EG_free(trix);
  if (status != EGADS_SUCCESS) {
    printf(" Warning: EG_setTessFace Partial %d = %d (recoverQuads)!\n",
           fIndex, status);
    return 0;
  }
  
  return nquad;
}


int OMLplus(ego body, double *tparms, ego *tesx)
{
  int          i, j, nfaces, nedges, status, alen, atype, nu, nv, iper;
  int          *nquads;
  double       trange[2];
  ego          context, tess, geom, *faces, *edges;
  const int    *ints;
  const char   *string;
  const double *reals;
  
  *tesx  = NULL;
  status = EG_getContext(body, &context);
  if (status != EGADS_SUCCESS) {
    printf(" Error: EG_getContext Face = %d (OMLplus)!\n", status);
    return status;
  }
  status = EG_getBodyTopos(body, NULL, FACE, &nfaces, &faces);
  if (status != EGADS_SUCCESS) {
    printf(" Error: EG_getBodyTopos Face = %d (OMLplus)!\n", status);
    return status;
  }
  status = EG_getBodyTopos(body, NULL, EDGE, &nedges, &edges);
  if (status != EGADS_SUCCESS) {
    printf(" Error: EG_getBodyTopos Edge = %d (OMLplus)!\n", status);
    EG_free(faces);
    return status;
  }
  
  /* initialize the tessellation object */
  status  = EG_initTessBody(body, &tess);
  if (status != EGADS_SUCCESS) {
    printf(" Error: EG_initTessBody = %d (OMLplus)!\n", status);
    EG_free(edges);
    EG_free(faces);
    return status;
  }
  
  /* fill in Edges */
  for (i = 0; i < nedges; i++) {
    if (edges[i]->mtype == DEGENERATE) continue;
    status = EG_attributeRet(edges[i], "meshT", &atype, &alen,
                             &ints, &reals, &string);
    if (status != EGADS_SUCCESS) {
      /* trimming Edge -- fill in by Face spacings */
      status = trimmingEdge(body, tess, i+1, edges[i]);
    } else {
      if (reals[alen-1] == reals[alen-2]) alen--;   /* seeing multiples! */
      (void) EG_getRange(edges[i], trange, &iper);
      if ((fabs(trange[0]-reals[0]) > PARMTOL) ||
          (fabs(trange[1]-reals[alen-1]) > PARMTOL)) {
        status = partialEdge(tess, i+1, edges[i], alen, reals);
      } else {
        status = fullEdge(tess, i+1, edges[i], alen, reals);
      }
    }
    if (status != EGADS_SUCCESS) {
      for (j = 0; j < alen; j++) printf("     %3d:  %le\n", j+1, reals[j]);
      EG_free(edges);
      EG_free(faces);
      EG_deleteObject(tess);
      return status;
    }
  }
  
  /* insert quad locations into the Faces */
  for (i = 0; i < nfaces; i++) {
    status = EG_attributeRet(faces[i], "meshSize", &atype, &alen,
                             &ints, &reals, &string);
    if (status != EGADS_SUCCESS) {
      printf(" Face %3d: No mesh    %d\n", i+1, status);
      continue;
    }
    nu = ints[0];
    nv = ints[1];
    EG_attributeRet(faces[i], "meshUV", &atype, &alen, &ints, &reals, &string);
    printf(" Face %3d: meshSize = %3d %3d  sense = %2d  nInsert = %d\n",
           i+1, nu, nv, faces[i]->mtype,
           addFaceInserts(body, tess, faces[i], i+1, nu, nv, reals));
  }
  EG_free(edges);

  /* tessellate any untouched Edges and Faces */
  status = EG_finishTess(tess, tparms);
  if (status != EGADS_SUCCESS) {
    printf(" Error: EG_finishTess = %d (OMLplus)!\n", status);
    EG_free(faces);
    EG_deleteObject(tess);
    return status;
  }
  
  /* reopen the tessellation and make the quads from the triangles */
  nquads = (int *) EG_alloc(nfaces*sizeof(int));
  if (nquads == NULL) {
    printf(" Error: Allocating %d ints (OMLplus)!\n", nfaces);
    EG_free(faces);
    EG_deleteObject(tess);
    return EGADS_MALLOC;
  }
  for (i = 0; i < nfaces; i++) nquads[i] = 0;

  status = EG_openTessBody(tess);
  if (status != EGADS_SUCCESS) {
    printf(" Error: EG_openTessBody = %d (OMLplus)!\n", status);
    EG_free(nquads);
    EG_free(faces);
    EG_deleteObject(tess);
    return status;
  }
  printf("\n");
  
  /* recover quads and fill the Faces in the tessellation object */
  for (i = 0; i < nfaces; i++) {
    status = EG_attributeRet(faces[i], "meshSize", &atype, &alen,
                             &ints, &reals, &string);
    if (status != EGADS_SUCCESS) continue;
    nu = ints[0];
    nv = ints[1];
    EG_attributeRet(faces[i], "meshUV", &atype, &alen, &ints, &reals, &string);
    nquads[i] = recoverQuads(body, tess, faces[i], i+1, nu, nv, reals);
  }
  
  /* close the tessellation */
  status = EG_statusTessBody(tess, &geom, &i, &j);
  if ((status != EGADS_SUCCESS) || (abs(i) != 1)) {
    printf(" Error: EG_statusTessBody %d = %d (OMLplus)!\n", i, status);
    EG_free(nquads);
    EG_free(faces);
    EG_deleteObject(tess);
    return EGADS_TESSTATE;
  }
  printf("\n");
  
  /* cleanup all of our large attributes */
  EG_setOutLevel(context, 0);
  for (i = 0; i < nfaces; i++) {
    EG_attributeDel(faces[i], "meshSize");
    EG_attributeDel(faces[i], "meshUV");
    EG_attributeDel(faces[i], ".insert!");
  }
  EG_setOutLevel(context, 1);
  EG_free(faces);
  
  /* reset mixed attribute with additional quad info */
  atype  = ATTRINT;
  status = EG_attributeRet(tess, ".tessType", &atype, &alen, &ints, &reals,
                           &string);
  if ((status != EGADS_SUCCESS) || (atype != ATTRSTRING)) {
    status = EG_attributeAdd(tess, ".tessType", ATTRSTRING, 5, NULL, NULL,
                             "Mixed");
    if (status != EGADS_SUCCESS) {
      printf(" Warning: EG_attributeAdd tessType = %d (OMLplus)!\n", status);
    } else {
      status = EG_attributeAdd(tess, ".mixed", ATTRINT, nfaces, nquads, NULL,
                               NULL);
      if (status != EGADS_SUCCESS)
        printf(" Warning: EG_attributeAdd mixed 0 = %d (OMLplus)!\n", status);
    }
  } else {
    atype  = ATTRINT;
    status = EG_attributeRet(tess, ".mixed", &atype, &alen, &ints, &reals,
                             &string);
    if ((status != EGADS_SUCCESS) || (atype != ATTRINT)) {
      status = EG_attributeAdd(tess, ".mixed", ATTRINT, nfaces, nquads, NULL,
                               NULL);
      if (status != EGADS_SUCCESS)
        printf(" Warning: EG_attributeAdd mixed 1 = %d (OMLplus)!\n", status);
    } else {
      for (i = 0; i < nfaces; i++)
        if (nquads[i] == 0) nquads[i] = ints[i];
      status = EG_attributeAdd(tess, ".mixed", ATTRINT, nfaces, nquads, NULL,
                               NULL);
      if (status != EGADS_SUCCESS)
        printf(" Warning: EG_attributeAdd mixed 2 = %d (OMLplus)!\n", status);
    }
  }
  EG_free(nquads);
  
  *tesx = tess;
  return EGADS_SUCCESS;
}
