/*
 ************************************************************************
 *                                                                      *
 * udfCrochet -- generate component-based quad grids and crochet together *
 *                                                                      *
 *            Written by John Dannenhoffer @ Syracuse University        *
 *            Patterned after code written by Bob Haimes  @ MIT         *
 *                                                                      *
 ************************************************************************
 */

/*
 * Copyright (C) 2011/2025  John F. Dannenhoffer, III (Syracuse University)
 *
 * This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *     MA  02110-1301  USA
 */

/* attributes set by user before using udfCrochet

   Body
      meshComponent    req component name
      meshInfo         opt default meshInfo
      meshInfoX        opt default meshInfo for Edges where dx > (MAX(dy,dx)
      meshInfoY        opt default meshInfo for Edges where yx > (MAX(dz,dy)
      meshInfoZ        opt default meshInfo for Edges where zx > (MAX(dx,dz)
   Face
      meshComponent    req component name
      meshSurfType     opt $none|$solid|$wake   (default=$solid)
      meshPriority     opt priority             (default=0)
   Edge
      meshGroup        opt group number (must be positive)
      meshMatch        causes mesh in subsequent component to match normal spacing
      meshInfo         opt dsbeg;dsend;dsmax;nmin;nmax;dirn

                           dsbeg=0   no spacing specified at beg (default=0)
                                >0   relative spacing at beg as fraction of Edge length
                                <0   absolute spacing at beg
                              -99999 signal to use VORLAX-style p parameter
                           dsend=0   no spacing specified at end (default=0)
                                >0   relative spacing at end as fraction of Edge length
                                <0   absolute spacing at end
                               any   VORLAX-stype pspace parameter
                                     +3.0  uniform
                                     +2.0  sine clustered at beg
                                     +1.0  cosine
                                      0    uniform
                                     -1.0  cosine
                                     -2.0  sine clustered at end
                                     -3.0  uniform
                           dsmax=0   no maximum spacing specified (default=1.01*stotal/4)
                                >0   relative maximum spacing as fraction of Edge length
                                <0   absolute maximum spacing
                                     ignored if dsbeg=-99999
                           nmin      minimum number of mesh points (default=5)
                                     ignored if dsbeg=-99999
                           nmax      maximum number of mesh points (default=1000)
                                     used when  dsbeg=-99999
                           dirn =0   Edge direction comes from EGADS (default=0)
                                =1   smaller end abs(x-coordinate) is considered "beg"
                                =2   smaller end abs(y-coordinate) is considered "beg"
                                =3   smaller end abs(z-coordinate) is considered "beg"

   attributes used during execution of udpPanel:

   BODY
      __hasTess__        1
   Face
      meshOrigin         jbody;jface
      meshSize           imax;jmax
      meshUV             UVarray
   Edge
      meshT              Tarray
      meshTint           Tarray on component intersections
      meshN              spacing normal to (away from) Edge
      __meshInfo__       MeshInfo actually used on Edge
      __meshGroup__      temporary copy of meshGroup
      __meshNorm__       normal spacing at beg and end of Edge
*/

#define NUMUDPARGS       1
#define NUMUDPINPUTBODYS 1
#include "udpUtilities.h"

/* shorthands for accessing argument values and velocities */
#define COMPFILE( IUDP)  ((char   *) (udps[IUDP].arg[0].val))

/* data about possible arguments */
static char  *argNames[NUMUDPARGS] = {"compfile", };
static int    argTypes[NUMUDPARGS] = {ATTRFILE,   };
static int    argIdefs[NUMUDPARGS] = {0,          };
static double argDdefs[NUMUDPARGS] = {0.,         };

/* get utility routines: udpErrorStr, udpInitialize, udpReset, udpSet,
                         udpGet, udpVel, udpClean, udpMesh */
#include "udpUtilities.c"

#define           MIN(A,B)        (((A) < (B)) ? (A) : (B))
#define           MAX(A,B)        (((A) < (B)) ? (B) : (A))

#define FACE_NONE   -1
#define FACE_SOLID   0
#define FACE_WAKE    2

#define MAX_NPNT     1000

/* prototype for functions defined below */
static int makeComponentMesh(modl_T *MODL, int ibody, int jbody, char message[]);
static int makeIntersectionMesh(modl_T *MODL, int ibody);
//$$$static int checkMeshInfo(modl_T *MODL, int ibody, int iedge, char message[]);
static int edgeSpacing(ego eedge, int *npnt, char message[]);
static int groupSpacing(modl_T *MODL, int meshGroup, int ibody, int ngroup, int group[], double meshInfo[]);
static int intersectSegs(CDOUBLE uva[], CDOUBLE uvb[], CDOUBLE uvc[], CDOUBLE uvd[], double *fracab, double *fraccd);
static int matchEdges(modl_T *modl, int ibody, int iedge, int jbody, int jedge, double toler);

/*
    udpExecute
        makeComponentMesh
            matchEdges
            edgeSpacing
        makeIntersectionMesh
            intersectSegs
        OMLplus
*/

/* prototype of unpublished EGADS function */
extern int EG_relPosTs(ego   ecurve,        /* (in)  curve or edge */
                       int   npnt,          /* (in)  number of points */
      /*@null@*/ const double newS[],       /* (in)  relative arclengths of interior points */
                       double newT[],       /* (in)  newT[0]=tmin and newT[npnt-1]=tmax */
                                            /* (out) interior newT are filled in */
                       double newXYZ[]);    /* (out) xyz evaluated at newT */

/* prototype for new quadder */
#include "OMLplus.h"

/* message buffer and outLevel shared by all functions in this file */
static int outLevel=0;


/*
 ************************************************************************
 *                                                                      *
 *   udpExecute - execute the primitive                                 *
 *                                                                      *
 ************************************************************************
 */

int
udpExecute(ego  emodel,                 /* (in)  Model containing Body */
           ego  *eoutModel,             /* (out) MODL pointer */
           int  *nMesh,                 /* (out) number of associated meshes */
           char *string[])              /* (out) error message */
{
    int     status = EGADS_SUCCESS;

    int     oclass, mtype, nchild, *senses, ntemp, ncomp;
    int     ibody, jbody, iface, iedge, i, ij;
    int     attrType, attrType2, attrLen, attrLen2, itrace, one=1;
    int     *cfid=NULL, *ceid=NULL;
    CINT    *tempIlist, *tempIlist2, *meshSize;
    double  data[18], bbox[6], size, params[3];
    CDOUBLE *tempRlist, *tempRlist2, *meshUV;
    char    *message=NULL;
    CCHAR   *tempClist, *tempClist2;
    ego     context, eref, *ebodys, eoutBodys[2], *etemp;
    udp_T   *udps;
    modl_T  *MODL;
    FILE    *fp_comp=NULL;
    void    *modl;

    ROUTINE(udpExecute);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    udps = *Udps;
    printf("udpExecute(emodel=%llx)\n", (long long)emodel);
    printf("compfile(0) = %s\n", COMPFILE(0));
#endif

    printf("\nBEGIN udfCrochet\n\n");

    /* default return values */
    *eoutModel = NULL;
    *nMesh   = 0;
    *string  = NULL;

    MALLOC(message, char, 100);
    message[0] = '\0';

    /* check that Model was input that contains one Body */
    status = EG_getTopology(emodel, &eref, &oclass, &mtype,
                            data, &nchild, &ebodys, &senses);
    CHECK_STATUS(EG_getTopology);

    if (oclass != MODEL) {
        snprintf(message, 100, "expecting a Model");
        status = EGADS_NOTMODEL;
        goto cleanup;
    } else if (nchild != 1) {
        snprintf(message, 100, "expecting Model to contain one Body (not %d)", nchild);
        status = EGADS_NOTBODY;
        goto cleanup;
    }

    status = EG_getContext(emodel, &context);
    CHECK_STATUS(EG_getContext);

    /* get the outLevel from OpenCSM */
    outLevel = ocsmSetOutLevel(-1);

    /* cache copy of arguments for future use */
    status = cacheUdp(NULL);
    CHECK_STATUS(cacheUdp);

    /* remove any previous compfiles */
    if (strlen(COMPFILE(0)) > 0) {
        SPRINT1(1, "removing \"%s\"\n", COMPFILE(0));
        remove(COMPFILE(0));
    }

    /* get a pointer to ocsm's MODL */
    status = EG_getUserPointer(context, (void**)(&(modl)));
    CHECK_STATUS(EG_getUserPointer);

    MODL = (modl_T *) modl;

#ifdef DEBUG
    printf("compfile(%d) = %s\n", numUdp, COMPFILE(numUdp));
#endif

    /* find Body index of Body */
    status = EG_attributeRet(emodel, "__bodyList__",
                             &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
    CHECK_STATUS(EG_attributeRet);

    if (attrType == ATTRINT && attrLen == 1) {
        ibody = tempIlist[0];
    } else {
        snprintf(message, 100, "bad __bodyList__");
        status = OCSM_UDP_ERROR1;
        goto cleanup;
    }

    SPRINT1(2, "BREP of ibody=%5d", ibody);
    for (iedge = 1; iedge <= MODL->body[ibody].nedge; iedge++) {
        SPRINT5(2, "iedge=%5d, ibeg=%5d, iend=%5d, ileft=%5d, irite=%5d", iedge,
                MODL->body[ibody].edge[iedge].ibeg,
                MODL->body[ibody].edge[iedge].iend,
                MODL->body[ibody].edge[iedge].ileft,
                MODL->body[ibody].edge[iedge].irite);
    }
    for (iface = 1; iface <= MODL->body[ibody].nface; iface++) {
        status = EG_getBodyTopos(MODL->body[ibody].ebody, MODL->body[ibody].face[iface].eface,
                                 EDGE, &ntemp, NULL);
        CHECK_STATUS(EG_getBodyTopos);
        SPRINT2x(2, "iface=%5d, nedge=%5d, edges=", iface, ntemp);

        status = EG_getBodyTopos(MODL->body[ibody].ebody, MODL->body[ibody].face[iface].eface,
                                 EDGE, &ntemp, &etemp);
        CHECK_STATUS(EG_getBodyTopos);
        for (i = 0; i < ntemp; i++) {
            SPRINT1x(2, " %5d", EG_indexBodyTopo(MODL->body[ibody].ebody, etemp[i]));
        }
        SPRINT0(2, " ");

        EG_free(etemp);
    }

    /* remove temporary attributes that might be left over */
    status = EG_attributeRet(MODL->body[ibody].ebody, "__hasTess__",
                             &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
    if (status == EGADS_SUCCESS) {
        status = EG_attributeDel(MODL->body[ibody].ebody, "__hasTess__");
        CHECK_STATUS(EG_attributeDel);
    }

    for (iedge = 1; iedge <= MODL->body[ibody].nedge; iedge++) {
        status = EG_attributeRet(MODL->body[ibody].edge[iedge].eedge, "__meshInfo__",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) {
            status = EG_attributeDel(MODL->body[ibody].edge[iedge].eedge, "__meshInfo__");
            CHECK_STATUS(EG_attributeDel);
        }

        status = EG_attributeRet(MODL->body[ibody].edge[iedge].eedge, "__meshGroup__",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) {
            status = EG_attributeDel(MODL->body[ibody].edge[iedge].eedge, "__meshGroup__");
            CHECK_STATUS(EG_attributeDel);
        }

        status = EG_attributeRet(MODL->body[ibody].edge[iedge].eedge, "__meshNorm__",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) {
            status = EG_attributeDel(MODL->body[ibody].edge[iedge].eedge, "__meshNorm__");
            CHECK_STATUS(EG_attributeDel);
        }
    }

    /* find the component Face IDs (cfid) for every Face in ibody */
    MALLOC(cfid, int, 2*MODL->body[ibody].nface+2);

    SPRINT0(2, "mapping of Faces in current Body with their IDs in the component");
    for (iface = 1; iface <= MODL->body[ibody].nface; iface++) {
        cfid[2*iface  ] = 0;
        cfid[2*iface+1] = 0;

        status = EG_attributeRet(MODL->body[ibody].face[iface].eface, "__trace__",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        CHECK_STATUS(EG_attributeRet);

        for (itrace = 0; itrace < attrLen; itrace+=2) {
            jbody = tempIlist[itrace];

            status = EG_attributeRet(MODL->body[jbody].ebody, "meshComponent",
                                     &attrType2, &attrLen2, &tempIlist2, &tempRlist2, &tempClist2);
            if (status == EGADS_SUCCESS) {
                cfid[2*iface  ] = jbody;
                cfid[2*iface+1] = tempIlist[itrace+1];
                break;
            }
        }

        if (cfid[2*iface] == 0 || cfid[2*iface+1] == 0) {
            SPRINT2(2, "iface=%5d:%5d, not from a component", ibody, iface);
        } else {
            SPRINT4(2, "iface=%5d:%5d, cfid=%5d:%5d", ibody, iface, cfid[2*iface], cfid[2*iface+1]);
        }
    }

    /* find the component Edge IDs (ceid) for every Edge in ibody */
    MALLOC(ceid, int, 2*MODL->body[ibody].nedge+2);

    SPRINT0(2, "mapping of Edges in current Body with their IDs in the component");
    for (iedge = 1; iedge <= MODL->body[ibody].nedge; iedge++) {
        ceid[2*iedge  ] = 0;
        ceid[2*iedge+1] = 0;

        if (MODL->body[ibody].edge[iedge].itype == DEGENERATE) continue;

        status = EG_attributeRet(MODL->body[ibody].edge[iedge].eedge, "__trace__",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        CHECK_STATUS(EG_attributeRet);

        for (itrace = 0; itrace < attrLen; itrace+=2) {
            jbody = tempIlist[itrace];

            status = EG_attributeRet(MODL->body[jbody].ebody, "meshComponent",
                                     &attrType2, &attrLen2, &tempIlist2, &tempRlist2, &tempClist2);
            if (status == EGADS_SUCCESS) {
                ceid[2*iedge  ] = jbody;
                ceid[2*iedge+1] = tempIlist[itrace+1];
                break;
            }
        }

        if (ceid[2*iedge] == 0 || ceid[2*iedge+1] == 0) {
            SPRINT2(2, "iedge=%5d:%5d, not from a component", ibody, iedge);
        } else {
            SPRINT4(2, "iedge=%5d:%5d, ceid=%5d:%5d", ibody, iedge, ceid[2*iedge], ceid[2*iedge+1]);
        }
    }

    /* loop through all Bodys prior to ibody and make a component mesh
       if there is a match in cfid */
    ncomp = 0;

    for (jbody = 1; jbody < ibody; jbody++) {
        for (iface = 1; iface <= MODL->body[ibody].nface; iface++) {
            if (cfid[2*iface] == jbody) {
                status = EG_attributeRet(MODL->body[jbody].ebody, "meshComponent",
                                         &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
                CHECK_STATUS(EG_attributeRet);

                SPRINT2(1, "making a component mesh for jbody=%d  (%s)", jbody, tempClist);

                SPRINT1(2, "BREP of jbody=%5d", jbody);
                for (iedge = 1; iedge <= MODL->body[jbody].nedge; iedge++) {
                    SPRINT5(2, "iedge=%5d, ibeg=%5d, iend=%5d, ileft=%5d, irite=%5d", iedge,
                            MODL->body[jbody].edge[iedge].ibeg,
                            MODL->body[jbody].edge[iedge].iend,
                            MODL->body[jbody].edge[iedge].ileft,
                            MODL->body[jbody].edge[iedge].irite);
                }
                for (iface = 1; iface <= MODL->body[jbody].nface; iface++) {
                    status = EG_getBodyTopos(MODL->body[jbody].ebody, MODL->body[jbody].face[iface].eface,
                                             EDGE, &ntemp, NULL);
                    CHECK_STATUS(EG_getBodyTopos);
                    SPRINT2(2, "iface=%5d, nedge=%5d", iface, ntemp);
                }

                ncomp++;
                status = makeComponentMesh(MODL, ibody, jbody, message);
                CHECK_STATUS(makeComponentMesh);

                break;
            }
        }
    }

    /* generate a plotfile of the component meshes */
    if (STRLEN(COMPFILE(0)) > 0) {
        fp_comp = fopen(COMPFILE(0), "w");
        if (fp_comp != NULL) {
            for (iface = 1; iface <= MODL->body[ibody].nface; iface++) {
                status = EG_attributeRet(MODL->body[ibody].face[iface].eface, "meshSize",
                                         &attrType, &attrLen, &meshSize, &tempRlist, &tempClist);
                if (status != EGADS_SUCCESS) continue;

                status = EG_attributeRet(MODL->body[ibody].face[iface].eface, "meshUV",
                                         &attrType, &attrLen, &tempIlist, &meshUV, &tempClist);
                if (status != EGADS_SUCCESS) continue;

                fprintf(fp_comp, "%5d %5d component_%d:%d\n", meshSize[0], meshSize[1], cfid[2*iface], iface);
                for (ij = 0; ij < meshSize[0]*meshSize[1]; ij++) {
                    status = EG_evaluate(MODL->body[ibody].face[iface].eface,
                                         &(meshUV[2*ij]), data);
                    CHECK_STATUS(EG_evaluate);

                    fprintf(fp_comp, "%15.8f %15.8f %15.8f\n", data[0], data[1], data[2]);
                }
            }

            fprintf(fp_comp, "%5d %5d end\n", 0, 0);
            fclose(fp_comp);   fp_comp = NULL;
        }
    }

    /* if there were 2 or more components, mark all the Edges that adjoin two
       components with the t's at their intersection with a component mesh */
    if (ncomp > 1) {
        status = makeIntersectionMesh(MODL, ibody);
        CHECK_STATUS(makeIntersectionMesh);
    }

    SPRINT0(1, "Done making meshes");

    /* now that we put mesh attributes on ibody, copy it for output */
    status = EG_copyObject(MODL->body[ibody].ebody, NULL, &eoutBodys[0]);
    CHECK_STATUS(EG_copyObject);

    status = EG_attributeAdd(eoutBodys[0], "__hasTess__", ATTRINT, 1,
                             &one, NULL, NULL);
    CHECK_STATUS(EG_attributeAdd);

    /* create the quad-dominant mesh on the Body */
    status = EG_getBoundingBox(eoutBodys[0], bbox);
    CHECK_STATUS(EG_getBoundingBox);

    size = sqrt( (bbox[0]-bbox[3]) * (bbox[0]-bbox[3])
               + (bbox[1]-bbox[4]) * (bbox[1]-bbox[4])
               + (bbox[2]-bbox[5]) * (bbox[2]-bbox[5]));

    params[0] = 0.025*size;
    params[1] = 0.001*size;
    params[2] = 12.0;

    SPRINT0(1, "\nBEFORE OMLplus\n");
    status = OMLplus(eoutBodys[0], params, &eoutBodys[1]);
    CHECK_STATUS(OMLplus);
    SPRINT0(1, "\nAFTER  OMLplus\n");

    /* make the Model that will be returned to OpenCSM */
    status = EG_makeTopology(MODL->context, NULL, MODEL, 2, NULL, 1,
                             eoutBodys, NULL, eoutModel);
    CHECK_STATUS(EG_makeTopology);

    /* the copy of the Model whose Body was attributed is returned and contains
       the mixed tessellation Ego */
    udps[numUdp].ebody = *eoutModel;

    printf("\nEND   udfCrochet\n\n");

    /* clear the message buffer if we got here */
    message[0] = '\0';

cleanup:
    FREE(cfid);
    FREE(ceid);

    if (fp_comp != NULL) fclose(fp_comp);

    if (strlen(message) > 0) {
        *string = message;
        printf("%s\n", message);
    } else if (status != EGADS_SUCCESS) {
        FREE(message);
        *string = udpErrorStr(status);
    } else {
        FREE(message);
    }

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   udpSensitivity - return sensitivity derivatives for the "real" argument *
 *                                                                      *
 ************************************************************************
 */

int
udpSensitivity(ego    ebody,            /* (in)  Body pointer */
   /*@unused@*/int    npnt,             /* (in)  number of points */
   /*@unused@*/int    entType,          /* (in)  OCSM entity type */
   /*@unused@*/int    entIndex,         /* (in)  OCSM entity index (bias-1) */
   /*@unused@*/double uvs[],            /* (in)  parametric coordinates for evaluation */
   /*@unused@*/double vels[])           /* (out) velocities */
{
    int iudp, judp;

    ROUTINE(udpSensitivity);

    /* --------------------------------------------------------------- */

    /* check that ebody matches one of the ebodys */
    iudp = 0;
    for (judp = 1; judp <= numUdp; judp++) {
        if (ebody == udps[judp].ebody) {
            iudp = judp;
            break;
        }
    }
    if (iudp <= 0) {
        return EGADS_NOTMODEL;
    }

    /* this routine is not written yet */
    return EGADS_NOLOAD;
}


/*
 ************************************************************************
 *                                                                      *
 *   makeComponentMesh - make quad mesh on component                    *
 *                                                                      *
 ************************************************************************
 */

static int
makeComponentMesh(modl_T  *MODL,        /* (in)  pointer to MODL */
                  int     ibody,        /* (in)  index of combined  Body */
                  int     jbody,        /* (in)  index of component Body */
                  char    message[])    /* (out) error message buffer */
{
    int       status = SUCCESS;

    int       iface, jface, iedge, jedge, nloop, nedge, npnt, npnt0, npnt1, npnt2, npnt3;
    int       attrType, attrType2, attrLen, attrLen2, itrace, oclass, mtype, ipass, nchange, imatch, *senses;
    int       i, j, imax, jmax, size[2], origin[2], periodic, status1, status2;
    int       meshGroup, ngroup, *group=NULL, haveInfo;
    int       ipnt0, ipnt1, kedge, kedgem, kedgep;
    int       *meshFixed=NULL;
    CINT      *tempIlist, *tempIlist2;
    double    data[18], fraci, fracj, t, trange[2], stotal, temp;
    double    defMeshInfo[6], groupMeshInfo[6];
    double    xyzi0[18], xyzi1[18], xyzj0[18], xyzj1[18], dotprod, dx, dy, dz;
    double    data0[18], data1[18], normspace, tempList[6];
    double    *uv=NULL, *flipRlist=NULL;
    CDOUBLE   *tempRlist, *tempRlist2, *pnt0, *pnt1, *pnt2, *pnt3;
    CCHAR     *tempClist, *tempClist2;
    ego       eref, *eloops, *eedges, esurf;
    body_T    myIbody, myJbody;

    ROUTINE(makeComponentMesh);

    /* --------------------------------------------------------------- */

    myIbody = MODL->body[ibody];
    myJbody = MODL->body[jbody];

    /* transfer the meshT and __meshNorm__ attributes from any Edge in ibody to jbody.
       these are the Edges shared between two components (such as wing/wake) */
    MALLOC(meshFixed, int, myJbody.nedge+1);

    for (jedge = 1; jedge <= myJbody.nedge; jedge++) {
        meshFixed[jedge] = 0;
    }

    for (iedge = 1; iedge <= myIbody.nedge; iedge++) {
        if (myIbody.edge[iedge].itype == DEGENERATE) continue;

        /* meshT */
        status = EG_attributeRet(myIbody.edge[iedge].eedge, "meshT",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) {
            for (jedge = 1; jedge <= myJbody.nedge; jedge++) {
                status = imatch = matchEdges(MODL, ibody, iedge, jbody, jedge, EPS03);
                CHECK_STATUS(matchEdges);

                if (imatch == 1 || imatch == 4) {
                    SPRINT4(1, "        transferring meshT  to jbody=%3d, jedge=%3d from ibody=%3d, iedge=%3d",
                            jbody, jedge, ibody, iedge);

                    status = EG_attributeAdd(myJbody.edge[jedge].eedge, "meshT",
                                             attrType, attrLen, tempIlist, tempRlist, tempClist);
                    CHECK_STATUS(EG_attributeAdd);

                    meshFixed[jedge] = 1;
                    break;
                } else if (imatch == 2 || imatch == 5) {
                    SPRINT4(1, "        transferring meshT   to jbody=%3d, jedge=%3d from ibody=%3d, iedge=%3d (reversed)",
                            jbody, jedge, ibody, iedge);

                    MALLOC(flipRlist, double, attrLen);

                    for (i = 0; i < attrLen; i++) {
                        flipRlist[i] = 1 - tempRlist[attrLen-1-i];
                    }

                    status = EG_attributeAdd(myJbody.edge[jedge].eedge, "meshT",
                                             attrType, attrLen, tempIlist, flipRlist, tempClist);
                    CHECK_STATUS(EG_attributeAdd);

                    FREE(flipRlist);

                    meshFixed[jedge] = 1;
                    break;
                }
            }
        }

        /* __meshNorm__ */
        status = EG_attributeRet(myIbody.edge[iedge].eedge, "__meshNorm__",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) {
            for (jedge = 1; jedge <= myJbody.nedge; jedge++) {
                status = imatch = matchEdges(MODL, ibody, iedge, jbody, jedge, EPS03);
                CHECK_STATUS(matchEdges);

                if (imatch == 1 || imatch == 4) {
                    status = EG_attributeAdd(myJbody.edge[jedge].eedge, "__meshNorm__",
                                             attrType, attrLen, tempIlist, tempRlist, tempClist);
                    CHECK_STATUS(EG_attributeAdd);

                    meshFixed[jedge] = 1;
                    break;
                } else if (imatch == 2 || imatch == 5) {
                    MALLOC(flipRlist, double, attrLen);

                    for (i = 0; i < attrLen; i++) {
                        flipRlist[i] = 1 - tempRlist[attrLen-1-i];
                    }

                    status = EG_attributeAdd(myJbody.edge[jedge].eedge, "__meshNorm__",
                                             attrType, attrLen, tempIlist, flipRlist, tempClist);
                    CHECK_STATUS(EG_attributeAdd);

                    FREE(flipRlist);

                    meshFixed[jedge] = 1;
                    break;
                }
            }
        }
    }

    /* set up a temporary meshGroup attribute on the Edges */
    for (jedge = 1; jedge <= myJbody.nedge; jedge++) {
        status = EG_attributeRet(myJbody.edge[jedge].eedge, "meshGroup",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) {
            if (NINT(tempRlist[0]) == 0) {
                printf("meshGroup cannot be 0\n");
                status = OCSM_UDP_ERROR1;
                goto cleanup;
            }

            temp = tempRlist[0];
            status = EG_attributeAdd(myJbody.edge[jedge].eedge, "__meshGroup__",
                                     ATTRREAL, 1, NULL, &temp, NULL);
            CHECK_STATUS(EG_attributeAdd);
        }
    }

    /* if any Edges have a __meshGroup__ attribute, process them now */
    MALLOC(group, int, myJbody.nedge);

    for (ipass = 0; ipass < myJbody.nedge; ipass++) {

        meshGroup = 0;            // current meshGroup (or 0 if none)
        ngroup    = 0;            // number of Edges in this group
        haveInfo  = 0;            // ==1 if groupMeshInfo has been defined

        for (jedge = 1; jedge <= myJbody.nedge; jedge++) {
            status = EG_attributeRet(myJbody.edge[jedge].eedge, "__meshGroup__",
                                     &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
            if (status != EGADS_SUCCESS || attrType != ATTRREAL) continue;

            /* make a new group */
            if (meshGroup == 0) {
                meshGroup       = NINT(tempRlist[0]);
                group[ngroup++] = jedge;

                status = EG_attributeDel(myJbody.edge[jedge].eedge, "__meshGroup__");
                CHECK_STATUS(EG_attributeDel);

                /* remember the groupMeshInfo if defined on this Edge */
                status = EG_attributeRet(myJbody.edge[jedge].eedge, "meshInfo",
                                         &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
                if (status == EGADS_SUCCESS && attrType == ATTRREAL && attrLen == 6) {
                    groupMeshInfo[0] = tempRlist[0];
                    groupMeshInfo[1] = tempRlist[1];
                    groupMeshInfo[2] = tempRlist[2];
                    groupMeshInfo[3] = tempRlist[3];
                    groupMeshInfo[4] = tempRlist[4];
                    groupMeshInfo[5] = tempRlist[5];

                    haveInfo = 1;
                }

                /* add to the current group */
            } else if (NINT(tempRlist[0]) == meshGroup) {
                group[ngroup++] = jedge;

                status = EG_attributeDel(myJbody.edge[jedge].eedge, "__meshGroup__");
                CHECK_STATUS(EG_attributeDel);

                status = EG_attributeRet(myJbody.edge[jedge].eedge, "meshInfo",
                                         &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);

                /* add groupMeshInfo if we do not have it yet */
                if (status == EGADS_SUCCESS && attrType == ATTRREAL && attrLen == 6) {
                    if (haveInfo == 0) {
                        groupMeshInfo[0] = tempRlist[0];
                        groupMeshInfo[1] = tempRlist[1];
                        groupMeshInfo[2] = tempRlist[2];
                        groupMeshInfo[3] = tempRlist[3];
                        groupMeshInfo[4] = tempRlist[4];
                        groupMeshInfo[5] = tempRlist[5];

                        haveInfo = 1;

                        /* make sure that this groupMeshInfo is consistent with the
                           info we already have */
                    } else {
                        if (fabs(groupMeshInfo[0]-tempRlist[0]) > EPS06 ||
                            fabs(groupMeshInfo[1]-tempRlist[1]) > EPS06 ||
                            fabs(groupMeshInfo[2]-tempRlist[2]) > EPS06 ||
                            fabs(groupMeshInfo[3]-tempRlist[3]) > EPS06 ||
                            fabs(groupMeshInfo[4]-tempRlist[4]) > EPS06 ||
                            fabs(groupMeshInfo[5]-tempRlist[5]) > EPS06   ) {
                            snprintf(message, 100, "inconsistent groupMeshInfo");
                            status = OCSM_UDP_ERROR1;
                            goto cleanup;
                        }
                    }
                }
            }
        }

        if (meshGroup == 0) break;

        status = groupSpacing(MODL, meshGroup, jbody, ngroup, group, groupMeshInfo);
        CHECK_STATUS(groupSpacing);
    }

    /* for every Edge that does not yet have a meshT or __meshInfo__ attribute,
       use the meshInfo Edge attribute if it exists */
    for (jedge = 1; jedge <= myJbody.nedge; jedge++) {
        if (myJbody.edge[jedge].itype == DEGENERATE) continue;

        status = EG_attributeRet(myJbody.edge[jedge].eedge, "meshT",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) continue;

        status = EG_attributeRet(myJbody.edge[jedge].eedge, "__meshInfo__",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) continue;

        status = EG_attributeRet(myJbody.edge[jedge].eedge, "meshInfo",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status != EGADS_SUCCESS) continue;

        SPRINT2(1, "        adding __meshInfo__ to jbody=%3d, jedge=%3d from own meshInfo", jbody, jedge);
        status = EG_attributeAdd(myJbody.edge[jedge].eedge, "__meshInfo__",
                                 ATTRREAL, 6, NULL, tempRlist, NULL);
        CHECK_STATUS(EG_attributeAdd);
    }

    /* for every Edge that does not yet have a meshT or __meshInfo__ attribute,
       use the meshInfoX/Y/Z Body attribute if it exists */
    for (jedge = 1; jedge <= myJbody.nedge; jedge++) {
        if (myJbody.edge[jedge].itype == DEGENERATE) continue;

        status = EG_attributeRet(myJbody.edge[jedge].eedge, "meshT",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) continue;

        status = EG_attributeRet(myJbody.edge[jedge].eedge, "__meshInfo__",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) continue;

        status = EG_getRange(myJbody.edge[jedge].eedge, trange, &periodic);
        CHECK_STATUS(EG_getRange);

        status = EG_evaluate(myJbody.edge[jedge].eedge, &trange[0], xyzi0);
        CHECK_STATUS(EG_evaluate);

        status = EG_evaluate(myJbody.edge[jedge].eedge, &trange[1], xyzi1);
        CHECK_STATUS(EG_evaluate);

        dx = fabs(xyzi1[0] - xyzi0[0]);
        dy = fabs(xyzi1[1] - xyzi0[1]);
        dz = fabs(xyzi1[2] - xyzi0[2]);

        if        (dx > MAX(dy,dz)) {
            status = EG_attributeRet(myJbody.ebody, "meshInfoX",
                                     &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
            if (status != EGADS_SUCCESS) continue;
        } else if (dy > MAX(dz,dx)) {
            status = EG_attributeRet(myJbody.ebody, "meshInfoY",
                                     &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
            if (status != EGADS_SUCCESS) continue;
        } else if (dz > MAX(dx,dy)) {
            status = EG_attributeRet(myJbody.ebody, "meshInfoZ",
                                     &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
            if (status != EGADS_SUCCESS) continue;
        } else {
            continue;
        }

        /* getting here means that we want to use the meshInfoX/Y/Z Body attribute */
        SPRINT2(1, "        adding __meshInfo__ to jbody=%3d, jedge=%3d from Body's meshInfoX/Y/Z", jbody, jedge);

        status = EG_attributeAdd(myJbody.edge[jedge].eedge, "__meshInfo__",
                                 ATTRREAL, 6, NULL, tempRlist, NULL);
        CHECK_STATUS(EG_attributeAdd);
    }

    /* for every Edge that does not yet have a meshT or __meshInfo__ attribute,
       use the meshInfo Body attribute if it exists */
    for (jedge = 1; jedge <= myJbody.nedge; jedge++) {
        if (myJbody.edge[jedge].itype == DEGENERATE) continue;

        status = EG_attributeRet(myJbody.edge[jedge].eedge, "meshT",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) continue;

        status = EG_attributeRet(myJbody.edge[jedge].eedge, "__meshInfo__",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) continue;

        status = EG_attributeRet(myJbody.ebody, "meshInfo",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status != EGADS_SUCCESS) continue;

        /* getting here means that we want to use the meshInfo Body attribute */
        SPRINT2(1, "        adding __meshInfo__ to jbody=%3d, jedge=%3d from Body's meshInfo", jbody, jedge);

        status = EG_attributeAdd(myJbody.edge[jedge].eedge, "__meshInfo__",
                                 ATTRREAL, 6, NULL, tempRlist, NULL);
        CHECK_STATUS(EG_attributeAdd);
    }

    /* for every Edge that does not yet have a meshT or __meshInfo__ attribute,
       copy the meshT attribute from the Edge on the other side of a Face
       (if it has a meshT attribute) */
    for (jface = 1; jface <= myJbody.nface; jface++) {
        status = EG_attributeRet(myJbody.face[jface].eface, "meshSurfType",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS && attrType == ATTRSTRING && strcmp(tempClist, "none") == 0) {
            continue;
        }

        status = EG_getTopology(myJbody.face[jface].eface, &eref,
                                &oclass, &mtype, data, &nloop, &eloops, &senses);
        CHECK_STATUS(EG_getTopology);

        status = EG_getTopology(eloops[0], &eref,
                                &oclass, &mtype, data, &nedge, &eedges, &senses);
        CHECK_STATUS(EG_getTopology);

        if (nedge != 4) {
            SPRINT3(0, "Face %d:%d has %d Edges", jbody, jface, nedge);
            snprintf(message, 100, "Face %d:%d has %d Edges", jbody, jface, nedge);
            continue;
        }

        status1 = EG_attributeRet(eedges[0], "__meshInfo__",
                                  &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        status2 = EG_attributeRet(eedges[0], "meshT",
                                  &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status1 == EGADS_NOTFOUND && status2 == EGADS_NOTFOUND) {
            status = EG_attributeRet(eedges[2], "meshT",
                                     &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
            if (status == EGADS_SUCCESS) {
                if (senses[0]*senses[2] < 0) {
                    SPRINT3(1, "        copying meshT       to jbody=%3d, jedge=%3d from Edge %3d (2->0)", jbody,
                            EG_indexBodyTopo(myJbody.ebody, eedges[0]),
                            EG_indexBodyTopo(myJbody.ebody, eedges[2]));

                    status = EG_attributeAdd(eedges[0], "meshT", attrType, attrLen,
                                             tempIlist, tempRlist, tempClist);
                    CHECK_STATUS(EG_attributeAdd);
                } else {
                    SPRINT3(1, "        copying meshT       to jbody=%3d, jedge=%3d from Edge %3d (2->0 reversed)", jbody,
                            EG_indexBodyTopo(myJbody.ebody, eedges[0]),
                            EG_indexBodyTopo(myJbody.ebody, eedges[2]));

                    MALLOC(flipRlist, double, attrLen);

                    for (i = 0; i < attrLen; i++) {
                        flipRlist[i] = 1 - tempRlist[attrLen-1-i];
                    }

                    status = EG_attributeAdd(eedges[0], "meshT", attrType, attrLen,
                                             tempIlist, flipRlist, tempClist);
                    CHECK_STATUS(EG_attributeAdd);

                    FREE(flipRlist);
                }
            }
        }

        status1 = EG_attributeRet(eedges[1], "__meshInfo__",
                                  &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        status2 = EG_attributeRet(eedges[1], "meshT",
                                  &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status1 == EGADS_NOTFOUND && status2 == EGADS_NOTFOUND) {
            status = EG_attributeRet(eedges[3], "meshT",
                                     &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
            if (status == EGADS_SUCCESS) {
                if (senses[1]*senses[3] < 0) {
                    SPRINT3(1, "        copying meshT       to jbody=%3d, jedge=%3d from Edge %3d (3->1)", jbody,
                            EG_indexBodyTopo(myJbody.ebody, eedges[1]),
                            EG_indexBodyTopo(myJbody.ebody, eedges[3]));

                    status = EG_attributeAdd(eedges[1], "meshT", attrType, attrLen,
                                             tempIlist, tempRlist, tempClist);
                    CHECK_STATUS(EG_attributeAdd);
                } else {
                    SPRINT3(1, "        copying meshT       to jbody=%3d, jedge=%3d from Edge %3d (3->1 reversed", jbody,
                            EG_indexBodyTopo(myJbody.ebody, eedges[1]),
                            EG_indexBodyTopo(myJbody.ebody, eedges[3]));

                    MALLOC(flipRlist, double, attrLen);

                    for (i = 0; i < attrLen; i++) {
                        flipRlist[i] = 1 - tempRlist[attrLen-1-i];
                    }

                    status = EG_attributeAdd(eedges[1], "meshT", attrType, attrLen,
                                             tempIlist, flipRlist, tempClist);
                    CHECK_STATUS(EG_attributeAdd);

                    FREE(flipRlist);
                }
            }
        }

        status1 = EG_attributeRet(eedges[2], "__meshInfo__",
                                  &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        status2 = EG_attributeRet(eedges[2], "meshT",
                                  &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status1 == EGADS_NOTFOUND && status2 == EGADS_NOTFOUND) {
            status = EG_attributeRet(eedges[0], "meshT",
                                     &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
            if (status == EGADS_SUCCESS) {
                if (senses[2]*senses[0] < 0) {
                    SPRINT3(1, "        copying meshT       to jbody=%3d, jedge=%3d from Edge %3d (0->2)", jbody,
                            EG_indexBodyTopo(myJbody.ebody, eedges[2]),
                            EG_indexBodyTopo(myJbody.ebody, eedges[0]));

                    status = EG_attributeAdd(eedges[2], "meshT", attrType, attrLen,
                                             tempIlist, tempRlist, tempClist);
                    CHECK_STATUS(EG_attributeAdd);
                } else {
                    SPRINT3(1, "        copying meshT       to jbody=%3d, jedge=%3d from Edge %3d (0->2 reversed)", jbody,
                            EG_indexBodyTopo(myJbody.ebody, eedges[2]),
                            EG_indexBodyTopo(myJbody.ebody, eedges[0]));

                    MALLOC(flipRlist, double, attrLen);

                    for (i = 0; i < attrLen; i++) {
                        flipRlist[i] = 1 - tempRlist[attrLen-1-i];
                    }

                    status = EG_attributeAdd(eedges[2], "meshT", attrType, attrLen,
                                             tempIlist, flipRlist, tempClist);
                    CHECK_STATUS(EG_attributeAdd);

                    FREE(flipRlist);
                }
            }
        }

        status1 = EG_attributeRet(eedges[3], "__meshInfo__",
                                  &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        status2 = EG_attributeRet(eedges[3], "meshT",
                                  &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status1 == EGADS_NOTFOUND && status2 == EGADS_NOTFOUND) {
            status = EG_attributeRet(eedges[1], "meshT",
                                     &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
            if (status == EGADS_SUCCESS) {
                if (senses[3]*senses[1] < 0) {
                    SPRINT3(1, "        copying meshT       to jbody=%3d, jedge=%3d from Edge %3d (1->3)", jbody,
                            EG_indexBodyTopo(myJbody.ebody, eedges[3]),
                            EG_indexBodyTopo(myJbody.ebody, eedges[1]));

                    status = EG_attributeAdd(eedges[3], "meshT", attrType, attrLen,
                                             tempIlist, tempRlist, tempClist);
                    CHECK_STATUS(EG_attributeAdd);
                } else {
                    SPRINT3(1, "        copying meshT       to jbody=%3d, jedge=%3d from Edge %3d (1->3 reversed)", jbody,
                            EG_indexBodyTopo(myJbody.ebody, eedges[3]),
                            EG_indexBodyTopo(myJbody.ebody, eedges[1]));

                    MALLOC(flipRlist, double, attrLen);

                    for (i = 0; i < attrLen; i++) {
                        flipRlist[i] = 1 - tempRlist[attrLen-1-i];
                    }

                    status = EG_attributeAdd(eedges[3], "meshT", attrType, attrLen,
                                             tempIlist, flipRlist, tempClist);
                    CHECK_STATUS(EG_attributeAdd);

                    FREE(flipRlist);
                }
            }
        }
    }

    /* for every Edge (including DEGENERATE Edges) that does not yet have a meshT or
       __meshInfo__ attribute, use the defaults */
    for (jedge = 1; jedge <= myJbody.nedge; jedge++) {
        status = EG_attributeRet(myJbody.edge[jedge].eedge, "meshT",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) continue;

        status = EG_attributeRet(myJbody.edge[jedge].eedge, "__meshInfo__",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) continue;

        /* get total arclenth of Edge */
        status = EG_getRange(myJbody.edge[jedge].eedge, trange, &periodic);
        CHECK_STATUS(EG_getRange);

        status = EG_arcLength(myJbody.edge[jedge].eedge, trange[0], trange[1], &stotal);
        CHECK_STATUS(EG_arcLength);

        defMeshInfo[0] = 0;
        defMeshInfo[1] = 0;
        defMeshInfo[2] = 1.01 * stotal / 4;    // nmin-1
        defMeshInfo[3] = 5;
        defMeshInfo[4] = MAX_NPNT;
        defMeshInfo[5] = 0;

        SPRINT2(1, "        adding __meshInfo__ to jbody=%3d, jedge=%3d from defaultsn", jbody, jedge);

        status = EG_attributeAdd(myJbody.edge[jedge].eedge, "__meshInfo__",
                                 ATTRREAL, 6, NULL, defMeshInfo, NULL);
        CHECK_STATUS(EG_attributeAdd);
    }

    /* if any of the Edges gave a meshMatch attribute, modify the end
       spacings of the adjacent Edges */
    for (jface = 1; jface <= myJbody.nface; jface++) {
        status = EG_getTopology(myJbody.face[jface].eface, &eref,
                                &oclass, &mtype, data, &nloop, &eloops, &senses);
        CHECK_STATUS(EG_getTopology);

        status = EG_getTopology(eloops[0], &eref,
                                &oclass, &mtype, data, &nedge, &eedges, &senses);
        CHECK_STATUS(EG_getTopology);

        if (nedge != 4) continue;

        for (kedge = 0; kedge < 4; kedge++) {
            status = EG_attributeRet(eedges[kedge], "meshMatch",
                                     &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
            if (status != EGADS_SUCCESS) continue;

            status = EG_attributeRet(eedges[kedge], "__meshNorm__",
                                     &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
            CHECK_STATUS(EG_attributeRet);

            /*
                     |   A            |   B            |   C            |   D
              kedgem v         kedgem ^         kedgem v         kedgem ^
                     |                |                |                |
                     +--->---         +--->---         +---<---         +--->---
                       kedge            kedge            kedge            kedge
            */

            kedgem = (kedge + 3) % 4;

            status = EG_attributeRet(eedges[kedgem], "__meshInfo__",
                                     &attrType, &attrLen, &tempIlist, &tempRlist2, &tempClist);
            CHECK_STATUS(EG_attributeRet);

            if (attrLen > 6) {
                printf("tempList is not big enough\n");
                status = OCSM_UDP_ERROR1;
                goto cleanup;
            }

            for (i = 0; i < attrLen; i++) {
                tempList[i] = tempRlist2[i];
            }

            if (senses[kedge] > 0) {
                if (senses[kedgem] > 0) {    // case A
                    SPRINT2(1, "        changing dsend      on jbody=%3d, jedge=%3d (meshMatch)",
                            jbody, EG_indexBodyTopo(myJbody.ebody, eedges[kedgem]));
                    tempList[1] = -tempRlist[0];
                } else {                     // case B
                    SPRINT2(1, "        changing dsbeg      on jbody=%3d, jedge=%3d (meshMatch)",
                            jbody, EG_indexBodyTopo(myJbody.ebody, eedges[kedgem]));
                    tempList[0] = -tempRlist[0];
                }
            } else {
                if (senses[kedgem] > 0) {    // case C
                    SPRINT2(1, "        changing dsend      on jbody=%3d, jedge=%3d (meshMatch)",
                            jbody, EG_indexBodyTopo(myJbody.ebody, eedges[kedgem]));
                    tempList[1] = -tempRlist[1];
                } else {                     // case D
                    SPRINT2(1, "        changing dsbeg      on jbody=%3d, jedge=%3d (meshMatch)",
                            jbody, EG_indexBodyTopo(myJbody.ebody, eedges[kedgem]));
                    tempList[0] = -tempRlist[1];
                }
            }

            status = EG_attributeAdd(eedges[kedgem], "__meshInfo__",
                                     ATTRREAL, attrLen, NULL, tempList, NULL);
            CHECK_STATUS(EG_attributeAdd);

            /*

                 E   |            F   |            G   |            H   |
                     ^ kedgep         v kedgep         ^ kedgep         v kedgep
                     |                |                |                |
              --->---+         --->---+         ---<---+         ---<---+
               kedge            kedge            kedge            kedge
            */

            kedgep = (kedge + 1) % 4;

            status = EG_attributeRet(eedges[kedgep], "__meshInfo__",
                                     &attrType, &attrLen, &tempIlist, &tempRlist2, &tempClist);
            CHECK_STATUS(EG_attributeRet);

            if (attrLen > 6) {
                printf("tempList is not big enough\n");
                status = OCSM_UDP_ERROR1;
                goto cleanup;
            }

            for (i = 0; i < attrLen; i++) {
                tempList[i] = tempRlist2[i];
            }

            if (senses[kedge] > 0) {
                if (senses[kedgep] > 0) {    // case E
                    SPRINT2(1, "        changing dsbeg      on jbody=%3d, jedge=%3d (meshMatch)",
                            jbody, EG_indexBodyTopo(myJbody.ebody, eedges[kedgep]));
                    tempList[0] = -tempRlist[1];
                } else {                     // case F
                    SPRINT2(1, "        changing dsend      on jbody=%3d, jedge=%3d (meshMatch)",
                            jbody, EG_indexBodyTopo(myJbody.ebody, eedges[kedgep]));
                    tempList[1] = -tempRlist[1];
                }
            } else {
                if (senses[kedgem] > 0) {    // case G
                    SPRINT2(1, "        changing dsbeg      on jbody=%3d, jedge=%3d (meshMatch)",
                            jbody, EG_indexBodyTopo(myJbody.ebody, eedges[kedgep]));
                    tempList[0] = -tempRlist[0];
                } else {                     // case H
                    SPRINT2(1, "        changing dsend      on jbody=%3d, jedge=%3d (meshMatch)",
                            jbody, EG_indexBodyTopo(myJbody.ebody, eedges[kedgep]));
                    tempList[1] = -tempRlist[0];
                }
            }

            status = EG_attributeAdd(eedges[kedgep], "__meshInfo__",
                                     ATTRREAL, attrLen, NULL, tempList, NULL);
            CHECK_STATUS(EG_attributeAdd);
        }
    }

    /* apply initial spacing on all Edges that do not have a meshT attribute */
    for (jedge = 1; jedge <= myJbody.nedge; jedge++) {
        status = EG_attributeRet(myJbody.edge[jedge].eedge, "meshT",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) continue;

        npnt = 0;
        status = edgeSpacing(myJbody.edge[jedge].eedge, &npnt, message);
        if (status != EGADS_SUCCESS) {
            SPRINT3(0, "   was working on jbody=%d, jedge=%d, npnt=%d when error occurred",
                    jbody, jedge, npnt);
        }
        CHECK_STATUS(edgeSpacing);
    }

    /* loop through all the Faces and bump up the mesh size to make
       opposite Edges have the same number of mesh points.  this make
       take multiple passes since we may have to propogate a size change
       across multiple Faces */
    nchange = 0;
    for (ipass = 0; ipass < myJbody.nedge; ipass++) {
        nchange = 0;

        for (jface = 1; jface <= myJbody.nface; jface++) {
            status = EG_attributeRet(myJbody.face[jface].eface, "meshSurfType",
                                     &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
            if (status == EGADS_SUCCESS && attrType == ATTRSTRING && strcmp(tempClist, "none") == 0) {
                continue;
            }

            status = EG_getTopology(myJbody.face[jface].eface, &eref,
                                    &oclass, &mtype, data, &nloop, &eloops, &senses);
            CHECK_STATUS(EG_getTopology);

            status = EG_getTopology(eloops[0], &eref,
                                    &oclass, &mtype, data, &nedge, &eedges, &senses);
            CHECK_STATUS(EG_getTopology);

            if (nedge != 4) {
                snprintf(message, 100, "Face %d:%d has %d Edges", jbody, jface, nedge);
                continue;
            }

            /* get the number of points on each Edge */
            status = EG_attributeRet(eedges[0], "meshT",
                                     &attrType, &npnt0, &tempIlist, &tempRlist, &tempClist);
            CHECK_STATUS(EG_attributeRet);

            status = EG_attributeRet(eedges[1], "meshT",
                                     &attrType, &npnt1, &tempIlist, &tempRlist, &tempClist);
            CHECK_STATUS(EG_attributeRet);

            status = EG_attributeRet(eedges[2], "meshT",
                                     &attrType, &npnt2, &tempIlist, &tempRlist, &tempClist);
            CHECK_STATUS(EG_attributeRet);

            status = EG_attributeRet(eedges[3], "meshT",
                                     &attrType, &npnt3, &tempIlist, &tempRlist, &tempClist);
            CHECK_STATUS(EG_attributeRet);

            SPRINT1(2, "making opposite Edges of jface=%d the same", jface);
            SPRINT2(2, "   south (%3d) has %5d points", EG_indexBodyTopo(myJbody.ebody, eedges[0]), npnt0);
            SPRINT2(2, "   north (%3d) has %5d points", EG_indexBodyTopo(myJbody.ebody, eedges[2]), npnt2);
            SPRINT2(2, "   west  (%3d) has %5d points", EG_indexBodyTopo(myJbody.ebody, eedges[3]), npnt3);
            SPRINT2(2, "   east  (%3d) has %5d points", EG_indexBodyTopo(myJbody.ebody, eedges[1]), npnt1);

            /* bump up size on eedges[0] */
            if (npnt0 < npnt2) {
                status = jedge = EG_indexBodyTopo(myJbody.ebody, eedges[0]);
                CHECK_STATUS(EG_indexBodyTopo);

                if (meshFixed[jedge] == 0) {
                    SPRINT2(2, "    bumping up size on eedges[0], jedge=%3d, npnt=%3d", jedge, npnt2);

                    status = edgeSpacing(eedges[0], &npnt2, message);
                    if (status != EGADS_SUCCESS) {
                        SPRINT4(0, "   was working on jbody=%d, jface=%d, jedge=%d, npnt2=%d when error occurred",
                                jbody, jface, jedge, npnt2);
                    }
                    CHECK_STATUS(edgeSpacing);

                    nchange++;
                }

            /* bump up size on eedges[2] */
            } else if (npnt0 > npnt2) {
                status = jedge = EG_indexBodyTopo(myJbody.ebody, eedges[2]);
                CHECK_STATUS(EG_indexBodyTopo);

                if (meshFixed[jedge] == 0) {
                    SPRINT2(2, "    bumping up size on eedges[2], jedge=%3d, npnt=%3d", jedge, npnt0);

                    status = edgeSpacing(eedges[2], &npnt0, message);
                    if (status != EGADS_SUCCESS) {
                        SPRINT4(0, "   was working on jbody=%d, jface=%d, jedge=%d, npnt2=%d when error occurred",
                                jbody, jface, jedge, npnt2);
                    }
                    CHECK_STATUS(edgeSpacing);

                    nchange++;
                }
            }

            /* bump up size on eedges[1] */
            if (npnt1 < npnt3) {
                status = jedge = EG_indexBodyTopo(myJbody.ebody, eedges[1]);
                CHECK_STATUS(EG_indexBodyTopo);

                if (meshFixed[jedge] == 0) {
                    SPRINT2(2, "    bumping up size on eedges[1], jedge=%3d, npnt=%3d", jedge, npnt3);

                    status = edgeSpacing(eedges[1], &npnt3, message);
                    if (status != EGADS_SUCCESS) {
                        SPRINT4(0, "   was working on jbody=%d, jface=%d, jedge=%d, npnt2=%d when error occurred",
                                jbody, jface, jedge, npnt2);
                    }
                    CHECK_STATUS(edgeSpacing);

                    nchange++;
                }

            /* bump up size on eedges[3] */
            } else if (npnt1 > npnt3) {
                status = jedge = EG_indexBodyTopo(myJbody.ebody, eedges[3]);
                CHECK_STATUS(EG_indexBodyTopo);

                if (meshFixed[jedge] == 0) {
                    SPRINT2(2, "    bumping up size on eedges[3], jedge=%3d, npnt=%3d", jedge, npnt1);

                    status = edgeSpacing(eedges[3], &npnt1, message);
                    if (status != EGADS_SUCCESS) {
                        SPRINT4(0, "   was working on jbody=%d, jface=%d, jedge=%d, npnt2=%d when error occurred",
                                jbody, jface, jedge, npnt2);
                    }
                    CHECK_STATUS(edgeSpacing);

                    nchange++;
                }
            }
        }

        SPRINT2(2, "ipass=%d, nchange=%d", ipass, nchange);
        if (nchange == 0) break;
    }

    /* getting here with nchange>0 implies that we ran out of passes.
       it is possible that this happened because of an infitinite loop */
    if (nchange > 0) {
        snprintf(message, 100, "Edge size propogation failed");
        status = -997;
        goto cleanup;
    }

    for (jedge = 1; jedge <= myJbody.nedge; jedge++) {
        if (myJbody.edge[jedge].itype == DEGENERATE) {
            SPRINT2(1, "        jedge=%5d: %5d points (degenerate)", jedge, 0);
        } else {
            status = EG_attributeRet(myJbody.edge[jedge].eedge, "meshT",
                                     &attrType, &npnt0, &tempIlist, &pnt0, &tempClist);
            CHECK_STATUS(EG_attributeRet);

            status = EG_attributeRet(myJbody.edge[jedge].eedge, "__meshInfo__",
                                     &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
            if (status != EGADS_SUCCESS) {
                SPRINT2(1, "        jedge=%5d: %5d points", jedge, npnt0);
            } else {
                SPRINT8(1, "        jedge=%5d: %5d points (%10.5f %10.5f %10.5f %6.0f %6.0f %3.0f)",
                        jedge, npnt0, tempRlist[0], tempRlist[1], tempRlist[2], tempRlist[3], tempRlist[4], tempRlist[5]);
            }
        }
    }

    /* loop through the Edges and store the normal spacings (in XYZ) at
       their ends */
    for (jface = 1; jface <= myJbody.nface; jface++) {
        status = EG_getTopology(myJbody.face[jface].eface, &esurf,
                                &oclass, &mtype, data, &nloop, &eloops, &senses);
        CHECK_STATUS(EG_getTopology);

        status = EG_getTopology(eloops[0], &eref,
                                &oclass, &mtype, data, &nedge, &eedges, &senses);
        CHECK_STATUS(EG_getTopology);

        if (nedge != 4) continue;

        /* spacings normal to each of the Edges */
        for (kedge = 0; kedge < 4; kedge++) {
            jedge = EG_indexBodyTopo(myJbody.ebody, eedges[kedge]);
            if (myJbody.edge[jedge].itype == DEGENERATE) continue;

            kedgem = (kedge + 3) % 4;
            kedgep = (kedge + 1) % 4;

            /* look at the previous Edge in  the loop */
            status = EG_getInfo(eedges[kedgem], &oclass, &mtype, NULL, NULL, NULL);
            CHECK_STATUS(EG_getInfo);

            if (mtype != DEGENERATE) {
                status = EG_attributeRet(eedges[kedgem], "meshT",
                                         &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
                CHECK_STATUS(EG_attributeRet);

                if (senses[kedgem] > 0) {
                    ipnt0 = attrLen - 1;
                    ipnt1 = attrLen - 2;
                } else {
                    ipnt0 = 0;
                    ipnt1 = 1;
                }

                status = EG_evaluate(eedges[kedgem], &tempRlist[ipnt0], data0);
                CHECK_STATUS(EG_evaluate);

                status = EG_evaluate(eedges[kedgem], &tempRlist[ipnt1], data1);
                CHECK_STATUS(EG_evaluate);

                normspace = sqrt((data0[0]-data1[0]) * (data0[0]-data1[0])
                                +(data0[1]-data1[1]) * (data0[1]-data1[1])
                                +(data0[2]-data1[2]) * (data0[2]-data1[2]));

                if (senses[kedge] > 0) {
                    tempList[0] = normspace;
                    tempList[2] = normspace;
                } else {
                    tempList[1] = normspace;
                }
            }

            /* look at the next Edge in the loop */
            status = EG_getInfo(eedges[kedgep], &oclass, &mtype, NULL, NULL, NULL);
            CHECK_STATUS(EG_getInfo);

            if (mtype != DEGENERATE) {
                status = EG_attributeRet(eedges[kedgep], "meshT",
                                         &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
                CHECK_STATUS(EG_attributeRet);

                if (senses[kedgep] > 0) {
                    ipnt0 = 0;
                    ipnt1 = 1;
                } else {
                    ipnt0 = attrLen - 1;
                    ipnt1 = attrLen - 2;
                }

                status = EG_evaluate(eedges[kedgep], &tempRlist[ipnt0], data0);
                CHECK_STATUS(EG_evaluate);

                status = EG_evaluate(eedges[kedgep], &tempRlist[ipnt1], data1);
                CHECK_STATUS(EG_evaluate);

                normspace = sqrt((data0[0]-data1[0]) * (data0[0]-data1[0])
                                +(data0[1]-data1[1]) * (data0[1]-data1[1])
                                +(data0[2]-data1[2]) * (data0[2]-data1[2]));

                if (senses[kedge] > 0) {
                    tempList[1] = normspace;
                } else {
                    tempList[0] = normspace;
                    tempList[2] = normspace;
                }
            }

            /* put the normals on the the Edge in ibody */
            for (iedge = 1; iedge <= myIbody.nedge; iedge++) {
                if (myIbody.edge[iedge].itype == DEGENERATE) continue;

                status = EG_attributeRet(myIbody.edge[iedge].eedge, "__meshNorm__",
                                         &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
                if (status == EGADS_SUCCESS) continue;

                status = imatch = matchEdges(MODL, ibody, iedge, jbody, jedge, EPS03);
                CHECK_STATUS(matchEdges);

                if (imatch == 1 || imatch == 4) {
                    SPRINT6(1, "        putting __meshNorm__ (%12.6f, %12.6f) on ibody=%3d, iedge=%3d from jbody=%3d, jedge=%3d",
                            tempList[0], tempList[1], ibody, iedge, jbody, jedge);

                    status = EG_attributeAdd(myIbody.edge[iedge].eedge, "__meshNorm__",
                                             ATTRREAL, 2, NULL, &tempList[0], NULL);
                    CHECK_STATUS(EG_attributeAdd);
                    break;
                } else if (imatch == 2 || imatch == 5) {
                    SPRINT6(1, "        putting __meshNorm__ (%12.6f, %12.6f) on ibody=%3d, iedge=%3d from jbody=%3d, jedge=%3d",
                            tempList[1], tempList[2], ibody, iedge, jbody, jedge);

                    status = EG_attributeAdd(myIbody.edge[iedge].eedge, "__meshNorm__",
                                             ATTRREAL, 2,NULL, &tempList[1], NULL);
                    CHECK_STATUS(EG_attributeAdd);
                    break;
                }
            }
        }
    }

    /* generate TFI grid on each Face with 4 Edges */
    SPRINT0(1, "    number of mesh points associated with each Face");
    for (jface = 1; jface <= myJbody.nface; jface++) {
        SPRINT2(2, "\nBody %3d, Face %3d\n", jbody, jface);

        status = EG_getTopology(myJbody.face[jface].eface, &esurf,
                                &oclass, &mtype, data, &nloop, &eloops, &senses);
        CHECK_STATUS(EG_getTopology);

        status = EG_getTopology(eloops[0], &eref,
                                &oclass, &mtype, data, &nedge, &eedges, &senses);
        CHECK_STATUS(EG_getTopology);

        if (nedge != 4) continue;

        SPRINT4(2, "jedges= %d %d %d %d",
                senses[0]*EG_indexBodyTopo(myJbody.ebody, eedges[0]),
                senses[1]*EG_indexBodyTopo(myJbody.ebody, eedges[1]),
                senses[2]*EG_indexBodyTopo(myJbody.ebody, eedges[2]),
                senses[3]*EG_indexBodyTopo(myJbody.ebody, eedges[3]));

        status = EG_attributeRet(eedges[0], "meshT",
                                 &attrType, &npnt0, &tempIlist, &pnt0, &tempClist);
        CHECK_STATUS(EG_attributeRet);

        status = EG_attributeRet(eedges[1], "meshT",
                                 &attrType, &npnt1, &tempIlist, &pnt1, &tempClist);
        CHECK_STATUS(EG_attributeRet);

        status = EG_attributeRet(eedges[2], "meshT",
                                 &attrType, &npnt2, &tempIlist, &pnt2, &tempClist);
        CHECK_STATUS(EG_attributeRet);

        status = EG_attributeRet(eedges[3], "meshT",
                                 &attrType, &npnt3, &tempIlist, &pnt3, &tempClist);
        CHECK_STATUS(EG_attributeRet);

        if (npnt0 == npnt2) {
            imax = npnt0;
        } else {
            SPRINT5(1, "        jface=%5d:   **mismatch**     (%5d %5d %5d %5d)", jface,
                    senses[0]*EG_indexBodyTopo(myJbody.ebody, eedges[0]),
                    senses[1]*EG_indexBodyTopo(myJbody.ebody, eedges[1]),
                    senses[2]*EG_indexBodyTopo(myJbody.ebody, eedges[2]),
                    senses[3]*EG_indexBodyTopo(myJbody.ebody, eedges[3]));
            continue;
        }

        if (npnt1 == npnt3) {
            jmax = npnt1;
        } else {
            SPRINT5(1, "        jface=%5d:   **mismatch**     (%5d %5d %5d %5d)", jface,
                    senses[0]*EG_indexBodyTopo(myJbody.ebody, eedges[0]),
                    senses[1]*EG_indexBodyTopo(myJbody.ebody, eedges[1]),
                    senses[2]*EG_indexBodyTopo(myJbody.ebody, eedges[2]),
                    senses[3]*EG_indexBodyTopo(myJbody.ebody, eedges[3]));
            continue;
        }

        MALLOC(uv, double, 2*imax*jmax);

#define U(I,J) uv[2*((I)+(J)*imax)  ]
#define V(I,J) uv[2*((I)+(J)*imax)+1]

        /* south (eedges[0]) */
        j = 0;

        for (i = 0; i < imax; i++) {
            if (senses[0] > 0) {
                t = pnt0[i];
            } else {
                t = pnt0[npnt0-1-i];
            }

            status = EG_getEdgeUV(myJbody.face[jface].eface, eedges[0], 0, t, &U(i,j));
            CHECK_STATUS(EG_getEdgeUV);
        }

        /* north (eedges[2])*/
        j = jmax - 1;

        for (i = 0; i < imax; i++) {
            if (senses[2] > 0) {
                t = pnt2[npnt2-1-i];
            } else {
                t = pnt2[i];
            }

            status = EG_getEdgeUV(myJbody.face[jface].eface, eedges[2], 0, t, &U(i,j));
            CHECK_STATUS(EG_getEdgeUV);
        }

        /* west (eedges[3]) */
        i = 0;

        for (j = 1; j < jmax-1; j++) {
            if (senses[3] > 0) {
                t = pnt3[npnt3-1-j];
            } else {
                t = pnt3[j];
            }

            status = EG_getEdgeUV(myJbody.face[jface].eface, eedges[3], 0, t, &U(i,j));
            CHECK_STATUS(EG_getEdgeUV);
        }

        /* east (eedges[1]) */
        i = imax - 1;

        for (j = 1; j < jmax-1; j++) {
            if (senses[1] > 0) {
                t = pnt1[j];
            } else {
                t = pnt1[npnt1-1-j];
            }

            status = EG_getEdgeUV(myJbody.face[jface].eface, eedges[1], 0, t, &U(i,j));
            CHECK_STATUS(EG_getEdgeUV);
        }

        /* TFI */
        for (i = 1; i < imax-1; i++) {
            fraci = (double)(i) / (double)(imax-1);
            for (j = 1; j < jmax-1; j++) {
                fracj = (double)(j) / (double)(jmax-1);

                U(i,j)
                    = (1-fraci)             * U(0     ,j     )
                    +    fraci              * U(imax-1,j     )
                    +             (1-fracj) * U(i     ,0     )
                    +                fracj  * U(i     ,jmax-1)
                    - (1-fraci) * (1-fracj) * U(0     ,0     )
                    -    fraci  * (1-fracj) * U(imax-1,0     )
                    - (1-fraci) *    fracj  * U(0     ,jmax-1)
                    -    fraci  *    fracj  * U(imax-1,jmax-1);
                V(i,j)
                    = (1-fraci)             * V(0     ,j     )
                    +    fraci              * V(imax-1,j     )
                    +             (1-fracj) * V(i     ,0     )
                    +                fracj  * V(i     ,jmax-1)
                    - (1-fraci) * (1-fracj) * V(0     ,0     )
                    -    fraci  * (1-fracj) * V(imax-1,0     )
                    - (1-fraci) *    fracj  * V(0     ,jmax-1)
                    -    fraci  *    fracj  * V(imax-1,jmax-1);
            }
        }

        SPRINT0(2, "** U **");
        for (i = imax-1; i >= 0; i--) {
            SPRINT1x(2, "i=%2d:", i);
            for (j = 0; j < jmax; j++) {
                SPRINT1x(2, " %6.3f", U(i,j));
            }
            SPRINT0(2, " ");
        }
        SPRINT0(2, "** V **");
        for (i = imax-1; i >= 0; i--) {
            SPRINT1x(2, "i=%2d:", i);
            for (j = 0; j < jmax; j++) {
                SPRINT1x(2, " %6.3f", V(i,j));
            }
            SPRINT0(2, " ");
        }

        /* put mesh info attributes on the Faces in ibody */
        size[0] = imax;
        size[1] = jmax;

        origin[0] = jbody;
        origin[1] = jface;

        SPRINT2(2, "jbody=%3d, jface=%3d", jbody, jface);

        for (iface = 1; iface <= myIbody.nface; iface++) {
            status = EG_attributeRet(myIbody.face[iface].eface, "__trace__",
                                     &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
            CHECK_STATUS(EG_attributeRet);

            for (itrace = 0; itrace < attrLen; itrace+=2) {
                if (tempIlist[itrace] == jbody && tempIlist[itrace+1] == jface) {
                    SPRINT2(2, "       putting attributes on ibody=%3d, iface=%3d", ibody, iface);

                    status = EG_attributeAdd(myIbody.face[iface].eface, "meshOrigin",
                                             ATTRINT, 2, origin, NULL, NULL);
                    CHECK_STATUS(EG_attributeAdd);

                    status = EG_attributeAdd(myIbody.face[iface].eface, "meshSize",
                                             ATTRINT, 2, size, NULL, NULL);
                    CHECK_STATUS(EG_attributeAdd);

                    status = EG_attributeAdd(myIbody.face[iface].eface, "meshUV",
                                             ATTRREAL, 2*imax*jmax, NULL, uv, NULL);
                    CHECK_STATUS(EG_attributeAdd);

                    break;
                }
            }
        }

        FREE(uv);

        SPRINT7(1, "        jface=%5d: %5d*%5d points (%5d %5d %5d %5d)", jface, imax, jmax,
                senses[0]*EG_indexBodyTopo(myJbody.ebody, eedges[0]),
                senses[1]*EG_indexBodyTopo(myJbody.ebody, eedges[1]),
                senses[2]*EG_indexBodyTopo(myJbody.ebody, eedges[2]),
                senses[3]*EG_indexBodyTopo(myJbody.ebody, eedges[3]));
    }

    /* transfer the mesh info attributes on the Edges from the component
       Body to the final Body */
    for (jedge = 1; jedge <= myJbody.nedge; jedge++) {
        if (myJbody.edge[jedge].itype == DEGENERATE) continue;

        for (iedge = 1; iedge <= myIbody.nedge; iedge++) {
            if (myIbody.edge[iedge].itype == DEGENERATE) continue;

            status = EG_attributeRet(myIbody.edge[iedge].eedge, "__trace__",
                                     &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
            CHECK_STATUS(EG_attributeRet);

            for (itrace = 0; itrace < attrLen; itrace+=2) {
                if (tempIlist[itrace] == jbody && tempIlist[itrace+1] == jedge) {
                    SPRINT4(2, "transferring meshT from jbody=%3d, jedge=%3d to ibody=%3d, iedge=%3d",
                            jbody, jedge, ibody, iedge);
                    status = EG_attributeRet(myJbody.edge[jedge].eedge, "meshT",
                                             &attrType2, &attrLen2, &tempIlist2, &tempRlist2, &tempClist2);
                    CHECK_STATUS(EG_attributeRet);

                    status = EG_getRange(myIbody.edge[iedge].eedge, trange, &periodic);
                    CHECK_STATUS(EG_getRange);

                    status = EG_evaluate(myIbody.edge[iedge].eedge, &trange[0], xyzi0);
                    CHECK_STATUS(EG_evaluate);

                    status = EG_evaluate(myIbody.edge[iedge].eedge, &trange[1], xyzi1);
                    CHECK_STATUS(EG_evaluate);

                    status = EG_getRange(myJbody.edge[jedge].eedge, trange, &periodic);
                    CHECK_STATUS(EG_getRange);

                    status = EG_evaluate(myJbody.edge[jedge].eedge, &trange[0], xyzj0);
                    CHECK_STATUS(EG_evaluate);

                    status = EG_evaluate(myJbody.edge[jedge].eedge, &trange[1], xyzj1);
                    CHECK_STATUS(EG_evaluate);

                    dotprod = (xyzi1[0] - xyzi0[0]) * (xyzj1[0] - xyzj0[0])
                        +     (xyzi1[1] - xyzi0[1]) * (xyzj1[1] - xyzj0[1])
                        +     (xyzi1[2] - xyzi0[2]) * (xyzj1[2] - xyzj0[2]);

                    if (dotprod > 0) {
                        status = EG_attributeAdd(myIbody.edge[iedge].eedge, "meshT",
                                                 attrType2, attrLen2, tempIlist2, tempRlist2, tempClist2);
                        CHECK_STATUS(EG_attributeAdd);
                    } else {
                        MALLOC(flipRlist, double, attrLen2);

                        for (i = 0; i < attrLen2; i++) {
                            flipRlist[i] = 1 - tempRlist2[attrLen2-1-i];
                        }

                        status = EG_attributeAdd(myIbody.edge[iedge].eedge, "meshT",
                                                 attrType2, attrLen2, tempIlist, flipRlist, tempClist);
                        CHECK_STATUS(EG_attributeAdd);

                        FREE(flipRlist);
                    }

                    break;
                }
            }
        }

        /* this attribute might not exist */
        (void) EG_attributeDel(myJbody.edge[jedge].eedge, "meshT");
    }

cleanup:
    FREE(uv);
    FREE(group);
    FREE(meshFixed);
    FREE(flipRlist);

#undef U
#undef V

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   makeIntersectionMesh - set up mesh on intersection Edges           *
 *                                                                      *
 ************************************************************************
 */

static int
makeIntersectionMesh(modl_T  *MODL,     /* (in)  pointer to MODL */
                     int     ibody)     /* (in)  index of combined  Body */
{
    int       status = SUCCESS;

    int       iedge, attrType, attrLen, ileft, irite, jbody, ibody1, ibody2, ibeg, iend, nface;
    int       i, j, periodic, ipnt, npnt=100, icross, jcross, ncross, nchange, special, found;
    CINT      *tempIlist, *sizeleft, *sizerite;
    double    trange[2], t1, uv1[2], t2, uv2[2], frac1, frac2, *cross1=NULL, *cross2=NULL, swap;
    double    priority1, priority2;
    CDOUBLE   *tempRlist, *uvleft, *uvrite;
    CCHAR     *tempClist, *tempClist1, *tempClist2;
    ego       *efaces;
    body_T    myBody;

#define Uleft(I,J) uvleft[2*((I)+(J)*sizeleft[0])  ]
#define Vleft(I,J) uvleft[2*((I)+(J)*sizeleft[0])+1]
#define Urite(I,J) uvrite[2*((I)+(J)*sizerite[0])  ]
#define Vrite(I,J) uvrite[2*((I)+(J)*sizerite[0])+1]

    ROUTINE(makeIntersectionMesh);

    /* --------------------------------------------------------------- */

    SPRINT1(2, "makeIntersctionMesh(ibody=%d)", ibody);

    myBody = MODL->body[ibody];

    MALLOC(cross1, double, MAX_NPNT);
    MALLOC(cross2, double, MAX_NPNT);

    SPRINT0(1, "making intersection meshes");

    /* loop through all non-degenerate Edges and see if it
       spans two components */
    for (iedge = 1; iedge <= myBody.nedge; iedge++) {
        if (myBody.edge[iedge].itype == DEGENERATE) continue;

        /* find special Edges that were split and which are non-manifold.
           these come from intersecting an Edge in the SolidBody with
           a wake SheetBody that happen to be coincident */
        special = 0;

        if (myBody.edge[iedge].nface > 2) {
            ibeg = myBody.edge[iedge].ibeg;
            iend = myBody.edge[iedge].iend;

            status = EG_attributeRet(myBody.node[ibeg].enode, "_Keep",
                                     &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
            if (status == EGADS_SUCCESS && attrType == ATTRSTRING &&
                strcmp(tempClist, "subtract") == 0                  ) {
                special = 1;
            }

            status = EG_attributeRet(myBody.node[iend].enode, "_Keep",
                                     &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
            if (status == EGADS_SUCCESS && attrType == ATTRSTRING &&
                strcmp(tempClist, "subtract") == 0                  ) {
                special = 1;
            }
        }

        /* if not special, get meshT, ileft, and irite */
        if (special == 0) {
            status = EG_attributeRet(myBody.edge[iedge].eedge, "meshT",
                                     &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
            if (status == EGADS_SUCCESS) continue;

            ileft = myBody.edge[iedge].ileft;
            irite = myBody.edge[iedge].irite;

        /* this is special, so find an ileft and irite that come from different
           components.  also remove the meshT attribute so that a meshTint
           attribute is added below */
        } else {
            status = EG_getBodyTopos(myBody.ebody, myBody.edge[iedge].eedge,
                                     FACE, &nface, &efaces);
            CHECK_STATUS(EG_getBodyTopos);

            ileft = EG_indexBodyTopo(myBody.ebody, efaces[0]);

            status = EG_attributeRet(myBody.face[ileft].eface, "meshComponent",
                                     &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist1);
            if (status != EGADS_SUCCESS || attrType != ATTRSTRING) {
                EG_free(efaces);
                continue;
            }

            irite = 0;
            for (i = 1; i < nface; i++) {
                irite = EG_indexBodyTopo(myBody.ebody, efaces[i]);

                status = EG_attributeRet(myBody.face[irite].eface, "meshComponent",
                                         &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist2);
                if (status != EGADS_SUCCESS || attrType != ATTRSTRING) continue;

                if (strcmp(tempClist1, tempClist2) != 0) break;
            }

            EG_free(efaces);

            /* this attribute might not exist */
            (void) EG_attributeDel(myBody.edge[iedge].eedge, "meshT");
        }

        if (ileft <= 0 || irite <= 0) continue;

        status = EG_attributeRet(myBody.face[ileft].eface, "meshComponent",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist1);
        if (status != EGADS_SUCCESS || attrType != ATTRSTRING) continue;

        status = EG_attributeRet(myBody.face[irite].eface, "meshComponent",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist2);
        if (status != EGADS_SUCCESS || attrType != ATTRSTRING) continue;

        if (strcmp(tempClist1, tempClist2) == 0) continue;

        /* get the priority associated with the two components */
        status = EG_attributeRet(myBody.face[ileft].eface, "meshPriority",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS && attrType == ATTRREAL) {
            priority1 = tempRlist[0];
        } else {
            priority1 = 0;
        }

        status = EG_attributeRet(myBody.face[irite].eface, "meshPriority",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS && attrType == ATTRREAL) {
            priority2 = tempRlist[0];
        } else {
            priority2 = 0;
        }

        SPRINT2(2, "    tempClist1=%-20s, priority1=%f", tempClist1, priority1);
        SPRINT2(2, "    tempClist2=%-20s, priority2=%f", tempClist2, priority2);


        SPRINT3(1, "iedge=%5d spans %-20s (priority=%5.2f)", iedge, tempClist1, priority1);
        SPRINT2(1, "                  %-20s (priority=%5.2f)",      tempClist2, priority2);

        /* find component Bodys that made this Edge */
        ibody1 = 0;
        ibody2 = 0;
        for (jbody = 1; jbody < ibody; jbody++) {
            status = EG_attributeRet(MODL->body[jbody].ebody, "meshComponent",
                                     &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
            if (status == EGADS_SUCCESS) {
                if (ibody1 == 0 && strcmp(tempClist, tempClist1) == 0) {
                    ibody1 = jbody;
                }
                if (ibody2 == 0 && strcmp(tempClist, tempClist2) == 0) {
                    ibody2 = jbody;
                }
            }
        }

        if (ibody1 == 0) {
            SPRINT1(1, "could not find body with meshComponent=%s", tempClist1);
            continue;
        } else if (ibody2 == 0) {
            SPRINT1(1, "could not find body with meshComponent=%s", tempClist2);
            continue;
        }

        /* no intersection if ileft or irite has meshSurfType=none */
        status = EG_attributeRet(myBody.face[ileft].eface, "meshSurfType",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS && attrType == ATTRSTRING && strcmp(tempClist,"none") == 0) {
            continue;
        }

        status = EG_attributeRet(myBody.face[irite].eface, "meshSurfType",
                                 &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS && attrType == ATTRSTRING && strcmp(tempClist,"none") == 0) {
            continue;
        }

        /* get the meshes for ileft and irite */
        status = EG_attributeRet(myBody.face[ileft].eface, "meshSize",
                                 &attrType, &attrLen, &sizeleft,     &tempRlist, &tempClist);
        CHECK_STATUS(EG_attributeRet);

        status = EG_attributeRet(myBody.face[ileft].eface, "meshUV",
                                 &attrType, &attrLen, &tempIlist, &uvleft,    &tempClist);
        CHECK_STATUS(EG_attributeRet);

        status = EG_attributeRet(myBody.face[irite].eface, "meshSize",
                                 &attrType, &attrLen, &sizerite,     &tempRlist, &tempClist);
        CHECK_STATUS(EG_attributeRet);

        status = EG_attributeRet(myBody.face[irite].eface, "meshUV",
                                 &attrType, &attrLen, &tempIlist, &uvrite,    &tempClist);
        CHECK_STATUS(EG_attributeRet);

        /* look at 100 segments along the Edge */
        status = EG_getRange(myBody.edge[iedge].eedge, trange, &periodic);
        CHECK_STATUS(EG_getRange);

        ncross = 0;

        for (ipnt = 1; ipnt <= npnt; ipnt++) {
            t1 = trange[0] + (trange[1] - trange[0]) * (double)(ipnt-1) / (double)(npnt);
            t2 = trange[0] + (trange[1] - trange[0]) * (double)(ipnt  ) / (double)(npnt);

            /* add crossings for ileft */
            if (priority1 >= priority2) {
                status = EG_getEdgeUV(myBody.face[ileft].eface,
                                      myBody.edge[iedge].eedge, +1, t1, uv1);
                CHECK_STATUS(EG_getEdgeUV);

                status = EG_getEdgeUV(myBody.face[ileft].eface,
                                      myBody.edge[iedge].eedge, +1, t2, uv2);
                CHECK_STATUS(EG_getEdgeUV);

                for (i = 0; i < sizeleft[0]; i++) {
                    for (j = 1; j < sizeleft[1]; j++) {
                        status = intersectSegs(uv1, uv2, &Uleft(i,j-1), &Uleft(i,j), &frac1, &frac2);
                        if (status == EGADS_SUCCESS) {
                            cross1[ncross++] = (1-frac1) * t1 + frac1 * t2;
                        }
                    }
                }

                for (j = 0; j < sizeleft[1]; j++) {
                    for (i = 1; i < sizeleft[0]; i++) {
                        status = intersectSegs(uv1, uv2, &Uleft(i-1,j), &Uleft(i,j), &frac1, &frac2);
                        if (status == EGADS_SUCCESS) {
                            cross1[ncross++] = (1-frac1) * t1 + frac1 * t2;
                        }
                    }
                }
            }

            /* add crossings for irite */
            if (priority2 >= priority1) {
                status = EG_getEdgeUV(myBody.face[irite].eface,
                                      myBody.edge[iedge].eedge, -1, t1, uv1);
                CHECK_STATUS(EG_getEdgeUV);

                status = EG_getEdgeUV(myBody.face[irite].eface,
                                      myBody.edge[iedge].eedge, -1, t2, uv2);
                CHECK_STATUS(EG_getEdgeUV);

                for (i = 0; i < sizerite[0]; i++) {
                    for (j = 1; j < sizerite[1]; j++) {
                        status = intersectSegs(uv1, uv2, &Urite(i,j-1), &Urite(i,j), &frac1, &frac2);
                        if (status == EGADS_SUCCESS) {
                            cross1[ncross++] = (1-frac1) * t1 + frac1 * t2;
                        }
                    }
                }

                for (j = 0; j < sizerite[1]; j++) {
                    for (i = 1; i < sizerite[0]; i++) {
                        status = intersectSegs(uv1, uv2, &Urite(i-1,j), &Urite(i,j), &frac1, &frac2);
                        if (status == EGADS_SUCCESS) {
                            cross1[ncross++] = (1-frac1) * t1 + frac1 * t2;
                        }
                    }
                }
            }
        }

        /* add trange if not already in cross array */
        found = 0;
        for (jcross = 0; jcross < ncross; jcross++) {
            if (fabs(trange[0]-cross1[jcross]) < EPS06) {
                found = 1;
                break;
            }
        }
        if (found == 0) {
            cross1[ncross++] = trange[0];
        }

        found = 0;
        for (jcross = 0; jcross < ncross; jcross++) {
            if (fabs(trange[1]-cross1[jcross]) < EPS06) {
                found = 1;
                break;
            }
        }
        if (found == 0) {
            cross1[ncross++] = trange[1];
        }

        /* sort cross */
        for (jcross = 0; jcross < 2*ncross; jcross++) {
            nchange = 0;
            for (icross = 0; icross < ncross-1; icross++) {
                if (cross1[icross+1] < cross1[icross]) {
                    swap             = cross1[icross+1];
                    cross1[icross+1] = cross1[icross  ];
                    cross1[icross  ] = swap;
                    nchange++;
                }
            }
            if (nchange == 0) break;
        }

        /* smooth the interior points */
        for (icross = 0; icross < ncross; icross++) {
            cross2[icross] = cross1[icross];
        }
        for (icross = 1; icross < ncross-1; icross++) {
            cross1[icross] = (cross2[icross-1] + 2*cross2[icross] + cross2[icross+1]) / 4;
        }

        status = EG_attributeAdd(myBody.edge[iedge].eedge, "meshTint",
                                 ATTRREAL, ncross, NULL, cross1, NULL);
        CHECK_STATUS(EG_attributeAdd);

        SPRINT2(1, "        putting meshTint on ibody=%3d, iedge=%3d", ibody, iedge);

        SPRINT2x(2, "putting meshTint ibody=%3d, iedge=%3d:", ibody, iedge);
        for (icross = 0; icross < ncross; icross++) {
            SPRINT1x(2, " %10.5f", cross1[icross]);
        }
        SPRINT0(2, " ");

        /* verify that the cross value are monotonic */
        for (icross = 1; icross < ncross; icross++) {
            if (cross1[icross] <= cross1[icross-1]) {
                printf("not monotonic, iedge=%d, icross=%d\n", iedge, icross);
            }
        }

        SPRINT1(1, "            npnt=%5d", ncross);
    }

    status = EGADS_SUCCESS;

cleanup:
    FREE(cross1);
    FREE(cross2);

#undef Uleft
#undef Vleft
#undef Urite
#undef Urite

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   checkMeshInfo - check for consistency in meshInfo                  *
 *                                                                      *
 ************************************************************************
 */

#ifdef UNUSED
//$$$static int
//$$$checkMeshInfo(modl_T   *MODL,           /* (in)  pointer to MODL */
//$$$              int      ibody,           /* (in)  Body index */
//$$$              int      iedge,           /* (in)  Edge index */
//$$$              char     message[])       /* (out) error message */
//$$${
//$$$    int       status = SUCCESS;
//$$$
//$$$    int     attrType, attrLen, periodic, nmin, nmax;
//$$$    CINT    *tempIlist;
//$$$    double  trange[2], stotal, dsbeg, dsend, dsmax;
//$$$    CDOUBLE *tempRlist;
//$$$    CCHAR   *tempClist;
//$$$
//$$$    ROUTINE(checkMeshInfo);
//$$$
//$$$    /* --------------------------------------------------------------- */
//$$$
//$$$    SPRINT2(2, "checkMeshInfo(ibody=%d, iedge=%d", ibody, iedge);
//$$$
//$$$    if (MODL->body[ibody].edge[iedge].itype == DEGENERATE) goto cleanup;
//$$$
//$$$    status = EG_attributeRet(MODL->body[ibody].edge[iedge].eedge, "meshInfo",
//$$$                             &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
//$$$    if (status == EGADS_SUCCESS) {
//$$$        status = EG_getRange(MODL->body[ibody].edge[iedge].eedge, trange, &periodic);
//$$$        CHECK_STATUS(EG_getRange);
//$$$
//$$$        status = EG_arcLength(MODL->body[ibody].edge[iedge].eedge, trange[0], trange[1], &stotal);
//$$$        CHECK_STATUS(EG_arcLength);
//$$$
//$$$        if (tempRlist[0] <= -99998) goto cleanup; // VORLAX-type spacing specified
//$$$
//$$$        if (tempRlist[0] >= 0) {
//$$$            dsbeg =  tempRlist[0] * stotal;
//$$$        } else {
//$$$            dsbeg = -tempRlist[0];
//$$$        }
//$$$        if (tempRlist[1] >= 0) {
//$$$            dsend =  tempRlist[1] * stotal;
//$$$        } else {
//$$$            dsend = -tempRlist[1];
//$$$        }
//$$$        if (tempRlist[2] >= 0) {
//$$$            dsmax =  tempRlist[2] * stotal;
//$$$        } else {
//$$$            dsmax = -tempRlist[2];
//$$$        }
//$$$        nmin  = NINT(tempRlist[3]);
//$$$        nmax  = NINT(tempRlist[4]);
//$$$
//$$$        if        (dsbeg*(nmin-1) > stotal) {
//$$$            snprintf(message, 100, "illegal meshInfo: ibody=%d, iedge=%d, dsbeg=%10.5f, nmin=%d, stotal=%10.5f",
//$$$                     ibody, iedge, dsbeg, nmin, stotal);
//$$$            status = OCSM_ILLEGAL_VALUE;
//$$$            goto cleanup;
//$$$        } else if (dsend*(nmin-1) > stotal) {
//$$$            snprintf(message, 100, "illegal meshInfo: ibody=%d, iedge=%d, dsend=%10.5f, nmin=%d, stotal=%10.5f",
//$$$                     ibody, iedge, dsend, nmin, stotal);
//$$$            status = OCSM_ILLEGAL_VALUE;
//$$$            goto cleanup;
//$$$        } else if (dsmax*(nmax-1) < stotal) {
//$$$            snprintf(message, 100, "illegal meshInfo: ibody=%d, iedge=%d, dsmax=%10.5f, nmax=%d, stotal=%10.5f",
//$$$                     ibody, iedge, dsmax, nmax, stotal);
//$$$            status = OCSM_ILLEGAL_VALUE;
//$$$            goto cleanup;
//$$$        }
//$$$    }
//$$$
//$$$cleanup:
//$$$    return status;
//$$$}
#endif


/*
 ************************************************************************
 *                                                                      *
 *   edgeSpacing - set up stretched distribution on an Edge             *
 *                                                                      *
 ************************************************************************
 */

static int
edgeSpacing(ego     eedge,              /* (in)  Edge */
            int     *npnt,              /* (in)  target number of points (or zero) */
                                        /* (out) actual number of points */
            char    message[])          /* (out) error message buffer */
{
    int       status = SUCCESS;

    int       nmin, nmax, dirn;
    int       periodic, newNpnt, ipnt, icycle, iter, nabs;
    int       oclass, mtype, nchild, *senses;
    int       attrType, attrLen;
    CINT      *tempIlist;
    double    dsbeg, dsend, dsmax, swap;
    double    trange[2], tbeg, tend, send, dstest, dsbig, xyzbeg[18], xyzend[18];
    double    pabs, pequ, pcos, psin, frac, theta;
    double    a, da, b, db, f, dfda, dfdb, g, dgda, dgdb, u, v, duda, ri, rj;
    double    data[18], *newS=NULL, *newT=NULL, *xyzs=NULL;
    CDOUBLE   *tempRlist;
    CCHAR     *tempClist;
    ego       ecurve, *echilds;
    void      *realloc_temp=NULL;       /* used by RALLOC macro */

    int       ncycle=MAX_NPNT, niter=100;
    double    omega=0.90, damin=-10.0, damax=+10.0;

    ROUTINE(edgeSpacing);

    /* --------------------------------------------------------------- */

    SPRINT1(2, "edgeSpacing(*npnt=%d)", *npnt);

    /* get t at the beginning and end */
    status = EG_getRange(eedge, trange, &periodic);
    CHECK_STATUS(EG_getRange);

    tbeg = trange[0];
    tend = trange[1];
    SPRINT2(2, "    tbeg=%f, tend=%f", tbeg, tend);

    /* get the total arclength (note that sbeg is assumed to be 0) */
    status = EG_arcLength(eedge, tbeg, tend, &send);
    CHECK_STATUS(EG_arcLength);
    SPRINT1(2, "    send=%f", send);

    /* extract the Edge's meshing parameters */
    status = EG_attributeRet(eedge, "__meshInfo__", &attrType, &attrLen,
                             &tempIlist, &tempRlist, &tempClist);
    CHECK_STATUS(EG_attributeRet);

    if (tempRlist[0] <= -99998) {
        dsbeg = tempRlist[0];
        dsend = tempRlist[1];
        dsmax = 0;
    } else {
        if (tempRlist[0] >= 0) {
            dsbeg =  tempRlist[0] * send;
        } else {
            dsbeg = -tempRlist[0];
        }
        if (tempRlist[1] >= 0) {
            dsend =  tempRlist[1] * send;
        } else {
            dsend = -tempRlist[1];
        }
        if (tempRlist[2] >= 0) {
            dsmax =  tempRlist[2] * send;
        } else {
            dsmax = -tempRlist[2];
        }
    }
    nmin  = NINT(tempRlist[3]);
    nmax  = NINT(tempRlist[4]);
    dirn  = NINT(tempRlist[5]);
    SPRINT3(2, "    dsbeg=%f, dsend=%f, dsmax=%f", dsbeg, dsend, dsmax);
    SPRINT3(2, "    nmin=%d, nmax=%d, dirn=%d", nmin, nmax, dirn);

    /* if the Edge is degenerate, use a uniform distribution */
    status = EG_getInfo(eedge, &oclass, &mtype, NULL, NULL, NULL);
    CHECK_STATUS(EG_getInfo);

    if (mtype == DEGENERATE) {
        SPRINT0(2, "    DEGENERATE");
        if (*npnt > 0) {
            newNpnt = *npnt;
        } else {
            newNpnt = nmin;
        }

        MALLOC(newT, double, newNpnt);

        for (ipnt = 0; ipnt < newNpnt; ipnt++) {
            newT[ipnt] = tbeg + (tend - tbeg) * (double)(ipnt) / (double)(newNpnt-1);
        }

        /* put the t distribution on the Edge */
        status = EG_attributeAdd(eedge, "meshT", ATTRREAL, newNpnt, NULL, newT, NULL);
        CHECK_STATUS(EG_attributeAdd);

        goto cleanup;
    }

    /* if dirn is 1, 2, or 3, possibly reverse dsbeg and dsend if the
       Edge direction is not as specified */
    status = EG_evaluate(eedge, &tbeg, xyzbeg);
    CHECK_STATUS(EG_evaluate);

    status = EG_evaluate(eedge, &tend, xyzend);
    CHECK_STATUS(EG_evaluate);

    if        (dirn == 1) {
        if (fabs(xyzbeg[0]) > fabs(xyzend[0])) {
            SPRINT0(2, "    swapping");
            swap  = dsbeg;
            dsbeg = dsend;
            dsend = swap;
        }
    } else if (dirn == 2) {
        if (fabs(xyzbeg[1]) > fabs(xyzend[1])) {
            SPRINT0(2, "    swapping");
            swap  = dsbeg;
            dsbeg = dsend;
            dsend = swap;
        }
    } else if (dirn == 3) {
        if (fabs(xyzbeg[2]) > fabs(xyzend[2])) {
            SPRINT0(2, "    swapping");
            swap  = dsbeg;
            dsbeg = dsend;
            dsend = swap;
        }
    }

    /* if either dsbeg or dsbeg <= -99998 (which can happen because of
       swapping above) and *npnt is zero, use nmax */
    if (*npnt == 0 && (dsbeg <= -99998 || dsend <= -99998)) {
        newNpnt = nmax;

    /* if *npnt is zero, find the newNpnt that would correspond to a
       uniform distribution with ds <= dsmax */
    } else if (*npnt == 0) {
        newNpnt = 1 + send / dsmax;

    /* oherwise use the given npnt */
    } else {
        newNpnt = *npnt;
    }

    if        (newNpnt < nmin) {
        newNpnt = nmin;
    } else if (newNpnt > nmax) {
        newNpnt = nmax;
    }

    /* do not allow dsbeg and dsend to be smaller than uniform spacing
       if the number of points was specified */
    if (dsbeg > -99998 && dsend > -99998) {
        if (dsbeg > send/(newNpnt-1)) {
            dsbeg = send / (newNpnt-1);
            SPRINT1(2, "changing dsbeg=%f", dsbeg);
        }

        if (dsend > send/(newNpnt-1)) {
            dsend = send / (newNpnt-1);
            SPRINT1(2, "changing dsend=%f", dsend);
        }
    }

    /* we need multiple cycles since we may have to bump up newNpnt so
       that the maximum ds <= dsmax */
    for (icycle = 0; icycle < ncycle; icycle++) {
        SPRINT2(2, "beg of icycle=%d, newNpnt=%d", icycle, newNpnt);

        /* get the storage at this newNpnt */
        RALLOC(newS, double, newNpnt);

        /* npnt = 2 yields just the endpoints (and ignores dsbeg and dsend) */
        if (newNpnt == 2) {
            if (icycle == 0 && (dsbeg != 0 || dsend != 0)) {
                snprintf(message, 100, "dsbeg=%f or dsend=%f ignored for npnt=2", dsbeg, dsend);
            }

            newS[0] = 0;
            newS[1] = send;

        /* no spacings are given, so use a uniform distribution */
        } else if (dsbeg == 0 && dsend == 0) {
            for (ipnt = 0; ipnt < newNpnt; ipnt++) {
                newS[ipnt] = (double)(ipnt) / (double)(newNpnt-1) * send;
            }

       /* VORLAX-style distribution */
        } else if (dsbeg <= -99998 || dsend <= -99998) {
            if (dsbeg <= -99998) {
                pabs = fabs(dsend);
                nabs = (int)pabs + 1;
            } else {
                pabs = fabs(dsbeg);
                nabs = (int)pabs + 1;
            }

            if (nabs == 1) {
                pequ = 1.0 - pabs;
                pcos = pabs;
                psin = 0;
            } else if (nabs == 2) {
                pequ = 0;
                pcos = 2.0 - pabs;
                psin = pabs - 1.0;
            } else {       // nabs >= 3
                pequ = pabs - 2.0;
                pcos = 0;
                psin = 3.0 - pabs;
            }

            for (ipnt = 0; ipnt < newNpnt; ipnt++) {
                frac = (double)(ipnt) / (double)(newNpnt-1);
                theta = PI * frac;
                if (dsend >= 0) {
                    newS[ipnt] = send * ( pequ * frac
                                        + pcos * (1.0 - cos(theta)  ) / 2.0
                                        + psin * (1.0 - cos(theta/2))      );
                } else {
                    newS[ipnt] = send * ( pequ * frac
                                        + pcos * (1.0 - cos(theta)  ) / 2.0
                                        + psin * sin(theta/2));
                }
            }

            break;

        /* only spacing at beginning is given (one-sided tanh) */
        } else if (dsend == 0) {
            a = 1.0;
            f = 1.0;

            for (iter = 0; iter < niter; iter++) {
                if (fabs(f) <= EPS06*send) break;

                ri   = (double)(1) / (double)(newNpnt-1) - 1;
                f    =  send * (1.0 + tanh(a*ri) / tanh(a)) - dsbeg;
                /*@-evalorder@*/
                dfda =  send * ( tanh(a   ) * ri * (1 - pow(tanh(a*ri),2))
                               - tanh(a*ri) *      (1 - pow(tanh(a   ),2))) / pow(tanh(a),2);
                /*@+evalorder@*/
                da = -f / dfda;
                a += omega * MIN(MAX(damin, da), damax);
            }

            for (ipnt = 0; ipnt < newNpnt; ipnt++) {
                ri         = (double)(ipnt) / (double)(newNpnt-1) - 1;
                newS[ipnt] = send * (1.0 + tanh(a*ri) / tanh(a));
            }

        /* only spacing at end is given (one-sided tanh) */
        } else if (dsbeg == 0) {
            a = 1.0;
            g = 1.0;

            for (iter = 0; iter < niter; iter++) {
                if (fabs(g) <= EPS06*send) break;

                rj   = (double)(newNpnt-2) / (double)(newNpnt-1);
                g    =  send * (1 - tanh(a*rj) / tanh(a)) - dsend;
                /*@-evalorder@*/
                dgda = -send * ( tanh(a   ) * rj * (1 - pow(tanh(a*rj),2))
                               - tanh(a*rj) *      (1 - pow(tanh(a   ),2))) / pow(tanh(a),2);
                /*@+evalorder@*/
                da = -g / dgda;
                a += omega * MIN(MAX(damin, da), damax);
            }

            for (ipnt = 0; ipnt < newNpnt; ipnt++) {
                rj         = (double)(ipnt) / (double)(newNpnt-1);
                newS[ipnt] = send * (tanh(a*rj) / tanh(a));
            }

        /* spacing given at both ends (two-sided tanh) */
        } else {
            a = 1.0;
            b = 0.5;
            f = 1.0;
            g = 1.0;

            for (iter = 0; iter < niter; iter++) {
                if (fabs(f) <= EPS06*send && fabs(g) <= EPS06*send) break;

                ri  = (double)(3-newNpnt) / (double)(newNpnt-1);
                u   = 0.50 * (1.0 + tanh(a*ri) / tanh(a));
                v   = b + (1.0 - b) * u;
                f   = send * u / v - dsbeg;

                /*@-evalorder@*/
                duda = 0.50 * (ri * sinh(a) * cosh(a)/pow(cosh(a*ri),2) - tanh(a*ri)) / pow(sinh(a),2);
                dfda =  send * b * duda    / pow(v,2);
                dfdb = -send * u * (1.0-u) / pow(v,2);
                /*@+evalorder@*/

                rj  = (double)(newNpnt-3) / (double)(newNpnt-1);
                u   = 0.50 * (1.0 + tanh(a*rj) / tanh(a));
                v   = b + (1.0 - b) * u;
                g   = send * (1.0 - u / v) - dsend;

                /*@-evalorder@*/
                duda = 0.50 * (rj * sinh(a) * cosh(a)/pow(cosh(a*rj),2) - tanh(a*rj)) / pow(sinh(a),2);
                dgda = -send * b * duda    / pow(v,2);
                dgdb =  send * u * (1.0-u) / pow(v,2);
                /*@-evalorder@*/

                da =(-f * dgdb + g * dfdb) / (dfda * dgdb - dgda * dfdb);
                db =( f * dgda - g * dfda) / (dfda * dgdb - dgda * dfdb);
                a += omega * MIN(MAX(damin, da), damax);
                b += omega * MIN(MAX(damin, db), damax);
            }

            for (ipnt = 0; ipnt < newNpnt; ipnt++) {
                ri         = (double)(2*ipnt-newNpnt+1) / (double)(newNpnt-1);
                u          = 0.50 * (1.0 + tanh(a*ri)/tanh(a));
                v          = b + (1.0 - b) * u;
                newS[ipnt] = send * u / v;
            }
        }

        /* find the maximum spacing */
        dsbig = 0;
        for (ipnt = 1; ipnt < newNpnt; ipnt++) {
            dstest = newS[ipnt] - newS[ipnt-1];
            if (dstest > dsbig) dsbig = dstest;
        }

        SPRINT3(2, "end of icycle=%d, dsbig=%f, dsmax=%f", icycle, dsbig, dsmax);

        if (newNpnt >= MAX_NPNT) break;

        /* if dstest <= dsmax we are done */
        if (dsmax == 0 || dsbig <= dsmax) break;

        /* if *npnt (the input value) is 0, increase newNpnt and go
           back for another cycle */
        if (*npnt == 0) {
            newNpnt++;

        /* otherwise return an error */
        } else {
            snprintf(message, 100, "pnt specified gives spacing (%10.5f) > dsmax (%10.5f)", dsbig, dsmax);
            status = OCSM_ILLEGAL_VALUE;
            goto cleanup;
        }
    }

    /* convert the (normalized) S distribution into a T distribution */
    SPLINT_CHECK_FOR_NULL(newS);

    MALLOC(newT, double,   newNpnt);
    MALLOC(xyzs, double, 3*newNpnt);

    newT[0        ] = tbeg;
    newT[newNpnt-1] = tend;

    for (ipnt = 0; ipnt < newNpnt; ipnt++) {
        newS[ipnt] /= send;
    }

    status = EG_getTopology(eedge, &ecurve, &oclass, &mtype,
                            data, &nchild, &echilds, &senses);
    CHECK_STATUS(EG_getTopology);

    status = EG_relPosTs(ecurve, newNpnt, &(newS[1]), newT, xyzs);
    CHECK_STATUS(EG_relPosTs);

    FREE(xyzs);

    SPRINT0x(2, "    meshT=");
    for (ipnt = 0; ipnt < newNpnt; ipnt++) {
        SPRINT1x(2, " %8.4f", newT[ipnt]);
    }
    SPRINT0(2, " ");

    /* put the t distribution on the Edge */
    status = EG_attributeAdd(eedge, "meshT", ATTRREAL, newNpnt, NULL, newT, NULL);
    CHECK_STATUS(EG_attributeAdd);

    /* return the final distribution */
    *npnt = newNpnt;

cleanup:
    FREE(newS);
    FREE(newT);

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   groupSpacing - set up spacing distribution for a group of Edges    *
 *                                                                      *
 ************************************************************************
 */

static int
groupSpacing(modl_T   *MODL,            /* (in)  pointer to MODL */
             int      meshGroup,        /* (in)  mesh group number (for printing) */
             int      ibody,            /* (in)  Body index */
             int      ngroup,           /* (in)  number of Edges in group */
             int      group[],          /* (in)  Edge indicies in the group */
             double   meshInfo[])       /* (in)  mesh information */
{
    int    status = EGADS_SUCCESS;

    int     igroup, jgroup, periodic, attrType, attrLen, iedge, i, ibest, npnt, ipnt, iswap, itgt;
    int     *ibeg=NULL, *iend=NULL;
    CINT    *tempIlist;
    double  trange[2], alen, xyz[6], dbest, dtest, dt, frac;
    double  xyzmin, xyzbeg, xyzend;
    double  *length=NULL, *meshS=NULL, *newT=NULL, *flipT=NULL, *newS=NULL, *xyzs=NULL;
    CDOUBLE *tempRlist;
    char    message[100];
    CCHAR   *tempClist;
    ego     enode[2], ecurve, eedge;
    body_T  myBody;

    ROUTINE(groupSpacing);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    printf("enter groupSpacing(ibody=%d, ngroup=%d, group=", ibody, ngroup);
    for (igroup = 0; igroup < ngroup; igroup++) {
        printf(" %3d", group[igroup]);
    }
    printf(")\n");
#endif

    myBody = MODL->body[ibody];

    /* create (over-sized) arrays */
    MALLOC(length, double, ngroup);
    MALLOC(ibeg,   int,    ngroup);
    MALLOC(iend,   int,    ngroup);

    MALLOC(meshS, double,   MAX_NPNT);
    MALLOC(newT,  double,   MAX_NPNT);
    MALLOC(flipT, double,   MAX_NPNT);
    MALLOC(newS,  double,   MAX_NPNT);
    MALLOC(xyzs,  double, 3*MAX_NPNT);

    /* arrange the Edges head to tail in the specified direction */
#ifdef DEBUG
    printf("before\n");
    for (igroup = 0; igroup < ngroup; igroup++) {
        printf("group[%3d]=%5d\n", igroup, group[igroup]);
    }
#endif

    xyzmin = HUGEQ;
    itgt   = 0;
    for (igroup = 0; igroup < ngroup; igroup++) {
        iedge = abs(group[igroup]);

        if        (NINT(meshInfo[5]) == 1) {
            xyzbeg = fabs(myBody.node[myBody.edge[iedge].ibeg].x);
            xyzend = fabs(myBody.node[myBody.edge[iedge].iend].x);
        } else if (NINT(meshInfo[5]) == 2) {
            xyzbeg = fabs(myBody.node[myBody.edge[iedge].ibeg].y);
            xyzend = fabs(myBody.node[myBody.edge[iedge].iend].y);
        } else if (NINT(meshInfo[5]) == 3) {
            xyzbeg = fabs(myBody.node[myBody.edge[iedge].ibeg].z);
            xyzend = fabs(myBody.node[myBody.edge[iedge].iend].z);
        } else {
            printf("meshInfo[6] is not 1, 2, or 3\n");
            status = -987;
            goto cleanup;
        }

        if (xyzbeg < xyzmin) {
            xyzmin = xyzbeg;
            itgt   = +(igroup + 1);
        }

        if (xyzend < xyzmin) {
            xyzmin = xyzend;
            itgt   = -(igroup + 1);
        }
    }

    if (itgt > 0) {
        iswap          = group[0];
        group[0]       = group[itgt-1];
        group[itgt-1]  = iswap;
    } else {
        iswap          = group[0];
        group[0]       = group[-1-itgt];
        group[-1-itgt] = iswap;

        group[0] *= -1;
    }

#ifdef DEBUG
    printf("during\n");
    for (igroup = 0; igroup < ngroup; igroup++) {
        printf("group[%3d]=%5d\n", igroup, group[igroup]);
    }
#endif

    for (igroup = 0; igroup < ngroup; igroup++) {
        if (group[igroup] > 0) {
            itgt = myBody.edge[+group[igroup]].iend;
        } else {
            itgt = myBody.edge[-group[igroup]].ibeg;
        }

        for (jgroup = igroup+1; jgroup < ngroup; jgroup++) {
            iedge = abs(group[jgroup]);

            if (myBody.edge[iedge].ibeg == itgt) {
                iswap           = group[igroup+1];
                group[igroup+1] = group[jgroup];
                group[jgroup  ] = iswap;

                break;
            } else if (myBody.edge[iedge].iend == itgt) {
                iswap           = group[igroup+1];
                group[igroup+1] = group[jgroup];
                group[jgroup  ] = iswap;

                group[igroup+1] *= -1;

                break;
            }
        }
    }

#ifdef DEBUG
    printf("after\n");
    for (igroup = 0; igroup < ngroup; igroup++) {
        printf("group[%3d]=%5d\n", igroup, group[igroup]);
    }
#endif

    /* find the total length of the Edges in the group */
    for (igroup = 0; igroup < ngroup; igroup++) {
        iedge = abs(group[igroup]);

        status = EG_getRange(myBody.edge[iedge].eedge, trange, &periodic);
        CHECK_STATUS(EG_getRange);

        status = EG_arcLength(myBody.edge[iedge].eedge, trange[0], trange[1], &alen);
        CHECK_STATUS(EG_arcLength);

        if (igroup == 0) {
            length[igroup] = alen;
        } else {
            length[igroup] = length[igroup-1] + alen;
        }
    }

    /* make a phony Edge on which the meshS attribute will be generated */
    xyz[0] = 0;
    xyz[1] = 0;
    xyz[2] = 0;

    status = EG_makeTopology(MODL->context, NULL, NODE, 0, &xyz[0], 0, NULL, NULL, &enode[0]);
    CHECK_STATUS(EG_makeTopology);

    xyz[3] = length[ngroup-1];
    xyz[4] = 0;
    xyz[5] = 0;

    status = EG_makeTopology(MODL->context, NULL, NODE, 0, &xyz[3], 0, NULL, NULL, &enode[1]);
    CHECK_STATUS(EG_makeTopology);

    status = EG_makeGeometry(MODL->context, CURVE, LINE, NULL, NULL, xyz, &ecurve);
    CHECK_STATUS(EG_makeGeometry);

    trange[0] = 0;
    trange[1] = length[ngroup-1];

    status = EG_makeTopology(MODL->context, ecurve, EDGE, TWONODE, trange, 2, enode, NULL, &eedge);
    CHECK_STATUS(EG_makeTopology);

    /* put the meshInfo on this new phony Edge */
    status = EG_attributeAdd(eedge, "__meshInfo__", ATTRREAL, 6, NULL, meshInfo, NULL);
    CHECK_STATUS(EG_attributeAdd);

    /* apply the edgeSpacing to these Edges */
    npnt = 0;
    status = edgeSpacing(eedge, &npnt, message);
    CHECK_STATUS(edgeSpacing);

    status = EG_attributeRet(eedge, "meshT",
                             &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
    CHECK_STATUS(EG_attributeRet);

    for (i = 0; i < attrLen; i++) {
        meshS[i] = tempRlist[i];
    }

#ifdef DEBUG
    for (i = 0; i < attrLen; i++) {
        printf("meshS[%3d]=%10.5f\n", i, meshS[i]);
    }
#endif

    /* adjust the spacings so that there is a point at each of the Nodes */
    ibeg[0] = 0;

    for (igroup = 0; igroup < ngroup-1; igroup++) {

        /* find closest meshS */
        ibest = ibeg[0];
        dbest = fabs(length[igroup]);

        for (i = ibeg[igroup]+1; i < attrLen; i++) {
            dtest = fabs(length[igroup] - meshS[i]);
            if (dtest < dbest) {
                dbest = dtest;
                ibest = i;
            }
        }

        iend[igroup] = ibest;

        dt = length[igroup] - meshS[iend[igroup]];
        for (i = ibeg[igroup]+1; i < iend[igroup]; i++) {
            frac = (double)(i-ibeg[igroup]) / (double)(iend[igroup]-ibeg[igroup]);

            meshS[i] += frac * dt;
        }
        meshS[iend[igroup]] = length[igroup];
        for (i = iend[igroup]+1; i < attrLen; i++) {
            frac = (double)(i-iend[igroup]) / (double)(attrLen-1-iend[igroup]);

            meshS[i] += (1-frac) * dt;
        }
        ibeg[igroup+1] = iend[igroup];
    }

    iend[ngroup-1] = attrLen - 1;

#ifdef DEBUG
    for (igroup = 0; igroup < ngroup; igroup++) {
        printf("igroup=%2d, length=%10.5f, ibeg=%5d, iend=%5d\n",
               igroup, length[igroup], ibeg[igroup], iend[igroup]);
    }
    for (i = 0; i < attrLen; i++) {
        printf("meshS[%3d]=%10.5f\n", i, meshS[i]);
    }
#endif

    /* apply the meshS (which was generated on the whole group) into a
       meshT distribution for each of the Edges in the group */
    for (igroup = 0; igroup < ngroup; igroup++) {
        iedge = abs(group[igroup]);

#ifdef DEBUG
        printf("igroup=%d\n", igroup);
#endif

        status = EG_getRange(myBody.edge[iedge].eedge, trange, &periodic);
        CHECK_STATUS(EG_getRange);

        npnt = iend[igroup] - ibeg[igroup] + 1;
        for (ipnt = 0; ipnt < npnt; ipnt++) {
            newS[ipnt] = (meshS[ipnt+ibeg[igroup]] - meshS[ibeg[igroup]])
                       / (meshS[     iend[igroup]] - meshS[ibeg[igroup]]);
            newT[ipnt] = trange[0] + (trange[1] - trange[0]) * ipnt / (npnt-1);
        }

#ifdef DEBUG
        for (ipnt = 0; ipnt < npnt; ipnt++) {
            printf("ipnt=%3d, newS=%10.5f newT=%10.5f\n", ipnt, newS[ipnt], newT[ipnt]);
        }
#endif

        status = EG_relPosTs(ecurve, npnt, &newS[1], newT, xyzs);
        CHECK_STATUS(EG_relPosTs);

#ifdef DEBUG
        for (ipnt = 0; ipnt < npnt; ipnt++) {
            printf("ipnt=%3d, newS=%10.5f newT=%10.5f\n", ipnt, newS[ipnt], newT[ipnt]);
        }
#endif

        /* put the t distribution on the Edge */
        if (group[igroup] > 0) {
            status = EG_attributeAdd(myBody.edge[iedge].eedge, "meshT",
                                     ATTRREAL, npnt, NULL, newT, NULL);
            CHECK_STATUS(EG_attributeAdd);

            SPRINT3(1, "        adding __meshInfo__ to jbody=%3d, jedge=%3d from meshGroup %3d",
                    ibody, iedge, meshGroup);
        } else {
            for (ipnt = 0; ipnt < npnt; ipnt++) {
                flipT[ipnt] = 1 - newT[npnt-ipnt-1];
            }

            status = EG_attributeAdd(myBody.edge[iedge].eedge, "meshT",
                                     ATTRREAL, npnt, NULL, flipT, NULL);
            CHECK_STATUS(EG_attributeAdd);

            SPRINT3(1, "        adding __meshInfo__ to jbody=%3d, jedge=%3d from meshGroup %3d (flipped)",
                    ibody, iedge, meshGroup);
        }
    }

    /* clean up the temporary egos that were built */
    status = EG_deleteObject(eedge);
    CHECK_STATUS(EG_deleteObject);

    status = EG_deleteObject(enode[0]);
    CHECK_STATUS(EG_deleteObject);

    status = EG_deleteObject(enode[1]);
    CHECK_STATUS(EG_deleteObject);

    status = EG_deleteObject(ecurve);
    CHECK_STATUS(EG_deleteObject);

cleanup:
    FREE(length);
    FREE(ibeg  );
    FREE(iend  );

    FREE(meshS);
    FREE(newT );
    FREE(flipT);
    FREE(newS );
    FREE(xyzs );

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   intersectSegs - check if two line segments intersect               *
 *                                                                      *
 ************************************************************************
 */
static int
intersectSegs(CDOUBLE uva[],            /* (in)  UV at beg of first  segment */
              CDOUBLE uvb[],            /* (in)  UV at end of first  segment */
              CDOUBLE uvc[],            /* (in)  UV at beg of second segment */
              CDOUBLE uvd[],            /* (in)  UV at end of second segment */
              double  *fracab,          /* (out) fractional distance from a to b */
              double  *fraccd)          /* (out) fractional distance from c to d */
{
    int    status = EGADS_SUCCESS;      /* (out) return status */
                                        /* =0  intersects */
                                        /* =1  does not intersect */

    double   det;

    ROUTINE(intersectSegs);

    /* --------------------------------------------------------------- */

    *fracab = -1;
    *fraccd = -1;

    det = (uvb[0]-uva[0]) * (uvc[1]-uvd[1]) - (uvc[0]-uvd[0]) * (uvb[1]-uva[1]);
    if (fabs(det) < EPS12) {
        status = 1;
        goto cleanup;
    }

    *fracab = ((uvc[0]-uva[0]) * (uvc[1]-uvd[1]) - (uvc[0]-uvd[0]) * (uvc[1]-uva[1])) / det;
    *fraccd = ((uvb[0]-uva[0]) * (uvc[1]-uva[1]) - (uvc[0]-uva[0]) * (uvb[1]-uva[1])) / det;

    if (*fracab < 0-EPS12 || *fracab > 1+EPS12 ||
        *fraccd < 0-EPS12 || *fraccd > 1+EPS12   ) {
        status = 1;
    }

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   matchEdges - check if two Edges "match"                            *
 *                                                                      *
 ************************************************************************
 */
static int
matchEdges(modl_T   *MODL,              /* (in)  pointer to MODL */
           int      ibody,              /* (in)  first  Body index */
           int      iedge,              /* (in)  first  Edge index */
           int      jbody,              /* (in)  second Body index */
           int      jedge,              /* (in)  second Edge index */
           double   toler)              /* (in)  tolerance */
{
    int    status = EGADS_SUCCESS;      /* (out) return status */
                                        /* =0  no match */
                                        /* =1     match */
                                        /* =2  reversed match */
                                        /* =3  degenerate Edge */
                                        /* =4  eedge matches subset of jedge */
                                        /* =5  eedge matches reversed subset of jedge */

    int    periodic, ipnt, npnt=17;
    int    oclass, mtype, *senses, nchild;
    double datai[18], dataj[18], ti[2], tj[2], trangei[4], trangej[4];
    double datai0[18], datai1[18], dataj0[18], dataj1[18], dotprod;
    ego    eedge, eref, *echilds;

    ROUTINE(matchEdges);

    /* --------------------------------------------------------------- */

    /* check inputs */
    if (ibody < 1 || ibody > MODL->nbody) {
        status = OCSM_BODY_NOT_FOUND;
        goto cleanup;

    } else if (iedge < 1 || iedge > MODL->body[ibody].nedge) {
        status = OCSM_EDGE_NOT_FOUND;
        goto cleanup;

    } else if (MODL->body[ibody].edge[iedge].itype == DEGENERATE) {
        status = 3;
        goto cleanup;

    } else if (jbody < 1 || jbody > MODL->nbody) {
        status = OCSM_BODY_NOT_FOUND;
        goto cleanup;

    } else if (jedge < 1 || jedge > MODL->body[jbody].nedge) {
        status = OCSM_EDGE_NOT_FOUND;
        goto cleanup;

    } else if (MODL->body[jbody].edge[jedge].itype == DEGENERATE) {
        status = 3;
        goto cleanup;
    }

    eedge = MODL->body[ibody].edge[iedge].eedge;

    status = EG_getTopology(eedge, &eref, &oclass, &mtype,
                            datai, &nchild, &echilds, &senses);
    CHECK_STATUS(EG_getTopology);

    if (mtype == DEGENERATE) {
        status = 3;
        goto cleanup;
    }

    /* default tolerance */
    if (toler <= 0) toler = EPS06;

    /* get ranges for the two Edges */
    status = EG_getRange(eedge, trangei, &periodic);
    CHECK_STATUS(EG_getRange);

    status = EG_getRange(MODL->body[jbody].edge[jedge].eedge, trangej, &periodic);
    CHECK_STATUS(EG_getRange);

    /* make sure that npnt points along eedge are within tolerance of jedge */
    for (ipnt = 0; ipnt <= npnt; ipnt++) {
        ti[0] = trangei[0] + (trangei[1] - trangei[0]) * (double)(ipnt) / (double)(npnt);

        status = EG_evaluate(eedge, ti, datai);
        CHECK_STATUS(EG_evaluate);

        status = EG_invEvaluate(MODL->body[jbody].edge[jedge].eedge, datai, tj, dataj);
        CHECK_STATUS(EG_invEvaluate);

        if (tj[0] < trangej[0]-EPS06 ||
            tj[0] > trangej[1]+EPS06   ) {
            status = 0;
            goto cleanup;
        } else if (fabs(datai[0]-dataj[0]) > toler ||
                   fabs(datai[1]-dataj[1]) > toler ||
                   fabs(datai[2]-dataj[2]) > toler   ) {
            status = 0;
            goto cleanup;
        }
    }

    /* since the Edges (at least partially) match, if the endpoints do
       not match, then we have a subset */
    status = EG_evaluate(                              eedge, &trangei[0], datai0);
    CHECK_STATUS(EG_evaluate);

    status = EG_evaluate(                              eedge, &trangei[1], datai1);
    CHECK_STATUS(EG_evaluate);

    status = EG_evaluate(MODL->body[jbody].edge[jedge].eedge, &trangej[0], dataj0);
    CHECK_STATUS(EG_evaluate);

    status = EG_evaluate(MODL->body[jbody].edge[jedge].eedge, &trangej[1], dataj1);
    CHECK_STATUS(EG_evaluate);

    /* check if Edges are in same direction */
    if        (fabs(datai0[0]-dataj0[0]) < toler &&
               fabs(datai0[1]-dataj0[1]) < toler &&
               fabs(datai0[2]-dataj0[2]) < toler &&
               fabs(datai1[0]-dataj1[0]) < toler &&
               fabs(datai1[1]-dataj1[1]) < toler &&
               fabs(datai1[2]-dataj1[2]) < toler   ) {
        status = +1;
        goto cleanup;

    /* check if Edges are in opposite direction */
    } else if (fabs(datai0[0]-dataj1[0]) < toler &&
               fabs(datai0[1]-dataj1[1]) < toler &&
               fabs(datai0[2]-dataj1[2]) < toler &&
               fabs(datai1[0]-dataj0[0]) < toler &&
               fabs(datai1[1]-dataj0[1]) < toler &&
               fabs(datai1[2]-dataj0[2]) < toler   ) {
        status = +2;
        goto cleanup;

    /* endpoints do not match, so eedge is a subset of jedge */
    } else {
        dotprod = (datai1[0]-datai0[0])*(dataj1[0]-dataj0[0])
                + (datai1[1]-datai0[1])*(dataj1[1]-dataj0[1])
                + (datai1[2]-datai0[2])*(dataj1[2]-dataj0[2]);
        if (dotprod > 0) {
            status = +4;
        } else {
            status = +5;
        }
        goto cleanup;
    }

cleanup:
    return status;
}
