/*
 *************************************************************************
 *                                                                       *
 * udpSplineDesPmtr -- udf for b-spline design parameter                 *
 *                                                                       *
 *            Written by Marshall Galbraith @ MIT                        *
 *            Patterned after .c code written by John Dannenhoffer       *
 *                                                @ Syracuse University  *
 *                                                                       *
 *************************************************************************
 */

/*
 * Copyright (C) 2011/2026 Marshall Galbraith @ MIT
 *
 * This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *     MA  02110-1301  USA
 */

extern "C" {
#define NUMUDPARGS 5
#include "udpUtilities.h"
}

#define PARAM_ARG(UDP   )                    (UDP)->arg[0]
#define PARAM_VAL(UDP, I) ((const double *) ((UDP)->arg[0].val))[I]
#define PARAM_DOT(UDP, I) ((const double *) ((UDP)->arg[0].dot))[I]
#define VALUE_ARG(UDP   )                    (UDP)->arg[1]
#define VALUE_VAL(UDP, I) ((const double *) ((UDP)->arg[1].val))[I]
#define VALUE_DOT(UDP, I) ((const double *) ((UDP)->arg[1].dot))[I]
#define CONT_ARG( UDP   )                    (UDP)->arg[2]
#define CONT_VAL( UDP, I) ((const int *)    ((UDP)->arg[2].val))[I]
#define TS_ARG(   UDP   )                    (UDP)->arg[3]
#define TS_VAL(   UDP, I) ((const double *) ((UDP)->arg[3].val))[I]
#define TS_DOT(   UDP, I) ((const double *) ((UDP)->arg[3].dot))[I]
#define INTRP_ARG(UDP   )                    (UDP)->arg[4]
#define INTRP_VAL(UDP, I) ((double *)       ((UDP)->arg[4].val))[I]
#define INTRP_DOT(UDP, I) ((double *)       ((UDP)->arg[4].dot))[I]

#define PARAM_SURREAL(UDP, I) SurrealS<1>( PARAM_VAL(UDP, I), PARAM_DOT(UDP, I))
#define VALUE_SURREAL(UDP, I) SurrealS<1>( VALUE_VAL(UDP, I), VALUE_DOT(UDP, I))
#define TS_SURREAL(   UDP, I) SurrealS<1>( TS_VAL(   UDP, I), TS_DOT(   UDP, I))
#define INTRP_SURREAL(UDP, I) SurrealS<1>( INTRP_VAL(UDP, I), INTRP_DOT(UDP, I))

/* data about possible arguments */
static const char  *argNames[NUMUDPARGS] = {"param",     "value",     "cont",  "ts",        "interp"    };
static int          argTypes[NUMUDPARGS] = {ATTRREALSEN, ATTRREALSEN, ATTRINT, ATTRREALSEN, -ATTRREALSEN};
static int          argIdefs[NUMUDPARGS] = {0 ,          0 ,          2 ,      0,           0,          };
static double       argDdefs[NUMUDPARGS] = {0.,          0.,          0.,      0.,          0.,         };


#include "egadsSplineFit.h"
#include "egads_dot.h"

template<class T> T    PARAM(udp_T *udp, int i);
template<> double      PARAM< double      >(udp_T *udp, int i ) { return PARAM_VAL(udp,i); }
template<> SurrealS<1> PARAM< SurrealS<1> >(udp_T *udp, int i ) { return PARAM_SURREAL(udp,i); }

template<class T> T    VALUE(udp_T *udp, int i);
template<> double      VALUE< double      >(udp_T *udp, int i ) { return VALUE_VAL(udp,i); }
template<> SurrealS<1> VALUE< SurrealS<1> >(udp_T *udp, int i ) { return VALUE_SURREAL(udp,i); }

template<class T> T    TS(udp_T *udp, int i);
template<> double      TS< double      >( udp_T *udp, int i       ) { return TS_VAL(udp, i); }
template<> SurrealS<1> TS< SurrealS<1> >( udp_T *udp, int i       ) { return TS_SURREAL(udp, i); }

template<class T> T    INTRP(udp_T *udp, int i );
template<> double      INTRP< double      >( udp_T *udp, int i ) { return INTRP_VAL(udp, i); }
template<> SurrealS<1> INTRP< SurrealS<1> >( udp_T *udp, int i ) { return INTRP_SURREAL(udp, i); }

template class SurrealS<1>;
typedef SurrealS<1> SurrealS1;

#include <vector>
#include <float.h>

#define LENMSG 512

extern "C" {
/* get utility routines: udpErrorStr, udpInitialize, udpReset, udpSet,
                         udpGet, udpVel, udpClean, udpMesh */
#include "udpUtilities.c"
}

// Spline fitting tolerance
#define TOL 1e-6
#define NEWTONMAX 21
#define NITER    10000
#define RELAX    0.15

template<class T>
static int
splineMonotone1dFit(int end1x, int endnx,
                    const std::vector<int>& cn,
                    const std::vector<T>& xyz,
                    double tol, int *header, std::vector<T>& rdata);

/*
 ************************************************************************
 *                                                                      *
 *   udpExecute - execute the primitive                                 *
 *                                                                      *
 ************************************************************************
 */
extern "C"
int
udpExecute(ego context,                 /* (in)  context */
           ego  *ebody,                 /* (out) Body pointer */
           int  *nMesh,                 /* (out) number of associated meshes */
           char *string[])              /* (out) error message */
{
  int    status = EGADS_SUCCESS;
  ego    ecurve, spline_body = NULL;
  double data[18], tdata[2], t, x, dp;
  int    header[4], sense[1], end1c, endnc;
  ego    eedge = NULL, enodes[2], eloop;
  int    i, j;
//  udpDotCache_T *cache;
  char          *message=NULL;
  void    *realloc_temp=NULL;
  udp_T   *udps = *Udps;
  udp_T   *udp = &udps[0];

  std::vector<int> cn;
  std::vector<double> xyz;
  std::vector<double> rdata;

  ROUTINE(udpExecute);

  /* --------------------------------------------------------------- */

#ifdef DEBUG
  printf("udpSplineDesPmtr.udpExecute\n");
#endif

  /* default return values */
  *ebody  = NULL;
  *nMesh  = 0;
  *string = NULL;

  MALLOC(message, char, LENMSG);
  message[0] = '\0';

  if (PARAM_ARG(udp).nrow != 1 && PARAM_ARG(udp).ncol != 1) {
    snprintf(message, LENMSG, "Expecting 'param' vector. There are %d rows and %d columns\n", PARAM_ARG(udp).nrow, PARAM_ARG(udp).ncol);
    status = EGADS_RANGERR;
    goto cleanup;
  }
  if (PARAM_ARG(udp).size < 2) {
    snprintf(message, LENMSG, "Expecting at least length 2 'param'. There are %d rows and %d columns\n", PARAM_ARG(udp).nrow, PARAM_ARG(udp).ncol);
    status = EGADS_RANGERR;
    goto cleanup;
  }
  if (VALUE_ARG(udp).nrow != 1 && VALUE_ARG(udp).ncol != 1) {
    snprintf(message, LENMSG, "Expecting 'value' vector. There are %d rows and %d columns\n", VALUE_ARG(udp).nrow, VALUE_ARG(udp).ncol);
    status = EGADS_RANGERR;
    goto cleanup;
  }
  if (VALUE_ARG(udp).size < 2) {
    snprintf(message, LENMSG, "Expecting at least length 2 'value'. There are %d rows and %d columns\n", VALUE_ARG(udp).nrow, VALUE_ARG(udp).ncol);
    status = EGADS_RANGERR;
    goto cleanup;
  }
  if (PARAM_ARG(udp).size != VALUE_ARG(udp).size) {
    snprintf(message, LENMSG, "Expecting 'param' (%d) and 'value' (%d) to be equal\n", PARAM_ARG(udp).size, VALUE_ARG(udp).size);
    status = EGADS_RANGERR;
    goto cleanup;
  }
  if (CONT_ARG(udp).size > 1) {
    if (CONT_ARG(udp).nrow != 1 && CONT_ARG(udp).ncol != 1) {
      snprintf(message, LENMSG, "Expecting 'cont' vector. There are %d rows and %d columns\n", CONT_ARG(udp).nrow, CONT_ARG(udp).ncol);
      status = EGADS_RANGERR;
      goto cleanup;
    }
    if (PARAM_ARG(udp).size != CONT_ARG(udp).size) {
      snprintf(message, LENMSG, "Expecting 'param' (%d) and 'cont' (%d) to be equal\n", PARAM_ARG(udp).size, CONT_ARG(udp).size);
      status = EGADS_RANGERR;
      goto cleanup;
    }
    for (i = 0; i < CONT_ARG(udp).size; i++) {
      if (CONT_VAL(udp, i) < 0 || CONT_VAL(udp, i) > 2) {
        snprintf(message, LENMSG, "Expecting 0 <= 'cont[%d]' = %d <= 2\n", i+1, CONT_VAL(udp, i));
        status = EGADS_RANGERR;
        goto cleanup;
      }
    }
  }
  if (TS_ARG(udp).size < 1) {
    snprintf(message, LENMSG, "Expecting at least 1 'ts'. There are %d \n", TS_ARG(udp).size);
    status = EGADS_RANGERR;
    goto cleanup;
  }

  /* Cache copy of arguments for future use.
   * This also increments numUdp and copies udps[numUdp] to udps[numUdp].
   * Caching should only be performed after checking for valid inputs.
   */
  status = cacheUdp(NULL);
  CHECK_STATUS(cacheUdp);
  udp = &udps[numUdp];

  /* make enough room for the interpolated output */
  INTRP_ARG(udp).size = TS_ARG(udp).size;
  INTRP_ARG(udp).nrow = TS_ARG(udp).nrow;
  INTRP_ARG(udp).ncol = TS_ARG(udp).ncol;

  RALLOC(INTRP_ARG(udp).val, double, INTRP_ARG(udp).size);
  RALLOC(INTRP_ARG(udp).dot, double, INTRP_ARG(udp).size);

  cn.resize(PARAM_ARG(udp).size, 0);
  if (CONT_ARG(udp).size > 1) {
    for (i = 0; i < CONT_ARG(udp).size; i++) {
      cn[i] = 2-CONT_VAL(udp,i);
    }
  }

  end1c = cn.front();
  endnc = cn.back();

  /* use finite difference for linear or natural with only 3 points */
  if (PARAM_ARG(udp).size == 2) {
    end1c = endnc = 1;
  } else if (PARAM_ARG(udp).size == 3) {
    if (end1c == 2) end1c = 1;
    if (endnc == 2) endnc = 1;
  } else {
    if (CONT_VAL(udp,1)                    <= 1 && end1c == 2) end1c = 1;
    if (CONT_VAL(udp,CONT_ARG(udp).size-2) <= 1 && endnc == 2) endnc = 1;
  }

  cn.front() = 0;
  cn.back() = 0;

  xyz.resize(3*PARAM_ARG(udp).size);
  for (i = 0; i < PARAM_ARG(udp).size; i++) {
    xyz[3*i+0] = PARAM_VAL(udp, i);
    xyz[3*i+1] = VALUE_VAL(udp, i);
    xyz[3*i+2] = 0;
  }

  status = splineMonotone1dFit( end1c, endnc, cn, xyz, TOL, header, rdata );
  CHECK_STATUS(EG_spline1dFit2);

  status = EG_makeGeometry(context, CURVE, BSPLINE, NULL, header, rdata.data(), &ecurve);
  CHECK_STATUS(EG_makeGeometry);

  /* create Topology */

  /* first Node */
  i = 0;
  data[0] = PARAM_VAL(udp, i);
  data[1] = VALUE_VAL(udp, i);
  data[2] = 0;
  status = EG_makeTopology(context, NULL, NODE, 0,
                           data, 0, NULL, NULL, &(enodes[0]));
  CHECK_STATUS(EG_makeTopology);

  /* second Node */
  i = PARAM_ARG(udp).size-1;
  data[0] = PARAM_VAL(udp, i);
  data[1] = VALUE_VAL(udp, i);
  data[2] = 0;
  status = EG_makeTopology(context, NULL, NODE, 0,
                           data, 0, NULL, NULL, &(enodes[1]));
  CHECK_STATUS(EG_makeTopology);

  /* Edge */
  tdata[0] = rdata[0];
  tdata[1] = rdata[header[3]-1];
  status = EG_makeTopology(context, ecurve, EDGE, TWONODE,
                           tdata, 2, enodes, NULL, &(eedge));
  CHECK_STATUS(EG_makeTopology);

  /* create Loop of the Edge */
  sense[0] = SFORWARD;
  status = EG_makeTopology(context, NULL, LOOP, OPEN,
                           NULL, 1, &eedge, sense, &eloop);
  CHECK_STATUS(EG_makeTopology);

  /* create the WireBody */
  status = EG_makeTopology(context, NULL, BODY, WIREBODY,
                           NULL, 1, &eloop, NULL, &spline_body);
  CHECK_STATUS(EG_makeTopology);

  dp = PARAM_VAL(udp, PARAM_ARG(udp).size-1) - PARAM_VAL(udp, 0);

  /* Evaluate the spline */
  for (i = 0; i < TS_ARG(udp).size; i++) {
    x = TS_VAL(udp, i);
    t = (TS_VAL(udp, i)-PARAM_VAL(udp, 0))/dp;
    for (j = 0; j < NEWTONMAX; j++) {
      status = EG_evaluate(ecurve, &t, data);
      CHECK_STATUS(EG_evaluate);

      if (fabs(x - data[0]) < TOL) break;

      // Newton update the t-value until x and data[0] match
      double R   = x - data[0];
      double R_t =   - data[3];
      double dt  = R/R_t;
      t -= dt;
    }
    INTRP_VAL(udp, i) = data[1];
    //std::cout << "interp val: " << i << " : " << t << " " << data[0] << " : " << data[1] << std::endl;
  }

  /* cleanup in reverse order */
  status = EG_deleteObject(eloop);
  CHECK_STATUS(EG_deleteObject);
  status = EG_deleteObject(eedge);
  CHECK_STATUS(EG_deleteObject);
  status = EG_deleteObject(ecurve);
  CHECK_STATUS(EG_deleteObject);
  status = EG_deleteObject(enodes[1]);
  CHECK_STATUS(EG_deleteObject);
  status = EG_deleteObject(enodes[0]);
  CHECK_STATUS(EG_deleteObject);

  /* remember this model (body) */
  udp->ebody = *ebody = spline_body;

#ifdef DEBUG
  printf("udpExecute -> *ebody=%lx\n", (long)(*ebody));
#endif

cleanup:
  if (strlen(message) > 0) {
    *string = message;
    printf("%s\n", message);
  } else if (status != EGADS_SUCCESS) {
    printf("ERROR: status = %d \n", status);
    // freePrivateData(&ffdBox);
    FREE(message);
    *string = udpErrorStr(status);
  } else {
    FREE(message);
  }

  return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   udpSensitivity - return sensitivity derivatives for the "real" argument *
 *                                                                      *
 ************************************************************************
 */
extern "C"
int
udpSensitivity(ego    ebody,            /* (in)  Body pointer */
               int    npnt,             /* (in)  number of points */
               int    entType,          /* (in)  OCSM entity type */
               int    entIndex,         /* (in)  OCSM entity index (bias-1) */
               double uvs[],            /* (in)  parametric coordinates for evaluation */
               double vels[])           /* (out) velocities */
{
  int    status = EGADS_SUCCESS;
  int    iudp = 0, judp, stride, ipnt, header[4], *senses;
  int    nchild, nedge, nnode;
  int    oclass, mtype, end1c, endnc;
  double point[18], point_dot[18], data[4], x;
  ego    eent, eref, eloop, *eedges, ecurve, *enodes, *echildren;
  SurrealS<1> t, dp, pnt[3], tdata[2], sdata[18];
  udp_T   *udp = NULL;

  std::vector<SurrealS<1>> rdata;

  int    i, j;
  char   *message=NULL;

  ROUTINE(udpSensitivity);

  /* --------------------------------------------------------------- */

  /* check that ebody matches one of the ebodys */
  iudp = 0;
  for (judp = 1; judp <= numUdp; judp++) {
    if (ebody == udps[judp].ebody) {
      iudp = judp;
      break;
    }
  }
  if (iudp <= 0) {
    return EGADS_NOTMODEL;
  }

  udp = &udps[iudp];

#ifdef DEBUG
  printf("udpSplineDesPmtr.udpSensitivity iudp = %d\n", iudp);

  for (i = 0; i < PARAM_ARG(udp).nrow; i++) {
    printf("PARAM_VAL(%d) = %f %f\n", i, PARAM_VAL(udp,i) );
  }
  for (i = 0; i < VALUE_ARG(udp).nrow; i++) {
    printf("VALUE_VAL(%d) = %f %f\n", i, VALUE_VAL(udp,i) );
  }
  for (i = 0; i < CONT_ARG(udp).nrow; i++) {
    printf("CONT_VAL(%d) = %f %f\n", i, CONT_VAL(udp,i) );
  }
  for (i = 0; i < TS_ARG(udp).nrow; i++) {
      printf("TS_VAL(%d) = %f\n", i, TS_VAL(udp,i) );
  }
  for (i = 0; i < INTRP_ARG(udp).nrow; i++) {
      printf("INTRP_VAL(%d) = %f\n", i, INTRP_VAL(udp,i) );
  }
  printf("\n");
#endif

  MALLOC(message, char, LENMSG);
  message[0] = '\0';

  /* get the loop from the body */
  status = EG_getTopology(ebody, &eref, &oclass, &mtype, data, &nchild, &echildren,
                          &senses);
  CHECK_STATUS(EG_getTopology);
  eloop = echildren[0];

  /* get the edges from the loop */
  status = EG_getTopology(eloop, &eref, &oclass, &mtype, data, &nedge, &eedges,
                          &senses);
  CHECK_STATUS(EG_getTopology);

  /* get the nodes and the curve from the first edge */
  status = EG_getTopology(eedges[0], &ecurve, &oclass, &mtype, data, &nnode, &enodes,
                          &senses);
  CHECK_STATUS(EG_getTopology);


  /* build the sensitivity if needed */
  if (udp->ndotchg > 0 || EG_hasGeometry_dot(ebody) != EGADS_SUCCESS) {

    std::vector<int> cn(PARAM_ARG(udp).size, 0);
    if (CONT_ARG(udp).nrow > 1) {
      for (i = 0; i < CONT_ARG(udp).size; i++) {
        cn[i] = 2-CONT_VAL(udp,i);
      }
    }

    end1c = cn.front();
    endnc = cn.back();

    /* use finite difference for linear or natural with only 3 points */
    if (PARAM_ARG(udp).size == 2) {
      end1c = endnc = 1;
    } else if (PARAM_ARG(udp).size == 3) {
      if (end1c == 2) end1c = 1;
      if (endnc == 2) endnc = 1;
    } else {
      if (CONT_VAL(udp,1)                    <= 1 && end1c == 2) end1c = 1;
      if (CONT_VAL(udp,CONT_ARG(udp).size-2) <= 1 && endnc == 2) endnc = 1;
    }


    cn.front() = 0;
    cn.back() = 0;

    std::vector<SurrealS1> xyz(3*PARAM_ARG(udp).size);
    for (i = 0; i < PARAM_ARG(udp).size; i++) {
      xyz[3*i+0] = PARAM_SURREAL(udp, i);
      xyz[3*i+1] = VALUE_SURREAL(udp, i);
      xyz[3*i+2] = 0;
    }

    status = splineMonotone1dFit( end1c, endnc, cn, xyz, TOL, header, rdata );
    CHECK_STATUS(EG_spline1dFit2);

    /* set the sensitivity of the Curve */
    status = EG_setGeometry_dot(ecurve, CURVE, BSPLINE, header, rdata.data());
    CHECK_STATUS(EG_setGeometry_dot);

    /* set the sensitivity of the Node at trailing edge */
    i = 0;
    pnt[0] = PARAM_SURREAL(udp, i);
    pnt[1] = VALUE_SURREAL(udp, i);
    pnt[2] = 0;
    status = EG_setGeometry_dot(enodes[0], NODE, 0, NULL, pnt);
    CHECK_STATUS(EG_setGeometry_dot);

    /* set the sensitivity of the Node at leading edge */
    i = PARAM_ARG(udp).size-1;
    pnt[0] = PARAM_SURREAL(udp, i);
    pnt[1] = VALUE_SURREAL(udp, i);
    pnt[2] = 0;
    status = EG_setGeometry_dot(enodes[1], NODE, 0, NULL, pnt);
    CHECK_STATUS(EG_setGeometry_dot);

    /* set Edge t-range sensitivity */
    tdata[0] = rdata[0];
    tdata[1] = rdata[header[3]-1];
    status = EG_setRange_dot(eedges[0], EDGE, tdata);
    CHECK_STATUS(EG_setRange_dot);

    dp = PARAM_SURREAL(udp, PARAM_ARG(udp).size-1) - PARAM_SURREAL(udp, 0);

    /* update sensitivities for the interpolated data points*/
    for (i = 0; i < TS_ARG(udp).size; i++) {
      x = TS_VAL(udp, i);
      t = (TS_SURREAL(udp, i)-PARAM_SURREAL(udp, 0))/dp;
      for (j = 0; j < NEWTONMAX; j++) {
        status = EG_evaluate(ecurve, &t, sdata);
        CHECK_STATUS(EG_evaluate);

        if (fabs(x - sdata[0]) < TOL) break;

        // Newton update the t-value until x and data[0] match
        SurrealS<1> R   = x - sdata[0];
        SurrealS<1> R_t =   - sdata[3];
        SurrealS<1> dt  = R/R_t;
        t -= dt;
      }
      INTRP_SURREAL(udp, i) = sdata[1];
      //std::cout << "interp sens: " << i << " : " << t << " " << sdata[0] << " : " << sdata[1] << std::endl;
    }
  }


  /* find the ego entity */
  if (entType == OCSM_NODE) {
    status = EG_getBodyTopos(ebody, NULL, NODE, &nchild, &echildren);
    CHECK_STATUS(EG_getBodyTopos);

    stride = 0;
    eent = echildren[entIndex-1];

    FREE(echildren);
  } else if (entType == OCSM_EDGE) {
    status = EG_getBodyTopos(ebody, NULL, EDGE, &nchild, &echildren);
    CHECK_STATUS(EG_getBodyTopos);

    stride = 1;
    eent = echildren[entIndex-1];

    FREE(echildren);
  } else if (entType == OCSM_FACE) {
    status = EG_getBodyTopos(ebody, NULL, FACE, &nchild, &echildren);
    CHECK_STATUS(EG_getBodyTopos);

    stride = 2;
    eent = echildren[entIndex-1];

    FREE(echildren);
  } else {
    printf("udpSensitivity: bad entType=%d\n", entType);
    status = EGADS_GEOMERR;
    goto cleanup;
  }


  /* get the velocities from the entity */
  for (ipnt = 0; ipnt < npnt; ipnt++) {
    status = EG_evaluate_dot(eent, &(uvs[stride*ipnt]), NULL, point, point_dot);
    CHECK_STATUS(EG_evaluate_dot);

    /* return the point velocity */
    vels[3*ipnt  ] = point_dot[0];
    vels[3*ipnt+1] = point_dot[1];
    vels[3*ipnt+2] = point_dot[2];
  }


cleanup:
  if (strlen(message) > 0) {
   printf("%s\n", message);
  } else if (status != EGADS_SUCCESS) {
   printf("ERROR: status = %d \n", status);
  }

  FREE(message);

  return status;

}


/*
 *******************************************************************************
 *                                                                             *
 *   splineMonotone1dFit - create 1d monotonic cubic spline from input data    *
 *                                                                             *
 *   https://en.wikipedia.org/wiki/Monotone_cubic_interpolation                *
 *                                                                             *
 *******************************************************************************
 */

template<class T>
static int
splineMonotone1dFit(int end1x, int endnx,
                    const std::vector<int>& cn,
                    const std::vector<T>& xyz,
                    double tol, int *header, std::vector<T>& rdata)
{
  int status = EGADS_SUCCESS;
  int i, j, kk, iknot, icp, iter, end1c, endnc, imax;
  T   du, dx, dy, dz, dxyzmax, rj[3], u21, u20, data[9];
  T   *knots, *cps;

  end1c = end1x;
  endnc = endnx;
  imax = cn.size();
  if (imax < 2)                     return EGADS_DEGEN;
  if ((end1x < -1) || (end1x >  2)) return EGADS_RANGERR;
  if ((endnx < -1) || (endnx >  2)) return EGADS_RANGERR;
  if ((imax == 2) && (end1c == 2)) end1c = 1;
  if ((imax == 2) && (endnc == 2)) endnc = 1;

  /* indices associated with various arrays:
            xyz       knot       cp
                       0*
                       1*
                       2*
             0         3*         0
             .         4*         1       (for end condition)
             1         5          2
             2         6          3

                       :

           imax-2    imax+1     imax-1
             .       imax+2*    imax      (for end condition)
           imax-1    imax+3*    imax+1
                     imax+4*
                     imax+5*
                     imax+6*

      *note: there are 5 repeated knots at beginning and
                       5 repeated knots at end */

  std::vector<T> delx(imax-1, 0);
  std::vector<T> dely(imax-1, 0);

  std::vector<T> mdelx(imax, 0);
  std::vector<T> mdely(imax, 0);

  std::vector<T> u(imax, 0);

  // compute parameter space
  for (i = 0; i < imax; i++) {
    u[i] = xyz[3*i+0];
  }

  /* normalize */
  for (i = 0; i < imax; i++) u[i] /= u[imax-1];


  // compute difference between points
  for (i = 0; i < imax-1; i++) {
    delx[i] = (xyz[3*(i+1)+0]-xyz[3*i+0]);
    dely[i] = (xyz[3*(i+1)+1]-xyz[3*i+1]);
  }

  // compute central differences
  mdelx[0     ] = delx[0     ];
  mdely[0     ] = dely[0     ]/delx[0     ];
  mdelx[imax-1] = delx[imax-2];
  mdely[imax-1] = dely[imax-2]/delx[imax-2];
  for (i = 1; i < imax-1; i++) {
    mdelx[i] = delx[i]+delx[i-1];
    mdely[i] = (xyz[3*(i+1)+1]-xyz[3*(i-1)+1])/mdelx[i];
  }


  // limit the slopes
  T del1 = dely[0]/delx[0];
  for (i = 1; i < imax-1; i++) {

    T del2 = (xyz[3*(i+1)+1]-xyz[3*(i)+1])/delx[i];

    if (fabs(del1) >= DBL_EPSILON) {

      if (del1/fabs(del1) * del2 < 0.0)
        mdely[i] = 0.0;

      T a = mdely[i-1]/del1;
      T b = mdely[i  ]/del1;

      T r = a*a + b*b;

      if (r > 9.0) {
        T tau = 3.0/sqrt(r);
        mdely[i-1] *= tau;
        mdely[i  ] *= tau;
      }
    } else {
      mdely[i-1] = 0;
      mdely[i  ] = 0;
    }
    del1 = del2;
  }

  // adjust the last interval
  if (fabs(del1) >= DBL_EPSILON) {

    i = imax-1;
    T a = mdely[i-1]/del1;
    T b = mdely[i  ]/del1;

    T r = a*a + b*b;

    if (r > 9.0) {
      T tau = 3.0/sqrt(r);
      mdely[i-1] *= tau;
      mdely[i  ] *= tau;
    }
  } else {
    mdely[i-1] = 0;
    mdely[i  ] = 0;
  }

  // scale the dy by dx
  for (i = 0; i < imax; i++) {
    mdely[i] *= mdelx[i];
  }


  // add points for duplicity
  for (i = 0; i < (int)cn.size(); i++) {
    imax += cn[i];
  }

  icp   = imax + 2;
  iknot = imax + 6;

  rdata.resize(iknot+3*icp);

  knots =  rdata.data();
  cps   = &rdata[iknot];

  std::vector<T> cp(3*icp);

  /* create spline curve */
  header[0] = 0;
  header[1] = 3;
  header[2] = icp;
  header[3] = iknot;

  /* knots based only on x */
  kk          = 0;
  knots[kk++] = xyz[0];
  knots[kk++] = xyz[0];
  knots[kk++] = xyz[0];

  for (i = 0; i < (int)cn.size(); i++) {
    for (j = 0; j <= cn[i]; j++) {
      knots[kk++] = xyz[3*i];
    }
  }

  knots[kk] = knots[kk-1]; kk++;
  knots[kk] = knots[kk-1]; kk++;
  knots[kk] = knots[kk-1]; kk++;

  /* normalize */
  for (i = 0; i < kk; i++) knots[i] /= knots[kk-1];

  std::vector<T> xcp(3*imax);

  /* duplicate points */
  kk        = 0;
  xcp[kk] = xyz[0]; kk++;
  xcp[kk] = xyz[1]; kk++;
  xcp[kk] = xyz[2]; kk++;

  /* interior control points */
  for (i = 1; i < (int)cn.size()-1; i++) {
    for (j = 0; j <= cn[i]; j++) {
      xcp[kk] = xyz[3*i  ]; kk++;
      xcp[kk] = xyz[3*i+1]; kk++;
      xcp[kk] = xyz[3*i+2]; kk++;
    }
  }

  /* final control point */
  xcp[kk] = xyz[3*(cn.size()-1)  ]; kk++;
  xcp[kk] = xyz[3*(cn.size()-1)+1]; kk++;
  xcp[kk] = xyz[3*(cn.size()-1)+2]; kk++;


  std::vector<int> mdata(imax, 1);
  std::vector<T> mdx(imax, 0);
  std::vector<T> mdy(imax, 0);

  /* mark knots with multiplicity 2 or 3 */
  mdx[0] = mdelx[0];
  mdy[0] = mdely[0];

  kk = 1;
  for (i = 1; i < (int)cn.size()-1; i++) {
    if (cn[i] == 2) {
      mdata[kk++] = -3;
      mdata[kk++] =  1;
      mdata[kk++] = +3;
    } else if (cn[i] == 1) {
      mdx[kk] = mdelx[i];
      mdy[kk] = mdely[i];
      mdata[kk++] = -2;
      mdata[kk++] = +2;
    } else {
      mdata[kk++] =  1;
    }
  }
  mdx[kk] = mdelx.back();
  mdy[kk] = mdely.back();

  /* initial control point */
  kk        = 0;
  cps[kk] = cp[kk] = xyz[0]; kk++;
  cps[kk] = cp[kk] = xyz[1]; kk++;
  cps[kk] = cp[kk] = xyz[2]; kk++;

  /* initial interior control point (for slope) */
  cps[kk] = cp[kk] = (3 * xyz[0] + xyz[3]) / 4; kk++;
  cps[kk] = cp[kk] = (3 * xyz[1] + xyz[4]) / 4; kk++;
  cps[kk] = cp[kk] = (3 * xyz[2] + xyz[5]) / 4; kk++;

  /* interior control points */
  for (i = 1; i < (int)cn.size()-1; i++) {
    for (j = 0; j <= cn[i]; j++) {
      cps[kk] = cp[kk] = xyz[3*i  ]; kk++;
      cps[kk] = cp[kk] = xyz[3*i+1]; kk++;
      cps[kk] = cp[kk] = xyz[3*i+2]; kk++;
    }
  }

  /* penultimate interior control point (for slope) */
  cps[kk] = cp[kk] = (3 * xyz[3*(cn.size()-1)  ] + xyz[3*(cn.size()-2)  ]) / 4; kk++;
  cps[kk] = cp[kk] = (3 * xyz[3*(cn.size()-1)+1] + xyz[3*(cn.size()-2)+1]) / 4; kk++;
  cps[kk] = cp[kk] = (3 * xyz[3*(cn.size()-1)+2] + xyz[3*(cn.size()-2)+2]) / 4; kk++;

  /* final control point */
  cps[kk] = cp[kk] = xyz[3*(cn.size()-1)  ]; kk++;
  cps[kk] = cp[kk] = xyz[3*(cn.size()-1)+1]; kk++;
  cps[kk] = cp[kk] = xyz[3*(cn.size()-1)+2]; kk++;


  /* iterate to have knot evaluations match data points */
  for (iter = 0; iter < NITER; iter++) {
      dxyzmax = 0.0;

      /* condition at beginning */
      EG_spline1dDeriv(header, rdata.data(), 2, knots[3], data);
      du = knots[4] - knots[3];
      if (end1c == 0) {
          /* natural end */
          dx = du * du * data[6];
          dy = du * du * data[7];
          dz = du * du * data[8];
      } else if (end1c == 1) {
          /* FD slope */
          dx = mdx[0] - du * data[3];
          dy = mdy[0] - du * data[4];
          dz = 0;
      } else {
          /* quadratic fit */
          u20    = knots[5] - knots[3];
          u21    = knots[5] - knots[4];
          rj[0]  = xcp[3]*u20*u20 - xcp[0]*u21*u21 - xcp[6]*du*du;
          rj[1]  = xcp[4]*u20*u20 - xcp[1]*u21*u21 - xcp[7]*du*du;
          rj[2]  = xcp[5]*u20*u20 - xcp[2]*u21*u21 - xcp[8]*du*du;
          rj[0] /= 2.0*u21*du;
          rj[1] /= 2.0*u21*du;
          rj[2] /= 2.0*u21*du;
          dx     = rj[0] - xcp[0] - 0.5 * u20 * data[3];
          dy     = rj[1] - xcp[1] - 0.5 * u20 * data[4];
          dz     = rj[2] - xcp[2] - 0.5 * u20 * data[5];
      }
      if (fabs(dx) > dxyzmax) dxyzmax = fabs(dx);
      if (fabs(dy) > dxyzmax) dxyzmax = fabs(dy);
      if (fabs(dz) > dxyzmax) dxyzmax = fabs(dz);
      cp[3] = cps[3] + RELAX * dx;
      cp[4] = cps[4] + RELAX * dy;
      cp[5] = cps[5] + RELAX * dz;

      /* match interior spline points */
      for (i = 1; i < imax-1; i++) {

          if (mdata[i] == +2) {
              /* multiplicity 2 with flat spot on right
                 note: i is the second repeated data point */
              EG_spline1dDeriv(header, rdata.data(), 1, knots[i+2], data);
              dx = xcp[3*i  ] - data[0];
              dy = xcp[3*i+1] - data[1];
              dz = xcp[3*i+2] - data[2];

              if (fabs(dx) > dxyzmax) dxyzmax = fabs(dx);
              if (fabs(dy) > dxyzmax) dxyzmax = fabs(dy);
              if (fabs(dz) > dxyzmax) dxyzmax = fabs(dz);

              cp[3*(i-1)+3] += RELAX * dx;
              cp[3*(i-1)+4] += RELAX * dy;
              cp[3*(i-1)+5] += RELAX * dz;

              cp[3*(i  )+3] += RELAX * dx;
              cp[3*(i  )+4] += RELAX * dy;
              cp[3*(i  )+5] += RELAX * dz;
              continue;
          }

          if (mdata[i] == -2) {
              /* multiplicity 2 with flat spot on left
                 note: i is the first repeated data point */
              EG_spline1dDeriv(header, rdata.data(), 1, knots[i+4], data);
              du =  knots[i+5] - knots[i+2];
              //du = 1;
              dx = -mdx[i] + du * data[3];
              dy = -mdy[i] + du * data[4];
              dz = 0;

              if (fabs(dx) > dxyzmax) dxyzmax = fabs(dx);
              if (fabs(dy) > dxyzmax) dxyzmax = fabs(dy);
              if (fabs(dz) > dxyzmax) dxyzmax = fabs(dz);

              cp[3*(i  )+3] += +RELAX * dx;
              cp[3*(i  )+4] += +RELAX * dy;
              cp[3*(i  )+5] += +RELAX * dz;

              cp[3*(i+1)+3] += -RELAX * dx;
              cp[3*(i+1)+4] += -RELAX * dy;
              cp[3*(i+1)+5] += -RELAX * dz;
              continue;
          }

          EG_spline1dEval(header, rdata.data(), knots[i+3], data);
          dx = xcp[3*i  ] - data[0];
          dy = xcp[3*i+1] - data[1];
          dz = xcp[3*i+2] - data[2];
          if (fabs(dx) > dxyzmax) dxyzmax = fabs(dx);
          if (fabs(dy) > dxyzmax) dxyzmax = fabs(dy);
          if (fabs(dz) > dxyzmax) dxyzmax = fabs(dz);
          cp[3*i+3] = cps[3*i+3] + dx;
          cp[3*i+4] = cps[3*i+4] + dy;
          cp[3*i+5] = cps[3*i+5] + dz;
      }

      /* condition at end */
      EG_spline1dDeriv(header, rdata.data(), 2, knots[imax+2], data);
      du = knots[imax+2] - knots[imax+1];
      if (endnc == 0) {
        /* natural end */
        dx = du * du * data[6];
        dy = du * du * data[7];
        dz = du * du * data[8];
      } else if (endnc == 1) {
        /* FD slope */
        dx = -mdx[imax-1] + du * data[3];
        dy = -mdy[imax-1] + du * data[4];
        dz = 0;
      } else {
        /* quadratic fit */
        u20    = knots[imax+2] - knots[imax];
        u21    = knots[imax+1] - knots[imax];
        rj[0]  = xcp[3*(imax-2)  ]*u20*u20 - xcp[3*(imax-1)  ]*u21*u21 -
                 xcp[3*(imax-3)  ]*du*du;
        rj[1]  = xcp[3*(imax-2)+1]*u20*u20 - xcp[3*(imax-1)+1]*u21*u21 -
                 xcp[3*(imax-3)+1]*du*du;
        rj[2]  = xcp[3*(imax-2)+2]*u20*u20 - xcp[3*(imax-1)+2]*u21*u21 -
                 xcp[3*(imax-3)+2]*du*du;
        rj[0] /= 2.0*u21*du;
        rj[1] /= 2.0*u21*du;
        rj[2] /= 2.0*u21*du;
        dx     = rj[0] - xcp[3*(imax-1)  ] + 0.5 * u20 * data[3];
        dy     = rj[1] - xcp[3*(imax-1)+1] + 0.5 * u20 * data[4];
        dz     = rj[2] - xcp[3*(imax-1)+2] + 0.5 * u20 * data[5];
      }
      if (fabs(dx) > dxyzmax) dxyzmax = fabs(dx);
      if (fabs(dy) > dxyzmax) dxyzmax = fabs(dy);
      if (fabs(dz) > dxyzmax) dxyzmax = fabs(dz);
      cp[3*imax  ] = cps[3*imax  ] + RELAX * dx;
      cp[3*imax+1] = cps[3*imax+1] + RELAX * dy;
      cp[3*imax+2] = cps[3*imax+2] + RELAX * dz;

      /* update the control points */
      for (i = 0; i < imax; i++) {
        cps[3*i+3] = cp[3*i+3];
        cps[3*i+4] = cp[3*i+4];
        cps[3*i+5] = cp[3*i+5];
      }

      /* convergence check */
      if (dxyzmax < tol) break;

  }
  if (dxyzmax >= tol)
      printf(" Warning: Not Converged (EG_spline1dFit)!\n");

//cleanup:
  return status;
}

