# Import pyCAPS module
import pyCAPS

# Import os module
import os

# Import argparse module
import argparse

# Setup and read command line options. Please note that this isn't required for pyCAPS
parser = argparse.ArgumentParser(description = 'AFLR4 and AFLR3 Pytest Example',
                                 prog = 'aflr4_and_aflr3_PyTest',
                                 formatter_class = argparse.ArgumentDefaultsHelpFormatter)

#Setup the available commandline options
parser.add_argument('-workDir', default = ["." + os.sep], nargs=1, type=str, help = 'Set working/run directory')
parser.add_argument("-outLevel", default = 1, type=int, choices=[0, 1, 2], help="Set output verbosity")
args = parser.parse_args()

# Set analysis directory
workDir = os.path.join(str(args.workDir[0]), "AFLRAnalysisTest")

# Load CSM file
geometryScript = os.path.join("..","csmData","cfdMultiBody.csm")
capsProblem = pyCAPS.Problem(problemName=workDir,
                             capsFile=geometryScript,
                             outLevel=args.outLevel)

# Load AFLR4 aim
aflr4 = capsProblem.analysis.create(aim = "aflr4AIM")

# Set AIM verbosity
aflr4.input.Mesh_Quiet_Flag = True if args.outLevel == 0 else False

# Optional: Explicitly write mesh files
aflr4.input.Mesh_Format = ["Tecplot", "stl", "ugrid"]

# Farfield growth factor
aflr4.input.ff_cdfr = 1.3

# Set maximum and minimum edge lengths relative to capsMeshLength
aflr4.input.max_scale = 0.7
aflr4.input.ideal_min_scale = 0.07
#aflr4.input.Mesh_Length_Factor = 2
aflr4.input.curv_angle = 25

# Incrase resoltion on the wake
aflr4.input.Mesh_Sizing = {"Wake": {"scaleFactor":0.5}}

######################################
## AFRL4 executes automatically     ##
######################################

######################################
## Build volume mesh off of surface ##
##  mesh(es) using AFLR3            ##
######################################

# Load AFLR3 aim
aflr3 = capsProblem.analysis.create(aim = "aflr3AIM")

# Link the surface mesh
aflr3.input["Surface_Mesh"].link(aflr4.output["Surface_Mesh"])

# Set AIM verbosity
aflr3.input.Mesh_Quiet_Flag = True if args.outLevel == 0 else False

# Optional: Explicitly write mesh files
Mesh_Format = ["Tecplot", "ugrid", "vtk", "bvtk"]
if os.path.isfile( os.path.join(os.environ["ESP_ROOT"],"lib","exodusWriter.so") ):
    Mesh_Format.append("EXODUS")
aflr3.input.Mesh_Format = Mesh_Format

# Print out mesh quality information and dump a CFD++ file
aflr3.input.Mesh_Gen_Input_String = "-qall -o .cfd++"

# Limit the maximum number of boundary layers
aflr3.input.BL_Max_Layers = 4

# Set either global or local boundary layer thickness spacings
# These are coarse numbers just as an example, not a recommenation for good CFD solutions
useGlobal = False

if useGlobal:
    aflr3.input.BL_Initial_Spacing = 0.01
    aflr3.input.BL_Thickness = 0.1

else:
    inviscidBC = {"boundaryLayerThickness" : 0.0, "boundaryLayerSpacing" : 0.0}
    viscousBC  = {"boundaryLayerThickness" : 0.05, "boundaryLayerSpacing" : 0.025}

    # Set mesh sizing parmeters
    aflr3.input.Mesh_Sizing = {"Wing1": viscousBC, "Wing2": inviscidBC, 
                               "Wake": viscousBC | {"bcType":"-TRANSP_UG3_GBC"}} # Grow boundary layer on the wake

# Run AIM
aflr3.runAnalysis()
