# Import pyCAPS class file
import pyCAPS

# Import os module
import os
import argparse

# Setup and read command line options. Please note that this isn't required for pyCAPS
parser = argparse.ArgumentParser(description = 'FlightStream X15 VSP Pytest Example',
                                 prog = 'cbaero_PyTest',
                                 formatter_class = argparse.ArgumentDefaultsHelpFormatter)

#Setup the available commandline options
parser.add_argument('-workDir', default = "./", nargs=1, type=str, help = 'Set working/run directory')
parser.add_argument("-outLevel", default = 1, type=int, choices=[0, 1, 2], help="Set output verbosity")
parser.add_argument('-noPlotData', action='store_true', default = False, help = "Don't plot data")
args = parser.parse_args()

# Define units
deg  = pyCAPS.Unit("degree")
Pa   = pyCAPS.Unit("Pa")
ft   = pyCAPS.Unit("feet")
m    = pyCAPS.Unit("m")
mile = pyCAPS.Unit("mile")
s    = pyCAPS.Unit("s")
hour = pyCAPS.Unit("hour")
kg   = pyCAPS.Unit("kg")
K    = pyCAPS.Unit("Kelvin")

# Create working directory variable
projectName = "FlightStreamX15ESPTest"
workDir = os.path.join(str(args.workDir[0]), projectName)

# Load CSM file
geometryScript = os.path.join("..","csmData","x15","x15.csm")
capsProblem = pyCAPS.Problem(problemName=workDir,
                             capsFile=geometryScript,
                             outLevel=args.outLevel)

# Turn on only OML and make speate bodies for stabilizers
capsProblem.geometry.cfgpmtr.VIEW.Concept = 0
capsProblem.geometry.cfgpmtr.VIEW.Oml = 1
capsProblem.geometry.cfgpmtr.VIEW.Iml = 0
capsProblem.geometry.cfgpmtr.COMP.Htailhinge = 1

# Load egadsTess aim to create surface mesh
aflr4 = capsProblem.analysis.create(aim = "aflr4AIM",
                                    name="aflr4")

# Set AIM verbosity
aflr4.input.Mesh_Quiet_Flag = True

# Dissable curvature refinement when AFLR4 cannot generate a mesh
# aflr4.input.Mesh_Gen_Input_String = "auto_mode=0"

# Set maximum and minimum edge lengths relative to capsMeshLength
aflr4.input.curv_angle = 20
aflr4.input.max_scale = 0.3
aflr4.input.Mesh_Length_Factor = 2

# Load FlightStream aim
flightstream = capsProblem.analysis.create(aim = "flightstreamAIM",
                                           name = "FlightStream")

# Link mesh
flightstream.input["Surface_Mesh"].link(aflr4.output["Surface_Mesh"])

# Specify the possibly full PATH to FlightStream executable (optional)
flightstream.input.FlightStream = "FlightStream"

# Set the solver model
flightstream.input.Solver_Model = "MODIFIED_NEWTONIAN"

# Set flow conditions
refSonicVelocity = 295.06 * m/s

flightstream.input.Mach     = 6.83
flightstream.input.Alpha    = 1.0 * deg
flightstream.input.Beta     = 0.0 * deg

flightstream.input.Fluid_Properties = {"density": 0.072 * kg/m**3,
                                       "pressure": 5692.95 * Pa,
                                       "sonic_velocity": refSonicVelocity,
                                       "temperature": 216.67 * K,
                                       "viscosity": 0.0000142 * Pa*s}


# Reference velocity
flightstream.input.ReferenceVelocity = refSonicVelocity * flightstream.input.Mach

# Export surface data in Tecplot and VTK file formats
flightstream.input.Export_Solver_Analysis = ["Tecplot", "VTK"]

# Morph the mesh instead of re-meshing
flightstream.input.Mesh_Morph = True

CD = []
CL = []
CM = []

incidences = range(-10,11,1) if args.noPlotData == False else range(-1,2,1)
for incidence in incidences:
    # Update the geometry incidence angle
    capsProblem.geometry.despmtr.Htail.incidence = incidence

    flightstream.input.Proj_Name = "x15_hstab_"+str(incidence)

    print("-"*20)
    print("HStabIncidence", incidence)
    print("-"*20)

    CD.append(flightstream.output["CDi"].value)
    CL.append(flightstream.output["CL"].value)
    CM.append(flightstream.output["CMy"].value)

    flightstream.input["Surface_Mesh"].unlink()

if args.noPlotData == False:
    try:
        # Import pyplot module
        from matplotlib import pyplot as plt

        def plot(C, c):

            fig, ax1 = plt.subplots()

            # Plot the functional
            lns1 = ax1.plot(incidences, C, 'o-', label = r"$"+c+r"$", color='blue')

            ax1.set_xlabel("H-stab Incidence (deg)")
            ax1.set_ylabel(r"$"+c+r"$")

            plt.savefig("x15_ESP_"+c+".png")

        plot(CD, 'C_D')
        plot(CL, 'C_L')
        plot(CM, 'C_M')

        plt.show()

    except ImportError:
        print ("Unable to import matplotlib.pyplot module.")

