/*
 *      CAPS: Computational Aircraft Prototype Syntheses
 *
 *             ugrid 3D Mesh Writer Code
 *
 *      Copyright 2014-2025, Massachusetts Institute of Technology
 *      Licensed under The GNU Lesser General Public License, version 2.1
 *      See http://www.opensource.org/licenses/lgpl-2.1.php
 *
 */

#include <string.h>
#include "aimUtil.h"
#include "aimMesh.h"

#include "ugridWriter.h"

#ifdef WIN32
#include <io.h>
#define access     _access
#define F_OK       0
#else
#include <unistd.h>
#endif

#define MAX(A,B)  (((A) < (B)) ? (B) : (A))

const char *meshExtension()
{
/*@-observertrans@*/
  return MESHEXTENSION;
/*@+observertrans@*/
}

/*****************************************************************************/

int meshWrite(void *aimInfo, aimMesh *mesh)
{
  int  status; // Function return status
  int  nLine=0, nTri=0, nQuad=0;
  int  nTet=0, nPyramid=0, nPrism=0, nHex=0;
  int  igroup, ielem, nPoint, elementTopo=aimUnknownElem, ielemTopo, elemID = 0, nElems;
  char filename[PATH_MAX];
  FILE *fp=NULL;
  aimMeshData *meshData = NULL;

  printf("\nWriting ugrid file ....\n");

  if (mesh == NULL) return CAPS_NULLVALUE;
  if (mesh->meshRef  == NULL) return CAPS_NULLVALUE;
  if (mesh->meshData == NULL) return CAPS_NULLVALUE;

  meshData = mesh->meshData;

  if (meshData->dim != 2 && meshData->dim != 3) {
    AIM_ERROR(aimInfo, "meshData dim = %d must be 2 or 3!!!", meshData->dim);
    return CAPS_BADVALUE;
  }

  snprintf(filename, PATH_MAX, "%s%s", mesh->meshRef->fileName, MESHEXTENSION);

  fp = fopen(filename, "w");
  if (fp == NULL) {
    AIM_ERROR(aimInfo, "Cannot open file: %s\n", filename);
    return CAPS_IOERR;
  }

  for (igroup = 0; igroup < meshData->nElemGroup; igroup++) {
    if (meshData->elemGroups[igroup].order != 1) {
      AIM_ERROR(aimInfo, "SU2 only supports linear mesh elements! group %d order = %d",
                igroup, meshData->elemGroups[igroup].order);
      status = CAPS_IOERR;
      goto cleanup;
    }

    // count the number of element types
         if (meshData->elemGroups[igroup].elementTopo == aimLine   ) nLine    += meshData->elemGroups[igroup].nElems;
    else if (meshData->elemGroups[igroup].elementTopo == aimTri    ) nTri     += meshData->elemGroups[igroup].nElems;
    else if (meshData->elemGroups[igroup].elementTopo == aimQuad   ) nQuad    += meshData->elemGroups[igroup].nElems;
    else if (meshData->elemGroups[igroup].elementTopo == aimTet    ) nTet     += meshData->elemGroups[igroup].nElems;
    else if (meshData->elemGroups[igroup].elementTopo == aimPyramid) nPyramid += meshData->elemGroups[igroup].nElems;
    else if (meshData->elemGroups[igroup].elementTopo == aimPrism  ) nPrism   += meshData->elemGroups[igroup].nElems;
    else if (meshData->elemGroups[igroup].elementTopo == aimHex    ) nHex     += meshData->elemGroups[igroup].nElems;
    else {
      AIM_ERROR(aimInfo, "Unknown group %d element topology: %d", igroup+1, meshData->elemGroups[igroup].elementTopo);
      status = CAPS_MISMATCH;
      goto cleanup;
    }
  }

  /* write a binary UGRID file */
  status = fwrite(&meshData->nVertex, sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
  status = fwrite(&nTri,     sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
  status = fwrite(&nQuad,    sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
  status = fwrite(&nTet,     sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
  status = fwrite(&nPyramid, sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
  status = fwrite(&nPrism,   sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
  status = fwrite(&nHex,     sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

  /* write all of the vertices */
  status = fwrite(meshData->verts, sizeof(aimMeshCoords), meshData->nVertex, fp);
  if (status != meshData->nVertex) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

  /* write triangles and quads */
  for (ielemTopo = 0; ielemTopo < 2; ielemTopo++) {
         if (ielemTopo == 0) elementTopo = aimTri;
    else if (ielemTopo == 1) elementTopo = aimQuad;

    for (igroup = 0; igroup < meshData->nElemGroup; igroup++) {
      if (meshData->elemGroups[igroup].elementTopo != elementTopo) continue;

      nPoint = meshData->elemGroups[igroup].nPoint;
      nElems = meshData->elemGroups[igroup].nElems;

      /* write the element connectivity */
      status = fwrite(meshData->elemGroups[igroup].elements, sizeof(int), nPoint*nElems, fp);
      if (status != nPoint*nElems) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
    }
  }

  /* write triangles and quads IDs*/
  for (ielemTopo = 0; ielemTopo < 2; ielemTopo++) {
         if (ielemTopo == 0) elementTopo = aimTri;
    else if (ielemTopo == 1) elementTopo = aimQuad;

    for (igroup = 0; igroup < meshData->nElemGroup; igroup++) {
      if (meshData->elemGroups[igroup].elementTopo != elementTopo) continue;

      nElems = meshData->elemGroups[igroup].nElems;
      elemID = meshData->elemGroups[igroup].ID;

      /* write the element ID */
      for (ielem = 0; ielem < nElems; ielem++) {
        status = fwrite(&elemID,  sizeof(int), 1, fp);
        if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
      }
    }
  }

  /* write volume elements*/
  for (ielemTopo = 0; ielemTopo < 4; ielemTopo++) {
         if (ielemTopo == 0) elementTopo = aimTet;
    else if (ielemTopo == 1) elementTopo = aimPyramid;
    else if (ielemTopo == 2) elementTopo = aimPrism;
    else if (ielemTopo == 3) elementTopo = aimHex;

    for (igroup = 0; igroup < meshData->nElemGroup; igroup++) {
      if (meshData->elemGroups[igroup].elementTopo != elementTopo) continue;

      nPoint = meshData->elemGroups[igroup].nPoint;
      nElems = meshData->elemGroups[igroup].nElems;

      /* write the element connectivity */
      status = fwrite(meshData->elemGroups[igroup].elements, sizeof(int), nPoint*nElems, fp);
      if (status != nPoint*nElems) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
    }
  }

  if (nTet+nPyramid+nPrism+nHex == 0) {

    status = fwrite(&nLine, sizeof(int), 1, fp);
    if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

    for (igroup = 0; igroup < meshData->nElemGroup; igroup++) {
      if (meshData->elemGroups[igroup].elementTopo != aimLine) continue;

      nPoint = meshData->elemGroups[igroup].nPoint;
      nElems = meshData->elemGroups[igroup].nElems;
      elemID = meshData->elemGroups[igroup].ID;

      /* write the element connectivity and ID */
      for (ielem = 0; ielem < nElems; ielem++) {
        status = fwrite(&meshData->elemGroups[igroup].elements[nPoint*ielem], sizeof(int), nPoint, fp);
        if (status != nPoint) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
        status = fwrite(&elemID, sizeof(int), 1, fp);
        if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
      }
    }
  }

  printf("Finished writing ugrid file\n\n");

  status = CAPS_SUCCESS;

cleanup:

  if (fp != NULL) fclose(fp);
  return status;
}

/*****************************************************************************/

typedef struct
{
  char *name;
  int ID;
} NameID;


static int
readElements(void *aimInfo, FILE *fp, /*@null@*/ FILE *fpMV,
             int nName, /*@null@*/ NameID *names,
             int volID, int nPoint, enum aimMeshElem elementTopo, int nElems,
             int *elementIndex, aimMeshData *meshData)
{
  int status = CAPS_SUCCESS;
  int i, j, ID, igroup;
  int nMapGroupID = 0;
  int *mapGroupID = NULL, *IDs = NULL;
  char *name = NULL;

  if (nElems > 0) {

    if (fpMV != NULL) {
      AIM_ALLOC(IDs, nElems, int, aimInfo, status);

      for (i = 0; i < nElems; i++) {

        /* read the volume ID */
        status = fread(&ID, sizeof(int), 1, fpMV);
        if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
        if (ID <= 0) {
          AIM_ERROR(aimInfo, "ID must be a positive number: %d!", ID);
          status = CAPS_IOERR;
          goto cleanup;
        }

        /* add the volume group if necessary */
        if (ID > nMapGroupID) {
          AIM_REALL(mapGroupID, ID, int, aimInfo, status);
          for (j = nMapGroupID; j < ID; j++) mapGroupID[j] = -1;
          nMapGroupID = ID;
        }
        if (mapGroupID[ID-1] == -1) {
          if (names != NULL) {
            j = 0;
            while (names[j].ID != ID) {
              j++;
              if (nName == j) {
                AIM_ERROR(aimInfo, "Failed to find 'name' for ID %d!", ID);
                status = CAPS_IOERR;
                goto cleanup;
              }
            }
            name = names[j].name;
          }
          status = aim_addMeshElemGroup(aimInfo, name, ID, elementTopo, 1, nPoint, meshData);
          AIM_STATUS(aimInfo, status);
          mapGroupID[ID-1] = meshData->nElemGroup-1;
        }

        igroup = mapGroupID[ID-1];

        IDs[i] = ID;

        /* add the element to the group */
        meshData->elemGroups[igroup].nElems++;
      }

      for (ID = 0; ID < nMapGroupID; ID++) {
        igroup = mapGroupID[ID];
        if (igroup == -1) continue;

        /* resize the element group */
        AIM_REALL(meshData->elemGroups[igroup].elements, meshData->elemGroups[igroup].nPoint*(meshData->elemGroups[igroup].nElems), int, aimInfo, status);
        meshData->elemGroups[igroup].nElems = 0;
      }

      for (i = 0; i < nElems; i++) {

        /* get the volume group */
        igroup = mapGroupID[IDs[i]-1];

        /* read the element connectivity */
        status = fread(&meshData->elemGroups[igroup].elements[nPoint*(meshData->elemGroups[igroup].nElems)],
                       sizeof(int), nPoint, fp);
        if (status != nPoint) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

        meshData->elemMap[*elementIndex][0] = igroup;
        meshData->elemMap[*elementIndex][1] = meshData->elemGroups[igroup].nElems;

        meshData->elemGroups[igroup].nElems += 1;
        *elementIndex += 1;
      }

    } else {

      status = aim_addMeshElemGroup(aimInfo, NULL, volID, elementTopo, 1, nPoint, meshData);
      AIM_STATUS(aimInfo, status);

      igroup = meshData->nElemGroup-1;

      /* add the elements to the group */
      status = aim_addMeshElem(aimInfo, nElems, &meshData->elemGroups[igroup]);
      AIM_STATUS(aimInfo, status);

      /* read the element connectivity */
      status = fread(meshData->elemGroups[igroup].elements,
                     sizeof(int), nPoint*nElems, fp);
      if (status != nPoint*nElems) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

      for (i = 0; i < nElems; i++) {
        meshData->elemMap[*elementIndex][0] = igroup;
        meshData->elemMap[*elementIndex][1] = i;

        *elementIndex += 1;
      }
    }
  }

  status = CAPS_SUCCESS;
cleanup:

  AIM_FREE(mapGroupID);
  AIM_FREE(IDs);

  return status;
}

/*****************************************************************************/

int meshRead(void *aimInfo, aimMesh *mesh)
{
  int    status = CAPS_SUCCESS;

  int    nLine, nTri, nQuad;
  int    nTet, nPyramid, nPrism, nHex;
  int    i, j, elementIndex, nPoint, nElems, ID, igroup;
  int    line[2];
  int    maxVolID = 0, nVolName=0, nBCName=0, volID=1;
  int    nMapGroupID = 0, *mapGroupID = NULL;
  enum aimMeshElem elementTopo;
  char filename[PATH_MAX], groupName[PATH_MAX];
  char *name = NULL;
  NameID *volName=NULL, *bcName=NULL;
  size_t len;
  FILE *fp = NULL, *fpID = NULL, *fpMV=NULL;
  aimMeshData *meshData = NULL;

  if (mesh           == NULL) return CAPS_NULLOBJ;
  if (mesh->meshRef  == NULL) return CAPS_NULLOBJ;
  if (mesh->meshRef->fileName  == NULL) return CAPS_NULLOBJ;

  status = aim_freeMeshData(mesh->meshData);
  AIM_STATUS(aimInfo, status);
  AIM_FREE(mesh->meshData);

  AIM_ALLOC(meshData, 1, aimMeshData, aimInfo, status);
  status = aim_initMeshData(meshData);
  AIM_STATUS(aimInfo, status);

  /* read in the groupName from mapbc file */
  snprintf(filename, PATH_MAX, "%s%s", mesh->meshRef->fileName, ".mapbc");

  if (access(filename, F_OK) == 0) {
    // Correct the groupID's to be consistent with groupMap
    fpID = fopen(filename, "r");
    if (fpID == NULL) {
      AIM_ERROR(aimInfo, "Failed to open %s", filename);
      status = CAPS_IOERR;
      goto cleanup;
    }
    status = fscanf(fpID, "%d", &nBCName);
    if (status != 1) {
      AIM_ERROR(aimInfo, "Failed to read %s", filename);
      status = CAPS_IOERR;
      goto cleanup;
    }

    AIM_ALLOC(bcName, nBCName, NameID, aimInfo, status);
    for (i = 0; i < nBCName; i++) { bcName[i].name = NULL; bcName[i].ID = 0; }

    for (i = 0; i < nBCName; i++) {
      status = fscanf(fpID, "%d %d %s", &bcName[i].ID, &j, groupName);
      if (status != 3) {
        AIM_ERROR(aimInfo, "Failed to read %s", filename);
        status = CAPS_IOERR;
        goto cleanup;
      }

      AIM_STRDUP(bcName[i].name, groupName, aimInfo, status);

      volID = MAX(volID, bcName[i].ID+1);
    }

    /*@-dependenttrans@*/
    fclose(fpID); fpID = NULL;
    /*@+dependenttrans@*/
  }


  snprintf(filename, PATH_MAX, "%s%s", mesh->meshRef->fileName, MESHEXTENSION);

  fp = fopen(filename, "rb");
  if (fp == NULL) {
    AIM_ERROR(aimInfo, "Cannot open file: %s\n", filename);
    status = CAPS_IOERR;
    goto cleanup;
  }

  // Open a 2nd copy to read in the BC ID's
  fpID = fopen(filename, "rb");
  if (fpID == NULL) {
    AIM_ERROR(aimInfo, "Cannot open file: %s\n", filename);
    status = CAPS_IOERR;
    goto cleanup;
  }

  /* read a binary UGRID file */
  status = fread(&meshData->nVertex, sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
  status = fread(&nTri,     sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
  status = fread(&nQuad,    sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
  status = fread(&nTet,     sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
  status = fread(&nPyramid, sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
  status = fread(&nPrism,   sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
  status = fread(&nHex,     sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

  /* skip the header */
  status = fseek(fpID, 7*sizeof(int), SEEK_CUR);
  if (status != 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

  /*
  printf("\n Header from UGRID file: %d  %d %d  %d %d %d %d\n", numNode,
         numTriangle, numQuadrilateral, numTetrahedral, numPyramid, numPrism,
         numHexahedral);
   */

  snprintf(filename, PATH_MAX, "%s%s", mesh->meshRef->fileName, ".mapvol");

  // File for volume IDs
  fpMV = fopen(filename, "rb");
  if (fpMV != NULL) {
    status = fread(&nVolName, sizeof(int), 1, fpMV);
    if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

    /* read the maximum ID value */
    status = fread(&maxVolID, sizeof(int), 1, fpMV);
    if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

    if (nVolName == 0) {
      AIM_ERROR(aimInfo, "Invalid mapvol file with zero nVolName!");
      status = CAPS_IOERR;
      goto cleanup;
    }

    AIM_ALLOC(volName, nVolName, NameID, aimInfo, status);
    for (i = 0; i < nVolName; i++) { volName[i].name = NULL; volName[i].ID = 0; }

    for (i = 0; i < nVolName; i++) {
      status = fread(&volName[i].ID, sizeof(int), 1, fpMV);
      if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
      volID = MAX(volID, volName[i].ID+1);

      status = fread(&len, sizeof(size_t), 1, fpMV);
      if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

      AIM_ALLOC(volName[i].name, len, char, aimInfo, status);

      status = fread(volName[i].name, sizeof(char), len, fpMV);
      if (status != len) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
    }

    status = fread(&nElems, sizeof(int), 1, fpMV);
    if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

    if (nElems != nTet+nPyramid+nPrism+nHex) {
      AIM_ERROR(aimInfo, "Element count missmatch in mapvol file!");
      status = CAPS_IOERR;
      goto cleanup;
    }
  }

  AIM_ALLOC(meshData->verts, meshData->nVertex, aimMeshCoords, aimInfo, status);

  /* read all of the vertices */
  status = fread(meshData->verts, sizeof(aimMeshCoords), meshData->nVertex, fp);
  if (status != meshData->nVertex) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

  /* skip the verts */
  status = fseek(fpID, meshData->nVertex*sizeof(aimMeshCoords), SEEK_CUR);
  if (status != 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }


  // Numbers
  meshData->nTotalElems =  nTri     +
                           nQuad    +
                           nTet     +
                           nPyramid +
                           nPrism   +
                           nHex;

  // allocate the element map that maps back to the original element numbering
  AIM_ALLOC(meshData->elemMap, meshData->nTotalElems, aimMeshIndices, aimInfo, status);

  /*
  printf("Volume mesh:\n");
  printf("\tNumber of nodes          = %d\n", meshData->nVertex);
  printf("\tNumber of elements       = %d\n", meshData->nTotalElems);
  printf("\tNumber of triangles      = %d\n", numTriangle);
  printf("\tNumber of quadrilatarals = %d\n", numQuadrilateral);
  printf("\tNumber of tetrahedrals   = %d\n", numTetrahedral);
  printf("\tNumber of pyramids       = %d\n", numPyramid);
  printf("\tNumber of prisms         = %d\n", numPrism);
  printf("\tNumber of hexahedrals    = %d\n", numHexahedral);
*/

  /* skip the Tri+Quad elements for the ID */
  status = fseek(fpID, (3*nTri+4*nQuad)*sizeof(int), SEEK_CUR);
  if (status != 0) { status = CAPS_IOERR; goto cleanup; }

  /* Start of element index */
  elementIndex = 0;

  /* Elements triangles */
  nPoint = 3;
  elementTopo = aimTri;
  nElems = nTri;

  status = readElements(aimInfo, fp, nTet+nPyramid+nPrism+nHex == 0 ? NULL : fpID,
                        nBCName, nTet+nPyramid+nPrism+nHex == 0 ? NULL : bcName,
                        volID, nPoint, elementTopo, nElems,
                        &elementIndex, meshData);
  AIM_STATUS(aimInfo, status);

  /* Elements quadrilateral */
  nPoint = 4;
  elementTopo = aimQuad;
  nElems = nQuad;

  status = readElements(aimInfo, fp, nTet+nPyramid+nPrism+nHex == 0 ? NULL : fpID,
                        nBCName, nTet+nPyramid+nPrism+nHex == 0 ? NULL : bcName,
                        volID, nPoint, elementTopo, nElems,
                        &elementIndex, meshData);
  AIM_STATUS(aimInfo, status);

  /*@-dependenttrans@*/
  fclose(fpID); fpID = NULL;
  /*@+dependenttrans@*/

  // skip face ID section of the file
  status = fseek(fp, (nTri + nQuad)*sizeof(int), SEEK_CUR);
  if (status != 0) { status = CAPS_IOERR; goto cleanup; }

  // Elements Tetrahedral
  nPoint = 4;
  elementTopo = aimTet;
  nElems = nTet;

  status = readElements(aimInfo, fp, fpMV,
                        nVolName, volName,
                        volID, nPoint, elementTopo, nElems,
                        &elementIndex, meshData);
  AIM_STATUS(aimInfo, status);

  // Elements Pyramid
  nPoint = 5;
  elementTopo = aimPyramid;
  nElems = nPyramid;

  status = readElements(aimInfo, fp, fpMV,
                        nVolName, volName,
                        volID, nPoint, elementTopo, nElems,
                        &elementIndex, meshData);
  AIM_STATUS(aimInfo, status);

  // Elements Prism
  nPoint = 6;
  elementTopo = aimPrism;
  nElems = nPrism;

  status = readElements(aimInfo, fp, fpMV,
                        nVolName, volName,
                        volID, nPoint, elementTopo, nElems,
                        &elementIndex, meshData);
  AIM_STATUS(aimInfo, status);

  // Elements Hex
  nPoint = 8;
  elementTopo = aimHex;
  nElems = nHex;

  status = readElements(aimInfo, fp, fpMV,
                        nVolName, volName,
                        volID, nPoint, elementTopo, nElems,
                        &elementIndex, meshData);
  AIM_STATUS(aimInfo, status);

  // 2D grid
  if (nTet+nPyramid+nPrism+nHex == 0) {
    meshData->dim = 2;

    status = fread(&nLine, sizeof(int), 1, fp);
    if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

    meshData->nTotalElems += nLine;
    AIM_REALL(meshData->elemMap, meshData->nTotalElems, aimMeshIndices, aimInfo, status);

    // Elements Line
    nPoint = 2;
    elementTopo = aimLine;
    nElems = nLine;

    for (i = 0; i < nElems; i++) {
      /* read the element connectivity */
      status = fread(line, sizeof(int), nPoint, fp);
      if (status != nPoint) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
      status = fread(&ID, sizeof(int), 1, fp);
      if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
      if (ID <= 0) {
        AIM_ERROR(aimInfo, "BC ID must be a positive number: %d!", ID);
        status = CAPS_IOERR;
        goto cleanup;
      }

      /* add the BC group if necessary */
      if (ID > nMapGroupID) {
        AIM_REALL(mapGroupID, ID, int, aimInfo, status);
        for (j = nMapGroupID; j < ID; j++) mapGroupID[j] = -1;
        nMapGroupID = ID;
      }
      if (mapGroupID[ID-1] == -1) {
        j = 0;
        if (bcName != NULL) {
          while (bcName[j].ID != ID) {
            j++;
            if (nBCName == j) {
              AIM_ERROR(aimInfo, "Failed to find 'name' for ID %d!", ID);
              status = CAPS_IOERR;
              goto cleanup;
            }
          }
          name = bcName[j].name;
        }
        status = aim_addMeshElemGroup(aimInfo, name, ID, elementTopo, 1, nPoint, meshData);
        AIM_STATUS(aimInfo, status);
        mapGroupID[ID-1] = meshData->nElemGroup-1;
      }

      igroup = mapGroupID[ID-1];

      /* add the elements to the group */
      status = aim_addMeshElem(aimInfo, 1, &meshData->elemGroups[igroup]);
      AIM_STATUS(aimInfo, status);

      meshData->elemGroups[igroup].elements[nPoint*meshData->elemGroups[igroup].nElems-2] = line[0];
      meshData->elemGroups[igroup].elements[nPoint*meshData->elemGroups[igroup].nElems-1] = line[1];

      meshData->elemMap[elementIndex][0] = igroup;
      meshData->elemMap[elementIndex][1] = meshData->elemGroups[igroup].nElems-1;

      elementIndex += 1;
    }

  } else {
    meshData->dim = 3;
  }

  mesh->meshData = meshData;
  meshData = NULL;

  status = CAPS_SUCCESS;

cleanup:
  if (status != CAPS_SUCCESS) {
    aim_freeMeshData(meshData);
    AIM_FREE(meshData);
  }
  AIM_FREE(mapGroupID);

  if (volName != NULL) {
    for (i = 0; i < nVolName; i++)
      AIM_FREE(volName[i].name);
    AIM_FREE(volName);
  }

  if (bcName != NULL) {
    for (i = 0; i < nBCName; i++)
      AIM_FREE(bcName[i].name);
    AIM_FREE(bcName);
  }

/*@-dependenttrans@*/
  if (fp   != NULL) fclose(fp);
  if (fpID != NULL) fclose(fpID);
  if (fpMV != NULL) fclose(fpMV);
/*@+dependenttrans@*/

  return status;
}


/*****************************************************************************/

int meshCounts(void *aimInfo, aimMesh *mesh)
{
  int    status = CAPS_SUCCESS;

  int    nLine, nTri, nQuad;
  int    nTet, nPyramid, nPrism, nHex;
  int    nPoint, nElems, ID;
  enum aimMeshElem elementTopo;
  char filename[PATH_MAX];
  char *name = NULL;
  FILE *fp = NULL;
  aimMeshData *meshData = NULL;

  if (mesh           == NULL) return CAPS_NULLOBJ;
  if (mesh->meshRef  == NULL) return CAPS_NULLOBJ;
  if (mesh->meshRef->fileName  == NULL) return CAPS_NULLOBJ;

  AIM_ALLOC(meshData, 1, aimMeshData, aimInfo, status);
  status = aim_initMeshData(meshData);
  AIM_STATUS(aimInfo, status);

  snprintf(filename, PATH_MAX, "%s%s", mesh->meshRef->fileName, MESHEXTENSION);

  fp = fopen(filename, "rb");
  if (fp == NULL) {
    AIM_ERROR(aimInfo, "Cannot open file: %s\n", filename);
    status = CAPS_IOERR;
    goto cleanup;
  }

  /* read a binary UGRID file */
  status = fread(&meshData->nVertex, sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
  status = fread(&nTri,     sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
  status = fread(&nQuad,    sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
  status = fread(&nTet,     sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
  status = fread(&nPyramid, sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
  status = fread(&nPrism,   sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
  status = fread(&nHex,     sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

  snprintf(filename, PATH_MAX, "%s%s", mesh->meshRef->fileName, ".mapvol");

  // Numbers
  meshData->nTotalElems =  nTri     +
                           nQuad    +
                           nTet     +
                           nPyramid +
                           nPrism   +
                           nHex;

  /* Start of element index */
  ID = 0;

  /* Elements triangles */
  nPoint = 3;
  elementTopo = aimTri;
  nElems = nTri;
  ID++;
  status = aim_addMeshElemGroup(aimInfo, name, ID, elementTopo, 1, nPoint, meshData);
  AIM_STATUS(aimInfo, status);
  meshData->elemGroups[meshData->nElemGroup-1].nElems = nElems;

  /* Elements quadrilateral */
  nPoint = 4;
  elementTopo = aimQuad;
  nElems = nQuad;
  ID++;
  status = aim_addMeshElemGroup(aimInfo, name, ID, elementTopo, 1, nPoint, meshData);
  AIM_STATUS(aimInfo, status);
  meshData->elemGroups[meshData->nElemGroup-1].nElems = nElems;

  // Elements Tetrahedral
  nPoint = 4;
  elementTopo = aimTet;
  nElems = nTet;
  ID++;
  status = aim_addMeshElemGroup(aimInfo, name, ID, elementTopo, 1, nPoint, meshData);
  AIM_STATUS(aimInfo, status);
  meshData->elemGroups[meshData->nElemGroup-1].nElems = nElems;

  // Elements Pyramid
  nPoint = 5;
  elementTopo = aimPyramid;
  nElems = nPyramid;
  ID++;
  status = aim_addMeshElemGroup(aimInfo, name, ID, elementTopo, 1, nPoint, meshData);
  AIM_STATUS(aimInfo, status);
  meshData->elemGroups[meshData->nElemGroup-1].nElems = nElems;

  // Elements Prism
  nPoint = 6;
  elementTopo = aimPrism;
  nElems = nPrism;
  ID++;
  status = aim_addMeshElemGroup(aimInfo, name, ID, elementTopo, 1, nPoint, meshData);
  AIM_STATUS(aimInfo, status);
  meshData->elemGroups[meshData->nElemGroup-1].nElems = nElems;

  // Elements Hex
  nPoint = 8;
  elementTopo = aimHex;
  nElems = nHex;
  ID++;
  status = aim_addMeshElemGroup(aimInfo, name, ID, elementTopo, 1, nPoint, meshData);
  AIM_STATUS(aimInfo, status);
  meshData->elemGroups[meshData->nElemGroup-1].nElems = nElems;

  // 2D grid
  if (nTet+nPyramid+nPrism+nHex == 0) {
    meshData->dim = 2;

    status = fread(&nLine, sizeof(int), 1, fp);
    if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

    meshData->nTotalElems += nLine;

    // Elements Line
    nPoint = 2;
    elementTopo = aimLine;
    nElems = nLine;
    ID++;
    status = aim_addMeshElemGroup(aimInfo, name, ID, elementTopo, 1, nPoint, meshData);
    AIM_STATUS(aimInfo, status);
    meshData->elemGroups[meshData->nElemGroup-1].nElems = nElems;

  } else {
    meshData->dim = 3;
  }

  mesh->meshData = meshData;
  meshData = NULL;

  status = CAPS_SUCCESS;

cleanup:
  if (status != CAPS_SUCCESS) {
    aim_freeMeshData(meshData);
    AIM_FREE(meshData);
  }

/*@-dependenttrans@*/
  if (fp   != NULL) fclose(fp);
/*@+dependenttrans@*/

  return status;
}

