// Tetgen interface functions - Written by Dr. Ryan Durscher AFRL/RQVC
// This software has been cleared for public release on 05 Nov 2020, case number 88ABW-2020-3462.

#include <vector>
#include <set>
#include <iostream>

#include "aimUtil.h"

#include "tetgen.h"
#include "meshTypes.h"
#include "egads.h"
#include "capsTypes.h"

#include "meshUtils.h"

#include "tetgen_Interface.hpp"

#define CROSS(a,b,c)  a[0] = (b[1]*c[2]) - (b[2]*c[1]);\
                      a[1] = (b[2]*c[0]) - (b[0]*c[2]);\
                      a[2] = (b[0]*c[1]) - (b[1]*c[0])
#define DOT(a,b)     (a[0]*b[0] + a[1]*b[1] + a[2]*b[2])
#define MAX(a,b)     ((a) > (b) ? (a) : (b))


static int
tetgen_to_MeshStruct(tetgenio *mesh, meshStruct *genUnstrMesh)
{
    int status; // Function return status

    int i, j, elementIndex; // Indexing

    int numPoint;
    int defaultVolID = 1; // Default volume ID

    meshAnalysisTypeEnum analysisType;

    analysisType = genUnstrMesh->analysisType;

    // Cleanup existing node and elements
    (void) destroy_meshNodes(genUnstrMesh);

    (void) destroy_meshElements(genUnstrMesh);

    (void) destroy_meshQuickRefStruct(&genUnstrMesh->meshQuickRef);

    genUnstrMesh->meshType = VolumeMesh;

    // Numbers
    genUnstrMesh->numNode = mesh->numberofpoints;
    genUnstrMesh->numElement = mesh->numberoftrifaces + mesh->numberoftetrahedra;

    genUnstrMesh->meshQuickRef.useStartIndex = (int) true;
    genUnstrMesh->meshQuickRef.numTriangle = mesh->numberoftrifaces;
    genUnstrMesh->meshQuickRef.numTetrahedral = mesh->numberoftetrahedra;
    genUnstrMesh->meshQuickRef.startIndexTriangle = 0;
    genUnstrMesh->meshQuickRef.startIndexTetrahedral = mesh->numberoftrifaces;

    // Nodes - allocate
    genUnstrMesh->node = (meshNodeStruct *) EG_alloc(genUnstrMesh->numNode*sizeof(meshNodeStruct));
    if (genUnstrMesh->node == NULL) return EGADS_MALLOC;

    // Nodes - set
    for (i = 0; i < genUnstrMesh->numNode; i++) {

        // Initiate node
        status = initiate_meshNodeStruct(&genUnstrMesh->node[i], analysisType);
        if (status != CAPS_SUCCESS) return status;

        // Copy node data
        genUnstrMesh->node[i].nodeID = i+1;

        genUnstrMesh->node[i].xyz[0] = mesh->pointlist[3*i+0];
        genUnstrMesh->node[i].xyz[1] = mesh->pointlist[3*i+1];
        genUnstrMesh->node[i].xyz[2] = mesh->pointlist[3*i+2];

        /*
        if (mesh->numberofpointattributes != 0) {
        mesh->pointmarkerlist != NULL
        }
         */
    }

    // Elements - allocate
    genUnstrMesh->element = (meshElementStruct *) EG_alloc(genUnstrMesh->numElement*sizeof(meshElementStruct));
    if (genUnstrMesh->element == NULL) return EGADS_MALLOC;

    elementIndex = 0;
    numPoint = 0;
    // Elements -Set triangles
    for (i = 0; i < mesh->numberoftrifaces; i++) {
        status = initiate_meshElementStruct(&genUnstrMesh->element[elementIndex], analysisType);
        if (status != CAPS_SUCCESS) return status;

        genUnstrMesh->element[elementIndex].elementType = Triangle;
        genUnstrMesh->element[elementIndex].elementID   = elementIndex+1;
        genUnstrMesh->element[elementIndex].markerID = mesh->trifacemarkerlist[i];

        status = mesh_allocMeshElementConnectivity(&genUnstrMesh->element[elementIndex]);
        if (status != CAPS_SUCCESS) return status;

        if (i == 0) { // Only need this once
            numPoint = mesh_numMeshElementConnectivity(&genUnstrMesh->element[elementIndex]);
        }

        for (j = 0; j < numPoint; j++ ) {
            genUnstrMesh->element[elementIndex].connectivity[j] = mesh->trifacelist[numPoint*i+j];
        }

        elementIndex += 1;
    }

    //Set tetrahedral
    numPoint = 0;
    for (i = 0; i < mesh->numberoftetrahedra; i++) {
        status = initiate_meshElementStruct(&genUnstrMesh->element[elementIndex], analysisType);
        if (status != CAPS_SUCCESS) return status;

        genUnstrMesh->element[elementIndex].elementType = Tetrahedral;
        genUnstrMesh->element[elementIndex].elementID   = elementIndex+1;

        if (mesh->numberoftetrahedronattributes != 0) {
            genUnstrMesh->element[elementIndex].markerID = (int) mesh->tetrahedronattributelist[mesh->numberoftetrahedronattributes*i + 0];
        } else {
            genUnstrMesh->element[elementIndex].markerID = defaultVolID;
        }

        status = mesh_allocMeshElementConnectivity(&genUnstrMesh->element[elementIndex]);
        if (status != CAPS_SUCCESS) return status;

        if (i == 0) { // Only need this once
            numPoint = mesh_numMeshElementConnectivity(&genUnstrMesh->element[elementIndex]);
        }

        for (j = 0; j < numPoint; j++ ) {
            genUnstrMesh->element[elementIndex].connectivity[j] = mesh->tetrahedronlist[numPoint*i+j];
        }

        elementIndex += 1;
    }

    return CAPS_SUCCESS;
}

static int
writeBinaryUgrid(void *aimInfo, const char *fileName,
                 const mapAttrToIndexStruct *groupMap,
                 const tetgenRegionsStruct *regions, tetgenio *mesh)
{
  int    status = CAPS_SUCCESS;
  int    i, maxID=0, ID;
  int    nTri, nQuad;
  int    nTet, nPyramid, nPrism, nHex;
  size_t len;
  char   aimFile[PATH_MAX];
  FILE *fp = NULL;

  snprintf(aimFile, PATH_MAX, "%s.lb8.ugrid", fileName);

  fp = fopen(aimFile, "wb");
  if (fp == NULL) {
    AIM_ERROR(aimInfo, "Cannot open file: %s", aimFile);
    status = CAPS_IOERR;
    goto cleanup;
  }

  nTri     = mesh->numberoftrifaces;
  nQuad    = 0;
  nTet     = mesh->numberoftetrahedra;
  nPyramid = 0;
  nPrism   = 0;
  nHex     = 0;

  /* read a binary UGRID file */
  status = fwrite(&mesh->numberofpoints, sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
  status = fwrite(&nTri,     sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
  status = fwrite(&nQuad,    sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
  status = fwrite(&nTet,     sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
  status = fwrite(&nPyramid, sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
  status = fwrite(&nPrism,   sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
  status = fwrite(&nHex,     sizeof(int), 1, fp);
  if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

  /* write all of the vertices */
  status = fwrite(mesh->pointlist, sizeof(double), 3*mesh->numberofpoints, fp);
  if (status != 3*mesh->numberofpoints) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

  /* write the triangle connectivity */
  status = fwrite(mesh->trifacelist, sizeof(int), 3*nTri, fp);
  if (status != 3*nTri) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

  /* write the BC ID */
  status = fwrite(mesh->trifacemarkerlist, sizeof(int), nTri, fp);
  if (status != nTri) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

  /* write the tetrahedral connectivity */
  status = fwrite(mesh->tetrahedronlist, sizeof(int), 4*nTet, fp);
  if (status != 4*nTet) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

  fclose(fp); fp = NULL;

  snprintf(aimFile, PATH_MAX, "%s.mapbc", fileName);
  fp = fopen(aimFile, "w");
  if (fp == NULL) {
    AIM_ERROR(aimInfo, "Cannot open file: %s", aimFile);
    status = CAPS_IOERR;
    goto cleanup;
  }

  fprintf(fp, "%d\n", groupMap->numAttribute);
  for (i = 0; i < groupMap->numAttribute; i++) {
    fprintf(fp, "%d 0 %s\n", groupMap->attributeIndex[i], groupMap->attributeName[i]);
  }

  fclose(fp); fp = NULL;

  /* write out element groups */
  if (regions->size > 0) {
    snprintf(aimFile, PATH_MAX, "%s.mapvol", fileName);

    fp = fopen(aimFile, "wb");
    if (fp == NULL) {
      AIM_ERROR(aimInfo, "Cannot open file: %s", aimFile);
      status = CAPS_IOERR;
      goto cleanup;
    }

    /* write the number of groups */
    status = fwrite(&regions->size, sizeof(int), 1, fp);
    if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

    for (i = 0; i < regions->size; i++) {
      maxID = MAX(maxID, regions->attribute[i]);
    }
    status = fwrite(&maxID, sizeof(int), 1, fp);
    if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

    for (i = 0; i < regions->size; i++) {
      /* write the group IDs */
      status = fwrite(&regions->attribute[i], sizeof(int), 1, fp);
      if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

      /* write the group name */
      len = strlen(regions->names[i])+1;
      status = fwrite(&len, sizeof(size_t), 1, fp);
      if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

      status = fwrite(regions->names[i], sizeof(char), len, fp);
      if (status != (int)len) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
    }

    status = fwrite(&nTet, sizeof(int), 1, fp);
    if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

    /* write the tetrahedral attributes */
    for (i = 0; i < nTet; i++) {
      ID = (int)mesh->tetrahedronattributelist[i];
      status = fwrite(&ID, sizeof(int), 1, fp);
      if (status != 1) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
    }
  }

  status = CAPS_SUCCESS;

cleanup:
  if (fp != NULL) fclose(fp);

  return status;
}


extern "C"
int tetgen_VolumeMesh(void *aimInfo,
                      meshInputStruct meshInput,
                      const mapAttrToIndexStruct *groupMap,
                      const char *fileName,
                      const int numSurfMesh,
                      meshStruct *surfaceMesh,
                      meshStruct *volumeMesh)
{
    // tetgen documentation:
    //
    // -p Generate tetrahedra
    // -Y Preserves the input surface mesh (does not modify it).
    // -V verbose
    // -q mesh quality (maximum radius-edge ratio)/(minimum dihedral angle)
    // -a maximum volume constraint
    // -f provides the interior+boundry triangular faces
    // -nn to get tet neighbors for each triangular face
    // -k dumps to paraview when last argument is NULL
    // -m Applies a mesh sizing function

    // Initialize variables
    int status = 0; //Function status return
    int i, j; // Indexing
    int ID;
    char *inputString = NULL; // Input string to tetgen

    // Create Tetgen input string
    char temp[120] = "p"; // Tetrahedralize a piecewise linear complex flag
    char q[80];
    const char* tetgenDebugSurface = "tetgenDebugSurface";
    char aimFile[PATH_MAX], volFile[PATH_MAX];
    FILE *fp = NULL;

    int i1=1, i2=0;
    tetgenio tmpMesh[2];

    tetgenRegionsStruct regions;
    const tetgenHolesStruct* holes = &meshInput.tetgenInput.holes;

    status = initiate_regions(&regions);
    AIM_STATUS(aimInfo, status);

    status = copy_regions(aimInfo, &meshInput.tetgenInput.regions, &regions);
    AIM_STATUS(aimInfo, status);

    snprintf(volFile, PATH_MAX, "%s.txt", fileName);
    fp = fopen(volFile, "w");
    if (fp == NULL) {
        AIM_ERROR(aimInfo, "Failed to open '%s'!", volFile);
        status = CAPS_IOERR;
        goto cleanup;
    }


    // If no input string is provided create a simple one based on exposed parameters
    if (meshInput.tetgenInput.meshInputString == NULL) {

        if (meshInput.preserveSurfMesh !=0) strcat(temp, "Y"); // Preserve surface mesh flag

        if ((meshInput.tetgenInput.meshQuality_rad_edge != 0) ||
            (meshInput.tetgenInput.meshQuality_angle != 0)) {

            strcat(temp, "q"); // Mesh quality flag

            if (meshInput.tetgenInput.meshQuality_rad_edge >= 0) {

                snprintf(q, 80, "%.3f",meshInput.tetgenInput.meshQuality_rad_edge);
                strcat(temp,q);

            } else if (meshInput.tetgenInput.meshQuality_rad_edge < 0)  {

                printf("Not setting meshQuality radius-edge ratio. Value needs to be positive\n");
            }

            if (meshInput.tetgenInput.meshQuality_angle >= 0) {

                snprintf(q, 80, "/%.3f",meshInput.tetgenInput.meshQuality_angle);
                strcat(temp,q);

            } else if (meshInput.tetgenInput.meshQuality_angle < 0) {

                printf("Not setting meshQuality dihedral angle. Value needs to be positive\n");
            }

        }

        if ((meshInput.quiet != 0) && (meshInput.tetgenInput.verbose == 0)) strcat(temp, "Q"); // Quiet: No terminal output except errors
        if (meshInput.tetgenInput.verbose != 0) strcat(temp, "V"); // Verbose: Detailed information, more terminal output.

        if (meshInput.tetgenInput.meshTolerance > 0) {
            snprintf(q, 80, "T%.2e",meshInput.tetgenInput.meshTolerance);
            strcat(temp,q);
        }

        //if (meshInput.tetgenInput.regions.size > 0
        //    || meshInput.tetgenInput.holes.size > 0) {
            strcat(temp, "A");
        //}

        // Transfer temp character array to input
        inputString = temp;

    } else {

        inputString = meshInput.tetgenInput.meshInputString;
        // Dont think this is needed - maybe p is the default
        // Check to make sure a 'p' is included in the input string provided by a user
        /*if (strchr(input,'p') == NULL) {
            printf("Warning: No \'p\' was found in the Tetgen input string. Input into Tetgen is currently based on PLC's\n");
        }*/
    }

    for (int isurf = 0; isurf < numSurfMesh; isurf++) {

        // swap in the beginning so the final mesh is always in i1
        std::swap(i1,i2);

        // TetGen variables
        tetgenio in, out;
        tetgenio::facet *f;
        tetgenio::polygon *p;

        printf("\nGenerating volume mesh using TetGen.....\n");

        // First index - either 0 or 1
        in.firstnumber = 1;

        // Check inputs
        if (surfaceMesh[isurf].numNode    == 0) { fclose(fp); return CAPS_BADVALUE; }
        if (surfaceMesh[isurf].numElement == 0) { fclose(fp); return CAPS_BADVALUE; }

        // Set the number of surface nodes
        in.numberofpoints = surfaceMesh[isurf].numNode;

        // Create surface point array
        in.pointlist = new REAL[in.numberofpoints *3];

        // Transfer input coordinates to tetgen array
        for(i = 0; i < surfaceMesh[isurf].numNode; i++) {
            in.pointlist[3*i  ] = surfaceMesh[isurf].node[i].xyz[0];
            in.pointlist[3*i+1] = surfaceMesh[isurf].node[i].xyz[1];
            in.pointlist[3*i+2] = surfaceMesh[isurf].node[i].xyz[2];
        }

        // Set the number of surface tri
        in.numberoffacets = surfaceMesh[isurf].numElement;

        // Create surface tri arrays
        in.facetlist = new tetgenio::facet[in.numberoffacets];

        // Create surface marker/BC arrays
        in.facetmarkerlist = new int[in.numberoffacets];

        std::set<int> uniqueMarker;

        // Transfer input surfaceMesh[isurf].localTriFaceList array to tetgen array
        for(i = 0; i < surfaceMesh[isurf].numElement; i++) {
            f = &in.facetlist[i];
            f->numberofpolygons = 1;
            f->polygonlist = new tetgenio::polygon[f->numberofpolygons];
            f->numberofholes = 0;
            f->holelist = NULL;
            p = &f->polygonlist[0];

            p->numberofvertices = mesh_numMeshElementConnectivity(&surfaceMesh[isurf].element[i]);
            p->vertexlist = new int[p->numberofvertices];

            for (j = 0; j < p->numberofvertices; j++) {
                p->vertexlist[j] = surfaceMesh[isurf].element[i].connectivity[j];
            }

            // Transfer input BC array to tetgen array
            in.facetmarkerlist[i] = surfaceMesh[isurf].element[i].markerID;
            uniqueMarker.insert(in.facetmarkerlist[i]);
        }

        if (regions.size > 0)
        {
            in.numberofregions = regions.size;
            in.regionlist = new REAL[5 * regions.size];
            for (int n = 0; n < regions.size; ++n)
            {
                in.regionlist[5 * n + 0] = regions.x[n];
                in.regionlist[5 * n + 1] = regions.y[n];
                in.regionlist[5 * n + 2] = regions.z[n];
                in.regionlist[5 * n + 3] = (REAL) regions.attribute[n];
                in.regionlist[5 * n + 4] = regions.volume_constraint[n];
            }
        }

        /*==============================================================*/

        //Create an "empty mesh" where only surface nodes are connected to create the volume
        //Tet centers of the empty mesh will be used to identify holes
        //
        // -p Generate tetrahedra
        // -Y Preserves the input surface mesh (does not modify it).
        tetgenio emptymesh;
        try {
            tetrahedralize((char*)"pYQ", &in, &emptymesh);
        } catch (...){
            aim_file(aimInfo, tetgenDebugSurface, aimFile);
            mesh_writeTecplot(aimInfo,tetgenDebugSurface, 1, surfaceMesh, 1.0);
            AIM_ERROR  (aimInfo, "Tetgen failed to generate an empty volume mesh......!!!");
            AIM_ADDLINE(aimInfo, "  See Tecplot file %s.dat for the surface mesh", aimFile);
            return CAPS_EXECERR;
        }

        std::vector<REAL> holepoints;

        // Only solid bodies can have holes
        for ( std::set<int>::const_iterator marker = uniqueMarker.begin(); marker != uniqueMarker.end(); marker++ )
        {
            int globaltri = 0;
            while ( globaltri < in.numberoffacets && in.facetmarkerlist[globaltri] != *marker ) globaltri++;
            tetgenio::facet *f = in.facetlist + globaltri;
            tetgenio::polygon *p = f->polygonlist;

            // Look for two tets attached to a polygon
            int tet = 0;
            int twotets[2][4] = {{-1, -1, -1, -1}, {-1, -1, -1, -1}};
            for ( int k = 0; k < emptymesh.numberoftetrahedra; k++ )
            {
                bool match[3] = {false, false, false};
                for (int v0 = 0; v0 < 3; v0++)
                {
                    for (int n0 = 0; n0 < 4; n0++)
                    {
                        if ( p->vertexlist[v0] == emptymesh.tetrahedronlist[4*k+n0] )
                        {
                            match[v0] = true;
                            break;
                        }
                    }
                    //Found a match, save it of. There should be two tets for a face with holes
                    if ( match[0] && match[1] && match[2] )
                    {
                        for (int n0 = 0; n0 < 4; n0++)
                            twotets[tet][n0] = emptymesh.tetrahedronlist[4*k+n0];
                        tet++;

                    }
                }
                if ( tet == 2 )
                    break;
            }

            if ( tet == 2 )
            {
                //Found two tets, the one with a postive normal vector to the cell center is a hole

                //Compute the center of the polygon on the surface
                REAL polycenter[3] = {0, 0, 0};
                REAL polynormal[3] = {0, 0, 0};
                REAL polyedge0[3] = {0, 0, 0};
                REAL polyedge1[3] = {0, 0, 0};
                for (int n = 0; n < 3; n++)
                {
                    for (int v = 0; v < 3; v++)
                        polycenter[n] += in.pointlist[(p->vertexlist[v]-1)*3 + n];
                    polycenter[n] /= 3;

                    polyedge0[n] = in.pointlist[(p->vertexlist[1]-1)*3 + n] - in.pointlist[(p->vertexlist[0]-1)*3 + n];
                    polyedge1[n] = in.pointlist[(p->vertexlist[2]-1)*3 + n] - in.pointlist[(p->vertexlist[0]-1)*3 + n];
                }
                CROSS(polynormal, polyedge0, polyedge1);

                //Compute the center of the two tetrahedra
                REAL tetcenters[2][3] = {{0, 0, 0}, {0, 0, 0}};
                for (tet = 0; tet < 2; tet++)
                    for (int n = 0; n < 3; n++)
                    {
                        for (int v = 0; v < 4; v++)
                            tetcenters[tet][n] += emptymesh.pointlist[(twotets[tet][v]-1)*3+n];
                        tetcenters[tet][n] /= 4;
                    }

                REAL diffc[3] = {0, 0, 0};
                //Compute the dot product between the normal vector and the vector to the tet-center
                for (tet = 0; tet < 2; tet++)
                {
                    for (int n = 0; n < 3; n++)
                        diffc[n] = tetcenters[tet][n] - polycenter[n];

                    //Positive dot product means hole
                    if ( DOT(diffc, polynormal) > 0 )
                    {
                        for (int n = 0; n < 3; n++)
                            holepoints.push_back(tetcenters[tet][n]);
                    }
                }
            }
        }

        if (numSurfMesh > 1 && emptymesh.numberoftetrahedra > 1) {

            ego body;
            int state, np;

            int atype, len;
            const int *ints;
            const double *reals;
            const char *str;
            char tmpName[42];

            status = EG_statusTessBody(surfaceMesh[isurf].egadsTess, &body, &state, &np);
            AIM_STATUS(aimInfo, status);

            status = EG_attributeRet(body, "_name", &atype, &len, &ints, &reals, &str);
            if (status != CAPS_SUCCESS) {
              snprintf(tmpName, 42, "Body %d", isurf+1);
              str = tmpName;
            }

            //Compute the center of the one tetrahedra
            REAL tetcenter[3] = {0, 0, 0};
            for (int n = 0; n < 3; n++)
            {
                for (int v = 0; v < 4; v++)
                    tetcenter[n] += emptymesh.pointlist[(emptymesh.tetrahedronlist[v]-1)*3+n];
                tetcenter[n] /= 4;
            }

            int maxAttr = 0;
            for (int i = 0; i < regions.size; i++)
                maxAttr = MAX(regions.attribute[i], maxAttr);
            maxAttr++;

            AIM_REALL(regions.names, regions.size+1, char*, aimInfo, status);
            regions.names[regions.size] = NULL;
            AIM_REALL(regions.x, regions.size+1, REAL, aimInfo, status);
            AIM_REALL(regions.y, regions.size+1, REAL, aimInfo, status);
            AIM_REALL(regions.z, regions.size+1, REAL, aimInfo, status);
            AIM_REALL(regions.attribute, regions.size+1, int, aimInfo, status);
            AIM_REALL(regions.volume_constraint, regions.size+1, REAL, aimInfo, status);

            AIM_STRDUP(regions.names[regions.size], str, aimInfo, status);
            regions.x[regions.size] = tetcenter[0];
            regions.y[regions.size] = tetcenter[1];
            regions.z[regions.size] = tetcenter[2];
            regions.attribute[regions.size] = maxAttr;
            regions.volume_constraint[regions.size] = -1;

            regions.size++;
        }

        if (regions.size > 0)
        {
            delete [] in.regionlist;
            in.numberofregions = regions.size;
            in.regionlist = new REAL[5 * regions.size];
            for (int n = 0; n < regions.size; ++n)
            {
                in.regionlist[5 * n + 0] = regions.x[n];
                in.regionlist[5 * n + 1] = regions.y[n];
                in.regionlist[5 * n + 2] = regions.z[n];
                in.regionlist[5 * n + 3] = (REAL) regions.attribute[n];
                in.regionlist[5 * n + 4] = regions.volume_constraint[n];
            }
        }

        if (regions.size > 0 || holes->size > 0)
        {
          in.numberofholes = holes->size;
          if (in.holelist != NULL)
          {
            delete [] in.holelist;
            in.holelist = NULL;
          }
          if (holes->size > 0)
          {
            in.holelist = new REAL[3 * holes->size];
            for (int n = 0; n < holes->size; ++n)
            {
              in.holelist[3 * n + 0] = holes->x[n];
              in.holelist[3 * n + 1] = holes->y[n];
              in.holelist[3 * n + 2] = holes->z[n];
            }
          }
        }
        else
        {
          if ( holepoints.size() > 0 )
          {
              in.numberofholes = holepoints.size()/3;
              in.holelist = new REAL[holepoints.size()];
              for ( std::size_t n = 0; n < holepoints.size(); n++ )
                  in.holelist[n] = holepoints[n];
          }
        }

        /*==============================================================*/


        printf("\nTetgen input string = %s""\n", inputString);

        // Create volume mesh
        try {
            tetrahedralize(inputString, &in, &out);
        } catch (...){
            aim_file(aimInfo, tetgenDebugSurface, aimFile);
            mesh_writeTecplot(aimInfo,tetgenDebugSurface, 1, surfaceMesh, 1.0);
            AIM_ERROR  (aimInfo, "Tetgen failed to generate a volume mesh......!!!");
            AIM_ADDLINE(aimInfo, "  See Tecplot file %s.dat for the surface mesh", aimFile);
            return CAPS_EXECERR;
        }

        // Save data
        //in.save_nodes((char *)"TETGEN_Test");
        //in.save_poly((char *) "TETGEN_Test");
        //out.save_faces((char *) "TETGEN_Test");

        if (out.numberoftetrahedra == 0) {
            fclose(fp);
            aim_file(aimInfo, tetgenDebugSurface, aimFile);
            mesh_writeTecplot(aimInfo,tetgenDebugSurface, 1, surfaceMesh, 1.0);
            AIM_ERROR  (aimInfo, "Tetgen failed to generate a volume mesh......!!!");
            AIM_ADDLINE(aimInfo, "  See Tecplot file %s.dat for the surface mesh", aimFile);
            return CAPS_EXECERR;
        }

        fprintf(fp, "%d\n", out.numberofpoints);

        if (numSurfMesh == 1) {
            tmpMesh[i1] = out;
            out.initialize();
        } else {

            // append points
            tmpMesh[i1].pointlist = new REAL[(tmpMesh[i2].numberofpoints + out.numberofpoints) * 3];

            for (i = 0; i < tmpMesh[i2].numberofpoints*3; i++)
              tmpMesh[i1].pointlist[i] = tmpMesh[i2].pointlist[i];

            for (i = 0; i < out.numberofpoints*3; i++)
              tmpMesh[i1].pointlist[tmpMesh[i2].numberofpoints*3 + i] = out.pointlist[i];

            tmpMesh[i1].numberofpoints = tmpMesh[i2].numberofpoints + out.numberofpoints;

            // append triface
            tmpMesh[i1].trifacelist = new int[(tmpMesh[i2].numberoftrifaces + out.numberoftrifaces) * 3];

            for (i = 0; i < tmpMesh[i2].numberoftrifaces*3; i++)
              tmpMesh[i1].trifacelist[i] = tmpMesh[i2].trifacelist[i];

            for (i = 0; i < out.numberoftrifaces*3; i++)
              tmpMesh[i1].trifacelist[tmpMesh[i2].numberoftrifaces*3 + i] = out.trifacelist[i] + tmpMesh[i2].numberofpoints;

            // append trifacemarker
            tmpMesh[i1].trifacemarkerlist = new int[tmpMesh[i2].numberoftrifaces + out.numberoftrifaces];

            for (i = 0; i < tmpMesh[i2].numberoftrifaces; i++)
              tmpMesh[i1].trifacemarkerlist[i] = tmpMesh[i2].trifacemarkerlist[i];

            for (i = 0; i < out.numberoftrifaces; i++)
              tmpMesh[i1].trifacemarkerlist[tmpMesh[i2].numberoftrifaces + i] = out.trifacemarkerlist[i];

            tmpMesh[i1].numberoftrifaces = tmpMesh[i2].numberoftrifaces + out.numberoftrifaces;

            // append tetrahedronlist
            tmpMesh[i1].tetrahedronlist = new int[(tmpMesh[i2].numberoftetrahedra + out.numberoftetrahedra)*4];

            for (i = 0; i < tmpMesh[i2].numberoftetrahedra*4; i++)
              tmpMesh[i1].tetrahedronlist[i] = tmpMesh[i2].tetrahedronlist[i];

            for (i = 0; i < out.numberoftetrahedra*4; i++)
              tmpMesh[i1].tetrahedronlist[tmpMesh[i2].numberoftetrahedra*4 + i] = out.tetrahedronlist[i] + tmpMesh[i2].numberofpoints;

            // append tetrahedronattributelist
            if (out.numberoftetrahedronattributes == 1) {
              tmpMesh[i1].tetrahedronattributelist = new REAL[tmpMesh[i2].numberoftetrahedra + out.numberoftetrahedra];

              for (i = 0; i < tmpMesh[i2].numberoftetrahedra; i++)
                tmpMesh[i1].tetrahedronattributelist[i] = tmpMesh[i2].tetrahedronattributelist[i];

              for (i = 0; i < out.numberoftetrahedra; i++)
                tmpMesh[i1].tetrahedronattributelist[tmpMesh[i2].numberoftetrahedra + i] = out.tetrahedronattributelist[i];
            }

            tmpMesh[i1].numberoftetrahedronattributes = out.numberoftetrahedronattributes;

            tmpMesh[i1].numberoftetrahedra = tmpMesh[i2].numberoftetrahedra + out.numberoftetrahedra;

            tmpMesh[i2].clean_memory();
            tmpMesh[i2].initialize();
        }
    }

    /* check if tetgen made new regions */
    ID = -1;
    if (tmpMesh[i1].numberoftetrahedra > 0) {
      ID = tmpMesh[i1].tetrahedronattributelist[0];
      for (j = 0; j < regions.size; j++) {
        if (ID == regions.attribute[j]) break;
      }
      if (j == regions.size) {
        snprintf(temp, 120, "region_%d",ID);
        status = add_regions(aimInfo, temp, 0,0,0, ID, 0, &regions);
        AIM_STATUS(aimInfo, status);
      }
    }
    for (i = 0; i < tmpMesh[i1].numberoftetrahedra; i++) {
      if (ID == (int)tmpMesh[i1].tetrahedronattributelist[i]) continue;

      for (j = 0; j < regions.size; j++) {
        if (ID == regions.attribute[j]) break;
      }
      if (j == regions.size) {
        snprintf(temp, 120, "region_%d",ID);
        status = add_regions(aimInfo, temp, 0,0,0, ID, 0, &regions);
        AIM_STATUS(aimInfo, status);
      }
      ID = (int)tmpMesh[i1].tetrahedronattributelist[i];
    }

    // Transfer tetgen mesh structure to genUnstrMesh format
    status = tetgen_to_MeshStruct(&tmpMesh[i1], volumeMesh);
    AIM_STATUS(aimInfo, status);

    status = writeBinaryUgrid(aimInfo, fileName, groupMap, &regions, &tmpMesh[i1]);
    AIM_STATUS(aimInfo, status);

    status = CAPS_SUCCESS;
    printf("Done meshing using TetGen!\n");
cleanup:
    if (fp != NULL) fclose(fp);
    destroy_regions(&regions);

    return status;
}
