/*
 ************************************************************************
 *                                                                      *
 * udfMatchBodys -- finds matching Nodes, Edges, and Faces              *
 *                                                                      *
 *            Written by John Dannenhoffer@ Syracuse University         *
 *                                                                      *
 ************************************************************************
 */

/*
 * Copyright (C) 2013/2025  John F. Dannenhoffer, III (Syracuse University)
 *
 * This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *     MA  02110-1301  USA
 */

#define NUMUDPINPUTBODYS 2
#define NUMUDPARGS       6

/* set up the necessary structures (uses NUMUDPARGS) */
#include "udpUtilities.h"

/* shorthands for accessing argument values and velocities */
#define TOLER(   IUDP)     ((double *) (udps[IUDP].arg[0].val))[0]
#define ATTR(    IUDP)     ((char   *) (udps[IUDP].arg[1].val))
#define MAKEMAPS(IUDP)     ((int    *) (udps[IUDP].arg[2].val))[0]
#define NNODES(  IUDP)     ((int    *) (udps[IUDP].arg[3].val))[0]
#define NEDGES(  IUDP)     ((int    *) (udps[IUDP].arg[4].val))[0]
#define NFACES(  IUDP)     ((int    *) (udps[IUDP].arg[5].val))[0]

static char  *argNames[NUMUDPARGS] = {"toler",  "attr",     "makemaps", "nnodes", "nedges", "nfaces", };
static int    argTypes[NUMUDPARGS] = {ATTRREAL, ATTRSTRING, ATTRINT,    -ATTRINT, -ATTRINT, -ATTRINT, };
static int    argIdefs[NUMUDPARGS] = {0,        0,          0,          0,        0,        0,        };
static double argDdefs[NUMUDPARGS] = {0.,       0.,         0.,         0.,       0.,       0.,       };

/* get utility routines: udpErrorStr, udpInitialize, udpReset, udpSet,
                         udpGet, udpVel, udpClean, udpMesh */
#include "udpUtilities.c"

#include "OpenCSM.h"

static int oldMethod(ego ebody0, ego ebody1, int *NumUdp, udp_T *udps);
static int newMethod(ego ebody0, ego ebody1, int *NumUdp, udp_T *udps);


/*
 ************************************************************************
 *                                                                      *
 *   udpExecute - execute the primitive                                 *
 *                                                                      *
 ************************************************************************
 */

int
udpExecute(ego  emodel,                 /* (in)  input model */
           ego  *ebody,                 /* (out) Body pointer */
           int  *nMesh,                 /* (out) number of associated meshes */
           char *string[])              /* (out) error message */
{
    int     status = EGADS_SUCCESS;

    ego     context, *ebodys;

    int     oclass, mtype;
    int     nchild, *senses;
    double  data[18];
    ego     eref;
    udp_T   *udps = *Udps;

    ROUTINE(udpExecute);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    printf("udpExecute(emodel=%llx)\n", (long long)emodel);
    printf("toler(   0) = %f\n", TOLER(   0));
    printf("attr(    0) = %s\n", ATTR(    0));
    printf("makemaps(0) = %d\n", MAKEMAPS(0));
#endif

    /* default return values */
    *ebody  = NULL;
    *nMesh  = 0;
    *string = NULL;

    /* check/process arguments */
    if (udps[0].arg[0].size > 1) {
        printf(" udpExecute: toler should be a scalar\n");
        status  = EGADS_RANGERR;
        goto cleanup;

    } else if (TOLER(0) < 0) {
        printf(" udpExecute: toler = %f < 0\n", TOLER(0));
        status  =  EGADS_RANGERR;
        goto cleanup;
    }

    /* check that Model was input that contains two Bodys */
    status = EG_getTopology(emodel, &eref, &oclass, &mtype,
                            data, &nchild, &ebodys, &senses);
    CHECK_STATUS(EG_getTopology);

    if (oclass != MODEL) {
        printf(" udpExecute: expecting a Model\n");
        status = EGADS_NOTMODEL;
        goto cleanup;
    } else if (nchild != 2) {
        printf(" udpExecute: expecting Model to contain one Body (not %d)\n", nchild);
        status = EGADS_NOTBODY;
        goto cleanup;
    }

#ifdef DEBUG
    printf("emodel\n");
    ocsmPrintEgo(emodel);
#endif

    /* cache copy of arguments for future use */
    status = cacheUdp(emodel);
    CHECK_STATUS(cacheUdp);

#ifdef DEBUG
    printf("toler(   %d) = %f\n", numUdp, TOLER(   numUdp));
    printf("attr(    %d) = %s\n", numUdp, ATTR(    numUdp));
    printf("makemaps(%d) = %d\n", numUdp, MAKEMAPS(numUdp));
#endif

    /* default output value(s) */
    NNODES(numUdp) = 0;
    NEDGES(numUdp) = 0;
    NFACES(numUdp) = 0;

    status = EG_getContext(emodel, &context);
    CHECK_STATUS(EG_getContext);

    /* make a copy of the input emodel */
    status = EG_copyObject(emodel, NULL, ebody);
    CHECK_STATUS(EG_copyObject);

    SPLINT_CHECK_FOR_NULL(*ebody);

    status = EG_getTopology(*ebody, &eref, &oclass, &mtype,
                            data, &nchild, &ebodys, &senses);
    CHECK_STATUS(EG_getTopology);

    if (MAKEMAPS(0) == 0) {
        status = oldMethod(ebodys[0], ebodys[1], NumUdp, udps);
        CHECK_STATUS(oldMethod);
    } else {
        status = newMethod(ebodys[0], ebodys[1], NumUdp, udps);
        CHECK_STATUS(newMethod);
    }

    /* return the modfied Model that contains the two input Bodys */

#ifdef DEBUG
    printf("*ebody\n");
    (void) ocsmPrintEgo(*ebody);
#endif

    /* return output value(s) --- done above */

    /* remember this model (Body) */
    udps[numUdp].ebody = *ebody;
    goto cleanup;

#ifdef DEBUG
    printf("udpExecute -> *ebody=%llx\n", (long long)(*ebody));
#endif

cleanup:
    if (status != EGADS_SUCCESS) {
        *string = udpErrorStr(status);
    }

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   oldMethod - use old-style Attributes                               *
 *                                                                      *
 ************************************************************************
 */

static int
oldMethod(ego    ebody0,                /* (in)  first  Body */
          ego    ebody1,                /* (in)  second Body */
          int    *NumUdp,
          udp_T  *udps)
{
    int     status = EGADS_SUCCESS;

    int     oclass, mtype, inode0, inode1, nnode0, nnode1, nedge0, nedge1, nface0, nface1;
    int     nchild, *senses, attrType, attrLen, ibest;
    int     nmatch, i, *matches=NULL, *list0=NULL, *list1=NULL;
    CINT    *tempIlist;
    double  xyz1[4], xyz2[4], toler1, toler2, dx, dy, dz, tbest;
    CDOUBLE *tempRlist;
    CCHAR   *tempClist;
    ego     *enode0, *enode1, *eedge0, *eedge1, *eface0, *eface1, eref, *echilds;

    ROUTINE(oldMethod);

    /* --------------------------------------------------------------- */

    /* get a list of the Nodes in each Body */
    status = EG_getBodyTopos(ebody0, NULL, NODE, &nnode0, &enode0);
    CHECK_STATUS(EG_getBodyTopos);

    status = EG_getBodyTopos(ebody1, NULL, NODE, &nnode1, &enode1);
    CHECK_STATUS(EG_getBodyTopos);

    /* make an array to hold the possible Node matches */
    MALLOC(list0, int, nnode0);
    MALLOC(list1, int, nnode1);

    /* get tolerances for the Bodys */
    status = EG_getTolerance(ebody0, &toler1);
    CHECK_STATUS(EG_getTolerance);

    status = EG_getTolerance(ebody1, &toler2);
    CHECK_STATUS(EG_getTolerance);

    /* find Node matches */
    nmatch = 0;

    for (inode0 = 0; inode0 < nnode0; inode0++) {
        status = EG_getTopology(enode0[inode0], &eref, &oclass, &mtype,
                                xyz1, &nchild, &echilds, &senses);
        CHECK_STATUS(EG_getTopology);

        if (TOLER(numUdp) > 0) {
            tbest = TOLER(numUdp);
            ibest = -1;
        } else {
            tbest = MAX(toler1, toler2);
            ibest = -1;
        }

        for (inode1 = 0; inode1 < nnode1; inode1++) {
            status = EG_getTopology(enode1[inode1], &eref, &oclass, &mtype,
                                    xyz2, &nchild, &echilds, &senses);
            CHECK_STATUS(EG_getTopology);

            dx = fabs(xyz1[0] - xyz2[0]);
            dy = fabs(xyz1[1] - xyz2[1]);
            dz = fabs(xyz1[2] - xyz2[2]);

            if (dx < tbest && dy < tbest && dz < tbest) {
                tbest = MAX(MAX(dx, dy), dz);
                ibest = inode1;
            }
        }

        if (ibest >= 0) {
            list0[nmatch] = inode0 + 1;
            list1[nmatch] = ibest  + 1;
            nmatch++;

            if (STRLEN(ATTR(numUdp)) > 0) {
                status = EG_attributeRet(enode0[inode0], ATTR(numUdp), &attrType, &attrLen,
                                         &tempIlist, &tempRlist, &tempClist);
                if (status == EGADS_SUCCESS) {
                    printf("copying \"%s\" from inode0=%d to inode1=%d\n", ATTR(numUdp), inode0, inode1);
                    status = EG_attributeAdd(enode1[inode1], ATTR(numUdp), attrType, attrLen,
                                             tempIlist, tempRlist, tempClist);
                    CHECK_STATUS(EG_attributeAdd);
                }
                status = EGADS_SUCCESS;
            }
        }
    }

    EG_free(enode0);
    EG_free(enode1);

    /* add Attributes to the two Bodys that identify the matches */
    if (nmatch > 0) {
        NNODES(numUdp) = nmatch;

        status = EG_attributeAdd(ebody0, "_nodeMatches_", ATTRINT,
                                 nmatch, list0, NULL, NULL);
        CHECK_STATUS(EG_attributeAdd);

        status = EG_attributeAdd(ebody1, "_nodeMatches_", ATTRINT,
                                 nmatch, list1, NULL, NULL);
        CHECK_STATUS(EG_attributeAdd);
    }

    FREE(list0);
    FREE(list1);

    /* find the Edge matches */
    if (nmatch > 0) {
        status = EG_matchBodyEdges(ebody0, ebody1, TOLER(numUdp), &nmatch, &matches);
        CHECK_STATUS(EG_matchBodyEdges);

        if (nmatch > 0) {
            if (matches == NULL) goto cleanup;    // needed for splint

            status = EG_getBodyTopos(ebody0, NULL, EDGE, &nedge0, &eedge0);
            CHECK_STATUS(EG_getBodyTopos);

            status = EG_getBodyTopos(ebody1, NULL, EDGE, &nedge1, &eedge1);
            CHECK_STATUS(EG_getBodyTopos);

            MALLOC(list0, int, nmatch);
            MALLOC(list1, int, nmatch);

            for (i = 0; i < nmatch; i++) {
                list0[i] = matches[2*i  ];
                list1[i] = matches[2*i+1];

                if (STRLEN(ATTR(numUdp)) > 0) {
                    status = EG_attributeRet(eedge0[list0[i]-1], ATTR(numUdp), &attrType, &attrLen,
                                             &tempIlist, &tempRlist, &tempClist);
                    if (status == EGADS_SUCCESS) {
                        printf("copying \"%s\" from iedge0=%d to iedge1=%d\n", ATTR(numUdp), list0[i], list1[i]);
                        status = EG_attributeAdd(eedge1[list1[i]-1], ATTR(numUdp), attrType, attrLen,
                                                 tempIlist, tempRlist, tempClist);
                        CHECK_STATUS(EG_attributeAdd);
                    }
                }
            }

            NEDGES(numUdp) = nmatch;

            status = EG_attributeAdd(ebody0, "_edgeMatches_", ATTRINT,
                                     nmatch, list0, NULL, NULL);
            CHECK_STATUS(EG_attributeAdd);

            status = EG_attributeAdd(ebody1, "_edgeMatches_", ATTRINT,
                                     nmatch, list1, NULL, NULL);
            CHECK_STATUS(EG_attributeAdd);

            EG_free(eedge0);
            EG_free(eedge1);
            EG_free(matches);
            FREE(list0);
            FREE(list1);
        }
    }

    /* find the Face matches */
    if (nmatch > 0) {
        status = EG_matchBodyFaces(ebody0, ebody1, TOLER(numUdp), &nmatch, &matches);
        CHECK_STATUS(EG_matchBodyFaces);

        if (nmatch > 0) {
            if (matches == NULL) goto cleanup;    // needed for splint

            status = EG_getBodyTopos(ebody0, NULL, FACE, &nface0, &eface0);
            CHECK_STATUS(EG_getBodyTopos);

            status = EG_getBodyTopos(ebody1, NULL, FACE, &nface1, &eface1);
            CHECK_STATUS(EG_getBodyTopos);

            MALLOC(list0, int, nmatch);
            MALLOC(list1, int, nmatch);

            for (i = 0; i < nmatch; i++) {
                list0[i] = matches[2*i  ];
                list1[i] = matches[2*i+1];

                if (STRLEN(ATTR(numUdp)) > 0) {
                    status = EG_attributeRet(eface0[list0[i]-1], ATTR(numUdp), &attrType, &attrLen,
                                             &tempIlist, &tempRlist, &tempClist);
                    if (status == EGADS_SUCCESS) {
                        printf("copying \"%s\" from iface0=%d to iface1=%d\n", ATTR(numUdp), list0[i], list1[i]);
                        status = EG_attributeAdd(eface1[list1[i]-1], ATTR(numUdp), attrType, attrLen,
                                                 tempIlist, tempRlist, tempClist);
                        CHECK_STATUS(EG_attributeAdd);
                    }
                }
            }

            NFACES(numUdp) = nmatch;

            status = EG_attributeAdd(ebody0, "_faceMatches_", ATTRINT,
                                     nmatch, list0, NULL, NULL);
            CHECK_STATUS(EG_attributeAdd);

            status = EG_attributeAdd(ebody1, "_faceMatches_", ATTRINT,
                                     nmatch, list1, NULL, NULL);
            CHECK_STATUS(EG_attributeAdd);

            EG_free(eface0);
            EG_free(eface1);
            EG_free(matches);
            FREE(list0);
            FREE(list1);
        }
    }

cleanup:
    FREE(list0);
    FREE(list1);

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   newMethod - use new-style Attributes                               *
 *                                                                      *
 ************************************************************************
 */

static int
newMethod(ego    ebody0,                /* (in)  first  Body */
          ego    ebody1,                /* (in)  second Body */
          int    *NumUdp,
          udp_T  *udps)
{
    int     status = EGADS_SUCCESS;

    int     oclass, mtype;
    int     inode0, inode1, nnode0, nnode1;
    int     iedge0, iedge1, nedge0, nedge1;
    int     iface0, iface1, nface0, nface1;
    int     nchild, *senses, imatch, nmatch, *matches=NULL;
    int     *map0=NULL, *map1=NULL;
    double  toler, xyz0[18], xyz1[18];
    ego     *enode0=NULL, *enode1=NULL, eref, *echilds;

    ROUTINE(newMethod);

    /* --------------------------------------------------------------- */

    /* get a list of the Nodes in each Body */
    status = EG_getBodyTopos(ebody0, NULL, NODE, &nnode0, &enode0);
    CHECK_STATUS(EG_getBodyTopos);

    status = EG_getBodyTopos(ebody1, NULL, NODE, &nnode1, &enode1);
    CHECK_STATUS(EG_getBodyTopos);

    if (TOLER(numUdp) != 0) {
        toler = TOLER(numUdp);
    } else {
        toler = EPS06;
    }

    /* make an array to hold the possible Node matches */
    MALLOC(map0, int, nnode0);
    MALLOC(map1, int, nnode1);

    /* find a Node in ebody1 that matches each Node in ebody0 */
    for (inode0 = 0; inode0 < nnode0; inode0++) {
        SPLINT_CHECK_FOR_NULL(enode0);

        map0[inode0] = 0;

        status = EG_getTopology(enode0[inode0], &eref, &oclass, &mtype,
                                xyz0, &nchild, &echilds, &senses);
        CHECK_STATUS(EG_getTopology);

        for (inode1 = 0; inode1 < nnode1; inode1++) {
            SPLINT_CHECK_FOR_NULL(enode1);

            status = EG_getTopology(enode1[inode1], &eref, &oclass, &mtype,
                                    xyz1, &nchild, &echilds, &senses);
            CHECK_STATUS(EG_getTopology);

            if (fabs(xyz0[0]-xyz1[0]) < toler &&
                fabs(xyz0[1]-xyz1[1]) < toler &&
                fabs(xyz0[2]-xyz1[2]) < toler   ) {
                map0[inode0] = inode1 + 1;
                break;
            }
        }
    }

    /* find a Node in ebody0 that matches each Node in ebody1 */
    for (inode1 = 0; inode1 < nnode1; inode1++) {
        SPLINT_CHECK_FOR_NULL(enode1);

        map1[inode1] = 0;

        status = EG_getTopology(enode1[inode1], &eref, &oclass, &mtype,
                                xyz1, &nchild, &echilds, &senses);
        CHECK_STATUS(EG_getTopology);

        for (inode0 = 0; inode0 < nnode0; inode0++) {
            SPLINT_CHECK_FOR_NULL(enode0);

            status = EG_getTopology(enode0[inode0], &eref, &oclass, &mtype,
                                    xyz0, &nchild, &echilds, &senses);
            CHECK_STATUS(EG_getTopology);

            if (fabs(xyz0[0]-xyz1[0]) < toler &&
                fabs(xyz0[1]-xyz1[1]) < toler &&
                fabs(xyz0[2]-xyz1[2]) < toler   ) {
                map1[inode1] = inode0 + 1;
                break;
            }
        }
    }

    /* add the Node maps to the two Bodys */
    status = EG_attributeAdd(ebody0, "_nodeMap_", ATTRINT, nnode0, map0, NULL, NULL);
    CHECK_STATUS(EG_attributeAdd);

    status = EG_attributeAdd(ebody1, "_nodeMap_", ATTRINT, nnode1, map1, NULL, NULL);
    CHECK_STATUS(EG_attributeAdd);

    FREE(map0);
    FREE(map1);

    /* --------------------------------------------------------------- */

    /* get a list of the Edges in each Body */
    status = EG_getBodyTopos(ebody0, NULL, EDGE, &nedge0, NULL);
    CHECK_STATUS(EG_getBodyTopos);

    status = EG_getBodyTopos(ebody1, NULL, EDGE, &nedge1, NULL);
    CHECK_STATUS(EG_getBodyTopos);

    /* make an array to hold the possible Edge matches */
    MALLOC(map0, int, nedge0);
    MALLOC(map1, int, nedge1);

    for (iedge0 = 0; iedge0 < nedge0; iedge0++) {
        map0[iedge0] = 0;
    }
    for (iedge1 = 0; iedge1 < nedge1; iedge1++) {
        map1[iedge1] = 0;
    }

    status = EG_matchBodyEdges(ebody0, ebody1, 0.0, &nmatch, &matches);
    CHECK_STATUS(EG_matchBodyEdges);

    for (imatch = 0; imatch < nmatch; imatch++) {
        SPLINT_CHECK_FOR_NULL(matches);

        map0[matches[2*imatch  ]-1] = matches[2*imatch+1];
        map1[matches[2*imatch+1]-1] = matches[2*imatch  ];
    }

    if (matches != NULL) {
        EG_free(matches);
        matches = NULL;
    }

    /* add the Edge maps to the two Bodys */
    status = EG_attributeAdd(ebody0, "_edgeMap_", ATTRINT, nedge0, map0, NULL, NULL);
    CHECK_STATUS(EG_attributeAdd);

    status = EG_attributeAdd(ebody1, "_edgeMap_", ATTRINT, nedge1, map1, NULL, NULL);
    CHECK_STATUS(EG_attributeAdd);

    FREE(map0);
    FREE(map1);

    /* --------------------------------------------------------------- */

    /* get a list of the Faces in each Body */
    status = EG_getBodyTopos(ebody0, NULL, FACE, &nface0, NULL);
    CHECK_STATUS(EG_getBodyTopos);

    status = EG_getBodyTopos(ebody1, NULL, FACE, &nface1, NULL);
    CHECK_STATUS(EG_getBodyTopos);

    /* make an array to hold the possible Face matches */
    MALLOC(map0, int, nface0);
    MALLOC(map1, int, nface1);

    for (iface0 = 0; iface0 < nface0; iface0++) {
        map0[iface0] = 0;
    }
    for (iface1 = 0; iface1 < nface1; iface1++) {
        map1[iface1] = 0;
    }

    status = EG_matchBodyFaces(ebody0, ebody1, 0.0, &nmatch, &matches);
    CHECK_STATUS(EG_matchBodyFaces);

    for (imatch = 0; imatch < nmatch; imatch++) {
        SPLINT_CHECK_FOR_NULL(matches);

        map0[matches[2*imatch  ]-1] = matches[2*imatch+1];
        map1[matches[2*imatch+1]-1] = matches[2*imatch  ];
    }

    if (matches != NULL) {
        EG_free(matches);
        matches = NULL;
    }

    /* add the Face maps to the two Bodys */
    status = EG_attributeAdd(ebody0, "_faceMap_", ATTRINT, nface0, map0, NULL, NULL);
    CHECK_STATUS(EG_attributeAdd);

    status = EG_attributeAdd(ebody1, "_faceMap_", ATTRINT, nface1, map1, NULL, NULL);
    CHECK_STATUS(EG_attributeAdd);

    FREE(map0);
    FREE(map1);

cleanup:
    FREE(map0);
    FREE(map1);

    if (enode0 != NULL) EG_free(enode0);
    if (enode1 != NULL) EG_free(enode1);

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   udpSensitivity - return sensitivity derivatives for the "real" argument *
 *                                                                      *
 ************************************************************************
 */

int
udpSensitivity(ego    ebody,            /* (in)  Body pointer */
   /*@unused@*/int    npnt,             /* (in)  number of points */
   /*@unused@*/int    entType,          /* (in)  OCSM entity type */
   /*@unused@*/int    entIndex,         /* (in)  OCSM entity index (bias-1) */
   /*@unused@*/double uvs[],            /* (in)  parametric coordinates for evaluation */
   /*@unused@*/double vels[])           /* (out) velocities */
{
    int iudp, judp;

    ROUTINE(udpSensitivity);

    /* --------------------------------------------------------------- */

    /* check that ebody matches one of the ebodys */
    iudp = 0;
    for (judp = 1; judp <= numUdp; judp++) {
        if (ebody == udps[judp].ebody) {
            iudp = judp;
            break;
        }
    }
    if (iudp <= 0) {
        return EGADS_NOTMODEL;
    }

    /* this routine is not written yet */
    return EGADS_NOLOAD;
}
