# Import pyCAPS and os module
## [import]
import pyCAPS

import os
## [import]

import argparse
# Setup and read command line options. Please note that this isn't required for pyCAPS
parser = argparse.ArgumentParser(description = 'FlightStream Glider Pytest Example',
                                 prog = 'flightstream_Glider_PyTest.py',
                                 formatter_class = argparse.ArgumentDefaultsHelpFormatter)

#Setup the available commandline options
parser.add_argument('-workDir', default = "." + os.sep, nargs=1, type=str, help = 'Set working/run directory')
parser.add_argument("-outLevel", default = 1, type=int, choices=[0, 1, 2], help="Set output verbosity")
args = parser.parse_args()

# Define units
deg  = pyCAPS.Unit("degree")
kg   = pyCAPS.Unit("kg")
m    = pyCAPS.Unit("meter")
ft   = pyCAPS.Unit("feet")
s    = pyCAPS.Unit("s")
Pa   = pyCAPS.Unit("Pa")
mile = pyCAPS.Unit("mile")
hour = pyCAPS.Unit("hour")
K    = pyCAPS.Unit("Kelvin")

# Create working directory variable
workDir = os.path.join(str(args.workDir[0]), "FlightStreamGliderSweepTest")

# Load the geometry
geometryScript = os.path.join("..","csmData","flightstreamGlider.csm")
capsProblem = pyCAPS.Problem(problemName=workDir,
                             capsFile=geometryScript,
                             outLevel=args.outLevel)

# Load desired aim
flightstream = capsProblem.analysis.create(aim = "flightstreamAIM",
                                           name = "flightstream",
                                           # Specify the FligthStream unit system (optional)
                                           unitSystem = {"mass": kg, "length" : m, "time" : s, "temperature" : K})

# Specify the possibly full PATH to FlightStream executable (optional)
flightstream.input.FlightStream = "FlightStream"

# Set the solver model
flightstream.input.Solver_Model = "SUBSONIC_PRANDTL_GLAUERT"

# Specify the capsGroup general surfaces
flightstream.input.CCS_GeneralSurface = {"Fuselage": {}}

# Specify the capsGroup lifting surfaces
flightstream.input.CCS_LiftingSurface = {"Wing": {}, "Htail": {}, "Vtail":{}}

# Set control surface parameters
AileronLeft = {"deflectionAngle" : 25.0 * deg, "slotGap" : 0.005} # Large angle: no-wake termination nodes
AileronRite = {"deflectionAngle" : -5.0 * deg, "slotGap" : 0.005} # Small angle: wake termination nodes

flightstream.input.CCS_Control = {"AileronLeft": AileronLeft,
                                  "AileronRite": AileronRite,
                                  "StabilizerLeft" : {"deflectionAngle":15*deg, "slotGap" : 0.005},
                                  "StabilizerRite" : {"deflectionAngle":15*deg, "slotGap" : 0.005},
                                  "Rudder"     : {"deflectionAngle":-10*deg, "slotGap" : 0.005}}

# Specify the capsGroup bodies of revolution
flightstream.input.CCS_Revolution = {"NacelleR": {}}

# Specify the capsGroup annular bodies
flightstream.input.CCS_Annular = {"NacelleL": {}}

# Set flow conditions
flightstream.input.SweepRangeMach  = [0.5, 0.7, 0.1]       #start, end, increment
flightstream.input.SweepRangeAlpha = [1.0, 3.0, 1.0] * deg #start, end, increment
flightstream.input.Beta            = 0.0 * deg
flightstream.input.Altitude        = 1000 * ft

# Reference velocity
flightstream.input.ReferenceVelocity = 375 * mile/hour

# Export surface data in Tecplot and VTK file formats
flightstream.input.Export_Solver_Analysis = ["Tecplot", "VTK"]

# Set the number of solver iterations (too few for convergence)
flightstream.input.SolverIterations = 10

# Explicitly run analysis (optional)
flightstream.runAnalysis()

# Get all ouputs
for i in flightstream.output.keys():
    print(str(i) + " = " + str(flightstream.output[i].value))
