/*
 *      CAPS: Computational Aircraft Prototype Syntheses
 *
 *             HI-Mach AIM
 *
 *     Written by Marshall Galbraith (MIT)
 *
 *      Copyright 2014-2026, Massachusetts Institute of Technology
 *      Licensed under The GNU Lesser General Public License, version 2.1
 *      See http://www.opensource.org/licenses/lgpl-2.1.php
 */

/*!\mainpage Introduction
 * \tableofcontents
 * \section overviewHIMACH FlightStream AIM Overview

 * A module in the Computational Aircraft Prototype Syntheses (CAPS)
 * has been developed to interact (primarily through input files) with
 * Research in Flight's FlightStream \cite FlightStream. FlightStream predicts
 * the aerodynamic performance of a vehicle via a panel methid, which
 * makes it very fast.
 *
 * An outline of the AIM's inputs and outputs are provided in \ref
 * aimInputsHIMACH and \ref aimOutputsHIMACH, respectively.
 *
 * Geometric attributes recognized by the AIM are provided in \ref
 * attributeHIMACH.
 *
 * The accepted and expected geometric representation are detailed in
 * \ref geomRepIntentHIMACH.
 *
 */
 /*! \page attributeHIMACH Attribution
 *
 * The following list of attributes drives the FlightStream geometric definition.
 *
 *  - <b> capsLength</b> This attribute defines the length units that
 *  the *.csm file is generated in and is not optional for FligtStream.
 *  The FlightStream input grid will be scaled to either the default length of METER
 *  or the user specified length unit (see \ref aimUnitsHIMACH).
 *
 *  - <b> capsReferenceChord</b> and <b> capsReferenceSpan</b>
 * [Optional] These attributes may exist on any <em> Body</em>. Their
 * value will be used as the reference moment lengths in FlightStream's
 * input file with their units assumed to be consistent with the
 * attribute "capsLength". These values may be alternatively set through an input
 * value, "ReferenceChord" (see \ref aimInputsHIMACH)
 *
 *  - <b> capsReferenceArea</b> [Optional] This attribute may exist on
 * any <em> Body</em>.  Its value will be used as the reference area
 * in FlightStream's input file with its units assumed to be consistent with
 * the attribute "capsLength". This value may be alternatively set
 * through an input value, "ReferenceArea" (see \ref aimInputsHIMACH)
 *
 *  - <b> capsReferenceX</b>, <b> capsReferenceY</b>,  and <b> capsReferenceZ</b>
 * [Optional] These attribute may exist on any <em> Body</em>. Their
 * value will be used as the reference moment lengths in FlightStream's
 * input file with their units assumed to be consistent with the
 * attribute "capsLength". These values may be alternatively set through an input
 * value, "ReferenceX" (see \ref aimInputsHIMACH)
 *

 *
 */

//#define DEBUG

#include <string.h>
#include <math.h>

#include "capsTypes.h"
#include "aimUtil.h"

#include "miscUtils.h"
#include "meshUtils.h"
#include "cfdUtils.h"

#include "vtkWriter.h"

#ifdef WIN32
   #define snprintf    _snprintf
   #define strcasecmp  stricmp
   #define strncasecmp _strnicmp
   #define strtok_r    strtok_s
#else
   #include <unistd.h>
   #include <limits.h>
#endif

/* define the indicies of the inputs and outputs */
enum aimInputs
{
    inProj_Name = 1,             /* index is 1-biased */
    inHIMach,
    inMach,
    inAlpha,
    inBeta,
    //inFreestream_Direction,
    inGamma,
    inNose_Axis,
    inPitch_Axis,
    inReferenceChord,
    inReferenceSpan,
    inReferenceArea,
    inReferenceX,
    inReferenceY,
    inReferenceZ,
   // inReferenceLength,
   // inReferenceVelocity,
    inPressure_Scale_Factor,
    inWindward_Method,
    inLeeward_Method,
    inBase_Pressure,
    inShielding_Effects,
    inDesign_Variable,
    inMesh_Morph,
    inSurface_Mesh,
    NUMINPUT = inSurface_Mesh    /* Total number of inputs */
};

enum aimOutputs
{
    outCx = 1,                   /* index is 1-based */
    outCy,
    outCz,
    outCL,
    outCD,
    outCMx,
    outCMy,
    outCMz,
    NUMOUTPUT = outCMz           /* Total number of outputs */
};

#define MXCHAR  255

#define ANALYSIS_VTK

#ifdef ANALYSIS_PLOAD_BDF
static const char* loadbdf = "surface_load.bdf";
#elif defined(ANALYSIS_VTK)
static const char* cpvtk = "surface_cp.vtk";
#elif defined(ANALYSIS_CSV)
static const char* dpcsv = "surface_dp.csv";
#else
#error "must define ANALYSIS_PLOAD_BDF, ANALYSIS_VTK, or ANALYSIS_CSV"
#endif

typedef struct {

  char input[PATH_MAX];
  char result[PATH_MAX];
  char report[PATH_MAX];
  char sens[PATH_MAX];

  // variables that could be obtained via attributes on the Body
  double Cref;       // capsReferenceChord
  double Bref;       // capsReferenceSpan
  double Sref;       // capsReferenceArea
  double Xref;       // capsReferenceX   (cg)
  double Yref;       // capsReferenceY   (cg)
  double Zref;       // capsReferenceZ   (cg)

  // outputs from HI-Mach
  capsValue Cx;         // X-force coefficient
  capsValue Cy;         // Y-force coefficient
  capsValue Cz;         // Z-force coefficient
  capsValue CL;         // lift coefficient
  capsValue CD;         // drag coefficient
  capsValue CMx;        // X-moment coefficient
  capsValue CMy;        // Y-moment coefficient
  capsValue CMz;        // Z-moment coefficient

  // Attribute to index map
  mapAttrToIndexStruct groupMap;

  // Design information
  cfdDesignStruct design;

  // Mesh reference obtained from meshing AIM
  const aimMeshRef *meshRefIn;
  aimMeshRef meshRefMorph, meshRefDisp;

} aimStorage;


static int initialize_aimStorage(aimStorage *himachInstance)
{
  // Set initial values for himachInstance
  himachInstance->Cref = 0;
  himachInstance->Bref = 0;
  himachInstance->Sref = 0;
  himachInstance->Xref = 0;
  himachInstance->Yref = 0;
  himachInstance->Zref = 0;

  aim_initValue(&himachInstance->Cx);
  aim_initValue(&himachInstance->Cy);
  aim_initValue(&himachInstance->Cz);
  aim_initValue(&himachInstance->CL);
  aim_initValue(&himachInstance->CD);
  aim_initValue(&himachInstance->CMx);
  aim_initValue(&himachInstance->CMy);
  aim_initValue(&himachInstance->CMz);

  // Container for attribute to index map
  (void)initiate_mapAttrToIndexStruct(&himachInstance->groupMap);

  // Design information
  initiate_cfdDesignStruct(&himachInstance->design);

  himachInstance->meshRefIn = NULL;
  aim_initMeshRef(&himachInstance->meshRefMorph, aimUnknownMeshType);
  aim_initMeshRef(&himachInstance->meshRefDisp, aimUnknownMeshType);

  return CAPS_SUCCESS;
}


static int destroy_aimStorage(aimStorage *himachInstance)
{
  aim_freeValue(&himachInstance->Cx);
  aim_freeValue(&himachInstance->Cy);
  aim_freeValue(&himachInstance->Cz);
  aim_freeValue(&himachInstance->CL);
  aim_freeValue(&himachInstance->CD);
  aim_freeValue(&himachInstance->CMx);
  aim_freeValue(&himachInstance->CMy);
  aim_freeValue(&himachInstance->CMz);

  // Attribute to index map
  (void) destroy_mapAttrToIndexStruct(&himachInstance->groupMap);

  // Design information
  (void) destroy_cfdDesignStruct(&himachInstance->design);

  // Surface Mesh
  aim_freeMeshRef(&himachInstance->meshRefMorph);
  aim_freeMeshRef(&himachInstance->meshRefDisp);
  himachInstance->meshRefIn = NULL;

  initialize_aimStorage(himachInstance);

  return CAPS_SUCCESS;
}


// Get FlightStream data transfer files
static int
fs_dataTransfer(void *aimInfo,
                capsValue  aimInputs[],
                const aimStorage *fsInstance)
{

    /*! \page dataTransferHIMACH FlightStream Data Transfer
     *
     * \section dataToCart3D Data transfer to FlightStream (FieldIn)
     *
     * <ul>
     *  <li> <B>"Displacement"</B> </li> <br>
     *   Retrieves nodal displacements (as from a structural solver)
     *   and updates FlightStream surface mesh.
     * </ul>
     */

    int status; // Function return status
    int i, ibound, iglobal; // Indexing

    // Discrete data transfer variables
    capsDiscr *discr;
    char **boundNames = NULL;
    int numBoundName = 0;
    enum capsdMethod dataTransferMethod;
    int numDataTransferPoint;
    int dataTransferRank;
    double *dataTransferData;
    char *units;

    const aimMeshRef *meshRefIn;

    aimMesh    mesh;

    // Data transfer variable
    double *dxyz = NULL;

    int foundDisplacement = (int) false;

    mesh.meshRef = NULL;
    mesh.meshData = NULL;

    status = aim_getBounds(aimInfo, &numBoundName, &boundNames);
    if (status == CAPS_NOTFOUND) return CAPS_SUCCESS;
    AIM_STATUS(aimInfo, status);

    // Get mesh
    meshRefIn = (aimMeshRef *) aimInputs[inSurface_Mesh-1].vals.AIMptr;

    if ( aimInputs[inMesh_Morph-1].vals.integer == (int) true &&
        meshRefIn == NULL) { // If we are mighty morphing
      meshRefIn = &fsInstance->meshRefMorph;
    }

    foundDisplacement = (int) false;
    for (ibound = 0; ibound < numBoundName; ibound++) {
      AIM_NOTNULL(boundNames, aimInfo, status);

      status = aim_getDiscr(aimInfo, boundNames[ibound], &discr);
      if (status != CAPS_SUCCESS) continue;

      status = aim_getDataSet(discr,
                              "Displacement",
                              &dataTransferMethod,
                              &numDataTransferPoint,
                              &dataTransferRank,
                              &dataTransferData,
                              &units);
      if (status != CAPS_SUCCESS) continue;

      foundDisplacement = (int) true;

      // Is the rank correct?
      if (dataTransferRank != 3) {
        AIM_ERROR(aimInfo, "Displacement transfer data found however rank is %d not 3!!!!", dataTransferRank);
        status = CAPS_BADRANK;
        goto cleanup;
      }

    } // Loop through bound names

    if (foundDisplacement != (int) true ) {
        status = CAPS_SUCCESS;
        goto cleanup;
    }

    status = aim_freeMeshRef(&((aimStorage *)fsInstance)->meshRefDisp);
    AIM_STATUS(aimInfo, status);

    /* Create a local mesh file name */
    status = aim_localMeshRef(aimInfo, meshRefIn, &((aimStorage *)fsInstance)->meshRefDisp);
    AIM_STATUS(aimInfo, status);

    /*@-immediatetrans@*/
    mesh.meshData = NULL;
    mesh.meshRef = &((aimStorage *)fsInstance)->meshRefDisp;
    /*@+immediatetrans@*/

    status = mesh_surfaceMeshData(aimInfo, &fsInstance->groupMap, &mesh);
    AIM_STATUS(aimInfo, status);
    AIM_NOTNULL(mesh.meshData, aimInfo, status);

    /*@-immediatetrans@*/
    ((aimStorage *)fsInstance)->meshRefIn = &((aimStorage *)fsInstance)->meshRefDisp;
    /*@+immediatetrans@*/

    AIM_ALLOC(dxyz, 3*mesh.meshData->nVertex, double, aimInfo, status);
    memset(dxyz, 0, 3*mesh.meshData->nVertex*sizeof(double));

    // gather displacements first to avoid double counting edges/nodes
    for (ibound = 0; ibound < numBoundName; ibound++) {
      AIM_NOTNULL(boundNames, aimInfo, status);

      status = aim_getDiscr(aimInfo, boundNames[ibound], &discr);
      if (status != CAPS_SUCCESS) continue;

      status = aim_getDataSet(discr,
                              "Displacement",
                              &dataTransferMethod,
                              &numDataTransferPoint,
                              &dataTransferRank,
                              &dataTransferData,
                              &units);
      if (status != CAPS_SUCCESS) continue;

      if (numDataTransferPoint != discr->nPoints &&
          numDataTransferPoint > 1) {
        AIM_ERROR(aimInfo, "Developer error!! %d != %d", numDataTransferPoint, discr->nPoints);
        status = CAPS_MISMATCH;
        goto cleanup;
      }

      for (i = 0; i < discr->nPoints; i++) {
        iglobal = discr->tessGlobal[2*i+1];

        if (numDataTransferPoint == 1) {
          // A single point means this is an initialization phase
          status = aim_convert(aimInfo, 3, units, dataTransferData, units, &dxyz[3*(iglobal-1)+0]);
          AIM_STATUS(aimInfo, status);
        } else {
          // Apply delta displacements
          status = aim_convert(aimInfo, 3, units, &dataTransferData[3*i+0], units, &dxyz[3*(iglobal-1)+0]);
          AIM_STATUS(aimInfo, status);
        }
      }
    } // Loop through bound names

    // Apply the displacements
    for (i = 0; i < mesh.meshData->nVertex; i++) {
      mesh.meshData->verts[i][0] += dxyz[3*i+0];
      mesh.meshData->verts[i][1] += dxyz[3*i+1];
      mesh.meshData->verts[i][2] += dxyz[3*i+2];
    }

    /* write the mesh */
    status = aim_writeMesh(aimInfo, MESHWRITER, NULL, &mesh);
    AIM_STATUS(aimInfo, status);

    status = CAPS_SUCCESS;

// Clean-up
cleanup:
    AIM_FREE(dxyz);
    AIM_FREE(boundNames);

    aim_freeMeshData(mesh.meshData);
    AIM_FREE(mesh.meshData);

    return status;
}

/**********************************************************************/
/* aimInitialize - initialize the AIM                                 */
/**********************************************************************/

int
aimInitialize(int        qFlag,         /* (in)  -1 indiactes query only */
 /*@unused@*/ const char unitSys[],     /* (in)  unit system */
              void       *aimInfo,      /* (in)  AIM context */
              void       **instStore,   /* (out) AIM instance storage */
              int        *major,        /* (out) major version number */
              int        *minor,        /* (out) minor version number */
              int        *nIn,          /* (out) number of inputs to AIM */
              int        *nOut,         /* (out) number of outputs from AIM */
              int        *nFields,      /* (out) number of DataSet fields */
              char       ***fnames,     /* (out) array  of DataSet names */
              int        **franks,      /* (out) array  of DataSet ranks */
              int        **fInOut)      /* (out) array  of field flags */
{

    int status = CAPS_SUCCESS;

    int  *ints=NULL, i;
    char **strs=NULL;

    aimStorage *himachInstance = NULL;

    /* -------------------------------------------------------------- */

#ifdef DEBUG
    printf("himachAIM/aimInitialize(qFlag=%d, unitSys=%s)\n", qFlag, unitSys);
#endif

    /* specify the version number */
    *major = 1;
    *minor = 0;

    /* specify the number of analysis input and out "parameters" */
    *nIn     = NUMINPUT;
    *nOut    = NUMOUTPUT;

    /* if this is simply a query, return now */
    if (qFlag == -1) {
        return CAPS_SUCCESS;
    }

    /*! \page geomRepIntentHIMACH Geometry Representation The geometric
     * representation for the FlightStream AIM requires that the body be
     * either a solid body (SOLIDBODY) or a manifold sheet body
     * (SHEETBODY).
     */

    /* specify the field variables this analysis can generate and consume */
    *nFields = 2;

    /* specify the name of each field variable */
    AIM_ALLOC(strs, *nFields, char *, aimInfo, status);
    strs[0]  = EG_strdup("Pressure");
    strs[1]  = EG_strdup("Displacement");
    for (i = 0; i < *nFields; i++)
      if (strs[i] == NULL) {
        status = EGADS_MALLOC;
        goto cleanup;
      }
    *fnames  = strs;

    /* specify the dimension of each field variable */
    AIM_ALLOC(ints, *nFields, int, aimInfo, status);

    ints[0]  = 1;
    ints[1]  = 3;
    *franks   = ints;
    ints = NULL;

    /* specify if a field is an input field or output field */
    AIM_ALLOC(ints, *nFields, int, aimInfo, status);

    ints[0]  = FieldOut;
    ints[1]  = FieldIn;
    *fInOut  = ints;
    ints = NULL;

    // Allocate himachInstance
    AIM_ALLOC(himachInstance, 1, aimStorage, aimInfo, status);

    initialize_aimStorage(himachInstance);

    *instStore = himachInstance;

cleanup:
  return status;
}


/**********************************************************************/
/* aimInputs - return information about index'th analysis input       */
/**********************************************************************/

int
aimInputs(
/*@unused@*/void    *instStore,         /* (in)  AIM instance storage */
          void      *aimInfo,           /* (in)  AIM context */
          int       index,              /* (in)  input index (1-nIn] */
          char      **ainame,           /* (out) name of analysis input */
          capsValue *inval)             /* (out) description of analysis input */
{
    int status = CAPS_SUCCESS;

    /* -------------------------------------------------------------- */

#ifdef DEBUG
    printf("himachAIM/aimInputs(index=%d)\n", index);
#endif

    *ainame = NULL;

    /*! \page aimInputsHIMACH AIM Inputs
     * The following list outlines the FlightStream inputs along with their
     * default values available through the AIM interface.
     */

    // entities to be defined for each aimInput
    // default values are in [brackets]

    // *ainame             = EG_strdup("name")
    // inval->type         = Boolean, Integer, Double, String, Tuple, Pointer, DoubleDeriv, or PointerMesh
    // inval->nullVal      = [NotNull], IsNull, NotAllowed, IsPartial
    // inval->units        = [NULL] or EG_strdup("bar")
    // inval->lfixed       = [Fixed] or Change (length is fixed)
    // inval->sfixed       = [Fixed] or Change (shape  is fixed)
    // inval->dim          = [0], 1, or 2 (maximum dimensions)
    // inval->nrow         = [1] or n (only if dim>0)
    // inval->ncol         = [1] or n (only if dim>1)
    // inval->length       = inval->nrow * inval->ncol
    // inval->units        = NULL or EG_strdup("meter") or EG_strdup("degree") or ...
    // inval->vals.real    = number or AIM_ALLOC(inval->vals.reals, inval->length, double, aimInfo, status)
    // inval->vals.integer = number or AIM_ALLOC(inval->vals.integer, inval->length, int, aimInfo, status)
    // inval->vals.string  = NULL or EG_strdup("himach_CAPS")

    // FlightStream Inputs
    if (index == inProj_Name) {
        *ainame             = EG_strdup("Proj_Name");
        inval->type         = String;
        inval->nullVal      = NotNull;
        inval->lfixed       = Change;
        AIM_STRDUP(inval->vals.string, "himach_CAPS", aimInfo, status);

        /*! \page aimInputsHIMACH
         * - <B> ProjName = "himach_CAPS"</B> <br>
         * Name for files generated by HI-Mach AIM.
         */
    } else if (index == inHIMach) {
        *ainame             = EG_strdup("HIMach");
        inval->type         = String;
        inval->nullVal      = NotNull;
        inval->lfixed       = Change;
        AIM_STRDUP(inval->vals.string, "HIMach", aimInfo, status);

        /*! \page aimInputsHIMACH
         * - <B> HIMach = "himach.exe"</B> <br>
         * The name of the HI-Mach executable. May include full path.
         */
    } else if (index == inMach) {
        *ainame             = EG_strdup("Mach"); // Mach number
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->units        = NULL;
        inval->lfixed       = Fixed;
        inval->sfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.real    = 0.0;

        /*! \page aimInputsHIMACH
         * - <B> Mach = NULL </B> <br>
         *  Mach number
         */

    } else if (index == inAlpha) {
        *ainame             = EG_strdup("Alpha");
        inval->type         = Double;
        inval->nullVal      = NotAllowed;
        //inval->units        = EG_strdup("degree");
        inval->lfixed       = Fixed;
        inval->sfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.real    = 0.0;

        /*! \page aimInputsHIMACH
         * - <B> Alpha = 0.0 (default) </B> <br>
         *  Angle of attack [degree]
         */

    } else if (index == inBeta) {
        *ainame             = EG_strdup("Beta");
        inval->type         = Double;
        inval->nullVal      = NotAllowed;
        //inval->units        = EG_strdup("degree");
        inval->lfixed       = Fixed;
        inval->sfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.real    = 0.0;

        /*! \page aimInputsHIMACH
         * - <B> Beta = 0.0 (default) </B> <br>
         *  Sideslip angle
         */

#if 0
    } else if (index == inFreestream_Direction) {
        *ainame             = EG_strdup("Freestream_Direction");
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->lfixed       = Fixed;
        inval->sfixed       = Fixed;
        inval->dim          = Vector;
        inval->nrow         = 3;
        AIM_ALLOC(inval->vals.reals, inval->nrow, double, aimInfo, status);
        inval->vals.reals[0] = 1.0;
        inval->vals.reals[1] = 0.0;
        inval->vals.reals[2] = 0.0;

        /*! \page aimInputsHIMACH
         * - <B> Freestream_Direction = [1.0, 0.0, 0.0] (default) </B> <br>
         *  Freestream direction vector
         */
#endif

    } else if (index == inGamma) {
        *ainame             = EG_strdup("gamma");
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->lfixed       = Fixed;
        inval->sfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.real    = 1.4;

        /*! \page aimInputsHIMACH
         * - <B> gamma = 1.4 (default) </B> <br>
         *  Specific heat ratio
         */

    } else if (index == inNose_Axis) {
        *ainame             = EG_strdup("Nose_Axis");
        inval->type         = String;
        inval->nullVal      = NotAllowed;
        inval->lfixed       = Fixed;
        inval->sfixed       = Fixed;
        inval->dim          = Scalar;
        AIM_STRDUP(inval->vals.string, "x-", aimInfo, status);

        /*! \page aimInputsHIMACH
         * - <B> Nose_Axis = "x-" (default) </B> <br>
         *  Geometric nose axis direction. Must be one of: "x-", "x+", "y-", "y+", "z-", "z+"
         */

    } else if (index == inPitch_Axis) {
        *ainame             = EG_strdup("Pitch_Axis");
        inval->type         = String;
        inval->nullVal      = NotAllowed;
        inval->lfixed       = Fixed;
        inval->sfixed       = Fixed;
        inval->dim          = Scalar;
        AIM_STRDUP(inval->vals.string, "z-", aimInfo, status);

        /*! \page aimInputsHIMACH
         * - <B> Pitch_Axis = "z-" (default) </B> <br>
         *  Geometric pitch axis direction. Must be one of: "x-", "x+", "y-", "y+", "z-", "z+"
         */

    } else if (index == inReferenceChord) {
        *ainame             = EG_strdup("ReferenceChord");
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->lfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.real    = 0.0;

        /*! \page aimInputsHIMACH
         * - <B>ReferenceChord = NULL </B> <br>
         * This sets the reference chord for used in force and moment
         *  calculations.  Alternatively, the geometry (body) attribute
         *  (see \ref attributeHIMACH) "capsReferenceChord" maybe used to
         *  specify this variable (note: values set through the AIM input
         *  will supersede the attribution value).
         */

    } else if (index == inReferenceSpan) {
        *ainame             = EG_strdup("ReferenceSpan");
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->lfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.real    = 0.0;

        /*! \page aimInputsHIMACH
         * - <B>ReferenceSpan = NULL </B> <br>
         * This sets the reference span for used in force and moment
         *  calculations.  Alternatively, the geometry (body) attribute
         *  (see \ref attributeHIMACH) "capsReferenceSpan" maybe used to
         *  specify this variable (note: values set through the AIM input
         *  will supersede the attribution value).
         */

    } else if (index == inReferenceArea) {
        *ainame             = EG_strdup("ReferenceArea");
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->lfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.real    = 0.0;

        /*! \page aimInputsHIMACH
         * - <B>ReferenceArea = NULL </B> <br>
         * This sets the reference area for used in force and moment
         *  calculations.  Alternatively, the geometry (body) attribute
         *  (see \ref attributeHIMACH) "capsReferenceArea" maybe used to
         *  specify this variable (note: values set through the AIM input
         *  will supersede the attribution value).
         */

    } else if (index == inReferenceX) {
        *ainame             = EG_strdup("ReferenceX");
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->lfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.real    = 0.0;

        /*! \page aimInputsHIMACH
         * - <B>ReferenceX = NULL </B> <br>
         * This sets the reference X for moment calculations.
         * Alternatively, the geometry (body) attribute
         *  (see \ref attributeHIMACH) "capsReferenceX" maybe used to
         *  specify this variable (note: values set through the AIM input
         *  will supersede the attribution value).
         */

    } else if (index == inReferenceY) {
        *ainame             = EG_strdup("ReferenceY");
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->lfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.real    = 0.0;

        /*! \page aimInputsHIMACH
         * - <B>ReferenceX = NULL </B> <br>
         * This sets the reference Y for moment calculations.
         * Alternatively, the geometry (body) attribute
         *  (see \ref attributeHIMACH) "capsReferenceY" maybe used to
         *  specify this variable (note: values set through the AIM input
         *  will supersede the attribution value).
         */

    } else if (index == inReferenceZ) {
        *ainame             = EG_strdup("ReferenceZ");
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->lfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.real    = 0.0;

        /*! \page aimInputsHIMACH
         * - <B>ReferenceX = NULL </B> <br>
         * This sets the reference Z for moment calculations.
         * Alternatively, the geometry (body) attribute
         *  (see \ref attributeHIMACH) "capsReferenceZ" maybe used to
         *  specify this variable (note: values set through the AIM input
         *  will supersede the attribution value).
         */

//    } else if (index == inReferenceVelocity) {
//        *ainame             = EG_strdup("ReferenceVelocity");
//        inval->type         = Double;
//        inval->nullVal      = IsNull;
//        inval->lfixed       = Fixed;
//        inval->dim          = Scalar;
//        inval->vals.real    = 0.0;
//
//        /*! \page aimInputsHIMACH
//         * - <B>ReferenceVelocity = NULL </B> <br>
//         * This sets the reference velocity
//         */

    } else if (index == inPressure_Scale_Factor) {
        *ainame             = EG_strdup("Pressure_Scale_Factor");
        inval->type         = Double;
        inval->vals.real    = 1.0;

        /*! \page aimInputsCART3D
         * - <B>Pressure_Scale_Factor = 1.0</B> <br>
         * Value to scale Pressure data when transferring data. Data is scaled based on Pressure = Pressure_Scale_Factor*Pressure.
         */

    } else if (index == inWindward_Method) {
        *ainame             = EG_strdup("Windward_Method");
        inval->type         = String;
        inval->nullVal      = IsNull;
        inval->lfixed       = Fixed;
        inval->dim          = Scalar;

        /*! \page aimInputsHIMACH
         * - <B>Windward_Method = "modified-newtonian" (default) </B> <br>
         * Solver windward method
         */

    } else if (index == inLeeward_Method) {
        *ainame             = EG_strdup("Leeward_Method");
        inval->type         = String;
        inval->nullVal      = IsNull;
        inval->lfixed       = Fixed;
        inval->dim          = Scalar;

        /*! \page aimInputsHIMACH
         * - <B>Windward_Method = NULL </B> <br>
         * Solver leeward method
         */

    } else if (index == inBase_Pressure) {
        *ainame             = EG_strdup("Base_Pressure");
        inval->type         = String;
        inval->nullVal      = IsNull;
        inval->lfixed       = Fixed;
        inval->dim          = Scalar;

        /*! \page aimInputsHIMACH
         * - <B>Base_Pressure = NULL </B> <br>
         * Solver base pressure method
         */

    } else if (index == inShielding_Effects) {
        *ainame             = EG_strdup("Shielding_Effects");
        inval->type         = Boolean;
        inval->dim          = Scalar;
        inval->vals.integer = (int)false;

        /*! \page aimInputsHIMACH
         * - <B>Shielding_Efects = false (default)</B> <br>
         * Enable shielding effects
         */

    } else if (index == inDesign_Variable) {
        *ainame             = EG_strdup("Design_Variable");
        inval->type         = Tuple;
        inval->nullVal      = IsNull;
        inval->lfixed       = Change;
        inval->vals.tuple   = NULL;
        inval->dim          = Vector;

        /*! \page aimInputsHIMACH
         * - <B> Design_Variable = NULL</B> <br>
         * List of AnalysisIn and/or GeometryIn variable names used to compute sensitivities for optimization, see \ref cfdDesignVariable for additional details.
         */

    } else if (index == inMesh_Morph) {
        *ainame              = EG_strdup("Mesh_Morph");
        inval->type          = Boolean;
        inval->lfixed        = Fixed;
        inval->vals.integer  = (int) false;
        inval->dim           = Scalar;
        inval->nullVal       = NotNull;

        /*! \page aimInputsHIMACH
         * - <B> Mesh_Morph = False</B> <br>
         * Project previous surface mesh onto new geometry.
         */

    } else if (index == inSurface_Mesh) {
        *ainame             = EG_strdup("Surface_Mesh");
        inval->type         = PointerMesh;
        inval->dim          = Vector;
        inval->lfixed       = Change;
        inval->sfixed       = Change;
        inval->vals.AIMptr  = NULL;
        inval->nullVal      = IsNull;
        AIM_STRDUP(inval->meshWriter, MESHWRITER, aimInfo, status);

        /*! \page aimInputsHIMACH
         * - <B>Surface_Mesh = NULL</B> <br>
         * A Surface_Mesh link.
         */

    } else {
        AIM_ERROR(aimInfo, "Unknown input index $%d", index);
        status = CAPS_RANGEERR;
        goto cleanup;
    }

    AIM_NOTNULL(*ainame, aimInfo, status);

cleanup:
    if (status != CAPS_SUCCESS) {
        AIM_FREE(*ainame);
    }

    return status;
}


/**********************************************************************/
/* aimOutputs - return information about index'th analysis outputs    */
/**********************************************************************/

int
aimOutputs(/*@unused@*/ void *instStore,    /* (in)  AIM instance storage */
           /*@unused@*/ void *aimInfo,      /* (in)  AIM context */
           int index,                       /* (in)  output index (1-nOut) */
           char **aoname,                   /* (out) name of analysis output */
           capsValue *outval)               /* (out) description of analysis output */
{
    int status = CAPS_SUCCESS;

    /*! \page aimOutputsHIMACH AIM Outputs
     * The following list outlines the FlightStream outputs available through
     * the AIM interface. All variables currently correspond to values
     * found in the *.plt file
     */

    /* -------------------------------------------------------------- */

#ifdef DEBUG
    printf("himachAIM/aimOutputs(index=%d)\n", index);
#endif

    outval->type       = Double;
    outval->lfixed     = Fixed;
    outval->sfixed     = Fixed;
    outval->units      = NULL;
    outval->dim        = 0;
    outval->length     = 1;
    outval->nrow       = 1;
    outval->ncol       = 1;
    outval->vals.real  = 0.0;
    outval->vals.reals = NULL;

    if        (index == outCx) {
        *aoname = EG_strdup("Cx");
    } else if (index == outCy) {
        *aoname = EG_strdup("Cy");
    } else if (index == outCz) {
        *aoname = EG_strdup("Cz");
    } else if (index == outCL) {
        *aoname = EG_strdup("CL");
    } else if (index == outCD) {
        *aoname = EG_strdup("CD");
    } else if (index == outCMx) {
        *aoname = EG_strdup("CMx");
    } else if (index == outCMy) {
        *aoname = EG_strdup("CMy");
    } else if (index == outCMz) {
        *aoname = EG_strdup("CMz");
    } else {
        printf("himachAIM/aimOutputs index = %d NOT Found\n", index);
        status = CAPS_NOTFOUND;
        goto cleanup;
    }

    /*! \page aimOutputsHIMACH
     * Aerodynamic coefficients:
     * - <B>Cx</B> = X-force coefficient
     * - <B>Cx</B> = X-force coefficient
     * - <B>Cx</B> = X-force coefficient
     * - <B>CL</B> = Lift coefficient
     * - <B>CD</B> = drag coefficient
     * - <B>CMx</B> = X-moment coefficient
     * - <B>CMy</B> = Y-moment coefficient
     * - <B>CMz</B> = Z-moment coefficient
     */

    AIM_NOTNULL(*aoname, aimInfo, status);

cleanup:
    return status;
}


/**********************************************************************/
/* aimUpdateState - update the AIM's internal state                   */
/**********************************************************************/

int
aimUpdateState(void      *instStore,    /* (in)  AIM instance storage */
               void      *aimInfo,      /* (in)  AIM context */
               capsValue aimInputs[])   /* (in)  array of analysis inputs */
{
    // Function return flag
    int status = CAPS_SUCCESS;

    int i;
    int foundSref=(int)false;
    int foundCref=(int)false;
    int foundBref=(int)false;
    int foundXref=(int)false;
    int foundYref=(int)false;
    int foundZref=(int)false;

    // AIM input bodies
    const char *intent;
    int  numBody;
    ego *bodies = NULL;

    // EGADS return values
    int          atype, alen;
    const int    *ints;
    const char   *string;
    const double *reals;

    aimStorage *himachInstance = (aimStorage *)instStore;

    /* -------------------------------------------------------------- */

#ifdef DEBUG
    printf("himachAIM/aimUpdateState()\n");
#endif

    AIM_NOTNULL(aimInputs, aimInfo, status);

    aim_freeValue(&himachInstance->Cx);
    aim_freeValue(&himachInstance->Cy);
    aim_freeValue(&himachInstance->Cz);
    aim_freeValue(&himachInstance->CL);
    aim_freeValue(&himachInstance->CD);
    aim_freeValue(&himachInstance->CMx);
    aim_freeValue(&himachInstance->CMy);
    aim_freeValue(&himachInstance->CMz);

    if (aimInputs[inSurface_Mesh-1].nullVal == IsNull &&
        aimInputs[inMesh_Morph-1].vals.integer == (int) false) {
      AIM_ANALYSISIN_ERROR(aimInfo, inSurface_Mesh, "'Surface_Mesh' input must be linked to an output 'Surface_Mesh'");
      status = CAPS_BADVALUE;
      goto cleanup;
    }

    // Get AIM bodies
    status = aim_getBodies(aimInfo, &intent, &numBody, &bodies);
    AIM_STATUS(aimInfo, status);

#ifdef DEBUG
    printf("himachAIM/aimUpdateState -> numBody=%d\n", numBody);
#endif

    if ((numBody <= 0) || (bodies == NULL)) {
        AIM_ERROR(aimInfo, "No body\n");
        status = CAPS_SOURCEERR;
        goto cleanup;

    } else if (numBody > 1) {
        AIM_ERROR(aimInfo, "FlightStream can only accept a single body\n");
        status = CAPS_SOURCEERR;
        goto cleanup;
    }

    if (aimInputs[inMach-1].nullVal == IsNull) {
      AIM_ERROR(aimInfo, "'Mach' must be specified");
      status = CAPS_BADVALUE;
      goto cleanup;
    }

    // Get design variables
    if (aimInputs[inDesign_Variable-1].nullVal == NotNull &&
        (himachInstance->design.numDesignVariable == 0 ||
         aim_newAnalysisIn(aimInfo, inDesign_Variable) == CAPS_SUCCESS)) {

/*@-nullpass@*/
        status = cfd_getDesignVariable(aimInfo,
                                       aimInputs[inDesign_Variable-1].length,
                                       aimInputs[inDesign_Variable-1].vals.tuple,
                                       &himachInstance->design.numDesignVariable,
                                       &himachInstance->design.designVariable);
/*@+nullpass@*/
        AIM_STATUS(aimInfo, status);
    }

#ifdef DEBUG
    printf("himachAIM/aimUpdateState -> scaleFactor=%f\n", scaleFactor);
#endif

    // Loop over bodies and look for reference quantity attributes
    for (i=0; i < numBody; i++) {
        status = EG_attributeRet(bodies[i], "capsReferenceChord",
                                 &atype, &alen, &ints, &reals, &string);
        if (status == EGADS_SUCCESS) {
            if (atype == ATTRREAL && alen == 1) {
                himachInstance->Cref = reals[0];
                foundCref = (int)true;
            } else {
                AIM_ERROR(aimInfo, "capsReferenceChord should be a scalar\n");
                status = CAPS_BADVALUE;
                goto cleanup;
            }
        }

        status = EG_attributeRet(bodies[i], "capsReferenceSpan",
                                 &atype, &alen, &ints, &reals, &string);
        if (status == EGADS_SUCCESS) {
            if (atype == ATTRREAL && alen == 1) {
                himachInstance->Bref = reals[0];
                foundBref = (int)true;
            } else {
                AIM_ERROR(aimInfo, "capsReferenceSpan should be a scalar\n");
                status = CAPS_BADVALUE;
                goto cleanup;
            }
        }

        status = EG_attributeRet(bodies[i], "capsReferenceArea",
                                 &atype, &alen, &ints, &reals, &string);
        if (status == EGADS_SUCCESS) {
            if (atype == ATTRREAL && alen == 1) {
                himachInstance->Sref = reals[0];
                foundSref = (int)true;
            } else {
                AIM_ERROR(aimInfo, "capsReferenceArea should be a scalar\n");
                status = CAPS_BADVALUE;
                goto cleanup;
            }
        }

        status = EG_attributeRet(bodies[i], "capsReferenceX",
                                 &atype, &alen, &ints, &reals, &string);
        if (status == EGADS_SUCCESS) {

            if (atype == ATTRREAL && alen == 1) {
                himachInstance->Xref = reals[0];
                foundXref = (int)true;
            } else {
                AIM_ERROR(aimInfo, "capsReferenceX should be followed by a single real value!\n");
                status = CAPS_BADVALUE;
                goto cleanup;
            }
        }

        status = EG_attributeRet(bodies[i], "capsReferenceY",
                                 &atype, &alen, &ints, &reals, &string);
        if (status == EGADS_SUCCESS) {

            if (atype == ATTRREAL && alen == 1) {
                himachInstance->Yref = reals[0];
                foundYref = (int)true;
            } else {
                AIM_ERROR(aimInfo, "capsReferenceY should be followed by a single real value!\n");
                status = CAPS_BADVALUE;
                goto cleanup;
            }
        }

        status = EG_attributeRet(bodies[i], "capsReferenceZ",
                                 &atype, &alen, &ints, &reals, &string);
        if (status == EGADS_SUCCESS){

            if (atype == ATTRREAL && alen == 1) {
                himachInstance->Zref = reals[0];
                foundZref = (int)true;
            } else {
                AIM_ERROR(aimInfo, "capsReferenceZ should be followed by a single real value!\n");
                status = CAPS_BADVALUE;
                goto cleanup;
            }
        }
    }

    if (aimInputs[inReferenceArea-1].nullVal == NotNull) {
        himachInstance->Sref = aimInputs[inReferenceArea-1].vals.real;
        foundSref = (int)true;
    }
    if (foundSref == (int)false) {
        AIM_ERROR(aimInfo, "capsReferenceArea is not set on any body and 'ReferenceArea' input not set!");
        status = CAPS_BADVALUE;
        goto cleanup;
    }

    if (aimInputs[inReferenceChord-1].nullVal == NotNull) {
        himachInstance->Cref = aimInputs[inReferenceChord-1].vals.real;
        foundCref = (int)true;
    }
    if (foundCref == (int)false) {
        AIM_ERROR(aimInfo, "capsReferenceChord is not set on any body and 'ReferenceChord' input not set!");
        status = CAPS_BADVALUE;
        goto cleanup;
    }

    if (aimInputs[inReferenceSpan-1].nullVal == NotNull) {
        himachInstance->Bref = aimInputs[inReferenceSpan-1].vals.real;
        foundBref = (int)true;
    }
    if (foundBref == (int)false) {
        AIM_ERROR(aimInfo, "capsReferenceSpan is not set on any body and 'ReferenceSpan' input not set!");
        status = CAPS_BADVALUE;
        goto cleanup;
    }

    if (aimInputs[inReferenceX-1].nullVal == NotNull) {
        himachInstance->Xref = aimInputs[inReferenceX-1].vals.real;
        foundXref = (int)true;
    }
    if (foundXref == (int)false) {
        AIM_ERROR(aimInfo, "capsReferenceX is not set on any body and 'ReferenceX' input not set!");
        status = CAPS_BADVALUE;
        goto cleanup;
    }

    if (aimInputs[inReferenceY-1].nullVal == NotNull) {
        himachInstance->Yref = aimInputs[inReferenceY-1].vals.real;
        foundYref = (int)true;
    }
    if (foundYref == (int)false) {
        AIM_ERROR(aimInfo, "capsReferenceY is not set on any body and 'ReferenceY' input not set!");
        status = CAPS_BADVALUE;
        goto cleanup;
    }

    if (aimInputs[inReferenceZ-1].nullVal == NotNull) {
        himachInstance->Zref = aimInputs[inReferenceZ-1].vals.real;
        foundZref = (int)true;
    }
    if (foundZref == (int)false) {
        AIM_ERROR(aimInfo, "capsReferenceZ is not set on any body and 'ReferenceZ' input not set!");
        status = CAPS_BADVALUE;
        goto cleanup;
    }

//    if (aimInputs[inReferenceVelocity-1].nullVal == IsNull) {
//      AIM_ERROR(aimInfo, "Input ReferenceVelocity not set!");
//      status = CAPS_BADVALUE;
//      goto cleanup;
//    }

    // Get mesh
    himachInstance->meshRefIn = (aimMeshRef *) aimInputs[inSurface_Mesh-1].vals.AIMptr;

    if ( aimInputs[inMesh_Morph-1].vals.integer == (int) true &&
        himachInstance->meshRefIn == NULL) { // If we are mighty morphing

      // Lets "load" the meshRef now since it's not linked
      status = aim_loadMeshRef(aimInfo, &himachInstance->meshRefMorph);
      AIM_STATUS(aimInfo, status);

      // Mighty Morph the mesh
      status = aim_morphMeshUpdate(aimInfo, &himachInstance->meshRefMorph, numBody, bodies);
      AIM_STATUS(aimInfo, status);

      AIM_ALLOC(himachInstance->meshRefMorph.fileName, PATH_MAX, char, aimInfo, status);
      status = aim_file(aimInfo, aimInputs[inProj_Name-1].vals.string, himachInstance->meshRefMorph.fileName);
      AIM_STATUS(aimInfo, status);

      /*@-immediatetrans@*/
      himachInstance->meshRefIn = &himachInstance->meshRefMorph;
      /*@+immediatetrans@*/
    }
    AIM_NOTNULL(himachInstance->meshRefIn, aimInfo, status);

    // Get attribute to index mapping
    status = create_MeshRefToIndexMap(aimInfo, himachInstance->meshRefIn, &himachInstance->groupMap);
    AIM_STATUS(aimInfo, status);

    snprintf(himachInstance->input , PATH_MAX, "%s_input.json" , aimInputs[inProj_Name-1].vals.string);
    snprintf(himachInstance->result, PATH_MAX, "%s_result.vtk" , aimInputs[inProj_Name-1].vals.string);
    snprintf(himachInstance->report, PATH_MAX, "%s_report.json", aimInputs[inProj_Name-1].vals.string);
    snprintf(himachInstance->sens  , PATH_MAX, "%s.sens", aimInputs[inProj_Name-1].vals.string);

#ifdef DEBUG
    printf("himachInstance->projectName = %s\n", himachInstance->projectName);
    printf("himachInstance->Cref        = %f\n", himachInstance->Cref       );
    printf("himachInstance->Bref        = %f\n", himachInstance->Bref       );
    printf("himachInstance->Sref        = %f\n", himachInstance->Sref       );
    printf("himachInstance->Xref        = %f\n", himachInstance->Xref       );
    printf("himachInstance->Yref        = %f\n", himachInstance->Yref       );
    printf("himachInstance->Zref        = %f\n", himachInstance->Zref       );
#endif

cleanup:
    return status;
}


/**********************************************************************/
/* aimPreAnalysis - generate FlightStream input file                  */
/**********************************************************************/

int
aimPreAnalysis(const void *instStore,   /* (in)  AIM instance storage */
               void       *aimInfo,     /* (in)  AIM context */
               capsValue  aimInputs[])  /* (in)  array of analysis inputs */
{
    // Function return flag
    int status = CAPS_SUCCESS;

    //double alt, temp, vel;
    FILE   *fp = NULL;

    int comma = (int)false;
    char meshFile[PATH_MAX];

    //const double *freestream_direction;
    aimMesh    mesh;

    const aimStorage *himachInstance = (const aimStorage *)instStore;

    /* -------------------------------------------------------------- */

#ifdef DEBUG
    printf("himachAIM/aimPreAnalysis()\n");
#endif

    mesh.meshRef = NULL;
    mesh.meshData = NULL;

    AIM_NOTNULL(aimInputs, aimInfo, status);

    status = aim_rmFile(aimInfo, himachInstance->input ); AIM_STATUS(aimInfo, status);
    status = aim_rmFile(aimInfo, himachInstance->result); AIM_STATUS(aimInfo, status);
    status = aim_rmFile(aimInfo, himachInstance->report); AIM_STATUS(aimInfo, status);
    status = aim_rmFile(aimInfo, himachInstance->sens  ); AIM_STATUS(aimInfo, status);

    if ( aimInputs[inMesh_Morph-1].vals.integer == (int) true) {
      if (aimInputs[inSurface_Mesh-1].nullVal == NotNull) { // If we are mighty morphing
        // store the current mesh for future iterations
        status = aim_storeMeshRef(aimInfo, (aimMeshRef *) aimInputs[inSurface_Mesh-1].vals.AIMptr, NULL);
        AIM_STATUS(aimInfo, status);
      } else {

        /*@-immediatetrans@*/
        mesh.meshData = NULL;
        mesh.meshRef = (aimMeshRef *)himachInstance->meshRefIn;
        /*@+immediatetrans@*/

        status = mesh_surfaceMeshData(aimInfo, &himachInstance->groupMap, &mesh);
        AIM_STATUS(aimInfo, status);
        AIM_NOTNULL(mesh.meshData, aimInfo, status);

        /* write the mesh */
        status = aim_writeMesh(aimInfo, MESHWRITER, NULL, &mesh);
        AIM_STATUS(aimInfo, status);
      }
    }

    snprintf(meshFile, PATH_MAX, "%s%s", himachInstance->meshRefIn->fileName, MESHEXTENSION);

#ifdef WIN32
    status = aim_dupSlash(meshFile);
    AIM_STATUS(aimInfo, status);
#endif

    status = fs_dataTransfer(aimInfo, aimInputs, himachInstance);
    AIM_STATUS(aimInfo, status);

    // write flightstream script file
    fp = aim_fopen(aimInfo, himachInstance->input, "w");
    if (fp == NULL) {
        AIM_ERROR(aimInfo, "Cannot open \"%s\"", himachInstance->input);
        status = CAPS_IOERR;
        goto cleanup;
    }

    //freestream_direction = aimInputs[inFreestream_Direction-1].vals.reals;

    fprintf(fp, "{\n");
    fprintf(fp, "    \"flow\" : {\n");
    //fprintf(fp, "        \"freestream_direction\" : [%16.12e, %16.12e, %16.12e],\n", freestream_direction[0], freestream_direction[1], freestream_direction[2]);
    fprintf(fp, "        \"mach_number\" : %16.12e,\n", aimInputs[inMach-1].vals.real);
    fprintf(fp, "        \"gamma\" : %16.12e,\n", aimInputs[inGamma-1].vals.real);
    fprintf(fp, "        \"alpha\" : %16.12e,\n", aimInputs[inAlpha-1].vals.real);
    fprintf(fp, "        \"beta\" : %16.12e\n", aimInputs[inBeta-1].vals.real);
    fprintf(fp, "    },\n");

    fprintf(fp, "    \"solver\" : {\n");
    comma = (int)false;
    if (aimInputs[inWindward_Method-1].nullVal == NotNull) {
      fprintf(fp, "        \"windward_method\" : \"%s\"", aimInputs[inWindward_Method-1].vals.string);
      comma = (int)true;
    }
    if (aimInputs[inLeeward_Method-1].nullVal == NotNull) {
      if (comma == (int)true) fprintf(fp,",\n");
      fprintf(fp, "        \"leeward_method\" : \"%s\"", aimInputs[inLeeward_Method-1].vals.string);
      comma = (int)true;
    }
    if (aimInputs[inBase_Pressure-1].nullVal == NotNull) {
      if (comma == (int)true) fprintf(fp,",\n");
      fprintf(fp, "        \"base_pressure\" : \"%s\"", aimInputs[inBase_Pressure-1].vals.string);
      comma = (int)true;
    }
    if (aimInputs[inShielding_Effects-1].nullVal == NotNull) {
      if (comma == (int)true) fprintf(fp,",\n");
      fprintf(fp, "        \"shielding_effects\" : \"%s\"", aimInputs[inShielding_Effects-1].vals.integer == (int)true ? "true" : "false");
    }
    if (aimInputs[inDesign_Variable-1].nullVal == NotNull) {
      if (comma == (int)true) fprintf(fp,",\n");
      fprintf(fp, "        \"calc_sensitivities\" : true");
      comma = (int)true;
    }
    if (comma == (int)true) fprintf(fp,"\n");
    fprintf(fp, "    },\n");


    fprintf(fp, "    \"geometry\" : {\n");
    fprintf(fp, "        \"file\" : \"%s\",\n", meshFile);
    fprintf(fp, "        \"reference\" : {\n");
    fprintf(fp, "            \"area\" : %16.12e,\n", himachInstance->Sref);
    fprintf(fp, "            \"length\" : %16.12e,\n", himachInstance->Cref);
    fprintf(fp, "            \"CG\" : [%16.12e, %16.12e, %16.12e]\n", himachInstance->Xref, himachInstance->Yref, himachInstance->Zref);
    fprintf(fp, "        },\n");
    fprintf(fp, "        \"nose_axis\" : \"%s\",\n", aimInputs[inNose_Axis-1].vals.string);
    fprintf(fp, "        \"pitch_axis\" : \"%s\"\n", aimInputs[inPitch_Axis-1].vals.string);
    fprintf(fp, "    },\n");

    fprintf(fp, "    \"output\" : {\n");
    fprintf(fp, "        \"verbose\" : false,\n");
    if (aimInputs[inDesign_Variable-1].nullVal == NotNull) {
      fprintf(fp, "        \"write_sensitivities\" : {\n");
      fprintf(fp, "            \"filename\": \"%s\",\n", himachInstance->sens);
      fprintf(fp, "            \"C_L\": true,\n");
      fprintf(fp, "            \"C_D\": true\n");
      fprintf(fp, "        },\n");
    }
    fprintf(fp, "        \"body_file\" : \"%s\",\n", himachInstance->result);
    fprintf(fp, "        \"report_file\" : \"%s\"\n", himachInstance->report);
    fprintf(fp, "    }\n");

    fprintf(fp, "}\n");

    fclose(fp); fp = NULL;

cleanup:
    if (fp != NULL) fclose(fp);

    aim_freeMeshData(mesh.meshData);
    AIM_FREE(mesh.meshData);

    return status;
}


/**********************************************************************/
/* aimExecute - execute FlightStream                                  */
/**********************************************************************/
int aimExecute(/*@unused@*/ const void *instStore, void *aimInfo,
               int *state)
{
  /*! \page aimExecuteHIMACH AIM Execution
   *
   * If auto execution is enabled when creating an FlightStream AIM,
   * the AIM will execute FlightStream just-in-time on Linux with the command line:
   *
   * \code{.sh}
   * FlightStream script.txt > flightstreamOut.txt
   * \endcode
   *
   * and on Windows with the command:
   *
   * \code{.sh}
   * FlightStream -hidden -script script.txt > flightstreamOut.txt
   * \endcode
   *
   * In both cases the FlightStream executable is assumed to in the PATH environment variable.
   *
   * The analysis can be also be explicitly executed with caps_execute in the C-API
   * or via Analysis.runAnalysis in the pyCAPS API.
   *
   * Calling preAnalysis and postAnalysis is NOT allowed when auto execution is enabled.
   *
   * Auto execution can also be disabled when creating an FlightStream AIM object.
   * In this mode, caps_execute and Analysis.runAnalysis can be used to run the analysis,
   * or FlightStream can be executed by calling preAnalysis, system, and posAnalysis as demonstrated
   * below with a pyCAPS example:
   *
   * \code{.py}
   * print ("\n\preAnalysis......")
   * flightstream.preAnalysis()
   *
   * print ("\n\nRunning......")
   * flightstream.system("FlightStream.exe -hidden -script script.txt"); # Run via system call in inputs analysis directory
   *
   * print ("\n\postAnalysis......")
   * flightstream.postAnalysis()
   * \endcode
   */
  int status = CAPS_SUCCESS;
  const aimStorage *himachInstance = (const aimStorage *)instStore;
  capsValue *himach = NULL;

#define fslen PATH_MAX+42
  char fscmd[fslen];

  *state = 0;

  aim_getValue(aimInfo, inHIMach, ANALYSISIN, &himach);
  AIM_STATUS(aimInfo, status);
  AIM_NOTNULL(himach, aimInfo, status);

  snprintf(fscmd, fslen, "%s %s", himach->vals.string, himachInstance->input);
  printf(" Executing: %s\n", fscmd);
  status = aim_system(aimInfo, "", fscmd);
  AIM_STATUS(aimInfo, status);

#undef fslen
cleanup:
  return status;
}


/**********************************************************************/
/* aimPostAnalysis - read FlightStream output file                    */
/**********************************************************************/

int
aimPostAnalysis(/*@unused@*/ void *instStore,    /* (in)  AIM instance storage */
                /*@unused@*/ void *aimInfo,      /* (in)  AIM context */
                /*@unused@*/ int restart,        /* (in)  0=normal, 1=restart */
                /*@unused@*/ capsValue inputs[]) /* (in)  array of analysis inputs */
{
  int    status = CAPS_SUCCESS;
  int    found = (int)false;

  int i, j, k, idv, irow, icol, ibody; // Indexing
  int index;

  char tmp[128];
  int numFunctional=0, nGeomIn = 0, numDesignVariable = 0;
  int **functional_map=NULL;
  double **functional_xyz=NULL;
  double functional_dvar;

  int numNode = 0, *numPoint=NULL;
  const char *name;
  char **names=NULL;
  double **dxyz = NULL;

  const char *projectName =NULL;

  capsValue *values[NUMOUTPUT], **funcs=NULL, *geomInVal=NULL;
  const char *valnames[NUMOUTPUT] = {"Cx", "Cy", "Cz", "C_L", "C_D", "CMx", "CMy", "CMz"};

//  cfdDesignVariableStruct *dvar=NULL;

  // Mesh reference obtained from meshing AIM
  const aimMeshRef *meshRef = NULL;

  size_t linecap=0;
  char   *line=NULL, *rest=NULL;

  FILE   *fp=NULL;
  aimStorage *himachInstance = (aimStorage *)instStore;

  char aimFile[PATH_MAX];
//  FILE   *tec=NULL;

  /* -------------------------------------------------------------- */
  AIM_NOTNULL(inputs, aimInfo, status);

#ifdef DEBUG
  printf("himachAIM/aimPostAnalysis()\n");
#endif

  fp = aim_fopen(aimInfo, himachInstance->report, "r");
  if (fp == NULL) {
    AIM_ERROR(aimInfo, "Unable to open \"%s\"\n", himachInstance->report);
    status = CAPS_IOERR;
    goto cleanup;
  }

  /* Find the coefficients */
  while (getline(&line, &linecap, fp) > 0) {
    AIM_NOTNULL(line, aimInfo, status);
    if (strstr(line, "Cx") != NULL) {
      strtok_r(line, ":", &rest);
      AIM_NOTNULL(rest, aimInfo, status);
      sscanf(rest, "%lf", &(himachInstance->Cx.vals.real));
    } else if (strstr(line, "Cy") != NULL) {
      strtok_r(line, ":", &rest);
      AIM_NOTNULL(rest, aimInfo, status);
      sscanf(rest, "%lf", &(himachInstance->Cy.vals.real));
    } else if (strstr(line, "Cz") != NULL) {
      strtok_r(line, ":", &rest);
      AIM_NOTNULL(rest, aimInfo, status);
      sscanf(rest, "%lf", &(himachInstance->Cz.vals.real));
    } else if (strstr(line, "C_L") != NULL) {
      strtok_r(line, ":", &rest);
      AIM_NOTNULL(rest, aimInfo, status);
      sscanf(rest, "%lf", &(himachInstance->CL.vals.real));
    } else if (strstr(line, "C_D") != NULL) {
      strtok_r(line, ":", &rest);
      AIM_NOTNULL(rest, aimInfo, status);
      sscanf(rest, "%lf", &(himachInstance->CD.vals.real));
    } else if (strstr(line, "CMx") != NULL) {
      strtok_r(line, ":", &rest);
      AIM_NOTNULL(rest, aimInfo, status);
      sscanf(rest, "%lf", &(himachInstance->CMx.vals.real));
    } else if (strstr(line, "CMy") != NULL) {
      strtok_r(line, ":", &rest);
      AIM_NOTNULL(rest, aimInfo, status);
      sscanf(rest, "%lf", &(himachInstance->CMy.vals.real));
    } else if (strstr(line, "CMz") != NULL) {
      strtok_r(line, ":", &rest);
      AIM_NOTNULL(rest, aimInfo, status);
      sscanf(rest, "%lf", &(himachInstance->CMz.vals.real));
    }
  }

  fclose(fp);
  fp = NULL;

  if (inputs[inDesign_Variable-1].nullVal == NotNull) {

    values[0] = &himachInstance->Cx;
    values[1] = &himachInstance->Cy;
    values[2] = &himachInstance->Cz;
    values[3] = &himachInstance->CL;
    values[4] = &himachInstance->CD;
    values[5] = &himachInstance->CMx;
    values[6] = &himachInstance->CMy;
    values[7] = &himachInstance->CMz;

    // Get mesh
    meshRef = himachInstance->meshRefIn;
    AIM_NOTNULL(meshRef, aimInfo, status);

    /* check for GeometryIn variables*/
    nGeomIn = 0;
    for (i = 0; i < himachInstance->design.numDesignVariable; i++) {

      name = himachInstance->design.designVariable[i].name;

      // Loop over the geometry in values and compute sensitivities for all bodies
      index = aim_getIndex(aimInfo, name, GEOMETRYIN);
      if (index == CAPS_NOTFOUND) continue;
      if (index < CAPS_SUCCESS ) {
        status = index;
        AIM_STATUS(aimInfo, status);
      }

      if(aim_getGeomInType(aimInfo, index) != 0) {
        AIM_ERROR(aimInfo, "GeometryIn value %s is a configuration parameter and not a valid design parameter - can't get sensitivity\n",
                  name);
        status = CAPS_BADVALUE;
        goto cleanup;
      }

      nGeomIn++;
    }

    projectName = inputs[inProj_Name-1].vals.string;

    // Read <Proj_Name>.sens
    snprintf(tmp, 128, "%s%s", projectName, ".sens");
    fp = aim_fopen(aimInfo, tmp, "r");
    if (fp == NULL) {
      AIM_ERROR(aimInfo, "Unable to open: %s", tmp);
      status = CAPS_IOERR;
      goto cleanup;
    }

    // Number of nodes and functionals and AnalysIn design variables in the file
    status = fscanf(fp, "%d %d", &numFunctional, &numDesignVariable);
    if (status == EOF || status != 2) {
      AIM_ERROR(aimInfo, "Failed to read sens file number of functionals and analysis design variables");
      status = CAPS_IOERR; goto cleanup;
    }
    if (himachInstance->design.numDesignVariable != numDesignVariable+nGeomIn) {
      AIM_ERROR(aimInfo, "Incorrect number of AnalysisIn derivatives in sens file. Expected %d and found %d",
                himachInstance->design.numDesignVariable-nGeomIn, numDesignVariable);
      status = CAPS_IOERR; goto cleanup;
    }

    AIM_ALLOC(numPoint, numFunctional, int, aimInfo, status);
    for (i = 0; i < numFunctional; i++) numPoint[i] = 0;

    AIM_ALLOC(functional_map, numFunctional, int*, aimInfo, status);
    for (i = 0; i < numFunctional; i++) functional_map[i] = NULL;

    AIM_ALLOC(functional_xyz, numFunctional, double*, aimInfo, status);
    for (i = 0; i < numFunctional; i++) functional_xyz[i] = NULL;

    AIM_ALLOC(names, numFunctional, char*, aimInfo, status);
    for (i = 0; i < numFunctional; i++) names[i] = NULL;

    AIM_ALLOC(funcs, numFunctional, capsValue*, aimInfo, status);
    for (i = 0; i < numFunctional; i++) funcs[i] = NULL;

    // Read in Functional name, value and dFunctinoal/dxyz
    for (i = 0; i < numFunctional; i++) {

      status = fscanf(fp, "%s", tmp);
      if (status == EOF) {
        AIM_ERROR(aimInfo, "Failed to read sens file functional name");
        status = CAPS_IOERR; goto cleanup;
      }

      AIM_STRDUP(names[i], tmp, aimInfo, status);

      /* Check if it's a static output */
      for (j = 0; j < NUMOUTPUT; j++) {
        if (strcmp(names[i],valnames[j]) != 0) continue;
        funcs[i] = values[j];
        break;
      }
      /* Not a static function, make it a dynamic output */
      if (funcs[i] == NULL) {
        AIM_ALLOC(funcs[i], 1, capsValue, aimInfo, status);
        aim_initValue(funcs[i]);
      }

      funcs[i]->type = DoubleDeriv;

      /* allocate derivatives */
      AIM_ALLOC(funcs[i]->derivs, himachInstance->design.numDesignVariable, capsDeriv, aimInfo, status);
      for (idv = 0; idv < himachInstance->design.numDesignVariable; idv++) {
        funcs[i]->derivs[idv].name  = NULL;
        funcs[i]->derivs[idv].deriv = NULL;
        funcs[i]->derivs[idv].len_wrt = 0;
      }
      funcs[i]->nderiv = himachInstance->design.numDesignVariable;

      status = fscanf(fp, "%lf", &funcs[i]->vals.real);
      if (status == EOF || status != 1) {
        AIM_ERROR(aimInfo, "Failed to read sens file functional value");
        status = CAPS_IOERR; goto cleanup;
      }

      status = fscanf(fp, "%d", &numPoint[i]);
      if (status == EOF || status != 1) {
        AIM_ERROR(aimInfo, "Failed to read sens file number of points");
        status = CAPS_IOERR; goto cleanup;
      }

      AIM_ALLOC(functional_map[i],   numPoint[i], int   , aimInfo, status);
      AIM_ALLOC(functional_xyz[i], 3*numPoint[i], double, aimInfo, status);

      for (j = 0; j < numPoint[i]; j++) {
        status = fscanf(fp, "%d %lf %lf %lf", &functional_map[i][j],
                        &functional_xyz[i][3*j+0],
                        &functional_xyz[i][3*j+1],
                        &functional_xyz[i][3*j+2]);
        if (status == EOF || status != 4) {
          AIM_ERROR(aimInfo, "Failed to read sens file data");
          status = CAPS_IOERR; goto cleanup;
        }
      }


      /* read additional derivatives from .sens file */
      for (k = nGeomIn; k < himachInstance->design.numDesignVariable; k++) {

        /* get derivative name */
        status = fscanf(fp, "%s", tmp);
        if (status == EOF) {
          AIM_ERROR(aimInfo, "Failed to read sens file design variable name");
          status = CAPS_IOERR; goto cleanup;
        }

        found = (int)false;
        for (idv = 0; idv < himachInstance->design.numDesignVariable; idv++)
          if ( strcasecmp(himachInstance->design.designVariable[idv].name, tmp) == 0) {
            found = (int)true;
            break;
          }
        if (found == (int)false) {
          AIM_ERROR(aimInfo, "Design variable '%s' in sens file not in Design_Varible input", tmp);
          status = CAPS_IOERR; goto cleanup;
        }

        AIM_STRDUP(funcs[i]->derivs[idv].name, tmp, aimInfo, status);

        status = fscanf(fp, "%d", &funcs[i]->derivs[idv].len_wrt);
        if (status == EOF || status != 1) {
          AIM_ERROR(aimInfo, "Failed to read sens file number of design variable derivatives");
          status = CAPS_IOERR; goto cleanup;
        }

        AIM_ALLOC(funcs[i]->derivs[idv].deriv, funcs[i]->derivs[idv].len_wrt, double, aimInfo, status);
        for (j = 0; j < funcs[i]->derivs[idv].len_wrt; j++) {

          status = fscanf(fp, "%lf", &funcs[i]->derivs[idv].deriv[j]);
          if (status == EOF || status != 1) {
            AIM_ERROR(aimInfo, "Failed to read sens file design variable derivative");
            status = CAPS_IOERR; goto cleanup;
          }
        }
      }
    }

    AIM_ALLOC(dxyz, meshRef->nmap, double*, aimInfo, status);
    for (ibody = 0; ibody < meshRef->nmap; ibody++) dxyz[ibody] = NULL;

    /* set derivatives */
    for (idv = 0; idv < himachInstance->design.numDesignVariable; idv++) {

      name = himachInstance->design.designVariable[idv].name;

      // Loop over the geometry in values and compute sensitivities for all bodies
      index = aim_getIndex(aimInfo, name, GEOMETRYIN);
      status = aim_getValue(aimInfo, index, GEOMETRYIN, &geomInVal);
      if (status == CAPS_BADINDEX) continue;
      AIM_STATUS(aimInfo, status);
      AIM_NOTNULL(geomInVal, aimInfo, status);

      for (i = 0; i < numFunctional; i++) {
        AIM_STRDUP(funcs[i]->derivs[idv].name, name, aimInfo, status);

        AIM_ALLOC(funcs[i]->derivs[idv].deriv, geomInVal->length, double, aimInfo, status);
        funcs[i]->derivs[idv].len_wrt  = geomInVal->length;
        for (j = 0; j < geomInVal->length; j++)
          funcs[i]->derivs[idv].deriv[j] = 0;
      }

      snprintf(aimFile, PATH_MAX, "%s.tec" , inputs[inProj_Name-1].vals.string);

//      tec = aim_fopen(aimInfo, aimFile, "w");
//      if (tec == NULL) {
//        AIM_ERROR(aimInfo, "Unable to open \"%s\"\n", aimFile);
//        status = CAPS_IOERR;
//        goto cleanup;
//      }
//      fprintf(tec, "VARIABLES=X, Y, Z, nx, ny, nz\n");

      for (irow = 0; irow < geomInVal->nrow; irow++) {
        for (icol = 0; icol < geomInVal->ncol; icol++) {

          // get the sensitivity for each body
          for (ibody = 0; ibody < meshRef->nmap; ibody++) {
            if (meshRef->maps[ibody].tess == NULL) continue;
            status = aim_tessSensitivity(aimInfo,
                                         name,
                                         irow+1, icol+1, // row, col
                                         meshRef->maps[ibody].tess,
                                         &numNode, &dxyz[ibody]);
            AIM_STATUS(aimInfo, status, "Sensitivity for: %s\n", name);
            AIM_NOTNULL(dxyz[ibody], aimInfo, status);

            for (i = 0; i < numFunctional; i++) {

//              fprintf(tec, "ZONE T=\"geom_%d\", DATAPACKING=POINT, I=%d, J=1, K=1\n",i, numPoint[i]);
//              for (j = 0; j < numPoint[i]; j++) {
//
//                int ptype, pindex;
//                double xyz[3];
//                status = EG_getGlobal(meshRef->maps[ibody].tess,
//                                      j+1, &ptype, &pindex, xyz);
//                AIM_STATUS(aimInfo, status);
//                fprintf(tec, "%le %le %le  %le %le %le\n", xyz[0], xyz[1], xyz[2],
//                                                           dxyz[ibody][3*j + 0],
//                                                           dxyz[ibody][3*j + 1],
//                                                           dxyz[ibody][3*j + 2]);
//              }
//
//              fprintf(tec, "ZONE T=\"Cf_%d\", DATAPACKING=POINT, I=%d, J=1, K=1\n",i, numPoint[i]);

              functional_dvar = funcs[i]->derivs[idv].deriv[geomInVal->ncol*irow + icol];

              for (j = 0; j < numPoint[i]; j++) {
                k = functional_map[i][j]-1; // 1-based indexing into surface mesh

                functional_dvar += functional_xyz[i][3*j+0]*dxyz[ibody][3*k + 0]  // dx/dGeomIn
                                 + functional_xyz[i][3*j+1]*dxyz[ibody][3*k + 1]  // dy/dGeomIn
                                 + functional_xyz[i][3*j+2]*dxyz[ibody][3*k + 2]; // dz/dGeomIn

                int ptype, pindex;
                double xyz[3];
                status = EG_getGlobal(meshRef->maps[ibody].tess,
                                      k+1, &ptype, &pindex, xyz);
                AIM_STATUS(aimInfo, status);

//                fprintf(tec, "%le %le %le  %le %le %le\n", xyz[0], xyz[1], xyz[2],
//                                                           functional_xyz[i][3*k + 0],
//                                                           functional_xyz[i][3*k + 1],
//                                                           functional_xyz[i][3*k + 2]);
              }
              funcs[i]->derivs[idv].deriv[geomInVal->ncol*irow + icol] = functional_dvar;
            }
          }

          for (ibody = 0; ibody < meshRef->nmap; ibody++)
            AIM_FREE(dxyz[ibody]);
        }
      }
      //fclose(tec); tec = NULL;
    }

    /* create the dynamic output */
    for (i = 0; i < numFunctional; i++) {
      for (j = 0; j < NUMOUTPUT; j++) {
        if (strcmp(names[i],valnames[j]) == 0) {
          funcs[i] = NULL;
          break;
        }
      }
      if (funcs[i] == NULL) continue;
      status = aim_makeDynamicOutput(aimInfo, names[i], funcs[i]);
      AIM_STATUS(aimInfo, status);
      AIM_FREE(funcs[i]);
    }
  }


cleanup:
  if (fp   != NULL) fclose(fp);
  if (line != NULL) free(line);

  if (functional_map != NULL) {
    for (i = 0; i < numFunctional; i++)
      AIM_FREE(functional_map[i]);
    AIM_FREE(functional_map);
  }
  if (functional_xyz != NULL) {
    for (i = 0; i < numFunctional; i++)
      AIM_FREE(functional_xyz[i]);
    AIM_FREE(functional_xyz);
  }
  AIM_FREE(numPoint);
  if (funcs != NULL) {
    if (names != NULL) {
      for (i = 0; i < numFunctional; i++) {
        for (j = 0; j < NUMOUTPUT; j++) {
          if (strcmp(names[i],valnames[j]) != 0) {
            funcs[i] = NULL;
            break;
          }
        }
        AIM_FREE(funcs[i]);
      }
    }
    AIM_FREE(funcs);
  }
  if (names != NULL) {
    for (i = 0; i < numFunctional; i++)
      AIM_FREE(names[i]);
    AIM_FREE(names);
  }
  if (meshRef != NULL && dxyz != NULL) {
    for (ibody = 0; ibody < meshRef->nmap; ibody++)
      AIM_FREE(dxyz[ibody]);
    AIM_FREE(dxyz);
  }


  return status;
}


/**********************************************************************/
/* aimCalcOutput - retreive FlightStream output information           */
/**********************************************************************/

int
aimCalcOutput(/*@unused@*/ void *instStore,    /* (in)  AIM instance storage */
              /*@unused@*/ void *aimInfo,      /* (in)  AIM context */
              /*@unused@*/ int index,          /* (in)  analysis output */
              /*@unused@*/ capsValue *outval)  /* (in)  pointer to capsValue to fill */
{
  int status = CAPS_SUCCESS;

  aimStorage *himachInstance = (aimStorage *)instStore;

  /* -------------------------------------------------------------- */

#ifdef DEBUG
  printf("himachAIM/aimCalcOutput(index=%d)\n", index);
#endif

  if        (index == outCx) {
    *outval = himachInstance->Cx;
    aim_initValue(&himachInstance->Cx);
  } else if (index == outCy) {
    *outval = himachInstance->Cy;
    aim_initValue(&himachInstance->Cy);
  } else if (index == outCz) {
    *outval = himachInstance->Cz;
    aim_initValue(&himachInstance->Cz);
  } else if (index == outCL) {
    *outval = himachInstance->CL;
    aim_initValue(&himachInstance->CL);
  } else if (index == outCD) {
    *outval = himachInstance->CD;
    aim_initValue(&himachInstance->CD);
  } else if (index == outCMx) {
    *outval = himachInstance->CMx;
    aim_initValue(&himachInstance->CMx);
  } else if (index == outCMy) {
    *outval = himachInstance->CMy;
    aim_initValue(&himachInstance->CMy);
  } else if (index == outCMz) {
    *outval = himachInstance->CMz;
    aim_initValue(&himachInstance->CMz);
  } else {
    status = CAPS_NOTFOUND;
    goto cleanup;
  }

cleanup:
  return status;
}


/**********************************************************************/
/* aimCleanup - free up memory allocated in aimInitialize             */
/**********************************************************************/

void
aimCleanup(void *instStore)             /* (in)  AIM instance storage */
{

  /* -------------------------------------------------------------- */

#ifdef DEBUG
  printf("himachAIM/aimCleanup()\n");
#endif

  aimStorage *himachInstance = (aimStorage *)instStore;

  destroy_aimStorage(himachInstance);

  AIM_FREE(himachInstance);
}


// ********************** AIM Function Break *****************************
int
aimDiscr(char *tname, capsDiscr *discr)
{

  int i; // Indexing

  int status; // Function return status

  int numBody;

  // EGADS objects
  ego *bodies = NULL, *tess = NULL;

  const char   *intents;

  // Volume Mesh obtained from meshing AIM
  const aimMeshRef *meshRef;

  aimStorage *himachInstance;

  himachInstance = (aimStorage *) discr->instStore;

#ifdef DEBUG
  printf(" himachAIM/aimDiscr: tname = %s!\n", tname);
#endif

  if (tname == NULL) return CAPS_NOTFOUND;

  // Currently this ONLY works if the capsTranfer lives on single body!
  status = aim_getBodies(discr->aInfo, &intents, &numBody, &bodies);
  if (status != CAPS_SUCCESS) {
    printf(" himachAIM/aimDiscr: aim_getBodies = %d!\n", status);
    return status;
  }
  if (bodies == NULL) {
    AIM_ERROR(discr->aInfo, "NULL Bodies!\n");
    return CAPS_NULLOBJ;
  }

  // Get mesh
  meshRef = himachInstance->meshRefIn;
  AIM_NOTNULL(meshRef, discr->aInfo, status);

  if (aim_newGeometry(discr->aInfo) == CAPS_SUCCESS) {
    // Get capsGroup name and index mapping to make sure all faces have a capsGroup value
    status = create_CAPSGroupAttrToIndexMap(numBody,
                                            bodies,
                                            1, // Only search down to the face level of the EGADS body
                                            &himachInstance->groupMap);
    AIM_STATUS(discr->aInfo, status);
  }

  // Lets check the volume mesh

  // Do we have an individual surface mesh for each body
  if (meshRef->nmap != numBody) {
    AIM_ERROR(  discr->aInfo, "Number of surface mesh in the linked volume mesh (%d) does not match the number");
    AIM_ADDLINE(discr->aInfo,"of bodies (%d) - data transfer is NOT possible.", meshRef->nmap,numBody);
    status = CAPS_MISMATCH;
    goto cleanup;
  }

  // Lets store away our tessellation now
  AIM_ALLOC(tess, meshRef->nmap, ego, discr->aInfo, status);
  for (i = 0; i < meshRef->nmap; i++) {
    tess[i] = meshRef->maps[i].tess;
  }

  status = mesh_fillDiscr(tname, &himachInstance->groupMap, meshRef->nmap, tess, discr);
  AIM_STATUS(discr->aInfo, status);

#ifdef DEBUG
  printf(" himachAIM/aimDiscr: Finished!!\n");
#endif

  status = CAPS_SUCCESS;

cleanup:
  AIM_FREE(tess);
  return status;
}

// ********************** AIM Function Break *****************************
void aimFreeDiscrPtr(void *ptr)
{
  // Extra information to store into the discr void pointer - just a int array
  EG_free(ptr);
}

// ********************** AIM Function Break *****************************
int
aimLocateElement(capsDiscr *discr, double *params, double *param,
                 int *bIndex, int *eIndex, double *bary)
{
    return aim_locateElement(discr, params, param, bIndex, eIndex, bary);
}


// ********************** AIM Function Break *****************************
int
aimTransfer(capsDiscr *discr, const char *name, int npts, int rank, double *data,
            /*@unused@*/char **units)
{
  /*! \page dataTransferHIMACH AIM Data Transfer
   *
   * The FlightStream AIM has the ability to transfer surface data (e.g. pressure distributions) to and from the AIM
   * using the conservative and interpolative data transfer schemes in CAPS.
   *
   * \section dataFromFlightsteram Data transfer from FlightStream (FieldOut)
   *
   * <ul>
   * <li> <B>"Pressure" </B> </li> <br>
   *  Loads the pressure distribution from FlightStream vtk file.
   *  This distribution may be scaled based on
   *  Pressure = Pressure_Scale_Factor*Pressure, where "Pressure_Scale_Factor"
   *  is an AIM input (\ref aimInputsHIMACH)
   * </ul>
   *
   */
  int    i, j, global, status, bIndex;
  double **rvec=NULL, scale = 1.0;
  capsValue *Pressure_Scale_Factor_Value=NULL;
  int state, nglobal;
  ego        body;
  FILE *fp = NULL;
#ifdef ANALYSIS_PLOAD_BDF
  int ID, inode, jnode;
  char str[10];
  size_t linecap = 0;
  char *line = NULL; // Temporary line holder
  const char *PLOAD4 = "$$  PLOAD4 Data";
#elif defined(ANALYSIS_VTK)
  size_t linecap = 0;
  char *line = NULL; // Temporary line holder
  const char *CP = "SCALARS Cp FLOAT";
//  double vref = 0, qref=0;
//  capsValue *ReferenceVelocity=NULL;
#elif defined(ANALYSIS_CSV)
  double x, y, z;
#else
#error "must define ANALYSIS_PLOAD_BDF, ANALYSIS_VTK, or, ANALYSIS_CSV"
#endif

  const aimStorage *himachInstance = (const aimStorage *)discr->instStore;
  const aimMeshRef *meshRef = himachInstance->meshRefIn;

#ifdef DEBUG
  printf(" himachAIM/aimTransfer name = %s  npts = %d/%d!\n",
         name, npts, len_wrt);
#endif

  if (strcmp(name, "Pressure") == 0) {

    if (rank != 1) {
      AIM_ERROR(discr->aInfo, "Rank (%d) must be 1 for 'Pressure'!", rank);
      status = CAPS_NOTIMPLEMENT;
      goto cleanup;
    }

#ifdef ANALYSIS_PLOAD_BDF
    fp = aim_fopen(discr->aInfo, loadbdf, "r");
    if (fp == NULL) {
      AIM_ERROR(discr->aInfo, "Cannot open \"%s\"", loadbdf);
      status = CAPS_IOERR;
      goto cleanup;
    }

    /* try and read the bdf file */
    while (getline(&line, &linecap, fp) > 0) {
      AIM_NOTNULL(line, discr->aInfo, status);
      if (strncmp(line, PLOAD4, strlen(PLOAD4)) == 0) {
        break;
      }
    }

    AIM_NOTNULL(line, discr->aInfo, status);
    if (strncmp(line, PLOAD4, strlen(PLOAD4)) != 0) {
      AIM_ERROR(discr->aInfo, "Could not find 'PLOAD4' data in \"%s\"", loadbdf);
      status = CAPS_IOERR;
      goto cleanup;
    }

    // Skip '$$'
    status = getline(&line, &linecap, fp);
    if (status <= 0) AIM_STATUS(discr->aInfo, (status = CAPS_IOERR));

    AIM_ALLOC(rvec, meshRef->nmap, double*, discr->aInfo, status);
    for (i = 0; i < meshRef->nmap; i++) {
      rvec[i] = NULL;
    }

    jnode = 1;
    for (i = 0; i < meshRef->nmap; i++) {
      status = EG_statusTessBody(meshRef->maps[i].tess, &body, &state, &nglobal);
      AIM_STATUS(discr->aInfo, status);
      AIM_ALLOC(rvec[i], nglobal, double, discr->aInfo, status);

      for (j = 0; j < nglobal; j++, jnode++) {
        status = getline(&line, &linecap, fp);
        if (status <= 0) AIM_STATUS(discr->aInfo, (status = CAPS_IOERR));
        status = sscanf(line, "%s %d %d %lf", str, &ID, &inode, &rvec[i][j]);
        if (status <= 0) AIM_STATUS(discr->aInfo, (status = CAPS_IOERR));
        if (inode != jnode) {
          AIM_ERROR(discr->aInfo, "While reading %s, %d != %d!", loadbdf, inode, jnode);
          status = CAPS_NOTIMPLEMENT;
          goto cleanup;
        }
        // subtract off the reference pressure
        // both BDF pressure and pref are in Pa
        rvec[i][j] -= himachInstance->pref;
      }
    }

    // Convert from Pa to working pressure
    scale = 1;
    status = aim_convert(discr->aInfo, 1, "Pa", &scale, himachInstance->units.pressure, &scale);
    AIM_STATUS(discr->aInfo, status);

#elif defined(ANALYSIS_VTK)
    fp = aim_fopen(discr->aInfo, cpvtk, "r");
    if (fp == NULL) {
      AIM_ERROR(discr->aInfo, "Cannot open \"%s\"", cpvtk);
      status = CAPS_IOERR;
      goto cleanup;
    }

    /* try and read the vtk file */
    while (getline(&line, &linecap, fp) > 0) {
      AIM_NOTNULL(line, discr->aInfo, status);
      if (strncmp(line, CP, strlen(CP)) == 0) {
        break;
      }
    }

    AIM_NOTNULL(line, discr->aInfo, status);
    if (strncmp(line, CP, strlen(CP)) != 0) {
      AIM_ERROR(discr->aInfo, "Could not find 'PLOAD4' data in \"%s\"", cpvtk);
      status = CAPS_IOERR;
      goto cleanup;
    }

    // Skip 'LOOKUP_TABLE default'
    status = getline(&line, &linecap, fp);
    if (status <= 0) AIM_STATUS(discr->aInfo, (status = CAPS_IOERR));

    AIM_ALLOC(rvec, meshRef->nmap, double*, discr->aInfo, status);
    for (i = 0; i < meshRef->nmap; i++) {
      rvec[i] = NULL;
    }

    for (i = 0; i < meshRef->nmap; i++) {
      status = EG_statusTessBody(meshRef->maps[i].tess, &body, &state, &nglobal);
      AIM_STATUS(discr->aInfo, status);
      AIM_ALLOC(rvec[i], nglobal, double, discr->aInfo, status);

      for (j = 0; j < nglobal; j++) {
        status = getline(&line, &linecap, fp);
        if (status <= 0) AIM_STATUS(discr->aInfo, (status = CAPS_IOERR));
        status = sscanf(line, "%lf", &rvec[i][j]);
        if (status <= 0) AIM_STATUS(discr->aInfo, (status = CAPS_IOERR));
      }
    }

//    status = aim_getValue(discr->aInfo, inReferenceVelocity, ANALYSISIN, &ReferenceVelocity);
//    AIM_STATUS(discr->aInfo, status);
//    AIM_NOTNULL(ReferenceVelocity, discr->aInfo, status);
//
//    // Compute dynamic pressure
//    status = aim_convert(discr->aInfo, 1,
//                         ReferenceVelocity->units,  &ReferenceVelocity->vals.real,
//                         "m/s", &vref);
//    AIM_STATUS(discr->aInfo, status);

//    status = aim_convert(discr->aInfo, 1, "Pa", &qref, himachInstance->units.pressure, &qref);
//    AIM_STATUS(discr->aInfo, status);
//    scale *= qref;

#elif defined(ANALYSIS_CSV)

    fp = aim_fopen(discr->aInfo, dpcsv, "r");
    if (fp == NULL) {
      AIM_ERROR(discr->aInfo, "Cannot open \"%s\"", dpcsv);
      status = CAPS_IOERR;
      goto cleanup;
    }

    AIM_ALLOC(rvec, meshRef->nmap, double*, discr->aInfo, status);
    for (i = 0; i < meshRef->nmap; i++) {
      rvec[i] = NULL;
    }

    for (i = 0; i < meshRef->nmap; i++) {
      status = EG_statusTessBody(meshRef->maps[i].tess, &body, &state, &nglobal);
      AIM_STATUS(discr->aInfo, status);
      AIM_ALLOC(rvec[i], nglobal, double, discr->aInfo, status);

      for (j = 0; j < nglobal; j++) {
        status = fscanf(fp, "%lf, %lf, %lf, %lf\n", &x, &y, &z, &rvec[i][j]);
        if (status != 4) AIM_STATUS(discr->aInfo, (status = CAPS_IOERR));
      }
    }
#else
#error "must define ANALYSIS_PLOAD_BDF, ANALYSIS_VTK, or ANALYSIS_CSV"
#endif


    // Custom additional scale factor
    status = aim_getValue(discr->aInfo, inPressure_Scale_Factor, ANALYSISIN, &Pressure_Scale_Factor_Value);
    AIM_STATUS(discr->aInfo, status);
    AIM_NOTNULL(Pressure_Scale_Factor_Value, discr->aInfo, status);
    scale *= Pressure_Scale_Factor_Value->vals.real;

    // set the units
//    AIM_STRDUP(*units, himachInstance->units.pressure, discr->aInfo, status);
  }

  /* move the appropriate parts of the tessellation to data */
  AIM_NOTNULL(rvec, discr->aInfo, status);
  for (i = 0; i < npts; i++) {
    /* points might span multiple bodies */
    bIndex = discr->tessGlobal[2*i  ];
    global = discr->tessGlobal[2*i+1];
    for (j = 0; j < rank; j++)
      data[rank*i+j] = rvec[bIndex-1][rank*(global-1)+j] * scale;
  }

  status = CAPS_SUCCESS;
cleanup:
  if (rvec != NULL) {
    for (i = 0; i < meshRef->nmap; ++i) {
      AIM_FREE(rvec[i]);
    }
  }
  AIM_FREE(rvec);
#if defined(ANALYSIS_PLOAD_BDF) || defined(ANALYSIS_VTK)
  if (line != NULL) free(line);
#endif

  if (fp != NULL) fclose(fp);

  return status;
}


// ********************** AIM Function Break *****************************
int
aimInterpolation(capsDiscr *discr, /*@unused@*/ const char *name, int bIndex,
                 int eIndex, double *bary, int rank, double *data,
                 double *result)
{
#ifdef DEBUG
    printf(" himachAIM/aimInterpolation  %s!\n", name);
#endif

    return  aim_interpolation(discr, name, bIndex, eIndex,
                              bary, rank, data, result);
}


// ********************** AIM Function Break *****************************
int
aimInterpolateBar(capsDiscr *discr, /*@unused@*/ const char *name, int bIndex,
                  int eIndex, double *bary, int rank, double *r_bar,
                  double *d_bar)
{
#ifdef DEBUG
    printf(" himachAIM/aimInterpolateBar  %s!\n", name);
#endif

    return  aim_interpolateBar(discr, name, bIndex, eIndex,
                               bary, rank, r_bar, d_bar);
}


// ********************** AIM Function Break *****************************
int
aimIntegration(capsDiscr *discr, /*@unused@*/ const char *name, int bIndex,
               int eIndex, int rank, double *data, double *result)
{
#ifdef DEBUG
    printf(" himachAIM/aimIntegration  %s!\n", name);
#endif

    return aim_integration(discr, name, bIndex, eIndex, rank,
                           data, result);
}


// ********************** AIM Function Break *****************************
int
aimIntegrateBar(capsDiscr *discr, /*@unused@*/ const char *name, int bIndex,
                int eIndex, int rank, double *r_bar, double *d_bar)
{
#ifdef DEBUG
    printf(" himachAIM/aimIntegrateBar  %s!\n", name);
#endif

    return aim_integrateBar(discr, name, bIndex, eIndex, rank,
                            r_bar, d_bar);
}
