import unittest

import os
import glob
import shutil

import sys

import pyCAPS

# Helper function to check if an executable exists
def which(program):
    import os
    def is_exe(fpath):
        return os.path.isfile(fpath) and os.access(fpath, os.X_OK)

    fpath, fname = os.path.split(program)
    if fpath:
        if is_exe(program):
            return program
    else:
        for path in os.environ["PATH"].split(os.pathsep):
            exe_file = os.path.join(path, program)
            if is_exe(exe_file):
                return exe_file

    return None

class TestCBAERO(unittest.TestCase):

    @classmethod
    def setUpClass(cls):

        cls.problemName = "workDir_cbaero"
        cls.iProb = 1
        cls.cleanUp()

        cornerFile = os.path.join("..","csmData","cfdX43a.csm")
        cls.capsProblem = pyCAPS.Problem(cls.problemName, capsFile=cornerFile)

        # Turn of the farfield domain
        cls.capsProblem.geometry.cfgpmtr.makeBoundBox = 0


        # Load egadsTess aim to create surface mesh
        cls.meshAIM = cls.capsProblem.analysis.create(aim = "egadsTessAIM",
                                                      name = "egadsTess" )
        # Set meshing parameters
        cls.meshAIM.input.Tess_Params = [0.025, 0.01, 10.0]


    @classmethod
    def tearDownClass(cls):
        del cls.capsProblem
        del cls.meshAIM
        #cls.cleanUp()
        pass

    @classmethod
    def cleanUp(cls):

        # Remove analysis directories
        dirs = glob.glob( cls.problemName + '*')
        for dir in dirs:
            if os.path.isdir(dir):
                shutil.rmtree(dir)

#==============================================================================
    # Test inputs
    def test_inputs(self):

        m   = pyCAPS.Unit("meter")
        deg = pyCAPS.Unit("degree")
        psi = pyCAPS.Unit("psi")

        cbaero = self.capsProblem.analysis.create(aim = "cbaeroAIM",autoExec=False)

        cbaero.input.Proj_Name = "cbaero_CAPS"
        cbaero.input.Mach = 0.5
        cbaero.input.Alpha = 10 * deg
        cbaero.input.Beta = 5 * deg
        cbaero.input.ReferenceArea = 2 * m*m
        cbaero.input.ReferenceChord = 0.6 * m
        cbaero.input.ReferenceSpan = 6 * m
        cbaero.input.Moment_Center = [1,0.1,0.2] * m
        cbaero.input.Flow_Type = "Laminar"
        cbaero.input.Critical_Transition = 9
        cbaero.input.Planet = "Jupiter"
        cbaero.input.Default_Body_Method = "TangentCone"
        cbaero.input.Default_Body_Method = "TangentWedge"
        cbaero.input.Default_Low_Speed_Method = "LowAR"
        cbaero.input.Leading_Edge_Suction = 0.8
        cbaero.input.Aero_Surface = {"horizontalCS":"Wing"}
        cbaero.input.Mesh_Morph = False
        cbaero.input.Mangler_Setting = "2D"
        cbaero.input["Surface_Mesh"].link(self.meshAIM.output["Surface_Mesh"])

        # Write input files
        cbaero.preAnalysis()

#==============================================================================
    def test_Output_v5(self):

        m      = pyCAPS.Unit("meter")
        cm     = pyCAPS.Unit("centimeter")
        watt   = pyCAPS.Unit("watt")
        kelvin = pyCAPS.Unit("kelvin")
        deg    = pyCAPS.Unit("degree")
        bar    = pyCAPS.Unit("bar")

        cbaero = self.capsProblem.analysis.create(aim = "cbaeroAIM", autoExec=False)

        cbaero.input["Surface_Mesh"].link(self.meshAIM.output["Surface_Mesh"])

        # Write input files
        cbaero.preAnalysis()

        # Create a dummy plt file
        filename = os.path.join(cbaero.analysisDir, cbaero.input.Proj_Name+".plt")
        with open(filename, "w") as f:
            f.write("BLOCK B0.000000_Q0.689476_M20.000000\n")
            f.write("  Alpha        Bars      Mach      PerTrb       CL          CD        Cm        LoD        CLp        CDp         CLf        CDf     StagTemp   QdotConv    QdotRad  StagRadius CL_Trefftz CD_Trefftz\n")
            f.write("   0.00000  0.6894757   20.00000   15.71943    0.00004    1.65778   -0.00000    0.00002    0.00004    1.65693    0.00000    0.00086    6638.99       0.10       0.01       0.10    0.10000    4.00000\n")
            f.write("   5.00000  0.6894757   20.00000   11.42210   -0.11959    1.64205   -0.02117   -0.07283   -0.11980    1.64120    0.00021    0.00086    6638.23       0.20       0.02       0.01    0.02000    0.30000\n")
            f.write("   7.00000  0.6894757   20.00000   12.70497   -0.16611    1.62714   -0.02950   -0.10209   -0.16641    1.62622    0.00029    0.00092    6639.11       0.30       0.03       0.20    0.00300    0.00200\n")
            f.write("  10.00000  0.6894757   20.00000   13.25314   -0.23335    1.59578   -0.04172   -0.14623   -0.23374    1.59478    0.00040    0.00100    6635.25       0.40       0.04       0.02    0.00040    0.00010\n")
            f.write("\n")
            f.write("\n")
            f.write("BLOCK B0.000000_Q0.689476_M25.000000\n")
            f.write("  Alpha        Bars      Mach      PerTrb       CL          CD        Cm        LoD        CLp        CDp         CLf        CDf     StagTemp   QdotConv    QdotRad  StagRadius CL_Trefftz CD_Trefftz\n")
            f.write("   0.00000  0.6894757   25.00000    1.92223    0.00004    1.66538   -0.00000    0.00002    0.00004    1.66467    0.00000    0.00070    7685.48       0.00       0.00       0.00    0.00000    0.00000\n")
            f.write("   5.00000  0.6894757   25.00000    1.67473   -0.12021    1.64958   -0.02130   -0.07287   -0.12035    1.64887    0.00014    0.00071    7684.17       0.00       0.00       0.00    0.00000    0.00000\n")
            f.write("   7.00000  0.6894757   25.00000    3.20561   -0.16694    1.63458   -0.02966   -0.10213   -0.16717    1.63382    0.00022    0.00076    7685.68       0.00       0.00       0.00    0.00000    0.00000\n")
            f.write("  10.00000  0.6894757   25.00000    4.44776   -0.23449    1.60305   -0.04195   -0.14628   -0.23481    1.60222    0.00031    0.00083    7679.05       0.00       0.00       0.00    0.00000    0.00000\n")

        filename = os.path.join(cbaero.analysisDir, cbaero.input.Proj_Name+".adb")
        with open(filename, "w") as f:
            f.write("Testing")

        cbaero.postAnalysis()

        TrueValues = {"Beta": None,
                      "Alpha": [0.0, 5.0, 7.0, 10.0, 0.0, 5.0, 7.0, 10.0]*deg,
                      "Dynamic_Pressure": [0.6894757, 0.6894757, 0.6894757, 0.6894757, 0.6894757, 0.6894757, 0.6894757, 0.6894757]*bar,
                      "Mach": [20.0, 20.0, 20.0, 20.0, 25.0, 25.0, 25.0, 25.0],
                      "PerTrb": [15.71943, 11.4221, 12.70497, 13.25314, 1.92223, 1.67473, 3.20561, 4.44776],
                      "CLtot": [4e-05, -0.11959, -0.16611, -0.23335, 4e-05, -0.12021, -0.16694, -0.23449],
                      "CDtot": [1.65778, 1.64205, 1.62714, 1.59578, 1.66538, 1.64958, 1.63458, 1.60305],
                      "CMYtot": [-0.0, -0.02117, -0.0295, -0.04172, -0.0, -0.0213, -0.02966, -0.04195],
                      "LoDtot": [2e-05, -0.07283, -0.10209, -0.14623, 2e-05, -0.07287, -0.10213, -0.14628],
                      "CL_p": [4e-05, -0.1198, -0.16641, -0.23374, 4e-05, -0.12035, -0.16717, -0.23481],
                      "CD_p": [1.65693, 1.6412, 1.62622, 1.59478, 1.66467, 1.64887, 1.63382, 1.60222],
                      "CL_v": [0.0, 0.00021, 0.00029, 0.0004, 0.0, 0.00014, 0.00022, 0.00031],
                      "CD_v": [0.00086, 0.00086, 0.00092, 0.001, 0.0007, 0.00071, 0.00076, 0.00083],
                      "Stagnation_Temperature": [6638.99, 6638.23, 6639.11, 6635.25, 7685.48, 7684.17, 7685.68, 7679.05]*kelvin,
                      "Stagnation_Radius": [0.1, 0.01, 0.2, 0.02, 0.0, 0.0, 0.0, 0.0]*m,
                      "Convective_Flux": [0.1, 0.2, 0.3, 0.4, 0.0, 0.0, 0.0, 0.0]*watt / cm**2,
                      "Radiative_Flux": [0.01, 0.02, 0.03, 0.04, 0.0, 0.0, 0.0, 0.0]*watt / cm**2,
                      "CL_Trefftz": [0.1, 0.02, 0.003, 0.0004, 0.0, 0.0, 0.0, 0.0],
                      "CD_Trefftz": [4.0, 0.3, 0.002, 0.0001, 0.0, 0.0, 0.0, 0.0]}

        self.assertEqual(TrueValues.keys(), cbaero.output.keys())

        for key in cbaero.output.keys():
            if key == "Beta":
               self.assertIsNone(cbaero.output[key].value)
            else:
               self.assertAlmostEqual(TrueValues[key], cbaero.output[key].value, 1e-5)

#==============================================================================
    def test_Output_v6(self):

        m      = pyCAPS.Unit("meter")
        cm     = pyCAPS.Unit("centimeter")
        watt   = pyCAPS.Unit("watt")
        kelvin = pyCAPS.Unit("kelvin")
        deg    = pyCAPS.Unit("degree")
        bar    = pyCAPS.Unit("bar")

        cbaero = self.capsProblem.analysis.create(aim = "cbaeroAIM", autoExec=False)

        cbaero.input["Surface_Mesh"].link(self.meshAIM.output["Surface_Mesh"])

        # Write input files
        cbaero.preAnalysis()

        # Create a dummy plt file
        filename = os.path.join(cbaero.analysisDir, cbaero.input.Proj_Name+".plt")
        with open(filename, "w") as f:
            f.write("  Beta       Alpha        Bars      Mach      PerTrb       CL          CD        Cm        LoD        CLp        CDp         CLf        CDf     StagTemp   QdotConv    QdotRad  StagRadius CL_Trefftz CD_Trefftz\n")
            f.write("BLOCK B0.000000_Q0.689476_M20.000000\n")
            f.write("   0.00001    0.00000  0.6894757   20.00000   15.71943    0.00004    1.65778   -0.00000    0.00002    0.00004    1.65693    0.00000    0.00086    6638.99       0.10       0.01       0.10    0.10000    4.00000\n")
            f.write("   0.00020    5.00000  0.6894757   20.00000   11.42210   -0.11959    1.64205   -0.02117   -0.07283   -0.11980    1.64120    0.00021    0.00086    6638.23       0.20       0.02       0.01    0.02000    0.30000\n")
            f.write("   0.00300    7.00000  0.6894757   20.00000   12.70497   -0.16611    1.62714   -0.02950   -0.10209   -0.16641    1.62622    0.00029    0.00092    6639.11       0.30       0.03       0.20    0.00300    0.00200\n")
            f.write("   0.04000   10.00000  0.6894757   20.00000   13.25314   -0.23335    1.59578   -0.04172   -0.14623   -0.23374    1.59478    0.00040    0.00100    6635.25       0.40       0.04       0.02    0.00040    0.00010\n")
            f.write("\n")
            f.write("\n")
            f.write("  Beta       Alpha        Bars      Mach      PerTrb       CL          CD        Cm        LoD        CLp        CDp         CLf        CDf     StagTemp   QdotConv    QdotRad  StagRadius CL_Trefftz CD_Trefftz\n")
            f.write("BLOCK B0.000000_Q0.689476_M25.000000\n")
            f.write("   0.00000    0.00000  0.6894757   25.00000    1.92223    0.00004    1.66538   -0.00000    0.00002    0.00004    1.66467    0.00000    0.00070    7685.48       0.00       0.00       0.00    0.00000    0.00000\n")
            f.write("   0.00000    5.00000  0.6894757   25.00000    1.67473   -0.12021    1.64958   -0.02130   -0.07287   -0.12035    1.64887    0.00014    0.00071    7684.17       0.00       0.00       0.00    0.00000    0.00000\n")
            f.write("   0.00000    7.00000  0.6894757   25.00000    3.20561   -0.16694    1.63458   -0.02966   -0.10213   -0.16717    1.63382    0.00022    0.00076    7685.68       0.00       0.00       0.00    0.00000    0.00000\n")
            f.write("   0.00000   10.00000  0.6894757   25.00000    4.44776   -0.23449    1.60305   -0.04195   -0.14628   -0.23481    1.60222    0.00031    0.00083    7679.05       0.00       0.00       0.00    0.00000    0.00000\n")

        filename = os.path.join(cbaero.analysisDir, cbaero.input.Proj_Name+".adb")
        with open(filename, "w") as f:
            f.write("Testing")

        cbaero.postAnalysis()

        TrueValues = {
            "Beta": [1e-05, 0.0002, 0.003, 0.04, 0.0, 0.0, 0.0, 0.0]*deg,
            "Alpha": [0.0, 5.0, 7.0, 10.0, 0.0, 5.0, 7.0, 10.0]*deg,
            "Dynamic_Pressure": [0.6894757, 0.6894757, 0.6894757, 0.6894757, 0.6894757, 0.6894757, 0.6894757, 0.6894757]*bar,
            "Mach": [20.0, 20.0, 20.0, 20.0, 25.0, 25.0, 25.0, 25.0],
            "PerTrb": [15.71943, 11.4221, 12.70497, 13.25314, 1.92223, 1.67473, 3.20561, 4.44776],
            "CLtot": [4e-05, -0.11959, -0.16611, -0.23335, 4e-05, -0.12021, -0.16694, -0.23449],
            "CDtot": [1.65778, 1.64205, 1.62714, 1.59578, 1.66538, 1.64958, 1.63458, 1.60305],
            "CMYtot": [-0.0, -0.02117, -0.0295, -0.04172, -0.0, -0.0213, -0.02966, -0.04195],
            "LoDtot": [2e-05, -0.07283, -0.10209, -0.14623, 2e-05, -0.07287, -0.10213, -0.14628],
            "CL_p": [4e-05, -0.1198, -0.16641, -0.23374, 4e-05, -0.12035, -0.16717, -0.23481],
            "CD_p": [1.65693, 1.6412, 1.62622, 1.59478, 1.66467, 1.64887, 1.63382, 1.60222],
            "CL_v": [0.0, 0.00021, 0.00029, 0.0004, 0.0, 0.00014, 0.00022, 0.00031],
            "CD_v": [0.00086, 0.00086, 0.00092, 0.001, 0.0007, 0.00071, 0.00076, 0.00083],
            "Stagnation_Temperature": [6638.99, 6638.23, 6639.11, 6635.25, 7685.48, 7684.17, 7685.68, 7679.05]*kelvin,
            "Stagnation_Radius": [0.1, 0.01, 0.2, 0.02, 0.0, 0.0, 0.0, 0.0]*m,
            "Convective_Flux": [0.1, 0.2, 0.3, 0.4, 0.0, 0.0, 0.0, 0.0]*watt / cm**2,
            "Radiative_Flux": [0.01, 0.02, 0.03, 0.04, 0.0, 0.0, 0.0, 0.0]*watt / cm**2,
            "CL_Trefftz": [0.1, 0.02, 0.003, 0.0004, 0.0, 0.0, 0.0, 0.0],
            "CD_Trefftz": [4.0, 0.3, 0.002, 0.0001, 0.0, 0.0, 0.0, 0.0]
            }

        self.assertEqual(TrueValues.keys(), cbaero.output.keys())

        for key in cbaero.output.keys():
            self.assertAlmostEqual(TrueValues[key], cbaero.output[key].value, 1e-5)

#==============================================================================
    @unittest.skipIf(which("cbaero") == None, "no cbaero executable")
    def test_MeshMorph(self):

        deg    = pyCAPS.Unit("degree")
        psi    = pyCAPS.Unit("psi")
        bar    = pyCAPS.Unit("bar")
        m      = pyCAPS.Unit("m")
        cm     = pyCAPS.Unit("cm")
        watt   = pyCAPS.Unit("watt")
        K      = pyCAPS.Unit("Kelvin")

        cbaero = self.capsProblem.analysis.create(aim = "cbaeroAIM", autoExec=True)

        cbaero.input["Surface_Mesh"].link(self.meshAIM.output["Surface_Mesh"])

        # Set number of cases to run simultaneously
        cbaero.input.NumParallelCase = 3;

        # Set number of threads to use to solve each case
        cbaero.input.NumThreadPerCase = 2;

        # Set AoA number
        cbaero.input.Alpha = [1.0, 3.0, 5.0] * deg

        # Set Mach number
        cbaero.input.Mach = [2.0, 2.5, 4.0]

        # Set Dynamic pressure
        cbaero.input.Dynamic_Pressure = 10 * psi

        # Set Mangler setting
        cbaero.input.Mangler_Setting = "2D"

        # Set aero surfaces
        cbaero.input.Aero_Surface = {"fuselage_base":"Base",
                                     "horizontalCS" :"Wing",
                                     "verticalCS"   :"Wing",
                                     "inlet"        :"Body",
                                     "hollowInlet"  :"Body"}

        # Set materials
        cbaero.input.Material_Group = {"mat1":{"surfaceType":"Fully Catalytic", "emissivity":0.8,"groupName":["fuselage", "fuselage_base"]},
                                       "mat2":{"surfaceType":"RCG"            , "emissivity":0.6,"groupName":["horizontalCS", "verticalCS"]},
                                       "mat3":{"surfaceType":"8"              , "emissivity":0.7,"groupName":"inlet"},
                                       "mat4":{"surfaceType":"Fully Catalytic", "emissivity":0.5,"groupName":"hollowInlet"}}

        # Turn on mesh morphing
        cbaero.input.Mesh_Morph = True

        # Explicitly run analysis
        cbaero.runAnalysis()

        cbaero.input["Surface_Mesh"].unlink()

        #for i in cbaero.output.keys():
        #    print("            \"" + str(i) + "\": " + str(cbaero.output[i].value) + ",")

        TrueValues = {
            "Beta": None,
            "Alpha": [1.0, 3.0, 5.0, 1.0, 3.0, 5.0, 1.0, 3.0, 5.0]* deg,
            "Dynamic_Pressure": [0.6894757, 0.6894757, 0.6894757, 0.6894757, 0.6894757, 0.6894757, 0.6894757, 0.6894757, 0.6894757] * bar,
            "Mach": [2.0, 2.0, 2.0, 2.5, 2.5, 2.5, 4.0, 4.0, 4.0],
            "PerTrb": [99.99365, 99.99365, 99.99365, 99.99365, 99.99365, 99.99365, 99.87326, 99.87326, 99.83023],
            "CLtot": [-406737.0, -190113.07812, 20322.45898, -269014.875, -84531.74219, 89448.09375, -134699.64062, 37123.75781, 208934.04688],
            "CDtot": [249976.875, 234501.1875, 236002.6875, 202456.875, 194977.5, 202441.28125, 141040.98438, 142102.07812, 157067.3125],
            "CMYtot": [1899404928.0, 1099663616.0, 313780736.0, 1314749312.0, 665966656.0, 48456660.0, 721284608.0, 161861696.0, -392302560.0],
            "LoDtot": [-1.6271, -0.81071, 0.08611, -1.32875, -0.43355, 0.44185, -0.95504, 0.26125, 1.33022],
            "CL_p": [-407165.8125, -190181.21875, 20619.0625, -269352.84375, -84551.41406, 89748.59375, -134883.3125, 37178.87109, 209232.28125],
            "CD_p": [234473.17188, 218905.26562, 220316.70312, 189052.65625, 181479.09375, 188835.0625, 131843.98438, 132840.65625, 147737.60938],
            "CL_v": [428.78583, 68.12714, -296.60181, 337.94275, 19.67566, -300.49786, 183.65225, -55.11328, -298.22186],
            "CD_v": [15503.69824, 15595.92969, 15685.97559, 13404.21484, 13498.40625, 13606.21484, 9196.99219, 9261.41992, 9329.69922],
            "Stagnation_Temperature": [395.92, 395.7, 395.26, 482.14, 481.9, 481.42, 873.21, 872.77, 871.88] * K,
            "Stagnation_Radius": [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0] * m,
            "Convective_Flux": [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0] * watt / cm**2,
            "Radiative_Flux": [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0] * watt / cm**2,
            "CL_Trefftz": [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            "CD_Trefftz": [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            }

        self.assertEqual(TrueValues.keys(), cbaero.output.keys())
        for key in cbaero.output.keys():
            self.assertAlmostEqual(TrueValues[key], cbaero.output[key].value, 1e-5)

        # Change the geometry
        self.capsProblem.geometry.despmtr.section1_x *= 1.1

        # Explicitly run analysis again with a morphed mesh
        cbaero.runAnalysis()

        # for i in cbaero.output.keys():
        #     print("            \"" + str(i) + "\": " + str(cbaero.output[i].value) + ",")

        TrueValues = {
            "Beta": None,
            "Alpha": [1.0, 3.0, 5.0, 1.0, 3.0, 5.0, 1.0, 3.0, 5.0] * deg,
            "Dynamic_Pressure": [0.6894757, 0.6894757, 0.6894757, 0.6894757, 0.6894757, 0.6894757, 0.6894757, 0.6894757, 0.6894757] * bar,
            "Mach": [2.0, 2.0, 2.0, 2.5, 2.5, 2.5, 4.0, 4.0, 4.0],
            "PerTrb": [99.99364, 99.99364, 99.99364, 99.99364, 99.99364, 99.99364, 99.87309, 99.87309, 99.83002],
            "CLtot": [-398254.71875, -181818.20312, 28389.18555, -260135.57812, -75853.78125, 97889.88281, -126436.25, 45459.90234, 217840.84375],
            "CDtot": [254089.01562, 239485.67188, 241841.9375, 206752.35938, 200181.42188, 208537.29688, 145571.625, 147515.1875, 163454.03125],
            "CMYtot": [1879025536.0, 1077418752.0, 289702720.0, 1293413760.0, 642695040.0, 23264888.0, 699311680.0, 138078160.0, -419467392.0],
            "LoDtot": [-1.56738, -0.7592, 0.11739, -1.2582, -0.37893, 0.46941, -0.86855, 0.30817, 1.33273],
            "CL_p": [-398681.96875, -181883.5625, 28689.63281, -260469.85938, -75869.90625, 98193.89062, -126612.875, 45523.35938, 218146.25],
            "CD_p": [238590.9375, 223896.48438, 226163.76562, 193346.65625, 186689.5, 194942.89062, 136367.01562, 138249.78125, 154130.57812],
            "CL_v": [427.24387, 65.37384, -300.45056, 334.27127, 16.12585, -304.00134, 176.61804, -63.45651, -305.39841],
            "CD_v": [15498.05957, 15589.19824, 15678.17676, 13405.71289, 13491.91895, 13594.42969, 9204.61426, 9265.41211, 9323.47949],
            "Stagnation_Temperature": [395.92, 395.7, 395.26, 482.14, 481.9, 481.42, 873.21, 872.77, 871.88] * K,
            "Stagnation_Radius": [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0] * m,
            "Convective_Flux": [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0] * watt/cm**2,
            "Radiative_Flux": [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0] * watt/cm**2,
            "CL_Trefftz": [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            "CD_Trefftz": [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            }

        self.assertEqual(TrueValues.keys(), cbaero.output.keys())
        for key in cbaero.output.keys():
            self.assertAlmostEqual(TrueValues[key], cbaero.output[key].value, 1e-5)

        # Change the geometry back
        self.capsProblem.geometry.despmtr.section1_x /= 1.1


#==============================================================================
    @unittest.skipIf(which("cbaero") == None, "no cbaero executable")
    def test_HeatFlux(self):

        deg    = pyCAPS.Unit("degree")
        psi    = pyCAPS.Unit("psi")
        bar    = pyCAPS.Unit("bar")
        m      = pyCAPS.Unit("m")
        cm     = pyCAPS.Unit("cm")
        watt   = pyCAPS.Unit("watt")
        K      = pyCAPS.Unit("Kelvin")

        cbaero = self.capsProblem.analysis.create(aim = "cbaeroAIM", autoExec=True)

        cbaero.input["Surface_Mesh"].link(self.meshAIM.output["Surface_Mesh"])

        # Set number of cases to run simultaneously
        cbaero.input.NumParallelCase = 3;

        # Set number of threads to use to solve each case
        cbaero.input.NumThreadPerCase = 2;

        # Set AoA number
        cbaero.input.Alpha = [1.0, 3.0, 5.0] * deg

        # Set Mach number
        cbaero.input.Mach = [2.0, 2.5, 4.0]

        # Set Dynamic pressure
        cbaero.input.Dynamic_Pressure = [10, 20] * psi

        # Set Mangler setting
        cbaero.input.Mangler_Setting = "2D"

        # Set aero surfaces
        cbaero.input.Aero_Surface = {"fuselage_base":"Base",
                                     "horizontalCS" :"Wing",
                                     "verticalCS"   :"Wing",
                                     "inlet"        :"Body",
                                     "hollowInlet"  :"Body"}

        boundNames = ["fuselage"]
        for boundName in boundNames:
            # Create the bound
            bound = self.capsProblem.bound.create(boundName)

            fluidVset = bound.vertexSet.create(cbaero)

            # Create HeatFlux data sets
            for IMach in range(len(cbaero.input.Mach)):
                for IDynp in range(len(cbaero.input.Dynamic_Pressure)):
                    for IAlpha in range(len(cbaero.input.Alpha)):
                        fluidVset.dataSet.create(f"HeatFlux_IMach_{IMach}_IDynp_{IDynp}_IAlpha_{IAlpha}")

            # Close the bound as complete (cannot create more vertex or data sets)
            bound.close()

        # Explicitly run analysis
        cbaero.runAnalysis()

        for boundName in boundNames:
            # Create the bound
            bound = self.capsProblem.bound[boundName]

            # Get data sets
            for IMach in range(len(cbaero.input.Mach)):
                for IDynp in range(len(cbaero.input.Dynamic_Pressure)):
                    for IAlpha in range(len(cbaero.input.Alpha)):
                        fluidVset_HeatFlux = fluidVset.dataSet[f"HeatFlux_IMach_{IMach}_IDynp_{IDynp}_IAlpha_{IAlpha}"]

                        # Simply try extracting the data
                        fluidVset_HeatFlux.data()

if __name__ == '__main__':
    unittest.main()
