/* ToDo
   * adjust internal control points for tangency
   * flend13 cases fail (normal parallel to A vector)
 */

/*
 ************************************************************************
 *                                                                      *
 * udfFlend3 -- create a flend on an attributed Body                    *
 *                                                                      *
 *            Written by John Dannenhoffer @ Syracuse University        *
 *            Patterned after code written by Bob Haimes  @ MIT         *
 *                                                                      *
 ************************************************************************
 */

/*
 * Copyright (C) 2011/2026  John F. Dannenhoffer, III (Syracuse University)
 *
 * This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *     MA  02110-1301  USA
 */

//#define DEBUG 1

#define NUMUDPARGS       5
#define NUMUDPINPUTBODYS 1
#include "udpUtilities.h"

/* shorthands for accessing argument values and velocities */
#define ATTR1IN(IUDP)  ((char   *) (udps[IUDP].arg[0].val))
#define ATTR2IN(IUDP)  ((char   *) (udps[IUDP].arg[1].val))
#define FRAC1(  IUDP)  ((double *) (udps[IUDP].arg[2].val))[0]
#define FRAC2(  IUDP)  ((double *) (udps[IUDP].arg[3].val))[0]
#define ATTROUT(IUDP)  ((char   *) (udps[IUDP].arg[4].val))

/* data about possible arguments */
static char  *argNames[NUMUDPARGS] = {"attr1in",  "attr2in",  "frac1",  "frac2",  "attrout",  };
static int    argTypes[NUMUDPARGS] = {ATTRSTRING, ATTRSTRING, ATTRREAL, ATTRREAL, ATTRSTRING, };
static int    argIdefs[NUMUDPARGS] = {0,          0,          0,        0,        0,          };
static double argDdefs[NUMUDPARGS] = {0.,         0.,         0.50,     0.50,     0.,         };

/* get utility routines: udpErrorStr, udpInitialize, udpReset, udpSet,
                         udpGet, udpVel, udpClean, udpMesh */
#include "udpUtilities.c"

#include "OpenCSM.h"
#include "mateContours.h"

/* prototype of unpublished EGADS function */
extern int EG_relPosTs(ego ecurve, int npnt, /*@null@*/ const double newS[],
                       double newT[], double newXYZ[]);

/* prototypes of routine defined below */
static int setupNodeInfo(ego ebody, char attr1[], char attr2[], int nodeCount[], double nodeInfo[]);

static int getXYZforEntity(ego ebody, int iedge, int isens, char attr[], double nodeInfo[], double xyz[]);

static int makeFlendCurve(ego context, double xyz1[], double xyz2[], double frac1, double frac2, double cpout[]);

/* message (to be shared by all functions) */
static char message[1024];


/*
 ************************************************************************
 *                                                                      *
 *   udpExecute - execute the primitive                                 *
 *                                                                      *
 ************************************************************************
 */

int
udpExecute(ego  emodel,                 /* (in)  Model containing Body */
           ego  *ebody,                 /* (out) Body pointer */
           int  *nMesh,                 /* (out) number of associated meshes */
           char *string[])              /* (out) error message */
{
    int     status = EGADS_SUCCESS;

    int     oclass, mtype, nchild, *senses, periodic, attrType, attrLen;
    int     ncontour, *nent=NULL, *contr1=NULL, *contr2=NULL;
    int     ipair, icontour, ient, iedge1, isens1, iedge2, isens2;
    int     count1, count2, nfoo, ifoo, icp;
    int     nnode, iedge, nedge, nface, brchattr[2], bodyattr[2];
    int     ibody, ntemp = 0, header[4], sense[4], fillStyle;
    int     nlist=0, iface, attrs[2], inode;
    int     *nodeCount=NULL;
    CINT    *tempIlist;
    double  xyz0[18], xyz1[18], xyz2[18], xyz3[18], data[18];
    double  value, dot, cps[12], rdata[20], trange[4], uvrange[4];
    double  *nodeInfo=NULL;
    CDOUBLE *tempRlist;
    char    tempstr[MAX_STR_LEN];
    CCHAR   *tempClist;
    ego     context, eref, *ebodys, *echilds;
    ego     *enodes=NULL, *eedges=NULL, *efaces=NULL, *efoo=NULL, *faceList=NULL;
    ego     etemp[4], enode, eedge, ecurve, eloop, esurface, eedge01, eedge23;
    udp_T   *udps = *Udps;
    void    *modl;

#ifdef DEBUG
    int     i, j, k;
#endif

    ROUTINE(udpExecute);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    printf("udpExecute(emodel=%llx)\n", (long long)emodel);
    printf("attr1in(0) = %s\n", ATTR1IN(0));
    printf("attr2in(0) = %s\n", ATTR2IN(0));
    printf("frac1(  0) = %f\n", FRAC1(  0));
    printf("frac2(  0) = %f\n", FRAC2(  0));
    printf("attrout(0) = %s\n", ATTROUT(0));
#endif

    /* default return values */
    *ebody  = NULL;
    *nMesh  = 0;
    *string = NULL;

    message[0] = '\0';

    /* get context and pointer to the OpenCSM MODL */
    status = EG_getContext(emodel, &context);
    CHECK_STATUS(EG_getContext);

    status = EG_getUserPointer(context, (void**)(&(modl)));
    CHECK_STATUS(EG_getUserPointer);

    /* check that Model was input that contains one Body */
    status = EG_getTopology(emodel, &eref, &oclass, &mtype,
                            data, &nchild, &ebodys, &senses);
    CHECK_STATUS(EG_getTopology);

    if (oclass != MODEL) {
        snprintf(message, 1024, "expecting a Model\n");
        status = EGADS_NOTMODEL;
        goto cleanup;
    } else if (nchild != 1) {
        snprintf(message, 1024, "expecting Model to contain one Body (not %d)\n", nchild);
        status = EGADS_NOTBODY;
        goto cleanup;
    }

    /* verify that the given attributes are different */
    if (strcmp(ATTR1IN(0), ATTR2IN(0)) == 0) {
        snprintf(message, 1024, "attr1 and attr2 are the same\n");
        status = EGADS_RANGERR;
        goto cleanup;
    }

    /* make sure input is a SheetBody or SolidBody */
    status = EG_getTopology(ebodys[0], &eref, &oclass, &mtype,
                            data, &nchild, &echilds, &senses);
    CHECK_STATUS(EG_getTopology);

    if (mtype != FACEBODY && mtype != SHEETBODY && mtype != SOLIDBODY) {
        snprintf(message, 1024, "Body is not a SheetBody or SolidBody\n");
        status = EGADS_NOTBODY;
        goto cleanup;
    }

    /* cache copy of arguments for future use */
    status = cacheUdp(NULL);
    CHECK_STATUS(cacheUdp);

#ifdef DEBUG
    printf("attr1in(%d) = %s\n", numUdp, ATTR1IN(numUdp));
    printf("attr2in(%d) = %s\n", numUdp, ATTR2IN(numUdp));
    printf("frac1(  %d) = %f\n", numUdp, FRAC1(  numUdp));
    printf("frac2(  %d) = %f\n", numUdp, FRAC2(  numUdp));
    printf("attrout(%d) = %s\n", numUdp, ATTROUT(numUdp));
#endif

    /* summarize Edges and Faces with ATTR1IN and ATTR2IN */
#ifdef DEBUG
    status = EG_getBodyTopos(ebodys[0], NULL, EDGE, &nedge, &eedges);
    CHECK_STATUS(EG_getBodyTopos);

    status = EG_getBodyTopos(ebodys[0], NULL, FACE, &nface, &efaces);
    CHECK_STATUS(EG_getBodyTopos);

    SPLINT_CHECK_FOR_NULL(eedges);
    SPLINT_CHECK_FOR_NULL(efaces);

    for (iedge = 0; iedge < nedge; iedge++) {
        status = EG_attributeRet(eedges[iedge], ATTR1IN(0), &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) {
            printf("Edge %5d has attribute %s\n", iedge+1, ATTR1IN(0));
        }
    }

    for (iface = 0; iface < nface; iface++) {
        status = EG_attributeRet(efaces[iface], ATTR1IN(0), &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) {
            printf("Face %5d has attribute %s\n", iface+1, ATTR1IN(0));
        }
    }

    for (iedge = 0; iedge < nedge; iedge++) {
        status = EG_attributeRet(eedges[iedge], ATTR2IN(0), &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) {
            printf("Edge %5d has attribute %s\n", iedge+1, ATTR2IN(0));
        }
    }

    for (iface = 0; iface < nface; iface++) {
        status = EG_attributeRet(efaces[iface], ATTR2IN(0), &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) {
            printf("Face %5d has attribute %s\n", iface+1, ATTR2IN(0));
        }
    }

    EG_free(eedges);   eedges = NULL;
    EG_free(efaces);   efaces = NULL;
#endif


    /* add Nodes to the input Body so that the Loops with the given
       attributes have matching Nodes (which may involve splitting
       Edges)*/
    status = EG_mateContours(ebodys[0], ATTR1IN(0), ATTR2IN(0),
                             ebody, &ncontour, &nent, &contr1, &contr2);
    CHECK_STATUS(EG_mateContours);

#ifdef DEBUG
    printf("*body after EG_mateContours:\n");
    ocsmPrintEgo(*ebody);
#endif

    SPLINT_CHECK_FOR_NULL(*ebody);
    SPLINT_CHECK_FOR_NULL(nent  );
    SPLINT_CHECK_FOR_NULL(contr1);
    SPLINT_CHECK_FOR_NULL(contr2);

    /* get tangency information at all Nodes associated with
       Edges that have either ATR1IN or ATTR2IN */
    status = EG_getBodyTopos(*ebody, NULL, NODE, &nnode, &enodes);
    CHECK_STATUS(EG_getBodyTopos);

    EG_free(enodes);   enodes = NULL;

    MALLOC(nodeCount, int,      nnode);
    MALLOC(nodeInfo,  double, 9*nnode);

    for (inode = 0; inode < nnode; inode++) {
        nodeCount[  inode  ] = 0;
        nodeInfo[ 9*inode  ] = 0;
        nodeInfo[ 9*inode+1] = 0;
        nodeInfo[ 9*inode+2] = 0;
        nodeInfo[ 9*inode+3] = 0;
        nodeInfo[ 9*inode+4] = 0;
        nodeInfo[ 9*inode+5] = 0;
        nodeInfo[ 9*inode+6] = 0;
        nodeInfo[ 9*inode+7] = 0;
        nodeInfo[ 9*inode+8] = 0;
    }

    status = setupNodeInfo(*ebody, ATTR1IN(0), ATTR2IN(0), nodeCount, nodeInfo);
    CHECK_STATUS(setupNodeInfo);


    /* print the matching Edges/Nodes */
#ifdef DEBUG
    printf("ncontour=%d\n", ncontour);
    k = 0;
    for (i = 0; i < ncontour; i++) {
        for (j = 0; j < nent[i]; j++) {
            printf("i=%3d, j=%3d, contr1=%5d,%5d contr2=%5d,%5d\n", i, j,
                   contr1[2*k], contr1[2*k+1], contr2[2*k], contr2[2*k+1]);
            k++;
        }
    }
#endif

    /* go to cleanup here to see the results of EG_mateContours (before
       the flend Faces are added */
//$$$    goto cleanup;

    /* get the Nodes, Edges, and Faces associated with the new Body */
    SPLINT_CHECK_FOR_NULL(*ebody);

    status = EG_getBodyTopos(*ebody, NULL, NODE, &nnode, &enodes);
    CHECK_STATUS(EG_getBodyTopos);

    status = EG_getBodyTopos(*ebody, NULL, EDGE, &nedge, &eedges);
    CHECK_STATUS(EG_getBodyTopos);

    status = EG_getBodyTopos(*ebody, NULL, FACE, &nface, &efaces);
    CHECK_STATUS(EG_getBodyTopos);

    SPLINT_CHECK_FOR_NULL(enodes);
    SPLINT_CHECK_FOR_NULL(eedges);
    SPLINT_CHECK_FOR_NULL(efaces);

    /* initialize the attributes that will be put on the new Faces */
    status = ocsmEvalExpr(modl, "@nbody", &value, &dot, tempstr);
    CHECK_STATUS(ocsmEvalExpr);

    ibody = value + 1.01;

    brchattr[0] = -1;                   // fixed in buildPrimitive because _markFaces_ is not set
    bodyattr[0] = ibody;

    /* any Edge that has both Faces associated with ATTR1IN should also get ATTR1IN.
       same for ATTR2IN */
    for (iedge = 0; iedge < nedge; iedge++) {
        status = EG_getBodyTopos(*ebody, eedges[iedge], FACE, &nfoo, &efoo);
        CHECK_STATUS(EG_getBodyTopos);

        SPLINT_CHECK_FOR_NULL(efoo);

        count1 = 0;
        count2 = 0;

        for (ifoo = 0; ifoo < nfoo; ifoo++) {
            status = EG_attributeRet(efoo[ifoo], ATTR1IN(0), &attrType, &attrLen,
                                     &tempIlist, &tempRlist, &tempClist);
            if (status == EGADS_SUCCESS) {
                count1++;
            }

            status = EG_attributeRet(efoo[ifoo], ATTR2IN(0), &attrType, &attrLen,
                                     &tempIlist, &tempRlist, &tempClist);
            if (status == EGADS_SUCCESS) {
                count2++;
            }
        }

        if (count1 >= nfoo) {
            status = EG_attributeAdd(eedges[iedge], ATTR1IN(0), ATTRINT, 1,
                                     &count1, NULL, NULL);
            CHECK_STATUS(EG_attributeAdd);
        }

        if (count2 >= nfoo) {
            status = EG_attributeAdd(eedges[iedge], ATTR2IN(0), ATTRINT, 1,
                                     &count2, NULL, NULL);
            CHECK_STATUS(EG_attributeAdd);
        }

        EG_free(efoo);   efoo = NULL;
    }

    /* allocate a list of the FAces to sew together */
    nlist = nface;
    for (icontour = 0; icontour < ncontour; icontour++) {
        nlist += nent[icontour];
    }

    MALLOC(faceList, ego, nlist);

    /* build a list of the Faces that are not associated with ATTR1IN or ATTR2IN */
    nlist = 0;
    for (iface = 0; iface < nface; iface++) {
        status = EG_attributeRet(efaces[iface], ATTR1IN(0), &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) continue;

        status = EG_attributeRet(efaces[iface], ATTR2IN(0), &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) continue;

        faceList[nlist++] = efaces[iface];
    }

    /* loop through each set of contours (eg, separate flends on the
       two sides of the aircraft) */
    ipair  = 0;
    for (icontour = 0; icontour < ncontour; icontour++) {

        /* make a Face for each entity pair */
        for (ient = 0; ient < nent[icontour]; ient++) {
            iedge1 = contr1[2*ipair  ];
            isens1 = contr1[2*ipair+1];
            iedge2 = contr2[2*ipair  ];
            isens2 = contr2[2*ipair+1];

            /* get the Nodes associated with this entity in loop1 */
            if (isens1 == 0) {
                status = EG_objectBodyTopo(*ebody, NODE, iedge1, &enode);
                CHECK_STATUS(EG_objectBodyTopo);

                etemp[3] = enode;
                etemp[0] = enode;
            } else {
                status = EG_objectBodyTopo(*ebody, EDGE, iedge1, &eedge);
                CHECK_STATUS(EG_objectBodyTopo);

                status = EG_getTopology(eedge, &eref, &oclass, &mtype,
                                        data, &nchild, &echilds, &senses);
                CHECK_STATUS(EG_getTopology);

                if (isens1 > 0) {
                    etemp[3] = echilds[0];
                    etemp[0] = echilds[1];
                } else {
                    etemp[0] = echilds[0];
                    etemp[3] = echilds[1];
                }
            }

            /* get the Nodes associated with this entity in loop2 */
            if (isens2 == 0) {
                status = EG_objectBodyTopo(*ebody, NODE, iedge2, &enode);
                CHECK_STATUS(EG_objectBodyTopo);

                etemp[1] = enode;
                etemp[2] = enode;
            } else {
                status = EG_objectBodyTopo(*ebody, EDGE, iedge2, &eedge);
                CHECK_STATUS(EG_objectBodyTopo);

                status = EG_getTopology(eedge, &eref, &oclass, &mtype,
                                        data, &nchild, &echilds, &senses);
                CHECK_STATUS(EG_getTopology);

                if (isens2 < 0) {
                    etemp[1] = echilds[0];
                    etemp[2] = echilds[1];
                } else {
                    etemp[2] = echilds[0];
                    etemp[1] = echilds[1];
                }
            }

            /* set up the control points for the curve between Nodes 0 and 1 */
            status = getXYZforEntity(*ebody, iedge1, +isens1, ATTR1IN(0), nodeInfo, xyz0);
            CHECK_STATUS(getXYZforEntity);

            status = getXYZforEntity(*ebody, iedge2, +isens2, ATTR2IN(0), nodeInfo, xyz1);
            CHECK_STATUS(getXYZforEntity);

#ifdef DEBUG
            printf("\nxyz0:  %10.5f %10.5f %10.5f\n", xyz0[0], xyz0[1], xyz0[2]);
            printf(  "       %10.5f %10.5f %10.5f\n", xyz0[3], xyz0[4], xyz0[5]);
            printf(  "xyz1:  %10.5f %10.5f %10.5f\n", xyz1[0], xyz1[1], xyz1[2]);
            printf(  "       %10.5f %10.5f %10.5f\n", xyz1[3], xyz1[4], xyz1[5]);
#endif

            status = makeFlendCurve(context, xyz0, xyz1, FRAC1(0), FRAC2(0), cps);
            CHECK_STATUS(makeFlendCurve);

#ifdef DEBUG
            printf(  "cps:   %10.5f %10.5f %10.5f\n", cps[ 0], cps[ 1], cps[ 2]);
            printf(  "       %10.5f %10.5f %10.5f\n", cps[ 3], cps[ 4], cps[ 5]);
            printf(  "       %10.5f %10.5f %10.5f\n", cps[ 6], cps[ 7], cps[ 8]);
            printf(  "       %10.5f %10.5f %10.5f\n", cps[ 9], cps[10], cps[11]);
#endif

            /* make the Bspline Edge (eedge01) between Nodes 0 and 1 */
            header[0] = 0;
            header[1] = 3;
            header[2] = 4;
            header[3] = 8;

            rdata[ 0] = 0;
            rdata[ 1] = 0;
            rdata[ 2] = 0;
            rdata[ 3] = 0;
            rdata[ 4] = 1;
            rdata[ 5] = 1;
            rdata[ 6] = 1;
            rdata[ 7] = 1;

            for (icp = 0; icp < 12; icp++) {
                rdata[8+icp] = cps[icp];
            }

#ifdef DEBUG
            printf("rdata: %10.5f %10.5f %10.5f\n", rdata[ 8], rdata[ 9], rdata[10]);
            printf("       %10.5f %10.5f %10.5f\n", rdata[11], rdata[12], rdata[13]);
            printf("       %10.5f %10.5f %10.5f\n", rdata[14], rdata[15], rdata[16]);
            printf("       %10.5f %10.5f %10.5f\n", rdata[17], rdata[18], rdata[19]);
#endif

            status = EG_makeGeometry(context, CURVE, BSPLINE, NULL, header, rdata, &ecurve);
            CHECK_STATUS(EG_makeGeometry);

            status = EG_getRange(ecurve, trange, &periodic);
            CHECK_STATUS(EG_getRange);

            status = EG_makeTopology(context, ecurve, EDGE, TWONODE, trange,
                                     2, &etemp[0], NULL, &eedge01);
            CHECK_STATUS(EG_makeTopology);

            /* set up the control points for the curve between Nodes 2 and 3 */
            status = getXYZforEntity(*ebody, iedge2, -isens2, ATTR2IN(0), nodeInfo, xyz2);
            CHECK_STATUS(getXYZforEntity);

            status = getXYZforEntity(*ebody, iedge1, -isens1, ATTR1IN(0), nodeInfo, xyz3);
            CHECK_STATUS(getXYZforEntity);

#ifdef DEBUG
            printf("\nxyz2:  %10.5f %10.5f %10.5f\n", xyz2[0], xyz2[1], xyz2[2]);
            printf(  "       %10.5f %10.5f %10.5f\n", xyz2[3], xyz2[4], xyz2[5]);
            printf(  "xyz3:  %10.5f %10.5f %10.5f\n", xyz3[0], xyz3[1], xyz3[2]);
            printf(  "       %10.5f %10.5f %10.5f\n", xyz3[3], xyz3[4], xyz3[5]);
#endif

            status = makeFlendCurve(context, xyz2, xyz3, FRAC2(0), FRAC1(0), cps);
            CHECK_STATUS(makeFlendCurve);

#ifdef DEBUG
            printf(  "cps:   %10.5f %10.5f %10.5f\n", cps[ 0], cps[ 1], cps[ 2]);
            printf(  "       %10.5f %10.5f %10.5f\n", cps[ 3], cps[ 4], cps[ 5]);
            printf(  "       %10.5f %10.5f %10.5f\n", cps[ 6], cps[ 7], cps[ 8]);
            printf(  "       %10.5f %10.5f %10.5f\n", cps[ 9], cps[10], cps[11]);
#endif

            /* make the Bspline Edge (eedge23) between Nodes 2 and 3 */
            header[0] = 0;
            header[1] = 3;
            header[2] = 4;
            header[3] = 8;

            rdata[ 0] = 0;
            rdata[ 1] = 0;
            rdata[ 2] = 0;
            rdata[ 3] = 0;
            rdata[ 4] = 1;
            rdata[ 5] = 1;
            rdata[ 6] = 1;
            rdata[ 7] = 1;

            for (icp = 0; icp < 12; icp++) {
                rdata[8+icp] = cps[icp];
            }

#ifdef DEBUG
            printf("rdata: %10.5f %10.5f %10.5f\n", rdata[ 8], rdata[ 9], rdata[10]);
            printf("       %10.5f %10.5f %10.5f\n", rdata[11], rdata[12], rdata[13]);
            printf("       %10.5f %10.5f %10.5f\n", rdata[14], rdata[15], rdata[16]);
            printf("       %10.5f %10.5f %10.5f\n", rdata[17], rdata[18], rdata[19]);
#endif

            status = EG_makeGeometry(context, CURVE, BSPLINE, NULL, header, rdata, &ecurve);
            CHECK_STATUS(EG_makeGeometry);

            status = EG_getRange(ecurve, trange, &periodic);
            CHECK_STATUS(EG_getRange);

            status = EG_makeTopology(context, ecurve, EDGE, TWONODE, trange,
                                     2, &etemp[2], NULL, &eedge23);
            CHECK_STATUS(EG_makeTopology);

            /* build the Loop */
            if (isens1 == 0) {
                ntemp = 3;

                etemp[0] = eedge01;
                sense[0] = SFORWARD;

                etemp[1] = eedges[iedge2-1];
                sense[1] = -isens2;

                etemp[2] = eedge23;
                sense[2] = SFORWARD;
            } else if (isens2 == 0) {
                ntemp = 3;

                etemp[0] = eedge23;
                sense[0] = SFORWARD;

                etemp[1] = eedges[iedge1-1];
                sense[1] = isens1;

                etemp[2] = eedge01;
                sense[2] = SFORWARD;
            } else {
                ntemp = 4;

                etemp[3] = eedges[iedge1-1];
                sense[3] = isens1;

                etemp[0] = eedge01;
                sense[0] = SFORWARD;

                etemp[1] = eedges[iedge2-1];
                sense[1] = -isens2;

                etemp[2] = eedge23;
                sense[2] = SFORWARD;
            }

            status = EG_makeTopology(context, NULL, LOOP, CLOSED, NULL,
                                     ntemp, etemp, sense, &eloop);
            CHECK_STATUS(EG_makeTopology);

            /* build the surface and flip its U if there is a Node
               in contour2 (to make the U's consistent with the
               other Faces in the flend */

            fillStyle = 0;

            /* note: fillStyle=1 looks better but cannot be applied
               when the top and/or bottom are lines.  if we want to
               use fillStyle=1, then a change will be need to be made
               in EG_isoCline to convert linear Bsplines to cubic
               Bsplines */
            status = EG_isoCline(eloop, fillStyle, 0.0, &esurface);
            CHECK_STATUS(EG_isoCline);

            if (isens2 == 0) {
                etemp[0] = esurface;

                status = EG_flipObject(etemp[0], &esurface);
                CHECK_STATUS(EG_flipObject);
            }

            /* build the Face and attribute it for OpenCSM */
            status = EG_getRange(esurface, uvrange, &periodic);
            CHECK_STATUS(EG_getRange);

            status = EG_makeFace(esurface, SFORWARD, uvrange, &faceList[nlist++]);
            CHECK_STATUS(EG_makeFace);

            brchattr[1] = icontour * 1000 + ient;
            bodyattr[1] = icontour * 1000 + ient;

            status = EG_attributeAdd(faceList[nlist-1], "_brch", ATTRINT, 2,
                                     brchattr, NULL, NULL);
            CHECK_STATUS(EG_attributeAdd);

            status = EG_attributeAdd(faceList[nlist-1], "_body", ATTRINT, 2,
                                     bodyattr, NULL, NULL);
            CHECK_STATUS(EG_attributeAdd);

            if (strlen(ATTROUT(0)) > 0) {
                attrs[0] = icontour;
                attrs[1] = ient;

                status = EG_attributeAdd(faceList[nlist-1], ATTROUT(0), ATTRINT, 2,
                                         attrs, NULL, NULL);
                CHECK_STATUS(EG_attributeAdd);
            }

            /* get ready for next entity */
            ipair++;
        }
    }

    /* sew the Faces together and extract the only Body */
    status = EG_sewFaces(nlist, faceList, 1e-4, 0, &emodel);
    CHECK_STATUS(EG_sewFaces);

    status = EG_getTopology(emodel, &eref, &oclass, &mtype,
                            data, &nchild, &echilds, &senses);
    CHECK_STATUS(EG_getTopology);

    if (nchild > 1) {
        printf("error: nchild=%d\n", nchild);
        exit(0);
    }

    /* update nnode, nedge, and nface */
    EG_free(enodes);
    EG_free(eedges);
    EG_free(efaces);

    status = EG_getBodyTopos(echilds[0], NULL, NODE, &nnode, &enodes);
    CHECK_STATUS(EG_getBodyTopos);

    status = EG_getBodyTopos(echilds[0], NULL, EDGE, &nedge, &eedges);
    CHECK_STATUS(EG_getBodyTopos);

    status = EG_getBodyTopos(echilds[0], NULL, FACE, &nface, &efaces);
    CHECK_STATUS(EG_getBodyTopos);

    SPLINT_CHECK_FOR_NULL(enodes);
    SPLINT_CHECK_FOR_NULL(eedges);
    SPLINT_CHECK_FOR_NULL(efaces);

    /* loop through all Edges that are between newly-created flend Faces */
    for (iedge = 0; iedge < nedge; iedge++) {
        status = EG_getInfo(eedges[iedge], &oclass, &mtype, NULL, NULL, NULL);
        CHECK_STATUS(EG_getInfo);
        if (mtype == DEGENERATE) continue;

        status = EG_getBodyTopos(echilds[0], eedges[iedge], FACE, &nfoo, &efoo);
        CHECK_STATUS(EG_getBodyTopos);

        if (nfoo < 2) {
            EG_free(efoo);
            continue;
        }

        SPLINT_CHECK_FOR_NULL(efoo);

        status = EG_attributeRet(efoo[0], "_body", &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        CHECK_STATUS(EG_attributeRet);
        if (attrType != ATTRINT || tempIlist[0] != ibody) {
            EG_free(efoo);
            continue;
        }

        status = EG_attributeRet(efoo[1], "_body", &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        CHECK_STATUS(EG_attributeRet);
        if (attrType != ATTRINT || tempIlist[0] != ibody) {
            EG_free(efoo);
            continue;
        }

        EG_free(efoo);

        if (strlen(ATTROUT(0)) > 0) {
            status = EG_attributeAdd(eedges[iedge], ATTROUT(0), ATTRINT, 1,
                                     &iedge, NULL, NULL);
            CHECK_STATUS(EG_attributeAdd);
        }
    }

    /* remove arrays created by EG_mateContours */
    EG_free(nent  );
    EG_free(contr1);
    EG_free(contr2);

    *ebody = echilds[0];
    /* set the output value */

    /* return the new Body */
    udps[numUdp].ebody = *ebody;

cleanup:
    FREE(nodeCount);
    FREE(nodeInfo );

    if (enodes != NULL) EG_free(enodes);
    if (eedges != NULL) EG_free(eedges);
    if (efaces != NULL) EG_free(efaces);

    FREE(faceList);

    if (strlen(message) > 0) {
        MALLOC(*string, char, 1024);
        strncpy(*string, message, 1023);
    } else if (status != EGADS_SUCCESS) {
        *string = udpErrorStr(status);
    }

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   udpSensitivity - return sensitivity derivatives for the "real" argument *
 *                                                                      *
 ************************************************************************
 */

int
udpSensitivity(ego    ebody,            /* (in)  Body pointer */
   /*@unused@*/int    npnt,             /* (in)  number of points */
   /*@unused@*/int    entType,          /* (in)  OCSM entity type */
   /*@unused@*/int    entIndex,         /* (in)  OCSM entity index (bias-1) */
   /*@unused@*/double uvs[],            /* (in)  parametric coordinates for evaluation */
   /*@unused@*/double vels[])           /* (out) velocities */
{
    int iudp, judp;

    ROUTINE(udpSensitivity);

    /* --------------------------------------------------------------- */

    /* check that ebody matches one of the ebodys */
    iudp = 0;
    for (judp = 1; judp <= numUdp; judp++) {
        if (ebody == udps[judp].ebody) {
            iudp = judp;
            break;
        }
    }
    if (iudp <= 0) {
        return EGADS_NOTMODEL;
    }

    /* this routine is not written yet */
    return EGADS_NOLOAD;
}


/*
 ************************************************************************
 *                                                                      *
 *   setupNodeInfo - set up XYZ, norm and tang at each Node             *
 *                                                                      *
 ************************************************************************
 */

static int
setupNodeInfo(ego    ebody,             /* (in)  Body */
              char   attr1[],           /* (in)  first  attribute */
              char   attr2[],           /* (in)  second attribute */
              int    nodeCount[],       /* (both) number of Edges incident to Node */
              double nodeInfo[])        /* (both) xyz, norm, and tang at each Node */
{
    int status = EGADS_SUCCESS;

    int     nloop, iloop, nface, iface, nedge, iedge, nnode, inode;
    int     oclass, mtype, attrType, attrLen, *senses;
    int     ntempEdge, ntempNode, itemp, mtypeFace, i;
    int     *sensesLoop, *sensesEdges;
    CINT    *tempIlist;
    double  dataFace[18], dataEdge[18], uv_best[2], xyz_best[3];
    double  tangx,tangy, tangz, tang, normx, normy, normz, norm;
    double  trange[2], uvrange[4];
    CDOUBLE *tempRlist;
    CCHAR   *tempClist;
    ego     *etempEdges, *etempNodes, *eloops, *enodes, *eedges, *efaces, eref;

    ROUTINE (setupNodeInfo);

    /* --------------------------------------------------------------- */

    status = EG_getBodyTopos(ebody, NULL, NODE, &nnode, &enodes);
    CHECK_STATUS(EG_getBodyTopos);

    EG_free(enodes);

    status = EG_getBodyTopos(ebody, NULL, EDGE, &nedge, &eedges);
    CHECK_STATUS(EG_getBodyTopos);

    SPLINT_CHECK_FOR_NULL(eedges);

    /* loop through the Edges with attr1 that are adjacent to a Face
       without attr1 and deposit at its Nodes the normal and tangential
       directions */
    for (iedge = 0; iedge < nedge; iedge++) {
#ifdef DEBUG
        printf("looking at iedge=%d\n", EG_indexBodyTopo(ebody, eedges[iedge]));
#endif

        status = EG_getInfo(eedges[iedge], &oclass, &mtype, NULL, NULL, NULL);
        CHECK_STATUS(EG_getInfo);
        if (mtype == DEGENERATE) continue;

        status = EG_attributeRet(eedges[iedge], attr1, &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status != EGADS_SUCCESS) continue;

        status = EG_getBodyTopos(ebody, eedges[iedge], FACE, &nface, &efaces);
        CHECK_STATUS(EG_getBodyTopos);

        SPLINT_CHECK_FOR_NULL(efaces);

        for (iface = 0; iface < nface; iface++) {
            status = EG_attributeRet(efaces[iface], attr1, &attrType, &attrLen,
                                     &tempIlist, &tempRlist, &tempClist);
            if (status == EGADS_SUCCESS) continue;

#ifdef DEBUG
            printf("     found iface=%d\n", EG_indexBodyTopo(ebody, efaces[iface]));
#endif
            status = EG_getTopology(efaces[iface], &eref, &oclass, &mtypeFace,
                                    uvrange, &nloop, &eloops, &sensesLoop);
            CHECK_STATUS(EG_getTopology);

            for (iloop = 0; iloop < nloop; iloop++) {
                status = EG_getTopology(eloops[iloop], &eref, &oclass, &mtype,
                                        uvrange, &ntempEdge, &etempEdges, &sensesEdges);
                CHECK_STATUS(EG_getTopology);

                for (itemp = 0; itemp < ntempEdge; itemp++) {
                    if (etempEdges[itemp] != eedges[iedge]) continue;

                    status = EG_getTopology(etempEdges[itemp], &eref, &oclass, &mtype,
                                            trange, &ntempNode, &etempNodes, &senses);
                    CHECK_STATUS(EG_getTopology);

                    if (mtype == DEGENERATE) continue;

                    for (i = 0; i < 2; i++) {
                        inode = EG_indexBodyTopo(ebody, etempNodes[i]) - 1;

                        status = EG_evaluate(etempEdges[itemp], &trange[i], dataEdge);
                        CHECK_STATUS(EG_evaluate);

                        status = EG_invEvaluate(efaces[iface], dataEdge, uv_best, xyz_best);
                        CHECK_STATUS(EG_invEvaluate);

                        status = EG_evaluate(efaces[iface], uv_best, dataFace);
                        CHECK_STATUS(EG_evaluate);

                        tangx = sensesLoop[iloop] * sensesEdges[itemp] * dataEdge[3];
                        tangy = sensesLoop[iloop] * sensesEdges[itemp] * dataEdge[4];
                        tangz = sensesLoop[iloop] * sensesEdges[itemp] * dataEdge[5];
                        tang  = sqrt(tangx * tangx + tangy * tangy + tangz * tangz);

                        normx = dataFace[4] * dataFace[8] - dataFace[5] * dataFace[7];
                        normy = dataFace[5] * dataFace[6] - dataFace[3] * dataFace[8];
                        normz = dataFace[3] * dataFace[7] - dataFace[4] * dataFace[6];
                        norm  = mtypeFace * sqrt(normx * normx + normy * normy + normz * normz);
#ifdef DEBUG
                        printf("        at inode=%3d, tang=%10.5f %10.5f %10.5f, norm=%10.5f %10.5f %10.5f\n",
                               inode+1, tangx/tang, tangy/tang, tangz/tang, normx/norm, normy/norm, normz/norm);
#endif
                        nodeCount[  inode  ] ++;
                        nodeInfo[ 9*inode  ] = dataEdge[0];
                        nodeInfo[ 9*inode+1] = dataEdge[1];
                        nodeInfo[ 9*inode+2] = dataEdge[2];
                        nodeInfo[ 9*inode+3] += normx / norm;
                        nodeInfo[ 9*inode+4] += normy / norm;
                        nodeInfo[ 9*inode+5] += normz / norm;
                        nodeInfo[ 9*inode+6] += tangx / tang;
                        nodeInfo[ 9*inode+7] += tangy / tang;
                        nodeInfo[ 9*inode+8] += tangz / tang;
                    }
                }
            }
        }

        EG_free(efaces);   efaces = NULL;
    }

    /* loop through the Edges with attr2 that are adjacent to a Face
       without attr2 and deposit at its Nodes the normal and tangential
       directions */
    for (iedge = 0; iedge < nedge; iedge++) {
#ifdef DEBUG
        printf("looking at iedge=%d\n", EG_indexBodyTopo(ebody, eedges[iedge]));
#endif

        status = EG_getInfo(eedges[iedge], &oclass, &mtype, NULL, NULL, NULL);
        CHECK_STATUS(EG_getInfo);
        if (mtype == DEGENERATE) continue;

        status = EG_attributeRet(eedges[iedge], attr2, &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status != EGADS_SUCCESS) continue;

        status = EG_getBodyTopos(ebody, eedges[iedge], FACE, &nface, &efaces);
        CHECK_STATUS(EG_getBodyTopos);

        SPLINT_CHECK_FOR_NULL(efaces);

        for (iface = 0; iface < nface; iface++) {
            status = EG_attributeRet(efaces[iface], attr2, &attrType, &attrLen,
                                     &tempIlist, &tempRlist, &tempClist);
            if (status == EGADS_SUCCESS) continue;

#ifdef DEBUG
            printf("     found iface=%d\n", EG_indexBodyTopo(ebody, efaces[iface]));
#endif
            status = EG_getTopology(efaces[iface], &eref, &oclass, &mtypeFace,
                                    uvrange, &nloop, &eloops, &sensesLoop);
            CHECK_STATUS(EG_getTopology);

            for (iloop = 0; iloop < nloop; iloop++) {
                status = EG_getTopology(eloops[iloop], &eref, &oclass, &mtype,
                                        uvrange, &ntempEdge, &etempEdges, &sensesEdges);
                CHECK_STATUS(EG_getTopology);

                for (itemp = 0; itemp < ntempEdge; itemp++) {
                    if (etempEdges[itemp] != eedges[iedge]) continue;

                    status = EG_getTopology(etempEdges[itemp], &eref, &oclass, &mtype,
                                            trange, &ntempNode, &etempNodes, &senses);
                    CHECK_STATUS(EG_getTopology);

                    if (mtype == DEGENERATE) continue;

                    for (i = 0; i < 2; i++) {
                        inode = EG_indexBodyTopo(ebody, etempNodes[i]) - 1;

                        status = EG_evaluate(etempEdges[itemp], &trange[i], dataEdge);
                        CHECK_STATUS(EG_evaluate);

                        status = EG_invEvaluate(efaces[iface], dataEdge, uv_best, xyz_best);
                        CHECK_STATUS(EG_invEvaluate);

                        status = EG_evaluate(efaces[iface], uv_best, dataFace);
                        CHECK_STATUS(EG_evaluate);

                        tangx = sensesLoop[iloop] * sensesEdges[itemp] * dataEdge[3];
                        tangy = sensesLoop[iloop] * sensesEdges[itemp] * dataEdge[4];
                        tangz = sensesLoop[iloop] * sensesEdges[itemp] * dataEdge[5];
                        tang  = sqrt(tangx * tangx + tangy * tangy + tangz * tangz);

                        normx = dataFace[4] * dataFace[8] - dataFace[5] * dataFace[7];
                        normy = dataFace[5] * dataFace[6] - dataFace[3] * dataFace[8];
                        normz = dataFace[3] * dataFace[7] - dataFace[4] * dataFace[6];
                        norm  = mtypeFace * sqrt(normx * normx + normy * normy + normz * normz);
#ifdef DEBUG
                        printf("        at inode=%3d, tang=%10.5f %10.5f %10.5f, norm=%10.5f %10.5f %10.5f\n",
                               inode+1, tangx/tang, tangy/tang, tangz/tang, normx/norm, normy/norm, normz/norm);
#endif
                        nodeCount[  inode  ] ++;
                        nodeInfo[ 9*inode  ] = dataEdge[0];
                        nodeInfo[ 9*inode+1] = dataEdge[1];
                        nodeInfo[ 9*inode+2] = dataEdge[2];
                        nodeInfo[ 9*inode+3] += normx / norm;
                        nodeInfo[ 9*inode+4] += normy / norm;
                        nodeInfo[ 9*inode+5] += normz / norm;
                        nodeInfo[ 9*inode+6] += tangx / tang;
                        nodeInfo[ 9*inode+7] += tangy / tang;
                        nodeInfo[ 9*inode+8] += tangz / tang;
                    }
                }
            }
        }

        EG_free(efaces);   efaces = NULL;
    }

    EG_free(eedges);   eedges = NULL;

#ifdef DEBUG
    for (inode = 0; inode < nnode; inode++) {
        printf("%3d count=%3d, xyz=%10.5f %10.5f %10.5f, norm=%10.5f %10.5f %10.5f, tang=%10.5f %10.5f %10.5f, t.x.n=%10.5f %10.5f %10.5f\n",
               inode+1, nodeCount[inode],
               nodeInfo[9*inode  ], nodeInfo[9*inode+1], nodeInfo[9*inode+2],
               nodeInfo[9*inode+3], nodeInfo[9*inode+4], nodeInfo[9*inode+5],
               nodeInfo[9*inode+6], nodeInfo[9*inode+7], nodeInfo[9*inode+8],
               nodeInfo[9*inode+5]*nodeInfo[9*inode+7]-nodeInfo[9*inode+4]*nodeInfo[9*inode+8],
               nodeInfo[9*inode+3]*nodeInfo[9*inode+8]-nodeInfo[9*inode+5]*nodeInfo[9*inode+6],
               nodeInfo[9*inode+4]*nodeInfo[9*inode+6]-nodeInfo[9*inode+3]*nodeInfo[9*inode+7]);
    }
#endif

    for (inode = 0; inode < nnode; inode++) {
        if (nodeCount[inode] == 0) {

        } else if (nodeCount[inode] == 2) {
            nodeInfo[9*inode+3] /= 2;
            nodeInfo[9*inode+4] /= 2;
            nodeInfo[9*inode+5] /= 2;
            nodeInfo[9*inode+6] /= 2;
            nodeInfo[9*inode+7] /= 2;
            nodeInfo[9*inode+8] /= 2;
        } else {
            printf("trouble here, inode=%d\n", inode);
            exit(0);
        }
    }

    status = EGADS_SUCCESS;

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   getXYZforEntity - find xyz for Edge or Face at end of entity       *
 *                                                                      *
 ************************************************************************
 */

static int
getXYZforEntity(ego    ebody,           /* (in)  Body */
                int    iedge,           /* (in)  Edge or Node index (bias-1) */
                int    isens,           /* (in)  =0 for Node, otherwise Edge */
                char   attr[],          /* (in)  attribute associated with Edge */
                double nodeInfo[],
                double xyz[])           /* (out) xyz, d(xyz)/d(t) */
{
    int status = EGADS_SUCCESS;

    int      inode, ntemp, nedge, jedge, nface, jface, count;
    int      oclass, mtype, *senses, attrType, attrLen;
    CINT     *tempIlist;
    double   trange[4], uv[2], data[18];
    double   xnorm1, ynorm1, znorm1, xnorm2, ynorm2, znorm2, temp, dot;
    CDOUBLE  *tempRlist;
    CCHAR    *tempClist;
    ego      enode, eedge, eref, *etemp, *eedges=NULL, *efaces=NULL;

    ROUTINE (getXYZforEntity);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    printf("enter getXYZforEntity(iedge=%d, isens=%d, attr=%s)\n", iedge, isens, attr);
#endif

    /* make sure the Edge has the specified attribute */
    if (isens != 0) {
        status = EG_objectBodyTopo(ebody, EDGE, iedge, &eedge);
        CHECK_STATUS(EG_objectBodyTopo);

        status = EG_attributeRet(eedge, attr, &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        CHECK_STATUS(EG_attributeRet);
    }

    /* find enode at beg of entity */
    if (isens < 0) {
        status = EG_getTopology(eedge, &eref, &oclass, &mtype,
                                trange, &ntemp, &etemp, &senses);
        CHECK_STATUS(EG_getTopology);

        enode = etemp[0];
    } else if (isens > 0) {
        status = EG_getTopology(eedge, &eref, &oclass, &mtype,
                                trange, &ntemp, &etemp, &senses);
        CHECK_STATUS(EG_getTopology);

        enode = etemp[1];
    } else {
        inode = iedge;

        status = EG_objectBodyTopo(ebody, NODE, inode, &enode);
        CHECK_STATUS(EG_objectBodyTopo);
    }
#ifdef DEBUG
    printf("inode=%d\n", EG_indexBodyTopo(ebody, enode));
#endif

    /* find count of Edge(s) that are not associated with attr and
       which are incident to enode */
    count = 0;

    status = EG_getBodyTopos(ebody, enode, EDGE, &nedge, &eedges);
    CHECK_STATUS(EG_getBodyTopos);

    SPLINT_CHECK_FOR_NULL(eedges);

    for (jedge = 0; jedge < nedge; jedge++) {
        status = EG_attributeRet(eedges[jedge], attr, &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status != EGADS_SUCCESS) {
            eedge = eedges[jedge];
            count ++;
        }
    }

    EG_free(eedges);   eedges = NULL;

    /* if there is one Edge, check to see if the normals on the two
       incident Faces are nearly parallel.  and if they are, change
       the count to 0 so that the flend curve is not forced to be
       parallel to the Edge */
    if (count == 1) {
        status = EG_getBodyTopos(ebody, enode, FACE, &nface, &efaces);
        CHECK_STATUS(EG_getBodyTopos);

        SPLINT_CHECK_FOR_NULL(efaces);

        xnorm1 = 0;
        ynorm1 = 0;
        znorm1 = 0;
        xnorm2 = 0;
        ynorm2 = 0;
        znorm2 = 0;

        for (jface = 0; jface < nface; jface++) {
            status = EG_attributeRet(efaces[jface], attr, &attrType, &attrLen,
                                     &tempIlist, &tempRlist, &tempClist);
            if (status != EGADS_SUCCESS) {
                xnorm2 = xnorm1;
                ynorm2 = ynorm1;
                znorm2 = znorm1;

                status = EG_getEdgeUV(efaces[jface], enode, 0, 0, uv);
                CHECK_STATUS(EG_getEdgeUV);

                status = EG_evaluate(efaces[jface], uv, data);
                CHECK_STATUS(EG_evaluate);

                xnorm1 = data[4] * data[8] - data[5] * data[7];
                ynorm1 = data[5] * data[6] - data[3] * data[8];
                znorm1 = data[3] * data[7] - data[4] * data[6];
                temp   = sqrt(xnorm1 * xnorm1 + ynorm1 * ynorm1 + znorm1 * znorm1);
                xnorm1 /= temp;
                ynorm1 /= temp;
                znorm1 /= temp;
            }
        }

        dot = xnorm1 * xnorm2 + ynorm1 * ynorm2 + znorm1 * znorm2;
        if (dot > 0.985) {              // approximatey 10 deg
            count = 0;
#ifdef DEBUG
            printf("faces almost normal, so changing count=0\n");
#endif
        }

        EG_free(efaces);   efaces = NULL;
    }

    /* if there are no Edges to set the direction, find Faces that are not
       associated with attr and which are indicent to enode */
    if (count == 0) {
        inode = EG_indexBodyTopo(ebody, enode) - 1;

        xyz[0] = nodeInfo[9*inode  ];
        xyz[1] = nodeInfo[9*inode+1];
        xyz[2] = nodeInfo[9*inode+2];

        /* vector out of Face is tang.cross.norm (assuming Face is on
           left of Edge (which was taken care of in setupNodeInfo) */
        xyz[3] = nodeInfo[9*inode+7] * nodeInfo[9*inode+5] - nodeInfo[9*inode+8] * nodeInfo[9*inode+4];
        xyz[4] = nodeInfo[9*inode+8] * nodeInfo[9*inode+3] - nodeInfo[9*inode+6] * nodeInfo[9*inode+5];
        xyz[5] = nodeInfo[9*inode+6] * nodeInfo[9*inode+4] - nodeInfo[9*inode+7] * nodeInfo[9*inode+3];

    /* if there was one Edge incident to enode and not associated
       with attr, evaluate the Edge at enode */
    } else if (count == 1) {
        SPLINT_CHECK_FOR_NULL(eedge);

        status = EG_getTopology(eedge, &eref, &oclass, &mtype,
                                trange, &ntemp, &etemp, &senses);
        CHECK_STATUS(EG_getTopology);

#ifdef DEBUG
        printf("enode   =%3d (%llx)\n", EG_indexBodyTopo(ebody, enode   ), (long long)enode   );
        printf("eedge   =%3d (%llx)\n", EG_indexBodyTopo(ebody, eedge   ), (long long)eedge   );
        printf("etemp[0]=%3d (%llx)\n", EG_indexBodyTopo(ebody, etemp[0]), (long long)etemp[0]);
        printf("etemp[1]=%3d (%llx)\n", EG_indexBodyTopo(ebody, etemp[1]), (long long)etemp[1]);
#endif

        if (etemp[0] == enode) {
            status = EG_evaluate(eedge, &trange[0], data);
            CHECK_STATUS(EG_evaluate);

            xyz[0] =  data[0];
            xyz[1] =  data[1];
            xyz[2] =  data[2];
            xyz[3] = -data[3];
            xyz[4] = -data[4];
            xyz[5] = -data[5];
        } else if (etemp[1] == enode) {
            status = EG_evaluate(eedge, &trange[1], data);
            CHECK_STATUS(EG_evaluate);

            xyz[0] =  data[0];
            xyz[1] =  data[1];
            xyz[2] =  data[2];
            xyz[3] = +data[3];
            xyz[4] = +data[4];
            xyz[5] = +data[5];
        }

        /* more than one Edge found */
    } else {
        printf("error2, count=%d\n", count); exit(0);
    }

cleanup:
    if (eedges != NULL) EG_free(eedges);
    if (efaces != NULL) EG_free(efaces);

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   makeFlendCurve - make a flend curve between two points             *
 *                                                                      *
 ************************************************************************
 */

static int
makeFlendCurve(ego     context,         /* (in)  context */
               double  xyz1[],          /* (in)  data at beg of flend curve */
               double  xyz2[],          /* (in)  data at end of flend curve */
               double  frac1,           /* (in)  flend strength at beg */
               double  frac2,           /* (in)  flend strength at end */
               double  cpout[])
{
    int status = EGADS_SUCCESS;

    int    header[4], i;
//#define NEWSTUFF
#ifdef  NEWSTUFF
    double dist, rhs1, rhs2, det, t1, t2;
#else
    double Ax, Ay, Az, Ax1, Ay1, Az1, Ax2, Ay2, Az2;
#endif
    double temp, knots[8], cps[12], rdata[20];
    ego    newCurve;

    ROUTINE (makeFlendCurve);

    /* --------------------------------------------------------------- */

    printf("frac1=%10.5f frac2=%10.5f\n", frac1, frac2);

    /* set up bspline header and knot vector */
    header[0] = 0;
    header[1] = 3;            /* degree */
    header[2] = 4;            /* ncp    */
    header[3] = 8;            /* nknot  */

    knots[0] = 0;
    knots[1] = 0;
    knots[2] = 0;
    knots[3] = 0;
    knots[4] = 1;
    knots[5] = 1;
    knots[6] = 1;
    knots[7] = 1;

    /* normalize the tangent vectors */
    temp     = sqrt(xyz1[3] * xyz1[3] + xyz1[4] * xyz1[4] + xyz1[5] * xyz1[5]);
    xyz1[3] /= temp;
    xyz1[4] /= temp;
    xyz1[5] /= temp;
    printf("xyz1: %10.5f %10.5f %10.5f   %10.5f %10.5f %10.5f\n",
           xyz1[0], xyz1[1], xyz1[2], xyz1[3], xyz1[4], xyz1[5]);

    temp     = sqrt(xyz2[3] * xyz2[3] + xyz2[4] * xyz2[4] + xyz2[5] * xyz2[5]);
    xyz2[3] /= temp;
    xyz2[4] /= temp;
    xyz2[5] /= temp;
    printf("xyz2: %10.5f %10.5f %10.5f   %10.5f %10.5f %10.5f\n",
           xyz2[0], xyz2[1], xyz2[2], xyz2[3], xyz2[4], xyz2[5]);

    /* find the distance between points */
#ifdef NEWSTUFF
    dist = sqrt((xyz1[0] - xyz2[0]) * (xyz1[0] - xyz2[0])
               +(xyz1[1] - xyz2[1]) * (xyz1[1] - xyz2[1])
               +(xyz1[2] - xyz2[2]) * (xyz1[2] - xyz2[2]));

    /* find the t1 and t2 that minimizes the distance between vectors
       out of xyz1 and xyz2 */
    temp = - xyz1[3] * xyz2[3] - xyz1[4] * xyz2[4] - xyz1[5] * xyz2[5];

    rhs1 = (xyz2[0] - xyz1[0]) * xyz1[3]
        +  (xyz2[1] - xyz1[1]) * xyz1[4]
        +  (xyz2[2] - xyz1[2]) * xyz1[5];
    rhs2 = (xyz1[0] - xyz2[0]) * xyz2[3]
        +  (xyz1[1] - xyz2[1]) * xyz2[4]
        +  (xyz1[2] - xyz2[2]) * xyz2[5];
    printf("mat=%10.5f %10.5f,   rhs=%10.5f\n", 1., temp, rhs1);
    printf("    %10.5f %10.5f,       %10.5f\n", temp, 1., rhs2);

    det = 1 - temp * temp;
    printf("dist=%10.5f, det=%10.5f\n", dist, det);


    if (fabs(det) > EPS06) {
        t1 = (rhs1 - temp * rhs2) / det;
        t2 = (rhs2 - temp * rhs1) / det;
    } else {
        t1 = dist;
        t2 = dist;
    }
    printf("t1=%10.5f, t2=%10.5f\n", t1, t2);

    if (t1 < dist/2) t1 = dist / 2;
    if (t1 > dist  ) t1 = dist;
    if (t2 < dist/2) t2 = dist / 2;
    if (t2 > dist  ) t2 = dist;

    printf("t1=%10.5f, t2=%10.5f\n", t1, t2);

    cps[ 0] = xyz1[0];
    cps[ 1] = xyz1[1];
    cps[ 2] = xyz1[2];
    cps[ 3] = xyz1[0] + frac1 * t1 * xyz1[3];
    cps[ 4] = xyz1[1] + frac1 * t1 * xyz1[4];
    cps[ 5] = xyz1[2] + frac1 * t1 * xyz1[5];
    cps[ 6] = xyz2[0] + frac2 * t2 * xyz2[3];
    cps[ 7] = xyz2[1] + frac2 * t2 * xyz2[4];
    cps[ 8] = xyz2[2] + frac2 * t2 * xyz2[5];
    cps[ 9] = xyz2[0];
    cps[10] = xyz2[1];
    cps[11] = xyz2[2];

#else
    /* if the vectors are nearly parallel, just create a straight line */
    /*@-evalorder@*/
    temp =    (xyz1[3] * xyz2[3] + xyz1[4] * xyz2[4] + xyz1[5] * xyz2[5])
        / sqrt(xyz1[3] * xyz1[3] + xyz1[4] * xyz1[4] + xyz1[5] * xyz1[5])
        / sqrt(xyz2[3] * xyz2[3] + xyz2[4] * xyz2[4] + xyz2[5] * xyz2[5]);
    /*@+evalorder@*/

    if (temp > 0.985) {
        cps[ 0] =      xyz1[0];
        cps[ 1] =      xyz1[1];
        cps[ 2] =      xyz1[2];
        cps[ 3] = (2 * xyz1[0] +     xyz2[0]) / 3;
        cps[ 4] = (2 * xyz1[1] +     xyz2[1]) / 3;
        cps[ 5] = (2 * xyz1[2] +     xyz2[2]) / 3;
        cps[ 6] = (    xyz1[0] + 2 * xyz2[0]) / 3;
        cps[ 7] = (    xyz1[1] + 2 * xyz2[1]) / 3;
        cps[ 8] = (    xyz1[2] + 2 * xyz2[2]) / 3;
        cps[ 9] =                    xyz2[0];
        cps[10] =                    xyz2[1];
        cps[11] =                    xyz2[2];

    } else {
        /* find the vector A between the two ends */
        Ax = xyz2[0] - xyz1[0];
        Ay = xyz2[1] - xyz1[1];
        Az = xyz2[2] - xyz1[2];

        /* get the slope at beg from tangent vector */
        /*@-evalorder@*/
        temp = sqrt(Ax      * Ax      + Ay      * Ay      + Az      * Az     )
            / sqrt(xyz1[3] * xyz1[3] + xyz1[4] * xyz1[4] + xyz1[5] * xyz1[5]);
        /*@+evalorder@*/
        Ax1 = frac1 * xyz1[3] * temp;
        Ay1 = frac1 * xyz1[4] * temp;
        Az1 = frac1 * xyz1[5] * temp;

        temp = Ax * xyz1[3] + Ay * xyz1[4] + Az * xyz1[5];
        if (temp < 0) {
            Ax1 *= -1;
            Ay1 *= -1;
            Az1 *= -1;
        }

        /* get the slope at end from tangent vector */
        /*@-evalorder@*/
        temp = sqrt(Ax      * Ax      + Ay      * Ay      + Az      * Az     )
            / sqrt(xyz2[3] * xyz2[3] + xyz2[4] * xyz2[4] + xyz2[5] * xyz2[5]);
        /*@+evalorder@*/
        Ax2 = frac2 * xyz2[3] * temp;
        Ay2 = frac2 * xyz2[4] * temp;
        Az2 = frac2 * xyz2[5] * temp;

        temp = Ax * xyz2[3] + Ay * xyz2[4] + Az * xyz2[5];
        if (temp < 0) {
            Ax2 *= -1;
            Ay2 *= -1;
            Az2 *= -1;
        }

        /* set up the control points */
        cps[ 0] = xyz1[0];
        cps[ 1] = xyz1[1];
        cps[ 2] = xyz1[2];
        cps[ 3] = xyz1[0] + Ax1;
        cps[ 4] = xyz1[1] + Ay1;
        cps[ 5] = xyz1[2] + Az1;
        cps[ 6] = xyz2[0] - Ax2;
        cps[ 7] = xyz2[1] - Ay2;
        cps[ 8] = xyz2[2] - Az2;
        cps[ 9] = xyz2[0];
        cps[10] = xyz2[1];
        cps[11] = xyz2[2];
    }
#endif

    for (i = 0; i < 12; i++) {
        cpout[i] = cps[i];
    }

    /* create the cubic bspline curve */
    for (i = 0; i < 8; i++) {
        rdata[i  ] = knots[i];
    }
    for (i = 0; i < 12; i++) {
        rdata[i+8] = cps[i];
    }
    status = EG_makeGeometry(context, CURVE, BSPLINE, NULL,
                             header, rdata, &newCurve);
    CHECK_STATUS(EG_makeGeometry);

    status = EG_deleteObject(newCurve);
    CHECK_STATUS(EG_deleteObject);

cleanup:
    return status;
}
