/*
 *      CAPS: Computational Aircraft Prototype Syntheses
 *
 *             libMeshb Mesh Writer Code
 *
 *      Copyright 2014-2026, Massachusetts Institute of Technology
 *      Licensed under The GNU Lesser General Public License, version 2.1
 *      See http://www.opensource.org/licenses/lgpl-2.1.php
 *
 */

#include <string.h>
#include <math.h>
#include "aimUtil.h"
#include "aimMesh.h"

#include "miscUtils.h"

#include "libmeshbWriter.h"
#include "libMeshb/sources/libmeshb7.h"

/* these values requested to emulate Feflo.a behavior */
#define EXPORT_MESHB_VERTEX_ID (1)
#define EXPORT_MESHB_2D_ID (1)
#define EXPORT_MESHB_3D_ID (0)
#define EXPORT_MESHB_VERTEX_3 (10000000)
#define EXPORT_MESHB_VERTEX_4 (200000000)


#define CROSS(a,b,c)      a[0] = ((b)[1]*(c)[2]) - ((b)[2]*(c)[1]);\
                          a[1] = ((b)[2]*(c)[0]) - ((b)[0]*(c)[2]);\
                          a[2] = ((b)[0]*(c)[1]) - ((b)[1]*(c)[0])
#define DOT(a,b)         ((a)[0]*(b)[0] + (a)[1]*(b)[1] + (a)[2]*(b)[2])

typedef struct {
  int    npts;
  double *xyz;
  double *t;
  int    *ivp;   // volume node index
} edgeData;


typedef struct {
  int    npts;
  double *xyz;
  double *uv;
  int    nquad;
  int    ntri;
  int    *tris;
  int    *ivp;   // volume node index
} faceData;


typedef struct {
  double **rvec;
  ego    *surfaces;
  ego    body;
  ego    *faces;
  ego    *edges;
  ego    *nodes;
  int    nfaces;
  int    nedges;
  int    nnodes;

  edgeData *tedges;
  faceData *tfaces;
} bodyData;


/*****************************************************************************/
static int initiate_bodyData(int numBody, bodyData *bodydata)
{
  int i;

  for (i = 0; i < numBody; i++) {
    bodydata[i].rvec = NULL;
    bodydata[i].surfaces = NULL;
    bodydata[i].faces = NULL;
    bodydata[i].edges = NULL;
    bodydata[i].nodes = NULL;
    bodydata[i].nfaces = 0;
    bodydata[i].nedges = 0;
    bodydata[i].nnodes = 0;
    bodydata[i].tedges = NULL;
    bodydata[i].tfaces = NULL;
  }

  return CAPS_SUCCESS;
}

static int destroy_bodyData(int numBody, bodyData *bodydata)
{
  int i, j;

  if (bodydata == NULL) return CAPS_SUCCESS;

  for (i = 0; i < numBody; i++) {
    for (j = 0; j < bodydata[i].nfaces; j++) {
      if (bodydata[i].surfaces != NULL)
        if (bodydata[i].surfaces[j+bodydata[i].nfaces] != NULL)
          EG_deleteObject(bodydata[i].surfaces[j+bodydata[i].nfaces]);
      if (bodydata[i].rvec != NULL)
        EG_free(bodydata[i].rvec[j]);
    }
    EG_free(bodydata[i].nodes);
    EG_free(bodydata[i].edges);
    EG_free(bodydata[i].faces);
    EG_free(bodydata[i].surfaces);
    EG_free(bodydata[i].rvec);

    if (bodydata[i].tedges != NULL) {
        for (j = 0; j < bodydata[i].nedges; j++) {
            EG_free(bodydata[i].tedges[j].xyz);
            EG_free(bodydata[i].tedges[j].t);
            EG_free(bodydata[i].tedges[j].ivp);
        }
        EG_free(bodydata[i].tedges);
    }

    if (bodydata[i].tfaces != NULL) {
        for (j = 0; j < bodydata[i].nfaces; j++) {
            EG_free(bodydata[i].tfaces[j].xyz);
            EG_free(bodydata[i].tfaces[j].uv);
            EG_free(bodydata[i].tfaces[j].tris);
            EG_free(bodydata[i].tfaces[j].ivp);
        }
        EG_free(bodydata[i].tfaces);
    }
  }

  return CAPS_SUCCESS;
}

/*****************************************************************************/

const char *meshExtension()
{
/*@-observertrans@*/
  return MESHEXTENSION;
/*@+observertrans@*/
}

/*****************************************************************************/

int meshWrite(void *aimInfo, aimMesh *mesh)
{
  int status; // Function return status
  int  i, j, igroup, ielem, nPoint;
  int state, nglobal, id;
  int nNode, nEdge, nFace;
  int iedge, iface;
  int nNodeOffset, nEdgeOffset, nFaceOffset;
  int nNodeVerts, nEdgeVerts, nFaceVerts;
  int nLine, nTri, nQuad;
  int ptype, pindex, plen, tlen, iglobal, atype, alen, stride;
  int elem[4];
  const int *ptypes, *pindexs, *tris, *tric, *tessFaceQuadMap=NULL;
  const double *points, *uv, *reals, *t;
  const char *string;
  double xyz[3], param[2];
  ego context, model=NULL, body, *bodies=NULL, *faces=NULL,*edges=NULL;
  size_t nbytes;
  char *stream=NULL;
  char filename[PATH_MAX];
  int xMeshConstant = (int)true, yMeshConstant = (int)true, zMeshConstant = (int)true;
  int64_t fileID=0;
  int meshVersion;
  aimMeshRef *meshRef = NULL;
  aimMeshData *meshData = NULL;

  if (mesh == NULL) return CAPS_NULLVALUE;
  if (mesh->meshRef  == NULL) return CAPS_NULLVALUE;
  if (mesh->meshData == NULL) return CAPS_NULLVALUE;

  meshRef  = mesh->meshRef;
  meshData = mesh->meshData;

  if (meshData->dim != 2 && meshData->dim != 3) {
    AIM_ERROR(aimInfo, "meshData dim = %d must be 2 or 3!!!", mesh->meshData->dim);
    return CAPS_BADVALUE;
  }

  snprintf(filename, PATH_MAX, "%s%s", mesh->meshRef->fileName, MESHEXTENSION);

  meshVersion = 2;
  if (EXPORT_MESHB_VERTEX_3 < meshData->nVertex) meshVersion = 3;
  if (EXPORT_MESHB_VERTEX_4 < meshData->nVertex) meshVersion = 4;

  fileID = GmfOpenMesh(filename, GmfWrite, meshVersion, meshData->dim);

  if (fileID == 0) {
    AIM_ERROR(aimInfo, "Cannot open file: %s\n", filename);
    return CAPS_IOERR;
  }

  status = GmfSetKwd(fileID, GmfVertices, meshData->nVertex);
  if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

  if (meshData->dim == 2) {

    for (i = 0; i < meshData->nVertex; i++) {
      // Constant x?
      if (fabs(meshData->verts[i][0] - meshData->verts[0][0]) > 1E-7) {
        xMeshConstant = (int) false;
      }

      // Constant y?
      if (fabs(meshData->verts[i][1] - meshData->verts[0][1] ) > 1E-7) {
        yMeshConstant = (int) false;
      }

      // Constant z?
      if (fabs(meshData->verts[i][2] - meshData->verts[0][2]) > 1E-7) {
        zMeshConstant = (int) false;
      }
    }

    if (zMeshConstant == (int) false) {
      printf("libMeshb expects 2D meshes in the x-y plane... attempting to rotate mesh!");

      if (xMeshConstant == (int) true && yMeshConstant == (int) false) {
        printf("Swapping z and x coordinates!\n");
        for (i = 0; i < meshData->nVertex; i++) {
          status = GmfSetLin(fileID, GmfVertices, meshData->verts[i][2],
                                                  meshData->verts[i][1], EXPORT_MESHB_VERTEX_ID);
          if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
        }

      } else if(xMeshConstant == (int) false && yMeshConstant == (int) true) {

        printf("Swapping z and y coordinates!");
        for (i = 0; i < meshData->nVertex; i++) {
          status = GmfSetLin(fileID, GmfVertices, meshData->verts[i][0],
                                                  meshData->verts[i][2], EXPORT_MESHB_VERTEX_ID);
          if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
        }

      } else {
        AIM_ERROR(aimInfo, "Unable to rotate mesh!");
        status = CAPS_NOTFOUND;
        goto cleanup;
      }

    } else { // zMeshConstant == true
      // Write nodal coordinates as is
      for (i = 0; i < meshData->nVertex; i++) {
        status = GmfSetLin(fileID, GmfVertices, meshData->verts[i][0],
                                                meshData->verts[i][1], EXPORT_MESHB_VERTEX_ID);
        if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
      }
    }

  } else {

    // Write nodal coordinates
    for (i = 0; i < meshData->nVertex; i++) {
      status = GmfSetLin(fileID, GmfVertices, meshData->verts[i][0],
                                              meshData->verts[i][1],
                                              meshData->verts[i][2], EXPORT_MESHB_VERTEX_ID);
      if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
    }
  }


  // write out elements

  // count the number of EDGE/FACE elements
  nQuad = nTri = nLine = 0;
  for (i = 0; i < meshRef->nmap; i++) {
    status = EG_statusTessBody(meshRef->maps[i].tess, &body, &state, &nglobal);
    AIM_STATUS(aimInfo, status);

    status = EG_getBodyTopos(body, NULL, EDGE, &nEdge, &edges);
    AIM_STATUS(aimInfo, status);

    for (iedge = 0; iedge < nEdge; iedge++) {
      if (edges[iedge]->mtype == DEGENERATE) continue;
      status = EG_getTessEdge(meshRef->maps[i].tess, iedge + 1, &plen, &points, &t);
      AIM_STATUS(aimInfo, status);
      nLine += plen-1;
    }
    AIM_FREE(edges);

    // No triangles for an area mesh
    if (meshRef->type == aimAreaMesh) continue;

    // check if the tessellation has a mixture of quad and tri
    status = EG_attributeRet(meshRef->maps[i].tess, ".mixed",
                             &atype, &alen, &tessFaceQuadMap, &reals, &string);
    AIM_NOTFOUND(aimInfo, status);

    status = EG_getBodyTopos(body, NULL, FACE, &nFace, NULL);
    AIM_STATUS(aimInfo, status);

    for (iface = 0; iface < nFace; iface++) {
      status = EG_getTessFace(meshRef->maps[i].tess, iface + 1, &plen, &points, &uv, &ptypes, &pindexs,
                              &tlen, &tris, &tric);
      AIM_STATUS(aimInfo, status);

      if (tessFaceQuadMap != NULL) {
        tlen -= 2*tessFaceQuadMap[iface];
        nQuad += tessFaceQuadMap[iface];
      }

      nTri += tlen;
    }
  }

  // Write out EDGE line elements
  status = GmfSetKwd(fileID, GmfEdges, nLine);
  if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

  nEdgeOffset = 0;
  for (i = 0; i < meshRef->nmap; i++) {
    status = EG_statusTessBody(meshRef->maps[i].tess, &body, &state, &nglobal);
    AIM_STATUS(aimInfo, status);

    status = EG_getBodyTopos(body, NULL, EDGE, &nEdge, NULL);
    AIM_STATUS(aimInfo, status);

    for (iedge = 0; iedge < nEdge; iedge++) {
      status = EG_getTessEdge(meshRef->maps[i].tess, iedge + 1, &plen, &points, &t);
      if (status == EGADS_DEGEN) continue;
      AIM_STATUS(aimInfo, status);

      for (j = 0; j < plen-1; j++) {

        status = EG_localToGlobal(meshRef->maps[i].tess, -(iedge + 1), j + 1, &elem[0]);
        if (status == EGADS_DEGEN) continue;
        AIM_STATUS(aimInfo, status);

        status = EG_localToGlobal(meshRef->maps[i].tess, -(iedge + 1), j + 2, &elem[1]);
        if (status == EGADS_DEGEN) continue;
        AIM_STATUS(aimInfo, status);

        status = GmfSetLin(fileID, GmfEdges, meshRef->maps[i].map[elem[0]-1],
                                             meshRef->maps[i].map[elem[1]-1],
                                             nEdgeOffset + iedge + 1);
        if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
      }
    }
    nEdgeOffset += nEdge;
  }

  // NOTE:
  // For Surface/Volume Meshes the Tri/Quad elements must be grouped based on
  // the EGADS Faces, not grouped by capsGroup in the meshData structure.
  //
  // For AreaMeshes the Tri/Quads should be grouped by capsGroup in the meshData structure.

  if (meshRef->type != aimAreaMesh) {
    // Write FACE triangle elements
    status = GmfSetKwd(fileID, GmfTriangles, nTri);
    if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

    nFaceOffset = 0;
    for (i = 0; i < meshRef->nmap; i++) {

      // check if the tessellation has a mixture of quad and tri
      status = EG_attributeRet(meshRef->maps[i].tess, ".mixed",
                               &atype, &alen, &tessFaceQuadMap, &reals, &string);
      AIM_NOTFOUND(aimInfo, status);

      status = EG_statusTessBody(meshRef->maps[i].tess, &body, &state, &nglobal);
      AIM_STATUS(aimInfo, status);

      status = EG_getBodyTopos(body, NULL, FACE, &nFace, NULL);
      AIM_STATUS(aimInfo, status);

      for (iface = 0; iface < nFace; iface++) {
        status = EG_getTessFace(meshRef->maps[i].tess, iface + 1, &plen, &points, &uv, &ptypes, &pindexs,
                                &tlen, &tris, &tric);
        AIM_STATUS(aimInfo, status);

        if (tessFaceQuadMap != NULL) {
          tlen -= 2*tessFaceQuadMap[iface];
        }

        for (j = 0; j < tlen; j++) {
          /* triangle orientation flipped, per refine convention */
          status = EG_localToGlobal(meshRef->maps[i].tess, iface + 1, tris[3*j + 0], &elem[1/*0*/]);
          AIM_STATUS(aimInfo, status);
          status = EG_localToGlobal(meshRef->maps[i].tess, iface + 1, tris[3*j + 1], &elem[0/*1*/]);
          AIM_STATUS(aimInfo, status);
          status = EG_localToGlobal(meshRef->maps[i].tess, iface + 1, tris[3*j + 2], &elem[2]);
          AIM_STATUS(aimInfo, status);

          status = GmfSetLin(fileID, GmfTriangles, meshRef->maps[i].map[elem[0]-1],
                                                   meshRef->maps[i].map[elem[1]-1],
                                                   meshRef->maps[i].map[elem[2]-1],
                                                   nFaceOffset + iface + 1);
          if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
        }
      }

      nFaceOffset += nFace;
    }

    if (nQuad > 0) {
      // Write FACE quad elements
      status = GmfSetKwd(fileID, GmfQuadrilaterals, nQuad);
      if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

      nFaceOffset = 0;
      for (i = 0; i < meshRef->nmap; i++) {

        // check if the tessellation has a mixture of quad and tri
        status = EG_attributeRet(meshRef->maps[i].tess, ".mixed",
                                 &atype, &alen, &tessFaceQuadMap, &reals, &string);
        AIM_NOTFOUND(aimInfo, status);

        if (tessFaceQuadMap == NULL) continue;

        status = EG_statusTessBody(meshRef->maps[i].tess, &body, &state, &nglobal);
        AIM_STATUS(aimInfo, status);

        status = EG_getBodyTopos(body, NULL, FACE, &nFace, NULL);
        AIM_STATUS(aimInfo, status);

        for (iface = 0; iface < nFace; iface++) {
          status = EG_getTessFace(meshRef->maps[i].tess, iface + 1, &plen, &points, &uv, &ptypes, &pindexs,
                                  &tlen, &tris, &tric);
          AIM_STATUS(aimInfo, status);

          tlen -= 2*tessFaceQuadMap[iface];
          stride = 3*tlen;

          for (j = 0; j < tessFaceQuadMap[iface]; j++, stride += 6) {
            /* quad orientation flipped, per refine convention */
            status = EG_localToGlobal(meshRef->maps[i].tess, iface + 1, tris[stride + 0], &elem[0]);
            AIM_STATUS(aimInfo, status);
            status = EG_localToGlobal(meshRef->maps[i].tess, iface + 1, tris[stride + 1], &elem[3/*1*/]);
            AIM_STATUS(aimInfo, status);
            status = EG_localToGlobal(meshRef->maps[i].tess, iface + 1, tris[stride + 2], &elem[2]);
            AIM_STATUS(aimInfo, status);
            status = EG_localToGlobal(meshRef->maps[i].tess, iface + 1, tris[stride + 5], &elem[1/*3*/]);
            AIM_STATUS(aimInfo, status);

            status = GmfSetLin(fileID, GmfQuadrilaterals, meshRef->maps[i].map[elem[0]-1],
                                                          meshRef->maps[i].map[elem[1]-1],
                                                          meshRef->maps[i].map[elem[2]-1],
                                                          meshRef->maps[i].map[elem[3]-1],
                                                          nFaceOffset + iface + 1);
            if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
          }
        }

        nFaceOffset += nFace;
      }
    }
  }

  // Write any remaining elements
  for (igroup = 0; igroup < meshData->nElemGroup; igroup++) {
    if (meshData->elemGroups[igroup].order != 1) {
      AIM_ERROR(aimInfo, "libMeshb writer currently only supports linear mesh elements! group %d order = %d",
                igroup, meshData->elemGroups[igroup].order);
      status = CAPS_IOERR;
      goto cleanup;
    }

    if (meshData->elemGroups[igroup].elementTopo == aimLine) {
      // written with geometry above
    }

    else if (meshData->elemGroups[igroup].elementTopo == aimTri) {
      if (meshRef->type == aimAreaMesh) {
        status = GmfSetKwd(fileID, GmfTriangles, meshData->elemGroups[igroup].nElems);
        if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

        // element connectivity 1-based
        nPoint = meshData->elemGroups[igroup].nPoint;
        for (ielem = 0; ielem < meshData->elemGroups[igroup].nElems; ielem++) {
          /* triangle orientation flipped, per refine convention */
          status = GmfSetLin(fileID, GmfTriangles, meshData->elemGroups[igroup].elements[nPoint*ielem+1/*0*/],
                                                   meshData->elemGroups[igroup].elements[nPoint*ielem+0/*1*/],
                                                   meshData->elemGroups[igroup].elements[nPoint*ielem+2],
                                                   igroup+1);
          if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
        }
      } else {
        // written with geometry above
      }
    }

    else if (meshData->elemGroups[igroup].elementTopo == aimQuad) {
      if (meshRef->type == aimAreaMesh) {
        status = GmfSetKwd(fileID, GmfQuadrilaterals, meshData->elemGroups[igroup].nElems);
        if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

        // element connectivity 1-based
        nPoint = meshData->elemGroups[igroup].nPoint;
        for (ielem = 0; ielem < meshData->elemGroups[igroup].nElems; ielem++) {
          /* quad orientation flipped, per refine convention */
          status = GmfSetLin(fileID, GmfQuadrilaterals, meshData->elemGroups[igroup].elements[nPoint*ielem+0],
                                                        meshData->elemGroups[igroup].elements[nPoint*ielem+1],
                                                        meshData->elemGroups[igroup].elements[nPoint*ielem+2],
                                                        meshData->elemGroups[igroup].elements[nPoint*ielem+3],
                                                        igroup+1);
          if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
        }
      } else {
        // written with geometry above
      }
    }

    else if (meshData->elemGroups[igroup].elementTopo == aimTet) {
      status = GmfSetKwd(fileID, GmfTetrahedra, meshData->elemGroups[igroup].nElems);
      if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

      // element connectivity 1-based
      nPoint = meshData->elemGroups[igroup].nPoint;
      for (ielem = 0; ielem < meshData->elemGroups[igroup].nElems; ielem++) {
        status = GmfSetLin(fileID, GmfTetrahedra, meshData->elemGroups[igroup].elements[nPoint*ielem+0],
                                                  meshData->elemGroups[igroup].elements[nPoint*ielem+1],
                                                  meshData->elemGroups[igroup].elements[nPoint*ielem+2],
                                                  meshData->elemGroups[igroup].elements[nPoint*ielem+3],
                                                  0); // to be consistent with refine
        if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
      }
    }

    else if (meshData->elemGroups[igroup].elementTopo == aimPyramid) {
        status = GmfSetKwd(fileID, GmfPyramids, meshData->elemGroups[igroup].nElems);
        if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

        // element connectivity 1-based
        nPoint = meshData->elemGroups[igroup].nPoint;
        for (ielem = 0; ielem < meshData->elemGroups[igroup].nElems; ielem++) {
          status = GmfSetLin(fileID, GmfPyramids, meshData->elemGroups[igroup].elements[nPoint*ielem+0],
                                                  meshData->elemGroups[igroup].elements[nPoint*ielem+3],
                                                  meshData->elemGroups[igroup].elements[nPoint*ielem+4],
                                                  meshData->elemGroups[igroup].elements[nPoint*ielem+1],
                                                  meshData->elemGroups[igroup].elements[nPoint*ielem+2],
                                                    0); // to be consistent with refine
          if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
        }
    }

    else if (meshData->elemGroups[igroup].elementTopo == aimPrism) {
        status = GmfSetKwd(fileID, GmfPrisms, meshData->elemGroups[igroup].nElems);
        if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

        // element connectivity 1-based
        nPoint = meshData->elemGroups[igroup].nPoint;
        for (ielem = 0; ielem < meshData->elemGroups[igroup].nElems; ielem++) {
          status = GmfSetLin(fileID, GmfPrisms, meshData->elemGroups[igroup].elements[nPoint*ielem+0],
                                                meshData->elemGroups[igroup].elements[nPoint*ielem+1],
                                                meshData->elemGroups[igroup].elements[nPoint*ielem+2],
                                                meshData->elemGroups[igroup].elements[nPoint*ielem+3],
                                                meshData->elemGroups[igroup].elements[nPoint*ielem+4],
                                                meshData->elemGroups[igroup].elements[nPoint*ielem+5],
                                                0); // to be consistent with refine
          if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
        }
    }

    else if (meshData->elemGroups[igroup].elementTopo == aimHex) {
        status = GmfSetKwd(fileID, GmfHexahedra, meshData->elemGroups[igroup].nElems);
        if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

        // element connectivity 1-based
        nPoint = meshData->elemGroups[igroup].nPoint;
        for (ielem = 0; ielem < meshData->elemGroups[igroup].nElems; ielem++) {
          status = GmfSetLin(fileID, GmfHexahedra, meshData->elemGroups[igroup].elements[nPoint*ielem+0],
                                                   meshData->elemGroups[igroup].elements[nPoint*ielem+1],
                                                   meshData->elemGroups[igroup].elements[nPoint*ielem+2],
                                                   meshData->elemGroups[igroup].elements[nPoint*ielem+3],
                                                   meshData->elemGroups[igroup].elements[nPoint*ielem+4],
                                                   meshData->elemGroups[igroup].elements[nPoint*ielem+5],
                                                   meshData->elemGroups[igroup].elements[nPoint*ielem+6],
                                                   meshData->elemGroups[igroup].elements[nPoint*ielem+7],
                                                   0); // to be consistent with refine
          if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
        }

    } else {
      AIM_ERROR(aimInfo, "libMeshb writer element type currently not implemented! group %d type = %d",
                igroup, meshData->elemGroups[igroup].elementTopo);
      status = CAPS_IOERR;
      goto cleanup;
    }
  }

  nNodeVerts = nEdgeVerts = nFaceVerts = 0;
  for (i = 0; i < meshRef->nmap; i++) {
    status = EG_statusTessBody(meshRef->maps[i].tess, &body, &state, &nglobal);
    AIM_STATUS(aimInfo, status);

    status = EG_getBodyTopos(body, NULL, NODE, &nNode, NULL);
    AIM_STATUS(aimInfo, status);
    nNodeVerts += nNode;

    status = EG_getBodyTopos(body, NULL, EDGE, &nEdge, &edges);
    AIM_STATUS(aimInfo, status);

    for (iedge = 0; iedge < nEdge; iedge++) {
      if (edges[iedge]->mtype == DEGENERATE) continue;
      status = EG_getTessEdge(meshRef->maps[i].tess, iedge + 1, &plen, &points, &t);
      AIM_STATUS(aimInfo, status);
      nEdgeVerts += plen;
    }
    AIM_FREE(edges);

    if (meshRef->type == aimAreaMesh) continue;

    status = EG_getBodyTopos(body, NULL, FACE, &nFace, NULL);
    AIM_STATUS(aimInfo, status);

    for (iface = 0; iface < nFace; iface++) {
      status = EG_getTessFace(meshRef->maps[i].tess, iface + 1, &plen, &points, &uv, &ptypes, &pindexs,
                              &tlen, &tris, &tric);
      AIM_STATUS(aimInfo, status);
      nFaceVerts += plen;
    }
  }

  // write out parametric coordinates

  // Write NODEs
  status = GmfSetKwd(fileID, GmfVerticesOnGeometricVertices, nNodeVerts);
  if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

  nNodeOffset = 0;
  for (i = 0; i < meshRef->nmap; i++) {
    status = EG_statusTessBody(meshRef->maps[i].tess, &body, &state, &nglobal);
    AIM_STATUS(aimInfo, status);

    status = EG_getBodyTopos(body, NULL, NODE, &nNode, NULL);
    AIM_STATUS(aimInfo, status);

    for (j = 0; j < nglobal; j++) {
      status = EG_getGlobal(meshRef->maps[i].tess, j + 1, &ptype, &pindex, xyz);
      AIM_STATUS(aimInfo, status);
      if (ptype == 0) {
        status = GmfSetLin(fileID, GmfVerticesOnGeometricVertices,
                           meshRef->maps[i].map[j],
                           nNodeOffset + pindex);
        if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
      }
    }

    nNodeOffset += nNode;
  }

  // Write EDGEs
  status = GmfSetKwd(fileID, GmfVerticesOnGeometricEdges, nEdgeVerts);
  if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

  nEdgeOffset = 0;
  for (i = 0; i < meshRef->nmap; i++) {
    status = EG_statusTessBody(meshRef->maps[i].tess, &body, &state, &nglobal);
    AIM_STATUS(aimInfo, status);

    status = EG_getBodyTopos(body, NULL, EDGE, &nEdge, NULL);
    AIM_STATUS(aimInfo, status);

    for (iedge = 0; iedge < nEdge; iedge++) {
      status = EG_getTessEdge(meshRef->maps[i].tess, iedge + 1, &plen, &points, &t);
      if (status == EGADS_DEGEN) continue;
      AIM_STATUS(aimInfo, status);

      for (j = 0; j < plen; j++) {

        status = EG_localToGlobal(meshRef->maps[i].tess, -(iedge + 1), j + 1, &iglobal);
        if (status == EGADS_DEGEN) continue;
        AIM_STATUS(aimInfo, status);

        id = nEdgeOffset + iedge + 1;
        status = GmfSetLin(fileID, GmfVerticesOnGeometricEdges,
                           meshRef->maps[i].map[iglobal-1],
                           id,
                           t[j],
                           (double)id);  // refine abuse of dist
        if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
      }
    }

    nEdgeOffset += nEdge;
  }


  if (meshRef->type == aimAreaMesh) {

    // Write Triangles
    status = GmfSetKwd(fileID, GmfVerticesOnGeometricTriangles, meshData->nVertex);
    if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

    status = EG_statusTessBody(meshRef->maps[0].tess, &body, &state, &nglobal);
    AIM_STATUS(aimInfo, status);

    status = EG_getBodyTopos(body, NULL, FACE, &nFace, &faces);
    AIM_STATUS(aimInfo, status);

    for (i = 0; i < meshData->nVertex; i++) {

      status = EG_invEvaluate(faces[0], meshData->verts[i], param, xyz);
      AIM_STATUS(aimInfo, status);

      id = 1;
      status = GmfSetLin(fileID, GmfVerticesOnGeometricTriangles,
                         i+1,
                         id,
                         param[0], param[1],
                         (double)id); // refine abuse of dist
      if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
    }

    AIM_FREE(faces);

  } else {

    // Write FACEs
    status = GmfSetKwd(fileID, GmfVerticesOnGeometricTriangles, nFaceVerts);
    if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

    nFaceOffset = 0;
    for (i = 0; i < meshRef->nmap; i++) {

      status = EG_statusTessBody(meshRef->maps[i].tess, &body, &state, &nglobal);
      AIM_STATUS(aimInfo, status);

      status = EG_getBodyTopos(body, NULL, FACE, &nFace, NULL);
      AIM_STATUS(aimInfo, status);

      for (iface = 0; iface < nFace; iface++) {
        status = EG_getTessFace(meshRef->maps[i].tess, iface + 1, &plen, &points, &uv, &ptypes, &pindexs,
                                &tlen, &tris, &tric);
        AIM_STATUS(aimInfo, status);

        for (j = 0; j < plen; j++) {

          status = EG_localToGlobal(meshRef->maps[i].tess, iface + 1, j + 1, &iglobal);
          AIM_STATUS(aimInfo, status);

          id = nFaceOffset + iface + 1;
          status = GmfSetLin(fileID, GmfVerticesOnGeometricTriangles,
                             meshRef->maps[i].map[iglobal-1],
                             id,
                             uv[2*j+0], uv[2*j+1],
                             (double)id); // refine abuse of dist
          if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
        }
      }

      nFaceOffset += nFace;
    }
  }

  // Write EGADS model
  // NOTE: This must be written at the end of the file.
  // refine does not conform to the libMeshb standard, and the
  // refineAIM will overwrite the ByteFlow section
  AIM_ALLOC(bodies, meshRef->nmap, ego, aimInfo, status);

  for (i = 0; i < meshRef->nmap; i++) {
    status = EG_statusTessBody(meshRef->maps[i].tess, &body, &state, &nglobal);
    AIM_STATUS(aimInfo, status);

    status = EG_copyObject(body, NULL, &bodies[i]);
    AIM_STATUS(aimInfo, status);
  }

  status = EG_getContext(bodies[0], &context);
  AIM_STATUS(aimInfo, status);

  status = EG_makeTopology(context, NULL, MODEL, 0, NULL,
                           meshRef->nmap, bodies, NULL, &model);
  AIM_STATUS(aimInfo, status);

  // export the bytestream of the model
  status = EG_exportModel(model, &nbytes, &stream);
  AIM_STATUS(aimInfo, status);

  // write to the libmeshb file
  status = GmfWriteByteFlow(fileID, stream, nbytes);
  if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
  AIM_FREE(stream);

  //printf("Finished writing libMeshb file\n\n");

  status = CAPS_SUCCESS;

cleanup:
  AIM_FREE(edges);
  AIM_FREE(faces);
  AIM_FREE(bodies);
  AIM_FREE(stream);
  EG_deleteObject(model);
  if (bodies != NULL) {
    for (i = 0; i < meshRef->nmap; i++)
      EG_deleteObject(bodies[i]);
    AIM_FREE(bodies);
  }
  if (fileID != 0) GmfCloseMesh(fileID);
  return status;
}


/*****************************************************************************/
static void swapd(double *xp, double *yp)
{
    double temp = *xp;
    *xp = *yp;
    *yp = temp;
}

static void swapi(int *xp, int *yp)
{
    int temp = *xp;
    *xp = *yp;
    *yp = temp;
}


/*****************************************************************************/
// A function to implement bubble sort
static void
bubbleSortEdge(edgeData *tedge)
{
  int i, j;
  for (i = 0; i < tedge->npts-1; i++)
    // Last i elements are already in place
    for (j = 0; j < tedge->npts-i-1; j++)
      if (tedge->t[j] > tedge->t[j+1]) {
        swapd(&tedge->t[j]      , &tedge->t[j+1]        );
        swapd(&tedge->xyz[3*j+0], &tedge->xyz[3*(j+1)+0]);
        swapd(&tedge->xyz[3*j+1], &tedge->xyz[3*(j+1)+1]);
        swapd(&tedge->xyz[3*j+2], &tedge->xyz[3*(j+1)+2]);
        swapi(&tedge->ivp[j]    , &tedge->ivp[j+1]      );
      }
}


static void
bubbleSortFace(faceData *tface)
{
  int i, j;
  for (i = 0; i < tface->npts-1; i++)
    // Last i elements are already in place
    for (j = 0; j < tface->npts-i-1; j++)
      if (tface->ivp[j] > tface->ivp[j+1]) {
        swapd(&tface->uv[2*j+0] , &tface->uv[2*(j+1)+0] );
        swapd(&tface->uv[2*j+1] , &tface->uv[2*(j+1)+1] );
        swapd(&tface->xyz[3*j+0], &tface->xyz[3*(j+1)+0]);
        swapd(&tface->xyz[3*j+1], &tface->xyz[3*(j+1)+1]);
        swapd(&tface->xyz[3*j+2], &tface->xyz[3*(j+1)+2]);
        swapi(&tface->ivp[j]    , &tface->ivp[j+1]      );
      }
}

// use bisection to find the face Index
static int
faceVertex(const int ivp, faceData *tface)
{
  int i0 = 0;
  int i1 = tface->npts/2;
  int i2 = tface->npts;

  while(tface->ivp[i1] != ivp) {
    if (ivp > tface->ivp[i1]) {
      i0 = i1;
      i1 = (i1 + i2)/2;
    } else {
      i2 = i1;
      i1 = (i0 + i1)/2;
    }
  }

  return i1+1;
}

/*****************************************************************************/

int meshRead(void *aimInfo, aimMesh *mesh)
{
  typedef int INT_2[2];

  int    status = CAPS_SUCCESS;

  char   attrname[128];
  int    nBody=0;
  int    i, j, k, elementIndex, nPoint, ibody, igroup, iglobal, nglobal, localIndex, topoIndex;
  int    nEdgeOffset, nFaceOffset;
  int    nEdgeTotal=0, nFaceTotal=0;
  int    elem[8], ivp, id, npts, ntri, iedge, iface;
  int    meshVersion, nEdgeVerts, nFaceVerts;
  int    nVol=0, nElemTopo[aimNumMeshElem], nPointTopo[aimNumMeshElem], groupTopo[aimNumMeshElem];
  const char *elemTopoName[aimNumMeshElem];
  int    *faceVertID=NULL, *face_tris, *tessFaceQuadMap=NULL;
  int    *edgeID=NULL, *edgeIndex=NULL, *edgeBodyIndex=NULL, *edgeGroups=NULL;
  int    *faceID=NULL, *faceIndex=NULL, *faceBodyIndex=NULL;
  INT_2  *faceGroups=NULL;
  double reference, t, uv[2], double_gref;
  double result[18], params[3];
  double     *face_uv = NULL, *face_xyz = NULL;
  double     v1[3], v2[3], faceNormal[3], triNormal[3], ndot;
  const int    *tris = NULL, *tric = NULL, *ptype = NULL, *pindex = NULL;
  const double *pxyz = NULL, *puv = NULL;
  const char   *groupName = NULL;
  enum aimMeshElem elementTopo;
  char filename[PATH_MAX];
  int64_t fileID=0;
  aimMeshData *meshData = NULL;
  mapAttrToIndexStruct groupMap;
  bodyData *bodydata=NULL;

  const char *intents;
  ego        *bodies, tess, ref, prev, next;

  int oclass, mtype, faceSense, ncild, *senses;
  double limits[4];
  ego surface, geom, *childs;

  int *header=NULL;
  double *gdata=NULL;
  double norm[3];

  if (mesh           == NULL) return CAPS_NULLOBJ;
  if (mesh->meshRef  == NULL) return CAPS_NULLOBJ;
  if (mesh->meshRef->fileName  == NULL) return CAPS_NULLOBJ;

  for (i = 0; i < aimNumMeshElem; i++) {
    nPointTopo[i] = 0;
    nElemTopo[i] = 0;
    groupTopo[i] = -1;
    elemTopoName[i] = NULL;
  }

  nPointTopo[aimLine   ] = 2;
  nPointTopo[aimTri    ] = 3;
  nPointTopo[aimQuad   ] = 4;
  nPointTopo[aimTet    ] = 4;
  nPointTopo[aimPyramid] = 5;
  nPointTopo[aimPrism  ] = 6;
  nPointTopo[aimHex    ] = 8;

/*@-observertrans@*/
  elemTopoName[aimLine   ] = "Line";
  elemTopoName[aimTri    ] = "Triangle";
  elemTopoName[aimQuad   ] = "Quadrilateral";
  elemTopoName[aimTet    ] = "Tetrahedron";
  elemTopoName[aimPyramid] = "Pyramid";
  elemTopoName[aimPrism  ] = "Prism";
  elemTopoName[aimHex    ] = "Hexahedron";
/*@+observertrans@*/

  status = initiate_mapAttrToIndexStruct(&groupMap);
  AIM_STATUS(aimInfo, status);

  status = aim_getBodies(aimInfo, &intents, &nBody, &bodies);
  AIM_STATUS(aimInfo, status);

  status = create_CAPSGroupAttrToIndexMap(nBody,
                                          bodies,
                                          2, // Body, Faces, and Edges
                                          &groupMap);
  AIM_STATUS(aimInfo, status);

  AIM_ALLOC(bodydata, nBody, bodyData, aimInfo, status);
  status = initiate_bodyData(nBody, bodydata);
  AIM_STATUS(aimInfo, status);

  nEdgeTotal = nFaceTotal = 0;
  for (i = 0; i < nBody; i++) {
    status = EG_getBodyTopos(bodies[i], NULL, NODE, &bodydata[i].nnodes, NULL);
    AIM_STATUS(aimInfo, status);

    status = EG_getBodyTopos(bodies[i], NULL, EDGE, &bodydata[i].nedges, &bodydata[i].edges);
    AIM_STATUS(aimInfo, status);
    nEdgeTotal += bodydata[i].nedges;

    AIM_ALLOC(bodydata[i].tedges, bodydata[i].nedges, edgeData, aimInfo, status);
    for (j = 0; j < bodydata[i].nedges; j++) {
      bodydata[i].tedges[j].npts = 0;
      bodydata[i].tedges[j].xyz  = NULL;
      bodydata[i].tedges[j].t    = NULL;
      bodydata[i].tedges[j].ivp  = NULL;
    }

    status = EG_getBodyTopos(bodies[i], NULL, FACE, &bodydata[i].nfaces, &bodydata[i].faces);
    AIM_STATUS(aimInfo, status);
    nFaceTotal += bodydata[i].nfaces;

    AIM_ALLOC(bodydata[i].tfaces, bodydata[i].nfaces, faceData, aimInfo, status);
    for (j = 0; j < bodydata[i].nfaces; j++) {
      bodydata[i].tfaces[j].npts  = 0;
      bodydata[i].tfaces[j].xyz   = NULL;
      bodydata[i].tfaces[j].uv    = NULL;
      bodydata[i].tfaces[j].nquad = 0;
      bodydata[i].tfaces[j].ntri  = 0;
      bodydata[i].tfaces[j].tris  = NULL;
      bodydata[i].tfaces[j].ivp   = NULL;
    }
  }

  AIM_ALLOC(meshData, 1, aimMeshData, aimInfo, status);
  status = aim_initMeshData(meshData);
  AIM_STATUS(aimInfo, status);

  snprintf(filename, PATH_MAX, "%s%s", mesh->meshRef->fileName, MESHEXTENSION);

  fileID = GmfOpenMesh(filename, GmfRead, &meshVersion, &meshData->dim);

  if (fileID == 0) {
    AIM_ERROR(aimInfo, "Cannot open file: %s\n", filename);
    status = CAPS_IOERR;
    goto cleanup;
  }

  // Setup indexing arrays
  AIM_ALLOC(edgeID       , nEdgeTotal, int, aimInfo, status);
  AIM_ALLOC(edgeIndex    , nEdgeTotal, int, aimInfo, status);
  AIM_ALLOC(edgeBodyIndex, nEdgeTotal, int, aimInfo, status);
  AIM_ALLOC(edgeGroups   , nEdgeTotal, int, aimInfo, status);
  for (i = 0; i < nEdgeTotal; i++) {
    edgeID[i] = -1;
    edgeIndex[i] = -1;
    edgeBodyIndex[i] = -1;
    edgeGroups[i] = -1;
  }

  AIM_ALLOC(faceID       , nFaceTotal, int, aimInfo, status);
  AIM_ALLOC(faceIndex    , nFaceTotal, int, aimInfo, status);
  AIM_ALLOC(faceBodyIndex, nFaceTotal, int, aimInfo, status);
  AIM_ALLOC(faceGroups   , nFaceTotal, INT_2, aimInfo, status);
  for (i = 0; i < nFaceTotal; i++) {
    faceID[i] = -1;
    faceIndex[i] = -1;
    faceBodyIndex[i] = -1;
    faceGroups[i][0] = -1;
    faceGroups[i][1] = -1;
  }

  nEdgeOffset = nFaceOffset = 0;
  for (i = 0; i < nBody; i++) {

    for (iedge = 0; iedge < bodydata[i].nedges; iedge++) {

      edgeIndex[nEdgeOffset+iedge] = iedge;
      edgeBodyIndex[nEdgeOffset+iedge] = i;

      // Look for component/boundary ID for attribute mapper based on capsGroup
      status = retrieve_CAPSGroupAttr(bodydata[i].edges[iedge], &groupName);
      if (status == EGADS_NOTFOUND) {
        edgeID[iedge] = iedge;
        continue;
      }
      if (status != CAPS_SUCCESS) {
        AIM_ERROR(aimInfo, "No capsGroup attribute found on Edge %d, unable to assign a boundary index value",
                  iedge+1);
        print_AllAttr( aimInfo, bodydata[i].edges[iedge] );
        goto cleanup;
      }

      /*@-nullpass@*/
      status = get_mapAttrToIndexIndex(&groupMap, groupName, &id);
      AIM_STATUS(aimInfo, status, "Unable to retrieve index from capsGroup: %s",
                 groupName);
      /*@+nullpass@*/

      edgeID[nEdgeOffset+iedge] = id;
    }

    if (bodydata[i].nfaces == 1) {

      status = EG_getTopology(bodydata[i].faces[0], &surface, &oclass, &faceSense,
                              limits, &ncild, &childs, &senses);
      AIM_STATUS(aimInfo, status);

      status = EG_getGeometry(surface, &oclass, &mtype, &geom, &header, &gdata);
      AIM_STATUS(aimInfo, status);

      if (mtype == PLANE && nBody == 1) {
        CROSS(norm, gdata+3, gdata+6);
        norm[2] *= faceSense;
        if (fabs(norm[2] - 1) > 1e-7) {
          AIM_ERROR(aimInfo, "libMeshb requires 2D meshes in the x-y plane with the normal in positive z!");
          status = CAPS_BADVALUE;
          goto cleanup;
        }

        meshData->dim = 2;
      } else {
        meshData->dim = 3;
      }
      AIM_FREE(header);
      AIM_FREE(gdata);

      faceID[nFaceOffset] = 1;
      faceBodyIndex[nFaceOffset] = i;

    } else {

      meshData->dim = 3;

      for (iface = 0; iface < bodydata[i].nfaces; iface++) {

        faceIndex[nFaceOffset+iface] = iface;
        faceBodyIndex[nFaceOffset+iface] = i;

        // Look for component/boundary ID for attribute mapper based on capsGroup
        status = retrieve_CAPSGroupAttr(bodydata[i].faces[iface], &groupName);
        if (status != CAPS_SUCCESS) {
          AIM_ERROR(aimInfo, "No capsGroup attribute found on Face %d, unable to assign a boundary index value",
                    iface+1);
          print_AllAttr( aimInfo, bodydata[i].faces[iface] );
          goto cleanup;
        }

        /*@-nullpass@*/
        status = get_mapAttrToIndexIndex(&groupMap, groupName, &id);
        AIM_STATUS(aimInfo, status, "Unable to retrieve boundary index from capsGroup: %s",
                   groupName);
        /*@+nullpass@*/

        faceID[nFaceOffset+iface] = id;
        faceBodyIndex[nFaceOffset+iface] = i;
      }
    }

    nEdgeOffset += bodydata[i].nedges;
    nFaceOffset += bodydata[i].nfaces;
  }

  meshData->nVertex = GmfStatKwd(fileID, GmfVertices);
  AIM_ALLOC(meshData->verts, meshData->nVertex, aimMeshCoords, aimInfo, status);

  status = GmfGotoKwd(fileID, GmfVertices);
  if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

  // Read nodal coordinates
  if (meshData->dim == 2) {
    for (i = 0; i < meshData->nVertex; i++) {
      status = GmfGetLin(fileID, GmfVertices, &meshData->verts[i][0],
                                              &meshData->verts[i][1], &reference);
      meshData->verts[i][2] = 0;
      if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
    }

    nElemTopo[aimLine] = GmfStatKwd(fileID, GmfEdges);
    nElemTopo[aimTri ] = GmfStatKwd(fileID, GmfTriangles);
    nElemTopo[aimQuad] = GmfStatKwd(fileID, GmfQuadrilaterals);

    mesh->meshRef->type = aimAreaMesh;

  } else {
    for (i = 0; i < meshData->nVertex; i++) {
      status = GmfGetLin(fileID, GmfVertices, &meshData->verts[i][0],
                                              &meshData->verts[i][1],
                                              &meshData->verts[i][2], &reference);
      if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
    }

    nElemTopo[aimTri    ] = GmfStatKwd(fileID, GmfTriangles);
    nElemTopo[aimQuad   ] = GmfStatKwd(fileID, GmfQuadrilaterals);
    nElemTopo[aimTet    ] = GmfStatKwd(fileID, GmfTetrahedra);
    nElemTopo[aimPyramid] = GmfStatKwd(fileID, GmfPyramids);
    nElemTopo[aimPrism  ] = GmfStatKwd(fileID, GmfPrisms);

    nVol = 0;
    for (i = aimTet; i < aimNumMeshElem; i++)
      nVol += nElemTopo[i];

    if (nVol > 0)
      mesh->meshRef->type = aimVolumeMesh;
    else
      mesh->meshRef->type = aimSurfaceMesh;
  }

  meshData->nTotalElems = 0;
  for (i = 0; i < aimNumMeshElem; i++)
    meshData->nTotalElems += nElemTopo[i];

  // allocate the element map that maps back to the original element numbering
  AIM_ALLOC(meshData->elemMap, meshData->nTotalElems, aimMeshIndices, aimInfo, status);

  /* Start of element index */
  elementIndex = 0;

  if (mesh->meshRef->type == aimAreaMesh) {

    // Elements Line
    status = GmfGotoKwd(fileID, GmfEdges);
    if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

    nPoint = 2;
    elementTopo = aimLine;
    for (i = 0; i < nElemTopo[elementTopo]; i++) {

      /* read the element and group */
      status = GmfGetLin(fileID, GmfEdges, &elem[0],
                                           &elem[1], &igroup);
      if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

      if (igroup <= 0 || igroup > nEdgeTotal) {
        AIM_ERROR(aimInfo, "Edge ID %d is out of range [1, %d]!=", igroup, nEdgeTotal);
        status = CAPS_IOERR;
        goto cleanup;
      }
      igroup -= 1; // make zero based

      igroup = edgeID[igroup]-1;

      /* add the group if necessary */
      if (edgeGroups[igroup] == -1) {
        status = get_mapAttrToIndexKeyword(&groupMap, igroup+1, &groupName);
        AIM_NOTFOUND(aimInfo, status);

        status = aim_addMeshElemGroup(aimInfo, groupName, igroup+1, elementTopo, 1, nPoint, meshData);
        AIM_STATUS(aimInfo, status);
        edgeGroups[igroup] = meshData->nElemGroup-1;
      }
      igroup = edgeGroups[igroup];

      /* add the element to the group */
      status = aim_addMeshElem(aimInfo, 1, &meshData->elemGroups[igroup]);
      AIM_STATUS(aimInfo, status);

      /* set the element connectivity */
      for (j = 0; j < nPoint; j++)
        meshData->elemGroups[igroup].elements[nPoint*(meshData->elemGroups[igroup].nElems-1) + j] = elem[j];

      meshData->elemMap[elementIndex][0] = igroup;
      meshData->elemMap[elementIndex][1] = meshData->elemGroups[igroup].nElems-1;

      elementIndex += 1;
    }
  }


  /* Allocate EGADS associativity for Tri/Quad */
  nPoint = 3;
  elementTopo = aimTri;
  if (nElemTopo[elementTopo] > 0) {

    status = GmfGotoKwd(fileID, GmfTriangles);
    if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

    for (i = 0; i < nElemTopo[elementTopo]; i++) {

      /* read the element and group */
      status = GmfGetLin(fileID, GmfTriangles, &elem[0],
                                               &elem[1],
                                               &elem[2], &igroup);
      if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

      if (igroup <= 0) {
        AIM_ERROR(aimInfo, "Group for Triangle %d must be a positive number: %d!", i+1, igroup);
        status = CAPS_IOERR;
        goto cleanup;
      }
      igroup -= 1; // make zero based

      iface = faceIndex[igroup];
      ibody = faceBodyIndex[igroup];
      if (mesh->meshRef->type != aimAreaMesh) {
        bodydata[ibody].tfaces[iface].ntri++;
        igroup = faceID[igroup]-1;
      }

      /* add the group if necessary */
      if (faceGroups[igroup][0] == -1) {
        status = get_mapAttrToIndexKeyword(&groupMap, igroup+1, &groupName);
        if (status != CAPS_SUCCESS && nFaceTotal == 1)
          groupName = NULL;
        else
          AIM_STATUS(aimInfo, status);

        status = aim_addMeshElemGroup(aimInfo, groupName, igroup+1, elementTopo, 1, nPoint, meshData);
        AIM_STATUS(aimInfo, status);
        faceGroups[igroup][0] = meshData->nElemGroup-1;
      }
      igroup = faceGroups[igroup][0];

      meshData->elemGroups[igroup].nElems++;
    }
  }

  nPoint = 4;
  elementTopo = aimQuad;
  if (nElemTopo[elementTopo] > 0) {

    status = GmfGotoKwd(fileID, GmfQuadrilaterals);
    if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

    for (i = 0; i < nElemTopo[elementTopo]; i++) {

      /* read the element and group */
      status = GmfGetLin(fileID, GmfQuadrilaterals, &elem[0],
                                                    &elem[1],
                                                    &elem[2],
                                                    &elem[3], &igroup);
      if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

      if (igroup <= 0) {
        AIM_ERROR(aimInfo, "Group for Quadrilateral %d must be a positive number: %d!", i+1, igroup);
        status = CAPS_IOERR;
        goto cleanup;
      }
      igroup -= 1; // make zero based

      iface = faceIndex[igroup];
      ibody = faceBodyIndex[igroup];
      if (mesh->meshRef->type != aimAreaMesh) {
        bodydata[ibody].tfaces[iface].nquad += 1;
        bodydata[ibody].tfaces[iface].ntri  += 2;
        igroup = faceID[igroup]-1;
      }

      /* add the group if necessary */
      if (faceGroups[igroup][1] == -1) {
        status = get_mapAttrToIndexKeyword(&groupMap, igroup+1, &groupName);
        if (status != CAPS_SUCCESS && nFaceTotal == 1)
          groupName = NULL;
        else
          AIM_STATUS(aimInfo, status);

        status = aim_addMeshElemGroup(aimInfo, groupName, igroup+1, elementTopo, 1, nPoint, meshData);
        AIM_STATUS(aimInfo, status);
        faceGroups[igroup][1] = meshData->nElemGroup-1;
      }
      igroup = faceGroups[igroup][1];

      meshData->elemGroups[igroup].nElems++;
    }
  }

  // allocate the element groups
  for (igroup = 0; igroup < meshData->nElemGroup; igroup++) {
    if (meshData->elemGroups[igroup].elementTopo != aimTri &&
        meshData->elemGroups[igroup].elementTopo != aimQuad) continue;
    status = aim_sizeMeshElem(aimInfo, meshData->elemGroups[igroup].nElems, &meshData->elemGroups[igroup]);
    AIM_STATUS(aimInfo, status);
    meshData->elemGroups[igroup].nElems = 0;
  }

  if (mesh->meshRef->type != aimAreaMesh) {
    for (i = 0; i < nBody; i++) {
      for (j = 0; j < bodydata[i].nfaces; j++) {
        ntri = bodydata[i].tfaces[j].ntri;
        AIM_ALLOC(bodydata[i].tfaces[j].tris, 3*ntri, int, aimInfo, status);
        bodydata[i].tfaces[j].ntri = 0;
      }
    }
  }

  /* Elements triangles */
  nPoint = 3;
  elementTopo = aimTri;
  if (nElemTopo[elementTopo] > 0) {

    status = GmfGotoKwd(fileID, GmfTriangles);
    if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

    for (i = 0; i < nElemTopo[elementTopo]; i++) {

      /* read the element and group */
      status = GmfGetLin(fileID, GmfTriangles, &elem[0],
                                               &elem[1],
                                               &elem[2], &igroup);
      if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

      if (igroup <= 0) {
        AIM_ERROR(aimInfo, "Group for Triangle %d must be a positive number: %d!", i+1, igroup);
        status = CAPS_IOERR;
        goto cleanup;
      }
      igroup -= 1; // make zero based

      if (mesh->meshRef->type != aimAreaMesh) {

        iface = faceIndex[igroup];
        ibody = faceBodyIndex[igroup];
        ntri = bodydata[ibody].tfaces[iface].ntri;
        bodydata[ibody].tfaces[iface].tris[3*ntri+0] = elem[0];
        bodydata[ibody].tfaces[iface].tris[3*ntri+1] = elem[1];
        bodydata[ibody].tfaces[iface].tris[3*ntri+2] = elem[2];
        bodydata[ibody].tfaces[iface].ntri++;

        igroup = faceID[igroup]-1;
      }

      /* get the group */
      igroup = faceGroups[igroup][0];

      /* set the element connectivity */
      for (j = 0; j < nPoint; j++)
        meshData->elemGroups[igroup].elements[nPoint*meshData->elemGroups[igroup].nElems + j] = elem[j];

      meshData->elemGroups[igroup].nElems++;
      meshData->elemMap[elementIndex][0] = igroup;
      meshData->elemMap[elementIndex][1] = meshData->elemGroups[igroup].nElems-1;

      elementIndex += 1;
    }
  }

  /* Elements quadrilaterals */
  nPoint = 4;
  elementTopo = aimQuad;
  if (nElemTopo[elementTopo] > 0) {

    status = GmfGotoKwd(fileID, GmfQuadrilaterals);
    if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

    for (i = 0; i < nElemTopo[elementTopo]; i++) {

      /* read the element and group */
      status = GmfGetLin(fileID, GmfQuadrilaterals, &elem[0],
                                                    &elem[1],
                                                    &elem[2],
                                                    &elem[3], &igroup);
      if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

      if (igroup <= 0) {
        AIM_ERROR(aimInfo, "Group for Triangle %d must be a positive number: %d!", i+1, igroup);
        status = CAPS_IOERR;
        goto cleanup;
      }
      igroup -= 1; // make zero based

      if (mesh->meshRef->type != aimAreaMesh) {

        iface = faceIndex[igroup];
        ibody = faceBodyIndex[igroup];
        ntri = bodydata[ibody].tfaces[iface].ntri;
        bodydata[ibody].tfaces[iface].tris[3*(ntri+0)+0] = elem[0];
        bodydata[ibody].tfaces[iface].tris[3*(ntri+0)+1] = elem[1];
        bodydata[ibody].tfaces[iface].tris[3*(ntri+0)+2] = elem[2];
        bodydata[ibody].tfaces[iface].tris[3*(ntri+1)+0] = elem[0];
        bodydata[ibody].tfaces[iface].tris[3*(ntri+1)+1] = elem[2];
        bodydata[ibody].tfaces[iface].tris[3*(ntri+1)+2] = elem[3];
        bodydata[ibody].tfaces[iface].ntri += 2;

        igroup = faceID[igroup]-1;
      }

      /* get the group */
      igroup = faceGroups[igroup][1];

      /* set the element connectivity */
      for (j = 0; j < nPoint; j++)
        meshData->elemGroups[igroup].elements[nPoint*meshData->elemGroups[igroup].nElems + j] = elem[j];

      meshData->elemGroups[igroup].nElems++;
      meshData->elemMap[elementIndex][0] = igroup;
      meshData->elemMap[elementIndex][1] = meshData->elemGroups[igroup].nElems-1;

      elementIndex += 1;
    }
  }

  // Read volume elements
  if (mesh->meshRef->type == aimVolumeMesh) {

    // Loop over all volume element topologies and read them in
    for (elementTopo = aimTet; elementTopo < aimNumMeshElem; elementTopo++) {

      if (nElemTopo[elementTopo] > 0) {

        /* get the number of points for this topology */
        nPoint = nPointTopo[elementTopo];

        /* add the group */
        if (groupTopo[elementTopo] == -1) {
          status = aim_addMeshElemGroup(aimInfo, NULL, igroup, elementTopo, 1, nPoint, meshData);
          AIM_STATUS(aimInfo, status);
          groupTopo[elementTopo] = meshData->nElemGroup-1;
        }
        igroup = groupTopo[elementTopo];

        /* allocate the element group storage */
        status = aim_addMeshElem(aimInfo, nElemTopo[elementTopo], &meshData->elemGroups[igroup]);
        AIM_STATUS(aimInfo, status);

        if (elementTopo == aimTet) {
          status = GmfGotoKwd(fileID, GmfTetrahedra);
          if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
        } else if (elementTopo == aimPyramid) {
          status = GmfGotoKwd(fileID, GmfPyramids);
          if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
        } else if (elementTopo == aimPrism) {
          status = GmfGotoKwd(fileID, GmfPrisms);
          if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
        } else if (elementTopo == aimHex) {
          status = GmfGotoKwd(fileID, GmfHexahedra);
          if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
        } else {
          AIM_ERROR(aimInfo, "Developer error: Unknown element topo!");
          status = CAPS_NOTIMPLEMENT;
          goto cleanup;
        }

        for (i = 0; i < nElemTopo[elementTopo]; i++) {

          /* read the element and group */
          if (elementTopo == aimTet) {
            status = GmfGetLin(fileID, GmfTetrahedra,
                               &elem[0],
                               &elem[1],
                               &elem[2],
                               &elem[3], &igroup);
            if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
          } else if (elementTopo == aimPyramid) {
            status = GmfGetLin(fileID, GmfPyramids,
                               &elem[0],
                               &elem[3],
                               &elem[4],
                               &elem[1],
                               &elem[2], &igroup);
            if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
          } else if (elementTopo == aimPrism) {
            status = GmfGetLin(fileID, GmfPrisms,
                               &elem[0],
                               &elem[1],
                               &elem[2],
                               &elem[3],
                               &elem[4],
                               &elem[5], &igroup);
            if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
          } else if (elementTopo == aimHex) {
            status = GmfGetLin(fileID, GmfHexahedra,
                               &elem[0],
                               &elem[1],
                               &elem[2],
                               &elem[3],
                               &elem[4],
                               &elem[5],
                               &elem[6],
                               &elem[7], &igroup);
            if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }
          } else {
            AIM_ERROR(aimInfo, "Developer error: Unknown element topo!");
            status = CAPS_NOTIMPLEMENT;
            goto cleanup;
          }
          if (igroup == 1) igroup = 0;

          if (igroup != 0) {
            AIM_ERROR(aimInfo, "%s group must be 0 or 1: %d!", elemTopoName[elementTopo], igroup);
            status = CAPS_IOERR;
            goto cleanup;
          }
          igroup = groupTopo[elementTopo];

          /* set the element connectivity */
          for (j = 0; j < nPoint; j++)
            meshData->elemGroups[igroup].elements[nPoint*i + j] = elem[j];

          meshData->elemMap[elementIndex][0] = igroup;
          meshData->elemMap[elementIndex][1] = i;

          elementIndex += 1;
        }
      }
    }
  }

  // generate tessellation

  // read parametric coordinates

  // Read EDGEs
  nEdgeVerts = GmfStatKwd(fileID, GmfVerticesOnGeometricEdges);
  status = GmfGotoKwd(fileID, GmfVerticesOnGeometricEdges);
  if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

  // first count points on each edge
  for (j = 0; j < nEdgeVerts; j++) {
    status = GmfGetLin(fileID, GmfVerticesOnGeometricEdges,
                       &ivp,
                       &id,
                       &t,
                       &double_gref);  // refine abuse of dist
    if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

    if (id <= 0 || id > nEdgeTotal) {
      AIM_ERROR(aimInfo, "Edge ID %d is out of range [1, %d]", id, nEdgeTotal);
      status = CAPS_IOERR;
      goto cleanup;
    }
    iedge = edgeIndex[id-1];
    ibody = edgeBodyIndex[id-1];
    bodydata[ibody].tedges[iedge].npts++;
  }

  for (i = 0; i < nBody; i++) {
    for (j = 0; j < bodydata[i].nedges; j++) {
      npts = bodydata[i].tedges[j].npts;
      AIM_ALLOC(bodydata[i].tedges[j].xyz, 3*npts, double, aimInfo, status);
      AIM_ALLOC(bodydata[i].tedges[j].t  ,   npts, double, aimInfo, status);
      AIM_ALLOC(bodydata[i].tedges[j].ivp,   npts, int   , aimInfo, status);
      bodydata[i].tedges[j].npts = 0;
    }
  }

  status = GmfGotoKwd(fileID, GmfVerticesOnGeometricEdges);
  if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

  // read the data
  for (j = 0; j < nEdgeVerts; j++) {
    status = GmfGetLin(fileID, GmfVerticesOnGeometricEdges,
                       &ivp,
                       &id,
                       &t,
                       &double_gref);  // refine abuse of dist
    if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

    iedge = edgeIndex[id-1];
    ibody = edgeBodyIndex[id-1];
    npts = bodydata[ibody].tedges[iedge].npts;

    bodydata[ibody].tedges[iedge].t[npts] = t;

    bodydata[ibody].tedges[iedge].xyz[3*npts+0] = meshData->verts[ivp-1][0];
    bodydata[ibody].tedges[iedge].xyz[3*npts+1] = meshData->verts[ivp-1][1];
    bodydata[ibody].tedges[iedge].xyz[3*npts+2] = meshData->verts[ivp-1][2];

    bodydata[ibody].tedges[iedge].ivp[npts] = ivp;

    bodydata[ibody].tedges[iedge].npts++;
  }

  for (i = 0; i < nBody; i++) {
    for (j = 0; j < bodydata[i].nedges; j++) {
      bubbleSortEdge(&bodydata[i].tedges[j]);
    }
  }

  if (mesh->meshRef->type != aimAreaMesh) {
    // Count face points first
    nFaceVerts = GmfStatKwd(fileID, GmfVerticesOnGeometricTriangles);
    status = GmfGotoKwd(fileID, GmfVerticesOnGeometricTriangles);
    if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

    for (j = 0; j < nFaceVerts; j++) {

      status = GmfGetLin(fileID, GmfVerticesOnGeometricTriangles,
                         &ivp,
                         &id,
                         &uv[0], &uv[1],
                         &double_gref); // refine abuse of dist
      if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

      if (id <= 0 || id > nFaceTotal) {
        AIM_ERROR(aimInfo, "Face ID %d is out of range [1, %d]", id, nFaceTotal);
        status = CAPS_IOERR;
        goto cleanup;
      }
      iface = faceIndex[id-1];
      ibody = faceBodyIndex[id-1];
      bodydata[ibody].tfaces[iface].npts++;
    }

    for (i = 0; i < nBody; i++) {
      for (j = 0; j < bodydata[i].nfaces; j++) {
        npts = bodydata[i].tfaces[j].npts;
        AIM_ALLOC(bodydata[i].tfaces[j].xyz , 3*npts, double, aimInfo, status);
        AIM_ALLOC(bodydata[i].tfaces[j].uv  , 2*npts, double, aimInfo, status);
        AIM_ALLOC(bodydata[i].tfaces[j].ivp ,   npts, int   , aimInfo, status);
        bodydata[i].tfaces[j].npts = 0;
      }
    }

    status = GmfGotoKwd(fileID, GmfVerticesOnGeometricTriangles);
    if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

    // read the data
    for (j = 0; j < nFaceVerts; j++) {
      status = GmfGetLin(fileID, GmfVerticesOnGeometricTriangles,
                         &ivp,
                         &id,
                         &uv[0], &uv[1],
                         &double_gref); // refine abuse of dist
      if (status <= 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

      iface = faceIndex[id-1];
      ibody = faceBodyIndex[id-1];

      npts = bodydata[ibody].tfaces[iface].npts;

      bodydata[ibody].tfaces[iface].uv[2*npts+0] = uv[0];
      bodydata[ibody].tfaces[iface].uv[2*npts+1] = uv[1];

      bodydata[ibody].tfaces[iface].xyz[3*npts+0] = meshData->verts[ivp-1][0];
      bodydata[ibody].tfaces[iface].xyz[3*npts+1] = meshData->verts[ivp-1][1];
      bodydata[ibody].tfaces[iface].xyz[3*npts+2] = meshData->verts[ivp-1][2];

      bodydata[ibody].tfaces[iface].ivp[npts] = ivp;

      bodydata[ibody].tfaces[iface].npts++;
    }

    for (i = 0; i < nBody; i++) {
      for (j = 0; j < bodydata[i].nfaces; j++) {
        bubbleSortFace(&bodydata[i].tfaces[j]);

        // get the face triangulation
        for (k = 0; k < bodydata[i].tfaces[j].ntri; k++) {
          bodydata[i].tfaces[j].tris[3*k+0] = faceVertex(bodydata[i].tfaces[j].tris[3*k+0], &bodydata[i].tfaces[j]);
          bodydata[i].tfaces[j].tris[3*k+1] = faceVertex(bodydata[i].tfaces[j].tris[3*k+1], &bodydata[i].tfaces[j]);
          bodydata[i].tfaces[j].tris[3*k+2] = faceVertex(bodydata[i].tfaces[j].tris[3*k+2], &bodydata[i].tfaces[j]);
        }
      }
    }
  }

  aim_freeMeshRefMaps(mesh->meshRef);

  // Allocate surfaceMesh from number of bodies
  AIM_ALLOC(mesh->meshRef->maps, nBody, aimMeshTessMap, aimInfo, status);
  mesh->meshRef->nmap = nBody;
  for (i = 0; i < nBody; i++) {
    mesh->meshRef->maps[i].tess = NULL;
    mesh->meshRef->maps[i].map = NULL;
  }


  for (ibody = 0; ibody < nBody; ibody++) {

    // Build up the body tessellation object
    status = EG_initTessBody(bodies[ibody], &tess);
    AIM_STATUS(aimInfo, status);
    AIM_NOTNULL(tess, aimInfo, status);

    for ( iedge = 0; iedge < bodydata[ibody].nedges; iedge++ ) {

      // Check if the edge is degenerate
      if (bodydata[ibody].edges[iedge]->mtype == DEGENERATE) continue;

      status = EG_setTessEdge(tess, iedge+1,
                              bodydata[ibody].tedges[iedge].npts,
                              bodydata[ibody].tedges[iedge].xyz,
                              bodydata[ibody].tedges[iedge].t);
      AIM_STATUS(aimInfo, status, "Failed to set tessellation on Edge %d!", iedge+1);

      // Add the unique indexing of the edge tessellation
      snprintf(attrname, 128, "edgeVertID_%d",iedge+1);
      status = EG_attributeAdd(tess, attrname, ATTRINT,
                               bodydata[ibody].tedges[iedge].npts,
                               bodydata[ibody].tedges[iedge].ivp, NULL, NULL);
      AIM_STATUS(aimInfo, status);
    }

    if (mesh->meshRef->type == aimAreaMesh) {

      params[0] = -1;
      params[1] = 1;
      params[2] = 20;

      status = EG_finishTess( tess, params );
      AIM_STATUS(aimInfo, status);

    } else {

      for (iface = 0; iface < bodydata[ibody].nfaces; iface++) {

        ntri  = bodydata[ibody].tfaces[iface].ntri;

        face_tris = bodydata[ibody].tfaces[iface].tris;
        face_uv   = bodydata[ibody].tfaces[iface].uv;
        face_xyz  = bodydata[ibody].tfaces[iface].xyz;

        AIM_NOTNULL(face_tris, aimInfo, status);
        AIM_NOTNULL(face_uv  , aimInfo, status);
        AIM_NOTNULL(face_xyz , aimInfo, status);

        // check the normals of the elements match the geometry normals
        // only need to check one element per face to decide for all
        elem[0] = face_tris[0]-1;
        elem[1] = face_tris[1]-1;
        elem[2] = face_tris[2]-1;

        // get the uv centroid
        uv[0] = (face_uv[2*elem[0]+0] + face_uv[2*elem[1]+0] + face_uv[2*elem[2]+0])/3.;
        uv[1] = (face_uv[2*elem[0]+1] + face_uv[2*elem[1]+1] + face_uv[2*elem[2]+1])/3.;

        // get the normal of the face
        status = EG_evaluate(bodydata[ibody].faces[iface], uv, result);
        AIM_STATUS(aimInfo, status);

        // use cross dX/du x dX/dv to get geometry normal
        v1[0] = result[3]; v1[1] = result[4]; v1[2] = result[5];
        v2[0] = result[6]; v2[1] = result[7]; v2[2] = result[8];
        CROSS(faceNormal, v1, v2);

        // get mtype=SFORWARD or mtype=SREVERSE for the face to get topology normal
        status = EG_getInfo(bodydata[ibody].faces[iface], &oclass, &mtype, &ref, &prev, &next);
        AIM_STATUS(aimInfo, status);
        faceNormal[0] *= mtype;
        faceNormal[1] *= mtype;
        faceNormal[2] *= mtype;

        // get the normal of the mesh triangle
        v1[0] = face_xyz[3*elem[1]+0] - face_xyz[3*elem[0]+0];
        v1[1] = face_xyz[3*elem[1]+1] - face_xyz[3*elem[0]+1];
        v1[2] = face_xyz[3*elem[1]+2] - face_xyz[3*elem[0]+2];

        v2[0] = face_xyz[3*elem[2]+0] - face_xyz[3*elem[0]+0];
        v2[1] = face_xyz[3*elem[2]+1] - face_xyz[3*elem[0]+1];
        v2[2] = face_xyz[3*elem[2]+2] - face_xyz[3*elem[0]+2];

        CROSS(triNormal, v1, v2);

        // get the dot product between the triangle and face normals
        ndot = DOT(faceNormal,triNormal);

        // if the normals are opposite, swap all triangles
        if (ndot < 0) {
          // swap two vertices to reverse the normal
          for (i = 0; i < ntri; i++) {
            swapi(&face_tris[3*i+0], &face_tris[3*i+2]);
          }
        }

        status = EG_setTessFace(tess,
                                iface+1,
                                bodydata[ibody].tfaces[iface].npts,
                                face_xyz,
                                face_uv,
                                ntri,
                                face_tris);
        AIM_STATUS(aimInfo, status);


        // The points get reindexed to be consistent with loops in EG_setTessFace
        // This uses the new triangulation to map that index change.
        status = EG_getTessFace(tess, iface+1, &npts, &pxyz, &puv, &ptype,
                                &pindex, &ntri, &tris, &tric);
        AIM_STATUS(aimInfo, status);
        AIM_NOTNULL(tris, aimInfo, status);

        AIM_ALLOC(faceVertID, npts, int, aimInfo, status);

        for (i = 0; i < ntri; i++) {
          for (j = 0; j < 3; j++) {
            faceVertID[tris[3*i+j]-1] = bodydata[ibody].tfaces[iface].ivp[face_tris[3*i+j]-1];
          }
        }

        // Add the unique indexing of the tessellation
        snprintf(attrname, 128, "faceVertID_%d",iface+1);
        status = EG_attributeAdd(tess, attrname, ATTRINT,
                                 bodydata[ibody].tfaces[iface].npts,
                                 faceVertID, NULL, NULL);
        AIM_STATUS(aimInfo, status);

        // replace the shuffled volume ID's
        AIM_FREE(bodydata[ibody].tfaces[iface].ivp);
        bodydata[ibody].tfaces[iface].ivp = faceVertID;
        faceVertID = NULL;
      }
    }

    // finalize the tessellation
    status = EG_statusTessBody(tess, &bodies[ibody], &i, &nglobal);
    AIM_STATUS(aimInfo, status, "Tessellation object was not built correctly!!!");

    // save the tessellation with caps
    status = aim_newTess(aimInfo, tess);
    AIM_STATUS(aimInfo, status);

    /*@-kepttrans@*/
    // reference the surface mesh object
    mesh->meshRef->maps[ibody].tess = tess;
    tess = NULL;
    /*@+kepttrans@*/


    if (nElemTopo[aimQuad] > 0) {
      AIM_ALLOC(tessFaceQuadMap, bodydata[ibody].nfaces, int, aimInfo, status);
      nVol = 0;
      for (iface = 0; iface < bodydata[ibody].nfaces; iface++)
        nVol += tessFaceQuadMap[iface] = bodydata[ibody].tfaces[iface].nquad;

      if (nVol > 0) {
        // check if the tessellation has a mixture of quad and tri
        status = EG_attributeAdd(mesh->meshRef->maps[ibody].tess, ".mixed",
                                 ATTRINT, bodydata[ibody].nfaces, tessFaceQuadMap, NULL, NULL);
        AIM_STATUS(aimInfo, status);
      }
      AIM_FREE(tessFaceQuadMap);
    }


    // Create the map from the tessellation global vertex index to the volume mesh vertex index
    AIM_ALLOC(mesh->meshRef->maps[ibody].map, nglobal, int, aimInfo, status);

    if (mesh->meshRef->type == aimSurfaceMesh) {

      /* construct global vertex indices
       * EGADS re-ordres vertexes, so the meshb file must be written
       * back out to be consistent with the tess object. This implies
       * the mapping is simply an identity.
       */
      for (i = 0; i < nglobal; i++) {
        mesh->meshRef->maps[ibody].map[i] = i+1;
      }

    } else if (mesh->meshRef->type == aimAreaMesh) {

      // Find the boundary mesh in the global tessellation
      for (i = 0; i < nglobal; i++) {

        // Get the local indexes from the boundary mesh
        status = EG_getGlobal(mesh->meshRef->maps[ibody].tess, i+1, &localIndex, &topoIndex, NULL);
        AIM_STATUS(aimInfo, status);

        // Get the global index in the full 2D mesh
        if (localIndex == 0) {
          status = EG_localToGlobal(mesh->meshRef->maps[ibody].tess, 0, topoIndex, &iglobal);
          AIM_STATUS(aimInfo, status);
        } else if (topoIndex > 0) {
          status = EG_localToGlobal(mesh->meshRef->maps[ibody].tess, -topoIndex, localIndex, &iglobal);
          AIM_STATUS(aimInfo, status);
        } else {
          AIM_ERROR(aimInfo, "Developer exception! Should not find Face index!");
          status = CAPS_NOTIMPLEMENT;
          goto cleanup;
        }

        mesh->meshRef->maps[ibody].map[i] = iglobal;
      }

    } else {

      for (iface = 0; iface < bodydata[ibody].nfaces; iface++) {
        status = EG_getTessFace(mesh->meshRef->maps[ibody].tess, iface+1, &npts, &pxyz, &puv, &ptype,
                                &pindex, &ntri, &tris, &tric);
        AIM_STATUS(aimInfo, status);

        /* construct global vertex indices */
        for (i = 0; i < npts; i++) {
          status = EG_localToGlobal(mesh->meshRef->maps[ibody].tess, iface+1, i+1, &iglobal);
          AIM_STATUS(aimInfo, status);
          mesh->meshRef->maps[ibody].map[iglobal-1] = bodydata[ibody].tfaces[iface].ivp[i];
        }
      }
    }

  } // ibody

  mesh->meshData = meshData;
  meshData = NULL;

  status = CAPS_SUCCESS;

cleanup:
  if (status != CAPS_SUCCESS) {
    aim_freeMeshData(meshData);
    AIM_FREE(meshData);
  }

  if (fileID != 0) GmfCloseMesh(fileID);

  destroy_mapAttrToIndexStruct(&groupMap);

  destroy_bodyData(nBody, bodydata);
  AIM_FREE(bodydata);

  AIM_FREE(header);
  AIM_FREE(gdata);

  AIM_FREE(tessFaceQuadMap);

  AIM_FREE(edgeID       );
  AIM_FREE(edgeIndex    );
  AIM_FREE(edgeBodyIndex);
  AIM_FREE(edgeGroups   );
  AIM_FREE(faceID       );
  AIM_FREE(faceIndex    );
  AIM_FREE(faceBodyIndex);
  AIM_FREE(faceGroups   );


  return status;
}
