/*
 ************************************************************************
 *                                                                      *
 * udpArraySort -- sort arrays of data based on a column index          *
 *                                                                      *
 *            Written by Marshall Galbraith @ MIT                       *
 *            Patterned after code written by                           *
 *            John Dannenhoffer @ Syracuse University and               *
 *            Bob Haimes  @ MIT                                         *
 *                                                                      *
 ************************************************************************
 */

/*
 * Copyright (C) 2013/2026  Marshall Galbraith @ (MIT)
 *
 * This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *     MA  02110-1301  USA
 */

#define NUMUDPARGS       4
#define NUMUDPINPUTBODYS 1
#include "udpUtilities.h"
#include "common.h"

/* shorthands for accessing argument values and velocities */
#define ARRAY_ARG(UDP   )                   (UDP)->arg[0]
#define ARRAY(   UDP,I,J) ((const double *) (UDP->arg[0].val))[(I)*UDP->arg[0].ncol+(J)]
#define ICOL(       UDP ) ((const    int *) (UDP->arg[1].val))[0]-1
#define REVERSED(   UDP ) ((const    int *) (UDP->arg[2].val))[0]
#define SORTED_ARG(UDP  )                   (UDP)->arg[3]
#define SORTED(  UDP,I,J) ((      double *) (UDP->arg[3].val))[(I)*UDP->arg[3].ncol+(J)]

/* data about possible arguments */
static char  *argNames[NUMUDPARGS] = {"array",  "icol",  "reversed", "sorted",  };
static int    argTypes[NUMUDPARGS] = {ATTRREAL, ATTRINT, ATTRINT,    -ATTRREAL, };
static int    argIdefs[NUMUDPARGS] = {0,        1,       0,          0,         };
static double argDdefs[NUMUDPARGS] = {0.,       0.,      0.,         0.,        };

/* get utility routines: udpErrorStr, udpInitialize, udpReset, udpSet,
                         udpGet, udpVel, udpClean, udpMesh */
#include "udpUtilities.c"
#include <assert.h>

static void
swap(double *v0, double *v1)
{
  double tmp = *v1;
  *v1 = *v0;
  *v0 = tmp;
}

static void
bubbleSort(udp_T *udp)
{
  int i, j, k;
  for (i = 0; i < SORTED_ARG(udp).nrow-1; i++)
    // Last i elements are already in place
    for (j = 0; j < SORTED_ARG(udp).nrow-i-1; j++)
      if (SORTED(udp,j,ICOL(udp)) > SORTED(udp,j+1,ICOL(udp)))
        for (k = 0; k < SORTED_ARG(udp).ncol; k++)
          swap(&SORTED(udp,j,k), &SORTED(udp,j+1,k));
}

static void
bubbleSortRev(udp_T *udp)
{
  int i, j, k;
  for (i = 0; i < SORTED_ARG(udp).nrow-1; i++)
    // Last i elements are already in place
    for (j = 0; j < SORTED_ARG(udp).nrow-i-1; j++)
      if (SORTED(udp,j,ICOL(udp)) < SORTED(udp,j+1,ICOL(udp)))
        for (k = 0; k < SORTED_ARG(udp).ncol; k++)
          swap(&SORTED(udp,j,k), &SORTED(udp,j+1,k));
}


/*
 ************************************************************************
 *                                                                      *
 *   udpExecute - execute the primitive                                 *
 *                                                                      *
 ************************************************************************
 */

int
udpExecute(ego  emodel,                 /* (in)  Model containing Body */
           ego  *ebody,                 /* (out) Body pointer */
           int  *nMesh,                 /* (out) number of associated meshes */
           char *string[])              /* (out) error message */
{
    int     status = EGADS_SUCCESS;

    int     i, j;
    int     oclass, mtype, nchild, *senses;
    double  data[18];
    char    *message=NULL;
    ego     eref, *ebodys;
    void    *realloc_temp = NULL;            /* used by RALLOC macro */
    udp_T   *udps = *Udps;
    udp_T   *udp = &udps[0];

    ROUTINE(udpExecute);

    /* --------------------------------------------------------------- */

    /* default return values */
    *ebody  = NULL;
    *nMesh  = 0;
    *string = NULL;

    MALLOC(message, char, 100);
    message[0] = '\0';

    /* check that Model was input that contains one Body */
    status = EG_getTopology(emodel, &eref, &oclass, &mtype,
                            data, &nchild, &ebodys, &senses);
    CHECK_STATUS(EG_getTopology);

    if (oclass != MODEL) {
        snprintf(message, 100, "expecting a Model\n");
        status = EGADS_NOTMODEL;
        goto cleanup;
    } else if (nchild != 1) {
        snprintf(message, 100, "expecting Model to contain one Body (not %d)\n", nchild);
        status = EGADS_NOTBODY;
        goto cleanup;
    }

    if (ICOL(udp) < 0) {
        snprintf(message, 100, "expecting icol >= 1, icol = %d\n", ICOL(udp)+1);
        status = EGADS_NODATA;
        goto cleanup;
    } else if (ICOL(udp) >= ARRAY_ARG(udp).ncol) {
        snprintf(message, 100, "expecting icol <= array.ncol, icol = %d\n", ICOL(udp)+1);
        status = EGADS_NODATA;
        goto cleanup;
    }

#ifdef DEBUG
    printf("ARRAY=\n");
    for (i = 0; i < ARRAY_ARG(udp).nrow; i++) {
        for (j = 0; j < ARRAY_ARG(udp).ncol; j++) {
            printf(" %10.5f", ARRAY(udp,i,j));
        }
        printf("\n");
    }
    printf("REVERSED=%d\n", REVERSED(udp));
#endif

    /* make enough room for the interpolated output */
    SORTED_ARG(udp).size = ARRAY_ARG(udp).size;
    SORTED_ARG(udp).nrow = ARRAY_ARG(udp).nrow;
    SORTED_ARG(udp).ncol = ARRAY_ARG(udp).ncol;

    RALLOC(SORTED_ARG(udp).val, double, SORTED_ARG(udp).size);
    RALLOC(SORTED_ARG(udp).dot, double, SORTED_ARG(udp).size);

    for (i = 0; i < ARRAY_ARG(udp).nrow; i++) {
        for (j = 0; j < ARRAY_ARG(udp).ncol; j++) {
          SORTED(udp,i,j) = ARRAY(udp,i,j);
        }
    }

    if (REVERSED(udp) == 0) {
      bubbleSort(udp);
    } else {
      bubbleSortRev(udp);
    }


#ifdef DEBUG
    printf("SORTED=\n");
    for (i = 0; i < SORTED_ARG(udp).nrow; i++) {
        for (j = 0; j < SORTED_ARG(udp).ncol; j++) {
            printf(" %10.5f", SORTED(udp,i,j));
        }
        printf("\n");
    }
#endif

    /* cache copy of arguments for future use */
    status = cacheUdp(NULL);
    CHECK_STATUS(cacheUdp);

    /* make a copy of the Body (so that it does not get removed
     when OpenCSM deletes emodel) */
    status = EG_copyObject(ebodys[0], NULL, ebody);
    CHECK_STATUS(EG_copyObject);
    if (*ebody == NULL) goto cleanup;   // needed for splint

    /* add a special Attribute to the Body to tell OpenCSM that there
       is no topological change and hence it should not adjust the
       Attributes on the Body in finishBody() */
    status = EG_attributeAdd(*ebody, "__noTopoChange__", ATTRSTRING,
                             0, NULL, NULL, "udfEditAttr");
    CHECK_STATUS(EG_attributeAdd);

    /* remember this model (body) */
    udps[numUdp].ebody = *ebody;

cleanup:
#ifdef DEBUG
    printf("udpExecute -> *ebody=%llx\n", (long long)(*ebody));
#endif

    if (strlen(message) > 0) {
        *string = message;
        printf("%s\n", message);
    } else if (status != EGADS_SUCCESS) {
        FREE(message);
        *string = udpErrorStr(status);
    } else {
        FREE(message);
    }

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   udpSensitivity - return sensitivity derivatives for the "real" argument *
 *                                                                      *
 ************************************************************************
 */

int
udpSensitivity(ego    ebody,            /* (in)  Body pointer */
   /*@unused@*/int    npnt,             /* (in)  number of points */
   /*@unused@*/int    entType,          /* (in)  OCSM entity type */
   /*@unused@*/int    entIndex,         /* (in)  OCSM entity index (bias-1) */
   /*@unused@*/double uvs[],            /* (in)  parametric coordinates for evaluation */
   /*@unused@*/double vels[])           /* (out) velocities */
{
    int    status = EGADS_SUCCESS;

    int    iudp, judp;

    ROUTINE(udpSensitivity);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    printf("udpSensitivity(ebody=%llx, npnt=%d, entType=%d, entIndex=%d)\n",
           (long long)ebody, npnt, entType, entIndex);
#endif

    /* check that ebody matches one of the ebodys */
    iudp = 0;
    for (judp = 1; judp <= numUdp; judp++) {
        if (ebody == udps[judp].ebody) {
            iudp = judp;
            break;
        }
    }
    if (iudp <= 0) {
        status = EGADS_NOTMODEL;
        goto cleanup;
    }


    /* this routine is not written yet */
    status = EGADS_NOLOAD;

cleanup:
    return status;
}

