
# Import pyCAPS module
import pyCAPS

# Import modules
import os
import sys
import shutil
import argparse

# Setup and read command line options. Please note that this isn't required for pyCAPS
parser = argparse.ArgumentParser(description = 'Astros AGARD445.6 Pytest Example',
                                 prog = 'astros_AGARD445_PyTest',
                                 formatter_class = argparse.ArgumentDefaultsHelpFormatter)

#Setup the available commandline options
parser.add_argument('-workDir', default = ["."+os.sep], nargs=1, type=str, help = 'Set working/run directory')
parser.add_argument('-noAnalysis', action='store_true', default = False, help = "Don't run analysis code")
parser.add_argument("-outLevel", default = 1, type=int, choices=[0, 1, 2], help="Set output verbosity")
args = parser.parse_args()

#------------------------------------------------------------------
# ASTROS_ROOT should be the path containing ASTRO.D01 and ASTRO.IDX
#------------------------------------------------------------------
if args.noAnalysis == False:
    try:
       ASTROS_ROOT = os.environ["ASTROS_ROOT"]
       os.putenv("PATH", ASTROS_ROOT + os.pathsep + os.getenv("PATH"))
    except KeyError:
       print("Please set the environment variable ASTROS_ROOT")
       sys.exit(1)
#------------------------------------------------------------------

# Create project name
projectName = "AstrosModalAGARD445"

workDir = str(args.workDir[0]) + projectName

# Load CSM file
geometryScript = os.path.join("..","csmData","feaAGARD445.csm")
capsProblem = pyCAPS.Problem(problemName=workDir,
                             capsFile=geometryScript, 
                             outLevel=args.outLevel)

# Change the sweepAngle and span of the Geometry - Demo purposes
#capsProblem.geometry.despmtr.sweepAngle = 5 # From 45 to 5 degrees
#capsProblem.geometry.despmtr.semiSpan   = 5 # From 2.5 ft to 5 ft

# Load astros aim
astros = capsProblem.analysis.create(aim = "astrosAIM",
                                     name = "astros",
                                     autoExec = not args.noAnalysis)

# Set project name
astros.input.Proj_Name = projectName

# Set meshing parameters
astros.input.Edge_Point_Max = 10
astros.input.Edge_Point_Min = 6

astros.input.Quad_Mesh = True

astros.input.Tess_Params = [.25,.01,15]

# Set analysis type
astros.input.Analysis_Type = "Modal"

# Set analysis inputs
eigen = { "extractionMethod"     : "MGIV", # "Lanczos",
          "frequencyRange"       : [0.1, 200],
          "numEstEigenvalue"     : 1,
          "numDesiredEigenvalue" : 2,
          "eigenNormalization"   : "MASS",
	      "lanczosMode"          : 2,  # Default - not necesssary
          "lanczosType"          : "DPB"} # Default - not necesssary

astros.input.Analysis = {"EigenAnalysis": eigen}

# Set materials
mahogany    = {"materialType"        : "orthotropic",
               "youngModulus"        : 0.457E6 ,
               "youngModulusLateral" : 0.0636E6,
               "poissonRatio"        : 0.31,
               "shearModulus"        : 0.0637E6,
               "shearModulusTrans1Z" : 0.00227E6,
               "shearModulusTrans2Z" : 0.00227E6,
               "density"             : 3.5742E-5}

astros.input.Material = {"Mahogany": mahogany}

# Set properties
shell  = {"propertyType" : "Shell",
          "membraneThickness" : 0.82,
          "material"        : "mahogany",
          "bendingInertiaRatio" : 1.0, # Default - not necesssary
          "shearMembraneRatio"  : 5.0/6.0} # Default - not necesssary

astros.input.Property = {"yatesPlate": shell}

# Set constraints
constraint = {"groupName" : "constEdge",
              "dofConstraint" : 123456}

astros.input.Constraint = {"edgeConstraint": constraint}

# astros is executed automatically just-in-time
if args.noAnalysis:
    astros.preAnalysis()
    exit()

# Get Eigen-frequencies
print ("\nGetting results natural frequencies.....")
natrualFreq = astros.output.EigenFrequency

mode = 1
for i in natrualFreq:
    print ("Natural freq (Mode {:d}) = ".format(mode) + '{:.2f} '.format(i) + "(Hz)")
    mode += 1
