/*
 *      CAPS: Computational Aircraft Prototype Syntheses
 *
 *             vtu 3D Mesh Writer Code
 *
 *      Copyright 2014-2026, Massachusetts Institute of Technology
 *      Licensed under The GNU Lesser General Public License, version 2.1
 *      See http://www.opensource.org/licenses/lgpl-2.1.php
 *
 */

#include <vtkPoints.h>
#include <vtkUnsignedCharArray.h>
#include <vtkCellArray.h>
#include <vtkFloatArray.h>
#include <vtkCellData.h>
#include <vtkUnstructuredGrid.h>
#include <vtkXMLUnstructuredGridWriter.h>

#include <string.h>
#include "aimUtil.h"
#include "aimMesh.h"

#include "vtuWriter.h"

extern "C"
const char *meshExtension()
{
/*@-observertrans@*/
  return MESHEXTENSION;
/*@+observertrans@*/
}

/* =================================================================== */
extern "C"
int meshWrite(void *aimInfo, aimMesh *mesh)
{
  int  status; // Function return status

  int         i, nPoint, nElems, elemType;
  vtkIdType   numCells=0, nTable=0;
  int         ivrtx, ipnt, igroup, ielem;
  int         PyramidVTK2UGRID[5] = {0,3,4,1,2};
  char        filename[PATH_MAX];
  aimMeshData *meshData = NULL;
  vtkIdType   ptIds[8];
  vtkNew<vtkPoints> points;
  vtkNew<vtkUnsignedCharArray> shapes;
  vtkNew<vtkCellArray> connectivity;
  vtkNew<vtkFloatArray> cellField;
  vtkNew<vtkUnstructuredGrid> ugrid;
  vtkNew<vtkXMLUnstructuredGridWriter> writer;

  /* ---------------------------------------- */

  printf("\nWriting VTU file ....\n");

  if (mesh == NULL) return CAPS_NULLVALUE;
  if (mesh->meshRef  == NULL) return CAPS_NULLVALUE;
  if (mesh->meshData == NULL) return CAPS_NULLVALUE;

  meshData = mesh->meshData;

  for (igroup = 0; igroup < meshData->nElemGroup; igroup++) {
    if (meshData->elemGroups[igroup].order != 1) {
      AIM_ERROR(aimInfo, "VTK files only supports linear mesh elements! group %d order = %d",
                igroup, meshData->elemGroups[igroup].order);
      status = CAPS_IOERR;
      goto cleanup;
    }

    // count the number of elements and connectivity table size
    nPoint = meshData->elemGroups[igroup].nPoint;
    nElems = meshData->elemGroups[igroup].nElems;

    numCells += nElems;
    nTable += nElems*(nPoint+1);
  }

  /* set all of the verticies */
  points->SetNumberOfPoints(meshData->nVertex);
  for (ivrtx = 0; ivrtx < meshData->nVertex; ivrtx++) {
    points->SetPoint(ivrtx, meshData->verts[ivrtx]);
  }
  points->Modified();

  shapes->SetNumberOfComponents(1);
  shapes->SetNumberOfTuples(numCells);
  connectivity->AllocateExact(numCells, nTable);
  cellField->SetName("ID");
  cellField->SetNumberOfComponents(1);
  cellField->SetNumberOfTuples(numCells);


  /* write out each of the cells */
  if (meshData->elemMap != NULL) {

    /* Write elements in order from the mesh generator */
    for (i = 0; i < meshData->nTotalElems; i++) {

      igroup = meshData->elemMap[i][0];
      ielem  = meshData->elemMap[i][1];

           if (meshData->elemGroups[igroup].elementTopo == aimLine)    elemType = 3;
      else if (meshData->elemGroups[igroup].elementTopo == aimTri)     elemType = 5;
      else if (meshData->elemGroups[igroup].elementTopo == aimQuad)    elemType = 9;
      else if (meshData->elemGroups[igroup].elementTopo == aimTet)     elemType = 10;
      else if (meshData->elemGroups[igroup].elementTopo == aimPyramid) elemType = 14;
      else if (meshData->elemGroups[igroup].elementTopo == aimPrism)   elemType = 13;
      else if (meshData->elemGroups[igroup].elementTopo == aimHex)     elemType = 12;
      else { AIM_ERROR(aimInfo, "Unknown element type"); status = CAPS_NOTIMPLEMENT; goto cleanup; }

      /* set the element type */
      shapes->SetValue(i, elemType);

      /* set the element ID */
      cellField->SetValue(i, meshData->elemGroups[igroup].ID);

      /* number of points in the element */
      nPoint = meshData->elemGroups[igroup].nPoint;

      /* VTK is 0-based indexed */
      if (meshData->elemGroups[igroup].elementTopo == aimPyramid) {
        for (ipnt = 0; ipnt < nPoint; ipnt++) {
          ptIds[ipnt] = meshData->elemGroups[igroup].elements[nPoint*ielem+PyramidVTK2UGRID[ipnt]]-1;
        }
      } else {
        for (ipnt = 0; ipnt < nPoint; ipnt++) {
          ptIds[ipnt] = meshData->elemGroups[igroup].elements[nPoint*ielem+ipnt]-1;
        }
      }

      /* insert the cell */
      connectivity->InsertNextCell(nPoint, ptIds);
    }

  } else {

    /* Write elements by grouping */
    for (igroup = 0; igroup < meshData->nElemGroup; igroup++) {

           if (meshData->elemGroups[igroup].elementTopo == aimLine)    elemType = 3;
      else if (meshData->elemGroups[igroup].elementTopo == aimTri)     elemType = 5;
      else if (meshData->elemGroups[igroup].elementTopo == aimQuad)    elemType = 9;
      else if (meshData->elemGroups[igroup].elementTopo == aimTet)     elemType = 10;
      else if (meshData->elemGroups[igroup].elementTopo == aimPyramid) elemType = 14;
      else if (meshData->elemGroups[igroup].elementTopo == aimPrism)   elemType = 13;
      else if (meshData->elemGroups[igroup].elementTopo == aimHex)     elemType = 12;
      else { AIM_ERROR(aimInfo, "Unknown element type"); status = CAPS_NOTIMPLEMENT; goto cleanup; }

      nPoint = meshData->elemGroups[igroup].nPoint;
      nElems = meshData->elemGroups[igroup].nElems;

      for (ielem = 0; ielem < nElems; ielem++) {

        /* set the element type */
        shapes->SetValue(ielem, elemType);

        /* set the element ID */
        cellField->SetValue(ielem, meshData->elemGroups[igroup].ID);

        /* VTK is 0-based indexed */
        if (meshData->elemGroups[igroup].elementTopo == aimPyramid) {
          for (ipnt = 0; ipnt < nPoint; ipnt++) {
            ptIds[ipnt] = meshData->elemGroups[igroup].elements[nPoint*ielem+PyramidVTK2UGRID[ipnt]]-1;
          }
        } else {
          for (ipnt = 0; ipnt < nPoint; ipnt++) {
            ptIds[ipnt] = meshData->elemGroups[igroup].elements[nPoint*ielem+ipnt]-1;
          }
        }

        /* insert the cell */
        connectivity->InsertNextCell(nPoint, ptIds);
      }
    }
  }

  ugrid->SetPoints(points);
  ugrid->SetCells(shapes, connectivity);
  ugrid->GetCellData()->AddArray(cellField);

  /* the file name */
  (void) snprintf(filename, PATH_MAX, "%s%s", mesh->meshRef->fileName, MESHEXTENSION);

  writer->SetInputData(ugrid);
  writer->SetFileName(filename);
  writer->SetDataModeToBinary();
  writer->Update();

  printf("Finished writing VTU file\n\n");

  status = CAPS_SUCCESS;

cleanup:

  return status;
}
