/*
 ************************************************************************
 *                                                                      *
 * timMiniopt -- Tool Integration Module for MiniOpt                    *
 *                                                                      *
 *            Written by John Dannenhoffer@ Syracuse University         *
 *                                                                      *
 ************************************************************************
 */

/*
 * Copyright (C) 2013/2026  John F. Dannenhoffer, III (Syracuse University)
 *
 * This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *     MA  02110-1301  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <math.h>
#include <assert.h>

#include "nlopt.h"
#include "OpenCSM.h"
#include "tim.h"
#include "wsserver.h"

#define CINT     const int
#define CDOUBLE  const double
#define CCHAR    const char

#define MAX_OPTVARS  100
#define MAX_CONSTRS  100

static int       outLevel   = 1;

typedef struct {
    modl_T    *MODL;          /* pointer to MODL */
    nlopt_opt opt;            /* nlopt context */

    int       iobjfun;        /* pmtr index of "objfun" */
    double    objfun_value;   /* current objective function value */

    int       ioptvar;        /* pmtr index of "optvar" */
    int       noptvar;        /* size       of "optvar" */
    double    *optvar_orig;   /* original values of "optvar" */
    double    *optvar_new;    /* new      values of "optvar" */

    int       iconstr;        /* pmtr index of "constr" */
    int       nconstr;        /* size       of "constr" */
    double    constr_max;     /* maximum contraint value */

    int       istep;          /* iteration step */
    int       verify;         /* verification flag in MODL */
} miniopt_T;

static double myObjfun(unsigned n, const double optvar[], double grad[], void *userdata);
static void   myConstraints(unsigned m, double constr[], unsigned n, const double optvar[], double grad[], void *userdata);


/***********************************************************************/
/*                                                                     */
/*   timLoad - open a tim instance                                     */
/*                                                                     */
/***********************************************************************/

int
timLoad(esp_T *ESP,                     /* (in)  pointer to ESP structure */
/*@unused@*/void  *dummy)               /* (in)  not used */
{
    int    status=0;                    /* (out) return status */

    int     itype, nrow, ncol, ioptvar, iconstr;
    int     major, minor, bugfix;
    double  dot, optvar[MAX_OPTVARS], lb[MAX_OPTVARS], ub[MAX_OPTVARS], tol[MAX_CONSTRS];
    char    name[MAX_NAME_LEN], response[MAX_EXPR_LEN];

    modl_T     *MODL   = NULL;
    miniopt_T  *miniopt= NULL;

    ROUTINE(timLoad(miniopt));

    /* --------------------------------------------------------------- */

    outLevel = ocsmSetOutLevel(-1);

    if (ESP == NULL) {
        printf("ERROR:: cannot run timMitten without serveESP\n");
        status = EGADS_SEQUERR;
        goto cleanup;
    }

    /* create the miniopt_T structure */
    if (ESP->nudata >= MAX_TIM_NESTING) {
        printf("ERROR:: cannot nest more than %d TIMs\n", MAX_TIM_NESTING);
        exit(0);
    }

    ESP->nudata++;
    MALLOC(ESP->udata[ESP->nudata-1], miniopt_T, 1);

    strcpy(ESP->timName[ESP->nudata-1], "miniopt");

    miniopt = (miniopt_T *) (ESP->udata[ESP->nudata-1]);

    MODL    = (modl_T    *) (ESP->MODL);

    /* initialize the structure */
    miniopt->MODL         = MODL;
    miniopt->opt          = NULL;

    miniopt->iobjfun      = 0;
    miniopt->objfun_value = 1e+6;

    miniopt->ioptvar      = 0;
    miniopt->noptvar      = 0;
    miniopt->optvar_orig  = NULL;
    miniopt->optvar_new   = NULL;

    miniopt->iconstr      = 0;
    miniopt->nconstr      = 0;
    miniopt->constr_max   = 1e+6;

    miniopt->istep        = 0;

    /* find the indicies of "objfun", "optvar", and "constr" */
    status = ocsmFindPmtr(MODL, "objfun", 0, 0, 0, &(miniopt->iobjfun));
    if (status != SUCCESS) {
        status = OCSM_NAME_NOT_FOUND;
        SPRINT0(0, "ERROR:: \"objfun\" not found");
        goto cleanup;
    }

    status = ocsmGetPmtr(MODL, miniopt->iobjfun, &itype, &nrow, &ncol, name);
    CHECK_STATUS(ocsmGetPmtr);

    if (itype != OCSM_OUTPMTR) {
        status = OCSM_WRONG_PMTR_TYPE;
        SPRINT0(0, "ERROR:: \"objfun\" is not an OUTPMTR");
        goto cleanup;
    }


    status = ocsmFindPmtr(MODL, "optvar", 0, 0, 0, &(miniopt->ioptvar));
    if (status != SUCCESS) {
        status = OCSM_NAME_NOT_FOUND;
        SPRINT0(0, "ERROR:: \"optvar\" not found");
        goto cleanup;
    }

    status = ocsmGetPmtr(MODL, miniopt->ioptvar, &itype, &nrow, &ncol, name);
    CHECK_STATUS(ocsmGetPmtr);

    if (itype != OCSM_DESPMTR) {
        status = OCSM_WRONG_PMTR_TYPE;
        SPRINT0(0, "ERROR:: \"optvar\" is not an DESPMTR");
        goto cleanup;
    } else if (nrow > 1) {
        status = OCSM_ILLEGAL_VALUE;
        SPRINT0(0, "ERROR:: \"optvar\" must be a row vector");
        goto cleanup;
    } else if (miniopt->noptvar > MAX_OPTVARS) {
        status = OCSM_ILLEGAL_VALUE;
        SPRINT0(0, "ERROR:: increase size of MAX_OPTVARS and recompile");
        goto cleanup;
    } else {
        miniopt->noptvar = ncol;
    }


    status = ocsmFindPmtr(MODL, "constr", 0, 0, 0, &(miniopt->iconstr));
    if (status != SUCCESS) {
        status = EGADS_NOTFOUND;
        SPRINT0(0, "ERROR:: \"constr\" not found");
        goto cleanup;
    }

    status = ocsmGetPmtr(MODL, miniopt->iconstr, &itype, &nrow, &ncol, name);
    CHECK_STATUS(ocsmGetPmtr);

    if (MODL->pmtr[miniopt->iconstr].type != OCSM_OUTPMTR) {
        status = OCSM_WRONG_PMTR_TYPE;
        SPRINT0(0, "ERROR:: \"constr\" is not an OUTPMTR");
        goto cleanup;
    } else if (nrow > 1) {
        status = OCSM_ILLEGAL_VALUE;
        SPRINT0(0, "ERROR:: \"constr\" must be a row vector");
        goto cleanup;
    } else if (miniopt->nconstr > MAX_CONSTRS) {
        status = OCSM_ILLEGAL_VALUE;
        SPRINT0(0, "ERROR:: increase size of MAX_CONSTRS and recompile");
        goto cleanup;
    } else {
        miniopt->nconstr = ncol;
    }

    /* remember the original and new "optvar" values */
    MALLOC(miniopt->optvar_orig, double, miniopt->noptvar);
    MALLOC(miniopt->optvar_new,  double, miniopt->noptvar);

    for (ioptvar = 0; ioptvar < miniopt->noptvar; ioptvar++) {
        status = ocsmGetValu(MODL, miniopt->ioptvar, 1, ioptvar+1,
                             &(optvar[ioptvar]), &dot);
        CHECK_STATUS(ocsmGetValu);

        miniopt->optvar_orig[ioptvar] = optvar[ioptvar];
        miniopt->optvar_new[ ioptvar] = optvar[ioptvar];
    }

    /* lower and upper bounds */
    for (ioptvar = 0; ioptvar < miniopt->noptvar; ioptvar++) {
        status = ocsmGetBnds(miniopt->MODL, miniopt->ioptvar, 1, ioptvar+1,
                             &lb[ioptvar], &ub[ioptvar]);
        CHECK_STATUS(ocsmGetBnds);
    }

    /* tell the user the version of nlopt that is being used */
    nlopt_version(&major, &minor, &bugfix);
    printf("\nStarting optimization using NLopt %d.%d.%d\n", major, minor, bugfix);

    printf("\nThere are %d optimization variables\n", miniopt->noptvar);
    for (ioptvar = 0; ioptvar < miniopt->noptvar; ioptvar++) {
        printf("    optvar[%2d]=%12.6f, lb=%12.6f, ub=%12.6f\n",
               ioptvar+1, miniopt->optvar_orig[ioptvar], lb[ioptvar], ub[ioptvar]);
    }
    printf("There are %d constraints\n\n", miniopt->nconstr);

    /* create the optimization object */
    miniopt->opt = nlopt_create(NLOPT_LD_MMA, miniopt->noptvar);
    if (miniopt->opt == NULL) {
        printf("nlopt_create failed\n");
        exit(EXIT_FAILURE);
    }

    /* define the objective function */
    status = nlopt_set_min_objective(miniopt->opt, myObjfun, miniopt);
    CHECK_STATUS(nlopt_set_min_objective);

    /* apply the bound constraints */
    status = nlopt_set_lower_bounds(miniopt->opt, lb);
    CHECK_STATUS(nlopt_set_lower_bounds);

    status = nlopt_set_upper_bounds(miniopt->opt, ub);
    CHECK_STATUS(nlopt_set_upper_bounds);

    /* define the nonlinear constraints */
    for (iconstr = 0; iconstr < miniopt->nconstr; iconstr++) {
        tol[iconstr] = 1e-8;
    }

    status = nlopt_add_inequality_mconstraint(miniopt->opt, miniopt->nconstr,
                                              myConstraints, miniopt, tol);
    CHECK_STATUS(nlopt_add_inequality_mconstraint);

    /* set the stopping criterion */
    status = nlopt_set_xtol_rel(miniopt->opt, 1e-5);
    CHECK_STATUS(nlopt_set_xtol_rel);

    status = nlopt_set_maxeval(miniopt->opt, 1000);
    CHECK_STATUS(nlopt_set_maxeval);

    /* unset the verification flag (so that verification is not
       done during the builds on myObjfun and myConstraints) */
    miniopt->verify = MODL->verify;
    MODL->verify    = 0;

    /* hold the UI when executing */
    status = 1;

    /* tell the browser the number of optvars anc constrs */
    snprintf(response, MAX_EXPR_LEN, "timLoad|miniopt|%d|%d|",
             miniopt->noptvar, miniopt->nconstr);
    tim_bcst("miniopt", response);

cleanup:
    return status;
}


/***********************************************************************/
/*                                                                     */
/*   timMesg - get command, process, and return response               */
/*                                                                     */
/***********************************************************************/

int
timMesg(esp_T *ESP,                     /* (in)  pointer to ESP structure */
        char  command[])                /* (in)  command */
{
    int    status = EGADS_SUCCESS;      /* (out) return status */

    int     neval, ioptvar, iconstr, buildTo, builtTo, nbody;
    double  value, dot, optvar[MAX_OPTVARS], minobj;
    char    *arg1=NULL, *pEnd, response[MAX_EXPR_LEN+1], temp[MAX_EXPR_LEN+1];

    modl_T  *MODL      =              ESP->MODL;
    miniopt_T *miniopt = (miniopt_T *)(ESP->udata[ESP->nudata-1]);

    ROUTINE(timMesg(miniopt));

    /* --------------------------------------------------------------- */

    /* "exec" */
    if (strncmp(command, "exec|", 5) == 0) {

        /* run the optimizer (from the last step) */
        for (ioptvar = 0; ioptvar < miniopt->noptvar; ioptvar++) {
            optvar[ioptvar] = miniopt->optvar_new[ioptvar];
        }

        /* the maximum number of evaluations */
        neval = 100;
        GetToken(command, 1, '|', &arg1);
        SPLINT_CHECK_FOR_NULL(arg1);
        if (strlen(arg1) > 0) neval = strtol(arg1, &pEnd, 10);
        FREE(arg1);

        status = nlopt_set_maxeval(miniopt->opt, neval);
        CHECK_STATUS(nlopt_set_maxeval);

        /* run the optimizer and report the values to the browser */
        status = nlopt_optimize(miniopt->opt, optvar, &minobj);
        printf("status=%d\n", status);
        CHECK_STATUS(nlopt_optimize);

        snprintf(response, MAX_EXPR_LEN, "timMesg|miniopt|exec|%d|%11.4e|%11.4e|%d|",
                 miniopt->istep, miniopt->objfun_value, miniopt->constr_max, status);
        tim_bcst("miniopt", response);

        strncpy(response, "timMesg|miniopt|optvar| ", MAX_EXPR_LEN);
        for (ioptvar = 0; ioptvar < miniopt->noptvar; ioptvar++) {
            snprintf(temp, MAX_EXPR_LEN, " %11.4e", optvar[ioptvar]);
            strncat(response, temp, MAX_EXPR_LEN);
        }
        strncat(response, "|", MAX_EXPR_LEN);
        tim_bcst("miniopt", response);

        /* remember the results of the optimization */
        for (ioptvar = 0; ioptvar < miniopt->noptvar; ioptvar++) {
            miniopt->optvar_new[ioptvar] = optvar[ioptvar];

            status = ocsmSetValuD(MODL, miniopt->ioptvar, 1, ioptvar+1,
                                  optvar[ioptvar]);
            CHECK_STATUS(ocsmSetValuD);
        }

        /* find the maximum constraint violation */
        miniopt->constr_max = -1e+6;
        for (iconstr = 0; iconstr < miniopt->nconstr; iconstr++) {
            status = ocsmGetValu(MODL, miniopt->iconstr, 1, iconstr+1,
                                 &value, &dot);
            CHECK_STATUS(ocsmGetValu);

            if (value > miniopt->constr_max) {
                miniopt->constr_max = value;
            }
        }

        /* turn the sensitivities off and rebuild with the "best" optvars */
        status = ocsmSetVelD(MODL, 0, 0, 0, 0.0);
        CHECK_STATUS(ocsmSetVelD);

        buildTo = 0;
        builtTo = 0;
        nbody   = 0;

        (void) ocsmSetOutLevel(0);
        status = ocsmBuild(miniopt->MODL, buildTo, &builtTo, &nbody, NULL);
        (void) ocsmSetOutLevel(1);
        CHECK_STATUS(ocsmBuild);

        /* return the Solve button to "Minimize" so that the
           optimization could be restarted */
        tim_bcst("miniopt", "updateSolveBtn|1|Minimize|");
    }

cleanup:
    return status;
}


/***********************************************************************/
/*                                                                     */
/*   timSave - save tim data and close tim instance                    */
/*                                                                     */
/***********************************************************************/

int
timSave(esp_T *ESP)                     /* (in)  pointer to ESP structure */
{
    int    status = EGADS_SUCCESS;      /* (out) return status */

    int       i, ioptvar;
    modl_T    *MODL  = ESP->MODL;
    miniopt_T *miniopt;

    ROUTINE(timSave(miniopt));

    /* --------------------------------------------------------------- */

    if (ESP->nudata <= 0) {
        goto cleanup;
    } else if (strcmp(ESP->timName[ESP->nudata-1], "miniopt") != 0) {
        printf("WARNING:: TIM on top of stack is not \"miniopt\"\n");
        for (i = 0; i < ESP->nudata; i++) {
            printf("   timName[%d]=%s\n", i, ESP->timName[i]);
        }
        goto cleanup;
    } else {
        miniopt = (miniopt_T *)(ESP->udata[ESP->nudata-1]);
    }

    if (miniopt == NULL) {
        goto cleanup;
    }

    /* reset the verification flag */
    MODL->verify = miniopt->verify;

    /* set all DESPMTRs to their new values */
    for (ioptvar = 0; ioptvar < miniopt->noptvar; ioptvar++) {
        status = ocsmSetValuD(MODL, miniopt->ioptvar, 1, ioptvar+1,
                              miniopt->optvar_new[ioptvar]);
        CHECK_STATUS(ocsmSetValuD);
    }

    /* free nlopt's context */
    nlopt_destroy(miniopt->opt);

    /* remove the allocated data from the MINIOPT structure */
    FREE(miniopt->optvar_orig);
    FREE(miniopt->optvar_new );

    FREE(ESP->udata[ESP->nudata-1]);
    ESP->timName[   ESP->nudata-1][0] = '\0';
    ESP->nudata--;

    tim_bcst("miniopt", "timSave|miniopt|");

cleanup:
    return status;
}


/***********************************************************************/
/*                                                                     */
/*   timQuit - close tim instance without saving                       */
/*                                                                     */
/***********************************************************************/

int
timQuit(esp_T *ESP,                     /* (in)  pointer to ESP structure */
/*@unused@*/int   unload)               /* (in)  flag to unload */
{
    int    status = EGADS_SUCCESS;      /* (out) return status */

    int       i, ioptvar;
    modl_T    *MODL  = ESP->MODL;
    miniopt_T *miniopt;

    ROUTINE(timQuit(miniopt));

    /* --------------------------------------------------------------- */

    if (ESP->nudata <= 0) {
        goto cleanup;
    } else if (strcmp(ESP->timName[ESP->nudata-1], "miniopt") != 0) {
        printf("WARNING:: TIM on top of stack is not \"miniopt\"\n");
        for (i = 0; i < ESP->nudata; i++) {
            printf("   timName[%d]=%s\n", i, ESP->timName[i]);
        }
        goto cleanup;
    } else {
        miniopt = (miniopt_T *)(ESP->udata[ESP->nudata-1]);
    }

    if (miniopt == NULL) {
        goto cleanup;
    }

    /* reset the verification flag */
    MODL->verify = miniopt->verify;

    /* return all DESPMTRs to their original values */
    for (ioptvar = 0; ioptvar < miniopt->noptvar; ioptvar++) {
        status = ocsmSetValuD(MODL, miniopt->ioptvar, 1, ioptvar+1,
                              miniopt->optvar_orig[ioptvar]);
        CHECK_STATUS(ocsmSetValuD);
    }

    /* free nlopt's context */
    nlopt_destroy(miniopt->opt);

    /* remove the allocated data from the MINIOPT structure */
    FREE(miniopt->optvar_orig);
    FREE(miniopt->optvar_new );

    FREE(ESP->udata[ESP->nudata-1]);
    ESP->timName[   ESP->nudata-1][0] = '\0';
    ESP->nudata--;

    tim_bcst("miniopt", "timQuit|miniopt|");

cleanup:
    return status;
}


/***********************************************************************/
/*                                                                     */
/*   myObjfun - objective function                                     */
/*                                                                     */
/***********************************************************************/

static double
myObjfun(unsigned     noptvar,          /* (in)  number of design variables */
         const double optvar[],         /* (in)  array  of design variables */
         double       grad[],           /* (out) gradient of objective function */
         void         *userdata)        /* (in)  problem-specific data */
{
    double   objfun;

    int        status, ioptvar, buildTo, builtTo, nbody;
    double     objfun_dot, temp;
    miniopt_T   *miniopt = (miniopt_T *)userdata;

    ROUTINE(myObjfun);

    /* --------------------------------------------------------------- */

    /* increment number of objective function evaluations */
    (miniopt->istep)++;

    /* default objective function */
    objfun = -HUGEQ;

    /* remember the new values of optvar */
    for (ioptvar = 0; ioptvar < noptvar; ioptvar++) {
        miniopt->optvar_new[ioptvar] = optvar[ioptvar];
    }

    /* transfer the optimization variables to OCSM */
    for (ioptvar = 0; ioptvar < noptvar; ioptvar++) {
        status = ocsmSetValuD(miniopt->MODL, miniopt->ioptvar, 1, ioptvar+1,
                              optvar[ioptvar]);
        CHECK_STATUS(ocsmSetValuD);
    }

    /* set up for first sensitivity */
    status = ocsmSetVelD(miniopt->MODL, 0, 0, 0, 0.0);
    CHECK_STATUS(ocsmSetVelD);

    status = ocsmSetVelD(miniopt->MODL, miniopt->ioptvar, 1, 1, 1);
    CHECK_STATUS(ocsmSetVelD);

    /* build the MODL (with the first sensitivity) */
    buildTo = 0;
    builtTo = 0;
    nbody   = 0;

    (void) ocsmSetOutLevel(0);
    status = ocsmBuild(miniopt->MODL, buildTo, &builtTo, &nbody, NULL);
    (void) ocsmSetOutLevel(1);
    CHECK_STATUS(ocsmBuild);

    /* get the objective function */
    status = ocsmGetValu(miniopt->MODL, miniopt->iobjfun, 1, 1,
                         &objfun, &objfun_dot);
    CHECK_STATUS(ocsmGetValu);

    miniopt->objfun_value = objfun;

    /* gradient of objective function */
    if (grad != NULL) {

        /* first gradient */
        grad[0] = objfun_dot;

        /* get the rest of the sensitivities */
        for (ioptvar = 1; ioptvar < noptvar; ioptvar++) {
            status = ocsmSetVelD(miniopt->MODL, 0, 0, 0, 0.0);
            CHECK_STATUS(ocsmSetVelD);

            status = ocsmSetVelD(miniopt->MODL, miniopt->ioptvar, 1, ioptvar+1, 1);
            CHECK_STATUS(ocsmSetVelD);

            (void) ocsmSetOutLevel(0);
            status = ocsmBuild(miniopt->MODL, buildTo, &builtTo, &nbody, NULL);
            (void) ocsmSetOutLevel(1);
            CHECK_STATUS(ocsmBuild);

            status = ocsmGetValu(miniopt->MODL, miniopt->iobjfun, 1, 1,
                                 &temp, &objfun_dot);
            CHECK_STATUS(ocsmGetValu);

            grad[ioptvar] = objfun_dot;
        }
    }

    printf("%5d, desvars=", miniopt->istep);
    for (ioptvar = 0; ioptvar < noptvar; ioptvar++) {
        printf(" %12.7f", optvar[ioptvar]);
    }
    printf(", objfun=%12.7f\n", objfun);

cleanup:
    return objfun;
}


/***********************************************************************/
/*                                                                     */
/*   myConstraints - constraint function                               */
/*                                                                     */
/***********************************************************************/

static void
myConstraints(unsigned     nconstr,     /* (in)  number of constraints */
              double       constr[],    /* (out) constraint values */
              unsigned     noptvar,     /* (in)  number of design variables */
              const double optvar[],    /* (in)  array  of design variables */
              double       grad[],      /* (out) gradient of constraints */
              void         *userdata)   /* (in)  problem-specific data */
{
    int        status, ioptvar, iconstr, buildTo, builtTo, nbody;
    double     constr_dot[MAX_CONSTRS], temp;
    char       response[MAX_EXPR_LEN];
    miniopt_T   *miniopt = (miniopt_T *)userdata;

    ROUTINE(myConstraints);

    /* --------------------------------------------------------------- */

    /* transfer the design variables to OCSM */
    for (ioptvar = 0; ioptvar < noptvar; ioptvar++) {
        status = ocsmSetValuD(miniopt->MODL, miniopt->ioptvar, 1, ioptvar+1,
                              optvar[ioptvar]);
        CHECK_STATUS(ocsmSetValuD);
    }

    /* set up for the sensitivity w.r.t. the first optimization variable */
    status = ocsmSetVelD(miniopt->MODL, 0,                0, 0, 0.0);
    CHECK_STATUS(ocsmSetVelD);

    status = ocsmSetVelD(miniopt->MODL, miniopt->ioptvar, 1, 1, 1.0);
    CHECK_STATUS(ocsmSetVelD);

    /* build the MODL (with the first sensitivity) */
    buildTo = 0;
    builtTo = 0;
    nbody   = 0;

    (void) ocsmSetOutLevel(0);
    status = ocsmBuild(miniopt->MODL, buildTo, &builtTo, &nbody, NULL);
    (void) ocsmSetOutLevel(1);
    CHECK_STATUS(ocsmBuild);

    /* get the constraints */
    for (iconstr = 0; iconstr < nconstr; iconstr++) {
        status = ocsmGetValu(miniopt->MODL, miniopt->iconstr, 1, iconstr+1,
                             &constr[iconstr], &constr_dot[iconstr]);
        CHECK_STATUS(ocsmGetValu);
    }

    /* gradient of the constraints */
    if (grad != NULL) {

        /* first gradient w.r.t. first optimization variable */
        ioptvar = 0;
        for (iconstr = 0; iconstr < nconstr; iconstr++) {
            grad[ioptvar+iconstr*noptvar] = constr_dot[iconstr];
        }

        /* get the rest of the gradients */
        for (ioptvar = 1; ioptvar < noptvar; ioptvar++) {

            status = ocsmSetVelD(miniopt->MODL, 0,                    0, 0,         0.0);
            CHECK_STATUS(ocsmSetVelD);

            status = ocsmSetVelD(miniopt->MODL, miniopt->ioptvar, 1, ioptvar+1, 1.0);
            CHECK_STATUS(ocsmSetVelD);

            (void) ocsmSetOutLevel(0);
            status = ocsmBuild(miniopt->MODL, buildTo, &builtTo, &nbody, NULL);
            (void) ocsmSetOutLevel(1);
            CHECK_STATUS(ocsmBuild);

            for (iconstr = 0; iconstr < nconstr; iconstr++) {
                status = ocsmGetValu(miniopt->MODL, miniopt->iconstr, 1, iconstr+1,
                                     &temp, &constr_dot[iconstr]);
                CHECK_STATUS(ocsmGetValu);

                grad[ioptvar+iconstr*noptvar] = constr_dot[iconstr];
            }
        }
    }

    printf("       desvars=");
    for (ioptvar = 0; ioptvar < noptvar; ioptvar++) {
        printf(" %12.7f", optvar[ioptvar]);
    }
    printf("                      constr=");
    for (iconstr = 0; iconstr < nconstr; iconstr++) {
        printf(" %12.7f", constr[iconstr]);
    }
    printf("\n");

    /* maximum constraint value */
    miniopt->constr_max = -HUGEQ;
    for (iconstr = 0; iconstr < nconstr; iconstr++) {
        if (constr[iconstr] > miniopt->constr_max) {
            miniopt->constr_max = constr[iconstr];
        }
    }

    /* report this step to  the browser */
    snprintf(response, MAX_EXPR_LEN, "timMesg|miniopt|exec|%d|%11.4e|%11.4e|%d|",
             miniopt->istep, miniopt->objfun_value, miniopt->constr_max, 0);
    tim_bcst("miniopt", response);

cleanup:
    return;
}

