 /*
 ************************************************************************
 *                                                                      *
 * udfFlend3 -- create a flend on an attributed Body                    *
 *                                                                      *
 *            Written by John Dannenhoffer @ Syracuse University        *
 *            Patterned after code written by Bob Haimes  @ MIT         *
 *                                                                      *
 ************************************************************************
 */

/*
 * Copyright (C) 2011/2025  John F. Dannenhoffer, III (Syracuse University)
 *
 * This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *     MA  02110-1301  USA
 */

#define NUMUDPARGS 2
#define NUMUDPINPUTBODYS 1
#include "udpUtilities.h"

/* shorthands for accessing argument values and velocities */
#define ATTR1(IUDP)  ((char   *) (udps[IUDP].arg[0].val))
#define ATTR2(IUDP)  ((char   *) (udps[IUDP].arg[1].val))

/* data about possible arguments */
static char  *argNames[NUMUDPARGS] = {"attr1",    "attr2",    };
static int    argTypes[NUMUDPARGS] = {ATTRSTRING, ATTRSTRING, };
static int    argIdefs[NUMUDPARGS] = {0,          0,          };
static double argDdefs[NUMUDPARGS] = {0.,         0.,         };

/* get utility routines: udpErrorStr, udpInitialize, udpReset, udpSet,
                         udpGet, udpVel, udpClean, udpMesh */
#include "udpUtilities.c"

#include "OpenCSM.h"
#include "mateContours.h"
#include <assert.h>

#define EPS12     1.0e-12
#define EPS30     1.0e-30

#define MIN(A,B)  (((A) < (B)) ? (A) : (B))
#define MAX(A,B)  (((A) < (B)) ? (B) : (A))

#ifdef GRAFIC
   #include "grafic.h"
#endif

/* prototype of unpublished EGADS function */
extern int EG_relPosTs(ego ecurve, int npnt, /*@null@*/ const double newS[], double newT[], double newXYZ[]);

/* message (to be shared by all functions) */
static char message[1024];


/*
 ************************************************************************
 *                                                                      *
 *   udpExecute - execute the primitive                                 *
 *                                                                      *
 ************************************************************************
 */

int
udpExecute(ego  emodel,                 /* (in)  Model containing Body */
           ego  *ebody,                 /* (out) Body pointer */
           int  *nMesh,                 /* (out) number of associated meshes */
           char *string[])              /* (out) error message */
{
    int     status = EGADS_SUCCESS;

    int     oclass, mtype, nchild, *senses;
    double  data[18];
    ego     eref, *ebodys, *echilds;
    udp_T   *udps = *Udps;

    int ncontour, *nent, *contr1, *contr2;
    int    ipair, icontour, ient, iedge1, isens1, iedge2, isens2, ndum, nlink, nslink, nsub, isub, jsub;
    double t1[18], xyz1[18], t2[18], xyz2[18], xyzlink[1000], xyzslink[10000], s1, s2, len;
    double ts1[50], xyzs1[150], ts2[50], xyzs2[150];
    ego    enode1, enode2, eedge1, eedge2, *edum;
    int  ilink;
    FILE *fp;

#ifdef DEBUG
    int     i;
#endif

    ROUTINE(udpExecute);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    printf("udpExecute(emodel=%llx)\n", (long long)emodel);
    printf("attr1(0) = %s\n", ATTR1(0));
    printf("attr2(0) = %s\n", ATTR2(0));
#endif

    /* default return values */
    *ebody  = NULL;
    *nMesh  = 0;
    *string = NULL;

    message[0] = '\0';

    /* check that Model was input that contains one Body */
    status = EG_getTopology(emodel, &eref, &oclass, &mtype,
                            data, &nchild, &ebodys, &senses);
    CHECK_STATUS(EG_getTopology);

    if (oclass != MODEL) {
        snprintf(message, 1024, "expecting a Model\n");
        status = EGADS_NOTMODEL;
        goto cleanup;
    } else if (nchild != 1) {
        snprintf(message, 1024, "expecting Model to contain one Body (not %d)\n", nchild);
        status = EGADS_NOTBODY;
        goto cleanup;
    }

    /* verify that the given attributes are different */
    if (strcmp(ATTR1(0), ATTR2(0)) == 0) {
        snprintf(message, 1024, "attr1 and attr2 are the same\n");
        status = EGADS_RANGERR;
        goto cleanup;
    }

    /* make sure input is a SheetBody or SolidBody */
    status = EG_getTopology(ebodys[0], &eref, &oclass, &mtype,
                            data, &nchild, &echilds, &senses);
    CHECK_STATUS(EG_getTopology);

    if (mtype != FACEBODY && mtype != SHEETBODY && mtype != SOLIDBODY) {
        snprintf(message, 1024, "Body is not a SheetBody or SOlidBody\n");
        status = EGADS_NOTBODY;
        goto cleanup;
    }

    /* cache copy of arguments for future use */
    status = cacheUdp(NULL);
    CHECK_STATUS(cacheUdp);

#ifdef DEBUG
    printf("attr1(%d) = %s\n", numUdp, ATTR1(numUdp));
    printf("attr2(%d) = %s\n", numUdp, ATTR2(numUdp));
#endif

    /* add Nodes to the input Body so that the Loops with the
       given attributes have matching Nodes */
    status = EG_mateContours(ebodys[0], ATTR1(0), ATTR2(0),
                             ebody, &ncontour, &nent, &contr1, &contr2);
    CHECK_STATUS(EG_mateContours);

    SPLINT_CHECK_FOR_NULL(*ebody);

    /* create links between matching Nodes in the two contours */
    ipair = 0;
    nlink = 0;
    nslink = 0;
    for (icontour = 0; icontour < ncontour; icontour++) {
        for (ient = 0; ient < nent[icontour]; ient++) {
            iedge1 = contr1[2*ipair  ];
            isens1 = contr1[2*ipair+1];
            iedge2 = contr2[2*ipair  ];
            isens2 = contr2[2*ipair+1];

            if (isens1 == 0) {
                status = EG_objectBodyTopo(*ebody, NODE, iedge1, &enode1);
                CHECK_STATUS(EG_objectBodyTopo);

                status = EG_getTopology(enode1, &eref, &oclass, &mtype,
                                        xyz1, &nchild, &echilds, &senses);
                CHECK_STATUS(EG_getTopology);

                s1 = 0;
            } else if (isens1 > 0) {
                status = EG_objectBodyTopo(*ebody, EDGE, iedge1, &eedge1);
                CHECK_STATUS(EG_objectBodyTopo);

                status = EG_getTopology(eedge1, &eref, &oclass, &mtype,
                                        t1, &nchild, &echilds, &senses);
                CHECK_STATUS(EG_getTopology);

                status = EG_getTopology(echilds[0], &eref, &oclass, &mtype,
                                        xyz1, &ndum, &edum, &senses);
                CHECK_STATUS(EG_getTopology);

                status = EG_arcLength(eedge1, t1[0], t1[1], &s1);
                CHECK_STATUS(EG_arcLength);
            } else {
                status = EG_objectBodyTopo(*ebody, EDGE, iedge1, &eedge1);
                CHECK_STATUS(EG_objectBodyTopo);

                status = EG_getTopology(eedge1, &eref, &oclass, &mtype,
                                        t1, &nchild, &echilds, &senses);
                CHECK_STATUS(EG_getTopology);

                status = EG_getTopology(echilds[1], &eref, &oclass, &mtype,
                                        xyz1, &ndum, &edum, &senses);
                CHECK_STATUS(EG_getTopology);

                status = EG_arcLength(eedge1, t1[0], t1[1], &s1);
                CHECK_STATUS(EG_arcLength);
            }

            if (isens2 == 0) {
                status = EG_objectBodyTopo(*ebody, NODE, iedge2, &enode2);
                CHECK_STATUS(EG_objectBodyTopo);

                status = EG_getTopology(enode2, &eref, &oclass, &mtype,
                                        xyz2, &nchild, &echilds, &senses);
                CHECK_STATUS(EG_getTopology);

                s2 = 0;
            } else if (isens2 > 0) {
                status = EG_objectBodyTopo(*ebody, EDGE, iedge2, &eedge2);
                CHECK_STATUS(EG_objectBodyTopo);

                status = EG_getTopology(eedge2, &eref, &oclass, &mtype,
                                        t2, &nchild, &echilds, &senses);
                CHECK_STATUS(EG_getTopology);

                status = EG_getTopology(echilds[0], &eref, &oclass, &mtype,
                                        xyz2, &ndum, &edum, &senses);
                CHECK_STATUS(EG_getTopology);

                status = EG_arcLength(eedge2, t2[0], t2[1], &s2);
                CHECK_STATUS(EG_arcLength);
            } else {
                status = EG_objectBodyTopo(*ebody, EDGE, iedge2, &eedge2);
                CHECK_STATUS(EG_objectBodyTopo);

                status = EG_getTopology(eedge2, &eref, &oclass, &mtype,
                                        t2, &nchild, &echilds, &senses);
                CHECK_STATUS(EG_getTopology);

                status = EG_getTopology(echilds[1], &eref, &oclass, &mtype,
                                        xyz2, &ndum, &edum, &senses);
                CHECK_STATUS(EG_getTopology);

                status = EG_arcLength(eedge2, t2[0], t2[1], &s2);
                CHECK_STATUS(EG_arcLength);
            }

            /* determine the number of sublinks */
            len = sqrt(  (xyz2[0]-xyz1[0]) * (xyz2[0]-xyz1[0])
                       + (xyz2[1]-xyz1[1]) * (xyz2[1]-xyz1[1])
                       + (xyz2[2]-xyz1[2]) * (xyz2[2]-xyz1[2]));
            nsub = MIN(2 + MAX(s1,s2) / (len/4), 50);

            printf("s1=%10.5f, s2=%10.5f, len=%10.5f\n", s1, s2, len);

            printf("beg=%10.5f %10.5f %10.5f   end=%10.5f %10.5f %10.5f   nsub=%d\n",
                   xyz1[0], xyz1[1], xyz1[2], xyz2[0], xyz2[1], xyz2[2], nsub);

            /* create the links */
            xyzlink[6*nlink  ] = xyz1[0];
            xyzlink[6*nlink+1] = xyz1[1];
            xyzlink[6*nlink+2] = xyz1[2];
            xyzlink[6*nlink+3] = xyz2[0];
            xyzlink[6*nlink+4] = xyz2[1];
            xyzlink[6*nlink+5] = xyz2[2];
            nlink++;

            /* create the sublinks */
            if (isens1 == 0) {
                for (isub = 0; isub < nsub; isub++) {
                    xyzs1[3*isub  ] = xyz1[0];
                    xyzs1[3*isub+1] = xyz1[1];
                    xyzs1[3*isub+2] = xyz1[2];
                }
            } else {
                ts1[0     ] = t1[0];
                ts1[nsub-1] = t1[1];
                status = EG_relPosTs(eedge1, nsub, NULL, ts1, xyzs1);
                CHECK_STATUS(EG_relPosT);
            }

            if (isens2 == 0) {
                for (isub = 0; isub < nsub; isub++) {
                    xyzs2[3*isub  ] = xyz2[0];
                    xyzs2[3*isub+1] = xyz2[1];
                    xyzs2[3*isub+2] = xyz2[2];
                }
            } else {
                ts2[0     ] = t2[0];
                ts2[nsub-1] = t2[1];
                status = EG_relPosTs(eedge2, nsub, NULL, ts2, xyzs2);
                CHECK_STATUS(EG_relPosT);
            }

#ifndef __clang_analyzer__
            for (isub = 1; isub < nsub-1; isub++) {
                jsub = nsub-1 - isub;
                xyzslink[6*nslink  ] = xyzs1[3*isub  ];
                xyzslink[6*nslink+1] = xyzs1[3*isub+1];
                xyzslink[6*nslink+2] = xyzs1[3*isub+2];
                if (isens2 >= 0) {
                    xyzslink[6*nslink+3] = xyzs2[3*isub  ];
                    xyzslink[6*nslink+4] = xyzs2[3*isub+1];
                    xyzslink[6*nslink+5] = xyzs2[3*isub+2];
                } else {
                    xyzslink[6*nslink+3] = xyzs2[3*jsub  ];
                    xyzslink[6*nslink+4] = xyzs2[3*jsub+1];
                    xyzslink[6*nslink+5] = xyzs2[3*jsub+2];
                }
                nslink++;
            }
#endif

            ipair++;
        }
    }

    EG_free(nent);
    EG_free(contr1);
    EG_free(contr2);

    fp = fopen("flend.plot", "w");
    if (fp != NULL) {
        fprintf(fp, "%5d %5d links\n", nlink, -1);
        for (ilink = 0; ilink < nlink; ilink++) {
            fprintf(fp, "%12.6f %12.6f %12.6f   %12.6f %12.6f %12.6f\n",
                    xyzlink[6*ilink  ], xyzlink[6*ilink+1], xyzlink[6*ilink+2],
                    xyzlink[6*ilink+3], xyzlink[6*ilink+4], xyzlink[6*ilink+5]);
        }
        fprintf(fp, "%5d %5d sublinks\n", nslink, -1);
        for (ilink = 0; ilink < nslink; ilink++) {
            fprintf(fp, "%12.6f %12.6f %12.6f   %12.6f %12.6f %12.6f\n",
                    xyzslink[6*ilink  ], xyzslink[6*ilink+1], xyzslink[6*ilink+2],
                    xyzslink[6*ilink+3], xyzslink[6*ilink+4], xyzslink[6*ilink+5]);
        }
        fprintf(fp, "%5d %5d end\n", 0, 0);
        fclose(fp);
    }

    /* set the output value */

    /* return the new Body */
    udps[numUdp].ebody = *ebody;

cleanup:
    if (strlen(message) > 0) {
        MALLOC(*string, char, 1024);
        strncpy(*string, message, 1023);
    } else if (status != EGADS_SUCCESS) {
        *string = udpErrorStr(status);
    }

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   udpSensitivity - return sensitivity derivatives for the "real" argument *
 *                                                                      *
 ************************************************************************
 */

int
udpSensitivity(ego    ebody,            /* (in)  Body pointer */
   /*@unused@*/int    npnt,             /* (in)  number of points */
   /*@unused@*/int    entType,          /* (in)  OCSM entity type */
   /*@unused@*/int    entIndex,         /* (in)  OCSM entity index (bias-1) */
   /*@unused@*/double uvs[],            /* (in)  parametric coordinates for evaluation */
   /*@unused@*/double vels[])           /* (out) velocities */
{
    int iudp, judp;

    ROUTINE(udpSensitivity);

    /* --------------------------------------------------------------- */

    /* check that ebody matches one of the ebodys */
    iudp = 0;
    for (judp = 1; judp <= numUdp; judp++) {
        if (ebody == udps[judp].ebody) {
            iudp = judp;
            break;
        }
    }
    if (iudp <= 0) {
        return EGADS_NOTMODEL;
    }

    /* this routine is not written yet */
    return EGADS_NOLOAD;
}
